package CryptoTron::GetAccount;

# Load the Perl pragmas.
use 5.010000;
use strict;
use warnings;

# Load the Perl pragma Exporter.
use vars qw(@ISA @EXPORT @EXPORT_OK);
use Exporter 'import';

# Exporting the implemented subroutine.
our @EXPORT = qw(GetAccount);

# Base class of this (tron_addr) module.
our @ISA = qw(Exporter);

# Set the package version. 
our $VERSION = '0.10';

# Load the required Perl modules or packages.
use Try::Catch;
use URI;
use POSIX;
use JSON::PP;
use LWP::UserAgent;

# Define the global variables.
our($HEADER, $SERVICE_URL);

# Set api url and api path.
our $API_URL = 'https://api.trongrid.io';
our $API_PATH = '/walletsolidity/getaccount';

# Set the request header.
$HEADER = [Accept => 'application/json',
           Content_Type => 'application/json'];

# Assemble the service url.
$SERVICE_URL = $API_URL.$API_PATH;

# ---------------------------------------------------------------------------- #
# Function encode()                                                            #
#                                                                              #
# Description:                                                                 # 
# The subroutine is decoding and encoding the given content.                   #
#                                                                              #
# @argument $content      Response content  (scalar)                           #
# @returns  $json_encode  Encoded content   (scalar)                           #
# ---------------------------------------------------------------------------- #
sub encode {
    # Assign the argument to the local variable.
    my $content = $_[0];
    # Set up the options for the Perl module.
    my $json = 'JSON::PP'->new->pretty;
    # Decode the content of the response.
    my $json_decode = $json->decode($content);
    # Encode the decoded content of the response.
    my $json_encode = $json->encode($json_decode);
    # Return encoded and decoded data.
    return $json_encode;
};

# ---------------------------------------------------------------------------- #
# Function get_response()                                                      # 
#                                                                              #
# Description:                                                                 #
# The subroutine is using the HTTP method POST to retrieve the response from   #
# the given service url.                                                       #
#                                                                              #
# @argument $service_url  Service url       (scalar)                           #
# @return   $content      Response content  (scalar)                           #
# ---------------------------------------------------------------------------- #
sub get_response {
    # Assign the argument to the local variable.
    my $payload = $_[0];
    # Declare the variable $content.
    my $content = "";
    # Create a new uri object.
    my $uri = URI->new($SERVICE_URL);
    # Create a new user agent object.
    my $ua = LWP::UserAgent->new;
    # Get the response from the uri.
    my $response = $ua->post($uri, $HEADER, Content => $payload);
    # Check success of operation.
    if ($response->is_success) {
        # Get the content from the response.
        $content = $response->content;
    };
    # Return the content.
    return $content;
};

# ---------------------------------------------------------------------------- #
# Subroutine GetAccount()                                                      #
#                                                                              #
# Description:                                                                 #
# Get the account information of a Tron account from the Tron blockchain.      #
#                                                                              #
# @argument $address    Base58 address      (scalar)                           #
#           $flag       Output format flag  (scalar)                           # 
# @return   $json_data  Response content    (scalar)                           #
# ---------------------------------------------------------------------------- #
sub GetAccount {
    # Assign the argument(s) to the local variable.
    my ($args) = @_;
    # Set the local variables.
    my $address = $args->{address};
    my $flag = $args->{flag};
    # Check the first argument.
    $address = (defined $address ? $address : "");
    # Check the second argument.
    $flag = (defined $flag ? $flag : "");
    # Initialise the return variable.
    my $json_data = ""; 
    # Initialise the other variables. 
    my $payload = "";
    my $content = "";
    # Check address.
    if ($address ne "") {
        # Assemble the payload from the address.
        $payload = "\{\"address\":\"$address\",\"visible\":\"True\"\}";
        # Get the content from the service url.
        $content = get_response($payload);
        # Format the content for the output.
        if ($flag eq "RAW") {
            $json_data = $content;    
        } else {
            # Encode the content.
            $json_data = encode($content);
        };
    };
    # Return the json data.
    return $json_data;
};

1;

__END__

=head1 NAME

CryptoTron::GetAccount - Perl extension for use with the blockchain of the crypto coin Tron.

=head1 SYNOPSIS

  use CryptoTron::GetAccount;

  # Declare the public keys.
  my $PublicKeyBase58 = 'TQHgMpVzWkhSsRB4BzZgmV8uW4cFL8eaBr';

  # Set the output format flag.
  my $OutputFlag = ["RAW"|"STR"];

  # Get the account info as JSON string.
  my $account_info = GetAccount({
      address => $PublicKeyBase58,
      [flag    => $OutputFlag]
  });

  # Print the account info into the terminal window.
  print $account_info;

=head1 DESCRIPTION

The module requests the account information of a Tron account from the Tron
blockchain using the so-called full-node HTTP API from Tron. Requests can be
of type POST or GET. The Tron method is using POST. The switch visible is set
to True in the request, which means that Base58 Tron addresses are used. A
request results in a response in JSON format. The module is able to return raw
JSON data as well as string formated JSON data.

=head1 SEE ALSO

Try::Catch;

POSIX;

URI;

LWP::UserAgent;

JSON::PP;

=head1 AUTHOR

Dr. Peter Netz, E<lt>ztenretep@cpan.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2022 by Dr. Peter Netz

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.30.0 or,
at your option, any later version of Perl 5 you may have available.

=cut
