package Algorithm::Easing::Elastic;

use Moo;

use Math::Trig qw(:pi);

use constant EPSILON => 0.000001;

extends 'Algorithm::Easing::Ease';

use namespace::clean;

sub ease_in  {
    my $self = shift;
    my ($t,$b,$c,$d) = (shift,shift,shift,shift);

    return $b if ($t < EPSILON);
    return $c if ($d < EPSILON);

    if ($t == 0) {
        return $b;
    }

    if (($t /= $d) == 1) {
        return $b + $c;
    }

    my $p = $d * 0.3;
    my $a = $c;
    my $s = $p / 4;

    my $post_fix = $a * $self->pow(2, 10 * ($t -= 1));
    return -($post_fix * sin(($t * $d - $s) * ( 2 * pi)/ $p )) + $b;
}

sub ease_out {
    my $self = shift;
    my ($t,$b,$c,$d) = (shift,shift,shift,shift);

    return $b if ($t < EPSILON);
    return $c if ($d < EPSILON);

    if ($t==0) {
        return $b;
    }

    if (($t /= $d) == 1) {
        return $b + $c;
    }

    my $p = $d * 0.3;
    my $a = $c;
    my $s = $p / 4;

    return ($a * $self->pow(2, -10 * $t) * sin(($t * $d - $s) * (2 * pi) / $p ) + $c + $b);
}

sub ease_both {
    my $self = shift;
    my ($t,$b,$c,$d) = (shift,shift,shift,shift);

    return $b if ($t < EPSILON);
    return $c if ($d < EPSILON);

    if ($t == 0) {
        return $b;
    }

    if (($t /= $d / 2) == 2) {
        return $b + $c;
    }

    my $p = $d * (0.3 * 1.5);
    my $a = $c;
    my $s = $p / 4;

    if ($t < 1) {
        my $post_fix = $a * $self->pow(2,10 * ($t-=1));
        return -0.5*($post_fix * sin(($t * $d - $s) * (2 * pi) / $p )) + $b;
    }

    my $post_fix =  $a * $self->pow(2,-10 * ($t-=1));
    return $post_fix * sin(($t * $d - $s) * (2 * pi)/$p ) * 0.5 + $c + $b;
}

1;

__END__

# MAN3 POD

=head1 NAME

Algorithm::Easing::Elastic - Calculate eased translations between two positive whole integer values over time

=back
=head1 SYNOPSIS

        ## with mediator

        use Algorithm::Easing;
        use Algorithm::Easing::Elastic;

        # this example produces traditional 'elastic' output;

        my $translation = Algorithm::Easing::Elastic->new;

        # total time for eased translation as a real positive integer value
        my $d = 2.5;

        # begin
        my $b = 0;

        # change
        my $c = 200;

        # time passed in seconds as a real positive integer between each frame
        my $frame_time = 0.0625;

        my @p = [319,0];

        for(my $t = 0; $t < 2.5; $t += 0.0625) {
            $p[1] = $translation->ease_out($t,$b,$c,$d)

            # plot
            ...;
        }

=back
=over 2
=item ease_none
    Usage :
    
        Arguments : 
            t = time,
            b = begin,
            c = change,
            d = duration,
        Return :
            p = position,
            
        my $p = $obj->ease_both($t,$b,$c,$d);

This method is used for a linear translation between two positive real whole integers using a positive real integer as the parameter for time.

=back
=over 2
=item ease_in
    Usage :
    
        Arguments : 
            t = time,
            b = begin,
            c = change,
            d = duration,
        Return :
            p = position,
            
        my $p = $obj->ease_both($t,$b,$c,$d);

This method is used for a eased translation between two positive real whole integers as an inward tween using a positive real integer as the parameter for time.

=back
=over 2
=item ease_out
    Usage :
    
        Arguments : 
            t = time,
            b = begin,
            c = change,
            d = duration,
        Return :
            p = position,
            
        my $p = $obj->ease_both($t,$b,$c,$d);

This method is used for a eased translation between two positive real whole integers as an outward tween using a positive real integer as the parameter for time.

=back
=over 2
=item ease_both
    Usage :
    
        Arguments : 
            t = time,
            b = begin,
            c = change,
            d = duration,
        Return :
            p = position,
            
        my $p = $obj->ease_both($t,$b,$c,$d);

This method is used for a eased translation between two positive real whole integers as an inward then outward tween using a positive real integer as the parameter for time.

=head1 AUTHOR

Jason McVeigh, <jmcveigh@outlook.com>

=back
=head1 COPYRIGHT AND LICENSE

Copyright 2016 by Jason McVeigh

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=back
=cut