package App::Wubot::Web::NotifyWeb;
use strict;
use warnings;

# VERSION

use Mojo::Base 'Mojolicious::Controller';

use Lingua::Translate;
use Log::Log4perl;

use App::Wubot::Logger;
use App::Wubot::SQLite;
use App::Wubot::Util::Colors;
use App::Wubot::Util::TagPredict;
use App::Wubot::Util::TimeLength;
use App::Wubot::Util::Taskbot;
use App::Wubot::Web::Obj::NotifyObj;

=head1 NAME

App::Wubot::Web::NotifyWeb - web interface for wubot notifications

=head1 CONFIGURATION

   ~/wubot/config/webui.yaml

    ---
    plugins:
      notify:
        '/notify': notify
        '/notify/id/(.id)': item
        '/tags': tags
        '/colors': colors

=head1 DESCRIPTION

The wubot web interface is still under construction!

The notification web interface serves as your notification inbox.  You
can browse through the unseen notifications, mark them seen, limit the
display to specific plugins or usernames, apply tags, or mark them for
later review.

By default, items in the inbox are grouped and collapsed based on the
'mailbox' field defined in the message.  You can easily use rules to
alter the defaults.  For example, to do this directly in the monitor
config, add a 'react' section such as:

  ---
  enable: 1

  react:
    - name: categorize
      plugin: SetField
      config:
        field: mailbox
        value: test

By convention, wubot messages that are worthy of your attention will
contain a 'subject' field describing the event.  This could be the
subject of an email or rss feed, a tweet, a description of a disk
space problem, etc.  For more information on wubot notifications, see
also L<App::Wubot::Guide::Notifications>.

In order to use the notification web interface, you will first need to
define a rule in the reactor to store the message in the notifications
table.  This can be done with a rule such as:

  - name: notify sql table
    plugin: SQLite
    config:
      file: /Users/wu/wubot/sqlite/notify.sql
      tablename: notifications

Make sure you have copied the 'notifications.yaml' and 'tags.yaml'
schema files from config/schemas/ in the wubot distribution into your
~/wubot/schemas/.

Notifications will be displayed in reverse order with the newest
notification at the top.  The last 100 notifications will be
displayed.  Duplicate notifications will be collapsed into a single
line and will contain a 'count' field that will show how many of the
last 100 notifications were duplicates of this notification.  So, if
you have more than 100 unseen notifications, the notifications count
will be incomplete.  A fix is planned for this.

You can click on the 'seen' link to mark the item seen.  Also the
message 'key' will be broken into its two component fields, the plugin
name and the instance name.  Clicking on the instance name (key2) will
mark all notifications from that instance, and clicking on the plugin
name (key1) will mark all notifications generated by that plugin as
seen.

Scored items show up at the top of the list, with the highest score on
top.

=head1 COMMAND

In the notifications web ui, there is a 'cmd' field which allows you
to enter commands to be applied to one or more items.  The following
commands are supported:

  r         mark a single item as seen
  rr        mark all unseen items in the selected mailbox as seen
  r.*       mark all unseen notifications as seen
  r.regexp  mark all notifications whose subject matches /regexp/ as seen
  ##        entering a number will apply a score to the item, 00 removes score
  red       entering a color name will apply a color to the item, see L<App::Wubot::Util::Colors>
  m         apply 'readme' tag to the item and mark it read, for later perusal
  x         remove the 'readme' tag and update the 'seen' flag time to now
  tr xx     translate the subject from the specified language, see L<Lingua::Translate> for more info
  mytag     any non-command input will be treated as a tag which will be applied to the item
  -mytag    preceeding tag name with a minus sign removes the tag from the item

Multiple commands may be comma separated.

=head1 ICONS

To ensure that the icons show up in the notifications web ui, you
should do two things:

  1. ensure that the 'icon' rule runs before the notifications sqlite
  rule.  otherwise no icon will have been set in the message.

  2. ensure that your icons directory is symlinked into the public/
  directory.  for example, if your icons are in ~/.icons/ and you are
  running wubot from ~/projects/wubot, then set into the
  ~/projects/wubot/public directory and run 'ln -s ~/.icons ./'.  Then
  you should be able to see one of your images in your ~/.icons
  directory by navigating to, e.g.:

    http://localhost:3000/images/wubot.png

By default you can get to the notification web interface at:

  http://localhost:3000/notify/


Lots more to come here!


=cut

my $logger = Log::Log4perl::get_logger( __PACKAGE__ );

my $colors = App::Wubot::Util::Colors->new();

my $predict = App::Wubot::Util::TagPredict->new();

my $is_null = "IS NULL";
my $is_not_null = "IS NOT NULL";

my $timelength = App::Wubot::Util::TimeLength->new();

my $notify_file    = join( "/", $ENV{HOME}, "wubot", "sqlite", "notify.sql" );
my $sqlite_notify  = App::Wubot::SQLite->new( { file => $notify_file } );

my $taskbot      = App::Wubot::Util::Taskbot->new();

=head1 SUBROUTINES/METHODS

=over 8

=item notify

Display the notifications web interface

=cut

sub notify {
    my $self = shift;

    my $now = time;

    my $limit = 100;
    if ( $self->param('limit') ) {
        $limit = $self->param('limit');
    }

    my $expand;
    if ( $self->param('expand') ) {
        $expand = 1;
    }

    my $order = 'score DESC, lastupdate DESC, id DESC';
    if ( $self->param( 'order' ) ) {
        $order = $self->param( 'order' );
    }

    my $seen = \$is_null;

    my $old = $self->param( 'old' );
    if ( $old ) {
        $expand = 1;
        $seen = \$is_not_null;
        $order = 'seen DESC';
    }

    my $where = { seen => $seen };

    my $params = $self->req->params->to_hash;
    for my $param ( sort keys %{ $params } ) {
        next unless $params->{$param};
        next unless $param =~ m|^tag_|;

        $param =~ m|^tag_(\d+)|;
        my $id = $1;

        my $cmd = $params->{$param};

        $self->_cmd( $id, $cmd );

    }

    my $mailbox     = $self->param( "mailbox" );

    my $seen_id      = $self->param( "seen" );
    if ( $seen_id ) {
        my @seen = split /,/, $seen_id;
        $sqlite_notify->update( 'notifications',
                                { seen => $now   },
                                { id   => \@seen },
                            );

        if ( $mailbox ) {
            $self->redirect_to( "/notify?mailbox=$mailbox" );
        }
        else {
            $self->redirect_to( "/notify" );
        }
    }

    my $key      = $self->param( "key" );
    if ( $key ) {
        $expand = 1;
        $where = { key => $key, seen => $seen };
        if ( ! $old && ! $self->param( 'order' ) ) {
            $order = "lastupdate, id";
        }
    }

    my $username      = $self->param( "username" );
    if ( $username ) {
        $expand = 1;
        $where = { username => $username, seen => $seen };
        if ( ! $old && ! $self->param( 'order' ) ) {
            $order = "lastupdate, id";
        }
    }

    my $plugin      = $self->param( "plugin" );
    if ( $plugin ) {
        $expand = 1;
        $where = { key => { LIKE => "$plugin%" }, seen => $seen };
        if ( ! $old && ! $self->param( 'order' ) ) {
            $order = "lastupdate, id";
        }
    }

    if ( $mailbox ) {
        $expand = 1;
        if ( $mailbox eq "null" ) {
            $where = { mailbox => undef, seen => $seen };
        }
        else {
            $where = { mailbox => $mailbox, seen => $seen };
        }
        if ( ! $old && ! $self->param( 'order' ) ) {
            $order = "lastupdate, id";
        }
    }
    $self->stash( 'mailbox', $mailbox );

    my $seen_key      = $self->param( "seen_key" );
    if ( $seen_key ) {
        $sqlite_notify->update( 'notifications',
                                { seen => $now },
                                { key  => $seen_key },
                            );

        $self->redirect_to( "/notify" );
    }


    my $tag = $self->param( "tag" );
    if ( $tag ) {
        $expand = 1;
        my @ids;
        for my $row ( $sqlite_notify->select( { tablename => 'tags',
                                                fieldname => 'remoteid',
                                                where     => { tag => $tag },
                                                order     => $order,
                                            } ) ) {

            push @ids, $row->{remoteid};
        }

        $where = { id => \@ids };
    }

    my @messages;
    my @ids;

  MESSAGE:
    for my $message ( $sqlite_notify->select( { tablename => 'notifications',
                                                where     => $where,
                                                order     => $order,
                                                limit     => $limit,
                                            } ) ) {

        push @ids, $message->{id};

        push @messages, App::Wubot::Web::Obj::NotifyObj->new( { db_hash => $message, sql => $sqlite_notify } );
    }
    $self->stash( 'body_data', \@messages );

    $self->stash( 'headers', [ qw/cmd mailbox key1 key2 seen username icon id subject link score age/ ] );

    $self->stash( 'ids', join( ",", @ids ) );

    my ( $total ) = $sqlite_notify->select( { fields    => 'count(*) as count',
                                          tablename => 'notifications',
                                          where     => { seen => \$is_null },
                                      } );
    $self->stash( 'count', $total->{count} );

    my ( $readme ) = $sqlite_notify->select( { fields    => 'count(*) as count',
                                               tablename => 'tags',
                                               where     => { tag => 'readme' },
                                           } );
    $self->stash( 'readme', $readme->{count} );

    my ( $todo ) = $sqlite_notify->select( { fields    => 'count(*) as count',
                                             tablename => 'tags',
                                             where     => { tag => 'todo' },
                                           } );
    $self->stash( 'todo', $todo->{count} );

    my @mailboxes;
    $sqlite_notify->select( { fields => 'mailbox, lastupdate, count(*) as count',
                              tablename => 'notifications',
                              group => 'mailbox',
                              order => 'lastupdate DESC, count DESC',
                              where => { seen => \$is_null },
                              callback => sub {
                                  my $row = shift;
                                  $row->{color} = $timelength->get_age_color( $now - $row->{lastupdate} );
                                  unless ( $row->{mailbox} ) { $row->{mailbox} = "null" }
                                  push @mailboxes, $row;
                              },
                          } );
    $self->stash( 'mailboxes', \@mailboxes );

    my $showbody = $self->param('showbody') || 0;
    $self->stash( 'showbody', $showbody );

    $self->render( template => 'notify.list' );

};

=item item

Display a single item from the notification queue.

=cut

sub item {
    my $self = shift;

    $self->stash( 'edit' => $self->param('edit') ? 1 : 0 );

    my $id = $self->stash( 'id' );

    my $cmd = $self->param( 'cmd' );
    if ( $cmd ) {
        $self->_cmd( $id, $cmd );
    }

    my ( $item ) = $sqlite_notify->select( { tablename => 'notifications',
                                             where     => { id => $id },
                                      } );

    my $item_obj = App::Wubot::Web::Obj::NotifyObj->new( { id => $id, sql => $sqlite_notify } );
    $self->stash( item => $item_obj );

    my $subject = $self->param( 'subject' );

    if ( $self->param('edit') ) {
        my $update = 0;
        if ( $subject && $subject ne $item->{subject_text} ) {
            $sqlite_notify->update( 'notifications',
                                    { subject_text => $subject },
                                    { id           => $id  },
                                );
            $update = 1;
        }
        my $body = $self->param('body');
        if ( $body && $body ne $item->{body} ) {
            $sqlite_notify->update( 'notifications',
                                    { body => $body },
                                    { id   => $id  },
                                );
            $update = 1;
        }
        if ( $update ) {
            $self->redirect_to( "/notify/id/$id" );
        }
    }

    my @tags;
    $sqlite_notify->select( { tablename => 'tags',
                              fieldname => 'tag',
                              where     => { remoteid => $item->{id} },
                              order     => 'tag',
                              callback  => sub { my $entry = shift;
                                                 push @tags, $entry->{tag};
                                             },
                          } );
    $self->stash( tags => \@tags );

    my $words = $predict->count_words( $item->{subject_text}, $item->{body} );
    my $recs = $predict->predict_tags( $words, { limit => 5 } );
    $self->stash( predict_tags => $recs );

    my $now = time;
    my @mailboxes;
    $sqlite_notify->select( { fields => 'mailbox, lastupdate, count(*) as count',
                              tablename => 'notifications',
                              group => 'mailbox',
                              order => 'lastupdate DESC, count DESC',
                              where => { seen => \$is_null },
                              callback => sub {
                                  my $row = shift;
                                  $row->{color} = $timelength->get_age_color( $now - $row->{lastupdate} );
                                  push @mailboxes, $row;
                              },
                          } );
    $self->stash( 'mailboxes', \@mailboxes );

    $self->render( template => 'notify.item' );
}

sub _cmd {
    my ( $self, $id, $cmd ) = @_;

    $logger->error( "ID:id COMMAND:$cmd" );

    my $now = time;

    for my $tag ( split /\s*,\s*/, $cmd ) {

        if ( $tag eq "r" ) {
            #print "Marking read: $id\n";
            $sqlite_notify->update( 'notifications',
                                    { seen => $now },
                                    { id   => $id  },
                                );

        }
        elsif ( $tag eq "rr" ) {
            my ( $entry ) = $sqlite_notify->select( { tablename => 'notifications',
                                                      fields    => 'mailbox',
                                                      where     => { id => $id },
                                                  } );
            $sqlite_notify->update( 'notifications',
                                    { seen => $now },
                                    { mailbox => $entry->{mailbox}, seen => undef },
                                );

        }
        elsif ( $tag eq "r.*" ) {
            $sqlite_notify->update( 'notifications',
                                    { seen => $now },
                                    {},
                                );
        }
        elsif ( $tag =~ m|^r\.(.*)$| ) {
            $sqlite_notify->update( 'notifications',
                                    { seen => $now },
                                    { subject => { 'LIKE' => "%$1%" } },
                                );
        }
        elsif ( $tag =~ m|^\d+$| ) {
            if ( $tag eq "00" ) { $tag = undef }
            $sqlite_notify->update( 'notifications',
                                    { score => $tag },
                                    { id => $id },
                                );
        }
        elsif ( $colors->get_color( $tag ) ne $tag ) {
            $sqlite_notify->update( 'notifications',
                                    { color => $tag },
                                    { id    => $id  },
                                );
        }
        elsif ( $tag =~ m|^-| ) {
            $tag =~ s|^\-||;
            print "Removing tag $tag on id $id\n";
            $sqlite_notify->delete( 'tags',
                                    { remoteid => $id, tag => $tag, tablename => 'notifications' },
                                );
        }
        elsif ( $tag eq "x" ) {
            print "Removing tag readme from id $id\n";
            $sqlite_notify->delete( 'tags',
                                    { remoteid => $id, tag => 'readme', tablename => 'notifications' },
                                );
            $sqlite_notify->update( 'notifications',
                                    { seen => $now },
                                    { id   => $id  },
                                );
        }
        elsif ( $tag eq "m" ) {
            print "Setting README tag on id $id and marking seen\n";
            $sqlite_notify->insert( 'tags',
                                    { remoteid => $id, tag => 'readme', tablename => 'notifications', lastupdate => time },
                                );
            $sqlite_notify->update( 'notifications',
                                    { seen => $now },
                                    { id   => $id  },
                                );
        }
        elsif ( $tag =~ m|tr (\w+)| ) {
            my $src_lang = $1;
            print "Translating subject from $src_lang\n";
            my ( $entry ) = $sqlite_notify->select( { tablename => 'notifications',
                                                      fields    => 'subject',
                                                      where     => { id => $id },
                                                  } );

            my $xl8r = Lingua::Translate->new(src => $src_lang,
                                              dest => "en" )
                or die "No translation server available";

            my $english = $xl8r->translate($entry->{subject});
            chomp $english;

            if ( $english ) {
                print "TRANSLATED: $english\n";

                $sqlite_notify->update( 'notifications',
                                        { subject_text => $english   },
                                        { id   => $id },
                                    );
            }

        }
        else {
            print "Setting tag $tag on id $id\n";
            $sqlite_notify->insert( 'tags',
                                    { remoteid => $id, tag => $tag, tablename => 'notifications', lastupdate => time },
                                );

        }
    }
}

=item tags

Display the tags web interface.

=cut

sub tags {
    my $self = shift;

    my @tags;

    my $now = time;

    for my $tag ( $sqlite_notify->select( { tablename => 'tags',
                                            fields    => 'count(*) as count, tag, lastupdate',
                                            group     => 'tag',
                                            order     => 'lastupdate DESC, id DESC',
                                        } ) ) {

        my $age = $now - $tag->{lastupdate};

        $tag->{age} = $timelength->get_human_readable( $age );
        $tag->{age_color} = $timelength->get_age_color( $age );

        push @tags, $tag;
    }

    $self->stash( 'tags', \@tags );

    $self->render( template => 'tags' );

};

=item colors

Display the range of the age colors used in the timeline.

=cut

sub colors {
    my $self = shift;

    my @times;

    push @times, map { $_ * 60 } ( 0 .. 59 );

    push @times, map { $_ * 60 * 24 + 60*60 } ( 0 .. 59 );
    push @times, map { $_ * 60 * 24 * 7 + 60*60*24 } ( 0 .. 59 );
    push @times, map { $_ * 60 * 24 * 30 + 60*60*24*7  } ( 0 .. 59 );
    push @times, map { $_ * 60 * 24 * 365 * 2 + 60*60*24*30 } ( 0 .. 59 );

    my @results;

    for my $age ( @times ) {

        my $time = $timelength->get_human_readable( $age );
        my $color = $timelength->get_age_color( $age );

        push @results, { time => $time, color => $color };
    }

    $self->render( template => 'colors', results => \@results );

}

1;

__END__


=back
