/*
 *  Copyright 1999-2001 Vizdom Software, Inc. All Rights Reserved.
 * 
 *  This program is free software; you can redistribute it and/or 
 *  modify it under the same terms as the Perl Kit, namely, under 
 *  the terms of either:
 *
 *      a) the GNU General Public License as published by the Free
 *      Software Foundation; either version 1 of the License, or 
 *      (at your option) any later version, or
 *
 *      b) the "Artistic License" that comes with the Perl Kit.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See either
 *  the GNU General Public License or the Artistic License for more 
 *  details.
 */

package com.vizdom.util;

import java.util.Hashtable;
import com.vizdom.util.UnreachableCodeException;

/**
 * This class provides an interface to character encoding functionality.
 * If Java exposed its character encoding classes and allowed an
 * application to add new encodings, we wouldn't need to do this. Note that
 * this class only works for single-byte character encodings.
 * <p>
 * The great comedy here is that this class is not currently extensible,
 * either. It is hard-coded for the extra character encodings we happen
 * to need.
 *   
 * @author Gennis Emerson
 * @version $Revision: 1.15 $
 */
public abstract class CharacterEncoder
{
    /** Used in the cache when no encoder has yet been instantiated. */
    private static Object gPLACE_HOLDER = new Object();
    /** A cache of instantiated encoders. */
    private static Hashtable gEncoders;

    /** 
     * The name of the ASCII-equivalent character encoding being 
     * used in this JVM.
     */
    private static String gAsciiEncoding = null;
    /** 
     * The name of the Latin-1 character encoding being 
     * used in this JVM.
     */
    private static String gLatin1Encoding = null;

    // All supported non-Java encodings must be listed here.
    static
    {
        gEncoders = new Hashtable();
        gEncoders.put("DEC_MCS", gPLACE_HOLDER);
        gEncoders.put("HP_Roman8", gPLACE_HOLDER);
        gEncoders.put("Europa_3", gPLACE_HOLDER);
    }


    /**
     * Returns the name of the platform's default encoding.
     *
     * @return the name of the default character encoding on 
     *      this platform
     */
    public static String getDefaultEncoding()
    {
        byte[] b = new byte[0];
        java.io.InputStreamReader reader = 
            new java.io.InputStreamReader(new java.io.ByteArrayInputStream(b));
        String encoding = reader.getEncoding();
        try { reader.close(); } catch (java.io.IOException e) {}
        return encoding;
    }

    
    /**
     * Returns ASCII, if the current JVM supports the ASCII encoding.
     * If not, this method will try to get the Latin-1 encoding, on 
     * the theory that ASCII is a subset of Latin-1 and Latin-1 should
     * therefore be somewhat reasonable as a fallback position.
     * This exists because some older JVMs (the 1.0 JVMs in older 
     * web browsers, the Netscape JVM in their web servers) don't
     * support ASCII.
     *
     * @return the name of a character encoding supported on 
     *      this platform
     * @exception UnsupportedEncodingException if neither ASCII
     *      nor the designated fallback encodings are supported
     * @see getLatin1Encoding     
     */
    public static String getAsciiEncoding() 
        throws java.io.UnsupportedEncodingException
    {
        if (gAsciiEncoding != null)
            return gAsciiEncoding;

        try
        {
            "hello world".getBytes("ASCII");
            if (Debug.DEBUG)
            {
                Debug.getLogWriter(Debug.BRIEF).
                    println("Using ASCII character encoding");
            }
            gAsciiEncoding = "ASCII";
            return gAsciiEncoding;
        }
        catch (java.io.UnsupportedEncodingException encEx)
        {
            if (Debug.DEBUG)
            {
                Debug.getLogWriter(Debug.BRIEF).
                    println("ASCII character encoding not supported");
            }
            gAsciiEncoding = getLatin1Encoding();
            return gAsciiEncoding;
        }
    }
    
    /**
     * Returns the name of a Latin-1 encoding supported in the 
     * current JVM. Some older JVMs don't support the name
     * ISO8859_1, but do support 8859_1, so this method
     * tries them both. If neither encoding name is known, 
     * this method throws an exception.
     *
     * @return the name of a Latin-1 encoding supported on this platform
     * @exception UnsupportedEncodingException if no Latin-1 encoding
     *      can be found for this platform
     * @see getAsciiEncoding
     */
    public static String getLatin1Encoding() 
        throws java.io.UnsupportedEncodingException
    {
        if (gLatin1Encoding != null)
            return gLatin1Encoding;

        String[] encodings = { "ISO8859_1", "8859_1" };
        for (int i = 0; i < encodings.length; i++)
        {
            try
            {
                "hello world".getBytes(encodings[i]);
                if (Debug.DEBUG)
                {
                    Debug.getLogWriter(Debug.BRIEF).
                        println("Using " + encodings[i] + 
                        " character encoding");
                }
                gLatin1Encoding = encodings[i];
                return gLatin1Encoding;
            }
            catch (java.io.UnsupportedEncodingException encEx)
            {
                if (Debug.DEBUG)
                {
                    Debug.getLogWriter(Debug.BRIEF).
                        println(encodings[i] + 
                        " character encoding not supported");
                }
                continue;
            }
        }
        if (Debug.DEBUG)
        {
            Debug.getLogWriter(Debug.BRIEF).
                println("Latin-1 character encoding not supported " +
                "under any known name");
        }
        throw new java.io.UnsupportedEncodingException();
    }

    
    /**
     * Converts the given string to a byte array, using the supplied 
     * encoding.
     *
     * @param aString a string to be converted to bytes
     * @param anEncoding an encoding name
     * @return the bytes corresponding to the characters in the string,
     *      using the given encoding
     * @exception java.io.UnsupportedEncodingException if the 
     *      encoding name is unknown or unsupported on the current platform
     */
    public static byte[] toByteArray(String aString, String anEncoding) 
        throws java.io.UnsupportedEncodingException
    {
        CharacterEncoder enc = getEncoder(anEncoding);
        if (enc == null)
            return aString.getBytes(anEncoding);
        else
            return enc.convertAll(aString);
    }

    /**
     * Converts the given byte array to a string, using the supplied 
     * encoding.
     *
     * @param aByteArray a byte array to be converted to a string
     * @param anEncoding an encoding name
     * @return the string corresponding to the bytes,
     *      using the given encoding
     * @exception java.io.UnsupportedEncodingException if the 
     *      encoding name is unknown or unsupported on the current platform
     */
    public static String toString(byte[] aByteArray, String anEncoding)
        throws java.io.UnsupportedEncodingException
    {
        CharacterEncoder enc = getEncoder(anEncoding);
        if (enc == null)
            return new String(aByteArray, anEncoding);
        else
            return enc.convertAll(aByteArray);
    }

    /**
     * Converts part of the given byte array to a string.
     *
     * @param aByteArray a byte array
     * @param anOffset the offset into the byte array at which to begin 
     *      converting bytes
     * @param aLength the number of bytes to convert
     * @param anEncoding an encoding name
     * @return the string corresponding to the bytes,
     *      using the given encoding
     * @exception java.io.UnsupportedEncodingException if the 
     *      encoding name is unknown or unsupported on the current platform
     */
    public static String toString(byte[] aByteArray, int anOffset, int aLength,
        String anEncoding) throws java.io.UnsupportedEncodingException
    {
        CharacterEncoder enc = getEncoder(anEncoding);
        if (enc == null)
            return new String(aByteArray, anOffset, aLength, anEncoding);
        else
            return enc.convert(aByteArray, anOffset, aLength);
    }

    /**
     * Wraps the supplied OutputStream in a Writer using the given character
     * encoding.
     *
     * @param anOut an OutputStream
     * @param anEncoding an encoding name
     * @return a Writer which writes to the given OutputStream
     *      using the given encoding
     * @exception java.io.UnsupportedEncodingException if the 
     *      encoding name is unknown or unsupported on the current platform
     */
    public static java.io.Writer toWriter(java.io.OutputStream anOut, 
        String anEncoding) throws java.io.UnsupportedEncodingException
    {
        CharacterEncoder enc = getEncoder(anEncoding);
        if (enc == null)
            return new java.io.OutputStreamWriter(anOut, anEncoding);
        else
            return new CharacterEncoderWriter(anOut, enc);
    }


    /**
     * Wraps the supplied InputStream in a Reader using the given character
     * encoding.
     *
     * @param anIn an InputStream
     * @param anEncoding an encoding name
     * @return a Reader which reads from the given InputStream
     *      using the given encoding
     * @exception java.io.UnsupportedEncodingException if the 
     *      encoding name is unknown or unsupported on the current platform
     */
    public static java.io.Reader toReader(java.io.InputStream anIn, 
        String anEncoding) throws java.io.UnsupportedEncodingException
    {
        CharacterEncoder enc = getEncoder(anEncoding);
        if (enc == null)
            return new java.io.InputStreamReader(anIn, anEncoding);
        else
            return new CharacterEncoderReader(anIn, enc);
    }


    /**
     * Returns a CharacterEncoder which uses the given encoding, if the
     * encoding name is a custom encoding. Returns null otherwise.
     *
     * @param anEncoding an encoding name
     * @return a CharacterEncoder which uses the given encoding, or
     *      null if the encoding is not a custom encoding (for example,
     *      if the encoding is a standard Java encoding)
     */
    private static synchronized CharacterEncoder getEncoder(String anEncoding)
    {
        Object enc = gEncoders.get(anEncoding);
        // All supported encodings must return either a valid encoder or
        // gPLACE_HOLDER. Java or unknown encodings will return null.
        if (enc == gPLACE_HOLDER)
        {
            if (anEncoding.equals("DEC_MCS"))
            {
                enc = new CharacterEncoderDecMcs();
                gEncoders.put(anEncoding, enc);
            }
            else if (anEncoding.equals("HP_Roman8"))
            {
                enc = new CharacterEncoderHpRoman8();
                gEncoders.put(anEncoding, enc);
            }
            else if (anEncoding.equals("Europa_3"))
            {
                enc = new CharacterEncoderEuropa3();
                gEncoders.put(anEncoding, enc);
            }
            else
                throw new UnreachableCodeException();
        }
        return (CharacterEncoder) enc;
    }


    
    /** The byte to character mapping. */
    protected int[] mByteToCharTable;
    /** The character to byte mapping. */
    protected int[] mCharToByteTable;
    /** 
     * This character is used when there's no mapping for a byte value.
     * This should never happen; the tables should always have all 256
     * values defined.
     */
    protected char mUnicodeUnknownCharacter = '\ufffd';
    /**
     * This byte is sent to the server if a Unicode character is provided
     * for which no server byte is known. This should only happen
     * if a character above 255 is provided; other cases should already be 
     * handled in the mapping tables.
     */
    protected byte mNativeUnknownCharacter;

    // ??? Are the rest of these methods supposed to be private, protected,
    // package, or public? (The class used to be package, so they could have
    // been intended as any of the above.)

    /** 
     * Converts a string to bytes.
     *
     * @param aString a string to be converted
     * @return the bytes corresponding to the characters in the string
     */
    byte[] convertAll(String aString)
    {
        byte[] bytes = new byte[aString.length()];
        
        for (int i = 0; i < aString.length(); i++)
        {
            int index = (int) aString.charAt(i);
            if (index < 0 || index >= mCharToByteTable.length)
                bytes[i] = mNativeUnknownCharacter;
            else
                bytes[i] = (byte) (mCharToByteTable[index] & 0xFF);
        }
        return bytes;
    }

    /**
     * Converts a character array to bytes. This is useful for a Writer.
     *
     * @param aCharacterArray an array of characters to be converted
     *      to bytes
     * @return the bytes corresponding to the given characters
     */
    byte[] convertAll(char[] aCharacterArray)
    {
        return convert(aCharacterArray, 0, aCharacterArray.length);
    }

    /**
     * Converts a character array to bytes. This is useful for a Writer.
     *
     * @param aCharacterArray an array of characters to be converted
     *      to bytes
     * @param anOffset the offset in the character array at which to begin
     *      converting characters
     * @param aLength the number of characters to convert
     * @return the bytes corresponding to the given characters
     * @exception ArrayIndexOutOfBoundsException if the offset or
     *      length are invalid
     */
    byte[] convert(char[] aCharacterArray, int anOffset, int aLength)
    {
        byte[] bytes = new byte[aLength];
        
        for (int i = anOffset; i < anOffset + aLength; i++)
        {
            int index = (int) aCharacterArray[i];
            if (index < 0 || index >= mCharToByteTable.length)
                bytes[i] = mNativeUnknownCharacter;
            else
                bytes[i] = (byte) (mCharToByteTable[index] & 0xFF);
        }
        return bytes;
    }
    
    /**
     * Converts a byte array to a string.
     *
     * @param aByteArray the byte array to be converted
     * @return the String corresponding to the given bytes
     */
    String convertAll(byte[] aByteArray)
    {
        return convert(aByteArray, 0, aByteArray.length);
    }

    /**
     * Converts part of the given byte array to a string.
     *
     * @param aByteArray a byte array
     * @param anOffset the offset into the byte array at which to begin 
     *      converting bytes
     * @param aLength the number of bytes to convert
     * @return the string corresponding to the bytes,
     *      using the given encoding
     */
    String convert(byte[] aByteArray, int anOffset, int aLength)
    {
        StringBuffer buffer = new StringBuffer();
        for (int i = anOffset; i < anOffset + aLength; i++)
            buffer.append((char) mByteToCharTable[aByteArray[i] & 0xFF]);
        return buffer.toString();
    }

    /**
     * Returns the native byte representation of the given character.
     *
     * @param aCharacter a Unicode character
     * @return the native byte representation of the given character,
     *      or a default value if no encoding is known
     */
    byte getNative(char aCharacter)
    {
        int index = (int) aCharacter;
        if (index < 0 || index >= mCharToByteTable.length)
            return mNativeUnknownCharacter;
        else
            return (byte) (mCharToByteTable[index] & 0xFF);
    }

    /**
     * Returns the Unicode character corresponding to the given byte.
     *
     * @param aByte a native byte value
     * @return the Unicode character corresponding to the given byte,
     *      or a default value if no encoding is known
     */
    char getUnicode(byte aByte)
    {
        int index = aByte & 0xFF;
        return (char) mByteToCharTable[index];
    }
}


/**
 * Embodies the DEC_MCS character encoding. The encoding tables
 * were taken from the Resource File for CMLIB (cmlib.rc).
 *
 * @author John Lacey
 */
class CharacterEncoderDecMcs extends CharacterEncoder
{
    /** The server-to-client map. */
    private static final int[] gServerToClient = 
    {0000, 0001, 0002, 0003, 0004, 0005, 0006, 0007,
     0010, 0011, 0012, 0013, 0014, 0015, 0016, 0017,
     0020, 0021, 0022, 0023, 0024, 0025, 0026, 0027,
     0030, 0031, 0032, 0033, 0034, 0035, 0036, 0037,
     0040, 0041, 0042, 0043, 0044, 0045, 0046, 0047,
     0050, 0051, 0052, 0053, 0054, 0055, 0056, 0057,
     0060, 0061, 0062, 0063, 0064, 0065, 0066, 0067,
     0070, 0071, 0072, 0073, 0074, 0075, 0076, 0077,
     0100, 0101, 0102, 0103, 0104, 0105, 0106, 0107,
     0110, 0111, 0112, 0113, 0114, 0115, 0116, 0117,
     0120, 0121, 0122, 0123, 0124, 0125, 0126, 0127,
     0130, 0131, 0132, 0133, 0134, 0135, 0136, 0137,
     0140, 0141, 0142, 0143, 0144, 0145, 0146, 0147,
     0150, 0151, 0152, 0153, 0154, 0155, 0156, 0157,
     0160, 0161, 0162, 0163, 0164, 0165, 0166, 0167,
     0170, 0171, 0172, 0173, 0174, 0175, 0176, 0177,
     0056, 0056, 0056, 0056, 0056, 0056, 0056, 0056,
     0056, 0056, 0056, 0056, 0056, 0056, 0056, 0056,
     0056, 0056, 0056, 0056, 0056, 0056, 0056, 0056,
     0056, 0056, 0056, 0056, 0056, 0056, 0056, 0056,
     0056, 0241, 0242, 0243, 0056, 0245, 0056, 0247,
     0244, 0251, 0252, 0253, 0056, 0056, 0056, 0056,
     0260, 0261, 0262, 0263, 0056, 0265, 0266, 0267,
     0056, 0271, 0272, 0273, 0274, 0275, 0056, 0277,
     0300, 0301, 0302, 0303, 0304, 0305, 0306, 0307,
     0310, 0311, 0312, 0313, 0314, 0315, 0316, 0317,
     0056, 0321, 0322, 0323, 0324, 0325, 0326, 0056,
     0330, 0331, 0332, 0333, 0334, 0056, 0056, 0337,
     0340, 0341, 0342, 0343, 0344, 0345, 0346, 0347,
     0350, 0351, 0352, 0353, 0354, 0355, 0356, 0357,
     0056, 0361, 0362, 0363, 0364, 0365, 0366, 0056,
     0370, 0371, 0372, 0373, 0374, 0377, 0056, 0056
    };

    /** The client-to-server map. */
    private static final int[] gClientToServer = 
    {
     0000, 0001, 0002, 0003, 0004, 0005, 0006, 0007,    // 0
     0010, 0011, 0012, 0013, 0014, 0015, 0016, 0017,
     0020, 0021, 0022, 0023, 0024, 0025, 0026, 0027,    // 1
     0030, 0031, 0032, 0033, 0034, 0035, 0036, 0037,
     0040, 0041, 0042, 0043, 0044, 0045, 0046, 0047,    // 2
     0050, 0051, 0052, 0053, 0054, 0055, 0056, 0057,
     0060, 0061, 0062, 0063, 0064, 0065, 0066, 0067,    // 3
     0070, 0071, 0072, 0073, 0074, 0075, 0076, 0077,
     0100, 0101, 0102, 0103, 0104, 0105, 0106, 0107,    // 4
     0110, 0111, 0112, 0113, 0114, 0115, 0116, 0117,
     0120, 0121, 0122, 0123, 0124, 0125, 0126, 0127,    // 5
     0130, 0131, 0132, 0133, 0134, 0135, 0136, 0137,
     0140, 0141, 0142, 0143, 0144, 0145, 0146, 0147,    // 6
     0150, 0151, 0152, 0153, 0154, 0155, 0156, 0157,
     0160, 0161, 0162, 0163, 0164, 0165, 0166, 0167,    // 7
     0170, 0171, 0172, 0173, 0174, 0175, 0176, 0177,
     0056, 0056, 0056, 0056, 0056, 0056, 0056, 0056,    // 8
     0056, 0056, 0056, 0056, 0056, 0056, 0056, 0056,
     0056, 0056, 0056, 0056, 0056, 0056, 0056, 0056,    // 9
     0056, 0056, 0056, 0056, 0056, 0056, 0056, 0056,
     0056, 0241, 0242, 0243, 0250, 0245, 0056, 0247,    // A
     0056, 0251, 0252, 0253, 0056, 0056, 0056, 0056,
     0260, 0261, 0262, 0263, 0056, 0265, 0266, 0267,    // B
     0056, 0271, 0272, 0273, 0274, 0275, 0056, 0277,
     0300, 0301, 0302, 0303, 0304, 0305, 0306, 0307,    // C
     0310, 0311, 0312, 0313, 0314, 0315, 0316, 0317,
     0056, 0321, 0322, 0323, 0324, 0325, 0326, 0056,    // D
     0330, 0331, 0332, 0333, 0334, 0056, 0056, 0337,
     0340, 0341, 0342, 0343, 0344, 0345, 0346, 0347,    // E
     0350, 0351, 0352, 0353, 0354, 0355, 0356, 0357,
     0056, 0361, 0362, 0363, 0364, 0365, 0366, 0056,    // F
     0370, 0371, 0372, 0373, 0374, 0056, 0056, 0375
    };
    
    /** 
     * The character to send to the server in place of unmapped
     * Unicode characters.
     */
    private static final byte gUnknownCharacter = (byte) 0x2e;
    
    /**
     * Constructor. Initializes the mapping tables. 
     */
    CharacterEncoderDecMcs()
    {
        mCharToByteTable = gClientToServer;
        mByteToCharTable = gServerToClient;
        mNativeUnknownCharacter = gUnknownCharacter;
    }
}

/**
 * Embodies the HP-Roman8 character encoding. The encoding tables
 * were taken from the Resource File for CMLIB (cmlib.rc).
 *
 * @author John Lacey
 */
class CharacterEncoderHpRoman8 extends CharacterEncoder
{
    /** The server-to-client map. */
    private static final int[] gServerToClient = 
    {0000, 0001, 0002, 0003, 0004, 0005, 0006, 0007,
     0010, 0011, 0012, 0013, 0014, 0015, 0016, 0017,
     0020, 0021, 0022, 0023, 0024, 0025, 0026, 0027,
     0030, 0031, 0032, 0033, 0034, 0035, 0036, 0037,
     0040, 0041, 0042, 0043, 0044, 0045, 0046, 0047,
     0050, 0051, 0052, 0053, 0054, 0055, 0056, 0057,
     0060, 0061, 0062, 0063, 0064, 0065, 0066, 0067,
     0070, 0071, 0072, 0073, 0074, 0075, 0076, 0077,
     0100, 0101, 0102, 0103, 0104, 0105, 0106, 0107,
     0110, 0111, 0112, 0113, 0114, 0115, 0116, 0117,
     0120, 0121, 0122, 0123, 0124, 0125, 0126, 0127,
     0130, 0131, 0132, 0133, 0134, 0135, 0136, 0137,
     0140, 0141, 0142, 0143, 0144, 0145, 0146, 0147,
     0150, 0151, 0152, 0153, 0154, 0155, 0156, 0157,
     0160, 0161, 0162, 0163, 0164, 0165, 0166, 0167,
     0170, 0171, 0172, 0173, 0174, 0175, 0176, 0177,
     0056, 0056, 0056, 0056, 0056, 0056, 0056, 0056,
     0056, 0056, 0056, 0056, 0056, 0056, 0056, 0056,
     0056, 0056, 0056, 0056, 0056, 0056, 0056, 0056,
     0056, 0056, 0056, 0056, 0056, 0056, 0056, 0056,
     0240, 0300, 0302, 0310, 0312, 0313, 0316, 0317,
     0264, 0056, 0056, 0250, 0056, 0331, 0333, 0056,
     0257, 0335, 0375, 0260, 0307, 0347, 0321, 0361,
     0241, 0277, 0244, 0243, 0245, 0247, 0056, 0242,
     0342, 0352, 0364, 0373, 0341, 0351, 0363, 0372,
     0340, 0350, 0362, 0371, 0344, 0353, 0366, 0374,
     0305, 0356, 0330, 0306, 0345, 0355, 0370, 0346,
     0304, 0354, 0326, 0334, 0311, 0357, 0337, 0324,
     0301, 0303, 0343, 0320, 0360, 0315, 0314, 0323,
     0322, 0325, 0365, 0056, 0056, 0332, 0056, 0377,
     0336, 0376, 0267, 0265, 0266, 0276, 0255, 0274,
     0275, 0252, 0272, 0253, 0056, 0273, 0261, 0056
    };
    
    
    /** The client-to-server map. */
    private static final int[] gClientToServer = 
    {
     0000, 0001, 0002, 0003, 0004, 0005, 0006, 0007,
     0010, 0011, 0012, 0013, 0014, 0015, 0016, 0017, 
     0020, 0021, 0022, 0023, 0024, 0025, 0026, 0027,
     0030, 0031, 0032, 0033, 0034, 0035, 0036, 0037,
     0040, 0041, 0042, 0043, 0044, 0045, 0046, 0047,
     0050, 0051, 0052, 0053, 0054, 0055, 0056, 0057,
     0060, 0061, 0062, 0063, 0064, 0065, 0066, 0067,
     0070, 0071, 0072, 0073, 0074, 0075, 0076, 0077,
     0100, 0101, 0102, 0103, 0104, 0105, 0106, 0107,
     0110, 0111, 0112, 0113, 0114, 0115, 0116, 0117,
     0120, 0121, 0122, 0123, 0124, 0125, 0126, 0127,
     0130, 0131, 0132, 0133, 0134, 0135, 0136, 0137,
     0140, 0141, 0142, 0143, 0144, 0145, 0146, 0147,
     0150, 0151, 0152, 0153, 0154, 0155, 0156, 0157,
     0160, 0161, 0162, 0163, 0164, 0165, 0166, 0167,
     0170, 0171, 0172, 0173, 0174, 0175, 0176, 0177,
     0056, 0056, 0056, 0056, 0056, 0056, 0056, 0056,
     0056, 0056, 0056, 0056, 0056, 0056, 0056, 0056,
     0056, 0056, 0056, 0056, 0056, 0056, 0056, 0056,
     0056, 0056, 0056, 0056, 0056, 0056, 0056, 0056,
     0240, 0270, 0277, 0273, 0272, 0274, 0056, 0275,
     0253, 0056, 0371, 0373, 0056, 0366, 0056, 0260,
     0263, 0376, 0056, 0056, 0250, 0363, 0364, 0362,
     0056, 0056, 0372, 0375, 0367, 0370, 0365, 0271,
     0241, 0340, 0242, 0341, 0330, 0320, 0323, 0264,
     0243, 0334, 0244, 0245, 0346, 0345, 0246, 0247,
     0343, 0266, 0350, 0347, 0337, 0351, 0332, 0056,
     0322, 0255, 0355, 0256, 0333, 0261, 0360, 0336,
     0310, 0304, 0300, 0342, 0314, 0324, 0327, 0265,
     0311, 0305, 0301, 0315, 0331, 0325, 0321, 0335,
     0344, 0267, 0312, 0306, 0302, 0352, 0316, 0056,
     0326, 0313, 0307, 0303, 0317, 0262, 0361, 0357
    };

    /** 
     * The character to send to the server in place of unmapped
     * Unicode characters.
     */
    private static final byte gUnknownCharacter = (byte) 0x2e;

    /**
     * Constructor. Initializes the mapping tables. 
     */
    CharacterEncoderHpRoman8()
    {
        mCharToByteTable = gClientToServer;
        mByteToCharTable = gServerToClient;
        mNativeUnknownCharacter = gUnknownCharacter;
    }
}

/**
 * Embodies the Europa-3 character encoding. The encoding tables
 * were taken from the Resource File for CMLIB (cmlib.rc).
 *
 * @author John Lacey
 */
class CharacterEncoderEuropa3 extends CharacterEncoder
{
    /** The server-to-client map. */
    private static final int[] gServerToClient = 
    {0000, 0001, 0002, 0003, 0004, 0005, 0006, 0007,
     0010, 0011, 0012, 0013, 0014, 0015, 0016, 0017,
     0020, 0021, 0022, 0023, 0024, 0025, 0026, 0027,
     0030, 0031, 0032, 0033, 0034, 0035, 0036, 0037,
     0040, 0041, 0042, 0043, 0044, 0045, 0046, 0047,
     0050, 0051, 0052, 0053, 0054, 0055, 0056, 0057,
     0060, 0061, 0062, 0063, 0064, 0065, 0066, 0067,
     0070, 0071, 0072, 0073, 0074, 0075, 0076, 0077,
     0100, 0101, 0102, 0103, 0104, 0105, 0106, 0107,
     0110, 0111, 0112, 0113, 0114, 0115, 0116, 0117,
     0120, 0121, 0122, 0123, 0124, 0125, 0126, 0127,
     0130, 0131, 0132, 0133, 0134, 0135, 0136, 0137,
     0140, 0141, 0142, 0143, 0144, 0145, 0146, 0147,
     0150, 0151, 0152, 0153, 0154, 0155, 0156, 0157,
     0160, 0161, 0162, 0163, 0164, 0165, 0166, 0167,
     0170, 0171, 0172, 0173, 0174, 0175, 0176, 0056,
     0307, 0374, 0351, 0342, 0344, 0340, 0345, 0347,
     0352, 0353, 0350, 0357, 0356, 0354, 0304, 0305,
     0311, 0346, 0306, 0364, 0366, 0362, 0373, 0371,
     0056, 0326, 0334, 0370, 0056, 0330, 0056, 0056,
     0341, 0355, 0363, 0372, 0361, 0321, 0056, 0056,
     0056, 0056, 0056, 0056, 0056, 0056, 0056, 0056,
     0056, 0056, 0056, 0056, 0256, 0301, 0302, 0300,
     0056, 0056, 0056, 0056, 0056, 0056, 0317, 0056,
     0056, 0272, 0241, 0277, 0056, 0056, 0343, 0307,
     0056, 0056, 0056, 0056, 0056, 0056, 0056, 0056,
     0056, 0056, 0312, 0056, 0056, 0265, 0315, 0056,
     0056, 0056, 0056, 0056, 0056, 0056, 0056, 0056,
     0323, 0337, 0324, 0056, 0365, 0325, 0056, 0056,
     0056, 0332, 0056, 0056, 0056, 0056, 0056, 0056,
     0056, 0056, 0056, 0056, 0056, 0056, 0056, 0056,
     0056, 0056, 0267, 0056, 0056, 0056, 0056, 0056
    };

    /** The client-to-server map. */
    private static final int[] gClientToServer = 
    {
     0000, 0001, 0002, 0003, 0004, 0005, 0006, 0007,
     0010, 0011, 0012, 0013, 0014, 0015, 0016, 0017,
     0020, 0021, 0022, 0023, 0024, 0025, 0026, 0027,
     0030, 0031, 0032, 0033, 0034, 0035, 0036, 0037,
     0040, 0041, 0042, 0043, 0044, 0045, 0046, 0047,
     0050, 0051, 0052, 0053, 0054, 0055, 0056, 0057,
     0060, 0061, 0062, 0063, 0064, 0065, 0066, 0067,
     0070, 0071, 0072, 0073, 0074, 0075, 0076, 0077,
     0100, 0101, 0102, 0103, 0104, 0105, 0106, 0107,
     0110, 0111, 0112, 0113, 0114, 0115, 0116, 0117,
     0120, 0121, 0122, 0123, 0124, 0125, 0126, 0127,
     0130, 0131, 0132, 0133, 0134, 0135, 0136, 0137,
     0140, 0141, 0142, 0143, 0144, 0145, 0146, 0147,
     0150, 0151, 0152, 0153, 0154, 0155, 0156, 0157,
     0160, 0161, 0162, 0163, 0164, 0165, 0166, 0167,
     0170, 0171, 0172, 0173, 0174, 0175, 0176, 0056,
     0056, 0056, 0056, 0056, 0056, 0056, 0056, 0056,
     0056, 0056, 0056, 0056, 0056, 0056, 0056, 0056,
     0056, 0056, 0056, 0056, 0056, 0056, 0056, 0056,
     0056, 0056, 0056, 0056, 0056, 0056, 0056, 0056,
     0056, 0302, 0056, 0026, 0056, 0056, 0056, 0025,
     0056, 0056, 0264, 0056, 0056, 0056, 0056, 0056,
     0007, 0056, 0056, 0056, 0056, 0325, 0056, 0372,
     0056, 0056, 0301, 0056, 0056, 0056, 0056, 0303,
     0267, 0265, 0266, 0307, 0216, 0217, 0222, 0200,
     0056, 0220, 0322, 0034, 0056, 0326, 0056, 0276,
     0056, 0245, 0056, 0340, 0342, 0345, 0231, 0056,
     0235, 0056, 0351, 0056, 0232, 0056, 0056, 0341,
     0205, 0240, 0203, 0306, 0204, 0206, 0221, 0207,
     0212, 0202, 0210, 0211, 0215, 0241, 0214, 0213,
     0056, 0244, 0225, 0242, 0223, 0344, 0224, 0056,
     0233, 0227, 0243, 0226, 0201, 0056, 0056, 0056
    };

    /** 
     * The character to send to the server in place of unmapped
     * Unicode characters.
     */
    private static final byte gUnknownCharacter = (byte) 0x2e;

    /**
     * Constructor. Initializes the mapping tables. 
     */
    CharacterEncoderEuropa3()
    {
        mCharToByteTable = gClientToServer;
        mByteToCharTable = gServerToClient;
        mNativeUnknownCharacter = gUnknownCharacter;
    }
}

/**
 * This class is similar to java.io.OutputStreamWriter, except that
 * it uses a custom character mapping.
 *
 * @author John Lacey
 */
class CharacterEncoderWriter extends java.io.Writer
{
    /** This stream's character encoder. */
    private CharacterEncoder mEncoder;
    /** The underlying output stream. */
    private java.io.OutputStream mOutputStream;

    /**
     * Constructor.
     *
     * @param anOutputStream the output stream to which this Writer
     *      will write
     * @param anEncoder a character encoder to use
     */
    CharacterEncoderWriter(java.io.OutputStream anOutputStream, 
        CharacterEncoder anEncoder)
    {
        mEncoder = anEncoder;
        mOutputStream = anOutputStream;
    }
    
    /**
     * Throws an exception if this object has been closed.
     *
     * @exception java.io.IOException if this object is closed
     */
    private void mEnsureValid() throws java.io.IOException
    {
        if (mOutputStream == null || mEncoder == null)
            throw new java.io.IOException();
    }


    /**
     * Writes the character data to the underlying stream, converting 
     * the characters to bytes using this object's character encoder.
     *
     * @param aCharacterBuffer the data to be written
     * @param anOffset the offset into the buffer from which to begin
     *      reading the data to write
     * @param aLength the number of characters to write
     * @exception java.io.IOException if an error occurs writing to 
     *      the underlying stream
     */
    public void write(char aCharacterBuffer[], int anOffset, int aLength) 
        throws java.io.IOException
    {
        mEnsureValid();

        // As per the InputStream documentation.
        if (aCharacterBuffer == null)
            throw new NullPointerException();
        else if (anOffset < 0 || aLength < 0 || 
            anOffset + aLength > aCharacterBuffer.length)
        {
            throw new IndexOutOfBoundsException();
        }
        else if (aLength == 0)
            return;

        byte[] b = mEncoder.convert(aCharacterBuffer, anOffset, aLength);
        mOutputStream.write(b);
    }

    /**
     * Does nothing, but must be implemented when extending Writer.
     *
     * @exception java.IO.IOException never
     */
    public void flush() throws java.io.IOException
    {
    }

    /**
     * Closes the underlying stream.
     *
     * @exception java.io.IOException if an error occurs closing
     *      the underlying stream
     */
    public void close() throws java.io.IOException
    {
        mEnsureValid();
        mOutputStream.close();
        mOutputStream = null;
        mEncoder = null;
    }
}


/**
 * Reads data from an InputStream and converts the bytes to characters
 * using the supplied character encoder.
 *
 * @author John Lacey
 */
/* ??? Could improve performace here with buffering, I expect. */
class CharacterEncoderReader extends java.io.Reader
{
    /** The character encoder to use. */
    private CharacterEncoder mEncoder;
    /** The underlying input stream. */
    private java.io.InputStream mInputStream;

    /**
     * Constructor.
     *
     * @param anInputStream the input stream from which this object
     *      will read
     * @param anEncoder the character encoder which this object
     *      will use to convert the data
     */
    CharacterEncoderReader(java.io.InputStream anInputStream, 
        CharacterEncoder anEncoder)
    {
        mEncoder = anEncoder;
        mInputStream = anInputStream;
    }

    /**
     * Throws an exception if this object has been closed.
     *
     * @exception java.io.IOException if this object is closed
     */
    private void mEnsureValid() throws java.io.IOException
    {
        if (mInputStream == null || mEncoder == null)
            throw new java.io.IOException();
    }

    /**
     * Reads data into the given buffer.
     *
     * @param aCharacterBuffer the buffer into which to read data
     * @param anOffset the offset into the buffer at which to begin 
     *      storing data
     * @param aLength the maximum number of characters to read into
     *      the buffer
     * @return the number of characters read, or -1 if the end of the 
     *      stream was reached
     * @exception java.io.IOException if an error occurs reading from
     *      the underlying stream
     */
    public int read(char aCharacterBuffer[], int anOffset, int aLength) 
        throws java.io.IOException
    {
        mEnsureValid();

        // As per the InputStream documentation.
        if (aCharacterBuffer == null)
            throw new NullPointerException();
        else if (anOffset < 0 || aLength < 0 || 
            anOffset + aLength > aCharacterBuffer.length)
        {
            throw new IndexOutOfBoundsException();
        }
        else if (aLength == 0)
            return 0;

        byte[] bytes = new byte[aLength];
        int count = mInputStream.read(bytes);
        if (count == -1)
            return count;
        
        for (int i = 0; i < bytes.length; i++)
        {
            aCharacterBuffer[anOffset + i] = mEncoder.getUnicode(bytes[i]);
        }

        return count;
    }

    /**
     * Closes the underlying stream.
     *
     * @exception java.io.IOException if an error occurs closing
     *      the underlying stream
     */
    public void close() throws java.io.IOException
    {
        mEnsureValid();
        mInputStream.close();
        mInputStream = null;
        mEncoder = null;
    }
}

