#   ---------------------------------------------------------------------- copyright and license ---
#
#   file: lib/Version/Dotted/Odd.pm
#
#   Copyright © 2016 Van de Bugger.
#
#   This file is part of perl-Version-Dotted.
#
#   perl-Version-Dotted is free software: you can redistribute it and/or modify it under the terms
#   of the GNU General Public License as published by the Free Software Foundation, either version
#   3 of the License, or (at your option) any later version.
#
#   perl-Version-Dotted is distributed in the hope that it will be useful, but WITHOUT ANY
#   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#   PURPOSE. See the GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License along with
#   perl-Version-Dotted. If not, see <http://www.gnu.org/licenses/>.
#
#   ---------------------------------------------------------------------- copyright and license ---

#pod =for test_synopsis my $v;
#pod
#pod =head1 SYNOPSIS
#pod
#pod     use Version::Dotted::Odd;           # import nothing
#pod     use Version::Dotted::Odd qw{ qv };  # import qv
#pod
#pod     # Construct:
#pod     $v = Version::Dotted::Odd->new( v1.0 );     # v1.0.0 (at least 3 parts)
#pod     $v = qv( v1.0.2.5 );    # v1.0.2.5
#pod
#pod     # Release status:
#pod     if ( $v->is_trial ) {   # If the second part is odd.
#pod         ...
#pod     };
#pod
#pod     # Other methods are inherited from Version::Dotted.
#pod
#pod =head1 DESCRIPTION
#pod
#pod This is subclass of C<Version::Dotted>. Two features distinct it from the parent:
#pod
#pod =over
#pod
#pod =item *
#pod
#pod Version object always has at least 3 parts.
#pod
#pod     $v = qv( v1 );          # v1.0.0
#pod     $v->part( 0 ) == 1;     # Parts 0, 1, 2 are always defined.
#pod     $v->part( 1 ) == 0;     # Zero if not specified explicitly.
#pod     $v->part( 2 ) == 0;     # ditto
#pod     $v->part( 3 ) == undef; # But may be defined.
#pod
#pod =item *
#pod
#pod The second part defines the release status: odd numbers denotes a trial release.
#pod
#pod     $v = qv( v1.0 );        # $v == v1.0.0
#pod     $v->is_trial;           # false
#pod     $v += v0.1;             # $v == v1.1.0
#pod     $v->is_trial;           # true
#pod
#pod =back
#pod
#pod Such versioning scheme was used by Linux kernel (between 1.0 and 2.6) and still used by Perl.
#pod
#pod =cut

package Version::Dotted::Odd;

use strict;
use warnings;

# ABSTRACT: TODO
our $VERSION = 'v0.0.0_02'; # TRIAL VERSION

use parent 'Version::Dotted';

sub _min_len { 3 };     ## no critic ( ProhibitUnusedPrivateSubroutines, RequireFinalReturn )

# --------------------------------------------------------------------------------------------------

#pod =method is_trial
#pod
#pod Returns true if the second version part is an odd number, and false otherwise.
#pod
#pod =cut

sub is_trial {
    my ( $self ) = @_;
    my $v = $self->{ version };
    return $v->[ 1 ] % 2 != 0;
};

# --------------------------------------------------------------------------------------------------

1;

# --------------------------------------------------------------------------------------------------

#pod =head1 SEE ALSO
#pod
#pod =for :list
#pod = L<Version::Dotted>
#pod
#pod =head1 COPYRIGHT AND LICENSE
#pod
#pod Copyright (C) 2016 Van de Bugger
#pod
#pod License GPLv3+: The GNU General Public License version 3 or later
#pod <http://www.gnu.org/licenses/gpl-3.0.txt>.
#pod
#pod This is free software: you are free to change and redistribute it. There is
#pod NO WARRANTY, to the extent permitted by law.
#pod
#pod
#pod =cut

#   ------------------------------------------------------------------------------------------------
#
#   file: doc/what.pod
#
#   This file is part of perl-Version-Dotted.
#
#   ------------------------------------------------------------------------------------------------

#pod =encoding UTF-8
#pod
#pod =head1 WHAT?
#pod
#pod TODO
#pod
#pod =cut

# end of file #


# end of file #

__END__

=pod

=encoding UTF-8

=head1 NAME

Version::Dotted::Odd - TODO

=head1 VERSION

Version v0.0.0_02, released on 2016-12-25 22:38 UTC.
This is a B<trial release>.

=head1 WHAT?

TODO

=for test_synopsis my $v;

=head1 SYNOPSIS

    use Version::Dotted::Odd;           # import nothing
    use Version::Dotted::Odd qw{ qv };  # import qv

    # Construct:
    $v = Version::Dotted::Odd->new( v1.0 );     # v1.0.0 (at least 3 parts)
    $v = qv( v1.0.2.5 );    # v1.0.2.5

    # Release status:
    if ( $v->is_trial ) {   # If the second part is odd.
        ...
    };

    # Other methods are inherited from Version::Dotted.

=head1 DESCRIPTION

This is subclass of C<Version::Dotted>. Two features distinct it from the parent:

=over

=item *

Version object always has at least 3 parts.

    $v = qv( v1 );          # v1.0.0
    $v->part( 0 ) == 1;     # Parts 0, 1, 2 are always defined.
    $v->part( 1 ) == 0;     # Zero if not specified explicitly.
    $v->part( 2 ) == 0;     # ditto
    $v->part( 3 ) == undef; # But may be defined.

=item *

The second part defines the release status: odd numbers denotes a trial release.

    $v = qv( v1.0 );        # $v == v1.0.0
    $v->is_trial;           # false
    $v += v0.1;             # $v == v1.1.0
    $v->is_trial;           # true

=back

Such versioning scheme was used by Linux kernel (between 1.0 and 2.6) and still used by Perl.

=head1 OBJECT METHODS

=head2 is_trial

Returns true if the second version part is an odd number, and false otherwise.

=head1 SEE ALSO

=over 4

=item L<Version::Dotted>

=back

=head1 AUTHOR

Van de Bugger <van.de.bugger@gmail.com>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2016 Van de Bugger

License GPLv3+: The GNU General Public License version 3 or later
<http://www.gnu.org/licenses/gpl-3.0.txt>.

This is free software: you are free to change and redistribute it. There is
NO WARRANTY, to the extent permitted by law.

=cut
