# Makefile.PL #

use 5.014;   # /r search modifier appears in v5.14.
use strict;
use warnings FATAL => 'all';
use autodie ':all';

use FindBin;
use Getopt::Long;
use Inline::MakeMaker;

# --------------------------------------------------------------------------------------------------
# Attributes.
# --------------------------------------------------------------------------------------------------

#   Version info.
my %V = (
    name        => 'Systemd-Daemon',
    version     => '0.04',
    release     => '1',
    status      => 'testing',
    abstract    => 'Write systemd-aware daemons in Perl',
    license     => 'gpl_3',
    author      => 'Van de Bugger <van.de.bugger@liamg.moc>',
    requires    => {
        configure => {
            'autodie'               => 0,
            'CPAN::Meta'            => 0,
            'ExtUtils::MakeMaker'   => 6.52,
            'FindBin'               => 0,
            'Getopt::Long'          => 0,
            'Inline::MakeMaker'     => 0.45,
            'IPC::System::Simple'   => 0,       # Required by autodie ':all'.
            'Path::Class'           => 0,
            'strict'                => 0,
            'warnings'              => 0,
        },
        build => {
            'File::pushd'           => 1.009,
            'Inline'                => 0,
            'Inline::C'             => 0,
            'Inline::Filters'       => 0,       # POD is used in __DATA__.
        },
        run => {
            'Exporter'              => 5.57,    # use Exporter qw{ import };
            'Inline'                => 0,
            'Inline::C'             => 0,
            'Inline::Filters'       => 0,       # POD is used in __DATA__.
            'strict'                => 0,
            'warnings'              => 0,
        },
        test => {
            'autodie'               => 0,
            'File::Basename'        => 0,
            'File::Temp'            => 0,
            'FindBin'               => 0,
            'Getopt::Long'          => 0,
            'IPC::System::Simple'   => 0,       # Required by autodie ':all'.
            'Test::More'            => 0,
        },
    },
    hg_files => "--- Mercurial files\n^\\.hg/\n^\\.hgignore\$",
);

#   Convert requirements to a string form which can be utilized in rpm spec, for example:
#       "perl(autodie), perl(Inline::MakeMaker) >= 0.45, perl(warnings)"
foreach my $stage ( qw{ configure build run test } ) {
    my $reqs = $V{ requires }->{ $stage };
    $V{ $stage . '_requires' } =
        join(
            ', ',
            map(
                "perl($_)" . ( $reqs->{ $_ } eq '0' ? '' : " >= $reqs->{ $_ }" ),
                keys( %$reqs )
            )
        );
};

my $devel;      # True if we run in development mode.

# List of generated files.
my %generated = (
    # File                                         # Generated from
    "MANIFEST.SKIP"                             => ".hgignore",
    "lib/@{[ $V{ name } =~ s{-}{/}gr ]}.pm"     => '',      # Default source.
    "rpm/perl-$V{ name }.spec"                  => '',
);
#   Default source is a destination file with appended suffix ".in".
map( $generated{ $_ } = ( $generated{ $_ } or "$_.in" ), keys( %generated ) );

#   WriteMakefile parameters.
my %M = (
    NAME                => ( $V{ name } =~ s{-}{::}gr ),
    VERSION             => $V{ version },
    LICENSE             => $V{ license },
    ABSTRACT            => $V{ abstract },
    AUTHOR              => $V{ author },
    META_MERGE          => {
        release_status      => $V{ status },
        keywords            => [ qw{ systemd daemon } ],
        no_index            => {
            file                => [
                'Makefile.PL',          # metacpan.org wrongly treats it as documentation.
                values( %generated ),   # cpan treats `*.in` files as sources.
            ],
            directory           => [
                'bin',
                'rpm',
            ],
        },
    },
    PL_FILES            => { 'Makefile.PL' => [ keys( %generated ) ] },
        # The same `Makefile.PL` script is used for generating all files: `Makefile` and all others.
    PM => { 'lib/Systemd/Daemon.pm' => '$(INST_LIB)/Systemd/Daemon.pm' },
    CONFIGURE_REQUIRES  => $V{ requires }->{ configure },
    BUILD_REQUIRES      => $V{ requires }->{ build     },
    PREREQ_PM           => $V{ requires }->{ run       },
    TEST_REQUIRES       => $V{ requires }->{ test      },
    dist                => {
        COMPRESS    => 'gzip -f --best',
        SUFFIX      => '.gz',
    },
    clean               => {    # Temporary and intermediate files.
        FILES => [
            '_Inline',
            '_rpmbuild',
            "$V{ name }-$V{ version }",
            "$V{ name }.inl",
            keys( %generated ),
        ],
    },
    realclean           => {    # Output files,
        FILES => [
            "$V{ name }-*.tar.gz",
            "perl-$V{ name }-*.rpm"
        ],
    },
);

# --------------------------------------------------------------------------------------------------
# Helper functions.
# --------------------------------------------------------------------------------------------------

# Read entire file, return its content as a scalar.
sub gulp($) {
    my ( $name ) = @_;
    open( my $handle, '<', $name );
    local $/ = undef;
    my $bulk = <$handle>;
    close( $handle );
    return $bulk;
}; # sub slurp

# Write file.
sub barf($$) {
    my ( $name, $bulk ) = @_;
    open( my $handle, '>', $name );
    print( $handle $bulk );
    close( $handle );
}; # sub barf

#  Substitute `@VAR@` placeholders with appropriate values taken from hash `%V`.
sub subst($$) {
    my ( $name, $text ) = @_;
    $text =~
        s{
            \@ ( [A-Z] [A-Z_]* | ! ) \@
        }{
            if ( $1 eq '!' ) {
                'Generated by Makefile.PL from';
            } else {
                $V{ lc( $1 ) } // die "$name: Unknown placeholder `\@$1\@' used\n";
            };
        }gex;
    return $text;
}; # sub subst

# --------------------------------------------------------------------------------------------------
# Parse arguments and do the work.
# --------------------------------------------------------------------------------------------------

{
    local @ARGV = @ARGV;    # Important!
        # The rest of the script assumes @ARGV contains original command line arguments.
    {
        local $SIG{ __WARN__ } = sub { die "$FindBin::Script: $_[ 0 ]"; };
            #   `GetOptions` reports command-line errors by `warn`. Convert warnings to errors.
        GetOptions(
            'devel' => \$devel,
        ) or die;   # Just in case.
    }
    my @files = grep( $_ !~ m{^[A-Za-z_]+=}, @ARGV );  # Filter out variable assignments.
    if ( @files ) {
        #   If we have a non-option argument, the script is called to generate a file other than
        #   `Makefile`.
        if ( @files > 1 ) {
            die "Too many arguments; only one argument is expected.\n";
        }; # if
        my $dst = shift( @files );
        my $src = $generated{ $dst };
        barf( $dst, subst( $src, gulp( $src ) ) );
        exit( 0 );
    }; # if
}

# TODO: What if rpmbuild is not found?
if ( $devel ) {
    my $rpmbuild = ( $ENV{ RPMBUILD } // 'rpmbuild' );
    $V{ arch } = qx{ $rpmbuild --eval '%{_arch}' } or die "Failed to run rpmbuild";
    $V{ dist } = qx{ $rpmbuild --eval '%{dist}'  } or die "Failed to run rpmbuild";
    chomp( $V{ arch }, $V{ dist } );
};

WriteMakefile( %M );
exit( 0 );

# ==================================================================================================
# Package MY — MakeMaker customizations.
# ==================================================================================================

{ package MY;

use Path::Class;

our $Text;

# --------------------------------------------------------------------------------------------------
# Some helper methods.
# --------------------------------------------------------------------------------------------------

my ( $BoL, $S, $Variable, $Target, $Targets, $Col, $Eqv, $Tail );
BEGIN {
    $BoL      = qr{(?<!\\\n)^}m;            # Beginning of a logical line.
    $S        = qr{[ \t]};                  # One space or tab.
    $Variable = qr{[A-Za-z_0-9]+}; # Variable name.
    $Target   = qr{(?:[A-Za-z_0-9/.-]+|\$\($Variable\))+};
    $Targets  = qr{$Target(?:$S+$Target)*};
    $Col      = qr{::?};                    # Colon which separates target and prereqs in a rule.
    $Eqv      = qr{[:+]?=};                 # Assignment.
    $Tail     = qr{.*?(?:\\\n.*?)*\n};
}

#   Converts argument to a regular expression.
#       regexp( REGEXP )
#       regexp( STRING )
#   If argument is a regular expression, returns it. Otherwise converts argument to a regular
#   expression by quoting all the characters.
#
sub regexp($) {
    return ref( $_[ 0 ] ) eq 'Regexp' ? $_[ 0 ] : qr{^@{[ quotemeta( $_[ 0 ] ) ]}$};
}

#   Converts argument to a string.
#       string( STRING )
#       string( CODEREF )
#   If argument is not reference to code, returns it. Otherwise the code is called in scalar context
#   with no arguments, result is returned.
sub string($) {
    return ref( $_[ 0 ] ) eq 'CODE' ? scalar( $_[ 0 ]->() ) : ( $_[ 0 ] // '' );
};

#   Edit the text with specified edits (search/replace pairs).
#       edit( STRING, [ OLD => NEW, ... ] )
#
sub edit($$) {
    my ( $text, @edits ) = ( $_[ 0 ], @{ $_[ 1 ] // [] } );
    while ( @edits ) {
        my ( $old, $new ) = ( shift( @edits ), shift( @edits ) );
        if ( defined( $old ) or defined( $new ) ) {
            if ( defined( $old ) and $old ne '' ) {
                $old = regexp( $old );
                $text =~ s{$old}{string( $new )}ge;
            } else {
                $text = string( $new );
            }; # if
        }; # if
    }; # while
    return $text;
}; # sub edit

#   Comment out specified text.
#       comment( STRING )
#
sub comment($) {
    my ( $text ) = @_;
    if ( $text =~ m{\A\t} ) {
        $text =~ s{^\t}{\t\@# ××× }gm;
    } else {
        $text =~ s{^}{# ××× }gm;
    }; # if
    return $text;
};

#   Construct a recipe from list of commands.
#       recipe( COMMAND,… )
#
sub recipe(@) {
    return join( '', map( "\t$_\n", grep( $_ ne '', @_ ) ) );
};

#   Assemble a makefile rule from target, prerequisites, and commands.
#       rule( TARGET, PREREQ, COMMAND,… )
#       rule( TARGET, [ PREREQ,… ], COMMAND,… )
sub rule($$@) {
    my ( $target, $prereqs, @commands ) = @_;
    my $text =
        $target . ' : ' .
        ( ref( $prereqs ) eq 'ARRAY' ? join( ' ', @$prereqs ) : ( $prereqs // '' ) )  . "\n" .
        recipe( ( @commands ? '$(PROLOGUE)' : () ), @commands );
    return $text;
}; # sub rule

#   Create variable definition/assignment.
#       mk_var( NAME )
#       mk_var( NAME, VALUE )
#       mk_var( NAME, OP, VALUE )
#
sub mk_var($;$$) {
    my ( $name, $op, $value ) = ( shift( @_ ), @_ > 1 ? shift( @_ ) : '=', shift( @_ ) // '' );
    $Text .=
        "# +++\n" .
        "$name $op $value\n";
};

#   Edit variable assignment.
#       ch_var( NAME, name => [ OLD => NEW,… ], op => [ OLD => NEW,… ], value => [ OLD => NEW,… ] )
#   NAME may be a string (searched literally) or regular expression.
#
sub ch_var($@) {
    my ( $name, %edits ) = @_;
    $name = regexp( $name );
    $Text =~ s{
        #   All the assignments starts on the first column, and variable name is always literal.
        $BoL ( ( $Variable ) ( $S* ) ( $Eqv ) ( $S* ) ( $Tail ) )
    }{
        my ( $old_stmt, $old_name, $s0, $old_op, $s1, $old_value ) = ( $1, $2, $3, $4, $5, $6 );
        chomp( $old_stmt, $old_value );
        if ( $old_name =~ $name ) {
            my $new_name  = edit( $old_name,  $edits{ name  } );
            my $new_op    = edit( $old_op,    $edits{ op    } );
            my $new_value = edit( $old_value, $edits{ value } );
            my $new_stmt  = "$new_name$s0$new_op$s1$new_value";
            comment( "$old_stmt\n" ) .
            "# +++\n" .
            "$new_stmt\n";
        } else {
            "$old_stmt\n";
        };
    }gxe;
};

#   Remove existing variable definition.
#       rm_var( NAME )
#   NAME could be a string (searched literally) or regular expression.
#
sub rm_var($) {
    my ( $name ) = @_;
    $name = regexp( $name );
    $Text =~ s{
        $BoL ( ( $Variable ) ( $S* ) ( $Eqv ) ( $S* ) ( $Tail ) )
    }{
        my ( $old_stmt, $old_name, $s0, $old_op, $s1, $old_value ) = ( $1, $2, $3, $4, $5, $6 );
        chomp( $old_stmt, $old_value );
        ( $old_name =~ $name ? comment( $old_stmt ) : $old_stmt ) . "\n";
    }mgxe;
};

#   Add new rule.
#       mk_rule( TARGET, PREREQ, COMMANDS… )
#       mk_rule( TARGET, [ PREREQ… ], COMMANDS… )
#
sub mk_rule($@) {
    $Text .=
        "# +++\n" .
        &rule( @_ );
}; # sub mk_rule

#   Delete entire rule for the specified target.
#       rm_rule( TARGET )
#
sub rm_rule($) {
    my ( $target ) = @_;
    $target = regexp( $target );
    $Text =~ s{
        $BoL  ( ( $Targets ) ( $S* ) ( $Col ) ( $S* ) ( $Tail ) ( (?: \t $Tail )* ) )
    }{
        my ( $old_rule, $old_target, $s0, $old_col, $s1, $old_prereqs, $old_recipe ) = ( $1, $2, $3, $4, $5, $6, $7 );
        my @old_commands = ( $old_recipe =~ m{^\t($Tail)}gm );
        chomp( $old_prereqs, @old_commands );
        if ( $old_target =~ $target ) {
            comment( $old_rule );
        } else {
            $old_rule;
        }; # if
    }mgxe;
}; # sub rm_rule

#   Edit rule.
#       ch_rule( TARGET, target => [ OLD => NEW,… ], col => […], prereqs => […], commands => […], recipe => […] )
#
sub ch_rule($@) {
    my ( $target, %edits ) = @_;
    $target = regexp( $target );
    $Text =~ s{
        $BoL  ( ( $Targets ) ( $S* ) ( $Col ) ( $S* ) ( $Tail ) ( (?: \t $Tail )* ) )
    }{
        my ( $old_rule, $old_target, $s0, $old_col, $s1, $old_prereqs, $old_recipe ) = ( $1, $2, $3, $4, $5, $6, $7 );
        my @old_commands = ( $old_recipe =~ m{^\t($Tail)}gm );
        chomp( $old_prereqs, @old_commands );
        if ( $old_target =~ $target ) {
            if ( @old_commands and $old_commands[ 0 ] eq '$(PROLOGUE)' ) {
                shift( @old_commands );
            }; # if
            my $new_target   = edit( $old_target, $edits{ target } );
            my $new_col      = edit( $old_col, $edits{ col } );
            my $new_prereqs  = edit( $old_prereqs, $edits{ prereqs } );
            my @new_commands = map( edit( $_, $edits{ commands } ), @old_commands );
            my $new_recipe   = edit( recipe( @new_commands ), $edits{ recipe } );
            if ( $new_recipe ne '' and $new_recipe ne "\t\$(NOOP)\n" ) {
                $new_recipe = "\t\$(PROLOGUE)\n$new_recipe";
            };
            my $new_rule     = "$new_target$s0$new_col$s1$new_prereqs\n$new_recipe";
            ( $old_rule ne $new_rule ? comment( $old_rule ) . "# +++\n" : '' ) .
            "$new_rule";
        } else {
            "$old_rule";
        }; # if
    }mgxe;
}; # sub ch_rule

sub reveal {
    $Text = "# *** Section revealed.\n\n" . $Text;
    ch_rule(
        qr{.+},
        commands => [
            #   Do not ignore command failures.
            qr{^$S*-$S*} => '',
            #   Use macro `$(SAY)` instead of pair `$(NOECHO) $(ECHO)`.
            qr{^$S*\Q$(NOECHO)\E$S+\Q$(ECHO)\E$S+} => '$(SAY) ',
            #   Do not hide commands.
            qr{^$S*\Q$(NOECHO)\E$S+} => '',
        ],
    );
};

# --------------------------------------------------------------------------------------------------
# MakeMaker sections customization.
# --------------------------------------------------------------------------------------------------

#   Customization methods should start with `my_` and/or `My_' prefix. When a method is invoked,
#   variable $Text is preloaded with original section text. The method may change the variable
#   value: append some more text to the end, change it, fully replace, or leave it as is.
#   Methods with `my_` prefix are called unconditionally, methods with `My_` prefix are called
#   only if the script is invoked with `--devel` option. If both methods are defined for the same
#   section (for example, `my_constants` and `My_constants`), `my_` method is callled first.
#
#   Thus, `my_` methods should be used for simple and safe changes, while tricky and potentially
#   unsafe customizations (e. g. customizations which rely on MakeMaker internals) should be done
#   in `My_` methods.

#   post_initialize
#   const_config

sub my_constants {
    mk_var( 'RELEASE', $V{ release } );
};

#   platform_constants
#   tool_autosplit
#   tool_xsubpp

sub my_tools_other {
    mk_var( 'RPMBUILD',  'rpmbuild'          );
    mk_var( 'SAY',       '$(NOECHO) $(ECHO)' );
};

sub My_tools_other {
    #   Do not hide command output.
    ch_var( 'DEV_NULL', value => [ ''          => ''                        ] );
    ch_var( 'NOOP',     value => [ qr{^(.*)$}s => sub { "\$(NOECHO) $1"; }  ] );
    mk_var( 'PROLOGUE', '$(SAY) "\# ----- $@ ----- \#"' );
};

#   makemakerdflt
#   dist
#   macro
#   depend
#   cflags
#   const_loadlibs
#   const_cccmd
#   post_constants
#   pasthru

sub My_special_targets {
    #   Let GNU make delete target file in case of error.
    #   TODO: Add it if make is GNU Make.
    mk_rule( '.DELETE_ON_ERROR' );
};

#   c_o
#   xs_c
#   xs_o

sub my_top_targets {
    #   `help` target generated by MakeMaker shows help on MakeMaker itself. It is very long
    #   and not useful for a user who just wants to build the module.
    ch_rule(
        'help',
        recipe => [
            '' => recipe(
                '$(SAY) "Targets:"',
                '$(SAY) "    Building:"',
                '$(SAY) "        all       -- build everything"',
                '$(SAY) "   Testing:"',
                '$(SAY) "        test      -- run all the tests"',
                '$(SAY) "        distcheck -- check manifest"',
                #~ '$(SAY) "        disttest  -- "',
                '$(SAY) "   Packaging:"',
                '$(SAY) "        dist      -- build tarball package"',
                ( $devel ? (
                    '$(SAY) "        rpm       -- build rpm packages"',
                ) : () ),
                '$(SAY) "   Cleaning:"',
                '$(SAY) "        clean     -- delete temp files"',
                '$(SAY) "        realclean -- clean then delete packages"',
                '$(SAY) "        veryclean -- realclean then delete backup files"',
                '$(SAY) "        distclean -- realclean then distcheck"',
            ),
        ],
    );
};

#   blibdirs
#   linkext
#   dlsyms
#   dynamic_bs
#   dynamic
#   dynamic_lib
#   static
#   static_lib

sub My_manifypods {
};

sub My_processPL {
    foreach my $file ( keys( %generated ) ) {
        ch_rule(
            $file,
            prereqs => [
                qr{\A} => $generated{ $file } . ' ',
            ],
        );
    };
};

#   installbin
#   subdirs
#   clean_subdirs

sub My_clean {
    #   `mv Makefile Makefile.old` fails if `Makefile` does not exist. To avoid failure make sure
    #   `Makefile` exists before moving it.
    ch_rule(
        'clean',
        commands => [
            qr{^(\Q$(MV) $(FIRST_MAKEFILE) $(MAKEFILE_OLD) $(DEV_NULL)\E)$}
                => sub { "if test -e \$(FIRST_MAKEFILE); then $1; fi"; },
        ],
    );
};

#   realclean_subdirs

sub My_realclean {
};

sub My_metafile {
    ch_rule(
        'metafile',
        prereqs => [
            'create_distdir' => '$(DISTVNAME)/META.yml $(DISTVNAME)/META.json',
        ],
        commands => [
            qr{\bMETA_new\.(yml|json)\b} => sub { "META.$1.tmp"; },
        ],
        recipe => [
            qr{^\t\Q$(PROLOGUE)\E\n}
                => '',
            qr{^\t\Q$(SAY)\E$S+Generating$S+META\.(yml|json)$S*\n}m
                => sub { "\$(DISTVNAME)/META.$1 : \$(FIRSTMAKE)\n\t\$(PROLOGUE)\n" },
        ],
    );
};

#   signature

sub My_dist_basics {
    #   My `MANIFEST` is hand-made, `make manifest` will damage it,
    #   so delete rule for `manifest` target.
    rm_rule( 'manifest' );
};

sub My_dist_core {
    # Do not delete directory.
    ch_rule(
        '$(DISTVNAME).tar$(SUFFIX)',
        commands => [
            '$(RM_RF) $(DISTVNAME)' => '',
        ],
    );
};

sub My_distdir {
    #   I do not want to delete and fully recreate distdir every time.
    #   Instead, I want to populate it with changed files and remove any
    #   extra files.
    ch_rule(
        'create_distdir',
        commands => [
            '$(RM_RF) $(DISTVNAME)' => '',
        ],
        recipe => [
            #~ '' => recipe( '$(PERLRUN) -Ibin -MDistDir=copy -e \'copy( ".", "$(DISTVNAME)", "$(DIST_CP)" )\'' )
        ],
    );
    ch_rule(
        'distdir',
        prereqs => [
            qr{\z}    => 'clear_distdir',
        ],
    );
    mk_rule(
        'clear_distdir',
        [ 'create_distdir', 'distmeta' ],
        '$(PERLRUN) -Ibin -MDistDir=clear -e \'clear( "$(DISTVNAME)" )\''
    );
};

#   dist_test

sub my_dist_ci {
    #   Disable "ci" target. I do not use archaic RCS.
    $Text = comment( $Text );
};

sub My_distmeta {
    #   Workaround for bug #103042
    #       https://rt.cpan.org/Ticket/Display.html?id=103042
    ch_rule(
        'distmeta',
        commands => [
            qr{ or print }  => ' or die ',
            qr{\${4}}    => '$$',
        ],
    );
};

#   distsignature

sub My_install {
};

#   force
#   perldepend

sub My_makefile {
    #   GNU make automatically reloads Makefile if it was rebuilt.
    #   No need to print a message and stop make. Let it continue.
    #   TODO: Do replace if GNU Make.
    ch_rule(
        '$(FIRST_MAKEFILE)',
        commands => [
            qr{^\Q$(SAY) "==> Your Makefile has been rebuilt. <=="\E$}      => '',
            qr{^\Q$(SAY) "==> Please rerun the $(MAKE) command.  <=="\E$}   => '',
            qr{^\Q$(FALSE)\E$}                                              => '',
        ],
    )
};

#   staticmake
#   makeaperl

sub My_test {
    #   I want verbose testing.
    ch_var( 'TEST_VERBOSE', value => [ '' => '1' ] );
    #   And release testing too.
    foreach my $target ( qw{ test_dynamic testdb_dynamic } ) {
        ch_rule(
            $target,
            prereqs => [
                #`MANIFEST.SKIP` is required, otherwise test `manifest.t` will fail.
                qr{\z} => ' MANIFEST.SKIP',
            ],
            commands => [
                qr{^(.*?\Q$(FULLPERLRUN)\E.*)$} => sub { "RELEASE_TESTING=1 $1" },
            ]
        );
    };
};

#   test
#   ppd
#   pm_to_blib
#   selfdocument

sub My_postamble {
    $Text .= rpm() . rules();
};

# --------------------------------------------------------------------------------------------------
# Makefile.PL sections.
# --------------------------------------------------------------------------------------------------

sub rules {
    my $text =
        "\n" .
        "# --- " . __FILE__ . " rules section:\n" .
        rule( '%/.dir', [], '$(MKPATH) $(dir $@)', '$(TOUCH) $@' ) .
        "";
    return $text;
};

sub rpm {
    my $D = sub {
        return '-D"' . $_[ 0 ] . ' ' . $_[ 1 ]->absolute . '" ';
    };
    my $dir = dir();
    my $topdir = $dir->subdir( '_rpmbuild' );
    my $srcdir = $dir;
    my $spec   = $srcdir->subdir( 'rpm' )->file( "perl-$V{ name }.spec" );
    my $brpm   = "perl-$V{ name }-$V{ version }-$V{ release }.$V{ arch }.rpm";
    my $srpm   = "perl-$V{ name }-$V{ version }-$V{ release }.src.rpm";
    my $text =
        "# --- " . __FILE__ . " rpm section:\n" .
        rule( $srpm, [ $spec, '$(DISTVNAME).tar$(SUFFIX)' ],
            '$(RPMBUILD) ' . $D->( '_topdir', $topdir ) . $D->( '_sourcedir', $srcdir ) . '-bs $<',
            '$(CP) ' . $topdir->subdir( 'SRPMS' )->file( $srpm ) . ' $@'
        ) .
        rule( $brpm, [ $spec, '$(DISTVNAME).tar$(SUFFIX)' ],
            '$(RPMBUILD) ' . $D->( '_topdir', $topdir ) . $D->( '_sourcedir', $srcdir ) . '-bb $<',
            '$(CP) ' . $topdir->subdir( "RPMS/$V{ arch }" )->file( $brpm ) . ' $@'
        ) .
        rule( 'rpm', [ $srpm, $brpm ],
            '$(NOECHO) $(NOOP)'
        );
    return $text;
}; # rpm

# --------------------------------------------------------------------------------------------------
# Some magic.
# --------------------------------------------------------------------------------------------------

#   Customizing a make section requires non-trivial work for finding the original section text.
#   This method does all the stuff, so customizing methods (defined above) a free from repeating the
#   same boilerplate code: they just operate on $Text variable.
#
sub section {

    my ( $self, $name, @args ) = @_;

    #   Find out superclass method name.
    #   Some superclass methods named differently, for example: not "matafile"
    #   but "metafile_taget". Handle such a case (MakeMaker does the same).
    my $super = "SUPER::$name";
    if ( not $self->can( $super ) ) {
        $super .= '_target';
    }; # if

    #   Load original section text into $Text variable.
    local $Text = $self->$super( @args );

    #   Call customization method.
    {
        if ( exists( $MY::{ "my_$name" } ) ) {
            no strict 'refs';
            &{ "my_$name" }();
        }; # if
        if ( exists( $MY::{ "My_$name" } ) and $devel ) {
            reveal();
            no strict 'refs';
            &{ "My_$name" }();
        }; # if
    };

    #   Return customized text to MakeMaker.
    return $Text;

}; # sub section

BEGIN {
    my %methods;
    foreach my $symbol ( keys( %MY:: ) ) {
        if ( $symbol =~ m{^[Mm]y_(.*)$} ) {
            if ( not $methods{ lc( $symbol ) } ) {
                eval "sub $1 { section( shift( \@_ ), \"$1\", \@_ ); };";
                $methods{ lc( $symbol ) } = 1;
            }; # if
        }; # if
    }
} # BEGIN

} # package MY

__END__

=begin comment

Using `Module::Install` looks nice but does not work. `make` finishes with error:

    make: *** No rule to make target 'Daemon.inl', needed by 'pure_all'.  Stop.

This error can be workarounded by moving `Daemon.pm` to the top-level directory, but in such case
`make` each run rebuilds almost everything because `make` does not create `Daemon.inl`. I believe
it is a bug in the `Module::Install::Inline`.

=end comment

=cut

use 5.006;
use strict;
use warnings FATAL => 'all';
use inc::Module::Install;

name      'Systemd-Daemon';
all_from  'lib/Systemd/Daemon.pm';

configure_requires  'Inline::MakeMaker'     => '0.45',
configure_requires  'ExtUtils::MakeMaker'   => '6.52',

test_requires   'Test::More'    => 0;

requires  'Inline'          => '0.45';
requires  'Inline::C'       => 0;
requires  'Inline::Filters' => 0;

WriteAll( inline => 1 );

# end of file #
