#!perl

package AuthorVDBTesterBuild;

use if $ENV{AUTOMATED_TESTING}, 'Test::DiagINC';
use namespace::autoclean;
use utf8;
use version 0.77;
use open ':locale';

use Test::Routine;

with 'Test::Dist::Zilla::Build';
with 'Test::Dist::Zilla::BuiltFiles' => { -version => 'v0.3.1' };
    # ^ A bug in v0.3.0 affects the test.

use Path::Tiny;
use Test::Deep qw{ cmp_deeply re };
use Test::More;
use Test::Routine::Util;

# REQUIRE: Dist::Zilla 5.040
    #   Starting from version 5.040, `PodCoverageTests` and `PodSyntaxTests` generate author test,
    #   not release test as before. Older version will cause test failure.
# REQUIRE: Dist::Zilla::Plugin::Test::Version 1.05
    #   Starting from version 1.05, `Test::Version` generates author test, not release test as
    #   before. Older version will cause test failure.
# REQUIRE: Dist::Zilla::Plugin::Manifest::Write v0.9.0
    #   v0.9.0 does not show file mungers. The bundle can works with earlier versions, but tests
    #   will fail.
# REQUIRE: Dist::Zilla::Plugin::MojibakeTests 0.8
    #   MojibakeTests 0.8 generates author test, not release as before. Older version will cause
    #   test failure.
# REQUIRE: Dist::Zilla::Plugin::Test::Pod::No404s 1.003
    #   Test::Pod::No404s 1.003 generates author test, not release as before. Older version will
    #   cause test failure.
# REQUIRE: Dist::Zilla::Plugin::Test::Portability 2.000007
    #   Test::Portability 2.000007 generates author test, not release as before. Older version will
    #   cause test failure.

# I do not need `version` because `@Author::VDB` have a `Hook::VersionProvider`.
around _build_dist => sub {
    my ( $orig, $self, @args ) = @_;
    my $dist = $self->$orig( @args );
    delete( $dist->{ version } );
    return $dist;
};

sub _build_message_filter {
    my ( $self ) = @_;
    return sub {
        map( { ( my $r = $_ ) =~ s{^\[[^\]]*\] }{}; $r } grep( $_ =~ m{^\Q[Author::VDB]\E }, @_ ) );
    };
};

has options => (
    isa         => 'HashRef',
    is          => 'ro',
    default     => sub { {} },
);

has extra_files => (
    isa         => 'HashRef',
    is          => 'ro',
    default     => sub { {} },
);

sub _build_files {
    my ( $self ) = @_;
    return {
        'doc/copying.pod' => [
            '=encoding UTF-8',
            '',
            '=head1 COPYRIGHT AND LICENSE',
            '',
            'Copyright © 2015 John Doe',
            '',
            '=cut',
            '',
            '# end of file #',
        ],
        'doc/what.pod' => [
            '=encoding UTF-8',
            '',
            '=head1 WHAT?',
            '',
            'That!',
            '',
            '=cut',
        ],
        'doc/why.pod' => [
            '=encoding UTF-8',
            '',
            '=head1 WHY?',
            '',
            'Err...',
            '',
            '=cut',
        ],
        'lib/Dummy.pm' => [
            'package Dummy;',
            '# VERSION',
            '# {{ $dist->version }}',
            '1;',
        ],
        'lib/Dummy/Manual.pod' => [
            '=encoding UTF-8',
            '',
            '=head1 DESCRIPTION',
            '',
            'POD file is a {{ \'template\' }}.',
            # TODO: Use $MY::name, etc.
            '',
            '=cut',
            '',
            '# end of file',
        ],
        'Changes'  => '# Changes #',
        'VERSION'  => 'v0.3.1',
        'MANIFEST' => [
            'doc/                       / Documentation.',
            'doc/what.pod               - ',
            'doc/why.pod                - ',
            'doc/copying.pod            - Do not copy it to the distribution.',
            'lib/                       / Perl modules.',
            'lib/Dummy.pm               + Main module',
            'lib/Dummy/Manual.pod       + User manual.',
            '',
            'Changes                    + Release history.',
            'VERSION                    + Software version.',
        ],
        %{ $self->extra_files },
    };
};

sub _build_plugins {
    my ( $self ) = @_;
    return [
        [ '@Author::VDB' => $self->options ],
    ];
};

has check => (
    isa         => 'CodeRef',
    is          => 'ro',
    default     => sub { sub {}; },
);

test 'Post-build check' => sub {
    my ( $self ) = @_;
    if ( $self->exception ) {
        plan skip_all => 'exception occurred';
    };
    plan 'no_plan';
    $self->check->( $self );
    pass;
};

my $aborting = "Aborting...\n";

# --------------------------------------------------------------------------------------------------

run_me 'Special' => {
    extra_files => {
        'doc/assa.pod' => '©',
        'MANIFEST' => [
            'doc/assa.pod               +',
            'lib/Dummy.pm               + Main module',
            'lib/Dummy/Manual.pod       + User manual.',
            'Changes                    + Release history.',
            'VERSION                    + Software version.',
        ],
    },
    expected => {
    },
    check => sub {
        my ( $self ) = @_;
        {
            diag( "== root dir ==" );
            my $root = path( $self->tzil->root );
            my @assa = $root->child( 'doc/assa.pod' )->lines_utf8();
            diag( "doc/assa.pod: " . ( @assa + 0 ) . " lines:" );
            diag( "    $_" ) for @assa;
            if ( @assa ) {
                my $first = $assa[ 0 ];
                diag( "first line '$first'" );
                diag( "first line length: " . length( $first ) );
                diag( utf8::is_utf8( $first ) ? "first line IS utf8" : "first line is NOT utf8" );
            };
        };
        {
            diag( "== build dir ==" );
            my $built_in = path( $self->tzil->built_in );
            my @assa = $built_in->child( 'doc/assa.pod' )->lines_utf8();
            diag( "doc/assa.pod: " . ( @assa + 0 ) . " lines:" );
            diag( "    $_" ) for @assa;
            if ( @assa ) {
                my $first = $assa[ 0 ];
                diag( "first line '$first'" );
                diag( "first line length: " . length( $first ) );
                diag( utf8::is_utf8( $first ) ? "first line IS utf8" : "first line is NOT utf8" );
            };
        };
        {
            diag( "== environment ==" );
            diag( "LANG=$ENV{ LANG }" );
            diag( "$_=$ENV{ $_ }" ) for sort( grep( { $_ =~ m{^LC_} } keys( %ENV ) ) );
            diag( "== versions ==" );
            for my $m ( sort( qw{ Encode Encode::Encoding Path::Tiny Unicode::UTF8 Dist::Zilla Path::Class } ) ) {
                eval "require $m;\n";
                diag( "$m " . ( $@ ? "oops" : $m->VERSION ) );
            };
        }
    },
};

run_me 'No options' => {
    expected => {
        files => {
            'doc/copying.pod'   => undef,   #   These files are not copied to distribution
            'doc/what.pod'      => undef,   #   (`Manifest::Read` works).
            'doc/why.pod'       => undef,
            'COPYING' => [                  #   `COPYING` is generated; it is a plain text, not POD.
                'COPYRIGHT AND LICENSE',    #   BTW, real `doc/copying.pod` file overrides
                '',                         #   file embedded into the bundle.
                'Copyright © 2015 John Doe',
            ],
            'README' => re( qr{^WHAT\?\n\nThat!\n\nWHY\?\n\nErr\.\.\.\n\nNAMING\n} ),
            'lib/Dummy.pm' => [             # Perl code fragments expanded.
                'package Dummy;',
                'our $VERSION = \'v0.3.1\'; # VERSION',     # Hook::VersionProvider + OurPkgVersion.
                '# v0.3.1',
                #  ^^^^^^^ Perl code fragment expanded.
                '1;',
            ],
            'META.json' => re( qr{} ),  # MetaJSON
            'META.yml'  => re( qr{} ),  # MetaYAML
            # TODO: Test PodWeaver.
            'MANIFEST' => [
                re( qr{^# This file was generated with Dist::Zilla::Plugin::Manifest::Write\b} ),
                'Build.PL                     # 3rd party file built by ModuleBuildTiny',
                'COPYING                      #     Dummy file built by GenerateFile',
                'Changes                      #     Dummy file added by Manifest::Read',
                'MANIFEST                     #  metainfo file built by Manifest::Write',
                'META.json                    #  metainfo file built by MetaJSON',
                'META.yml                     #  metainfo file built by MetaYAML',
                'README                       #     Dummy file built by GenerateFile',
                'VERSION                      #     Dummy file added by Manifest::Read',
                'lib/Dummy.pm                 #     Dummy file added by Manifest::Read',
                'lib/Dummy/Manual.pod         #     Dummy file added by Manifest::Read',
                't/00-compile.t               # 3rd party file built by Test::Compile',
                'xt/author/critic.t           # 3rd party file built by Test::Perl::Critic',
                'xt/author/eol.t              # 3rd party file built by Test::EOL',
                'xt/author/mojibake.t         # 3rd party file built by MojibakeTests',
                'xt/author/no-tabs.t          # 3rd party file built by Test::NoTabs',
                'xt/author/pod-coverage.t     # 3rd party file built by PodCoverageTests',
                'xt/author/pod-no404s.t       # 3rd party file built by Test::Pod::No404s',
                'xt/author/pod-spell.t        # 3rd party file built by Test::PodSpelling',
                'xt/author/pod-syntax.t       # 3rd party file built by PodSyntaxTests',
                'xt/author/portability.t      # 3rd party file built by Test::Portability',
                'xt/author/test-version.t     # 3rd party file built by Test::Version',
                'xt/release/cpan-changes.t    # 3rd party file built by Test::CPAN::Changes',
                'xt/release/dist-manifest.t   # 3rd party file built by Test::DistManifest',
                'xt/release/distmeta.t        # 3rd party file built by MetaTests',
                'xt/release/fixme.t           # 3rd party file built by Test::Fixme',
                'xt/release/kwalitee.t        # 3rd party file built by Test::Kwalitee',
                'xt/release/meta-json.t       # 3rd party file built by Test::CPAN::Meta::JSON',
                'xt/release/minimum-version.t # 3rd party file built by Test::MinimumVersion',
                'xt/release/new-version.t     # 3rd party file built by Test::NewVersion',
                'xt/release/pod-linkcheck.t   # 3rd party file built by Test::Pod::LinkCheck',
                'xt/release/synopsis.t        # 3rd party file built by Test::Synopsis',
            ],
            'VERSION' => 'v0.3.1',
        },
    },
};

#   Non-default `copying` option.
run_me 'Copying option' => {
    options => {
        'copying' => 'doc/non-default-copying.pod',
    },
    extra_files => {
        'doc/non-default-copying.pod' => [
            '=encoding UTF-8',
            '',
            '=head1 LICENSE AND COPYRIGHT',
            '',
            'Copyleft © 2015 Jane Doe',
            '',
        ],
    },
    expected => {
        files => {
            'doc/non-default-copying.pod' => undef,
            'COPYING' => [
                # `COPYING` file is plain text, not POD.
                'LICENSE AND COPYRIGHT',
                '',
                'Copyleft © 2015 Jane Doe',
            ],
        },
    },
};

run_me 'Empty copying option' => {
    options => {
        'copying' => '',
    },
    expected => {
        files => {
            'COPYING' => undef, # If `copying` is empty, `COPYING` is not generated.
        },
    },
};

#   Non-default `readme` option.
run_me 'Readme option' => {
    options => {
        'readme' => [
            'doc/non-default-what.pod',
            'doc/non-default-copying.pod',
        ],
    },
    extra_files => {
        'doc/non-default-copying.pod' => [
            '=encoding UTF-8',
            '',
            '=head1 LICENSE AND COPYRIGHT',
            '',
            'Copyleft © 2015 Jane Doe',
        ],
        'doc/non-default-what.pod' => [
            '=encoding UTF-8',
            '',
            '=head1 WHAT?',
            '',
            'Nope.',
            '',
            '=cut',
        ],
    },
    expected => {
        files => {
            'doc/non-default-copying.pod' => undef,
            'doc/non-default-what.pod' => undef,
            'README' => [
                'WHAT?',
                '',
                'Nope.',
                '',
                'LICENSE AND COPYRIGHT',
                '',
                'Copyleft © 2015 Jane Doe',
            ],
        },
    },
};

#   Empty `readme` option.
run_me 'Empty readme option' => {
    options => {
        'readme' => '',
    },
    expected => {
        files => {
            'README' => undef,
        },
    },
};

#   Non-default `templates` option.
run_me 'Templates option' => {
    options => {
        templates => ':NoFiles',
        #             ^^^^^^^^ No files are treated as templates.
    },
    expected => {
        files => {
            'lib/Dummy.pm' => [
                'package Dummy;',
                'our $VERSION = \'v0.3.1\'; # VERSION',
                '# {{ $dist->version }}',
                #  ^^^^^^^^^^^^^^^^^^^^ Perl code is not evaluated.
                '1;',
            ],
        },
    },
};

#   Empty `templates` option.
run_me 'Empty templates option' => {
    options => {
        templates => '',
    },
    expected => {
        files => {
            'lib/Dummy.pm' => [
                'package Dummy;',
                'our $VERSION = \'v0.3.1\'; # VERSION',
                '# {{ $dist->version }}',
                #  ^^^^^^^^^^^^^^^^^^^^ Perl code is not evaluated.
                '1;',
            ],
        },
    },
};

done_testing;

exit( 0 );
