# t/ManifestWriteTester.pm #

#   - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
#
#   Copyright © 2015 Van de Bugger
#
#   This file is part of perl-Dist-Zilla-Plugin-Manifest-Write.
#
#   perl-Dist-Zilla-Plugin-Manifest-Write is free software: you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by the Free Software
#   Foundation, either version 3 of the License, or (at your option) any later version.
#
#   perl-Dist-Zilla-Plugin-Manifest-Write is distributed in the hope that it will be useful, but
#   WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
#   PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License along with
#   perl-Dist-Zilla-Plugin-Manifest-Write. If not, see <http://www.gnu.org/licenses/>.
#
#   - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

package ManifestWriteTester;

#   The test is written using `Moose`-based `Test::Routine`. It is not big deal, because we are
#   testing plugin for `Dist::Zilla`, and `Dist-Zilla` is also `Moose`-based.

use strict;
use warnings;
use autodie ':all';
use namespace::autoclean;

use File::Temp qw{ tempdir };
use Test::DZil;
use Test::Fatal;
use Test::More;
use Test::Routine;

# `AutoPrereqs` fails to detect these dependencies:
use IPC::System::Simple ();                     # Used by `autodie ':all'`.
use Software::License::GPL_3::or_later ();      # Used by `t/dist/dust.ini`.

has plugin => (
    is          => 'ro',
    isa         => 'Str',
    default     => 'Dist::Zilla::Plugin::Manifest::Write',
);

#   Content of `dist.ini` file, in form acceptable by `Builder->from_config`.
has ini_body => (
    is          => 'ro',
    isa         => 'ArrayRef',
    required    => 1,
);

#   Expected content of `MANIFEST`, array of lines without "\n".
has manifest => (
    is          => 'ro',
    isa         => 'ArrayRef',
    required    => 1,
);

#   Test `Dist::Zilla` instance.
has tzil => (
    is          => 'ro',
    isa         => 'Object',
    lazy        => 1,
    builder     => 'build_tzil',
    init_arg    => undef,
    handles     => [ 'build' ],
);

sub build_tzil {
    my ( $self ) = @_;
    return Builder->from_config(
        { dist_root => tempdir( CLEANUP => 1 ) },
        {
            add_files => {
                'source/lib/Dummy.pm' =>
                    "package Dummy;\n" .
                    "\n" .                  # Blank line for `PkgVersion`.
                    "# ABSTRACT: Dummy\n" .
                    "# VERSION\n" .
                    "1;\n",
                'source/dist.ini' => dist_ini(
                    {
                        name                => 'Dummy',
                        version             => '0.001',
                        author              => 'John Doe',
                        license             => 'GPL_3::or_later',
                        copyright_holder    => 'John Doe',
                        copyright_year      => '2007',
                    },
                    @{ $self->ini_body },
                ),
            },
        },
    );
};

#   Return build log as single string.
sub build_log {
    my ( $self ) = @_;
    return
        join(
            "\n",
            map( $_->{ message }, @{ $self->tzil->chrome->logger->events } )
        );
};

test 'test' => sub {
    my ( $self ) = @_;
    plan tests => 3;
    my $exception = exception { $self->build() };
    is( $exception, undef, 'build must pass' )
        or diag( "Build messages:\n" . $self->build_log )
            #   If build unexpectedly fail, print out all the build messages to make failure
            #   investigation easier.
        and skip 'build unexpectedly fail', 2;
            #   And skip the rest — they will fail anyway.
    my @manifest =  $self->tzil->built_in->file( 'MANIFEST' )->
        slurp( iomode => '<:encoding(UTF-8)', chomp => 1 );
    my $head = shift( @manifest );
    my $plugin = $self->plugin;
    like(
        $head,
        qr{
            \A
            \# \Q This file was generated with $plugin \E
                \d+\.\d\d\d(?:_\d\d\d)?
            \z
        }x,
        'manifest head'
    );
    is_deeply( \@manifest, $self->manifest, 'manifest body' );
};

1;

# end of file #
