#! /usr/bin/env false

use v6.c;

use Terminal::ANSIColor;

class Pod::To::Pager::BorderedBlock
{
	has Str %.box-characters =
		outer-top-left     => "┏",
		outer-top-right    => "┓",
		outer-bottom-left  => "┗",
		outer-bottom-right => "┛",
		outer-horizontal   => "━",
		outer-vertical     => "┃",
		seperator-left     => "┠",
		seperator-right    => "┨",
		seperator          => "─",
	;

	has Str $.content = "";
	has Str $.header = "";
	has Str $.footer = "";
	has Int $.minimum-width = 66;

	multi method render (
		--> Str
	) {
		self.render(
			$!content,
			header => $!header,
			footer => $!footer,
			minimum-width => $!minimum-width,
		);
	}

	multi method render (
		Str:D $content is copy,
		:$header = $!header,
		:$footer = $!header,
		:$minimum-width = $!minimum-width,
		--> Str
	) {
		# Trim content
		$content .= trim;

		# Calculate the maximum width of the content
		my Int $longest-string = 0;

		for $content.lines { $longest-string = colorstrip($_).chars if $longest-string < colorstrip($_).chars; }
		for $header.lines  { $longest-string = colorstrip($_).chars if $longest-string < colorstrip($_).chars; }
		for $footer.lines  { $longest-string = colorstrip($_).chars if $longest-string < colorstrip($_).chars; }

		$longest-string = $minimum-width if $longest-string < $minimum-width;

		# Create the top bar
		my Str $block = %!box-characters<outer-top-left>;
		$block ~= %!box-characters<outer-horizontal> x $longest-string;
		$block ~= %!box-characters<outer-top-right>;
		$block ~= "\n";

		# Add a header, if needed
		if ($header ne "") {
			for $header.lines {
				$block ~= self!wrap-line($_, $longest-string);
			}

			$block ~= %!box-characters<seperator-left>;
			$block ~= %!box-characters<seperator> x $longest-string;
			$block ~= %!box-characters<seperator-right>;
			$block ~= "\n";
		}

		# Add the main content
		for $content.lines {
			$block ~= self!wrap-line($_, $longest-string);
		}

		# Add a footer, if needed
		if ($footer ne "") {
			$block ~= %!box-characters<seperator-left>;
			$block ~= %!box-characters<seperator> x $longest-string;
			$block ~= %!box-characters<seperator-right>;
			$block ~= "\n";

			for $footer.lines {
				$block ~= self!wrap-line($_, $longest-string);
			}
		}

		# Create the bottom bar
		$block ~= %!box-characters<outer-bottom-left>;
		$block ~= %!box-characters<outer-horizontal> x $longest-string;
		$block ~= %!box-characters<outer-bottom-right>;
	}

	method !wrap-line (
		Str:D $line,
		Int:D $length,
		--> Str
	) {
		%!box-characters<outer-vertical> ~ $line ~ " " x ($length - colorstrip($line).chars) ~ %!box-characters<outer-vertical> ~ "\n";
	}
}

=begin pod

=NAME   Pod::To::Pager::BorderedBlock
=AUTHOR Patrick Spek
=VERSION 0.2.0
=LICENSE GNU Aferro General Public License, version 3

=head1 Description

This module is supporting for C<p6man>, to create the blocks of text surrounded
by a border. It is intended to be seperated into it's own module later on.

=head1 Examples

=head2 Default usage

=begin code
use Pod::To::Pager::BorderedBlock;

my Pod::To::Pager::BorderedBlock $block .= new;

say $block.render("Some content to put a border around")
=end code

=begin output
┏━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┓
┃Some content to put a border around                               ┃
┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛
=end output

=end pod

# vim: ft=perl6 noet
