=head1 NAME

AudioFile::Identify::MusicBrainz::Query

=head1 DESCRIPTION

The main query object. This is probably the thing you want to use.

=head1 METHODS

=cut

package AudioFile::Identify::MusicBrainz::Query;

use AudioFile::Identify::MusicBrainz;

use strict;
use warnings::register;
use base qw(AudioFile::Identify::MusicBrainz::Object);

use LWP;
use LWP::UserAgent;
use Data::Dumper;
use XML::DOM;

use AudioFile::Identify::MusicBrainz::RDF qw(rdf);
use AudioFile::Identify::MusicBrainz::Artist;
use AudioFile::Identify::MusicBrainz::Album;
use AudioFile::Identify::MusicBrainz::Result;
use AudioFile::Identify::MusicBrainz::Track;
use AudioFile::Identify::MusicBrainz::Store;

=head2 userAgent()

returns a singleton C<LWP::UserAgent> configured for use in retrieving
data from MusicBrainz.org.

=cut

my $_UA = undef;  # Private package scope singleton ref for LWP::UserAgent
sub userAgent {
  return $_UA if defined $_UA;

  # The _UA variable hasn't been initialized.  Do so now.
  my $VERSION = $AudioFile::Identify::MusicBrainz::VERSION;
  $_UA = LWP::UserAgent->new(agent => "AudioFile::Identify::MusicBrainz/$VERSION http://jerakeen.org/programming/musicbrainz",
			     keep_alive => 1,);

  # Enable the use of proxys configured through environment variables
  $_UA->env_proxy;

  return $_UA;
}


sub init {
  my $self = shift;
  $self->store( AudioFile::Identify::MusicBrainz::Store->new() )
    unless $self->store;
  1;
}

sub url {
  return "http://mm.musicbrainz.org/cgi-bin/mq_2_1.pl";
}


# The MusicBrainz server chokes when certain characters are passed in
# the query without being transformed into proper XML entities.  This
# method does such transformations.
sub _escape_mb {
    my $val = shift;
    $val =~ s/&/&amp;/g if defined $val;
    $val =~ s/</&lt;/g if defined $val;

    return $val;
}


sub _fileinfolookup_to_rdf {
  my $self = shift;
  my @fields = qw(
    trm
    artist
    album
    track
    tracknum
    secs
    filename
    artist_id
    album_id
    track_id
  );

  unshift(@fields, 'dummy'); # RDF templates number them from zero.

  my $query;
  if (ref($_[0])) {
    $query = shift;
  } else {
    $query = { @_ };
  }

  # MB wants length in milliseconds. Sigh.
  $query->{secs} = $query->{secs} * 1000 if $query->{secs};

  $query->{tracknum} =~ s/\D.*// if $query->{tracknum};
  $query->{track} ||= $query->{title}; # common error

  my $params = {};
  for (my $i = 0; $i < scalar(@fields); $i++) {
    # Copy query values into params hash, transforming anything
    # necessary for the MusicBrainz server.
    $params->{$i} = _escape_mb($query->{$fields[$i]});
  }
  $params->{MAX_ITEMS} = $query->{items} || 20;

  my $rdf = rdf('MBQ_FileInfoLookup', $params);

  return $rdf;
}


=head2 FileInfoLookup($query)

Perform a MusicBrainz FileInfoLookup query - this is the clever one that I care
about. $query here is a hash / hashref where the keys are the information you
know:

  use AudioFile::Identify::MusicBrainz::Query;
  my $query = AudioFile:::Identify::MusicBrainz::Query->new();
  $query->FileInfoLookup(
      artist => 'coldplay',
      title => 'yellow',
      items => 30,
  ) or die "Could not query: " . $query->error();
  print "I got ".scalar(@{$query->results})." results\n";

keys you can use in the query are:

=over 4

=item trm

the trm - the audio fingerprint of the song. Generating this is a Future Project.

=item artist

the artist name.  Either this key or L<artist_id> must be supplied for
a query to be valid.

=item artist_id

The artist ID assigned by MusicBrainz, without the 'http' portion.
For example, '3d2b98e5-556f-4451-a3ff-c50ea18d57cb'.

=item album

the album name

=item album_id

The album ID assigned by MusicBrainz, without the 'http' portion.
For example, '2d5d5d4c-d0d7-4772-962f-de7185605ff8'.

=item track

the track title

=item track_id

The track ID assigned by MusicBrainz, without the 'http' portion.
For example, 'f27a903c-f5cc-465f-91c4-5c67d8b3d6d6'.

=item tracknum

the track number

=item secs

the track length in seconds

=item filename

the filename of the track

=back

This function will return true on success, false on error, and an error will
be accessible through the C<error> function.

=cut

sub FileInfoLookup {
  my $self = shift;

  my $query;
  if (ref($_[0])) {
    $query = shift;
  } else {
    $query = { @_ };
  }

  my $rdf = $self->_fileinfolookup_to_rdf($query);

  #die $rdf;

  my $ua = userAgent();

  my $req = HTTP::Request->new(POST => $self->url,);
  $req->content($rdf);

  my $res = $ua->request($req);

  # Check the outcome of the response
  if ($res->is_success) {
    $self->response($res->content);
    return $self->parse();
  } else {
    $self->response('');
    $self->error('got bad http response: '.$res->code);
    return 0;
  }
}


=head2 response

returns the unparsed RDF text of the response generated by Query.

=cut

sub response {
  my $self = shift;
  my $set = shift;
  if (defined($set)) {
    $self->{response} = $set;
    return $self;
  } else {
    return $self->{response};
  }
}

=head2 error

returns the last error generated by Query.

=cut

sub error {
  my $self = shift;
  my $set = shift;
  if (defined($set)) {
    $self->{error} = $set;
    return $self;
  } else {
    return $self->{error};
  }
}


sub parse {
  my $self = shift;
  return unless $self->response;
  my $parser = new XML::DOM::Parser;
  my $doc = $parser->parse($self->response);
  my $n;

  # get any errors.
  my $error_nodes = $doc->getElementsByTagName('mq:error');
  $n = $error_nodes->getLength;
  for (my $i = 0; $i < $n; $i++) {
    my $node = $error_nodes->item($i);
    my $error = $node->getFirstChild->toString;
    $self->error("MusicBrainz server responded with error message: $error");
    return undef;
  }

  # Handle any Artist or Album nodes found in the response.  This
  # loads them into the Store.
  my $artist_nodes = $doc->getElementsByTagName('mm:Artist');

  $n = $artist_nodes->getLength;
  for (my $i = 0; $i < $n; $i++) {
    my $node = $artist_nodes->item($i);
    my $id = $node->getAttributeNode("rdf:about")->getValue;

    AudioFile::Identify::MusicBrainz::Artist->new($id, $self->store, $node);
  }

  my $album_nodes = $doc->getElementsByTagName('mm:Album');

  $n = $album_nodes->getLength;
  for (my $i = 0; $i < $n; $i++) {
    my $node = $album_nodes->item($i);
    my $id = $node->getAttributeNode("rdf:about")->getValue;

    AudioFile::Identify::MusicBrainz::Album->new($id, $self->store, $node);
  }

  my $results = [];

  # Figure out if the rest of the result set is a lookupResultList
  # (multiple albums/tracks returned of varrying relevance) or an
  # exact match (only a single track returned), typically as a result
  # of specifying a track_id in the query.
  my $rLen = $doc->getElementsByTagName('mq:lookupResultList')->getLength;

  if ($rLen > 0) {

    # Handle the lookupResultList response type by going after any
    # AlbumTrackResult or AlbumResult nodes.
    my $result_nodes = $doc->getElementsByTagName('mq:AlbumTrackResult');

    $n = $result_nodes->getLength;
    for (my $i = 0; $i < $n; $i++) {
      my $node = $result_nodes->item($i);
      my $result =
        AudioFile::Identify::MusicBrainz::Result->new()
                                                ->store($self->store)
                                                ->type('Track')
                                                ->parse($node);
      push @$results, $result;
    }

    $result_nodes = $doc->getElementsByTagName('mq:AlbumResult');

    $n = $result_nodes->getLength;
    for (my $i = 0; $i < $n; $i++) {
      my $node = $result_nodes->item($i);
      my $result =
        AudioFile::Identify::MusicBrainz::Result->new()
                                                ->store($self->store)
                                                ->type('Album')
                                                ->parse($node);
      push @$results, $result;
    }

  } else {

    # Handle non-resultList response type (exact match) by going after
    # the single Album and Track nodes and creating a Result object.

    # ---------------- Get the track
    my $result_nodes = $doc->getElementsByTagName('mm:Track');

    # There should only be one track in this situation, so return an
    # error if there is more than one.
    if ($result_nodes->getLength != 1) {
      $self->error("Should be exactly 1 track, but got " .
                   $result_nodes->getLength);
      return undef;
    }

    # Retrieve the track object into the Store
    my $track_id = $result_nodes->item(0)->getAttributeNode("rdf:about")
                                              ->getValue;
    AudioFile::Identify::MusicBrainz::Track->new($track_id, $self->store);

    # ---------------- Get the album
    my $album_nodes = $doc->getElementsByTagName('mm:Album');

    # There should only be one album in this situation, so return an
    # error if there is more than one.
    if ($album_nodes->getLength != 1) {
      $self->error("Should be exactly 1 album, but got " .
                   $album_nodes->getLength);
      return undef;
    }

    my $album_node = $album_nodes->item(0);
    my $album_id = $album_node->getAttributeNode("rdf:about")
                                            ->getValue;
    AudioFile::Identify::MusicBrainz::Album->new($album_id, $self->store,
						 $album_node);

    # ---------------- Build a result object
    my $result =
      AudioFile::Identify::MusicBrainz::Result->new()
                                              ->store($self->store)
                                              ->type('Track')
                                              ->track($track_id)
                                              ->album($album_id)
                                              ->relevance(100);

    push @$results, $result;
  }

  $self->{results} = $results;
  return 1;
}


=head2 results

returns a listref of C<AudioFile::Identify::MusicBrainz::Result> objects.

=cut

sub results {
  my $self = shift;
  return $self->{results};
}

=head2 result(index)

returns a given result, at a given index, or C<undef> if the result
set doesn't contain a result at that index value.

=cut

sub result {
  my $self = shift;
  my $index = shift;

  if (defined $self->results and $index < $self->resultCount()) {
    return $self->results->[$index];
  } else {
    return undef;
  }
}


=head2 resultCount()

returns the number of items in the results set.

=cut

sub resultCount {
  my $self = shift;

  if (defined $self->results) {
    return scalar @{$self->results};
  } else {
    return 0;
  }
}

1;

