package Pinto::Types;

# ABSTRACT: Moose types used within Pinto

use strict;
use warnings;

use MooseX::Types -declare => [ qw( Author Uri Dir File Files Io Vers StackName
                                    StackAll StackDefault PropertyName PkgSpec
                                    DistSpec Spec Specs) ];

use MooseX::Types::Moose qw( Str Num ScalarRef ArrayRef Undef
                             HashRef FileHandle Object Int );

use URI;
use Path::Class::Dir;
use Path::Class::File;
use IO::String;
use IO::Handle;
use IO::File;

use Pinto::SpecFactory;
use Pinto::Constants qw(:all);

use version;
use namespace::autoclean;

#-----------------------------------------------------------------------------

our $VERSION = '0.049'; # VERSION

#-----------------------------------------------------------------------------

subtype Author,
  as Str,
  where   { $_ =~ $PINTO_AUTHOR_REGEX },
  message { 'The author id (' . (defined() ? $_ : 'undef') . ') must be alphanumeric' };

#-----------------------------------------------------------------------------

subtype StackName,
  as      Str,
  where   { $_ =~ $PINTO_STACK_NAME_REGEX },
  message { 'The stack name (' . (defined() ? $_ : 'undef') . ') must be alphanumeric' };

#-----------------------------------------------------------------------------

subtype StackAll,
  as      Str,
  where   { $_ eq $PINTO_STACK_NAME_ALL },
  message { qq{The stack name must be '$PINTO_STACK_NAME_ALL'} };

#-----------------------------------------------------------------------------

subtype StackDefault,
  as      Undef;

#-----------------------------------------------------------------------------

subtype PropertyName,
  as      Str,
  where   { $_ =~ $PINTO_PROPERTY_NAME_REGEX },
  message { 'The property name (' . (defined() ? $_ : 'undef') . 'must be alphanumeric' };

#-----------------------------------------------------------------------------

class_type Vers, {class => 'version'};

coerce Vers,
  from Str,
  via { version->parse($_) };

coerce Vers,
  from Num,
  via { version->parse($_) };

#-----------------------------------------------------------------------------

class_type Uri, {class => 'URI'};

coerce Uri,
  from Str,
  via { URI->new($_) };

#-----------------------------------------------------------------------------

class_type Dir, {class => 'Path::Class::Dir'};

coerce Dir,
  from Str,             via { Path::Class::Dir->new($_) },
  from ArrayRef,        via { Path::Class::Dir->new(@{$_}) };

#-----------------------------------------------------------------------------

class_type File, {class => 'Path::Class::File'};

coerce File,
  from Str,             via { Path::Class::File->new($_) },
  from ArrayRef,        via { Path::Class::File->new(@{$_}) };

#-----------------------------------------------------------------------------

subtype Files, as ArrayRef[File];

coerce Files,
  from File,          via { [ $_ ] },
  from Str,           via { [ Path::Class::File->new($_) ] },
  from ArrayRef[Str], via { [ map { Path::Class::File->new($_) } @$_ ] };

#-----------------------------------------------------------------------------

class_type PkgSpec, {class => 'Pinto::PackageSpec'};

coerce PkgSpec,
  from Str,     via { Pinto::SpecFactory->make_spec($_) },
  from HashRef, via { Pinto::SpecFactory->make_spec($_) };

#-----------------------------------------------------------------------------

class_type DistSpec, {class => 'Pinto::DistributionSpec'};

coerce DistSpec,
  from Str,         via { Pinto::SpecFactory->make_spec($_) },
  from HashRef,     via { Pinto::SpecFactory->make_spec($_) };


#-----------------------------------------------------------------------------

subtype Specs,
  as ArrayRef[PkgSpec| DistSpec];    ## no critic qw(ProhibitBitwiseOperators);

coerce Specs,
  from  PkgSpec,            via { [ $_ ] },
  from  DistSpec,           via { [ $_ ] },
  from  Str,                via { [ Pinto::SpecFactory->make_spec($_) ] },
  from  ArrayRef[Str],      via { [ map { Pinto::SpecFactory->make_spec($_) } @$_ ] };

#-----------------------------------------------------------------------------

subtype Io, as Object;

coerce Io,
    from Str,        via { my $fh = IO::File->new(); $fh->open($_);   return $fh },
    from File,       via { my $fh = IO::File->new(); $fh->open("$_"); return $fh },
    from ArrayRef,   via { IO::Handle->new_from_fd( @$_ ) },
    from ScalarRef,  via { IO::String->new( ${$_} ) };

#-----------------------------------------------------------------------------

1;



=pod

=for :stopwords Jeffrey Ryan Thalhammer Imaginative Software Systems

=head1 NAME

Pinto::Types - Moose types used within Pinto

=head1 VERSION

version 0.049

=head1 AUTHOR

Jeffrey Ryan Thalhammer <jeff@imaginative-software.com>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2011 by Imaginative Software Systems.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut


__END__
