#include "../lib/test.h"

#define TEST(name) TEST_CASE("parse-compression: " name, "[parse-compression]")

/* ACCEPT-ENCODING */
TEST("Accept-Encoding: gzip") {
    RequestParser p;
    string raw = "GET / HTTP/1.1\r\n"
        "Accept-Encoding: gzip\r\n"
    "\r\n";
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::done);
    CHECK(req->headers.get("Accept-Encoding") == "gzip");
    CHECK((req->allowed_compression() & Compression::GZIP));
    CHECK((req->allowed_compression() & Compression::IDENTITY));
}

TEST("Accept-Encoding: gzip;q=1.0") {
    RequestParser p;
    string raw = "GET / HTTP/1.1\r\n"
        "Accept-Encoding: gzip;q=1.0\r\n"
    "\r\n";
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::done);
    CHECK((req->allowed_compression() & Compression::GZIP));
    CHECK((req->allowed_compression() & Compression::IDENTITY));
}

TEST("Accept-Encoding: gzip;q=0.002") {
    RequestParser p;
    string raw = "GET / HTTP/1.1\r\n"
        "Accept-Encoding: gzip;q=0.002\r\n"
    "\r\n";
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::done);
    CHECK((req->allowed_compression() & Compression::GZIP));
    CHECK((req->allowed_compression() & Compression::IDENTITY));
}

TEST("Accept-Encoding: gzip;q=0.0") {
    RequestParser p;
    string raw = "GET / HTTP/1.1\r\n"
        "Accept-Encoding: gzip;q=0.0\r\n"
    "\r\n";
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::done);
    CHECK(!(req->allowed_compression() & Compression::GZIP));
    CHECK((req->allowed_compression() & Compression::IDENTITY));
}

TEST("Accept-Encoding: gzip;q=0.000") {
    RequestParser p;
    string raw = "GET / HTTP/1.1\r\n"
        "Accept-Encoding: gzip;q=0.000\r\n"
    "\r\n";
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::done);
    CHECK(!(req->allowed_compression() & Compression::GZIP));
    CHECK((req->allowed_compression() & Compression::IDENTITY));
}

TEST("Accept-Encoding: (empty)") {
    RequestParser p;
    string raw = "GET / HTTP/1.1\r\n"
        "Accept-Encoding:\r\n"
    "\r\n";
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::done);
    CHECK((req->allowed_compression() == Compression::IDENTITY));
}

TEST("Accept-Encoding: gzip, deflate, br, *;q=0") {
    RequestParser p;
    string raw = "GET / HTTP/1.1\r\n"
        "Accept-Encoding: gzip, deflate, br, *;q=0\r\n"
    "\r\n";
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::done);
    CHECK((req->allowed_compression() & Compression::GZIP));
    CHECK((req->allowed_compression() & Compression::DEFLATE));
    CHECK((req->allowed_compression() & Compression::IDENTITY));
}


TEST("Accept-Encoding: gzip;q=1.0, deflate;q=0") {
    RequestParser p;
    string raw = "GET / HTTP/1.1\r\n"
        "Accept-Encoding: gzip;q=1.0, deflate;q=0\r\n"
    "\r\n";
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::done);
    CHECK((req->allowed_compression() & Compression::GZIP));
    CHECK(!(req->allowed_compression() & Compression::DEFLATE));
    CHECK((req->allowed_compression() & Compression::IDENTITY));
}


TEST("Accept-Encoding: gzip;q=0.123, deflate;q=0.120") {
    RequestParser p;
    string raw = "GET / HTTP/1.1\r\n"
        "Accept-Encoding: gzip;q=0.123, deflate;q=0.120\r\n"
    "\r\n";
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::done);
    CHECK((req->allowed_compression() & Compression::GZIP));
    CHECK((req->allowed_compression() & Compression::DEFLATE));
    CHECK((req->allowed_compression() & Compression::IDENTITY));
}

/* CONTENT-ENCODING */

TEST("Content-Encoding: empty") {
    RequestParser p;
    string raw =
        "POST /upload HTTP/1.1\r\n"
        "\r\n"
        ;
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::done);
    CHECK(req->compression.type == Compression::IDENTITY);
}

TEST("Content-Encoding: identity") {
    RequestParser p;
    string raw =
        "POST /upload HTTP/1.1\r\n"
        "Content-Encoding: identity\r\n"
        "\r\n"
        ;
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::done);
    CHECK(req->compression.type == Compression::IDENTITY);
}

TEST("Content-Encoding: gzip") {
    RequestParser p;
    string raw =
        "POST /upload HTTP/1.1\r\n"
        "Content-Encoding: gzip\r\n"
        "\r\n"
        ;
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::done);
    CHECK(req->compression.type == Compression::GZIP);
}

TEST("Content-Encoding: deflate") {
    RequestParser p;
    string raw =
        "POST /upload HTTP/1.1\r\n"
        "Content-Encoding: deflate\r\n"
        "\r\n"
        ;
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::done);
    CHECK(req->compression.type == Compression::DEFLATE);
}

TEST("Content-Encoding: rar") {
    RequestParser p;
    string raw =
        "POST /upload HTTP/1.1\r\n"
        "Content-Encoding: rar\r\n"
        "\r\n"
        ;
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::error);
    CHECK(result.error == errc::unsupported_compression);
}

TEST("Content-Encoding: rar (disabled uncompression)") {
    RequestParser p;
    p.uncompress_content = false;
    string raw =
        "POST /upload HTTP/1.1\r\n"
        "Content-Encoding: rar\r\n"
        "\r\n"
        ;
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::done);
    CHECK(req->compression.type == Compression::IDENTITY);
}

TEST("Content-Encoding: gzip, identity") {
    RequestParser p;
    string raw =
        "POST /upload HTTP/1.1\r\n"
        "Content-Encoding: gzip, identity\r\n"
        "\r\n"
        ;
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::done);
    CHECK(req->compression.type == Compression::GZIP);
}

TEST("Content-Encoding: gzip, gzip") {
    RequestParser p;
    string raw =
        "POST /upload HTTP/1.1\r\n"
        "Content-Encoding: gzip, gzip\r\n"
        "\r\n"
        ;
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::error);
    CHECK(result.error == errc::unsupported_compression);
}

/* compression gzip */
unsigned char gzip_sample[] = {
    0x50, 0x4f, 0x53, 0x54, 0x20, 0x2f, 0x20, 0x48,
    0x54, 0x54, 0x50, 0x2f, 0x31, 0x2e, 0x31, 0x0d,
    0x0a, 0x48, 0x6f, 0x73, 0x74, 0x3a, 0x20, 0x31,
    0x32, 0x37, 0x2e, 0x30, 0x2e, 0x30, 0x2e, 0x31,
    0x0d, 0x0a, 0x55, 0x73, 0x65, 0x72, 0x2d, 0x41,
    0x67, 0x65, 0x6e, 0x74, 0x3a, 0x20, 0x63, 0x75,
    0x72, 0x6c, 0x2f, 0x37, 0x2e, 0x36, 0x37, 0x2e,
    0x30, 0x0d, 0x0a, 0x41, 0x63, 0x63, 0x65, 0x70,
    0x74, 0x3a, 0x20, 0x2a, 0x2f, 0x2a, 0x0d, 0x0a,
    0x43, 0x6f, 0x6e, 0x74, 0x65, 0x6e, 0x74, 0x2d,
    0x45, 0x6e, 0x63, 0x6f, 0x64, 0x69, 0x6e, 0x67,
    0x3a, 0x20, 0x67, 0x7a, 0x69, 0x70, 0x0d, 0x0a,
    0x43, 0x6f, 0x6e, 0x74, 0x65, 0x6e, 0x74, 0x2d,
    0x4c, 0x65, 0x6e, 0x67, 0x74, 0x68, 0x3a, 0x20,
    0x33, 0x38, 0x0d, 0x0a, 0x43, 0x6f, 0x6e, 0x74,
    0x65, 0x6e, 0x74, 0x2d, 0x54, 0x79, 0x70, 0x65,
    0x3a, 0x20, 0x61, 0x70, 0x70, 0x6c, 0x69, 0x63,
    0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2f, 0x78, 0x2d,
    0x77, 0x77, 0x77, 0x2d, 0x66, 0x6f, 0x72, 0x6d,
    0x2d, 0x75, 0x72, 0x6c, 0x65, 0x6e, 0x63, 0x6f,
    0x64, 0x65, 0x64, 0x0d, 0x0a, 0x0d, 0x0a, 0x1f,
    0x8b, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0xf3, 0xc9, 0x2f, 0x4a, 0xcd, 0x55, 0xc8,
    0x2c, 0x28, 0x2e, 0xcd, 0x55, 0x48, 0xc9, 0xcf,
    0xc9, 0x2f, 0xe2, 0x02, 0x00, 0xe8, 0xd0, 0x4e,
    0xfa, 0x12, 0x00, 0x00, 0x00
};

TEST("request with gzip payload (small piece)") {

/*
wireshark captured piece. Generated via

echo 'Lorem ipsum dolor' | gzip | curl -v -i --data-binary @- -H "Content-Encoding: gzip" http://127.0.0.1/

POST / HTTP/1.1
Host: 127.0.0.1
User-Agent: curl/7.67.0
Content-Encoding: gzip
Content-Length: 38
Content-Type: application/x-www-form-urlencoded

............/J.U.,(..UH.../.....N.....
*/
    string raw(reinterpret_cast<char*>(gzip_sample), sizeof (gzip_sample));
    RequestParser p;
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::done);
    CHECK(result.error.value() == 0);
    CHECK(req->compression.type == Compression::GZIP);
    REQUIRE(req->body.parts.size() == 1);
    CHECK(req->body.parts[0] == "Lorem ipsum dolor\n");

    SECTION("check reset") {
        // p.reset() is implicitly called
        auto req = result.request;
        CHECK(result.state == State::done);
        CHECK(result.error.value() == 0);
        CHECK(req->compression.type == Compression::GZIP);
        REQUIRE(req->body.parts.size() == 1);
        CHECK(req->body.parts[0] == "Lorem ipsum dolor\n");
    }
}

TEST("request with corrupted gzip payload") {
    // the same sample as above, but changed byte in body
    unsigned char data[] = {
        0x50, 0x4f, 0x53, 0x54, 0x20, 0x2f, 0x20, 0x48,
        0x54, 0x54, 0x50, 0x2f, 0x31, 0x2e, 0x31, 0x0d,
        0x0a, 0x48, 0x6f, 0x73, 0x74, 0x3a, 0x20, 0x31,
        0x32, 0x37, 0x2e, 0x30, 0x2e, 0x30, 0x2e, 0x31,
        0x0d, 0x0a, 0x55, 0x73, 0x65, 0x72, 0x2d, 0x41,
        0x67, 0x65, 0x6e, 0x74, 0x3a, 0x20, 0x63, 0x75,
        0x72, 0x6c, 0x2f, 0x37, 0x2e, 0x36, 0x37, 0x2e,
        0x30, 0x0d, 0x0a, 0x41, 0x63, 0x63, 0x65, 0x70,
        0x74, 0x3a, 0x20, 0x2a, 0x2f, 0x2a, 0x0d, 0x0a,
        0x43, 0x6f, 0x6e, 0x74, 0x65, 0x6e, 0x74, 0x2d,
        0x45, 0x6e, 0x63, 0x6f, 0x64, 0x69, 0x6e, 0x67,
        0x3a, 0x20, 0x67, 0x7a, 0x69, 0x70, 0x0d, 0x0a,
        0x43, 0x6f, 0x6e, 0x74, 0x65, 0x6e, 0x74, 0x2d,
        0x4c, 0x65, 0x6e, 0x67, 0x74, 0x68, 0x3a, 0x20,
        0x33, 0x38, 0x0d, 0x0a, 0x43, 0x6f, 0x6e, 0x74,
        0x65, 0x6e, 0x74, 0x2d, 0x54, 0x79, 0x70, 0x65,
        0x3a, 0x20, 0x61, 0x70, 0x70, 0x6c, 0x69, 0x63,
        0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2f, 0x78, 0x2d,
        0x77, 0x77, 0x77, 0x2d, 0x66, 0x6f, 0x72, 0x6d,
        0x2d, 0x75, 0x72, 0x6c, 0x65, 0x6e, 0x63, 0x6f,
        0x64, 0x65, 0x64, 0x0d, 0x0a, 0x0d, 0x0a, 0x1f,
        0x8b, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x03, 0xf3, 0xc9, 0x2f, 0x4a, 0xcd, 0x55, 0xc8,
        0x2c, 0x28, 0x2e, 0xcd, 0x55, 0x48, 0xc9, 0xcf,
        0xc9, 0x2f, 0xe2, 0x02, 0x00, 0xe8, 0xd0, 0x4e,
        0xfa, 0x99, 0x00, 0x00, 0x00 };

    string raw(reinterpret_cast<char*>(data), sizeof (data));
    RequestParser p;
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::error);
    CHECK(req->compression.type == Compression::GZIP);
    CHECK(result.error == errc::uncompression_failure);
}

TEST("request with redundant gzip payload") {
    // the same sample as above + extra byte on
    unsigned char data[] = {
        0x50, 0x4f, 0x53, 0x54, 0x20, 0x2f, 0x20, 0x48,
        0x54, 0x54, 0x50, 0x2f, 0x31, 0x2e, 0x31, 0x0d,
        0x0a, 0x48, 0x6f, 0x73, 0x74, 0x3a, 0x20, 0x31,
        0x32, 0x37, 0x2e, 0x30, 0x2e, 0x30, 0x2e, 0x31,
        0x0d, 0x0a, 0x55, 0x73, 0x65, 0x72, 0x2d, 0x41,
        0x67, 0x65, 0x6e, 0x74, 0x3a, 0x20, 0x63, 0x75,
        0x72, 0x6c, 0x2f, 0x37, 0x2e, 0x36, 0x37, 0x2e,
        0x30, 0x0d, 0x0a, 0x41, 0x63, 0x63, 0x65, 0x70,
        0x74, 0x3a, 0x20, 0x2a, 0x2f, 0x2a, 0x0d, 0x0a,
        0x43, 0x6f, 0x6e, 0x74, 0x65, 0x6e, 0x74, 0x2d,
        0x45, 0x6e, 0x63, 0x6f, 0x64, 0x69, 0x6e, 0x67,
        0x3a, 0x20, 0x67, 0x7a, 0x69, 0x70, 0x0d, 0x0a,
        0x43, 0x6f, 0x6e, 0x74, 0x65, 0x6e, 0x74, 0x2d,
        0x4c, 0x65, 0x6e, 0x67, 0x74, 0x68, 0x3a, 0x20,
        0x33, 0x39, 0x0d, 0x0a, 0x43, 0x6f, 0x6e, 0x74,
        0x65, 0x6e, 0x74, 0x2d, 0x54, 0x79, 0x70, 0x65,
        0x3a, 0x20, 0x61, 0x70, 0x70, 0x6c, 0x69, 0x63,
        0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2f, 0x78, 0x2d,
        0x77, 0x77, 0x77, 0x2d, 0x66, 0x6f, 0x72, 0x6d,
        0x2d, 0x75, 0x72, 0x6c, 0x65, 0x6e, 0x63, 0x6f,
        0x64, 0x65, 0x64, 0x0d, 0x0a, 0x0d, 0x0a, 0x1f,
        0x8b, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x03, 0xf3, 0xc9, 0x2f, 0x4a, 0xcd, 0x55, 0xc8,
        0x2c, 0x28, 0x2e, 0xcd, 0x55, 0x48, 0xc9, 0xcf,
        0xc9, 0x2f, 0xe2, 0x02, 0x00, 0xe8, 0xd0, 0x4e,
        0xfa, 0x12, 0x00, 0x00, 0x00, 0x00 };

    string raw(reinterpret_cast<char*>(data), sizeof (data));
    RequestParser p;
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::error);
    CHECK(req->compression.type == Compression::GZIP);
    CHECK(result.error == errc::uncompression_failure);
}

TEST("request with gzip payload (byte-by-byte consumption)") {
    string raw(reinterpret_cast<char*>(gzip_sample), sizeof (gzip_sample));
    RequestParser p;
    RequestParser::Result result;
    for(size_t i = 1; i <= raw.size(); ++i){
        auto piece = raw.substr(i - 1, 1);
        result = p.parse(piece);
        if ((result.state == State::done) || (result.state == State::error)) { break; }
    }
    auto req = result.request;
    CHECK(result.state == State::done);
    CHECK(result.error.value() == 0);
    CHECK(req->compression.type == Compression::GZIP);

    string joined;
    for(auto& it: req->body.parts) { joined += it; }
    CHECK(joined == "Lorem ipsum dolor\n");
}

TEST("request with gzip payload, no content-length") {
    unsigned char data[] = {
        0x48, 0x54, 0x54, 0x50, 0x2f, 0x31, 0x2e, 0x31,
        0x20, 0x32, 0x30, 0x30, 0x20, 0x4f, 0x4b, 0x0d,
        0x0a,
        0x48, 0x6f, 0x73, 0x74, 0x3a, 0x20, 0x31,
        0x32, 0x37, 0x2e, 0x30, 0x2e, 0x30, 0x2e, 0x31,
        0x0d, 0x0a, 0x55, 0x73, 0x65, 0x72, 0x2d, 0x41,
        0x67, 0x65, 0x6e, 0x74, 0x3a, 0x20, 0x63, 0x75,
        0x72, 0x6c, 0x2f, 0x37, 0x2e, 0x36, 0x37, 0x2e,
        0x30, 0x0d, 0x0a, 0x41, 0x63, 0x63, 0x65, 0x70,
        0x74, 0x3a, 0x20, 0x2a, 0x2f, 0x2a, 0x0d, 0x0a,
        0x43, 0x6f, 0x6e, 0x74, 0x65, 0x6e, 0x74, 0x2d,
        0x45, 0x6e, 0x63, 0x6f, 0x64, 0x69, 0x6e, 0x67,
        0x3a, 0x20, 0x67, 0x7a, 0x69, 0x70, 0x0d, 0x0a,
        0x43, 0x6f, 0x6e, 0x74,
        0x65, 0x6e, 0x74, 0x2d, 0x54, 0x79, 0x70, 0x65,
        0x3a, 0x20, 0x61, 0x70, 0x70, 0x6c, 0x69, 0x63,
        0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2f, 0x78, 0x2d,
        0x77, 0x77, 0x77, 0x2d, 0x66, 0x6f, 0x72, 0x6d,
        0x2d, 0x75, 0x72, 0x6c, 0x65, 0x6e, 0x63, 0x6f,
        0x64, 0x65, 0x64, 0x0d, 0x0a, 0x0d, 0x0a, 0x1f,
        0x8b, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x03, 0xf3, 0xc9, 0x2f, 0x4a, 0xcd, 0x55, 0xc8,
        0x2c, 0x28, 0x2e, 0xcd, 0x55, 0x48, 0xc9, 0xcf,
        0xc9, 0x2f, 0xe2, 0x02, 0x00, 0xe8, 0xd0, 0x4e,
        0xfa, 0x12, 0x00, 0x00, 0x00 };

    string raw(reinterpret_cast<char*>(data), sizeof (data));
    ResponseParser p;
    p.set_context_request(new Request(Method::Get, new URI("/")));
    auto result = p.parse(raw);
    auto res = result.response;
    CHECK(result.state == State::body);
    CHECK(res->compression.type == Compression::GZIP);
    REQUIRE(res->body.parts.size() == 1);
    CHECK(res->body.parts[0] == "Lorem ipsum dolor\n");
    CHECK(p.eof().state == State::done);
}

TEST("request with gzip payload (max body size)") {
    unsigned char data[] = {
        0x50, 0x4f, 0x53, 0x54, 0x20, 0x2f, 0x20, 0x48,
        0x54, 0x54, 0x50, 0x2f, 0x31, 0x2e, 0x31, 0x0d,
        0x0a, 0x48, 0x6f, 0x73, 0x74, 0x3a, 0x20, 0x31,
        0x32, 0x37, 0x2e, 0x30, 0x2e, 0x30, 0x2e, 0x31,
        0x0d, 0x0a, 0x55, 0x73, 0x65, 0x72, 0x2d, 0x41,
        0x67, 0x65, 0x6e, 0x74, 0x3a, 0x20, 0x63, 0x75,
        0x72, 0x6c, 0x2f, 0x37, 0x2e, 0x36, 0x37, 0x2e,
        0x30, 0x0d, 0x0a, 0x41, 0x63, 0x63, 0x65, 0x70,
        0x74, 0x3a, 0x20, 0x2a, 0x2f, 0x2a, 0x0d, 0x0a,
        0x43, 0x6f, 0x6e, 0x74, 0x65, 0x6e, 0x74, 0x2d,
        0x45, 0x6e, 0x63, 0x6f, 0x64, 0x69, 0x6e, 0x67,
        0x3a, 0x20, 0x67, 0x7a, 0x69, 0x70, 0x0d, 0x0a,
        0x43, 0x6f, 0x6e, 0x74, 0x65, 0x6e, 0x74, 0x2d,
        0x4c, 0x65, 0x6e, 0x67, 0x74, 0x68, 0x3a, 0x20,
        0x32, 0x35, 0x0d, 0x0a, 0x43, 0x6f, 0x6e, 0x74,
        0x65, 0x6e, 0x74, 0x2d, 0x54, 0x79, 0x70, 0x65,
        0x3a, 0x20, 0x61, 0x70, 0x70, 0x6c, 0x69, 0x63,
        0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2f, 0x78, 0x2d,
        0x77, 0x77, 0x77, 0x2d, 0x66, 0x6f, 0x72, 0x6d,
        0x2d, 0x75, 0x72, 0x6c, 0x65, 0x6e, 0x63, 0x6f,
        0x64, 0x65, 0x64, 0x0d, 0x0a, 0x0d, 0x0a, 0x1f,
        0x8b, 0x08, 0x00, 0xa3, 0x9d, 0xf0, 0x5d, 0x02,
        0x03, 0x73, 0x74, 0xa4, 0x02, 0xe0, 0x02, 0x00,
        0xef, 0xa5, 0xc2, 0x7b, 0x46, 0x00, 0x00, 0x00 };

    string raw(reinterpret_cast<char*>(data), sizeof (data));
    RequestParser p;
    p.max_body_size = 30;
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::error);
    CHECK(req->compression.type == Compression::GZIP);
    CHECK(result.error == errc::body_too_large);
}

TEST("response with gzipped chunked response") {
    unsigned char data[] = {
        0x48, 0x54, 0x54, 0x50, 0x2f, 0x31, 0x2e, 0x31,
        0x20, 0x32, 0x30, 0x30, 0x20, 0x4f, 0x4b, 0x0d,
        0x0a, 0x53, 0x65, 0x72, 0x76, 0x65, 0x72, 0x3a,
        0x20, 0x6e, 0x67, 0x69, 0x6e, 0x78, 0x2f, 0x31,
        0x2e, 0x31, 0x36, 0x2e, 0x31, 0x0d, 0x0a, 0x44,
        0x61, 0x74, 0x65, 0x3a, 0x20, 0x57, 0x65, 0x64,
        0x2c, 0x20, 0x31, 0x31, 0x20, 0x44, 0x65, 0x63,
        0x20, 0x32, 0x30, 0x31, 0x39, 0x20, 0x31, 0x30,
        0x3a, 0x30, 0x39, 0x3a, 0x35, 0x36, 0x20, 0x47,
        0x4d, 0x54, 0x0d, 0x0a, 0x43, 0x6f, 0x6e, 0x74,
        0x65, 0x6e, 0x74, 0x2d, 0x54, 0x79, 0x70, 0x65,
        0x3a, 0x20, 0x74, 0x65, 0x78, 0x74, 0x2f, 0x70,
        0x6c, 0x61, 0x69, 0x6e, 0x0d, 0x0a, 0x4c, 0x61,
        0x73, 0x74, 0x2d, 0x4d, 0x6f, 0x64, 0x69, 0x66,
        0x69, 0x65, 0x64, 0x3a, 0x20, 0x57, 0x65, 0x64,
        0x2c, 0x20, 0x31, 0x31, 0x20, 0x44, 0x65, 0x63,
        0x20, 0x32, 0x30, 0x31, 0x39, 0x20, 0x31, 0x30,
        0x3a, 0x30, 0x39, 0x3a, 0x31, 0x38, 0x20, 0x47,
        0x4d, 0x54, 0x0d, 0x0a, 0x54, 0x72, 0x61, 0x6e,
        0x73, 0x66, 0x65, 0x72, 0x2d, 0x45, 0x6e, 0x63,
        0x6f, 0x64, 0x69, 0x6e, 0x67, 0x3a, 0x20, 0x63,
        0x68, 0x75, 0x6e, 0x6b, 0x65, 0x64, 0x0d, 0x0a,
        0x43, 0x6f, 0x6e, 0x6e, 0x65, 0x63, 0x74, 0x69,
        0x6f, 0x6e, 0x3a, 0x20, 0x6b, 0x65, 0x65, 0x70,
        0x2d, 0x61, 0x6c, 0x69, 0x76, 0x65, 0x0d, 0x0a,
        0x56, 0x61, 0x72, 0x79, 0x3a, 0x20, 0x41, 0x63,
        0x63, 0x65, 0x70, 0x74, 0x2d, 0x45, 0x6e, 0x63,
        0x6f, 0x64, 0x69, 0x6e, 0x67, 0x0d, 0x0a, 0x45,
        0x54, 0x61, 0x67, 0x3a, 0x20, 0x57, 0x2f, 0x22,
        0x35, 0x64, 0x66, 0x30, 0x63, 0x30, 0x34, 0x65,
        0x2d, 0x34, 0x35, 0x22, 0x0d, 0x0a, 0x43, 0x6f,
        0x6e, 0x74, 0x65, 0x6e, 0x74, 0x2d, 0x45, 0x6e,
        0x63, 0x6f, 0x64, 0x69, 0x6e, 0x67, 0x3a, 0x20,
        0x67, 0x7a, 0x69, 0x70, 0x0d, 0x0a, 0x0d, 0x0a,
        0x31, 0x38, 0x0d, 0x0a, 0x1f, 0x8b, 0x08, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x73, 0x74,
        0xa4, 0x02, 0x00, 0x00, 0xe6, 0x31, 0xe4, 0x7c,
        0x45, 0x00, 0x00, 0x00, 0x0d, 0x0a, 0x30, 0x0d,
        0x0a, 0x0d, 0x0a };

    string raw(reinterpret_cast<char*>(data), sizeof (data));
    ResponseParser p;
    p.set_context_request(new Request(Method::Get, new URI("/")));
    auto result = p.parse(raw);
    auto res = result.response;
    CHECK(result.state == State::done);
    CHECK(res->compression.type == Compression::GZIP);
    REQUIRE(res->body.parts.size() == 1);
    CHECK(res->body.to_string() == "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA");
}

TEST("response with corrupted gzipped chunked response") {
    unsigned char data[] = {
        0x48, 0x54, 0x54, 0x50, 0x2f, 0x31, 0x2e, 0x31,
        0x20, 0x32, 0x30, 0x30, 0x20, 0x4f, 0x4b, 0x0d,
        0x0a, 0x53, 0x65, 0x72, 0x76, 0x65, 0x72, 0x3a,
        0x20, 0x6e, 0x67, 0x69, 0x6e, 0x78, 0x2f, 0x31,
        0x2e, 0x31, 0x36, 0x2e, 0x31, 0x0d, 0x0a, 0x44,
        0x61, 0x74, 0x65, 0x3a, 0x20, 0x57, 0x65, 0x64,
        0x2c, 0x20, 0x31, 0x31, 0x20, 0x44, 0x65, 0x63,
        0x20, 0x32, 0x30, 0x31, 0x39, 0x20, 0x31, 0x30,
        0x3a, 0x30, 0x39, 0x3a, 0x35, 0x36, 0x20, 0x47,
        0x4d, 0x54, 0x0d, 0x0a, 0x43, 0x6f, 0x6e, 0x74,
        0x65, 0x6e, 0x74, 0x2d, 0x54, 0x79, 0x70, 0x65,
        0x3a, 0x20, 0x74, 0x65, 0x78, 0x74, 0x2f, 0x70,
        0x6c, 0x61, 0x69, 0x6e, 0x0d, 0x0a, 0x4c, 0x61,
        0x73, 0x74, 0x2d, 0x4d, 0x6f, 0x64, 0x69, 0x66,
        0x69, 0x65, 0x64, 0x3a, 0x20, 0x57, 0x65, 0x64,
        0x2c, 0x20, 0x31, 0x31, 0x20, 0x44, 0x65, 0x63,
        0x20, 0x32, 0x30, 0x31, 0x39, 0x20, 0x31, 0x30,
        0x3a, 0x30, 0x39, 0x3a, 0x31, 0x38, 0x20, 0x47,
        0x4d, 0x54, 0x0d, 0x0a, 0x54, 0x72, 0x61, 0x6e,
        0x73, 0x66, 0x65, 0x72, 0x2d, 0x45, 0x6e, 0x63,
        0x6f, 0x64, 0x69, 0x6e, 0x67, 0x3a, 0x20, 0x63,
        0x68, 0x75, 0x6e, 0x6b, 0x65, 0x64, 0x0d, 0x0a,
        0x43, 0x6f, 0x6e, 0x6e, 0x65, 0x63, 0x74, 0x69,
        0x6f, 0x6e, 0x3a, 0x20, 0x6b, 0x65, 0x65, 0x70,
        0x2d, 0x61, 0x6c, 0x69, 0x76, 0x65, 0x0d, 0x0a,
        0x56, 0x61, 0x72, 0x79, 0x3a, 0x20, 0x41, 0x63,
        0x63, 0x65, 0x70, 0x74, 0x2d, 0x45, 0x6e, 0x63,
        0x6f, 0x64, 0x69, 0x6e, 0x67, 0x0d, 0x0a, 0x45,
        0x54, 0x61, 0x67, 0x3a, 0x20, 0x57, 0x2f, 0x22,
        0x35, 0x64, 0x66, 0x30, 0x63, 0x30, 0x34, 0x65,
        0x2d, 0x34, 0x35, 0x22, 0x0d, 0x0a, 0x43, 0x6f,
        0x6e, 0x74, 0x65, 0x6e, 0x74, 0x2d, 0x45, 0x6e,
        0x63, 0x6f, 0x64, 0x69, 0x6e, 0x67, 0x3a, 0x20,
        0x67, 0x7a, 0x69, 0x70, 0x0d, 0x0a, 0x0d, 0x0a,
        0x31, 0x38, 0x0d, 0x0a, 0x1f, 0x8b, 0x08, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x73, 0x74,
        0xa4, 0x02, 0x00, 0x00, 0xe6, 0x31, 0xe4, 0x9c,
        0x45, 0x00, 0x00, 0x00, 0x0d, 0x0a, 0x30, 0x0d,
        0x0a, 0x0d, 0x0a };

    string raw(reinterpret_cast<char*>(data), sizeof (data));
    ResponseParser p;
    p.set_context_request(new Request(Method::Get, new URI("/")));
    auto result = p.parse(raw);
    auto res = result.response;
    CHECK(result.state == State::error);
    CHECK(res->compression.type == Compression::GZIP);
    CHECK(result.error == errc::uncompression_failure);
}

TEST("MEIACORE-1427 bugfix (1)") {

    string header = "HTTP/1.1 200 OK\r\n"
        "Vary: Accept-Encoding\r\n"
        "Content-Encoding: gzip\r\n"
        "ETag: \"1acff3335ef2a417cac345f60413bf9463f1f7b8\"\r\n"
        "x-app-usage: {\"call_count\":6,\"total_cputime\":0,\"total_time\":0}\r\n"
        "Content-Type: text/javascript; charset=UTF-8\r\n"
        "facebook-api-version: v3.2\r\n"
        "Strict-Transport-Security: max-age=15552000; preload\r\n"
        "Pragma: no-cache\r\n"
        "x-fb-rev: 1002341698\r\n"
        "Access-Control-Allow-Origin: *\r\n"
        "Cache-Control: private, no-cache, no-store, must-revalidate\r\n"
        "x-fb-trace-id: AWywTKgtaSX\r\n"
        "x-fb-request-id: AJyeB12F2XH8cFf2EC7ajoe\r\n"
        "Expires: Sat, 01 Jan 2000 00:00:00 GMT\r\n"
        "X-FB-Debug: QTW94OgKSrRzpTbv3jRaGWruYkAYKRS8kiXx4r/p1OjrRb5d3syZemACF4ALkqMb6tiXc1Z7few5DHE3nZCZMQ==\r\n"
        "Date: Wed, 08 Jul 2020 17:05:15 GMT\r\n"
        "Alt-Svc: h3-29=\":443\"; ma=3600,h3-27=\":443\"; ma=3600\r\n"
        "Connection: keep-alive\r\n"
        "Content-Length: 862\r\n"
        "\r\n";

    const unsigned char piese_1_raw[714] = {
        0x1F, 0x8B, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0xED, 0x97, 0x5D, 0x6F, 0xE2, 0x38,
        0x14, 0x86, 0xFF, 0x0A, 0xEB, 0xAB, 0x5D, 0x6D, 0xA6, 0xF5, 0x77, 0xEC, 0xDC, 0xED, 0x00, 0x53,
        0x25, 0x1A, 0x60, 0x68, 0x29, 0xA5, 0x08, 0xA9, 0x32, 0x89, 0x69, 0x03, 0xF9, 0x40, 0x24, 0x0C,
        0x85, 0x11, 0xFF, 0x7D, 0x15, 0x1B, 0x15, 0x34, 0xB4, 0xA5, 0x73, 0x31, 0xDA, 0x55, 0xD5, 0x9B,
        0x43, 0xEC, 0xF3, 0x9E, 0x73, 0xEC, 0x63, 0x3D, 0xB2, 0xF9, 0x01, 0x10, 0x87, 0x04, 0x21, 0x08,
        0x25, 0x41, 0x42, 0x4A, 0xEA, 0xDE, 0x21, 0x86, 0x5C, 0xC9, 0x70, 0x35, 0x46, 0x90, 0x40, 0x08,
        0xBC, 0x1F, 0x40, 0xCD, 0xE7, 0x49, 0x1C, 0xAA, 0x32, 0xCE, 0xB3, 0x6A, 0x18, 0xAA, 0x52, 0xDF,
        0xE7, 0x8B, 0x35, 0xF0, 0xC0, 0x85, 0x4A, 0x75, 0x01, 0x1C, 0x90, 0xC4, 0xD9, 0x0C, 0x78, 0xE0,
        0xA1, 0x2C, 0xE7, 0x85, 0x37, 0x3A, 0x1F, 0x9D, 0xAB, 0xF9, 0xBC, 0x38, 0x9B, 0xA8, 0x50, 0x8F,
        0xF3, 0x7C, 0x76, 0x16, 0xE6, 0xE9, 0xE8, 0x3C, 0xD2, 0xDF, 0xEF, 0x56, 0xF9, 0x22, 0x89, 0xE6,
        0xF9, 0x4C, 0x2F, 0xC2, 0x64, 0x39, 0x1E, 0x9D, 0x03, 0x07, 0x64, 0x2A, 0xD5, 0xC0, 0x03, 0x7F,
        0x36, 0x9A, 0xFD, 0xBF, 0x6A, 0x37, 0x95, 0xBF, 0xF6, 0xAD, 0x12, 0xD4, 0xEA, 0xC9, 0x72, 0xBC,
        0xF3, 0x17, 0x73, 0x15, 0x56, 0xA2, 0xE3, 0x14, 0xC0, 0x01, 0x71, 0x04, 0x3C, 0x80, 0x30, 0xC4,
        0x48, 0x40, 0x24, 0x08, 0xA2, 0xAE, 0x20, 0x14, 0x6C, 0x1D, 0x10, 0x2E, 0xB4, 0x2A, 0x75, 0x74,
        0x57, 0xC6, 0xA6, 0x02, 0x86, 0x18, 0x7E, 0x82, 0xEE, 0x27, 0x28, 0x7A, 0x08, 0x79, 0x88, 0x7A,
        0x84, 0xFD, 0x0D, 0x21, 0x84, 0xC0, 0x01, 0x91, 0x2A, 0x15, 0xF0, 0xC0, 0x67, 0xBD, 0x0E, 0x56,
        0xFE, 0x34, 0x8F, 0x5B, 0xBD, 0xEB, 0x55, 0x7B, 0xDA, 0x5C, 0xB7, 0x7A, 0xCD, 0x4D, 0xA7, 0x77,
        0x0B, 0xDB, 0x8D, 0xFB, 0x4D, 0x7B, 0xED, 0x17, 0x7E, 0x16, 0x40, 0x3D, 0xF8, 0x9C, 0x54, 0x9A,
        0xA8, 0x17, 0xA0, 0xAF, 0xE9, 0x70, 0xAD, 0x6E, 0xFA, 0xCB, 0xE1, 0x97, 0x20, 0x09, 0x07, 0xFD,
        0x24, 0x24, 0xDD, 0xF8, 0x6B, 0x3D, 0x98, 0xAA, 0xBA, 0xCF, 0xFD, 0x14, 0xB1, 0x90, 0x5C, 0x26,
        0x61, 0x96, 0x4C, 0x6E, 0x53, 0x49, 0x07, 0xA4, 0x9D, 0x8C, 0x53, 0xE3, 0x9F, 0xF9, 0xD3, 0xB9,
        0xBB, 0xF7, 0xCF, 0xE2, 0xCE, 0xF4, 0x8B, 0x9C, 0x74, 0x81, 0x03, 0x26, 0x8B, 0x3C, 0xAD, 0x5A,
        0x6C, 0xF7, 0xC4, 0x20, 0x47, 0x18, 0x21, 0x22, 0x39, 0xA5, 0x82, 0xB8, 0xFB, 0x66, 0x8D, 0x96,
        0x90, 0xA2, 0xA8, 0xB2, 0x44, 0x18, 0xAB, 0xF6, 0xDF, 0x14, 0x1B, 0x4B, 0x0E, 0xE6, 0xAD, 0x85,
        0x35, 0x13, 0xA6, 0xCD, 0xC0, 0x58, 0x3A, 0x31, 0xDF, 0x26, 0x80, 0x30, 0x33, 0xE3, 0x1E, 0x04,
        0xB0, 0xAA, 0x89, 0xA9, 0x2E, 0x0A, 0x75, 0xFF, 0x54, 0xD4, 0x46, 0xD8, 0x1C, 0xE3, 0x7D, 0xA1,
        0x5D, 0x9C, 0xA8, 0x85, 0x0B, 0xB5, 0x59, 0x8F, 0xF3, 0xC7, 0x3F, 0x0E, 0x8A, 0xD9, 0x15, 0xD9,
        0xA4, 0x14, 0x1E, 0x24, 0x90, 0x46, 0x64, 0x2B, 0x13, 0xBB, 0x60, 0xED, 0xD8, 0x39, 0x6C, 0x7F,
        0xEC, 0x1E, 0xAD, 0x20, 0xB4, 0x53, 0xE1, 0x41, 0x02, 0xBE, 0x77, 0xD3, 0x5D, 0x84, 0x75, 0xEC,
        0x76, 0x6C, 0xAC, 0x5D, 0x9A, 0xF9, 0xDE, 0xF5, 0xC6, 0xA4, 0xA0, 0xC8, 0xD8, 0x89, 0x0D, 0xA3,
        0x36, 0xB8, 0x66, 0x26, 0x77, 0x35, 0xCC, 0xFE, 0x48, 0x74, 0x66, 0x14, 0xD4, 0x2A, 0x6C, 0x8B,
        0x6D, 0xD3, 0xED, 0xB2, 0x14, 0x70, 0x40, 0x99, 0x1F, 0x1C, 0xDA, 0x4F, 0xFC, 0x3C, 0x1D, 0x5A,
        0x33, 0x8A, 0x62, 0x5D, 0xEB, 0xEB, 0x28, 0xD2, 0x8B, 0xAA, 0xB3, 0x56, 0x7E, 0x92, 0xBF, 0xAD,
        0x03, 0x88, 0xA4, 0x4C, 0x70, 0x89, 0xA8, 0x20, 0x0C, 0x61, 0x7C, 0xAC, 0x79, 0x97, 0x8C, 0x42,
        0x0F, 0x41, 0x8F, 0xC1, 0x97, 0x19, 0x6D, 0xAE, 0x5B, 0x9B, 0x26, 0x6D, 0x35, 0xBA, 0xA8, 0xD3,
        0xBB, 0xC7, 0x9D, 0x5E, 0x93, 0xF8, 0xF1, 0x2A, 0x0E, 0xB3, 0x4B, 0x16, 0x5E, 0x5C, 0xC7, 0x9D,
        0x38, 0x40, 0xAD, 0xEC, 0x7A, 0x39, 0xCC, 0x82, 0xF9, 0xF0, 0x86, 0xCD, 0xAE, 0xD3, 0xFE, 0x63,
        0x74, 0xD3, 0xDF, 0x44, 0x75, 0xBF, 0xF0, 0xD3, 0x6E, 0xDC, 0xC9, 0x8A, 0x78, 0x3C, 0x48, 0x36,
        0xD1, 0x45, 0x7F, 0xAD, 0xAF, 0x7C, 0xDE, 0x1A, 0xC0, 0xC2, 0x4F, 0xDB, 0x79, 0x95, 0xFB, 0x69,
        0xBE, 0x2F, 0xE3, 0x31, 0x79, 0x98, 0x84, 0xB8, 0xBF, 0x1C, 0xD6, 0x03, 0x79, 0x44, 0x28, 0xC2,
        0x04, 0x09, 0x48, 0x99, 0xAC, 0x8E, 0xE7, 0x00, 0xD0, 0x96, 0x5A, 0xCC, 0x6A, 0x97, 0x2A, 0xCE,
        0xF2, 0xEF, 0x1F, 0x10, 0xFD, 0x76, 0x88, 0x4E, 0xF2, 0xB1, 0x75, 0x00, 0xC7, 0x9C, 0x33, 0x2C,
        0xA4, 0xC0, 0x90, 0x09, 0x4E, 0x8F, 0x35, 0xEF, 0x94, 0x21, 0x28, 0x3D, 0x42, 0x9F, 0x61, 0xE8,
        0xD4, 0x1D, 0xF5, 0x2A, 0x63, 0xC3, 0xBA, 0xCF, 0xF5, 0x3A, 0x28, 0xF5, 0xA0, 0x0D, 0x87, 0x83,
        0x80, 0xF9, 0xD3, 0xFC, 0x71, 0x72, 0xF5, 0x16, 0xF6, 0x3E
    };

    const unsigned char piese_2_raw[148] = {
        0x18, 0xFA, 0x9F, 0x32, 0x74, 0x92, 0x8F, 0xAD, 0x03, 0x70, 0xE5, 0xE5, 0x4C, 0x0A, 0x24, 0xA9,
        0x24, 0xFC, 0x58, 0xF3, 0x2E, 0x19, 0xE2, 0x1E, 0x43, 0x1E, 0xE5, 0xCF, 0x30, 0xB4, 0x63, 0xA4,
        0xBB, 0x6A, 0x37, 0xFE, 0x61, 0x9D, 0xC6, 0x2D, 0xED, 0xF4, 0xEE, 0x49, 0xA7, 0x71, 0x0B, 0x5B,
        0xD5, 0x1D, 0xF3, 0xB6, 0xB7, 0xE2, 0xB3, 0x6F, 0xC1, 0x97, 0xDF, 0x90, 0x41, 0xF2, 0xE2, 0x5D,
        0x44, 0x21, 0x85, 0x52, 0x70, 0x21, 0x85, 0x2B, 0x38, 0x3D, 0x38, 0xEF, 0xBA, 0x5A, 0x24, 0xB5,
        0xAB, 0x34, 0x2E, 0x1F, 0x3E, 0x38, 0xFA, 0xED, 0x1C, 0x9D, 0x64, 0x64, 0xEB, 0x00, 0x97, 0x30,
        0x4E, 0xB1, 0x2B, 0x20, 0x93, 0xC2, 0x95, 0xEC, 0x58, 0xF3, 0x2E, 0x39, 0xC2, 0x1E, 0x96, 0x1E,
        0x91, 0xBF, 0xCC, 0xD1, 0xAB, 0x6F, 0xB2, 0x57, 0xDE, 0x72, 0x6F, 0xE0, 0xEF, 0x99, 0xFF, 0x5C,
        0x1F, 0x14, 0x19, 0xFB, 0x1F, 0x53, 0x74, 0x92, 0x90, 0xED, 0xF6, 0x5F, 0xBD, 0x77, 0xDC, 0xB9,
        0xBC, 0x10, 0x00, 0x00
    };

    string piece_1 = header + string((char*)piese_1_raw, sizeof(piese_1_raw));
    string piece_2 =string((char*)piese_2_raw, sizeof(piese_2_raw));

    ResponseParser p;
    p.set_context_request(new Request(Method::Get, new URI("/")));
    auto res1 = p.parse_shift(piece_1);
    REQUIRE(res1.state == State::body);

    for(size_t i = 0; i < piece_2.size() - 1; ++i) {
        char c  = piece_2[i];
        string s(&c, 1);
        auto r = p.parse_shift(s);
        CHECK(r.state == State::body);
    }

    char c  = piece_2.back();
    string s(&c, 1);
    auto res2 = p.parse_shift(s);
    CHECK(res2.state == State::done);
};


TEST("MEIACORE-1427 bugfix (2)") {
    const unsigned char raw[926] = {
        0x48, 0x54, 0x54, 0x50, 0x2F, 0x31, 0x2E, 0x31, 0x20, 0x32, 0x30, 0x30, 0x20, 0x4F, 0x4B, 0x0D,
        0x0A, 0x44, 0x61, 0x74, 0x65, 0x3A, 0x20, 0x54, 0x75, 0x65, 0x2C, 0x20, 0x31, 0x34, 0x20, 0x4A,
        0x75, 0x6C, 0x20, 0x32, 0x30, 0x32, 0x30, 0x20, 0x31, 0x33, 0x3A, 0x33, 0x33, 0x3A, 0x31, 0x37,
        0x20, 0x47, 0x4D, 0x54, 0x0D, 0x0A, 0x43, 0x6F, 0x6E, 0x74, 0x65, 0x6E, 0x74, 0x2D, 0x54, 0x79,
        0x70, 0x65, 0x3A, 0x20, 0x61, 0x70, 0x70, 0x6C, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6F, 0x6E, 0x2F,
        0x6A, 0x73, 0x6F, 0x6E, 0x3B, 0x20, 0x63, 0x68, 0x61, 0x72, 0x73, 0x65, 0x74, 0x3D, 0x55, 0x54,
        0x46, 0x2D, 0x38, 0x0D, 0x0A, 0x54, 0x72, 0x61, 0x6E, 0x73, 0x66, 0x65, 0x72, 0x2D, 0x45, 0x6E,
        0x63, 0x6F, 0x64, 0x69, 0x6E, 0x67, 0x3A, 0x20, 0x63, 0x68, 0x75, 0x6E, 0x6B, 0x65, 0x64, 0x0D,
        0x0A, 0x43, 0x6F, 0x6E, 0x6E, 0x65, 0x63, 0x74, 0x69, 0x6F, 0x6E, 0x3A, 0x20, 0x63, 0x6C, 0x6F,
        0x73, 0x65, 0x0D, 0x0A, 0x56, 0x61, 0x72, 0x79, 0x3A, 0x20, 0x41, 0x63, 0x63, 0x65, 0x70, 0x74,
        0x2D, 0x45, 0x6E, 0x63, 0x6F, 0x64, 0x69, 0x6E, 0x67, 0x0D, 0x0A, 0x43, 0x6F, 0x6E, 0x74, 0x65,
        0x6E, 0x74, 0x2D, 0x45, 0x6E, 0x63, 0x6F, 0x64, 0x69, 0x6E, 0x67, 0x3A, 0x20, 0x67, 0x7A, 0x69,
        0x70, 0x0D, 0x0A, 0x53, 0x65, 0x72, 0x76, 0x65, 0x72, 0x3A, 0x20, 0x4E, 0x4B, 0x0D, 0x0A, 0x0D,
        0x0A, 0x32, 0x63, 0x31, 0x0D, 0x0A, 0x1F, 0x8B, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03,
        0xED, 0x9A, 0x4D, 0x8B, 0x1C, 0x39, 0x0C, 0x86, 0xFF, 0x4A, 0xA8, 0x73, 0xC7, 0xE3, 0x2F, 0xD9,
        0xD6, 0xFC, 0x81, 0xBD, 0xED, 0x21, 0xB0, 0xA7, 0x21, 0x04, 0xC9, 0x92, 0x27, 0x4D, 0xAA, 0x3F,
        0xE8, 0xAE, 0x09, 0xCC, 0x0E, 0xF3, 0xDF, 0xD7, 0xDD, 0x2C, 0xBB, 0x90, 0x5B, 0x2E, 0x21, 0x07,
        0x9D, 0xEA, 0xC3, 0x2E, 0x95, 0xEB, 0xC1, 0xD6, 0xFB, 0xAA, 0xF0, 0xD3, 0xDB, 0x72, 0xD1, 0xEB,
        0xCB, 0xBA, 0x2D, 0x8F, 0x6F, 0xCB, 0x76, 0xDA, 0x68, 0xFD, 0x74, 0xBF, 0xBC, 0x2E, 0x8F, 0x09,
        0x77, 0xCB, 0xBA, 0xBF, 0xCE, 0x96, 0xA7, 0xB7, 0xE5, 0xFC, 0xF5, 0xB4, 0x9D, 0xAE, 0xF7, 0xD3,
        0xED, 0xF5, 0xAC, 0xCB, 0xE3, 0xB2, 0x7D, 0x7D, 0x39, 0xF0, 0x91, 0xF6, 0xEB, 0xB2, 0x5B, 0xBE,
        0xD3, 0xFA, 0x72, 0xBB, 0xF7, 0xF0, 0xE0, 0xDD, 0xF5, 0xE3, 0xF1, 0x9B, 0x3B, 0xAF, 0x0F, 0xFB,
        0xC3, 0xF3, 0x03, 0x7D, 0xA7, 0x8D, 0x2E, 0xFF, 0x1E, 0xBE, 0x88, 0x0E, 0x9A, 0xA1, 0xBF, 0x1C,
        0x68, 0xD5, 0xE5, 0xFD, 0xF3, 0x6E, 0xD9, 0xCB, 0x7C, 0xE6, 0xAC, 0x97, 0xEB, 0xE9, 0xE8, 0x10,
        0xE3, 0x88, 0x41, 0x0B, 0x6B, 0x04, 0x25, 0x0D, 0x33, 0xEA, 0x91, 0x0E, 0x7A, 0x1B, 0xD6, 0x38,
        0x5D, 0x0E, 0xB4, 0x6D, 0x7A, 0xEB, 0xFD, 0xC7, 0x85, 0xFA, 0xDF, 0x1F, 0x8E, 0xF4, 0xE1, 0xFE,
        0x92, 0xE5, 0x7D, 0xB7, 0x3C, 0xEB, 0x51, 0xF4, 0x32, 0x9B, 0xEE, 0x61, 0x77, 0x8B, 0xEC, 0xAF,
        0xE7, 0x95, 0x5E, 0xFF, 0xBC, 0x3F, 0xFC, 0x63, 0xFF, 0xDD, 0xFF, 0xC3, 0xFE, 0xEB, 0xB2, 0xFE,
        0xDC, 0x88, 0x77, 0x0B, 0x3D, 0xCF, 0x90, 0xD1, 0x47, 0xFF, 0xBE, 0xFB, 0x05, 0x44, 0xA4, 0x63,
        0x89, 0x5C, 0xD1, 0x33, 0x62, 0xC3, 0x01, 0x46, 0x84, 0x14, 0xEB, 0xF0, 0xB5, 0x88, 0x0F, 0x21,
        0x65, 0x35, 0x22, 0xCE, 0x7B, 0xCC, 0x18, 0x35, 0xA2, 0xC7, 0x2A, 0x4D, 0xA3, 0x11, 0x51, 0x2D,
        0x1A, 0x42, 0x29, 0x29, 0x10, 0x70, 0x16, 0x9B, 0x23, 0x4E, 0x39, 0x86, 0x9C, 0x61, 0xD4, 0x91,
        0x14, 0x18, 0xBB, 0x11, 0x89, 0x4D, 0x87, 0x46, 0xE1, 0xC1, 0x81, 0x42, 0x24, 0x5B, 0x35, 0x0E,
        0x1A, 0x56, 0x68, 0x9A, 0x20, 0xA2, 0x44, 0x2D, 0xC3, 0x88, 0xE4, 0x21, 0xB1, 0x64, 0xCD, 0x31,
        0xFB, 0x2E, 0x9C, 0xB3, 0x11, 0xE1, 0xD4, 0x7C, 0xCB, 0x14, 0xB5, 0xB7, 0x06, 0x83, 0xD4, 0x88,
        0x44, 0xE8, 0x2C, 0x05, 0x7C, 0x8F, 0x9A, 0x5B, 0x34, 0x87, 0x36, 0x5D, 0x3C, 0x77, 0x6A, 0xA0,
        0x23, 0x63, 0x10, 0xD1, 0x60, 0x73, 0xC4, 0x8D, 0x51, 0x75, 0x2E, 0x97, 0x20, 0x21, 0x71, 0x01,
        0xD3, 0x9A, 0xA3, 0x6B, 0xB9, 0x87, 0x69, 0xCC, 0x40, 0x7A, 0x1A, 0xB9, 0x62, 0x35, 0x22, 0xB9,
        0x37, 0xAA, 0x53, 0x6D, 0x72, 0xA8, 0x9D, 0x3B, 0x98, 0x43, 0x73, 0x15, 0x51, 0x4B, 0xF5, 0x3E,
        0xA1, 0xAF, 0x99, 0xD9, 0x56, 0x8D, 0xA3, 0x04, 0x7D, 0x0C, 0xE0, 0x82, 0xB5, 0x27, 0xB1, 0xBA,
        0x66, 0xD6, 0xBE, 0xE4, 0xA3, 0xF2, 0x80, 0x32, 0x8D, 0x09, 0x50, 0x4E, 0x46, 0x04, 0x32, 0x55,
        0xA6, 0x00, 0x0D, 0xB8, 0xF5, 0xCA, 0x68, 0x44, 0xFC, 0x98, 0x86, 0x15, 0x58, 0x51, 0xF1, 0x26,
        0xBE, 0xA6, 0x35, 0x6E, 0x96, 0x33, 0xD0, 0xA3, 0x87, 0x69, 0x49, 0xA2, 0x08, 0x59, 0x1E, 0x71,
        0x89, 0x8A, 0xA4, 0x82, 0x09, 0x4A, 0x4C, 0x1A, 0xCC, 0xB3, 0x4E, 0x22, 0xAD, 0x89, 0x2F, 0x18,
        0x73, 0xD0, 0xD6, 0x4A, 0x2E, 0x46, 0x84, 0x7A, 0xAB, 0x21, 0x72, 0x2E, 0xEA, 0x03, 0x95, 0x64,
        0x73, 0xC4, 0x25, 0xCC, 0x28, 0x3E, 0x40, 0x2A, 0xB9, 0x49, 0x40, 0xD3, 0x1A, 0xD7, 0x4A, 0x8A,
        0x71, 0xB2, 0xE8, 0xC9, 0x4B, 0x25, 0xB5, 0x7F, 0x68, 0x2E, 0x67, 0x1D, 0x51, 0x43, 0x49, 0x20,
        0xC9, 0x07, 0x34, 0xAD, 0x71, 0x29, 0x21, 0x60, 0xEE, 0x99, 0x20, 0xA4, 0x2C, 0xD5, 0x32, 0xAB,
        0x8B, 0xD3, 0x9A, 0xF9, 0x1E, 0x67, 0x2E, 0xF1, 0xEC, 0x7D, 0xB5, 0xDA, 0xD7, 0xD5, 0xD2, 0x47,
        0x48, 0xC5, 0x6B, 0x50, 0x91, 0xA2, 0x56, 0xD7, 0x38, 0xD6, 0x36, 0x1A, 0x36, 0x4F, 0xA9, 0x03,
        0x33, 0x58, 0x1E, 0x71, 0x8C, 0x43, 0x47, 0xCA, 0x41, 0x40, 0xF2, 0x50, 0x31, 0xF5, 0x75, 0xD8,
        0x39, 0x65, 0xF0, 0x23, 0x95, 0x32, 0x94, 0xC8, 0xF6, 0x8F, 0xB8, 0x48, 0x85, 0x29, 0x49, 0x47,
        0x48, 0xAD, 0x78, 0x6A, 0x46, 0x04, 0x3A, 0x71, 0x1F, 0x55, 0x6A, 0x44, 0x1E, 0xA8, 0x6C, 0x44,
        0x44, 0x83, 0x44, 0x80, 0xAE, 0x03, 0x92, 0x88, 0xE5, 0x91, 0x49, 0xE4, 0xB6, 0x39, 0xA0, 0x92,
        0x97, 0x16, 0x72, 0x9C, 0x49, 0xC5, 0x88, 0x4C, 0xE1, 0xE5, 0x2C, 0x3E, 0x65, 0x9E, 0x7E, 0xC4,
        0x17, 0xCB, 0x23, 0x8E, 0x83, 0xE7, 0x71, 0xDB, 0x65, 0xE4, 0x6B, 0x4B, 0xCC, 0xFE, 0xF7, 0x26,
        0xF2, 0xF9, 0xFD, 0xBF, 0xD1, 0x9F, 0xCE, 0xAB, 0xBA, 0x67, 0xDD, 0xE6, 0x17, 0xFD, 0x03, 0xEB,
        0x22, 0x72, 0x46, 0xF9, 0x29, 0x00, 0x00, 0x0D, 0x0A, 0x30, 0x0D, 0x0A, 0x0D, 0x0A
    };

    string piece = string((char*)raw, sizeof(raw));

    ResponseParser p;
    p.set_context_request(new Request(Method::Get, new URI("/")));
    auto res1 = p.parse_shift(piece);
    REQUIRE(res1.state == State::done);
};

