package DBICx::LookupRelation;

use strict;
use warnings;

=head1 NAME

DBICx::LookupRelation - A dbic component for building accessors for a lookup table.

=head1 VERSION

Version 0.01

=cut

our $VERSION = '0.01';

use base qw(DBIx::Class);
use Moose;

use Data::Dumper;
use Smart::Comments -ENV;
use Hash::Merge::Simple qw/merge/;

use DBICx::LookupManager;




=head1 SYNOPSIS

	__PACKAGE__->load_components( qw/+DBICx::LookupRelation/ );
	
	__PACKAGE__->table("user");
	
	__PACKAGE__->add_columns(
	      "user_id",	{ data_type => "integer", is_auto_increment => 1, is_nullable => 0 },
	      "first_name", { data_type => "varchar2", is_nullable => 0, size => 45 },
	      "last_name", { data_type => "varchar2", is_nullable => 0, size => 45 },
	      "permission_type_id", { data_type => "integer", is_nullable => 0 },
	);
	
	__PACKAGE__->set_primary_key("user_id");
	
	__PACKAGE__->add_lookup(  'permission', 'permission_type_id', 'PermissionType' );




=head1 DESCRIPTION

This module generates a few convenient methods for accessing data in a lookup table from an associated belongs_to table on top of L<DBIx::Class>. It plays with L<DBICx::LookupManager>.

What's meant as lookup table is a table containing some terms definition, such as Permission ( 1, 'Administrator'; 2, 'User'; 3, 'Reader' ) which 
is for instance associated with a client table such as User ( id, first_name, last_name, permission_id ).

=head1 EXPORT

	add_lookup




=head1 METHODS

=head2 add_lookup

=over4

=item Arguments : $relation_name, $foreign_key, $lookup_table, \%options?

=item Return Value: no return value

=item Example :
__PACKAGE__->add_lookup(  'permission', 'permission_type_id', 'PermissionType', {
			
			{	name_accessor => 'get_the_permission',
				name_setter   => 'set_the_permission,
				name_checker  => 'is_the_permission'
			} 
				
		});
=back



=head1 GENERATED METHODS

=head2 $relation_name

=over4

=item Arguments : no argument

=item Return Value: value in the related $name_field within the associated lookup table

=item Example : User->find( 1 )->permission() returns the permission set in the lookup table.

=back




=head2 set_$relation_name

=over4

=item Arguments : new_value matching a related value in the $field_name within the lookup table

=item Return Value: no return value

=item Example : 
User->find( 1 )->set_permission( 'Administrator' ) set the id related to the new_value within the lookup table in the L<DBIx::Class::Row> User object.
=back




=head2 is_$relation_name

=over4

=item Arguments : any value in the related $field_name within the lookup table

=item Return Value: boolean

=item Example : 
User->find( 1 )->is_permission( 'Administrator' ) returns a boolean value
=back


=cut




sub add_lookup {
    my ( $class, $relation_name, $foreign_key, $lookup_table, $options ) = @_;
 
    # as it suggests $options is an optionnal argument
   	$options ||= {};
        
    my $defaults = {  
    				name_accessor => ${relation_name},
    				name_setter   => "set_${relation_name}",
    				name_checker  => "is_${relation_name}",
    				field_name    => 'name',
        			}; 
    
    my $params = merge $defaults, $options;
    
    my $field_name	= $params->{field_name};
    
    my $fetch_id_by_name = sub { 
   		my ($self, $name) = @_;
   		DBICx::LookupManager->FETCH_ID_BY_NAME(  $self->result_source->schema, $lookup_table, $field_name, $name);
    };
    
    my $meta = $class->meta or die;
	
        # test if not already present
        foreach my $method ( @$params{qw/name_accessor name_setter name_checker/} ) {
            confess "ERROR: method $method already defined"
                if $meta->get_method($method);
        }

        $meta->add_method( $params->{name_accessor}, sub {
            my $self = shift; # $self isa Row
            my $schema = $self->result_source->schema;
            return DBICx::LookupManager->FETCH_NAME_BY_ID( $schema, $lookup_table, $field_name, $self->get_column($foreign_key) );
        });
        
        
        $meta->add_method( $params->{name_setter}, sub {
            my ($self, $new_name) = @_; 
            my $schema = $self->result_source->schema;
            my $id = $fetch_id_by_name->( $self, $new_name );
            $self->set_column($foreign_key, $id);
        });
        

         $meta->add_method( $params->{name_checker}, sub {
            my ($self, $name) = @_; # $self isa Row
            my $schema = $self->result_source->schema;
            my $id = $self->get_column( $foreign_key );
            return unless defined $id;
            return $fetch_id_by_name->( $self, $name ) eq $id;
        });
}



#sub guess_field_name{
#	my (  $schema, $lookup_table ) = @_;
#	 
#	my @columns = $schema->source( $lookup_table )->columns;
#	my @primary_columns = $schema->source(  $lookup_table )->primary_columns;
#	my @columns_without_primary_keys = grep{ !($_ ~~ @primary_columns) }  @columns;
#	my $guessed_field = undef;
#	
#	# classic lookup table with only two columns
#	if ( @columns == 2 && @columns_without_primary_keys == 1){
#		$guessed_field = $columns_without_primary_keys[0]; 
#	}
#	# lookup table with more than two columns
#	else{
#		foreach my $column ( @columns_without_primary_keys ){
#			my $column_metas = $schema->source( $lookup_table )->column_info( $column );
#			
#			if ( $column_metas->{data_type} =~ /(varchar)/ ){
#				#select the first varchar column 
#				$guessed_field = $column;
#				last;
#			 }
#		}
#	}
#	return $guessed_field;
#}







=head1 AUTHOR

"Karl Forner", C<< <"karl.forner@gmail.com"> >>

=head1 BUGS

Please report any bugs or feature requests to C<bug-dbicx-vocabularies at rt.cpan.org>, or through
the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=DBICx-Vocabularies>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.




=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc DBIx::Class::LookupRelation


You can also look for information at:

=over 4

=item * RT: CPAN's request tracker (report bugs here)

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=DBICx-Vocabularies>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/DBICx-Vocabularies>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/DBICx-Vocabularies>

=item * Search CPAN

L<http://search.cpan.org/dist/DBICx-Vocabularies/>

=back


=head1 ACKNOWLEDGEMENTS


=head1 LICENSE AND COPYRIGHT

Copyright 2012 "".

This program is free software; you can redistribute it and/or modify it
under the terms of either: the GNU General Public License as published
by the Free Software Foundation; or the Artistic License.

See http://dev.perl.org/licenses/ for more information.


=cut

1; # End of DBICx::LookupRelation
