# 2C-cannedstats_pairs.t version 0.01
# A script to run tests on the Lingua::Norms::USF module.
# Checks correct returns of ASSOCIATION data from DB fields for the following:
# FSG Forward strength
# BSG Backward strength
# MSG Mediated strength
# OSG Overlap strength
# MCT Number of mediators
# OCT Number of overlaps
# as well as their values returned from assoc_stat
use strict;
use warnings;

use Test::More tests => 17;
use constant EPS => 1e-9;

BEGIN { use_ok('Lingua::Norms::USF') };

my $usf = Lingua::Norms::USF->new();

my ($word1, $word2, $dat) = (qw/milk cow/);

my %dat_ref = (
	FSG => 0.388,
	BSG => 0.352,
	MSG => 0.0013,
	OSG => 0,
	MCT => 1,
	OCT => 0,
);

# Get stats by method name:
$dat = $usf->forward_strength($word1, $word2);
ok( is_equal($dat, $dat_ref{'FSG'}), "fsg calculated = $dat, should be $dat_ref{'FSG'}");

$dat = $usf->backward_strength($word1, $word2);
ok( is_equal($dat, $dat_ref{'BSG'}), "bsg calculated = $dat, should be $dat_ref{'BSG'}");

$dat = $usf->mediator_strength($word1, $word2);
ok( is_equal($dat, $dat_ref{'MSG'}), "msg calculated = $dat, should be $dat_ref{'MSG'}");

$dat = $usf->overlap_strength($word1, $word2);
ok( is_equal($dat, $dat_ref{'OSG'}), "osg calculated = $dat, should be $dat_ref{'OSG'}");

$dat = $usf->mediator_count($word1, $word2);
ok( is_equal($dat, $dat_ref{'MCT'}), "mct calculated = $dat, should be $dat_ref{'MCT'}");

$dat = $usf->overlap_count($word1, $word2);
ok( is_equal($dat, $dat_ref{'OCT'}), "oct calculated = $dat, should be $dat_ref{'OCT'}");

# Get stats directly from assoc_stat();
foreach my $s(qw/FSG BSG MSG OSG MCT OCT/) {
    $dat = $usf->assoc_stat($word1, $word2, $s);
    ok( is_equal($dat, $dat_ref{$s}), "$s calculated = $dat, should be $dat_ref{$s}");
}

# Get two stats at once using assoc_stat:
# FSG + BSG = .74
#* by word_stat as open array (as default):
my (@ari, $aref, $href) = ();
my $sum_want = .74;
my $sum_got = '';

@ari = $usf->assoc_stat($word1, $word2, [qw/FSG BSG/]);
$sum_got = sum(@ari);
ok( about_equal($sum_got, $sum_want), "FSG + BSG requested = $sum_got, should be $sum_want");

@ari = $usf->assoc_stat($word1, $word2, [qw/FSG BSG/], {ref => 0});
$sum_got = sum(@ari);
ok( about_equal($sum_got, $sum_want), "FSG + BSG requested = $sum_got, should be $sum_want");

$aref = $usf->assoc_stat($word1, $word2, [qw/FSG BSG/], {ref => 1});
$sum_got = sum(@{$aref});
ok( about_equal($sum_got, $sum_want), "FSG + BSG requested = $sum_got, should be $sum_want");

$href = $usf->assoc_stat($word1, $word2, [qw/FSG BSG/], {ref => 2});
$sum_got = sum(values %{$href});
ok( about_equal($sum_got, $sum_want), "FSG + BSG requested = $sum_got, should be $sum_want");

sub sum {
    my $d = 0;
    $d += $_ foreach @_;
    return $d;
}

sub is_equal {
    return 1 if $_[0] == $_[1];
    return 0;
}

sub char_equal {
    return 1 if $_[0] eq $_[1];
    return 0;
}

sub about_equal {
    return 1 if $_[0] + EPS > $_[1] and $_[0] - EPS < $_[1];
    return 0;
}
