package Nexmo::SMS::GetBalance;

use strict;
use warnings;

use LWP::UserAgent;
use JSON::PP;


our $VERSION = '0.01';

my %attrs = (
    server            => 'required',
    username          => 'required',
    password          => 'required',
);

for my $attr ( keys %attrs ) {
    no strict 'refs';
    *{ __PACKAGE__ . '::' . $attr } = sub {
        my ($self,$value) = @_;
        
        my $key = '__' . $attr . '__';
        $self->{$key} = $value if @_ == 2;
        return $self->{$key};
    };
}


sub new {
    my ($class,%param) = @_;
    
    my $self = bless {}, $class;
    
    for my $attr ( keys %attrs ) {
        if ( exists $param{$attr} ) {
            $self->$attr( $param{$attr} );
        }
    }
    
    $self->user_agent(
        LWP::UserAgent->new(
            agent => 'Perl module ' . __PACKAGE__ . ' ' . $VERSION,
        ),
    );
    
    return $self;
}


sub user_agent {
    my ($self,$ua) = @_;
    
    $self->{__ua__} = $ua if @_ == 2;
    return $self->{__ua__};
}


sub get_balance {
    my ($self) = shift;
    
    use Data::Dumper;
    
    my $url = sprintf "%saccount/get-balance/%s/%s",
        $self->server,
        $self->username,
        $self->password;
    
    my $ua = $self->user_agent;
    $ua->default_header( 'Accept' => 'application/json' );
    
    my $response = $ua->get(
        $url,
    );
    
    if ( !$response || !$response->is_success ) {
        return;
    }
    
    my $json  = $response->content;
    my $coder = JSON::PP->new->utf8->pretty->allow_nonref;
    my $perl  = $coder->decode( $json );
    
    return if !$perl || ref $perl ne 'HASH';    
    return $perl->{'value'};
}


1;


__END__
=pod

=head1 NAME

Nexmo::SMS::GetBalance

=head1 VERSION

version 0.03

=head1 SYNOPSIS

This module simplifies sending SMS through the Nexmo API.

    use Nexmo::SMS::GetBalance;

    my $nexmo = Nexmo::SMS::GetBalance->new(
        server   => 'http://test.nexmo.com/sms/json',
        username => 'testuser1',
        password => 'testpasswd2',
    );
        
    my $balance = $sms->get_balance;

=head1 NAME

Nexmo::SMS::GetBalance - Module to ask for the balance for the Nexmo SMS API!

=head1 VERSION

Version 0.01

=head1 METHODS

=head2 new

create a new object

    my $object = Nexmo::SMS::GetBalance->new(
        server   => 'http://test.nexmo.com/sms/json',
        username => 'testuser1',
        password => 'testpasswd2',
    );

This method recognises these parameters:

    server            => 'required',
    username          => 'required',
    password          => 'required',

=head2 user_agent

Getter/setter for the user_agent attribute of the object. By default a new
object of LWP::UserAgent is used, but you can use your own class as long as it
is compatible to LWP::UserAgent.

  $sms->user_agent( MyUserAgent->new );
  my $ua = $sms->user_agent;

=head2 get_balance

This actually calls the Nexmo SMS API. It returns the balance of the account.

   my $balance = $object->get_balance;

=head1 Attributes

These attributes are available for C<Nexmo::SMS::GetBalance> objects. For each
attribute there is a getter/setter:

  $nexmo->server( 'servername' );
  my $server = $nexmo->server;

=over 4

=item * password

=item * server

=item * username

=back

=head1 ACKNOWLEDGEMENTS

=head1 COPYRIGHT & LICENSE

Copyright 2011 Renee Baecker.

This program is released under the following license: artistic_2

=cut

=head1 AUTHOR

Renee Baecker <module@renee-baecker.de>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2011 by Renee Baecker.

This is free software, licensed under:

  The Artistic License 2.0

=cut

