package Statocles::App;
# ABSTRACT: Base role for Statocles applications
$Statocles::App::VERSION = '0.046';
use Statocles::Base 'Role';
use Statocles::Link;
requires 'pages';


has site => (
    is => 'rw',
    isa => InstanceOf['Statocles::Site'],
);


has data => (
    is => 'ro',
    isa => HashRef,
    default => sub { {} },
);


has url_root => (
    is => 'ro',
    isa => Str,
    required => 1,
);


around pages => sub {
    my ( $orig, $self, @args ) = @_;
    my @pages = $self->$orig( @args );

    # Add the url_root
    my $url_root = $self->url_root;
    for my $page ( @pages ) {
        next if $page->path =~ /^$url_root/;
        $page->path( join "/", $url_root, $page->path );
    }

    return @pages;
};


sub url {
    my ( $self, $url ) = @_;
    $url =~ s{/index[.]html$}{/};
    return join "/", $self->url_root, $url;
}


sub link {
    my ( $self, %args ) = @_;
    my $url_root = $self->url_root;
    if ( $args{href} !~ /^$url_root/ ) {
        $args{href} = $self->url( $args{href} );
    }
    return Statocles::Link->new( %args );
}

1;

__END__

=pod

=head1 NAME

Statocles::App - Base role for Statocles applications

=head1 VERSION

version 0.046

=head1 DESCRIPTION

A Statocles App turns L<documents|Statocles::Documents> into a set of
L<pages|Statocles::Pages> that can then be written to the filesystem (or served
directly, if desired).

=head1 ATTRIBUTES

=head2 site

The site this app is part of.

=head2 data

A hash of arbitrary data available to theme templates. This is a good place to
put extra structured data like social network links or make easy customizations
to themes like header image URLs.

=head2 url_root

The URL root of this application. All pages from this app will be under this
root. Use this to ensure two apps do not try to write the same path.

=head1 METHODS

=head2 pages()

Get the pages for this app. Must return a list of L<Statocles::Page> objects, with
the index page (the page intended to be the entry page) first.

=head2 url( $url )

Get a URL to a page in this application. Prepends the L</url_root> if necessary. Strips
"index.html" if possible.

=head2 link( %args )

Create a link to a page in this application. C<%args> are attributes to be given to
L<Statocles::Link> constructor.

=head1 AUTHOR

Doug Bell <preaction@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by Doug Bell.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
