# Copyright (C) 2005-2007, The Perl Foundation.
# $Id: luastring.pir 17146 2007-02-22 14:37:26Z fperrad $

=head1 NAME

lib/luastring.pir - Lua String Library

=head1 DESCRIPTION

This library provides generic functions for string manipulation, such as
finding and extracting substrings, and pattern matching. When indexing a
string in Lua, the first character is at position 1 (not at 0, as in C).
Indices are allowed to be negative and are interpreted as indexing backwards,
from the end of the string. Thus, the last character is at position I<-1>,
and so on.

The string library provides all its functions inside the table C<string>.
It also sets a metatable for strings where the C<__index> field points to
the metatable itself. Therefore, you can use the string functions in
object-oriented style. For instance, C<string.byte(s, i)> can be written as
C<s:byte(i)>.

See "Lua 5.1 Reference Manual", section 5.4 "String Manipulation".

=head2 Functions

=over 4

=cut

.HLL 'Lua', 'lua_group'

.sub 'init_string' :load :anon

    load_bytecode 'languages/lua/lib/luabasic.pbc'

#    print "init Lua String\n"

    .local pmc _lua__GLOBAL
    _lua__GLOBAL = global '_G'
    new $P1, .LuaString

    .local pmc _string
    new _string, .LuaTable
    set $P1, 'string'
    _lua__GLOBAL[$P1] = _string

    .const .Sub _string_byte = '_string_byte'
    set $P1, 'byte'
    _string[$P1] = _string_byte

    .const .Sub _string_char = '_string_char'
    set $P1, 'char'
    _string[$P1] = _string_char

    .const .Sub _string_dump = '_string_dump'
    set $P1, 'dump'
    _string[$P1] = _string_dump

    .const .Sub _string_find = '_string_find'
    set $P1, 'find'
    _string[$P1] = _string_find

    .const .Sub _string_format = '_string_format'
    set $P1, 'format'
    _string[$P1] = _string_format

    .const .Sub _string_gmatch = '_string_gmatch'
    set $P1, 'gmatch'
    _string[$P1] = _string_gmatch

    # LUA_COMPAT_GFIND
    set $P1, 'gfind'
    _string[$P1] = _string_gmatch

    .const .Sub _string_gsub = '_string_gsub'
    set $P1, 'gsub'
    _string[$P1] = _string_gsub

    .const .Sub _string_len = '_string_len'
    set $P1, 'len'
    _string[$P1] = _string_len

    .const .Sub _string_lower = '_string_lower'
    set $P1, 'lower'
    _string[$P1] = _string_lower

    .const .Sub _string_match = '_string_match'
    set $P1, 'match'
    _string[$P1] = _string_match

    .const .Sub _string_rep = '_string_rep'
    set $P1, 'rep'
    _string[$P1] = _string_rep

    .const .Sub _string_reverse = '_string_reverse'
    set $P1, 'reverse'
    _string[$P1] = _string_reverse

    .const .Sub _string_sub = '_string_sub'
    set $P1, 'sub'
    _string[$P1] = _string_sub

    .const .Sub _string_upper = '_string_upper'
    set $P1, 'upper'
    _string[$P1] = _string_upper


    .local pmc _lua_mt_string
    new _lua_mt_string, .LuaTable
    global 'mt_string' = _lua_mt_string
    set $P1, '__index'
    _lua_mt_string[$P1] = _string


    load_bytecode 'languages/lua/lib/luaregex.pbc'

.end


.sub 'posrelat' :anon
    .param int pos
    .param int len
    if pos >= 0 goto L0
    pos = len + pos
    inc pos
L0:
    .return (pos)
.end


=item C<string.byte (s [, i [, j]])>

Returns the internal numerical codes of the characters C<s[i]>, C<s[i+1]>,...,
C<s[j]>. The default value for C<i> is 1; the default value for C<j> is C<i>.

Note that numerical codes are not necessarily portable across platforms.

=cut

.sub '_string_byte' :anon
    .param pmc s :optional
    .param pmc i :optional
    .param pmc j :optional
    $S0 = checkstring(s)
    .local int l
    l = length $S0
    .local int posi
    posi = optint(i, 1)
    posi = posrelat(posi, l)
    .local int pose
    pose = optint(j, posi)
    pose = posrelat(pose, l)
    unless posi <= 0 goto L0
    posi = 1
L0:
    unless pose > l goto L1
    pose = l
L1:
    unless posi > pose goto L2
    .return ()
L2:
    .local int n
    n = pose - posi
    inc n
    .local pmc ret
    new ret, .Array
    set ret, n
    .local int i
    i = 0
L3:
    unless i < n goto L4
    $I0 = posi + i
    dec $I0
    $I1 = ord $S0, $I0
    new $P0, .LuaNumber
    set $P0, $I1
    ret[i] = $P0
    inc i
    goto L3
L4:
    .return (ret :flat)
.end


=item C<string.char (i1, i2, ...)>

Receives 0 or more integers. Returns a string with length equal to the number
of arguments, in which each character has the internal numerical code equal
to its correspondent argument.

Note that numerical codes are not necessarily portable across platforms.

=cut

.sub '_string_char' :anon
    .param pmc argv :slurpy
    .local pmc ret
    .local int argc
    .local int i
    .local int c
    .local pmc curr
    .local string b
    .local string s
    b = ''
    argc = argv
    i = 0
L1:
    if i >= argc goto L2
    curr = argv[i]
    c = checknumber(curr)
    s = chr c
    b = concat b, s
    inc i
    goto L1
L2:
    new ret, .LuaString
    set ret, b
    .return (ret)
.end


=item C<string.dump (function)>

Returns a string containing a binary representation of the given function,
so that a later C<loadstring> on this string returns a copy of the function.
C<function> must be a Lua function without upvalues.

NOT YET IMPLEMENTED.

=cut

.sub '_string_dump' :anon
    .param pmc function :optional
    checktype(function, 'function')
    not_implemented()
.end


=item C<string.find (s, pattern [, init [, plain]])>

Looks for the first I<match> of C<pattern> in the string C<s>. If it finds a match,
then C<find> returns the indices of C<s> where this occurrence starts and ends;
otherwise, it returns B<nil>. A third, optional numerical argument C<init>
specifies where to start the search; its default value is 1 and may be negative.
A value of B<true> as a fourth, optional argument C<plain> turns off the
pattern matching facilities, so the function does a plain "find substring"
operation, with no characters in C<pattern> being considered "magic". Note
that if C<plain> is given, then C<init> must be given as well.

If the pattern has captures, then in a successful match the captured values
are also returned, after the two indices.

STILL INCOMPLETE (see F<languages/lua/lib/luaregex.pir>).

=cut

.sub '_string_find' :anon
    .param pmc argv :slurpy
    .return str_find_aux(1, argv :flat)
.end

.sub 'str_find_aux' :anon
    .param int find
    .param pmc s :optional
    .param pmc pattern :optional
    .param pmc init :optional
    .param pmc plain :optional
    $S1 = checkstring(s)
    $I1 = length $S1
    $S2 = checkstring(pattern)
    $I2 = length $S2
    $I3 = optint(init, 1)
    $I3 = posrelat($I3, $I1)
    dec $I3
    unless $I3 < 0 goto L1
    $I3 = 0
    goto L2
L1:
    unless $I3 > $I1 goto L2
    $I3 = $I1
L2:
    $S1 = substr $S1, $I3
    unless find goto L3
    if null plain goto L4
    $I0 = istrue plain
    if $I0 goto L5
L4:
    .const string specials = "^$*+?.([%-"
    $P0 = split '', specials
L6:
    $I0 = $P0
    unless $I0 goto L5
    $S0 = shift $P0
    $I0 = index $S2, $S0
    if $I0 >= 0 goto L3
    goto L6
L5:
    # do a plain search
    .local int idx
    idx = index $S1, $S2
    if idx < 0 goto L7
    .local pmc start
    .local pmc end
    new start, .LuaNumber
    $I0 = $I3 + idx
    inc $I0
    set start, $I0
    new end, .LuaNumber
    $I0 = $I3 + idx
    $I0 += $I2
    set end, $I0
    .return (start, end)
L3:
    .local pmc regex_comp
    regex_comp = compreg 'PGE::LuaRegex'
    .local pmc rulesub
    rulesub = regex_comp($S2)
    .local pmc match
    match = rulesub($S1)
    unless match goto L7
    unless find goto L8
    .local pmc start
    .local pmc end
    new start, .LuaNumber
    $I0 = match.'from'()
    $I0 += $I3
    inc $I0
    set start, $I0
    new end, .LuaNumber
    $I0 = match.'to'()
    $I0 += $I3
    set end, $I0
    $P0 = captures(match, 0)
    .return (start, end, $P0 :flat)
L8:
    $P0 = captures(match, 1)
    .return ($P0 :flat)
L7:
    # not found
    .local pmc ret
    new ret, .LuaNil
    .return (ret)
.end

.sub 'captures' :anon
    .param pmc match
    .param int whole
    .local pmc ret
    new ret, .Array
    .local pmc capts
    capts = match.'get_array'()
    if null capts goto L1
    $I1 = capts
    set ret, $I1
    $I0 = 0
L2:
    unless $I0 < $I1 goto L3
    $P0 = capts[$I0]
    $S0 = $P0.'text'()
    new $P1, .LuaString
    set $P1, $S0
    ret[$I0] = $P1
    inc $I0
    goto L2
L3:
    .return (ret)
L1:
    unless whole == 1 goto L4
    set ret, 1
    $S0 = match.'text'()
    new $P1, .LuaString
    set $P1, $S0
    ret[0] = $P1
L4:
    .return (ret)
.end

=item C<string.format (formatstring, e1, e2, ...)>

Returns a formatted version of its variable number of arguments following the
description given in its first argument (which must be a string). The format
string follows the same rules as the C<printf> family of standard C functions.
The only differences are that the options/modifiers C<*>, C<l>, C<L>, C<n>,
C<p>, and C<h> are not supported, and there is an extra option, C<q>. The C<q>
option formats a string in a form suitable to be safely read back by the Lua
interpreter: The string is written between double quotes, and all double
quotes, newlines, and backslashes in the string are correctly escaped when
written. For instance, the call

    string.format(%q, a string with "quotes" and \n new line)

will produce the string:

    "a string with \"quotes\" and \
     new line"

The options C<c>, C<d>, C<E>, C<e>, C<f>, C<g>, C<G>, C<i>, C<o>, C<u>, C<X>,
and C<x> all expect a number as argument, whereas C<q> and C<s> expect a string.

This function does not accept string values containing embedded zeros.

=cut

.sub '_string_format' :anon
    .param pmc formatstring :optional
    .param pmc argv :slurpy
    .local string strfrmt
    .local string b
    .local int arg
    .local int idx
    strfrmt = checkstring(formatstring)
    $I1 = length strfrmt
    b = ''
    arg = 0
    new $P1, .Array
    set $P1, 1
    idx = 0
L1:
    unless idx < $I1 goto L2
    $S0 = substr strfrmt, idx, 1
    unless $S0 != '%' goto L3
    b .= $S0
    inc idx
    goto L1
L3:
    inc idx
    $S0 = substr strfrmt, idx, 1
    unless $S0 == '%' goto L4
    b .= $S0
    inc idx
L4:
    .local string form
    .local string buff
    $P0 = argv[arg]
    inc arg
    (idx, form) = scanformat(strfrmt, idx)
    $S0 = substr strfrmt, idx, 1
    unless $S0 == 'c' goto L5
    $I0 = checknumber($P0)
    $P1[0] = $I0
    buff = sprintf form, $P1
    goto L6
L5:
    $I0 = index 'diouxX', $S0
    unless $I0 >= 0 goto L7
    $I0 = checknumber($P0)
    $P1[0] = $I0
    buff = sprintf form, $P1
    goto L6
L7:
    $I0 = index 'eEfgG', $S0
    unless $I0 >= 0 goto L8
    $N0 = checknumber($P0)
    $P1[0] = $N0
    buff = sprintf form, $P1
    goto L6
L8:
    unless $S0 == 'q' goto L9
    $S0 = checkstring($P0)
    buff = quoted($S0)
    goto L6
L9:
    unless $S0 == 's' goto L10
    buff = checkstring($P0)
    $I0 = index form, '.'
    if $I0 >= 0 goto L11
    $I0 = length $S0
    unless $I0 >= 100 goto L11
    goto L6
L11:
    $P1[0] = $P0
    buff = sprintf form, $P1
    goto L6
L10:
    $S1 = "invalid option '%" . $S0
    $S1 .= "' to 'format'"
    error($S1)
L6:
    b .= buff
    inc idx
    goto L1
L2:
    .local pmc ret
    new ret, .LuaString
    set ret, b
    .return (ret)
.end

.const string digits = '0123456789'

.sub 'scanformat' :anon
    .param string strfrmt
    .param int start
    .const string flags = '-+ #0'
    .local int idx
    $I1 = length strfrmt
    idx = start
L1:
    unless idx < $I1 goto L2
    $S0 = substr strfrmt, idx, 1
    $I0 = index flags, $S0
    unless $I0 >= 0 goto L2
    inc idx
    goto L1
L2:
    $I0 = idx - start
    unless $I0 > 5 goto L3
    error("invalid format (repeated flags)")
L3:
    $S0 = substr strfrmt, idx, 1
    $I0 = index digits, $S0
    unless $I0 >= 0 goto L4
    inc idx
L4:
    $S0 = substr strfrmt, idx, 1
    $I0 = index digits, $S0
    unless $I0 >= 0 goto L5
    inc idx
L5:
    unless $S0 == '.' goto L6
    inc idx
    $S0 = substr strfrmt, idx, 1
    $I0 = index digits, $S0
    unless $I0 >= 0 goto L7
    inc idx
L7:
    $S0 = substr strfrmt, idx, 1
    $I0 = index digits, $S0
    unless $I0 >= 0 goto L6
    inc idx
L6:
    $S0 = substr strfrmt, idx, 1
    $I0 = index digits, $S0
    unless $I0 >= 0 goto L8
    error("invalid format (width or precision too long)")
L8:
    .local string form
    $I0 = idx - start
    inc $I0
    $S0 = substr strfrmt, start, $I0
    form = '%' . $S0
    .return (idx, form)
.end

.sub 'quoted' :anon
    .param string s
    .local string b
    .local int idx
    $I1 = length s
    b = '"'
    idx = 0
L1:
    unless idx < $I1 goto L2
    $S0 = substr s, idx, 1
    inc idx
    $I0 = index "\"\\\n", $S0
    unless $I0 >= 0 goto L3
    b .= "\\"
    b .= $S0
    goto L1
L3:
    unless $S0 == "\r" goto L4
    b .= "\\r"
    goto L1
L4:
    unless $S0 == "\0" goto L5
    b .= "\\000"
    goto L1
L5:
    b .= $S0
    goto L1
L2:
    b .= '"'
    .return (b)
.end

=item C<string.gmatch (s, pattern)>

Returns an iterator function that, each time it is called, returns the next
captures from pattern C<pattern> over string C<s>.

If C<pattern> specifies no captures, then the whole match is produced in each
call.

As an example, the following loop

    s = "hello world from Lua"
    for w in string.gmatch(s, "%a+") do
        print(w)
    end

will iterate over all the words from string C<s>, printing one per line. The
next example collects all pairs C<key=value> from the given string into a
table:

    t = {}
    s = "from=world, to=Lua"
    for k, v in string.gmatch(s, "(%w+)=(%w+)") do
        t[k] = v
    end

STILL INCOMPLETE (see F<languages/lua/lib/luaregex.pir>).

=cut

.sub '_string_gmatch' :anon :lex
    .param pmc s :optional
    .param pmc pattern :optional
    .local pmc ret
    checkstring(s)
    $S2 = checkstring(pattern)
    .local pmc regex_comp
    regex_comp = compreg 'PGE::LuaRegex'
    .local pmc rulesub
    rulesub = regex_comp($S2)
    .lex 'upvar_rulesub', rulesub
    .lex 'upvar_s', s
    .const .Sub gmatch_aux = 'gmatch_aux'
    ret = newclosure gmatch_aux
    .return (ret)
.end

.sub 'gmatch_aux' :anon :lex :outer(_string_gmatch)
    .local pmc s
    .local pmc rulesub
    .local pmc match
    .local pmc ret
    s = find_lex 'upvar_s'
    rulesub = find_lex 'upvar_rulesub'
    $S1 = s
    match = rulesub($S1)
    unless match goto L1
    $I0 = match.'to'()
    $S1 = substr $S1, $I0
    set s, $S1
    $P0 = captures(match, 1)
    .return ($P0 :flat)
L1:
    .local pmc ret
    new ret, .LuaNil
    .return (ret)
.end

=item C<string.gsub (s, pat, repl [, n])>

Returns a copy of C<s> in which all occurrences of the pattern C<pat> have
been replaced by a replacement string specified by C<repl>. C<gsub> also
returns, as a second value, the total number of substitutions made.

If C<repl> is a string, then its value is used for replacement. Any sequence
in C<repl> of the form %I<n>, with I<n> between 1 and 9, stands for the value
of the I<n>-th captured substring.

If C<repl> is a function, then this function is called every time a match
occurs, with all captured substrings passed as arguments, in order; if the
pattern specifies no captures, then the whole match is passed as a sole
argument. If the value returned by this function is a string, then it is used
as the replacement string; otherwise, the replacement string is the empty
string.

The optional last parameter C<n> limits the maximum number of substitutions
to occur. For instance, when C<n> is 1 only the first occurrence of C<pat>
is replaced.

STILL INCOMPLETE (see F<languages/lua/lib/luaregex.pir>).

=cut

.sub '_string_gsub' :anon
    .param pmc s :optional
    .param pmc pat :optional
    .param pmc repl :optional
    .param pmc max :optional
    .local string src
    src = checkstring(s)
    $I1 = length src
    $S2 = checkstring(pat)
    $I0 = $I1 + 1
    $I4 = optint(max, $I0)
    .local int anchor
    anchor = 0
    $S0 = substr $S2, 0, 1
    unless $S0 == '^' goto L1
    anchor = 1
L1:
    .local int n
    n = 0
    .local pmc regex_comp
    regex_comp = compreg 'PGE::LuaRegex'
    .local pmc rulesub
    rulesub = regex_comp($S2)
    .local pmc b
    new b, .LuaString
L2:
    unless n < $I4 goto L3
    .local pmc match
    match = rulesub(src)
    unless match goto L3
    inc n
    add_value(b, src, match, repl)
    $I0 = match.'to'()
    src = substr src, $I0
    if anchor goto L3
    goto L2
L3:
    $S0 = b
    $S0 .= src
    set b, $S0
    new $P0, .LuaNumber
    set $P0, n
    .return (b, $P0)
.end

.sub 'add_value' :anon
    .param pmc b
    .param string s
    .param pmc match
    .param pmc repl
    $I0 = isa repl, 'LuaNumber'
    unless $I0 goto L1
    $P0 = repl.'tostring'()
    .return add_s(b, s, match, $P0)
L1:
    $I0 = isa repl, 'LuaString'
    unless $I0 goto L2
    .return add_s(b, s, match, repl)
L2:
    $I0 = isa repl, 'LuaClosure'
    if $I0 goto L3
    $I0 = isa repl, 'LuaFunction'
    if $I0 goto L3
    goto L4
L3:
    $P0 = captures(match, 1)
    ($P1) = repl($P0 :flat)
    goto L5
L4:
    $I0 = isa repl, 'LuaTable'
    unless $I0 goto L6
    $S0 = onecapture(match, 0)
    new $P0, .LuaString
    set $P0, $S0
    $P1 = repl[$P0]
    goto L5
L6:
    error("string/function/table expected")
L5:
    if $P1 goto L7  # nil or false?
    # keep original text
    $S1 = b
    $I0 = match.'to'()
    $S0 = substr s, 0, $I0
    $S1 .= $S0
    set b, $S1
    .return ()
L7:
    $I0 = isa $P1, 'LuaString'
    if $I0 goto L8
    $I0 = isa $P1, 'LuaNumber'
    if $I0 goto L8
    $S0 = "invalid replacement value (a "
    $S1 = typeof $P1
    $S0 .= $S1
    $S0 .= ")"
    error($S0)
L8:
    $S1 = b
    $I0 = match.'from'()
    $S0 = substr s, 0, $I0
    $S1 .= $S0
    $S0 = $P1
    $S1 .= $S0
    set b, $S1
    .return ()
.end

.sub 'add_s' :anon
    .param pmc b
    .param string s
    .param pmc match
    .param pmc repl
    $S1 = b
    $I0 = match.'from'()
    $S0 = substr s, 0, $I0
    $S1 .= $S0
    $S4 = repl
    $I4 = length $S4
    .local int i
    i = 0
L1:
    unless i < $I4 goto L2
    $S0 = substr $S4, i, 1
    if $S0 != '%' goto L3
    inc i
    $S0 = substr $S4, i, 1
    $I0 = index digits, $S0
    if $I0 < 0 goto L3
    unless $S0 == '0' goto L4
    $S0 = match.'text'()
    goto L3
L4:
    dec $I0
    $S0 = onecapture(match, $I0)
L3:
    $S1 .= $S0
    inc i
    goto L1
L2:
    set b, $S1
.end

.sub 'onecapture' :anon
    .param pmc match
    .param int i
    push_eh _handler
    $P0 = match.'get_array'()
    $P1 = $P0[i]
    $S0 = $P1.'text'()
    .return ($S0)
_handler:
    error("invalid capture index")
.end

=item C<string.len (s)>

Receives a string and returns its length. The empty string C<""> has length 0.
Embedded zeros are counted, so C<"a\000b\000c"> has length 5.

=cut

.sub '_string_len' :anon
    .param pmc s :optional
    .local pmc ret
    $S0 = checkstring(s)
    $I0 = length $S0
    new ret, .LuaNumber
    set ret, $I0
    .return (ret)
.end


=item C<string.lower (s)>

Receives a string and returns a copy of that string with all uppercase letters
changed to lowercase. All other characters are left unchanged. The definition
of what is an uppercase letter depends on the current locale.

=cut

.sub '_string_lower' :anon
    .param pmc s :optional
    .local pmc ret
    $S0 = checkstring(s)
    downcase $S0
    new ret, .LuaString
    set ret, $S0
    .return (ret)
.end


=item C<string.match (s, pattern [, init])>

Looks for the first I<match> of C<pattern> in the string C<s>. If it finds
one, then C<match> returns the captures from the pattern; otherwise it
returns B<nil>. If C<pattern> specifies no captures, then the whole match is
returned. A third, optional numerical argument C<init> specifies where to
start the search; its default value is 1 and may be negative.

STILL INCOMPLETE (see F<languages/lua/lib/luaregex.pir>).

=cut

.sub '_string_match' :anon
    .param pmc argv :slurpy
    .return str_find_aux(0, argv :flat)
.end


=item C<string.rep (s, n)>

Returns a string that is the concatenation of C<n> copies of the string C<s>.

=cut

.sub '_string_rep' :anon
    .param pmc s :optional
    .param pmc n
    .local pmc ret
    $S0 = checkstring(s)
    $I0 = checknumber(n)
    if $I0 >= 0 goto L0
    $I0 = 0
L0:
    $S1 = repeat $S0, $I0
    new ret, .LuaString
    set ret, $S1
    .return (ret)
.end


=item C<string.reverse (s)>

Returns a string that is the string C<s> reversed.

=cut

.sub '_string_reverse' :anon
    .param pmc s :optional
    .local pmc ret
    $S0 = checkstring(s)
    $I0 = 0
    $I1 = length $S0
    dec $I1
    $P0 = split '', $S0
L1:
    unless $I0 < $I1 goto L2
    $S2 = $P0[$I0]
    $S3 = $P0[$I1]
    $P0[$I0] = $S3
    $P0[$I1] = $S2
    inc $I0
    dec $I1
    goto L1
L2:
    $S1 = join '', $P0
    new ret, .LuaString
    set ret, $S1
    .return (ret)
.end


=item C<string.sub (s, i [, j])>

Returns the substring of C<s> that starts at C<i> and continues until C<j>;
C<i> and C<j> may be negative. If C<j> is absent, then it is assumed to be
equal to -1 (which is the same as the string length). In particular, the call
C<string.sub(s,1,j)> returns a prefix of C<s> with length C<j>, and
C<string.sub(s, -i)> returns a suffix of C<s> with length C<i>.

=cut

.sub '_string_sub' :anon
    .param pmc s :optional
    .param pmc i :optional
    .param pmc j :optional
    .local pmc ret
    $S0 = checkstring(s)
    $I0 = length $S0
    $I1 = checknumber(i)
    $I1 = posrelat($I1, $I0)
    $I2 = optint(j, -1)
    $I2 = posrelat($I2, $I0)
    unless $I1 < 1 goto L0
    $I1 = 1
L0:
    unless $I2 > $I0 goto L1
    $I2 = $I0
L1:
    unless $I1 <= $I2 goto L2
    dec $I1
    $I2 -= $I1
    $S1 = substr $S0, $I1, $I2
    goto L3
L2:
    $S1 = ''
L3:
    new ret, .LuaString
    set ret, $S1
    .return (ret)
.end


=item C<string.upper (s)>

Receives a string and returns a copy of that string with all lowercase letters
changed to uppercase. All other characters are left unchanged. The definition
of what is a lowercase letter depends on the current locale.

=cut

.sub '_string_upper' :anon
    .param pmc s :optional
    .local pmc ret
    $S0 = checkstring(s)
    upcase $S0
    new ret, .LuaString
    set ret, $S0
    .return (ret)
.end

=back

=head1 AUTHORS

Francois Perrad

=cut

