package Log::Any::IfLOG;

our $DATE = '2015-04-02'; # DATE
our $VERSION = '0.03'; # VERSION

sub import {
    my $self = shift;

    my $caller = caller();

    my $log_enabled = $ENV{LOG} || $ENV{TRACE} || $ENV{DEBUG} ||
        $ENV{VERBOSE} || $ENV{QUIET} || $ENV{LOG_LEVEL};

    if ($log_enabled) {
        require Log::Any;
        Log::Any->_export_to_caller($caller, @_);
    } else {
        my $saw_log_param = grep { $_ eq '$log' } @_;
        if ($saw_log_param) {
            *{"$caller\::log"} = \Object::Dumb->new;
        }
    }
}

package
    Object::Dumb;
sub new { my $o = ""; bless \$o, shift }
sub AUTOLOAD { 0 }

1;
# ABSTRACT: Load Log::Any only if log-related environment variables are set

__END__

=pod

=encoding UTF-8

=head1 NAME

Log::Any::IfLOG - Load Log::Any only if log-related environment variables are set

=head1 VERSION

This document describes version 0.03 of Log::Any::IfLOG (from Perl distribution Log-Any-IfLOG), released on 2015-04-02.

=head1 SYNOPSIS

 use Log::Any::IfLOG '$log';

=head1 DESCRIPTION

This module will load L<Log::Any> only when C<LOG> environment variable is true
(or C<TRACE>, or C<DEBUG>, or C<VERBOSE>, or C<QUIET>, or C<LOG_LEVEL>; these
variables are used by L<Perinci::CmdLine>). Otherwise, the module is not loaded
and if user imports C<$log>, a dumb object will be returned instead that will
accept any method but return false.

This is a quick-hack solution to avoid the cost of loading Log::Any under
"normal condition" (when log-enabling variables/flags are not set to true).
Since Log::Any 1.00, startup overhead increases to about 7ms on my PC (from
under 1ms for the previous version). Since I want to keep startup overhead of
CLI apps under 50ms (see L<Perinci::CmdLine::Lite>) to keep tab completion from
getting a noticeable lag, every millisecond counts.

=for Pod::Coverage ^(.+)$

=head1 ENVIRONMENT

=head2 LOG => bool

If set to true, will load Log::Any as usual. Otherwise, won't load Log::Any and
will return a dumb object in C<$log> instead.

=head2 TRACE => bool

=head2 DEBUG => bool

=head2 VERBOSE => bool

=head2 QUIET => bool

=head2 LOG_LEVEL => str

These variables are used by L<Perinci::CmdLine> as a shortcut to set log level.
The setting of these variables indicate that user wants to see some logging, so
Log::Any will be loaded under the presence of these variables.

=head1 SEE ALSO

L<Log::Any>

L<http://github.com/dagolden/Log-Any/issues/24>

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Log-Any-IfLOG>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Log-Any-IfLOG>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Log-Any-IfLOG>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
