package Bencher::Scenario::Serializers;

our $DATE = '2016-01-20'; # DATE
our $VERSION = '0.09'; # VERSION

use 5.010001;
use strict;
use utf8;
use warnings;

our $scenario = {
    summary => 'Benchmark Perl data serialization modules',
    participants => [
        {
            tags => ['json', 'serialize'],
            module => 'JSON::PP',
            function => 'encode_json',
            code_template => 'state $json = JSON::PP->new->allow_nonref; $json->encode(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            module => 'JSON::PP',
            function => 'decode_json',
            code_template => 'state $json = JSON::PP->new->allow_nonref; $json->decode(<data>)',
        },
        {
            tags => ['json', 'serialize'],
            module => 'JSON::XS',
            function => 'encode_json',
            code_template => 'state $json = JSON::XS->new->allow_nonref; $json->encode(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            module => 'JSON::XS',
            function => 'decode_json',
            code_template => 'state $json = JSON::XS->new->allow_nonref; $json->decode(<data>)',
        },
        {
            tags => ['json', 'serialize'],
            module => 'Cpanel::JSON::XS',
            function => 'encode_json',
            code_template => 'state $json = Cpanel::JSON::XS->new->allow_nonref; $json->encode(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            module => 'Cpanel::JSON::XS',
            function => 'decode_json',
            code_template => 'state $json = Cpanel::JSON::XS->new->allow_nonref; $json->decode(<data>)',
        },
        {
            tags => ['json', 'serialize'],
            module => 'JSON::MaybeXS',
            function => 'encode_json',
            code_template => 'state $json = JSON::MaybeXS->new(allow_nonref=>1); $json->encode(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            module => 'JSON::MaybeXS',
            function => 'decode_json',
            code_template => 'state $json = JSON::MaybeXS->new(allow_nonref=>1); $json->decode(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            fcall_template => 'JSON::Decode::Regexp::from_json(<data>)',
        },
        {
            tags => ['json', 'deserialize', 'cant_handle_scalar'],
            fcall_template => 'JSON::Decode::Marpa::from_json(<data>)',
        },
        {
            name => 'Pegex::JSON',
            tags => ['json', 'deserialize'],
            module => 'Pegex::JSON',
            code_template => 'state $obj = Pegex::JSON->new; $obj->load(<data>);',
        },
        {
            tags => ['json', 'serialize'],
            fcall_template => 'JSON::Create::create_json(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            fcall_template => 'JSON::Parse::parse_json(<data>)',
        },

        {
            tags => ['yaml', 'serialize'],
            fcall_template => 'YAML::Old::Dump(<data>)',
        },
        {
            tags => ['yaml', 'deserialize'],
            fcall_template => 'YAML::Old::Load(<data>)',
        },
        {
            tags => ['yaml', 'serialize'],
            fcall_template => 'YAML::Syck::Dump(<data>)',
        },
        {
            tags => ['yaml', 'deserialize'],
            fcall_template => 'YAML::Syck::Load(<data>)',
        },
        {
            tags => ['yaml', 'serialize'],
            fcall_template => 'YAML::XS::Dump(<data>)',
        },
        {
            tags => ['yaml', 'deserialize'],
            fcall_template => 'YAML::XS::Load(<data>)',
        },

        {
            tags => ['binary', 'serialize', 'cant_handle_scalar'],
            fcall_template => 'Storable::freeze(<data>)',
        },
        {
            tags => ['binary', 'deserialize', 'cant_handle_scalar'],
            fcall_template => 'Storable::thaw(<data>)',
        },

        {
            tags => ['binary', 'serialize'],
            fcall_template => 'Sereal::encode_sereal(<data>)',
        },
        {
            tags => ['binary', 'deserialize'],
            fcall_template => 'Sereal::decode_sereal(<data>)',
        },

        {
            name => 'Data::MessagePack::pack',
            tags => ['binary', 'serialize'],
            module => 'Data::MessagePack',
            function => 'pack',
            code_template => 'state $obj = Data::MessagePack->new; $obj->pack(<data>)',
        },
        {
            name => 'Data::MessagePack::unpack',
            tags => ['binary', 'deserialize'],
            module => 'Data::MessagePack',
            function => 'unpack',
            code_template => 'state $obj = Data::MessagePack->new; $obj->unpack(<data>)',
        },
    ],

    # XXX: add more datasets (larger data, etc)
    datasets => [
        {
            name => 'undef',
            summary => 'undef',
            args => {data=>undef},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },
        {
            name => 'num',
            summary => 'A single number (-1.23)',
            args => {data=>-1.23},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },
        {
            name => 'str1k',
            summary => 'A non-Unicode string 1024 characters/bytes long',
            args => {data=>'a' x 1024},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },
        {
            name => 'str1k',
            summary => 'A Unicode string 1024 characters (3072-bytes) long',
            args => {data=>'我爱你爱你一辈子' x 128},
            tags => ['serialize', 'unicode'],
            include_participant_tags => ['serialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },

        {
            name => 'array_int_10',
            summary => 'A 10-element array containing ints',
            args => {data=>[1..10]},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'array_int_100',
            summary => 'A 100-element array containing ints',
            args => {data=>[1..100]},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'array_int_1000',
            summary => 'A 1000-element array containing ints',
            args => {data=>[1..1000]},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'array_str1k_10',
            summary => 'A 10-element array containing 1024-characters/bytes-long non-Unicode strings',
            args => {data=>[('a' x 1024) x 10]},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'array_ustr1k_10',
            summary => 'A 10-element array containing 1024-characters-long (3072-bytes long) Unicode strings',
            args => {data=>[('我爱你爱你一辈子' x 128) x 10]},
            tags => ['serialize', 'json'],
            include_participant_tags => ['serialize'],
        },

        {
            name => 'hash_int_10',
            summary => 'A 10-key hash {1=>0, ..., 10=>0}',
            args => {data=>{map {$_=>0} 1..10}},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'hash_int_100',
            summary => 'A 100-key hash {1=>0, ..., 100=>0}',
            args => {data=>{map {$_=>0} 1..100}},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'hash_int_1000',
            summary => 'A 1000-key hash {1=>0, ..., 1000=>0}',
            args => {data=>{map {$_=>0} 1..1000}},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },

        {
            name => 'json:null',
            summary => 'null',
            args => {data=>'null'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },
        {
            name => 'json:num',
            summary => 'A single number (-1.23)',
            args => {data=>-1.23},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },
        {
            name => 'json:str1k',
            summary => 'A non-Unicode (ASCII) string 1024-characters/bytes long',
            args => {data=>'"' . ('a' x 1024) . '"'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },

        {
            name => 'json:array_int_10',
            summary => 'A 10-element array containing ints',
            args => {data=>'['.join(',',1..10).']'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
        {
            name => 'json:array_int_100',
            summary => 'A 10-element array containing ints',
            args => {data=>'['.join(',',1..100).']'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
        {
            name => 'json:array_int_1000',
            summary => 'A 1000-element array containing ints',
            args => {data=>'['.join(',',1..1000).']'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
        {
            name => 'json:array_str1k_10',
            summary => 'A 10-element array containing 1024-characters/bytes-long non-Unicode strings',
            args => {data=>'['.join(',',(('"'.('a' x 1024).'"') x 10)).']'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },

        {
            name => 'json:hash_int_10',
            summary => 'A 10-key hash {"1":0, ..., "10":0}',
            args => {data=>'{'.join(',', map {qq("$_":0)} 1..10).'}'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
        {
            name => 'json:hash_int_100',
            summary => 'A 100-key hash {"1":0, ..., "100":0}',
            args => {data=>'{'.join(',', map {qq("$_":0)} 1..100).'}'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
        {
            name => 'json:hash_int_1000',
            summary => 'A 1000-key hash {"1":0, ..., "1000":0}',
            args => {data=>'{'.join(',', map {qq("$_":0)} 1..1000).'}'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
    ],
};

1;
# ABSTRACT: Benchmark Perl data serialization modules

__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::Serializers - Benchmark Perl data serialization modules

=head1 VERSION

This document describes version 0.09 of Bencher::Scenario::Serializers (from Perl distribution Bencher-Scenario-Serializers), released on 2016-01-20.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m Serializers

To run module startup overhead benchmark:

 % bencher --module-startup -m Serializers

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 BENCHMARKED MODULES

L<JSON::PP>

L<JSON::XS>

L<Cpanel::JSON::XS>

L<JSON::MaybeXS>

L<JSON::Decode::Regexp> 1

L<JSON::Decode::Marpa> 1

L<Pegex::JSON> 1

L<JSON::Create>

L<JSON::Parse> 1

L<YAML::Old>

L<YAML::Syck>

L<YAML::XS>

L<Storable>

L<Sereal>

L<Data::MessagePack>

=head1 BENCHMARK PARTICIPANTS

=over

=item * JSON::PP::encode_json (perl_code)

Code template:

 state $json = JSON::PP->new->allow_nonref; $json->encode(<data>)



=item * JSON::PP::decode_json (perl_code)

Code template:

 state $json = JSON::PP->new->allow_nonref; $json->decode(<data>)



=item * JSON::XS::encode_json (perl_code)

Code template:

 state $json = JSON::XS->new->allow_nonref; $json->encode(<data>)



=item * JSON::XS::decode_json (perl_code)

Code template:

 state $json = JSON::XS->new->allow_nonref; $json->decode(<data>)



=item * Cpanel::JSON::XS::encode_json (perl_code)

Code template:

 state $json = Cpanel::JSON::XS->new->allow_nonref; $json->encode(<data>)



=item * Cpanel::JSON::XS::decode_json (perl_code)

Code template:

 state $json = Cpanel::JSON::XS->new->allow_nonref; $json->decode(<data>)



=item * JSON::MaybeXS::encode_json (perl_code)

Code template:

 state $json = JSON::MaybeXS->new(allow_nonref=>1); $json->encode(<data>)



=item * JSON::MaybeXS::decode_json (perl_code)

Code template:

 state $json = JSON::MaybeXS->new(allow_nonref=>1); $json->decode(<data>)



=item * JSON::Decode::Regexp::from_json (perl_code)

Function call template:

 JSON::Decode::Regexp::from_json(<data>)



=item * JSON::Decode::Marpa::from_json (perl_code)

Function call template:

 JSON::Decode::Marpa::from_json(<data>)



=item * Pegex::JSON (perl_code)

Code template:

 state $obj = Pegex::JSON->new; $obj->load(<data>);



=item * JSON::Create::create_json (perl_code)

Function call template:

 JSON::Create::create_json(<data>)



=item * JSON::Parse::parse_json (perl_code)

Function call template:

 JSON::Parse::parse_json(<data>)



=item * YAML::Old::Dump (perl_code)

Function call template:

 YAML::Old::Dump(<data>)



=item * YAML::Old::Load (perl_code)

Function call template:

 YAML::Old::Load(<data>)



=item * YAML::Syck::Dump (perl_code)

Function call template:

 YAML::Syck::Dump(<data>)



=item * YAML::Syck::Load (perl_code)

Function call template:

 YAML::Syck::Load(<data>)



=item * YAML::XS::Dump (perl_code)

Function call template:

 YAML::XS::Dump(<data>)



=item * YAML::XS::Load (perl_code)

Function call template:

 YAML::XS::Load(<data>)



=item * Storable::freeze (perl_code)

Function call template:

 Storable::freeze(<data>)



=item * Storable::thaw (perl_code)

Function call template:

 Storable::thaw(<data>)



=item * Sereal::encode_sereal (perl_code)

Function call template:

 Sereal::encode_sereal(<data>)



=item * Sereal::decode_sereal (perl_code)

Function call template:

 Sereal::decode_sereal(<data>)



=item * Data::MessagePack::pack (perl_code)

Code template:

 state $obj = Data::MessagePack->new; $obj->pack(<data>)



=item * Data::MessagePack::unpack (perl_code)

Code template:

 state $obj = Data::MessagePack->new; $obj->unpack(<data>)



=back

=head1 BENCHMARK DATASETS

=over

=item * undef

undef

=item * num

A single number (-1.23)

=item * str1k

A non-Unicode string 1024 characters/bytes long

=item * str1k

A Unicode string 1024 characters (3072-bytes) long

=item * array_int_10

A 10-element array containing ints

=item * array_int_100

A 100-element array containing ints

=item * array_int_1000

A 1000-element array containing ints

=item * array_str1k_10

A 10-element array containing 1024-characters/bytes-long non-Unicode strings

=item * array_ustr1k_10

A 10-element array containing 1024-characters-long (3072-bytes long) Unicode strings

=item * hash_int_10

A 10-key hash {1=>0, ..., 10=>0}

=item * hash_int_100

A 100-key hash {1=>0, ..., 100=>0}

=item * hash_int_1000

A 1000-key hash {1=>0, ..., 1000=>0}

=item * json:null

null

=item * json:num

A single number (-1.23)

=item * json:str1k

A non-Unicode (ASCII) string 1024-characters/bytes long

=item * json:array_int_10

A 10-element array containing ints

=item * json:array_int_100

A 10-element array containing ints

=item * json:array_int_1000

A 1000-element array containing ints

=item * json:array_str1k_10

A 10-element array containing 1024-characters/bytes-long non-Unicode strings

=item * json:hash_int_10

A 10-key hash {"1":0, ..., "10":0}

=item * json:hash_int_100

A 100-key hash {"1":0, ..., "100":0}

=item * json:hash_int_1000

A 1000-key hash {"1":0, ..., "1000":0}

=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.22.0 >>, CPU: I<< Intel(R) Core(TM) i5-2400 CPU @ 3.10GHz (4 cores) >>, OS: I<< GNU/Linux Debian version 8.0 >>, OS kernel: I<< Linux version 3.16.0-4-amd64 >>.

Benchmark serializing (C<< bencher -m Serializers --include-participant-tags-json '["serialize"]' >>):

 +-------------------------------+-----------------+------------+-----------+---------+---------+
 | participant                   | dataset         | rate (/s)  | time (ms) | errors  | samples |
 +-------------------------------+-----------------+------------+-----------+---------+---------+
 | YAML::Old::Dump               | hash_int_1000   | 24.9       | 40.1      | 3.2e-05 | 21      |
 | YAML::Old::Dump               | array_int_1000  | 32.6       | 30.6      | 1.2e-05 | 20      |
 | JSON::PP::encode_json         | hash_int_1000   | 234        | 4.28      | 3.1e-06 | 20      |
 | YAML::Old::Dump               | hash_int_100    | 245.4      | 4.075     | 9.1e-07 | 20      |
 | YAML::Old::Dump               | array_int_100   | 318        | 3.14      | 1.7e-06 | 20      |
 | JSON::PP::encode_json         | array_int_1000  | 595        | 1.68      | 1.3e-06 | 22      |
 | YAML::XS::Dump                | hash_int_1000   | 671        | 1.49      | 3.4e-06 | 20      |
 | YAML::Syck::Dump              | hash_int_1000   | 679        | 1.47      | 4.2e-06 | 20      |
 | YAML::Old::Dump               | array_ustr1k_10 | 872        | 1.15      | 6.4e-07 | 20      |
 | YAML::Syck::Dump              | array_int_1000  | 1.63e+03   | 0.613     | 1.5e-06 | 20      |
 | YAML::XS::Dump                | array_int_1000  | 1.77e+03   | 0.565     | 1.5e-06 | 23      |
 | JSON::PP::encode_json         | array_ustr1k_10 | 1898       | 0.5269    | 1.6e-07 | 20      |
 | YAML::Old::Dump               | array_str1k_10  | 2.04e+03   | 0.491     | 4.2e-07 | 21      |
 | YAML::Old::Dump               | hash_int_10     | 2.05e+03   | 0.488     | 2.7e-07 | 20      |
 | JSON::PP::encode_json         | hash_int_100    | 2.25e+03   | 0.444     | 6.5e-07 | 23      |
 | YAML::Old::Dump               | array_int_10    | 2.56e+03   | 0.39      | 4.8e-07 | 20      |
 | YAML::XS::Dump                | array_ustr1k_10 | 3039.585   | 0.3289923 | 4.6e-11 | 20      |
 | Cpanel::JSON::XS::encode_json | array_ustr1k_10 | 3.59e+03   | 0.279     | 2.5e-07 | 23      |
 | JSON::XS::encode_json         | array_ustr1k_10 | 3619       | 0.2763    | 5.3e-08 | 20      |
 | Sereal::encode_sereal         | hash_int_1000   | 3.66e+03   | 0.273     | 8.5e-07 | 20      |
 | JSON::MaybeXS::encode_json    | array_ustr1k_10 | 3.73e+03   | 0.268     | 2.1e-07 | 20      |
 | Data::MessagePack::pack       | hash_int_1000   | 4.27e+03   | 0.234     | 2.1e-07 | 21      |
 | Storable::freeze              | hash_int_1000   | 4.35e+03   | 0.23      | 2.7e-07 | 20      |
 | Cpanel::JSON::XS::encode_json | hash_int_1000   | 4.85e+03   | 0.206     | 2.1e-07 | 20      |
 | JSON::XS::encode_json         | hash_int_1000   | 4.86e+03   | 0.206     | 2e-07   | 22      |
 | JSON::MaybeXS::encode_json    | hash_int_1000   | 4.9e+03    | 0.2       | 1.5e-06 | 20      |
 | JSON::Create::create_json     | hash_int_1000   | 5.07e+03   | 0.197     | 2.1e-07 | 21      |
 | JSON::PP::encode_json         | array_int_100   | 5.71e+03   | 0.175     | 2.1e-07 | 20      |
 | YAML::Old::Dump               | str1k           | 6.9e+03    | 0.14      | 4.6e-07 | 22      |
 | YAML::Syck::Dump              | array_ustr1k_10 | 7.12e+03   | 0.14      | 1.8e-07 | 27      |
 | YAML::XS::Dump                | hash_int_100    | 7.16e+03   | 0.14      | 4.3e-07 | 20      |
 | YAML::XS::Dump                | array_str1k_10  | 7.27e+03   | 0.138     | 5.3e-08 | 20      |
 | YAML::Syck::Dump              | hash_int_100    | 7.3e+03    | 0.137     | 2.1e-07 | 20      |
 | JSON::PP::encode_json         | array_str1k_10  | 1.03e+04   | 0.0968    | 1.1e-07 | 20      |
 | YAML::Old::Dump               | str1k           | 1.26e+04   | 0.0797    | 1.1e-07 | 20      |
 | YAML::Old::Dump               | num             | 1.31e+04   | 0.0763    | 2.7e-08 | 20      |
 | JSON::Create::create_json     | array_ustr1k_10 | 14265.9    | 0.070097  | 5.3e-11 | 23      |
 | Storable::freeze              | array_int_1000  | 1.46e+04   | 0.0683    | 2.7e-08 | 20      |
 | YAML::Old::Dump               | undef           | 1.51e+04   | 0.0662    | 2.7e-08 | 20      |
 | YAML::Syck::Dump              | array_int_100   | 1.63e+04   | 0.0612    | 2.7e-08 | 20      |
 | YAML::XS::Dump                | array_int_100   | 1.63e+04   | 0.0612    | 8e-08   | 20      |
 | JSON::PP::encode_json         | str1k           | 1.87e+04   | 0.0535    | 1e-07   | 21      |
 | JSON::PP::encode_json         | hash_int_10     | 1.96e+04   | 0.051     | 1.1e-07 | 20      |
 | YAML::Syck::Dump              | array_str1k_10  | 1.97e+04   | 0.0507    | 2.5e-08 | 23      |
 | Cpanel::JSON::XS::encode_json | array_int_1000  | 20033.1    | 0.0499174 | 4.6e-11 | 20      |
 | JSON::XS::encode_json         | array_int_1000  | 2.06e+04   | 0.0487    | 5.3e-08 | 20      |
 | JSON::MaybeXS::encode_json    | array_int_1000  | 2.06e+04   | 0.04853   | 1.3e-08 | 21      |
 | Cpanel::JSON::XS::encode_json | str1k           | 2.21e+04   | 0.0453    | 5.3e-08 | 20      |
 | JSON::XS::encode_json         | str1k           | 22811.4    | 0.0438377 | 5.5e-11 | 20      |
 | JSON::Create::create_json     | array_int_1000  | 2.31e+04   | 0.0433    | 5.3e-08 | 32      |
 | JSON::MaybeXS::encode_json    | str1k           | 25673.5    | 0.0389507 | 5.8e-11 | 20      |
 | Data::MessagePack::pack       | array_int_1000  | 2.73e+04   | 0.0367    | 4.9e-08 | 24      |
 | Sereal::encode_sereal         | array_int_1000  | 2.83e+04   | 0.0354    | 1.1e-07 | 20      |
 | YAML::XS::Dump                | str1k           | 2.89e+04   | 0.0346    | 6.5e-08 | 21      |
 | JSON::Create::create_json     | array_str1k_10  | 4.027e+04  | 0.02483   | 6.4e-09 | 22      |
 | Storable::freeze              | hash_int_100    | 4.23e+04   | 0.0237    | 2.7e-08 | 20      |
 | JSON::PP::encode_json         | array_int_10    | 4.26e+04   | 0.0235    | 2.4e-08 | 24      |
 | Cpanel::JSON::XS::encode_json | array_str1k_10  | 4.45e+04   | 0.0225    | 2.7e-08 | 20      |
 | Sereal::encode_sereal         | hash_int_100    | 4.45e+04   | 0.0224    | 5.3e-08 | 20      |
 | JSON::MaybeXS::encode_json    | array_str1k_10  | 4.6e+04    | 0.022     | 1.1e-07 | 20      |
 | Data::MessagePack::pack       | hash_int_100    | 4.7e+04    | 0.0213    | 2.4e-08 | 24      |
 | JSON::XS::encode_json         | array_str1k_10  | 4.71e+04   | 0.0212    | 2e-08   | 20      |
 | YAML::Syck::Dump              | hash_int_10     | 5.3e+04    | 0.019     | 6e-08   | 20      |
 | YAML::Syck::Dump              | str1k           | 5.44e+04   | 0.0184    | 2e-08   | 20      |
 | JSON::XS::encode_json         | hash_int_100    | 5.5e+04    | 0.0182    | 2.7e-08 | 20      |
 | Cpanel::JSON::XS::encode_json | hash_int_100    | 5.64e+04   | 0.0177    | 2.7e-08 | 20      |
 | JSON::MaybeXS::encode_json    | hash_int_100    | 5.67e+04   | 0.0176    | 2e-08   | 20      |
 | YAML::XS::Dump                | hash_int_10     | 5.94e+04   | 0.0168    | 2e-08   | 20      |
 | JSON::Create::create_json     | hash_int_100    | 6.08e+04   | 0.0165    | 2.6e-08 | 21      |
 | YAML::XS::Dump                | str1k           | 62023.4    | 0.016123  | 3.4e-11 | 25      |
 | YAML::Syck::Dump              | array_int_10    | 8.76e+04   | 0.0114    | 1.3e-08 | 20      |
 | JSON::PP::encode_json         | str1k           | 89831      | 0.011132  | 4.2e-11 | 20      |
 | YAML::Syck::Dump              | str1k           | 1.04e+05   | 0.00962   | 1.3e-08 | 20      |
 | Storable::freeze              | array_int_100   | 1.06e+05   | 0.00943   | 1.1e-08 | 27      |
 | YAML::XS::Dump                | array_int_10    | 1.06e+05   | 0.00941   | 1.6e-08 | 21      |
 | Storable::freeze              | array_ustr1k_10 | 1.22e+05   | 0.0082    | 3.1e-09 | 23      |
 | YAML::Syck::Dump              | num             | 1.41e+05   | 0.00709   | 1e-08   | 20      |
 | JSON::Create::create_json     | str1k           | 1.4286e+05 | 0.0069996 | 8.5e-11 | 20      |
 | Storable::freeze              | hash_int_10     | 1.4304e+05 | 0.0069912 | 4.3e-11 | 20      |
 | Storable::freeze              | array_str1k_10  | 1.784e+05  | 0.005605  | 1.5e-09 | 24      |
 | YAML::Syck::Dump              | undef           | 1.82e+05   | 0.00548   | 8.3e-09 | 20      |
 | JSON::MaybeXS::encode_json    | array_int_100   | 1.8e+05    | 0.0054    | 2.2e-08 | 20      |
 | Cpanel::JSON::XS::encode_json | array_int_100   | 1.86e+05   | 0.00538   | 6.5e-09 | 21      |
 | JSON::XS::encode_json         | array_int_100   | 1.91e+05   | 0.00523   | 5e-09   | 20      |
 | JSON::PP::encode_json         | num             | 1.97e+05   | 0.00508   | 1.3e-08 | 20      |
 | YAML::XS::Dump                | num             | 2.04e+05   | 0.00491   | 1.5e-08 | 20      |
 | Storable::freeze              | array_int_10    | 2.37e+05   | 0.00422   | 6.2e-09 | 23      |
 | JSON::Create::create_json     | array_int_100   | 2.39e+05   | 0.00418   | 1.7e-09 | 20      |
 | Data::MessagePack::pack       | array_int_100   | 2.43e+05   | 0.00412   | 6.7e-09 | 20      |
 | Sereal::encode_sereal         | array_int_100   | 2.54e+05   | 0.00394   | 6.7e-09 | 20      |
 | Data::MessagePack::pack       | array_ustr1k_10 | 2.8e+05    | 0.00358   | 1.7e-09 | 20      |
 | Sereal::encode_sereal         | array_ustr1k_10 | 2.83e+05   | 0.00354   | 6.7e-09 | 20      |
 | JSON::PP::encode_json         | undef           | 3.0104e+05 | 0.0033219 | 6.6e-11 | 20      |
 | Data::MessagePack::pack       | array_str1k_10  | 3.5e+05    | 0.0029    | 1.3e-08 | 20      |
 | Sereal::encode_sereal         | array_str1k_10  | 3.549e+05  | 0.002817  | 8.3e-10 | 20      |
 | JSON::Create::create_json     | str1k           | 3.8e+05    | 0.00263   | 7.5e-09 | 20      |
 | YAML::XS::Dump                | undef           | 4.03e+05   | 0.00248   | 3.3e-09 | 20      |
 | Cpanel::JSON::XS::encode_json | str1k           | 4.06e+05   | 0.00247   | 8.3e-10 | 20      |
 | JSON::MaybeXS::encode_json    | str1k           | 4.1e+05    | 0.00244   | 2.5e-09 | 20      |
 | Sereal::encode_sereal         | hash_int_10     | 4.17e+05   | 0.0024    | 4.2e-09 | 20      |
 | Data::MessagePack::pack       | hash_int_10     | 4.33e+05   | 0.00231   | 6.7e-09 | 20      |
 | JSON::XS::encode_json         | str1k           | 4.33e+05   | 0.00231   | 3.3e-09 | 20      |
 | JSON::MaybeXS::encode_json    | hash_int_10     | 4.68e+05   | 0.00214   | 6.5e-09 | 21      |
 | JSON::XS::encode_json         | hash_int_10     | 4.91e+05   | 0.00204   | 8.3e-10 | 20      |
 | Cpanel::JSON::XS::encode_json | hash_int_10     | 4.94e+05   | 0.00202   | 3.2e-09 | 22      |
 | JSON::Create::create_json     | hash_int_10     | 5.23e+05   | 0.00191   | 3.3e-09 | 20      |
 | Sereal::encode_sereal         | array_int_10    | 1.2e+06    | 0.00085   | 5.3e-09 | 21      |
 | JSON::MaybeXS::encode_json    | array_int_10    | 1.25e+06   | 0.000801  | 1.7e-09 | 20      |
 | Cpanel::JSON::XS::encode_json | array_int_10    | 1.25e+06   | 0.000798  | 4.2e-10 | 20      |
 | JSON::XS::encode_json         | array_int_10    | 1.26e+06   | 0.000794  | 1.7e-09 | 20      |
 | Data::MessagePack::pack       | array_int_10    | 1.39e+06   | 0.000719  | 1.7e-09 | 20      |
 | JSON::Create::create_json     | array_int_10    | 1.42e+06   | 0.000706  | 8.3e-10 | 20      |
 | JSON::MaybeXS::encode_json    | num             | 1.65e+06   | 0.000608  | 8e-10   | 22      |
 | Cpanel::JSON::XS::encode_json | num             | 1.65e+06   | 0.000605  | 9.5e-10 | 24      |
 | JSON::XS::encode_json         | num             | 1.67e+06   | 0.000597  | 2.1e-10 | 21      |
 | Sereal::encode_sereal         | str1k           | 1.78e+06   | 0.000562  | 1.9e-10 | 25      |
 | Sereal::encode_sereal         | str1k           | 2e+06      | 0.0005    | 1.9e-09 | 20      |
 | JSON::Create::create_json     | num             | 2.06e+06   | 0.000485  | 1e-09   | 20      |
 | Data::MessagePack::pack       | str1k           | 2.21e+06   | 0.000452  | 1e-09   | 20      |
 | Sereal::encode_sereal         | num             | 2.63e+06   | 0.00038   | 1.1e-09 | 20      |
 | Data::MessagePack::pack       | str1k           | 2.64e+06   | 0.000378  | 9.5e-10 | 24      |
 | Sereal::encode_sereal         | undef           | 3.16e+06   | 0.000317  | 8.6e-10 | 20      |
 | Data::MessagePack::pack       | num             | 4.04e+06   | 0.000248  | 3.6e-10 | 27      |
 | Data::MessagePack::pack       | undef           | 4.46e+06   | 0.000224  | 4.2e-10 | 20      |
 | Cpanel::JSON::XS::encode_json | undef           | 4.81e+06   | 0.000208  | 4.6e-10 | 20      |
 | JSON::MaybeXS::encode_json    | undef           | 4.91e+06   | 0.000204  | 4.3e-10 | 20      |
 | JSON::XS::encode_json         | undef           | 4.93e+06   | 0.000203  | 1.1e-10 | 20      |
 | JSON::Create::create_json     | undef           | 5.58e+06   | 0.000179  | 1.9e-10 | 20      |
 +-------------------------------+-----------------+------------+-----------+---------+---------+


Benchmark deserializing (C<< bencher -m Serializers --include-participant-tags-json '["deserialize"]' >>):

 +---------------------------------+---------------------+------------+-----------+---------+---------+
 | participant                     | dataset             | rate (/s)  | time (ms) | errors  | samples |
 +---------------------------------+---------------------+------------+-----------+---------+---------+
 | JSON::Decode::Regexp::from_json | json:hash_int_1000  | 6.4        | 1.6e+02   | 0.0014  | 20      |
 | Pegex::JSON                     | json:hash_int_1000  | 15         | 66.8      | 8.3e-05 | 20      |
 | JSON::Decode::Marpa::from_json  | json:hash_int_1000  | 17         | 58        | 0.00021 | 20      |
 | Pegex::JSON                     | json:array_int_1000 | 26         | 38.4      | 7.3e-05 | 20      |
 | JSON::Decode::Regexp::from_json | json:array_int_1000 | 36         | 28        | 0.00012 | 22      |
 | JSON::Decode::Marpa::from_json  | json:array_int_1000 | 38         | 27        | 0.00012 | 20      |
 | JSON::PP::decode_json           | json:hash_int_1000  | 87.3       | 11.5      | 1.7e-05 | 20      |
 | JSON::Decode::Marpa::from_json  | json:array_str1k_10 | 89         | 11        | 3.6e-05 | 20      |
 | JSON::PP::decode_json           | json:array_str1k_10 | 131        | 7.61      | 1.5e-05 | 20      |
 | Pegex::JSON                     | json:hash_int_100   | 144        | 6.95      | 8e-06   | 20      |
 | JSON::Decode::Marpa::from_json  | json:hash_int_100   | 148        | 6.76      | 1.7e-05 | 20      |
 | JSON::PP::decode_json           | json:array_int_1000 | 166        | 6.04      | 1.3e-05 | 20      |
 | Pegex::JSON                     | json:array_int_100  | 2.3e+02    | 4.3       | 2.2e-05 | 20      |
 | JSON::Decode::Marpa::from_json  | json:array_int_100  | 2.8e+02    | 3.6       | 6.6e-05 | 20      |
 | JSON::Decode::Regexp::from_json | json:hash_int_100   | 2.8e+02    | 3.5       | 2.8e-05 | 20      |
 | Pegex::JSON                     | json:array_str1k_10 | 607        | 1.65      | 8.3e-07 | 21      |
 | JSON::Decode::Marpa::from_json  | json:hash_int_10    | 7.2e+02    | 1.4       | 8.9e-06 | 21      |
 | JSON::Decode::Regexp::from_json | json:array_int_100  | 852        | 1.17      | 1.8e-06 | 20      |
 | Pegex::JSON                     | json:hash_int_10    | 904        | 1.11      | 1.3e-06 | 20      |
 | JSON::PP::decode_json           | json:hash_int_100   | 946        | 1.06      | 6.4e-07 | 20      |
 | JSON::Decode::Marpa::from_json  | json:array_int_10   | 956        | 1.05      | 2.1e-06 | 20      |
 | Pegex::JSON                     | json:array_int_10   | 1.2e+03    | 0.84      | 3.1e-06 | 20      |
 | JSON::PP::decode_json           | json:str1k          | 1.3e+03    | 0.76      | 2.5e-06 | 20      |
 | JSON::PP::decode_json           | json:array_int_100  | 1.8e+03    | 0.54      | 2e-06   | 20      |
 | Pegex::JSON                     | json:str1k          | 1.9e+03    | 0.52      | 2e-06   | 20      |
 | Pegex::JSON                     | json:num            | 2.1e+03    | 0.48      | 3.9e-06 | 22      |
 | Pegex::JSON                     | json:null           | 2.1e+03    | 0.47      | 1.8e-06 | 20      |
 | JSON::Decode::Regexp::from_json | json:hash_int_10    | 3.3e+03    | 0.303     | 6.6e-07 | 22      |
 | JSON::Decode::Regexp::from_json | json:array_str1k_10 | 3.3e+03    | 0.3       | 9.6e-07 | 20      |
 | JSON::Decode::Regexp::from_json | json:array_int_10   | 5.57e+03   | 0.18      | 2.7e-07 | 20      |
 | JSON::Parse::parse_json         | json:hash_int_1000  | 6.6e+03    | 0.151     | 2.1e-07 | 20      |
 | JSON::XS::decode_json           | json:hash_int_1000  | 6.93e+03   | 0.144     | 2.1e-07 | 20      |
 | JSON::MaybeXS::decode_json      | json:hash_int_1000  | 7.16e+03   | 0.14      | 1.6e-07 | 20      |
 | Cpanel::JSON::XS::decode_json   | json:hash_int_1000  | 7189.94    | 0.139083  | 5.5e-11 | 21      |
 | JSON::PP::decode_json           | json:hash_int_10    | 9.04e+03   | 0.111     | 5.1e-08 | 22      |
 | JSON::Decode::Regexp::from_json | json:str1k          | 1.1e+04    | 0.0906    | 1.1e-07 | 20      |
 | JSON::Decode::Regexp::from_json | json:num            | 1.17e+04   | 0.0851    | 2.1e-07 | 20      |
 | JSON::Decode::Regexp::from_json | json:null           | 1.44e+04   | 0.0696    | 2.1e-07 | 21      |
 | JSON::PP::decode_json           | json:array_int_10   | 1.7e+04    | 0.058     | 3.2e-07 | 20      |
 | JSON::Parse::parse_json         | json:array_int_1000 | 2.651e+04  | 0.03773   | 1.2e-08 | 25      |
 | Cpanel::JSON::XS::decode_json   | json:array_int_1000 | 2.89e+04   | 0.0346    | 5e-08   | 23      |
 | JSON::MaybeXS::decode_json      | json:array_int_1000 | 2.96e+04   | 0.0338    | 1.3e-08 | 20      |
 | JSON::XS::decode_json           | json:array_int_1000 | 3.04e+04   | 0.0329    | 1.2e-08 | 24      |
 | JSON::Parse::parse_json         | json:array_str1k_10 | 3.56e+04   | 0.0281    | 4e-08   | 20      |
 | JSON::MaybeXS::decode_json      | json:array_str1k_10 | 5.1e+04    | 0.0196    | 2e-08   | 20      |
 | Cpanel::JSON::XS::decode_json   | json:array_str1k_10 | 5.14e+04   | 0.0195    | 6.7e-09 | 20      |
 | JSON::XS::decode_json           | json:array_str1k_10 | 6.42e+04   | 0.0156    | 6.7e-09 | 20      |
 | JSON::PP::decode_json           | json:num            | 7.14e+04   | 0.014     | 2.5e-08 | 22      |
 | Cpanel::JSON::XS::decode_json   | json:hash_int_100   | 8.58e+04   | 0.0117    | 1.3e-08 | 20      |
 | JSON::Parse::parse_json         | json:hash_int_100   | 8.6e+04    | 0.0116    | 1.3e-08 | 22      |
 | JSON::MaybeXS::decode_json      | json:hash_int_100   | 8.9e+04    | 0.0112    | 1.3e-08 | 20      |
 | JSON::XS::decode_json           | json:hash_int_100   | 89736      | 0.011144  | 2.3e-10 | 20      |
 | JSON::PP::decode_json           | json:null           | 1.33e+05   | 0.00753   | 3e-09   | 24      |
 | Cpanel::JSON::XS::decode_json   | json:array_int_100  | 2.38e+05   | 0.0042    | 4.8e-09 | 22      |
 | JSON::MaybeXS::decode_json      | json:array_int_100  | 2.4e+05    | 0.00417   | 6.7e-09 | 20      |
 | JSON::XS::decode_json           | json:array_int_100  | 2.4e+05    | 0.0041    | 2e-08   | 20      |
 | JSON::Parse::parse_json         | json:array_int_100  | 2.4225e+05 | 0.0041279 | 3.5e-11 | 20      |
 | JSON::Parse::parse_json         | json:str1k          | 3.576e+05  | 0.0027964 | 5.5e-11 | 26      |
 | JSON::MaybeXS::decode_json      | json:str1k          | 4.98e+05   | 0.00201   | 8.5e-10 | 20      |
 | Cpanel::JSON::XS::decode_json   | json:str1k          | 5.06e+05   | 0.00198   | 8.3e-10 | 21      |
 | JSON::XS::decode_json           | json:str1k          | 6.5e+05    | 0.00154   | 8.5e-10 | 20      |
 | Cpanel::JSON::XS::decode_json   | json:hash_int_10    | 7.67e+05   | 0.0013    | 3.7e-09 | 20      |
 | JSON::MaybeXS::decode_json      | json:hash_int_10    | 8.1e+05    | 0.00123   | 1.6e-09 | 21      |
 | JSON::XS::decode_json           | json:hash_int_10    | 8.34e+05   | 0.0012    | 1.7e-09 | 20      |
 | JSON::Parse::parse_json         | json:hash_int_10    | 8.52e+05   | 0.00117   | 4e-10   | 22      |
 | JSON::XS::decode_json           | json:num            | 1.01e+06   | 0.000987  | 4.6e-10 | 20      |
 | JSON::Parse::parse_json         | json:num            | 1.02e+06   | 0.000983  | 1.7e-09 | 20      |
 | Cpanel::JSON::XS::decode_json   | json:num            | 1.07e+06   | 0.000937  | 1.3e-09 | 20      |
 | JSON::MaybeXS::decode_json      | json:num            | 1.07e+06   | 0.000932  | 1.7e-09 | 20      |
 | Cpanel::JSON::XS::decode_json   | json:array_int_10   | 1.35e+06   | 0.000741  | 4.1e-10 | 22      |
 | JSON::MaybeXS::decode_json      | json:array_int_10   | 1.35e+06   | 0.000739  | 1.3e-09 | 20      |
 | JSON::Parse::parse_json         | json:array_int_10   | 1.414e+06  | 0.0007071 | 5.2e-11 | 20      |
 | JSON::XS::decode_json           | json:array_int_10   | 1.45e+06   | 0.000689  | 8.5e-10 | 20      |
 | JSON::Parse::parse_json         | json:null           | 6.8e+06    | 0.00015   | 9.4e-10 | 20      |
 | JSON::MaybeXS::decode_json      | json:null           | 7e+06      | 0.000143  | 1.8e-10 | 23      |
 | JSON::XS::decode_json           | json:null           | 7.1e+06    | 0.000141  | 1.4e-10 | 22      |
 | Cpanel::JSON::XS::decode_json   | json:null           | 7.2e+06    | 0.00014   | 1.3e-09 | 20      |
 +---------------------------------+---------------------+------------+-----------+---------+---------+


Benchmark module startup overhead (C<< bencher -m Serializers --module-startup >>):

 +----------------------+-----------+------------------------+---------+---------+
 | participant          | time (ms) | mod_overhead_time (ms) | errors  | samples |
 +----------------------+-----------+------------------------+---------+---------+
 | JSON::Decode::Marpa  | 1.3e+02   | 123.6                  | 0.00043 | 20      |
 | YAML::XS             | 53        | 46.6                   | 0.00013 | 20      |
 | JSON::PP             | 27        | 20.6                   | 0.00011 | 20      |
 | Pegex::JSON          | 23.1      | 16.7                   | 6.8e-05 | 20      |
 | JSON::MaybeXS        | 19        | 12.6                   | 0.0001  | 20      |
 | Sereal               | 16.3      | 9.9                    | 4.7e-05 | 20      |
 | Storable             | 16.1      | 9.7                    | 5.1e-05 | 20      |
 | YAML::Old            | 15        | 8.6                    | 5.4e-05 | 20      |
 | JSON::XS             | 14        | 7.6                    | 0.00014 | 20      |
 | Cpanel::JSON::XS     | 13        | 6.6                    | 7.8e-05 | 20      |
 | YAML::Syck           | 13        | 6.6                    | 5.1e-05 | 20      |
 | JSON::Parse          | 13        | 6.6                    | 4.1e-05 | 20      |
 | JSON::Create         | 10        | 3.6                    | 5.9e-05 | 20      |
 | JSON::Decode::Regexp | 10        | 3.6                    | 0.00016 | 20      |
 | Data::MessagePack    | 9.8       | 3.4                    | 8.3e-05 | 20      |
 | perl -e1 (baseline)  | 6.4       | 0                      | 9.1e-05 | 20      |
 +----------------------+-----------+------------------------+---------+---------+

=head1 DESCRIPTION

Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenario-Serializers>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenario-Serializers>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenario-Serializers>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
