package Bencher::Scenario::Serializers;

our $DATE = '2016-01-19'; # DATE
our $VERSION = '0.08'; # VERSION

use 5.010001;
use strict;
use utf8;
use warnings;

our $scenario = {
    summary => 'Benchmark Perl data serialization modules',
    participants => [
        {
            tags => ['json', 'serialize'],
            module => 'JSON::PP',
            function => 'encode_json',
            code_template => 'state $json = JSON::PP->new->allow_nonref; $json->encode(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            module => 'JSON::PP',
            function => 'decode_json',
            code_template => 'state $json = JSON::PP->new->allow_nonref; $json->decode(<data>)',
        },
        {
            tags => ['json', 'serialize'],
            module => 'JSON::XS',
            function => 'encode_json',
            code_template => 'state $json = JSON::XS->new->allow_nonref; $json->encode(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            module => 'JSON::XS',
            function => 'decode_json',
            code_template => 'state $json = JSON::XS->new->allow_nonref; $json->decode(<data>)',
        },
        {
            tags => ['json', 'serialize'],
            module => 'Cpanel::JSON::XS',
            function => 'encode_json',
            code_template => 'state $json = Cpanel::JSON::XS->new->allow_nonref; $json->encode(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            module => 'Cpanel::JSON::XS',
            function => 'decode_json',
            code_template => 'state $json = Cpanel::JSON::XS->new->allow_nonref; $json->decode(<data>)',
        },
        {
            tags => ['json', 'serialize'],
            module => 'JSON::MaybeXS',
            function => 'encode_json',
            code_template => 'state $json = JSON::MaybeXS->new(allow_nonref=>1); $json->encode(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            module => 'JSON::MaybeXS',
            function => 'decode_json',
            code_template => 'state $json = JSON::MaybeXS->new(allow_nonref=>1); $json->decode(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            fcall_template => 'JSON::Decode::Regexp::from_json(<data>)',
        },
        {
            tags => ['json', 'deserialize', 'cant_handle_scalar'],
            fcall_template => 'JSON::Decode::Marpa::from_json(<data>)',
        },
        {
            name => 'Pegex::JSON',
            tags => ['json', 'deserialize'],
            module => 'Pegex::JSON',
            code_template => 'state $obj = Pegex::JSON->new; $obj->load(<data>);',
        },
        {
            tags => ['json', 'serialize'],
            fcall_template => 'JSON::Create::create_json(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            fcall_template => 'JSON::Parse::parse_json(<data>)',
        },

        {
            tags => ['yaml', 'serialize'],
            fcall_template => 'YAML::Old::Dump(<data>)',
        },
        {
            tags => ['yaml', 'deserialize'],
            fcall_template => 'YAML::Old::Load(<data>)',
        },
        {
            tags => ['yaml', 'serialize'],
            fcall_template => 'YAML::Syck::Dump(<data>)',
        },
        {
            tags => ['yaml', 'deserialize'],
            fcall_template => 'YAML::Syck::Load(<data>)',
        },
        {
            tags => ['yaml', 'serialize'],
            fcall_template => 'YAML::XS::Dump(<data>)',
        },
        {
            tags => ['yaml', 'deserialize'],
            fcall_template => 'YAML::XS::Load(<data>)',
        },

        {
            tags => ['binary', 'serialize', 'cant_handle_scalar'],
            fcall_template => 'Storable::freeze(<data>)',
        },
        {
            tags => ['binary', 'deserialize', 'cant_handle_scalar'],
            fcall_template => 'Storable::thaw(<data>)',
        },

        {
            tags => ['binary', 'serialize'],
            fcall_template => 'Sereal::encode_sereal(<data>)',
        },
        {
            tags => ['binary', 'deserialize'],
            fcall_template => 'Sereal::decode_sereal(<data>)',
        },

        {
            name => 'Data::MessagePack::pack',
            tags => ['binary', 'serialize'],
            module => 'Data::MessagePack',
            function => 'pack',
            code_template => 'state $obj = Data::MessagePack->new; $obj->pack(<data>)',
        },
        {
            name => 'Data::MessagePack::unpack',
            tags => ['binary', 'deserialize'],
            module => 'Data::MessagePack',
            function => 'unpack',
            code_template => 'state $obj = Data::MessagePack->new; $obj->unpack(<data>)',
        },
    ],

    # XXX: add more datasets (larger data, etc)
    datasets => [
        {
            name => 'undef',
            summary => 'undef',
            args => {data=>undef},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },
        {
            name => 'num',
            summary => 'A single number (-1.23)',
            args => {data=>-1.23},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },
        {
            name => 'str1k',
            summary => 'A non-Unicode string 1024 bytes long',
            args => {data=>'a' x 1024},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },
        {
            name => 'str1k',
            summary => 'A Unicode string 1024 bytes long',
            args => {data=>'我爱你爱你一辈子' x 128},
            tags => ['serialize', 'unicode'],
            include_participant_tags => ['serialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },

        {
            name => 'array_int_10',
            summary => 'A 10-element array containing ints',
            args => {data=>[1..10]},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'array_int_100',
            summary => 'A 100-element array containing ints',
            args => {data=>[1..100]},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'array_int_1000',
            summary => 'A 1000-element array containing ints',
            args => {data=>[1..1000]},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'array_str1k_10',
            summary => 'A 10-element array containing 1024-bytes-long non-Unicode strings',
            args => {data=>[('a' x 1024) x 10]},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'array_ustr1k_10',
            summary => 'A 10-element array containing 1024-bytes-long Unicode strings',
            args => {data=>[('我爱你爱你一辈子' x 128) x 10]},
            tags => ['serialize', 'json'],
            include_participant_tags => ['serialize'],
        },

        {
            name => 'hash_int_10',
            summary => 'A 10-key hash {1=>0, ..., 10=>0}',
            args => {data=>{map {$_=>0} 1..10}},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'hash_int_100',
            summary => 'A 100-key hash {1=>0, ..., 100=>0}',
            args => {data=>{map {$_=>0} 1..100}},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'hash_int_1000',
            summary => 'A 1000-key hash {1=>0, ..., 1000=>0}',
            args => {data=>{map {$_=>0} 1..1000}},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },

        {
            name => 'json:null',
            summary => 'null',
            args => {data=>'null'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },
        {
            name => 'json:num',
            summary => 'A single number (-1.23)',
            args => {data=>-1.23},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },
        {
            name => 'json:str1k',
            summary => 'A non-Unicode (ASCII) string 1024-byte long',
            args => {data=>'"' . ('a' x 1024) . '"'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },

        {
            name => 'json:array_int_10',
            summary => 'A 10-element array containing ints',
            args => {data=>'['.join(',',1..10).']'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
        {
            name => 'json:array_int_100',
            summary => 'A 10-element array containing ints',
            args => {data=>'['.join(',',1..100).']'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
        {
            name => 'json:array_int_1000',
            summary => 'A 1000-element array containing ints',
            args => {data=>'['.join(',',1..1000).']'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
        {
            name => 'json:array_str1k_10',
            summary => 'A 10-element array containing 1024-bytes-long non-Unicode strings',
            args => {data=>'['.join(',',(('"'.('a' x 1024).'"') x 10)).']'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },

        {
            name => 'json:hash_int_10',
            summary => 'A 10-key hash {"1":0, ..., "10":0}',
            args => {data=>'{'.join(',', map {qq("$_":0)} 1..10).'}'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
        {
            name => 'json:hash_int_100',
            summary => 'A 100-key hash {"1":0, ..., "100":0}',
            args => {data=>'{'.join(',', map {qq("$_":0)} 1..100).'}'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
        {
            name => 'json:hash_int_1000',
            summary => 'A 1000-key hash {"1":0, ..., "1000":0}',
            args => {data=>'{'.join(',', map {qq("$_":0)} 1..1000).'}'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
    ],
};

1;
# ABSTRACT: Benchmark Perl data serialization modules

__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::Serializers - Benchmark Perl data serialization modules

=head1 VERSION

This document describes version 0.08 of Bencher::Scenario::Serializers (from Perl distribution Bencher-Scenario-Serializers), released on 2016-01-19.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m Serializers

To run module startup overhead benchmark:

 % bencher --module-startup -m Serializers

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 BENCHMARKED MODULES

L<JSON::PP>

L<JSON::XS>

L<Cpanel::JSON::XS>

L<JSON::MaybeXS>

L<JSON::Decode::Regexp> 1

L<JSON::Decode::Marpa> 1

L<Pegex::JSON> 1

L<JSON::Create>

L<JSON::Parse> 1

L<YAML::Old>

L<YAML::Syck>

L<YAML::XS>

L<Storable>

L<Sereal>

L<Data::MessagePack>

=head1 BENCHMARK PARTICIPANTS

=over

=item * JSON::PP::encode_json (perl_code)

Code template:

 state $json = JSON::PP->new->allow_nonref; $json->encode(<data>)



=item * JSON::PP::decode_json (perl_code)

Code template:

 state $json = JSON::PP->new->allow_nonref; $json->decode(<data>)



=item * JSON::XS::encode_json (perl_code)

Code template:

 state $json = JSON::XS->new->allow_nonref; $json->encode(<data>)



=item * JSON::XS::decode_json (perl_code)

Code template:

 state $json = JSON::XS->new->allow_nonref; $json->decode(<data>)



=item * Cpanel::JSON::XS::encode_json (perl_code)

Code template:

 state $json = Cpanel::JSON::XS->new->allow_nonref; $json->encode(<data>)



=item * Cpanel::JSON::XS::decode_json (perl_code)

Code template:

 state $json = Cpanel::JSON::XS->new->allow_nonref; $json->decode(<data>)



=item * JSON::MaybeXS::encode_json (perl_code)

Code template:

 state $json = JSON::MaybeXS->new(allow_nonref=>1); $json->encode(<data>)



=item * JSON::MaybeXS::decode_json (perl_code)

Code template:

 state $json = JSON::MaybeXS->new(allow_nonref=>1); $json->decode(<data>)



=item * JSON::Decode::Regexp::from_json (perl_code)

Function call template:

 JSON::Decode::Regexp::from_json(<data>)



=item * JSON::Decode::Marpa::from_json (perl_code)

Function call template:

 JSON::Decode::Marpa::from_json(<data>)



=item * Pegex::JSON (perl_code)

Code template:

 state $obj = Pegex::JSON->new; $obj->load(<data>);



=item * JSON::Create::create_json (perl_code)

Function call template:

 JSON::Create::create_json(<data>)



=item * JSON::Parse::parse_json (perl_code)

Function call template:

 JSON::Parse::parse_json(<data>)



=item * YAML::Old::Dump (perl_code)

Function call template:

 YAML::Old::Dump(<data>)



=item * YAML::Old::Load (perl_code)

Function call template:

 YAML::Old::Load(<data>)



=item * YAML::Syck::Dump (perl_code)

Function call template:

 YAML::Syck::Dump(<data>)



=item * YAML::Syck::Load (perl_code)

Function call template:

 YAML::Syck::Load(<data>)



=item * YAML::XS::Dump (perl_code)

Function call template:

 YAML::XS::Dump(<data>)



=item * YAML::XS::Load (perl_code)

Function call template:

 YAML::XS::Load(<data>)



=item * Storable::freeze (perl_code)

Function call template:

 Storable::freeze(<data>)



=item * Storable::thaw (perl_code)

Function call template:

 Storable::thaw(<data>)



=item * Sereal::encode_sereal (perl_code)

Function call template:

 Sereal::encode_sereal(<data>)



=item * Sereal::decode_sereal (perl_code)

Function call template:

 Sereal::decode_sereal(<data>)



=item * Data::MessagePack::pack (perl_code)

Code template:

 state $obj = Data::MessagePack->new; $obj->pack(<data>)



=item * Data::MessagePack::unpack (perl_code)

Code template:

 state $obj = Data::MessagePack->new; $obj->unpack(<data>)



=back

=head1 BENCHMARK DATASETS

=over

=item * undef

undef

=item * num

A single number (-1.23)

=item * str1k

A non-Unicode string 1024 bytes long

=item * str1k

A Unicode string 1024 bytes long

=item * array_int_10

A 10-element array containing ints

=item * array_int_100

A 100-element array containing ints

=item * array_int_1000

A 1000-element array containing ints

=item * array_str1k_10

A 10-element array containing 1024-bytes-long non-Unicode strings

=item * array_ustr1k_10

A 10-element array containing 1024-bytes-long Unicode strings

=item * hash_int_10

A 10-key hash {1=>0, ..., 10=>0}

=item * hash_int_100

A 100-key hash {1=>0, ..., 100=>0}

=item * hash_int_1000

A 1000-key hash {1=>0, ..., 1000=>0}

=item * json:null

null

=item * json:num

A single number (-1.23)

=item * json:str1k

A non-Unicode (ASCII) string 1024-byte long

=item * json:array_int_10

A 10-element array containing ints

=item * json:array_int_100

A 10-element array containing ints

=item * json:array_int_1000

A 1000-element array containing ints

=item * json:array_str1k_10

A 10-element array containing 1024-bytes-long non-Unicode strings

=item * json:hash_int_10

A 10-key hash {"1":0, ..., "10":0}

=item * json:hash_int_100

A 100-key hash {"1":0, ..., "100":0}

=item * json:hash_int_1000

A 1000-key hash {"1":0, ..., "1000":0}

=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.22.0 >>, CPU: I<< Intel(R) Core(TM) i5-2400 CPU @ 3.10GHz (4 cores) >>, OS: I<< GNU/Linux Debian version 8.0 >>, OS kernel: I<< Linux version 3.16.0-4-amd64 >>.

Benchmark serializing (C<< bencher -m Serializers --include-participant-tags-json '["serialize"]' >>):

 +-------------------------------+-----------------+------------+------------+---------+---------+
 | participant                   | dataset         | rate (/s)  | time (ms)  | errors  | samples |
 +-------------------------------+-----------------+------------+------------+---------+---------+
 | YAML::Old::Dump               | hash_int_1000   | 24.4       | 41         | 4.1e-05 | 20      |
 | YAML::Old::Dump               | array_int_1000  | 32.1       | 31.1       | 1.5e-05 | 20      |
 | JSON::PP::encode_json         | hash_int_1000   | 224        | 4.47       | 1.2e-05 | 20      |
 | YAML::Old::Dump               | hash_int_100    | 238        | 4.2        | 1.1e-05 | 20      |
 | YAML::Old::Dump               | array_int_100   | 311        | 3.21       | 7.1e-06 | 20      |
 | JSON::PP::encode_json         | array_int_1000  | 603        | 1.66       | 2e-06   | 20      |
 | YAML::Syck::Dump              | hash_int_1000   | 664        | 1.5        | 3.8e-06 | 20      |
 | YAML::XS::Dump                | hash_int_1000   | 700        | 1.43       | 2.2e-06 | 20      |
 | YAML::Old::Dump               | array_ustr1k_10 | 870        | 1.15       | 1.3e-06 | 20      |
 | YAML::Syck::Dump              | array_int_1000  | 1.64e+03   | 0.609      | 1.1e-06 | 20      |
 | YAML::XS::Dump                | array_int_1000  | 1.76e+03   | 0.569      | 2.1e-07 | 20      |
 | JSON::PP::encode_json         | array_ustr1k_10 | 1.89e+03   | 0.53       | 4.2e-07 | 21      |
 | YAML::Old::Dump               | array_str1k_10  | 2.02e+03   | 0.495      | 4.3e-07 | 20      |
 | YAML::Old::Dump               | hash_int_10     | 2.04e+03   | 0.491      | 2.7e-07 | 20      |
 | JSON::PP::encode_json         | hash_int_100    | 2.17e+03   | 0.46       | 1.1e-06 | 20      |
 | YAML::Old::Dump               | array_int_10    | 2.56e+03   | 0.391      | 4.1e-07 | 22      |
 | YAML::XS::Dump                | array_ustr1k_10 | 2.99e+03   | 0.334      | 7.5e-07 | 20      |
 | Sereal::encode_sereal         | hash_int_1000   | 3.56e+03   | 0.281      | 4.8e-07 | 20      |
 | Cpanel::JSON::XS::encode_json | array_ustr1k_10 | 3.61e+03   | 0.277      | 4.8e-07 | 20      |
 | JSON::MaybeXS::encode_json    | array_ustr1k_10 | 3.63e+03   | 0.276      | 2.1e-07 | 21      |
 | JSON::XS::encode_json         | array_ustr1k_10 | 3632       | 0.2753     | 5.3e-08 | 20      |
 | Data::MessagePack::pack       | hash_int_1000   | 4.3e+03    | 0.233      | 2.1e-07 | 20      |
 | Storable::freeze              | hash_int_1000   | 4.4e+03    | 0.227      | 4.7e-07 | 21      |
 | JSON::MaybeXS::encode_json    | hash_int_1000   | 4.83e+03   | 0.207      | 2.1e-07 | 20      |
 | JSON::XS::encode_json         | hash_int_1000   | 4.8e+03    | 0.21       | 1.1e-06 | 20      |
 | Cpanel::JSON::XS::encode_json | hash_int_1000   | 4850.96    | 0.206145   | 1.8e-10 | 21      |
 | JSON::Create::create_json     | hash_int_1000   | 4.97e+03   | 0.201      | 2.1e-07 | 20      |
 | JSON::PP::encode_json         | array_int_100   | 5778       | 0.1731     | 5.3e-08 | 20      |
 | YAML::Syck::Dump              | array_ustr1k_10 | 6.79e+03   | 0.147      | 2.1e-07 | 20      |
 | YAML::Old::Dump               | str1k           | 6.91e+03   | 0.145      | 2.1e-07 | 20      |
 | YAML::Syck::Dump              | hash_int_100    | 6.92e+03   | 0.144      | 2.1e-07 | 20      |
 | YAML::XS::Dump                | array_str1k_10  | 7269       | 0.1376     | 4.2e-08 | 33      |
 | YAML::XS::Dump                | hash_int_100    | 7.32e+03   | 0.137      | 5.3e-08 | 20      |
 | JSON::PP::encode_json         | array_str1k_10  | 1.037e+04  | 0.09642    | 2.7e-08 | 20      |
 | YAML::Old::Dump               | str1k           | 1.26e+04   | 0.0793     | 1.2e-07 | 24      |
 | YAML::Old::Dump               | num             | 1.32e+04   | 0.0756     | 1.1e-07 | 20      |
 | Storable::freeze              | array_int_1000  | 1.41e+04   | 0.0709     | 1.1e-07 | 27      |
 | JSON::Create::create_json     | array_ustr1k_10 | 14283.8    | 0.0700096  | 4.5e-11 | 21      |
 | YAML::Old::Dump               | undef           | 1.5e+04    | 0.067      | 3.4e-07 | 21      |
 | YAML::Syck::Dump              | array_int_100   | 1.62e+04   | 0.0617     | 1e-07   | 21      |
 | YAML::XS::Dump                | array_int_100   | 1.65e+04   | 0.0607     | 2.6e-08 | 21      |
 | JSON::PP::encode_json         | str1k           | 1.83e+04   | 0.0547     | 2.5e-08 | 23      |
 | JSON::PP::encode_json         | hash_int_10     | 1.88e+04   | 0.0531     | 7.6e-08 | 22      |
 | YAML::Syck::Dump              | array_str1k_10  | 20010.97   | 0.04997259 | 1.2e-11 | 22      |
 | JSON::MaybeXS::encode_json    | array_int_1000  | 2e+04      | 0.049      | 1.7e-07 | 23      |
 | Cpanel::JSON::XS::encode_json | array_int_1000  | 2.06e+04   | 0.0485     | 5.1e-08 | 22      |
 | JSON::XS::encode_json         | array_int_1000  | 2.095e+04  | 0.04774    | 1.3e-08 | 21      |
 | JSON::MaybeXS::encode_json    | str1k           | 22200.2    | 0.0450447  | 4.6e-11 | 21      |
 | Cpanel::JSON::XS::encode_json | str1k           | 22236      | 0.044972   | 2e-10   | 21      |
 | JSON::Create::create_json     | array_int_1000  | 22307.5    | 0.0448279  | 5.6e-11 | 20      |
 | JSON::XS::encode_json         | str1k           | 22838.3    | 0.043786   | 4.5e-11 | 29      |
 | Data::MessagePack::pack       | array_int_1000  | 2.66e+04   | 0.0375     | 5.3e-08 | 20      |
 | YAML::XS::Dump                | str1k           | 2.8e+04    | 0.0357     | 5.3e-08 | 20      |
 | Sereal::encode_sereal         | array_int_1000  | 2.87e+04   | 0.0348     | 6.4e-08 | 22      |
 | JSON::Create::create_json     | array_str1k_10  | 3.9e+04    | 0.026      | 1.7e-07 | 20      |
 | Storable::freeze              | hash_int_100    | 4.23e+04   | 0.0237     | 3e-08   | 24      |
 | JSON::PP::encode_json         | array_int_10    | 4.351e+04  | 0.02298    | 6.7e-09 | 20      |
 | JSON::MaybeXS::encode_json    | array_str1k_10  | 4.4e+04    | 0.023      | 1.1e-07 | 21      |
 | Cpanel::JSON::XS::encode_json | array_str1k_10  | 4.41e+04   | 0.0227     | 5.2e-08 | 27      |
 | Sereal::encode_sereal         | hash_int_100    | 4.44e+04   | 0.0225     | 5.3e-08 | 20      |
 | JSON::XS::encode_json         | array_str1k_10  | 4.64e+04   | 0.02155    | 6.2e-09 | 23      |
 | Data::MessagePack::pack       | hash_int_100    | 4.9e+04    | 0.02       | 7.8e-08 | 21      |
 | YAML::Syck::Dump              | hash_int_10     | 5.16e+04   | 0.0194     | 2.7e-08 | 20      |
 | YAML::Syck::Dump              | str1k           | 5.45e+04   | 0.0183     | 6.4e-09 | 22      |
 | JSON::XS::encode_json         | hash_int_100    | 5.51e+04   | 0.0181     | 3.3e-08 | 21      |
 | Cpanel::JSON::XS::encode_json | hash_int_100    | 5.62e+04   | 0.0178     | 5.2e-08 | 21      |
 | JSON::MaybeXS::encode_json    | hash_int_100    | 5.63e+04   | 0.0178     | 3.3e-08 | 21      |
 | JSON::Create::create_json     | hash_int_100    | 5.94e+04   | 0.0168     | 2e-08   | 20      |
 | YAML::XS::Dump                | hash_int_10     | 59679.4    | 0.0167562  | 4.5e-11 | 20      |
 | YAML::XS::Dump                | str1k           | 6.22e+04   | 0.0161     | 6.5e-09 | 21      |
 | YAML::Syck::Dump              | array_int_10    | 8.46e+04   | 0.0118     | 1.3e-08 | 20      |
 | JSON::PP::encode_json         | str1k           | 8.6e+04    | 0.0116     | 1e-08   | 20      |
 | YAML::XS::Dump                | array_int_10    | 1.02e+05   | 0.00979    | 1.3e-08 | 20      |
 | YAML::Syck::Dump              | str1k           | 1.041e+05  | 0.0096059  | 5.4e-11 | 20      |
 | Storable::freeze              | array_int_100   | 1.05e+05   | 0.00948    | 3.2e-09 | 22      |
 | Storable::freeze              | array_ustr1k_10 | 1.26e+05   | 0.00792    | 3.3e-09 | 20      |
 | YAML::Syck::Dump              | num             | 1.39e+05   | 0.00719    | 3e-09   | 25      |
 | Storable::freeze              | hash_int_10     | 1.4e+05    | 0.00712    | 1.7e-08 | 20      |
 | JSON::Create::create_json     | str1k           | 1.4232e+05 | 0.0070262  | 8.6e-11 | 20      |
 | Storable::freeze              | array_str1k_10  | 1.68e+05   | 0.00594    | 6.7e-09 | 20      |
 | YAML::Syck::Dump              | undef           | 1.8e+05    | 0.00556    | 6.7e-09 | 20      |
 | JSON::MaybeXS::encode_json    | array_int_100   | 1.87e+05   | 0.00535    | 6.7e-09 | 20      |
 | JSON::XS::encode_json         | array_int_100   | 1.872e+05  | 0.005342   | 1.7e-09 | 20      |
 | JSON::PP::encode_json         | num             | 1.88e+05   | 0.00531    | 7.8e-09 | 23      |
 | Cpanel::JSON::XS::encode_json | array_int_100   | 1.9e+05    | 0.00526    | 6.7e-09 | 20      |
 | YAML::XS::Dump                | num             | 2.1e+05    | 0.00477    | 6.2e-09 | 23      |
 | Storable::freeze              | array_int_10    | 2.32e+05   | 0.00432    | 6.7e-09 | 20      |
 | JSON::Create::create_json     | array_int_100   | 2.326e+05  | 0.0043     | 1.3e-09 | 32      |
 | Data::MessagePack::pack       | array_int_100   | 2.4e+05    | 0.0041     | 1.3e-08 | 20      |
 | Sereal::encode_sereal         | array_int_100   | 2.448e+05  | 0.004085   | 4.7e-11 | 20      |
 | Data::MessagePack::pack       | array_ustr1k_10 | 2.7651e+05 | 0.0036165  | 4.4e-11 | 20      |
 | Sereal::encode_sereal         | array_ustr1k_10 | 2.85e+05   | 0.00351    | 4.9e-09 | 21      |
 | JSON::PP::encode_json         | undef           | 2.97e+05   | 0.00337    | 1.5e-09 | 25      |
 | Data::MessagePack::pack       | array_str1k_10  | 3.402e+05  | 0.00294    | 8e-10   | 22      |
 | Sereal::encode_sereal         | array_str1k_10  | 3.472e+05  | 0.00288    | 1.5e-10 | 20      |
 | JSON::Create::create_json     | str1k           | 3.7629e+05 | 0.0026575  | 4.5e-11 | 25      |
 | YAML::XS::Dump                | undef           | 4e+05      | 0.0025     | 2.5e-09 | 20      |
 | Cpanel::JSON::XS::encode_json | str1k           | 4.01e+05   | 0.0025     | 7.5e-09 | 20      |
 | Sereal::encode_sereal         | hash_int_10     | 4.03e+05   | 0.00248    | 8.4e-10 | 20      |
 | JSON::MaybeXS::encode_json    | str1k           | 4.04e+05   | 0.00247    | 3.3e-09 | 20      |
 | JSON::XS::encode_json         | str1k           | 4.3e+05    | 0.00233    | 3.3e-09 | 20      |
 | Data::MessagePack::pack       | hash_int_10     | 4.31e+05   | 0.00232    | 4.2e-09 | 20      |
 | JSON::Create::create_json     | hash_int_10     | 4.79e+05   | 0.00209    | 3.3e-09 | 20      |
 | Cpanel::JSON::XS::encode_json | hash_int_10     | 4.81e+05   | 0.00208    | 6.5e-09 | 21      |
 | JSON::MaybeXS::encode_json    | hash_int_10     | 4.92e+05   | 0.00203    | 2.5e-09 | 20      |
 | JSON::XS::encode_json         | hash_int_10     | 4.97e+05   | 0.00201    | 2.5e-09 | 20      |
 | Sereal::encode_sereal         | array_int_10    | 1.04e+06   | 0.000962   | 1.7e-09 | 21      |
 | JSON::MaybeXS::encode_json    | array_int_10    | 1.28e+06   | 0.000784   | 1.7e-09 | 20      |
 | Cpanel::JSON::XS::encode_json | array_int_10    | 1.285e+06  | 0.0007783  | 1.5e-10 | 21      |
 | JSON::XS::encode_json         | array_int_10    | 1.3e+06    | 0.000767   | 1.7e-09 | 20      |
 | JSON::Create::create_json     | array_int_10    | 1.333e+06  | 0.0007502  | 4.4e-11 | 25      |
 | Cpanel::JSON::XS::encode_json | num             | 1.44e+06   | 0.000696   | 1.3e-09 | 20      |
 | Data::MessagePack::pack       | array_int_10    | 1.44e+06   | 0.000694   | 8.3e-10 | 20      |
 | JSON::MaybeXS::encode_json    | num             | 1.48e+06   | 0.000677   | 9e-10   | 27      |
 | JSON::XS::encode_json         | num             | 1.63e+06   | 0.000612   | 8.3e-10 | 20      |
 | Sereal::encode_sereal         | str1k           | 1.7e+06    | 0.0006     | 2.3e-09 | 27      |
 | Sereal::encode_sereal         | str1k           | 1.94e+06   | 0.000515   | 1e-09   | 20      |
 | JSON::Create::create_json     | num             | 2.17e+06   | 0.000461   | 8.7e-10 | 20      |
 | Data::MessagePack::pack       | str1k           | 2.35e+06   | 0.000426   | 2.1e-10 | 20      |
 | Sereal::encode_sereal         | num             | 2.4e+06    | 0.00042    | 1.5e-09 | 24      |
 | Data::MessagePack::pack       | str1k           | 2.828e+06  | 0.0003536  | 5.7e-11 | 20      |
 | Sereal::encode_sereal         | undef           | 2.88e+06   | 0.000347   | 8.4e-10 | 20      |
 | Data::MessagePack::pack       | num             | 4.31e+06   | 0.000232   | 4.3e-10 | 20      |
 | Cpanel::JSON::XS::encode_json | undef           | 4.71e+06   | 0.000212   | 9.9e-11 | 22      |
 | Data::MessagePack::pack       | undef           | 5e+06      | 0.0002     | 1.4e-09 | 20      |
 | JSON::MaybeXS::encode_json    | undef           | 5.02e+06   | 0.000199   | 1.3e-10 | 26      |
 | JSON::XS::encode_json         | undef           | 5.07e+06   | 0.000197   | 4.2e-10 | 20      |
 | JSON::Create::create_json     | undef           | 5.44e+06   | 0.000184   | 9.1e-11 | 26      |
 +-------------------------------+-----------------+------------+------------+---------+---------+


Benchmark deserializing (C<< bencher -m Serializers --include-participant-tags-json '["deserialize"]' >>):

 +---------------------------------+---------------------+-----------+-----------+---------+---------+
 | participant                     | dataset             | rate (/s) | time (ms) | errors  | samples |
 +---------------------------------+---------------------+-----------+-----------+---------+---------+
 | JSON::Decode::Regexp::from_json | json:hash_int_1000  | 6.5       | 1.5e+02   | 0.0012  | 20      |
 | Pegex::JSON                     | json:hash_int_1000  | 15.1      | 66.2      | 7e-05   | 20      |
 | JSON::Decode::Marpa::from_json  | json:hash_int_1000  | 17.3      | 57.9      | 6.4e-05 | 20      |
 | Pegex::JSON                     | json:array_int_1000 | 26        | 38        | 0.00014 | 20      |
 | JSON::Decode::Regexp::from_json | json:array_int_1000 | 35        | 28        | 0.00012 | 20      |
 | JSON::Decode::Marpa::from_json  | json:array_int_1000 | 38.4      | 26.1      | 4.3e-05 | 20      |
 | JSON::PP::decode_json           | json:hash_int_1000  | 86.1      | 11.6      | 2.8e-05 | 20      |
 | JSON::Decode::Marpa::from_json  | json:array_str1k_10 | 90.2      | 11.1      | 2.2e-05 | 21      |
 | JSON::PP::decode_json           | json:array_str1k_10 | 1.3e+02   | 7.7       | 2.5e-05 | 20      |
 | Pegex::JSON                     | json:hash_int_100   | 145       | 6.91      | 1.5e-05 | 21      |
 | JSON::Decode::Marpa::from_json  | json:hash_int_100   | 156       | 6.43      | 1.8e-05 | 20      |
 | JSON::PP::decode_json           | json:array_int_1000 | 172       | 5.83      | 5.1e-06 | 20      |
 | Pegex::JSON                     | json:array_int_100  | 247       | 4.05      | 1.2e-05 | 20      |
 | JSON::Decode::Regexp::from_json | json:hash_int_100   | 3.1e+02   | 3.3       | 1.2e-05 | 20      |
 | JSON::Decode::Marpa::from_json  | json:array_int_100  | 317       | 3.15      | 6.5e-06 | 20      |
 | Pegex::JSON                     | json:array_str1k_10 | 6e+02     | 1.7       | 6.5e-06 | 20      |
 | JSON::Decode::Marpa::from_json  | json:hash_int_10    | 732       | 1.37      | 3.4e-06 | 20      |
 | JSON::Decode::Regexp::from_json | json:array_int_100  | 876       | 1.14      | 9.1e-07 | 20      |
 | Pegex::JSON                     | json:hash_int_10    | 903       | 1.11      | 1.3e-06 | 20      |
 | JSON::PP::decode_json           | json:hash_int_100   | 936       | 1.07      | 6.4e-07 | 20      |
 | JSON::Decode::Marpa::from_json  | json:array_int_10   | 968       | 1.03      | 3.1e-06 | 20      |
 | Pegex::JSON                     | json:array_int_10   | 1.2e+03   | 0.84      | 3.1e-06 | 20      |
 | JSON::PP::decode_json           | json:str1k          | 1.31e+03  | 0.765     | 6.4e-07 | 20      |
 | JSON::PP::decode_json           | json:array_int_100  | 1.8e+03   | 0.55      | 2.4e-06 | 20      |
 | Pegex::JSON                     | json:str1k          | 1.95e+03  | 0.513     | 4.3e-07 | 20      |
 | Pegex::JSON                     | json:null           | 2.15e+03  | 0.465     | 5.9e-07 | 20      |
 | Pegex::JSON                     | json:num            | 2.2e+03   | 0.45      | 1.5e-06 | 20      |
 | JSON::Decode::Regexp::from_json | json:hash_int_10    | 3.51e+03  | 0.285     | 2.1e-07 | 20      |
 | JSON::Decode::Regexp::from_json | json:array_str1k_10 | 3.51e+03  | 0.285     | 4.8e-07 | 20      |
 | JSON::Decode::Regexp::from_json | json:array_int_10   | 5.8e+03   | 0.17      | 6.8e-07 | 21      |
 | JSON::Parse::parse_json         | json:hash_int_1000  | 6.6e+03   | 0.15      | 4.8e-07 | 20      |
 | Cpanel::JSON::XS::decode_json   | json:hash_int_1000  | 7.06e+03  | 0.142     | 2.1e-07 | 20      |
 | JSON::MaybeXS::decode_json      | json:hash_int_1000  | 7.07e+03  | 0.141     | 2.1e-07 | 20      |
 | JSON::XS::decode_json           | json:hash_int_1000  | 7.19e+03  | 0.139     | 2.1e-07 | 20      |
 | JSON::PP::decode_json           | json:hash_int_10    | 9.01e+03  | 0.111     | 2.1e-07 | 20      |
 | JSON::Decode::Regexp::from_json | json:str1k          | 1.16e+04  | 0.086     | 8e-08   | 20      |
 | JSON::Decode::Regexp::from_json | json:num            | 1.25e+04  | 0.0802    | 2.1e-07 | 21      |
 | JSON::Decode::Regexp::from_json | json:null           | 1.5e+04   | 0.065     | 2.1e-07 | 20      |
 | JSON::PP::decode_json           | json:array_int_10   | 1.7e+04   | 0.0588    | 2.6e-08 | 21      |
 | JSON::Parse::parse_json         | json:array_int_1000 | 2.57e+04  | 0.0389    | 1.2e-07 | 20      |
 | Cpanel::JSON::XS::decode_json   | json:array_int_1000 | 2.97e+04  | 0.0336    | 5.3e-08 | 20      |
 | JSON::MaybeXS::decode_json      | json:array_int_1000 | 29788.6   | 0.0335699 | 4.7e-11 | 23      |
 | JSON::XS::decode_json           | json:array_int_1000 | 30101.6   | 0.0332208 | 4.6e-11 | 32      |
 | JSON::Parse::parse_json         | json:array_str1k_10 | 3.49e+04  | 0.0287    | 3.8e-08 | 22      |
 | Cpanel::JSON::XS::decode_json   | json:array_str1k_10 | 5.06e+04  | 0.0198    | 2.6e-08 | 21      |
 | JSON::MaybeXS::decode_json      | json:array_str1k_10 | 5.14e+04  | 0.0195    | 6.7e-09 | 20      |
 | JSON::XS::decode_json           | json:array_str1k_10 | 6.64e+04  | 0.0151    | 5.4e-09 | 30      |
 | JSON::PP::decode_json           | json:num            | 6.7e+04   | 0.015     | 8e-08   | 20      |
 | JSON::Parse::parse_json         | json:hash_int_100   | 8.63e+04  | 0.0116    | 1.3e-08 | 20      |
 | JSON::MaybeXS::decode_json      | json:hash_int_100   | 8.86e+04  | 0.0113    | 1.3e-08 | 20      |
 | Cpanel::JSON::XS::decode_json   | json:hash_int_100   | 8.93e+04  | 0.0112    | 1.3e-08 | 22      |
 | JSON::XS::decode_json           | json:hash_int_100   | 8.95e+04  | 0.0112    | 1.3e-08 | 20      |
 | JSON::PP::decode_json           | json:null           | 1.28e+05  | 0.00779   | 1.7e-08 | 20      |
 | JSON::MaybeXS::decode_json      | json:array_int_100  | 2.35e+05  | 0.00426   | 1.7e-09 | 20      |
 | JSON::XS::decode_json           | json:array_int_100  | 2.4e+05   | 0.00417   | 1.6e-09 | 21      |
 | JSON::Parse::parse_json         | json:array_int_100  | 2.422e+05 | 0.004129  | 1.2e-09 | 37      |
 | Cpanel::JSON::XS::decode_json   | json:array_int_100  | 2.43e+05  | 0.00411   | 6.7e-09 | 20      |
 | JSON::Parse::parse_json         | json:str1k          | 3.67e+05  | 0.00273   | 3.5e-09 | 28      |
 | JSON::MaybeXS::decode_json      | json:str1k          | 4.41e+05  | 0.00227   | 4.2e-09 | 20      |
 | Cpanel::JSON::XS::decode_json   | json:str1k          | 5e+05     | 0.002     | 7.3e-09 | 21      |
 | JSON::XS::decode_json           | json:str1k          | 6.34e+05  | 0.00158   | 8.3e-10 | 20      |
 | Cpanel::JSON::XS::decode_json   | json:hash_int_10    | 8.25e+05  | 0.00121   | 1.3e-09 | 20      |
 | JSON::Parse::parse_json         | json:hash_int_10    | 8.54e+05  | 0.00117   | 4.1e-10 | 22      |
 | JSON::MaybeXS::decode_json      | json:hash_int_10    | 8.64e+05  | 0.00116   | 4.2e-10 | 20      |
 | JSON::XS::decode_json           | json:hash_int_10    | 8.65e+05  | 0.00116   | 4.2e-10 | 20      |
 | JSON::MaybeXS::decode_json      | json:num            | 9.46e+05  | 0.00106   | 3.2e-09 | 28      |
 | JSON::XS::decode_json           | json:num            | 9.7e+05   | 0.001     | 8.5e-09 | 21      |
 | Cpanel::JSON::XS::decode_json   | json:num            | 9.86e+05  | 0.00101   | 1.7e-09 | 20      |
 | JSON::Parse::parse_json         | json:num            | 1.06e+06  | 0.000946  | 4.3e-10 | 21      |
 | JSON::MaybeXS::decode_json      | json:array_int_10   | 1.4e+06   | 0.00073   | 9.4e-09 | 20      |
 | JSON::XS::decode_json           | json:array_int_10   | 1.408e+06 | 0.0007104 | 4.7e-11 | 26      |
 | Cpanel::JSON::XS::decode_json   | json:array_int_10   | 1.46e+06  | 0.000687  | 6.8e-10 | 30      |
 | JSON::Parse::parse_json         | json:array_int_10   | 1.47e+06  | 0.000679  | 8.5e-10 | 20      |
 | JSON::MaybeXS::decode_json      | json:null           | 6.4e+06   | 0.00016   | 1.8e-09 | 20      |
 | Cpanel::JSON::XS::decode_json   | json:null           | 6.5e+06   | 0.00015   | 1.8e-09 | 20      |
 | JSON::XS::decode_json           | json:null           | 7.07e+06  | 0.000141  | 5.3e-11 | 20      |
 | JSON::Parse::parse_json         | json:null           | 7.7e+06   | 0.00013   | 1.3e-09 | 20      |
 +---------------------------------+---------------------+-----------+-----------+---------+---------+


Benchmark module startup overhead:

 +----------------------+-----------+------------------------+---------+---------+
 | participant          | time (ms) | mod_overhead_time (ms) | errors  | samples |
 +----------------------+-----------+------------------------+---------+---------+
 | JSON::Decode::Marpa  | 1.3e+02   | 124.2                  | 0.00058 | 22      |
 | YAML::XS             | 53        | 47.2                   | 0.00026 | 20      |
 | JSON::PP             | 25.7      | 19.9                   | 6.6e-05 | 20      |
 | Pegex::JSON          | 23        | 17.2                   | 7.2e-05 | 20      |
 | JSON::MaybeXS        | 18        | 12.2                   | 5.6e-05 | 21      |
 | Sereal               | 15.8      | 10                     | 2.7e-05 | 20      |
 | Storable             | 15.8      | 10                     | 3.9e-05 | 20      |
 | YAML::Old            | 15.1      | 9.3                    | 4.6e-05 | 20      |
 | JSON::XS             | 13        | 7.2                    | 6.3e-05 | 20      |
 | JSON::Parse          | 13        | 7.2                    | 6.6e-05 | 20      |
 | YAML::Syck           | 12.5      | 6.7                    | 3.2e-05 | 20      |
 | Cpanel::JSON::XS     | 12        | 6.2                    | 8e-05   | 20      |
 | JSON::Create         | 10        | 4.2                    | 6e-05   | 20      |
 | JSON::Decode::Regexp | 9.3       | 3.5                    | 3.2e-05 | 20      |
 | Data::MessagePack    | 9.14      | 3.34                   | 2.6e-05 | 20      |
 | perl -e1 (baseline)  | 5.8       | 0                      | 3.5e-05 | 20      |
 +----------------------+-----------+------------------------+---------+---------+

=head1 DESCRIPTION

Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenario-Serializers>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenario-Serializers>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenario-Serializers>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
