package App::lcpan::Cmd::debian_dist2deb;

our $DATE = '2017-01-16'; # DATE
our $VERSION = '0.002'; # VERSION

use 5.010001;
use strict;
use warnings;
use Log::Any::IfLOG '$log';

require App::lcpan;

our %SPEC;

$SPEC{handle_cmd} = {
    v => 1.1,
    summary => 'Convert dist name to Debian package name',
    description => <<'_',

This routine uses the simple rule of: converting the dist name to lowercase then
add "lib" prefix and "-perl" suffix. A small percentage of Perl distributions do
not follow this rule.

_
    args => {
        %App::lcpan::common_args,
        %App::lcpan::dists_args,
        check_exists => {
            summary => 'Check each distribution if its Debian package exists, using Dist::Util::Debian::dist_has_deb',
            schema => 'bool*',
        },
        use_allpackages => {
            summary => 'Will be passed to Dist::Util::Debian::dist_has_deb',
            description => <<'_',

Using this option is faster if you need to check existence for many Debian
packages. See <pm:Dist::Util::Debian> documentation for more details.

_
            schema => 'bool*',
        },
    },
};
sub handle_cmd {
    require Dist::Util::Debian;

    my %args = @_;

    my $state = App::lcpan::_init(\%args, 'ro');
    my $dbh = $state->{dbh};

    my @rows;
    my @fields = qw(dist deb);

    push @fields, "exists" if $args{check_exists};

    my $opts = {};
    $opts->{use_allpackages} = 1 if $args{use_allpackages};

    for my $dist (@{ $args{dists} }) {
        my $deb = Dist::Util::Debian::dist2deb($dist);
        my $row = {dist => $dist, deb => $deb};
        if ($args{check_exists}) {
            $row->{exists} = Dist::Util::Debian::deb_exists($opts, $deb);
        }
        push @rows, $row;
    }
    [200, "OK", \@rows, {'table.fields' => \@fields}];
}

1;
# ABSTRACT: Convert dist name to Debian package name

__END__

=pod

=encoding UTF-8

=head1 NAME

App::lcpan::Cmd::debian_dist2deb - Convert dist name to Debian package name

=head1 VERSION

This document describes version 0.002 of App::lcpan::Cmd::debian_dist2deb (from Perl distribution App-lcpan-CmdBundle-debian), released on 2017-01-16.

=head1 FUNCTIONS


=head2 handle_cmd(%args) -> [status, msg, result, meta]

Convert dist name to Debian package name.

This routine uses the simple rule of: converting the dist name to lowercase then
add "lib" prefix and "-perl" suffix. A small percentage of Perl distributions do
not follow this rule.

This function is not exported.

Arguments ('*' denotes required arguments):

=over 4

=item * B<check_exists> => I<bool>

Check each distribution if its Debian package exists, using Dist::Util::Debian::dist_has_deb.

=item * B<cpan> => I<dirname>

Location of your local CPAN mirror, e.g. /path/to/cpan.

Defaults to C<~/cpan>.

=item * B<dists>* => I<array[perl::distname]>

=item * B<index_name> => I<filename> (default: "index.db")

Filename of index.

=item * B<use_allpackages> => I<bool>

Will be passed to Dist::Util::Debian::dist_has_deb.

Using this option is faster if you need to check existence for many Debian
packages. See L<Dist::Util::Debian> documentation for more details.

=back

Returns an enveloped result (an array).

First element (status) is an integer containing HTTP status code
(200 means OK, 4xx caller error, 5xx function error). Second element
(msg) is a string containing error message, or 'OK' if status is
200. Third element (result) is optional, the actual result. Fourth
element (meta) is called result metadata and is optional, a hash
that contains extra information.

Return value:  (any)

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/App-lcpan-CmdBundle-debian>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-App-lcpan-CmdBundle-debian>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=App-lcpan-CmdBundle-debian>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2017 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
