package App::BashHistoryUtils;

our $DATE = '2015-11-04'; # DATE
our $VERSION = '0.01'; # VERSION

use 5.010001;
use strict;
use warnings;

our %SPEC;

our %arg_histfile = (
    histfile => {
        schema => 'str*',
        default => ($ENV{HISTFILE} // "$ENV{HOME}/.bash_history"),
        cmdline_aliases => {f=>{}},
        'x.completion' => ['filename'],
    },
);

our %args_filtering = (
    pattern => {
        summary => 'Match entries using a regex pattern',
        schema => 're*',
        cmdline_aliases => {p=>{}},
        tags => ['category:filtering'],
        pos => 0,
    },
    max_age => {
        summary => 'Match entries older than a certain age',
        schema => 'duration*',
        'x.perl.coerce_to' => 'int(secs)',
        tags => ['category:filtering'],
    },
    min_age => {
        summary => 'Match entries younger than a certain age',
        schema => 'duration*',
        'x.perl.coerce_to' => 'int(secs)',
        tags => ['category:filtering'],
    },
    ignore_case => {
        schema => ['bool', is=>1],
        cmdline_aliases => {i=>{}},
    },
    invert_match => {
        schema => ['bool', is=>1],
        #cmdline_aliases => {v=>{}}, # clashes with --version
    },
);

sub _do {
    require Bash::History::Read;
    require Capture::Tiny;
    require Cwd;
    #require File::Temp;

    my $which = shift;
    my %args = @_;

    my $histfile = $args{histfile};
    return [412, "Can't find '$histfile': $!"] unless -f $histfile;
    my $realhistfile = Cwd::realpath($args{histfile})
        or return [412, "Can't find realpath of '$histfile': $!"];

    my $pat;
    if (defined $args{pattern}) {
        if ($args{ignore_case}) {
            $pat = qr/$args{pattern}/i;
        } else {
            $pat = qr/$args{pattern}/;
        }
    }

    local @ARGV = ($histfile);
    my $now = time;
    my $stdout = Capture::Tiny::capture_stdout(
        sub {
            Bash::History::Read::each_hist(
                sub {
                    if (defined($args{max_age}) &&
                            $main::TS < $now-$args{max_age}) {
                        $main::PRINT = 0;
                    }
                    if (defined($args{min_age}) &&
                            $main::TS > $now-$args{min_age}) {
                        $main::PRINT = 0;
                    }
                    if ($pat && $_ =~ $pat) {
                        $main::PRINT = 0;
                    }

                    if ($which eq 'grep') {
                        $main::PRINT = !$main::PRINT;
                    }
                    if ($args{invert_match}) {
                        $main::PRINT = !$main::PRINT;
                    }
                });
        });

    if ($which eq 'grep' ||
            $which eq 'delete' && $args{-dry_run}) {
        return [200,"OK", $stdout, {'cmdline.skip_format'=>1}];
    } elsif ($which eq 'delete') {
        my $tempfile = "$realhistfile.tmp.$$";
        open my($fh), ">", $tempfile
            or return [500, "Can't open temporary file '$tempfile': $!"];

        print $fh $stdout
            or return [500, "Can't write (1) to temporary file '$tempfile': $!"];

        close $fh
            or return [500, "Can't write (2) to temporary file '$tempfile': $!"];

        rename $tempfile, $realhistfile
            or return [500, "Can't replace temporary file '$tempfile' to '$realhistfile': $!"];
    }

    [200,"OK"];
}

$SPEC{grep_bash_history_entries} = {
    v => 1.1,
    summary => 'Show matching entries from bash history file',
    args => {
        %arg_histfile,
        %args_filtering,
    },
};
sub grep_bash_history_entries {
    _do('grep', @_);
}

$SPEC{delete_bash_history_entries} = {
    v => 1.1,
    summary => 'Delete matching entries from bash history file',
    args => {
        %arg_histfile,
        %args_filtering,
    },
    features => {
        dry_run => 1,
    },
};
sub delete_bash_history_entries {
    _do('delete', @_);
}

1;
# ABSTRACT: CLI utilities related to bash history file

__END__

=pod

=encoding UTF-8

=head1 NAME

App::BashHistoryUtils - CLI utilities related to bash history file

=head1 VERSION

This document describes version 0.01 of App::BashHistoryUtils (from Perl distribution App-BashHistoryUtils), released on 2015-11-04.

=head1 DESCRIPTION

This distribution includes the following CLI utilities:

=over

=item * L<delete-bash-history-entries>

=item * L<grep-bash-history-entries>

=back

=head1 FUNCTIONS


=head2 delete_bash_history_entries(%args) -> [status, msg, result, meta]

Delete matching entries from bash history file.

This function is not exportable.

This function supports dry-run operation.


Arguments ('*' denotes required arguments):

=over 4

=item * B<histfile> => I<str> (default: "/home/s1/.bash_history")

=item * B<ignore_case> => I<bool>

=item * B<invert_match> => I<bool>

=item * B<max_age> => I<duration>

Match entries older than a certain age.

=item * B<min_age> => I<duration>

Match entries younger than a certain age.

=item * B<pattern> => I<re>

Match entries using a regex pattern.

=back

Special arguments:

=over 4

=item * B<-dry_run> => I<bool>

Pass -dry_run=>1 to enable simulation mode.

=back

Returns an enveloped result (an array).

First element (status) is an integer containing HTTP status code
(200 means OK, 4xx caller error, 5xx function error). Second element
(msg) is a string containing error message, or 'OK' if status is
200. Third element (result) is optional, the actual result. Fourth
element (meta) is called result metadata and is optional, a hash
that contains extra information.

Return value:  (any)


=head2 grep_bash_history_entries(%args) -> [status, msg, result, meta]

Show matching entries from bash history file.

This function is not exportable.

Arguments ('*' denotes required arguments):

=over 4

=item * B<histfile> => I<str> (default: "/home/s1/.bash_history")

=item * B<ignore_case> => I<bool>

=item * B<invert_match> => I<bool>

=item * B<max_age> => I<duration>

Match entries older than a certain age.

=item * B<min_age> => I<duration>

Match entries younger than a certain age.

=item * B<pattern> => I<re>

Match entries using a regex pattern.

=back

Returns an enveloped result (an array).

First element (status) is an integer containing HTTP status code
(200 means OK, 4xx caller error, 5xx function error). Second element
(msg) is a string containing error message, or 'OK' if status is
200. Third element (result) is optional, the actual result. Fourth
element (meta) is called result metadata and is optional, a hash
that contains extra information.

Return value:  (any)

=head1 SEE ALSO

L<Bash::History::Read>

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/App-BashHistoryUtils>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-App-BashHistoryUtils>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=App-BashHistoryUtils>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
