package Acme::CPANModules::TextTable;

our $AUTHORITY = 'cpan:PERLANCAR'; # AUTHORITY
our $DATE = '2021-07-26'; # DATE
our $DIST = 'Acme-CPANModules-TextTable'; # DIST
our $VERSION = '0.011'; # VERSION

use 5.010001;
use strict;
use warnings;
use utf8;

sub _make_table {
    my ($cols, $rows, $celltext) = @_;
    my $res = [];
    push @$res, [];
    for (0..$cols-1) { $res->[0][$_] = "col" . ($_+1) }
    for my $row (1..$rows) {
        push @$res, [ map { $celltext // "row$row.$_" } 1..$cols ];
    }
    $res;
}

our $LIST = {
    summary => 'Modules that generate text tables',
    entry_features => {
        wide_char => {summary => 'Whether the use of wide characters (e.g. Kanji) in cells does not cause the table to be misaligned'},
        color_data =>  {summary => 'Whether module supports ANSI colors (i.e. text with ANSI color codes can still be aligned properly)'},
        multiline_data => {summary => 'Whether module supports aligning data cells that contain newlines'},

        box_char => {summary => 'Whether module can utilize box-drawing characters'},
        custom_border => {summary => 'Whether module allows customizing border in some way'},

        align_row => {summary => "Whether module supports aligning text horizontally in a row (left/right/middle)"},
        align_column => {summary => "Whether module supports aligning text horizontally in a column (left/right/middle)"},
        align_cell => {summary => "Whether module supports aligning text horizontally in individual cells (left/right/middle)"},

        valign_row => {summary => "Whether module supports aligning text vertically in a row (top/bottom/middle)"},
        valign_column => {summary => "Whether module supports aligning text vertically in a column (top/bottom/middle)"},
        valign_cell => {summary => "Whether module supports aligning text vertically in individual cells (top/bottom/middle)"},

        rowspan => {summary => "Whether module supports row spans"},
        colspan => {summary => "Whether module supports column spans"},

        custom_color => {summary => 'Whether the module produces colored table and supports customizing color in some way'},
        color_theme => {summary => 'Whether the module supports color theme/scheme'},

        speed => {summary => "Rendering speed", schema=>'str*'},

        column_width => {summary => 'Whether module allows setting the width of columns'},
        per_column_width => {summary => 'Whether module allows setting column width on a per-column basis'},
        row_height => {summary => 'Whether module allows setting the height of rows'},
        per_row_height => {summary => 'Whether module allows setting row height on a per-row basis'},

        pad => {summary => 'Whether module allows customizing cell horizontal padding'},
        vpad => {summary => 'Whether module allows customizing cell vertical padding'},
    },
    entries => [
        {
            module => 'Text::Table::Any',
            description => <<'_',

This is a frontend for many text table modules as backends. The interface is
dead simple, following <pm:Text::Table::Tiny>. The main drawback is that it
currently does not allow passing (some, any) options to each backend.

_
            bench_code => sub {
                my ($table) = @_;
                Text::Table::Any::table(rows=>$table, header_row=>1);
            },
            features => {
                align_cell     => {value=>undef, summary=>"Depends on backend"},
                align_column   => {value=>undef, summary=>"Depends on backend"},
                align_row      => {value=>undef, summary=>"Depends on backend"},
                box_char       => {value=>undef, summary=>"Depends on backend"},
                color_data     => {value=>undef, summary=>"Depends on backend"},
                color_theme    => {value=>undef, summary=>"Depends on backend"},
                colspan        => {value=>undef, summary=>"Depends on backend"},
                custom_border  => {value=>undef, summary=>"Depends on backend"},
                custom_color   => {value=>undef, summary=>"Depends on backend"},
                multiline_data => {value=>undef, summary=>"Depends on backend"},
                rowspan        => {value=>undef, summary=>"Depends on backend"},
                speed          => {value=>undef, summary=>"Depends on backend"},
                valign_cell    => {value=>undef, summary=>"Depends on backend"},
                valign_column  => {value=>undef, summary=>"Depends on backend"},
                valign_row     => {value=>undef, summary=>"Depends on backend"},
                wide_char_data => {value=>undef, summary=>"Depends on backend"},
            },
        },

        {
            module => 'Text::UnicodeBox::Table',
            description => <<'_',

The main feature of this module is the various border style it provides drawn
using Unicode box-drawing characters. It allows per-row style. The rendering
speed is particularly slow compared to other modules.

_
            bench_code => sub {
                my ($table) = @_;
                my $t = Text::UnicodeBox::Table->new;
                $t->add_header(@{ $table->[0] });
                $t->add_row(@{ $table->[$_] }) for 1..$#{$table};
                $t->render;
            },
            features => {
                align_cell => 0,
                align_column => 1,
                box_char => 0,
                color_data => 1,
                color_theme => 0,
                colspan => 0,
                custom_border => 1,
                custom_color => 0,
                multiline_data => 0,
                rowspan => 0,
                wide_char_data => 1,
                speed => "slow",
            },
        },

        {
            module => 'Text::Table::Manifold',
            description => <<'_',

Two main features of this module is per-column aligning and wide character
support. This module, aside from doing its rendering, can also be told to pass
rendering to HTML, CSV, or other text table module like
<pm:Text::UnicodeBox::Table>); so in this way it is similar to
<pm:Text::Table::Any>.

_
            bench_code => sub {
                my ($table) = @_;
                my $t = Text::Table::Manifold->new;
                $t->headers($table->[0]);
                $t->data([ @{$table}[1 .. $#{$table}] ]);
                join("\n", @{$t->render(padding => 1)}) . "\n";
            },
            features => {
                align_cell => 0,
                align_column => 1,
                box_char => undef, # ?
                color_data => 1,
                color_theme => 0,
                colspan => 0,
                custom_border => {value=>0, summary=>"But this module can pass rendering to other module like Text::UnicodeBox::Table"},
                custom_color => 0,
                multiline_data => 0,
                rowspan => 0,
                wide_char_data => 1,
            },
        },

        {
            module => 'Text::ANSITable',
            description => <<'_',

This 2013 project was my take in creating a text table module that can handle
color, multiline text, wide characters. I also threw in various formatting
options, e.g. per-column/row/cell align/valign/pad/vpad, conditional formatting,
and so on. I even added a couple of features I never used: hiding rows and
specifying columns to display which can be in different order from the original
specified columns or can contain the same original columns multiple times. I
think this module offers the most formatting options on CPAN.

In early 2021, I needed colspan/rowspan and I implemented this in a new module:
<pm:Text::Table::Span> (later renamed to <pm:Text::Table::More>). I plan to add
this feature too to Text::ANSITable, but in the meantime I'm also adding more
formatting options which I need to Text::Table::More.

_
            bench_code => sub {
                my ($table) = @_;
                my $t = Text::ANSITable->new(
                    use_utf8 => 0,
                    use_box_chars => 0,
                    use_color =>  0,
                    columns => $table->[0],
                    border_style => 'ASCII::SingleLine',
                );
                $t->add_row($table->[$_]) for 1..@$table-1;
                $t->draw;
            },
            features => {
                align_cell => 1,
                align_column => 1,
                align_row => 1,
                box_char => 1,
                color_data =>  1,
                color_theme => 1,
                colspan => 0,
                column_width => 1,
                custom_border => 1,
                custom_color => 1,
                multiline_data => 1,
                pad => 1,
                per_column_width => 1,
                per_row_height => 1,
                row_height => 1,
                rowspan => 0,
                speed => "slow",
                valign_cell => 1,
                valign_column => 1,
                valign_row => 1,
                vpad => 1,
                wide_char_data => 1,
            },
        },

        {
            module => 'Text::ASCIITable',
            bench_code => sub {
                my ($table) = @_;
                my $t = Text::ASCIITable->new();
                $t->setCols(@{ $table->[0] });
                $t->addRow(@{ $table->[$_] }) for 1..@$table-1;
                "$t";
            },
            features => {
                wide_char_data => 0,
                color_data =>  0,
                box_char => 0,
                multiline_data => 1,
            },
        },
        {
            module => 'Text::FormatTable',
            bench_code => sub {
                my ($table) = @_;
                my $t = Text::FormatTable->new(join('|', ('l') x @{ $table->[0] }));
                $t->head(@{ $table->[0] });
                $t->row(@{ $table->[$_] }) for 1..@$table-1;
                $t->render;
            },
            features => {
                wide_char_data => 0,
                color_data =>  0,
                box_char => 0,
                multiline_data => 1,
            },
        },
        {
            module => 'Text::MarkdownTable',
            bench_code => sub {
                my ($table) = @_;
                my $out = "";
                my $t = Text::MarkdownTable->new(file => \$out);
                my $fields = $table->[0];
                foreach (1..@$table-1) {
                    my $row = $table->[$_];
                    $t->add( {
                        map { $fields->[$_] => $row->[$_] } 0..@$fields-1
                    });
                }
                $t->done;
                $out;
            },
            features => {
                wide_char_data => 0,
                color_data =>  0,
                box_char => 0,
                multiline_data => {value=>0, summary=>'Newlines stripped'},
            },
        },
        {
            module => 'Text::Table',
            bench_code => sub {
                my ($table) = @_;
                my $t = Text::Table->new(@{ $table->[0] });
                $t->load(@{ $table }[1..@$table-1]);
                $t;
            },
            features => {
                wide_char_data => 0,
                color_data =>  0,
                box_char => {value=>undef, summary=>'Does not draw borders'},
                multiline_data => 1,
            },
        },
        {
            module => 'Text::Table::Tiny',
            bench_code => sub {
                my ($table) = @_;
                Text::Table::Tiny::table(rows=>$table, header_row=>1);
            },
            features => {
                wide_char_data => 1,
                color_data =>  1,
                box_char => 1,
                multiline_data => 0,
            },
        },
        {
            module => 'Text::Table::TinyBorderStyle',
            bench_code => sub {
                my ($table) = @_;
                Text::Table::TinyBorderStyle::table(rows=>$table, header_row=>1);
            },
            features => {
                wide_char_data => 0,
                color_data =>  0,
                box_char => 1,
                multiline_data => 0,
            },
        },
        {
            module => 'Text::Table::More',
            description => <<'_',

A module I wrote in early 2021. Main distinguishing feature is support for
rowspan/clospan. I plan to add more features to this module on an as-needed
basic. This module is now preferred to <pm:Text::ANSITable>, although currently
it does not nearly as many formatting options as Text::ANSITable.

_
            bench_code => sub {
                my ($table) = @_;
                Text::Table::More::generate_table(rows=>$table, header_row=>1);
            },
            features => {
                align_cell => 1,
                align_column => 1,
                align_row => 1,
                box_char => 1,
                color_data =>  1,
                color_theme => 0,
                colspan => 1,
                custom_border => 1,
                custom_color => 0,
                multiline_data => 1,
                rowspan => 1,
                speed => "slow",
                valign_cell => 1,
                valign_column => 1,
                valign_row => 1,
                wide_char_data => 1,
                column_width => 0, # todo
                per_column_width => 0, # todo
                row_height => 0, # todo
                per_row_height => 0, # todo
                pad => 0, # todo
                vpad => 0, # todo
            },
        },
        {
            module => 'Text::Table::Sprintf',
            description => <<'_',

A performant (see benchmark result) and lightweight (a page of code, no use of
modules at all), but with minimal extra features.

_
            bench_code => sub {
                my ($table) = @_;
                Text::Table::Sprintf::table(rows=>$table, header_row=>1);
            },
            features => {
                box_char => 0,
                color_data =>  0,
                multiline_data => 0,
                speed => "fast",
                wide_char_data => 0,
            },
        },
        {
            module => 'Text::Table::TinyColor',
            bench_code => sub {
                my ($table) = @_;
                Text::Table::TinyColor::table(rows=>$table, header_row=>1);
            },
            features => {
                wide_char_data => 0,
                color_data =>  1,
                box_char => 0,
                multiline_data => 0,
            },
        },
        {
            module => 'Text::Table::TinyColorWide',
            bench_code => sub {
                my ($table) = @_;
                Text::Table::TinyColorWide::table(rows=>$table, header_row=>1);
            },
            features => {
                wide_char_data => 1,
                color_data =>  1,
                box_char => 0,
                multiline_data => 0,
            },
        },
        {
            module => 'Text::Table::TinyWide',
            bench_code => sub {
                my ($table) = @_;
                Text::Table::TinyWide::table(rows=>$table, header_row=>1);
            },
            features => {
                wide_char_data => 1,
                color_data =>  0,
                box_char => 0,
            },
        },
        {
            module => 'Text::Table::Org',
            bench_code => sub {
                my ($table) = @_;
                Text::Table::Org::table(rows=>$table, header_row=>1);
            },
            features => {
                wide_char_data => 0,
                color_data =>  0,
                box_char => 0,
                multiline_data => 0,
            },
        },
        {
            module => 'Text::Table::CSV',
            bench_code => sub {
                my ($table) = @_;
                Text::Table::CSV::table(rows=>$table, header_row=>1);
            },
            features => {
                wide_char_data => 1,
                color_data =>  0,
                box_char => {value=>undef, summary=>"Irrelevant"},
                multiline_data => {value=>1, summary=>"But make sure your CSV parser can handle multiline cell"},
            },
        },
        {
            module => 'Text::Table::HTML',
            bench_code => sub {
                my ($table) = @_;
                Text::Table::HTML::table(rows=>$table, header_row=>1);
            },
            features => {
                wide_char_data => 1,
                color_data =>  {value=>0, summary=>'Not converted to HTML color elements'},
                box_char => 0,
                multiline_data => 1,
            },
        },
        {
            module => 'Text::Table::HTML::DataTables',
            bench_code => sub {
                my ($table) = @_;
                Text::Table::HTML::DataTables::table(rows=>$table, header_row=>1);
            },
            features => {
                wide_char_data => 1,
                color_data =>  {value=>0, summary=>'Not converted to HTML color elements'},
                box_char => 0,
                multiline_data => 1,
            },
        },
        {
            module => 'Text::TabularDisplay',
            bench_code => sub {
                my ($table) = @_;
                my $t = Text::TabularDisplay->new(@{ $table->[0] });
                $t->add(@{ $table->[$_] }) for 1..@$table-1;
                $t->render; # doesn't add newline
            },
            features => {
                wide_char_data => 1,
                color_data =>  0,
                box_char => {value=>undef, summary=>"Irrelevant"},
                multiline_data => 1,
            },
        },
    ],

    bench_datasets => [
        {name=>'tiny (1x1)'          , argv => [_make_table( 1, 1)],},
        {name=>'small (3x5)'         , argv => [_make_table( 3, 5)],},
        {name=>'wide (30x5)'         , argv => [_make_table(30, 5)],},
        {name=>'long (3x300)'        , argv => [_make_table( 3, 300)],},
        {name=>'large (30x300)'      , argv => [_make_table(30, 300)],},
        {name=>'multiline data (2x1)', argv => [ [["col1", "col2"], ["foobar\nbaz\nqux\nquux","corge"]] ], include_by_default=>0 },
        {name=>'wide char data (1x2)', argv => [ [["col1"], ["no wide character"], ["宽字"]] ], include_by_default=>0 },
        {name=>'color data (1x2)'    , argv => [ [["col1"], ["no color"], ["\e[31m\e[1mwith\e[0m \e[32m\e[1mcolor\e[0m"]] ], include_by_default=>0 },
    ],

};

1;
# ABSTRACT: Modules that generate text tables

__END__

=pod

=encoding UTF-8

=head1 NAME

Acme::CPANModules::TextTable - Modules that generate text tables

=head1 VERSION

This document describes version 0.011 of Acme::CPANModules::TextTable (from Perl distribution Acme-CPANModules-TextTable), released on 2021-07-26.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher --cpanmodules-module TextTable

To run module startup overhead benchmark:

 % bencher --module-startup --cpanmodules-module TextTable

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 ACME::CPANMODULES ENTRIES

=over

=item L<Text::Table::Any>

Author: L<PERLANCAR|https://metacpan.org/author/PERLANCAR>

This is a frontend for many text table modules as backends. The interface is
dead simple, following L<Text::Table::Tiny>. The main drawback is that it
currently does not allow passing (some, any) options to each backend.


=item L<Text::UnicodeBox::Table>

Author: L<EWATERS|https://metacpan.org/author/EWATERS>

The main feature of this module is the various border style it provides drawn
using Unicode box-drawing characters. It allows per-row style. The rendering
speed is particularly slow compared to other modules.


=item L<Text::Table::Manifold>

Author: L<RSAVAGE|https://metacpan.org/author/RSAVAGE>

Two main features of this module is per-column aligning and wide character
support. This module, aside from doing its rendering, can also be told to pass
rendering to HTML, CSV, or other text table module like
L<Text::UnicodeBox::Table>); so in this way it is similar to
L<Text::Table::Any>.


=item L<Text::ANSITable>

Author: L<PERLANCAR|https://metacpan.org/author/PERLANCAR>

This 2013 project was my take in creating a text table module that can handle
color, multiline text, wide characters. I also threw in various formatting
options, e.g. per-column/row/cell align/valign/pad/vpad, conditional formatting,
and so on. I even added a couple of features I never used: hiding rows and
specifying columns to display which can be in different order from the original
specified columns or can contain the same original columns multiple times. I
think this module offers the most formatting options on CPAN.

In early 2021, I needed colspan/rowspan and I implemented this in a new module:
L<Text::Table::Span> (later renamed to L<Text::Table::More>). I plan to add
this feature too to Text::ANSITable, but in the meantime I'm also adding more
formatting options which I need to Text::Table::More.


=item L<Text::ASCIITable>

Author: L<LUNATIC|https://metacpan.org/author/LUNATIC>

=item L<Text::FormatTable>

Author: L<TREY|https://metacpan.org/author/TREY>

=item L<Text::MarkdownTable>

Author: L<VOJ|https://metacpan.org/author/VOJ>

=item L<Text::Table>

Author: L<SHLOMIF|https://metacpan.org/author/SHLOMIF>

=item L<Text::Table::Tiny>

Author: L<NEILB|https://metacpan.org/author/NEILB>

=item L<Text::Table::TinyBorderStyle>

Author: L<PERLANCAR|https://metacpan.org/author/PERLANCAR>

=item L<Text::Table::More>

Author: L<PERLANCAR|https://metacpan.org/author/PERLANCAR>

A module I wrote in early 2021. Main distinguishing feature is support for
rowspan/clospan. I plan to add more features to this module on an as-needed
basic. This module is now preferred to L<Text::ANSITable>, although currently
it does not nearly as many formatting options as Text::ANSITable.


=item L<Text::Table::Sprintf>

Author: L<PERLANCAR|https://metacpan.org/author/PERLANCAR>

A performant (see benchmark result) and lightweight (a page of code, no use of
modules at all), but with minimal extra features.


=item L<Text::Table::TinyColor>

Author: L<PERLANCAR|https://metacpan.org/author/PERLANCAR>

=item L<Text::Table::TinyColorWide>

Author: L<PERLANCAR|https://metacpan.org/author/PERLANCAR>

=item L<Text::Table::TinyWide>

Author: L<PERLANCAR|https://metacpan.org/author/PERLANCAR>

=item L<Text::Table::Org>

Author: L<PERLANCAR|https://metacpan.org/author/PERLANCAR>

=item L<Text::Table::CSV>

Author: L<PERLANCAR|https://metacpan.org/author/PERLANCAR>

=item L<Text::Table::HTML>

Author: L<PERLANCAR|https://metacpan.org/author/PERLANCAR>

=item L<Text::Table::HTML::DataTables>

Author: L<PERLANCAR|https://metacpan.org/author/PERLANCAR>

=item L<Text::TabularDisplay>

Author: L<DARREN|https://metacpan.org/author/DARREN>

=back

=head1 ACME::CPANMODULES FEATURE COMPARISON MATRIX

 +-------------------------------+----------------+------------------+---------------+--------------+----------------+-----------------+-------------+-------------------+------------------+---------------------+--------------+------------+------------------+--------------------+-----------------+----------------+-------------------+----------+-----------------------+---------------------+-----------------+-----------+
 | module                        | align_cell *1) | align_column *2) | align_row *3) | box_char *4) | color_data *5) | color_theme *6) | colspan *7) | custom_border *8) | custom_color *9) | multiline_data *10) | rowspan *11) | speed *12) | valign_cell *13) | valign_column *14) | valign_row *15) | wide_char_data | column_width *16) | pad *17) | per_column_width *18) | per_row_height *19) | row_height *20) | vpad *21) |
 +-------------------------------+----------------+------------------+---------------+--------------+----------------+-----------------+-------------+-------------------+------------------+---------------------+--------------+------------+------------------+--------------------+-----------------+----------------+-------------------+----------+-----------------------+---------------------+-----------------+-----------+
 | Text::Table::Any              | N/A *22)       | N/A *22)         | N/A *22)      | N/A *22)     | N/A *22)       | N/A *22)        | N/A *22)    | N/A *22)          | N/A *22)         | N/A *22)            | N/A *22)     | N/A *22)   | N/A *22)         | N/A *22)           | N/A *22)        | N/A *22)       | N/A               | N/A      | N/A                   | N/A                 | N/A             | N/A       |
 | Text::UnicodeBox::Table       | no             | yes              | N/A           | no           | yes            | no              | no          | yes               | no               | no                  | no           | slow       | N/A              | N/A                | N/A             | yes            | N/A               | N/A      | N/A                   | N/A                 | N/A             | N/A       |
 | Text::Table::Manifold         | no             | yes              | N/A           | N/A          | yes            | no              | no          | no *23)           | no               | no                  | no           | N/A        | N/A              | N/A                | N/A             | yes            | N/A               | N/A      | N/A                   | N/A                 | N/A             | N/A       |
 | Text::ANSITable               | yes            | yes              | yes           | yes          | yes            | yes             | no          | yes               | yes              | yes                 | no           | slow       | yes              | yes                | yes             | yes            | yes               | yes      | yes                   | yes                 | yes             | yes       |
 | Text::ASCIITable              | N/A            | N/A              | N/A           | no           | no             | N/A             | N/A         | N/A               | N/A              | yes                 | N/A          | N/A        | N/A              | N/A                | N/A             | no             | N/A               | N/A      | N/A                   | N/A                 | N/A             | N/A       |
 | Text::FormatTable             | N/A            | N/A              | N/A           | no           | no             | N/A             | N/A         | N/A               | N/A              | yes                 | N/A          | N/A        | N/A              | N/A                | N/A             | no             | N/A               | N/A      | N/A                   | N/A                 | N/A             | N/A       |
 | Text::MarkdownTable           | N/A            | N/A              | N/A           | no           | no             | N/A             | N/A         | N/A               | N/A              | no *24)             | N/A          | N/A        | N/A              | N/A                | N/A             | no             | N/A               | N/A      | N/A                   | N/A                 | N/A             | N/A       |
 | Text::Table                   | N/A            | N/A              | N/A           | N/A *25)     | no             | N/A             | N/A         | N/A               | N/A              | yes                 | N/A          | N/A        | N/A              | N/A                | N/A             | no             | N/A               | N/A      | N/A                   | N/A                 | N/A             | N/A       |
 | Text::Table::Tiny             | N/A            | N/A              | N/A           | yes          | yes            | N/A             | N/A         | N/A               | N/A              | no                  | N/A          | N/A        | N/A              | N/A                | N/A             | yes            | N/A               | N/A      | N/A                   | N/A                 | N/A             | N/A       |
 | Text::Table::TinyBorderStyle  | N/A            | N/A              | N/A           | yes          | no             | N/A             | N/A         | N/A               | N/A              | no                  | N/A          | N/A        | N/A              | N/A                | N/A             | no             | N/A               | N/A      | N/A                   | N/A                 | N/A             | N/A       |
 | Text::Table::More             | yes            | yes              | yes           | yes          | yes            | no              | yes         | yes               | no               | yes                 | yes          | slow       | yes              | yes                | yes             | yes            | no                | no       | no                    | no                  | no              | no        |
 | Text::Table::Sprintf          | N/A            | N/A              | N/A           | no           | no             | N/A             | N/A         | N/A               | N/A              | no                  | N/A          | fast       | N/A              | N/A                | N/A             | no             | N/A               | N/A      | N/A                   | N/A                 | N/A             | N/A       |
 | Text::Table::TinyColor        | N/A            | N/A              | N/A           | no           | yes            | N/A             | N/A         | N/A               | N/A              | no                  | N/A          | N/A        | N/A              | N/A                | N/A             | no             | N/A               | N/A      | N/A                   | N/A                 | N/A             | N/A       |
 | Text::Table::TinyColorWide    | N/A            | N/A              | N/A           | no           | yes            | N/A             | N/A         | N/A               | N/A              | no                  | N/A          | N/A        | N/A              | N/A                | N/A             | yes            | N/A               | N/A      | N/A                   | N/A                 | N/A             | N/A       |
 | Text::Table::TinyWide         | N/A            | N/A              | N/A           | no           | no             | N/A             | N/A         | N/A               | N/A              | N/A                 | N/A          | N/A        | N/A              | N/A                | N/A             | yes            | N/A               | N/A      | N/A                   | N/A                 | N/A             | N/A       |
 | Text::Table::Org              | N/A            | N/A              | N/A           | no           | no             | N/A             | N/A         | N/A               | N/A              | no                  | N/A          | N/A        | N/A              | N/A                | N/A             | no             | N/A               | N/A      | N/A                   | N/A                 | N/A             | N/A       |
 | Text::Table::CSV              | N/A            | N/A              | N/A           | N/A *26)     | no             | N/A             | N/A         | N/A               | N/A              | yes *27)            | N/A          | N/A        | N/A              | N/A                | N/A             | yes            | N/A               | N/A      | N/A                   | N/A                 | N/A             | N/A       |
 | Text::Table::HTML             | N/A            | N/A              | N/A           | no           | no *28)        | N/A             | N/A         | N/A               | N/A              | yes                 | N/A          | N/A        | N/A              | N/A                | N/A             | yes            | N/A               | N/A      | N/A                   | N/A                 | N/A             | N/A       |
 | Text::Table::HTML::DataTables | N/A            | N/A              | N/A           | no           | no *28)        | N/A             | N/A         | N/A               | N/A              | yes                 | N/A          | N/A        | N/A              | N/A                | N/A             | yes            | N/A               | N/A      | N/A                   | N/A                 | N/A             | N/A       |
 | Text::TabularDisplay          | N/A            | N/A              | N/A           | N/A *26)     | no             | N/A             | N/A         | N/A               | N/A              | yes                 | N/A          | N/A        | N/A              | N/A                | N/A             | yes            | N/A               | N/A      | N/A                   | N/A                 | N/A             | N/A       |
 +-------------------------------+----------------+------------------+---------------+--------------+----------------+-----------------+-------------+-------------------+------------------+---------------------+--------------+------------+------------------+--------------------+-----------------+----------------+-------------------+----------+-----------------------+---------------------+-----------------+-----------+
 


Notes:

=over

=item 1. align_cell: Whether module supports aligning text horizontally in individual cells (left/right/middle)

=item 2. align_column: Whether module supports aligning text horizontally in a column (left/right/middle)

=item 3. align_row: Whether module supports aligning text horizontally in a row (left/right/middle)

=item 4. box_char: Whether module can utilize box-drawing characters

=item 5. color_data: Whether module supports ANSI colors (i.e. text with ANSI color codes can still be aligned properly)

=item 6. color_theme: Whether the module supports color theme/scheme

=item 7. colspan: Whether module supports column spans

=item 8. custom_border: Whether module allows customizing border in some way

=item 9. custom_color: Whether the module produces colored table and supports customizing color in some way

=item 10. multiline_data: Whether module supports aligning data cells that contain newlines

=item 11. rowspan: Whether module supports row spans

=item 12. speed: Rendering speed

=item 13. valign_cell: Whether module supports aligning text vertically in individual cells (top/bottom/middle)

=item 14. valign_column: Whether module supports aligning text vertically in a column (top/bottom/middle)

=item 15. valign_row: Whether module supports aligning text vertically in a row (top/bottom/middle)

=item 16. column_width: Whether module allows setting the width of columns

=item 17. pad: Whether module allows customizing cell horizontal padding

=item 18. per_column_width: Whether module allows setting column width on a per-column basis

=item 19. per_row_height: Whether module allows setting row height on a per-row basis

=item 20. row_height: Whether module allows setting the height of rows

=item 21. vpad: Whether module allows customizing cell vertical padding

=item 22. Depends on backend

=item 23. But this module can pass rendering to other module like Text::UnicodeBox::Table

=item 24. Newlines stripped

=item 25. Does not draw borders

=item 26. Irrelevant

=item 27. But make sure your CSV parser can handle multiline cell

=item 28. Not converted to HTML color elements

=back

=head1 BENCHMARKED MODULES

Version numbers shown below are the versions used when running the sample benchmark.

L<Text::Table::Any> 0.104

L<Text::UnicodeBox::Table>

L<Text::Table::Manifold> 1.03

L<Text::ANSITable> 0.602

L<Text::ASCIITable> 0.22

L<Text::FormatTable> 1.03

L<Text::MarkdownTable> 0.3.1

L<Text::Table> 1.133

L<Text::Table::Tiny> 1.02

L<Text::Table::TinyBorderStyle> 0.004

L<Text::Table::More> 0.014

L<Text::Table::Sprintf> 0.003

L<Text::Table::TinyColor> 0.002

L<Text::Table::TinyColorWide> 0.001

L<Text::Table::TinyWide> 0.001

L<Text::Table::Org> 0.02

L<Text::Table::CSV> 0.023

L<Text::Table::HTML> 0.004

L<Text::Table::HTML::DataTables> 0.007

L<Text::TabularDisplay> 1.38

=head1 BENCHMARK PARTICIPANTS

=over

=item * Text::Table::Any (perl_code)

L<Text::Table::Any>



=item * Text::UnicodeBox::Table (perl_code)

L<Text::UnicodeBox::Table>



=item * Text::Table::Manifold (perl_code)

L<Text::Table::Manifold>



=item * Text::ANSITable (perl_code)

L<Text::ANSITable>



=item * Text::ASCIITable (perl_code)

L<Text::ASCIITable>



=item * Text::FormatTable (perl_code)

L<Text::FormatTable>



=item * Text::MarkdownTable (perl_code)

L<Text::MarkdownTable>



=item * Text::Table (perl_code)

L<Text::Table>



=item * Text::Table::Tiny (perl_code)

L<Text::Table::Tiny>



=item * Text::Table::TinyBorderStyle (perl_code)

L<Text::Table::TinyBorderStyle>



=item * Text::Table::More (perl_code)

L<Text::Table::More>



=item * Text::Table::Sprintf (perl_code)

L<Text::Table::Sprintf>



=item * Text::Table::TinyColor (perl_code)

L<Text::Table::TinyColor>



=item * Text::Table::TinyColorWide (perl_code)

L<Text::Table::TinyColorWide>



=item * Text::Table::TinyWide (perl_code)

L<Text::Table::TinyWide>



=item * Text::Table::Org (perl_code)

L<Text::Table::Org>



=item * Text::Table::CSV (perl_code)

L<Text::Table::CSV>



=item * Text::Table::HTML (perl_code)

L<Text::Table::HTML>



=item * Text::Table::HTML::DataTables (perl_code)

L<Text::Table::HTML::DataTables>



=item * Text::TabularDisplay (perl_code)

L<Text::TabularDisplay>



=back

=head1 BENCHMARK DATASETS

=over

=item * tiny (1x1)

=item * small (3x5)

=item * wide (30x5)

=item * long (3x300)

=item * large (30x300)

=item * multiline data (2x1) (not included by default)

=item * wide char data (1x2) (not included by default)

=item * color data (1x2) (not included by default)

=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.30.2 >>, CPU: I<< Intel(R) Core(TM) i7-4770 CPU @ 3.40GHz (4 cores) >>, OS: I<< GNU/Linux LinuxMint version 19 >>, OS kernel: I<< Linux version 5.3.0-68-generic >>.

Benchmark with default options (C<< bencher --cpanmodules-module TextTable >>):

 #table1#
 {dataset=>"large (30x300)"}
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+-----------+---------+
 | participant                   | rate (/s) | time (ms) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors   | samples |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+-----------+---------+
 | Text::UnicodeBox::Table       |       1.4 |    740    |                 0.00% |             40062.50% |   0.003   |      20 |
 | Text::ANSITable               |       3.2 |    320    |               132.98% |             17138.44% |   0.001   |      20 |
 | Text::Table::More             |       6.5 |    150    |               375.49% |              8346.48% |   0.00084 |      21 |
 | Text::ASCIITable              |      15   |     67    |              1005.89% |              3531.68% |   0.00011 |      20 |
 | Text::FormatTable             |      21   |     48    |              1430.74% |              2523.73% |   0.00036 |      20 |
 | Text::Table::TinyColorWide    |      21   |     48    |              1450.07% |              2491.01% |   0.00012 |      20 |
 | Text::Table::TinyWide         |      29   |     34    |              2065.62% |              1754.55% |   0.00017 |      20 |
 | Text::Table::Manifold         |      48   |     21    |              3447.53% |              1032.13% | 8.9e-05   |      20 |
 | Text::Table::Tiny             |      53.8 |     18.6  |              3864.28% |               913.11% | 8.3e-06   |      21 |
 | Text::TabularDisplay          |      57   |     17    |              4113.45% |               853.20% | 2.1e-05   |      20 |
 | Text::Table::TinyColor        |      76   |     13    |              5529.33% |               613.45% | 3.3e-05   |      26 |
 | Text::Table::HTML             |      78   |     13    |              5670.96% |               595.94% | 2.4e-05   |      20 |
 | Text::MarkdownTable           |     110   |      9.5  |              7672.22% |               416.74% |   6e-05   |      20 |
 | Text::Table                   |     140   |      7    |             10445.18% |               280.86% |   1e-05   |      20 |
 | Text::Table::HTML::DataTables |     160   |      6.3  |             11576.84% |               243.95% | 9.6e-06   |      20 |
 | Text::Table::CSV              |     283   |      3.53 |             20785.02% |                92.30% | 1.1e-06   |      20 |
 | Text::Table::Org              |     285   |      3.51 |             20926.58% |                91.01% | 2.9e-06   |      20 |
 | Text::Table::TinyBorderStyle  |     330   |      3.03 |             24216.60% |                65.16% | 2.2e-06   |      20 |
 | Text::Table::Any              |     520   |      1.9  |             38508.97% |                 4.02% | 4.3e-06   |      20 |
 | Text::Table::Sprintf          |     540   |      1.8  |             40062.50% |                 0.00% | 7.1e-06   |      20 |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+-----------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                                   Rate  Text::UnicodeBox::Table  Text::ANSITable  Text::Table::More  Text::ASCIITable  Text::FormatTable  Text::Table::TinyColorWide  Text::Table::TinyWide  Text::Table::Manifold  Text::Table::Tiny  Text::TabularDisplay  Text::Table::TinyColor  Text::Table::HTML  Text::MarkdownTable  Text::Table  Text::Table::HTML::DataTables  Text::Table::CSV  Text::Table::Org  Text::Table::TinyBorderStyle  Text::Table::Any  Text::Table::Sprintf 
  Text::UnicodeBox::Table         1.4/s                       --             -56%               -79%              -90%               -93%                        -93%                   -95%                   -97%               -97%                  -97%                    -98%               -98%                 -98%         -99%                           -99%              -99%              -99%                          -99%              -99%                  -99% 
  Text::ANSITable                 3.2/s                     131%               --               -53%              -79%               -85%                        -85%                   -89%                   -93%               -94%                  -94%                    -95%               -95%                 -97%         -97%                           -98%              -98%              -98%                          -99%              -99%                  -99% 
  Text::Table::More               6.5/s                     393%             113%                 --              -55%               -68%                        -68%                   -77%                   -86%               -87%                  -88%                    -91%               -91%                 -93%         -95%                           -95%              -97%              -97%                          -97%              -98%                  -98% 
  Text::ASCIITable                 15/s                    1004%             377%               123%                --               -28%                        -28%                   -49%                   -68%               -72%                  -74%                    -80%               -80%                 -85%         -89%                           -90%              -94%              -94%                          -95%              -97%                  -97% 
  Text::FormatTable                21/s                    1441%             566%               212%               39%                 --                          0%                   -29%                   -56%               -61%                  -64%                    -72%               -72%                 -80%         -85%                           -86%              -92%              -92%                          -93%              -96%                  -96% 
  Text::Table::TinyColorWide       21/s                    1441%             566%               212%               39%                 0%                          --                   -29%                   -56%               -61%                  -64%                    -72%               -72%                 -80%         -85%                           -86%              -92%              -92%                          -93%              -96%                  -96% 
  Text::Table::TinyWide            29/s                    2076%             841%               341%               97%                41%                         41%                     --                   -38%               -45%                  -50%                    -61%               -61%                 -72%         -79%                           -81%              -89%              -89%                          -91%              -94%                  -94% 
  Text::Table::Manifold            48/s                    3423%            1423%               614%              219%               128%                        128%                    61%                     --               -11%                  -19%                    -38%               -38%                 -54%         -66%                           -70%              -83%              -83%                          -85%              -90%                  -91% 
  Text::Table::Tiny              53.8/s                    3878%            1620%               706%              260%               158%                        158%                    82%                    12%                 --                   -8%                    -30%               -30%                 -48%         -62%                           -66%              -81%              -81%                          -83%              -89%                  -90% 
  Text::TabularDisplay             57/s                    4252%            1782%               782%              294%               182%                        182%                   100%                    23%                 9%                    --                    -23%               -23%                 -44%         -58%                           -62%              -79%              -79%                          -82%              -88%                  -89% 
  Text::Table::TinyColor           76/s                    5592%            2361%              1053%              415%               269%                        269%                   161%                    61%                43%                   30%                      --                 0%                 -26%         -46%                           -51%              -72%              -73%                          -76%              -85%                  -86% 
  Text::Table::HTML                78/s                    5592%            2361%              1053%              415%               269%                        269%                   161%                    61%                43%                   30%                      0%                 --                 -26%         -46%                           -51%              -72%              -73%                          -76%              -85%                  -86% 
  Text::MarkdownTable             110/s                    7689%            3268%              1478%              605%               405%                        405%                   257%                   121%                95%                   78%                     36%                36%                   --         -26%                           -33%              -62%              -63%                          -68%              -80%                  -81% 
  Text::Table                     140/s                   10471%            4471%              2042%              857%               585%                        585%                   385%                   200%               165%                  142%                     85%                85%                  35%           --                            -9%              -49%              -49%                          -56%              -72%                  -74% 
  Text::Table::HTML::DataTables   160/s                   11646%            4979%              2280%              963%               661%                        661%                   439%                   233%               195%                  169%                    106%               106%                  50%          11%                             --              -43%              -44%                          -51%              -69%                  -71% 
  Text::Table::CSV                283/s                   20863%            8965%              4149%             1798%              1259%                       1259%                   863%                   494%               426%                  381%                    268%               268%                 169%          98%                            78%                --                0%                          -14%              -46%                  -49% 
  Text::Table::Org                285/s                   20982%            9016%              4173%             1808%              1267%                       1267%                   868%                   498%               429%                  384%                    270%               270%                 170%          99%                            79%                0%                --                          -13%              -45%                  -48% 
  Text::Table::TinyBorderStyle    330/s                   24322%           10461%              4850%             2111%              1484%                       1484%                  1022%                   593%               513%                  461%                    329%               329%                 213%         131%                           107%               16%               15%                            --              -37%                  -40% 
  Text::Table::Any                520/s                   38847%           16742%              7794%             3426%              2426%                       2426%                  1689%                  1005%               878%                  794%                    584%               584%                 400%         268%                           231%               85%               84%                           59%                --                   -5% 
  Text::Table::Sprintf            540/s                   41011%           17677%              8233%             3622%              2566%                       2566%                  1788%                  1066%               933%                  844%                    622%               622%                 427%         288%                           250%               96%               94%                           68%                5%                    -- 
 
 Legends:
   Text::ANSITable: participant=Text::ANSITable
   Text::ASCIITable: participant=Text::ASCIITable
   Text::FormatTable: participant=Text::FormatTable
   Text::MarkdownTable: participant=Text::MarkdownTable
   Text::Table: participant=Text::Table
   Text::Table::Any: participant=Text::Table::Any
   Text::Table::CSV: participant=Text::Table::CSV
   Text::Table::HTML: participant=Text::Table::HTML
   Text::Table::HTML::DataTables: participant=Text::Table::HTML::DataTables
   Text::Table::Manifold: participant=Text::Table::Manifold
   Text::Table::More: participant=Text::Table::More
   Text::Table::Org: participant=Text::Table::Org
   Text::Table::Sprintf: participant=Text::Table::Sprintf
   Text::Table::Tiny: participant=Text::Table::Tiny
   Text::Table::TinyBorderStyle: participant=Text::Table::TinyBorderStyle
   Text::Table::TinyColor: participant=Text::Table::TinyColor
   Text::Table::TinyColorWide: participant=Text::Table::TinyColorWide
   Text::Table::TinyWide: participant=Text::Table::TinyWide
   Text::TabularDisplay: participant=Text::TabularDisplay
   Text::UnicodeBox::Table: participant=Text::UnicodeBox::Table

 #table2#
 {dataset=>"long (3x300)"}
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+-----------+---------+
 | participant                   | rate (/s) | time (ms) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors   | samples |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+-----------+---------+
 | Text::UnicodeBox::Table       |      11   |    95     |                 0.00% |             38088.56% |   0.00035 |      20 |
 | Text::ANSITable               |      28   |    36     |               166.83% |             14212.13% |   0.00022 |      20 |
 | Text::Table::More             |      62.2 |    16.1   |               489.79% |              6374.94% | 6.2e-06   |      20 |
 | Text::ASCIITable              |     160   |     6.2   |              1423.98% |              2405.85% | 1.3e-05   |      20 |
 | Text::FormatTable             |     190   |     5.2   |              1734.64% |              1981.53% | 1.8e-05   |      20 |
 | Text::Table::TinyColorWide    |     200   |     4.9   |              1817.28% |              1891.81% | 1.3e-05   |      21 |
 | Text::Table::TinyWide         |     300   |     3.3   |              2736.74% |              1246.21% | 3.8e-06   |      20 |
 | Text::TabularDisplay          |     400   |     2.5   |              3655.21% |               916.95% | 1.8e-05   |      20 |
 | Text::Table::Manifold         |     442   |     2.26  |              4093.21% |               810.72% | 1.8e-06   |      20 |
 | Text::Table::Tiny             |     480   |     2.1   |              4443.69% |               740.48% | 5.8e-06   |      20 |
 | Text::MarkdownTable           |     520   |     1.9   |              4848.83% |               671.67% | 4.3e-06   |      20 |
 | Text::Table                   |     630   |     1.6   |              5894.54% |               537.06% | 2.2e-06   |      20 |
 | Text::Table::TinyColor        |     710   |     1.4   |              6637.81% |               466.78% | 1.8e-06   |      20 |
 | Text::Table::HTML             |     720   |     1.4   |              6764.78% |               456.30% | 5.3e-06   |      20 |
 | Text::Table::HTML::DataTables |    1200   |     0.82  |             11455.83% |               230.47% | 1.4e-06   |      20 |
 | Text::Table::Org              |    2100   |     0.48  |             19592.78% |                93.92% | 1.5e-06   |      20 |
 | Text::Table::CSV              |    2170   |     0.46  |             20507.69% |                85.31% | 2.1e-07   |      20 |
 | Text::Table::TinyBorderStyle  |    2210   |     0.452 |             20884.52% |                81.98% | 2.1e-07   |      21 |
 | Text::Table::Any              |    3700   |     0.27  |             34898.56% |                 9.11% | 9.1e-07   |      20 |
 | Text::Table::Sprintf          |    4000   |     0.25  |             38088.56% |                 0.00% | 2.7e-07   |      20 |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+-----------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                                   Rate  Text::UnicodeBox::Table  Text::ANSITable  Text::Table::More  Text::ASCIITable  Text::FormatTable  Text::Table::TinyColorWide  Text::Table::TinyWide  Text::TabularDisplay  Text::Table::Manifold  Text::Table::Tiny  Text::MarkdownTable  Text::Table  Text::Table::TinyColor  Text::Table::HTML  Text::Table::HTML::DataTables  Text::Table::Org  Text::Table::CSV  Text::Table::TinyBorderStyle  Text::Table::Any  Text::Table::Sprintf 
  Text::UnicodeBox::Table          11/s                       --             -62%               -83%              -93%               -94%                        -94%                   -96%                  -97%                   -97%               -97%                 -98%         -98%                    -98%               -98%                           -99%              -99%              -99%                          -99%              -99%                  -99% 
  Text::ANSITable                  28/s                     163%               --               -55%              -82%               -85%                        -86%                   -90%                  -93%                   -93%               -94%                 -94%         -95%                    -96%               -96%                           -97%              -98%              -98%                          -98%              -99%                  -99% 
  Text::Table::More              62.2/s                     490%             123%                 --              -61%               -67%                        -69%                   -79%                  -84%                   -85%               -86%                 -88%         -90%                    -91%               -91%                           -94%              -97%              -97%                          -97%              -98%                  -98% 
  Text::ASCIITable                160/s                    1432%             480%               159%                --               -16%                        -20%                   -46%                  -59%                   -63%               -66%                 -69%         -74%                    -77%               -77%                           -86%              -92%              -92%                          -92%              -95%                  -95% 
  Text::FormatTable               190/s                    1726%             592%               209%               19%                 --                         -5%                   -36%                  -51%                   -56%               -59%                 -63%         -69%                    -73%               -73%                           -84%              -90%              -91%                          -91%              -94%                  -95% 
  Text::Table::TinyColorWide      200/s                    1838%             634%               228%               26%                 6%                          --                   -32%                  -48%                   -53%               -57%                 -61%         -67%                    -71%               -71%                           -83%              -90%              -90%                          -90%              -94%                  -94% 
  Text::Table::TinyWide           300/s                    2778%             990%               387%               87%                57%                         48%                     --                  -24%                   -31%               -36%                 -42%         -51%                    -57%               -57%                           -75%              -85%              -86%                          -86%              -91%                  -92% 
  Text::TabularDisplay            400/s                    3700%            1340%               544%              148%               108%                         96%                    31%                    --                    -9%               -15%                 -24%         -36%                    -44%               -44%                           -67%              -80%              -81%                          -81%              -89%                  -90% 
  Text::Table::Manifold           442/s                    4103%            1492%               612%              174%               130%                        116%                    46%                   10%                     --                -7%                 -15%         -29%                    -38%               -38%                           -63%              -78%              -79%                          -80%              -88%                  -88% 
  Text::Table::Tiny               480/s                    4423%            1614%               666%              195%               147%                        133%                    57%                   19%                     7%                 --                  -9%         -23%                    -33%               -33%                           -60%              -77%              -78%                          -78%              -87%                  -88% 
  Text::MarkdownTable             520/s                    4900%            1794%               747%              226%               173%                        157%                    73%                   31%                    18%                10%                   --         -15%                    -26%               -26%                           -56%              -74%              -75%                          -76%              -85%                  -86% 
  Text::Table                     630/s                    5837%            2150%               906%              287%               225%                        206%                   106%                   56%                    41%                31%                  18%           --                    -12%               -12%                           -48%              -70%              -71%                          -71%              -83%                  -84% 
  Text::Table::TinyColor          710/s                    6685%            2471%              1050%              342%               271%                        250%                   135%                   78%                    61%                50%                  35%          14%                      --                 0%                           -41%              -65%              -67%                          -67%              -80%                  -82% 
  Text::Table::HTML               720/s                    6685%            2471%              1050%              342%               271%                        250%                   135%                   78%                    61%                50%                  35%          14%                      0%                 --                           -41%              -65%              -67%                          -67%              -80%                  -82% 
  Text::Table::HTML::DataTables  1200/s                   11485%            4290%              1863%              656%               534%                        497%                   302%                  204%                   175%               156%                 131%          95%                     70%                70%                             --              -41%              -43%                          -44%              -67%                  -69% 
  Text::Table::Org               2100/s                   19691%            7400%              3254%             1191%               983%                        920%                   587%                  420%                   370%               337%                 295%         233%                    191%               191%                            70%                --               -4%                           -5%              -43%                  -47% 
  Text::Table::CSV               2170/s                   20552%            7726%              3400%             1247%              1030%                        965%                   617%                  443%                   391%               356%                 313%         247%                    204%               204%                            78%                4%                --                           -1%              -41%                  -45% 
  Text::Table::TinyBorderStyle   2210/s                   20917%            7864%              3461%             1271%              1050%                        984%                   630%                  453%                   399%               364%                 320%         253%                    209%               209%                            81%                6%                1%                            --              -40%                  -44% 
  Text::Table::Any               3700/s                   35085%           13233%              5862%             2196%              1825%                       1714%                  1122%                  825%                   737%               677%                 603%         492%                    418%               418%                           203%               77%               70%                           67%                --                   -7% 
  Text::Table::Sprintf           4000/s                   37900%           14300%              6340%             2380%              1980%                       1860%                  1220%                  900%                   803%               740%                 660%         540%                    459%               459%                           227%               92%               84%                           80%                8%                    -- 
 
 Legends:
   Text::ANSITable: participant=Text::ANSITable
   Text::ASCIITable: participant=Text::ASCIITable
   Text::FormatTable: participant=Text::FormatTable
   Text::MarkdownTable: participant=Text::MarkdownTable
   Text::Table: participant=Text::Table
   Text::Table::Any: participant=Text::Table::Any
   Text::Table::CSV: participant=Text::Table::CSV
   Text::Table::HTML: participant=Text::Table::HTML
   Text::Table::HTML::DataTables: participant=Text::Table::HTML::DataTables
   Text::Table::Manifold: participant=Text::Table::Manifold
   Text::Table::More: participant=Text::Table::More
   Text::Table::Org: participant=Text::Table::Org
   Text::Table::Sprintf: participant=Text::Table::Sprintf
   Text::Table::Tiny: participant=Text::Table::Tiny
   Text::Table::TinyBorderStyle: participant=Text::Table::TinyBorderStyle
   Text::Table::TinyColor: participant=Text::Table::TinyColor
   Text::Table::TinyColorWide: participant=Text::Table::TinyColorWide
   Text::Table::TinyWide: participant=Text::Table::TinyWide
   Text::TabularDisplay: participant=Text::TabularDisplay
   Text::UnicodeBox::Table: participant=Text::UnicodeBox::Table

 #table3#
 {dataset=>"small (3x5)"}
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                   | rate (/s) | time (ms) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | Text::UnicodeBox::Table       |     300   | 4         |                 0.00% |             45723.34% | 7.1e-05 |      21 |
 | Text::ANSITable               |    1050   | 0.956     |               291.77% |             11596.45% | 9.1e-07 |      20 |
 | Text::Table::More             |    2500   | 0.41      |               819.03% |              4886.03% | 4.6e-07 |      22 |
 | Text::ASCIITable              |    6100   | 0.16      |              2186.14% |              1904.40% | 2.4e-07 |      25 |
 | Text::FormatTable             |    8500   | 0.12      |              3077.08% |              1342.31% |   2e-07 |      22 |
 | Text::Table::Manifold         |    9380   | 0.107     |              3413.81% |              1204.09% | 5.3e-08 |      20 |
 | Text::Table                   |    9450   | 0.106     |              3439.12% |              1194.76% | 5.3e-08 |      20 |
 | Text::Table::TinyColorWide    |    9770   | 0.102     |              3559.53% |              1152.17% | 4.8e-08 |      25 |
 | Text::Table::TinyWide         |   13800   | 0.0724    |              5074.97% |               785.48% | 2.7e-08 |      20 |
 | Text::Table::TinyBorderStyle  |   14000   | 0.07      |              5255.07% |               755.70% |   9e-08 |      28 |
 | Text::MarkdownTable           |   15000   | 0.067     |              5484.57% |               720.54% | 2.7e-07 |      20 |
 | Text::Table::HTML::DataTables |   18000   | 0.055     |              6685.27% |               575.34% | 1.3e-07 |      20 |
 | Text::TabularDisplay          |   18000   | 0.055     |              6766.04% |               567.39% | 1.1e-07 |      20 |
 | Text::Table::Tiny             |   19000   | 0.053     |              6917.02% |               553.03% |   8e-08 |      20 |
 | Text::Table::TinyColor        |   29000   | 0.034     |             10939.56% |               315.08% | 3.5e-08 |      26 |
 | Text::Table::HTML             |   38585.9 | 0.0259162 |             14358.44% |               216.93% |   0     |      28 |
 | Text::Table::Org              |   67700   | 0.0148    |             25251.38% |                80.75% | 6.7e-09 |      20 |
 | Text::Table::CSV              |   94000   | 0.011     |             35148.46% |                30.00% | 2.6e-08 |      26 |
 | Text::Table::Any              |  109000   | 0.00919   |             40661.78% |                12.42% | 2.9e-09 |      26 |
 | Text::Table::Sprintf          |  122000   | 0.00818   |             45723.34% |                 0.00% | 3.3e-09 |      20 |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                                      Rate  Text::UnicodeBox::Table  Text::ANSITable  Text::Table::More  Text::ASCIITable  Text::FormatTable  Text::Table::Manifold  Text::Table  Text::Table::TinyColorWide  Text::Table::TinyWide  Text::Table::TinyBorderStyle  Text::MarkdownTable  Text::Table::HTML::DataTables  Text::TabularDisplay  Text::Table::Tiny  Text::Table::TinyColor  Text::Table::HTML  Text::Table::Org  Text::Table::CSV  Text::Table::Any  Text::Table::Sprintf 
  Text::UnicodeBox::Table            300/s                       --             -76%               -89%              -96%               -97%                   -97%         -97%                        -97%                   -98%                          -98%                 -98%                           -98%                  -98%               -98%                    -99%               -99%              -99%              -99%              -99%                  -99% 
  Text::ANSITable                   1050/s                     318%               --               -57%              -83%               -87%                   -88%         -88%                        -89%                   -92%                          -92%                 -92%                           -94%                  -94%               -94%                    -96%               -97%              -98%              -98%              -99%                  -99% 
  Text::Table::More                 2500/s                     875%             133%                 --              -60%               -70%                   -73%         -74%                        -75%                   -82%                          -82%                 -83%                           -86%                  -86%               -87%                    -91%               -93%              -96%              -97%              -97%                  -98% 
  Text::ASCIITable                  6100/s                    2400%             497%               156%                --               -25%                   -33%         -33%                        -36%                   -54%                          -56%                 -58%                           -65%                  -65%               -66%                    -78%               -83%              -90%              -93%              -94%                  -94% 
  Text::FormatTable                 8500/s                    3233%             696%               241%               33%                 --                   -10%         -11%                        -15%                   -39%                          -41%                 -44%                           -54%                  -54%               -55%                    -71%               -78%              -87%              -90%              -92%                  -93% 
  Text::Table::Manifold             9380/s                    3638%             793%               283%               49%                12%                     --           0%                         -4%                   -32%                          -34%                 -37%                           -48%                  -48%               -50%                    -68%               -75%              -86%              -89%              -91%                  -92% 
  Text::Table                       9450/s                    3673%             801%               286%               50%                13%                     0%           --                         -3%                   -31%                          -33%                 -36%                           -48%                  -48%               -50%                    -67%               -75%              -86%              -89%              -91%                  -92% 
  Text::Table::TinyColorWide        9770/s                    3821%             837%               301%               56%                17%                     4%           3%                          --                   -29%                          -31%                 -34%                           -46%                  -46%               -48%                    -66%               -74%              -85%              -89%              -90%                  -91% 
  Text::Table::TinyWide            13800/s                    5424%            1220%               466%              120%                65%                    47%          46%                         40%                     --                           -3%                  -7%                           -24%                  -24%               -26%                    -53%               -64%              -79%              -84%              -87%                  -88% 
  Text::Table::TinyBorderStyle     14000/s                    5614%            1265%               485%              128%                71%                    52%          51%                         45%                     3%                            --                  -4%                           -21%                  -21%               -24%                    -51%               -62%              -78%              -84%              -86%                  -88% 
  Text::MarkdownTable              15000/s                    5870%            1326%               511%              138%                79%                    59%          58%                         52%                     8%                            4%                   --                           -17%                  -17%               -20%                    -49%               -61%              -77%              -83%              -86%                  -87% 
  Text::Table::HTML::DataTables    18000/s                    7172%            1638%               645%              190%               118%                    94%          92%                         85%                    31%                           27%                  21%                             --                    0%                -3%                    -38%               -52%              -73%              -80%              -83%                  -85% 
  Text::TabularDisplay             18000/s                    7172%            1638%               645%              190%               118%                    94%          92%                         85%                    31%                           27%                  21%                             0%                    --                -3%                    -38%               -52%              -73%              -80%              -83%                  -85% 
  Text::Table::Tiny                19000/s                    7447%            1703%               673%              201%               126%                   101%         100%                         92%                    36%                           32%                  26%                             3%                    3%                 --                    -35%               -51%              -72%              -79%              -82%                  -84% 
  Text::Table::TinyColor           29000/s                   11664%            2711%              1105%              370%               252%                   214%         211%                        199%                   112%                          105%                  97%                            61%                   61%                55%                      --               -23%              -56%              -67%              -72%                  -75% 
  Text::Table::HTML              38585.9/s                   15334%            3588%              1482%              517%               363%                   312%         309%                        293%                   179%                          170%                 158%                           112%                  112%               104%                     31%                 --              -42%              -57%              -64%                  -68% 
  Text::Table::Org                 67700/s                   26927%            6359%              2670%              981%               710%                   622%         616%                        589%                   389%                          372%                 352%                           271%                  271%               258%                    129%                75%                --              -25%              -37%                  -44% 
  Text::Table::CSV                 94000/s                   36263%            8590%              3627%             1354%               990%                   872%         863%                        827%                   558%                          536%                 509%                           400%                  400%               381%                    209%               135%               34%                --              -16%                  -25% 
  Text::Table::Any                109000/s                   43425%           10302%              4361%             1641%              1205%                  1064%        1053%                       1009%                   687%                          661%                 629%                           498%                  498%               476%                    269%               182%               61%               19%                --                  -10% 
  Text::Table::Sprintf            122000/s                   48799%           11587%              4912%             1855%              1366%                  1208%        1195%                       1146%                   785%                          755%                 719%                           572%                  572%               547%                    315%               216%               80%               34%               12%                    -- 
 
 Legends:
   Text::ANSITable: participant=Text::ANSITable
   Text::ASCIITable: participant=Text::ASCIITable
   Text::FormatTable: participant=Text::FormatTable
   Text::MarkdownTable: participant=Text::MarkdownTable
   Text::Table: participant=Text::Table
   Text::Table::Any: participant=Text::Table::Any
   Text::Table::CSV: participant=Text::Table::CSV
   Text::Table::HTML: participant=Text::Table::HTML
   Text::Table::HTML::DataTables: participant=Text::Table::HTML::DataTables
   Text::Table::Manifold: participant=Text::Table::Manifold
   Text::Table::More: participant=Text::Table::More
   Text::Table::Org: participant=Text::Table::Org
   Text::Table::Sprintf: participant=Text::Table::Sprintf
   Text::Table::Tiny: participant=Text::Table::Tiny
   Text::Table::TinyBorderStyle: participant=Text::Table::TinyBorderStyle
   Text::Table::TinyColor: participant=Text::Table::TinyColor
   Text::Table::TinyColorWide: participant=Text::Table::TinyColorWide
   Text::Table::TinyWide: participant=Text::Table::TinyWide
   Text::TabularDisplay: participant=Text::TabularDisplay
   Text::UnicodeBox::Table: participant=Text::UnicodeBox::Table

 #table4#
 {dataset=>"tiny (1x1)"}
 +-------------------------------+-----------+------------+-----------------------+-----------------------+---------+---------+
 | participant                   | rate (/s) | time (μs)  | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------------------+-----------+------------+-----------------------+-----------------------+---------+---------+
 | Text::UnicodeBox::Table       |     700   | 1000       |                 0.00% |             50233.42% | 3.9e-05 |      20 |
 | Text::ANSITable               |    3300   |  300       |               364.21% |             10742.90% | 2.3e-06 |      20 |
 | Text::Table::More             |    9400   |  110       |              1197.65% |              3778.81% | 4.7e-07 |      21 |
 | Text::Table::Manifold         |   15000   |   65       |              2047.55% |              2243.76% | 2.1e-07 |      20 |
 | Text::Table::TinyBorderStyle  |   19000   |   53       |              2512.73% |              1826.47% | 1.1e-07 |      20 |
 | Text::ASCIITable              |   22000   |   46       |              2948.84% |              1550.90% | 6.7e-08 |      20 |
 | Text::Table::HTML::DataTables |   20000   |   40       |              3088.90% |              1478.39% | 4.4e-07 |      21 |
 | Text::Table                   |   25000   |   40       |              3390.95% |              1341.83% | 5.3e-08 |      20 |
 | Text::MarkdownTable           |   29000   |   35       |              3883.48% |              1163.56% | 1.7e-07 |      20 |
 | Text::FormatTable             |   42000   |   24       |              5787.36% |               754.94% | 2.7e-08 |      20 |
 | Text::Table::TinyColorWide    |   57494.9 |   17.3929  |              7877.79% |               530.92% |   0     |      20 |
 | Text::Table::Tiny             |   68000   |   15       |              9335.33% |               433.46% | 2.7e-08 |      20 |
 | Text::TabularDisplay          |   73000   |   14       |             10045.67% |               396.11% |   6e-08 |      20 |
 | Text::Table::TinyWide         |   75000   |   13       |             10300.65% |               383.94% | 2.7e-08 |      20 |
 | Text::Table::TinyColor        |  118000   |    8.49    |             16244.10% |               207.96% |   3e-09 |      25 |
 | Text::Table::HTML             |  184329   |    5.42508 |             25476.89% |                96.79% |   0     |      22 |
 | Text::Table::Org              |  187000   |    5.36    |             25782.27% |                94.47% | 1.6e-09 |      22 |
 | Text::Table::Any              |  259827   |    3.84871 |             35952.78% |                39.61% |   0     |      20 |
 | Text::Table::Sprintf          |  354000   |    2.83    |             49002.23% |                 2.51% | 8.3e-10 |      20 |
 | Text::Table::CSV              |  363000   |    2.76    |             50233.42% |                 0.00% | 8.3e-10 |      20 |
 +-------------------------------+-----------+------------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                                      Rate  Text::UnicodeBox::Table  Text::ANSITable  Text::Table::More  Text::Table::Manifold  Text::Table::TinyBorderStyle  Text::ASCIITable  Text::Table::HTML::DataTables  Text::Table  Text::MarkdownTable  Text::FormatTable  Text::Table::TinyColorWide  Text::Table::Tiny  Text::TabularDisplay  Text::Table::TinyWide  Text::Table::TinyColor  Text::Table::HTML  Text::Table::Org  Text::Table::Any  Text::Table::Sprintf  Text::Table::CSV 
  Text::UnicodeBox::Table            700/s                       --             -70%               -89%                   -93%                          -94%              -95%                           -96%         -96%                 -96%               -97%                        -98%               -98%                  -98%                   -98%                    -99%               -99%              -99%              -99%                  -99%              -99% 
  Text::ANSITable                   3300/s                     233%               --               -63%                   -78%                          -82%              -84%                           -86%         -86%                 -88%               -92%                        -94%               -95%                  -95%                   -95%                    -97%               -98%              -98%              -98%                  -99%              -99% 
  Text::Table::More                 9400/s                     809%             172%                 --                   -40%                          -51%              -58%                           -63%         -63%                 -68%               -78%                        -84%               -86%                  -87%                   -88%                    -92%               -95%              -95%              -96%                  -97%              -97% 
  Text::Table::Manifold            15000/s                    1438%             361%                69%                     --                          -18%              -29%                           -38%         -38%                 -46%               -63%                        -73%               -76%                  -78%                   -80%                    -86%               -91%              -91%              -94%                  -95%              -95% 
  Text::Table::TinyBorderStyle     19000/s                    1786%             466%               107%                    22%                            --              -13%                           -24%         -24%                 -33%               -54%                        -67%               -71%                  -73%                   -75%                    -83%               -89%              -89%              -92%                  -94%              -94% 
  Text::ASCIITable                 22000/s                    2073%             552%               139%                    41%                           15%                --                           -13%         -13%                 -23%               -47%                        -62%               -67%                  -69%                   -71%                    -81%               -88%              -88%              -91%                  -93%              -94% 
  Text::Table::HTML::DataTables    20000/s                    2400%             650%               175%                    62%                           32%               14%                             --           0%                 -12%               -40%                        -56%               -62%                  -65%                   -67%                    -78%               -86%              -86%              -90%                  -92%              -93% 
  Text::Table                      25000/s                    2400%             650%               175%                    62%                           32%               14%                             0%           --                 -12%               -40%                        -56%               -62%                  -65%                   -67%                    -78%               -86%              -86%              -90%                  -92%              -93% 
  Text::MarkdownTable              29000/s                    2757%             757%               214%                    85%                           51%               31%                            14%          14%                   --               -31%                        -50%               -57%                  -60%                   -62%                    -75%               -84%              -84%              -89%                  -91%              -92% 
  Text::FormatTable                42000/s                    4066%            1150%               358%                   170%                          120%               91%                            66%          66%                  45%                 --                        -27%               -37%                  -41%                   -45%                    -64%               -77%              -77%              -83%                  -88%              -88% 
  Text::Table::TinyColorWide     57494.9/s                    5649%            1624%               532%                   273%                          204%              164%                           129%         129%                 101%                37%                          --               -13%                  -19%                   -25%                    -51%               -68%              -69%              -77%                  -83%              -84% 
  Text::Table::Tiny                68000/s                    6566%            1900%               633%                   333%                          253%              206%                           166%         166%                 133%                60%                         15%                 --                   -6%                   -13%                    -43%               -63%              -64%              -74%                  -81%              -81% 
  Text::TabularDisplay             73000/s                    7042%            2042%               685%                   364%                          278%              228%                           185%         185%                 150%                71%                         24%                 7%                    --                    -7%                    -39%               -61%              -61%              -72%                  -79%              -80% 
  Text::Table::TinyWide            75000/s                    7592%            2207%               746%                   400%                          307%              253%                           207%         207%                 169%                84%                         33%                15%                    7%                     --                    -34%               -58%              -58%              -70%                  -78%              -78% 
  Text::Table::TinyColor          118000/s                   11678%            3433%              1195%                   665%                          524%              441%                           371%         371%                 312%               182%                        104%                76%                   64%                    53%                      --               -36%              -36%              -54%                  -66%              -67% 
  Text::Table::HTML               184329/s                   18332%            5429%              1927%                  1098%                          876%              747%                           637%         637%                 545%               342%                        220%               176%                  158%                   139%                     56%                 --               -1%              -29%                  -47%              -49% 
  Text::Table::Org                187000/s                   18556%            5497%              1952%                  1112%                          888%              758%                           646%         646%                 552%               347%                        224%               179%                  161%                   142%                     58%                 1%                --              -28%                  -47%              -48% 
  Text::Table::Any                259827/s                   25882%            7694%              2758%                  1588%                         1277%             1095%                           939%         939%                 809%               523%                        351%               289%                  263%                   237%                    120%                40%               39%                --                  -26%              -28% 
  Text::Table::Sprintf            354000/s                   35235%           10500%              3786%                  2196%                         1772%             1525%                          1313%        1313%                1136%               748%                        514%               430%                  394%                   359%                    200%                91%               89%               35%                    --               -2% 
  Text::Table::CSV                363000/s                   36131%           10769%              3885%                  2255%                         1820%             1566%                          1349%        1349%                1168%               769%                        530%               443%                  407%                   371%                    207%                96%               94%               39%                    2%                -- 
 
 Legends:
   Text::ANSITable: participant=Text::ANSITable
   Text::ASCIITable: participant=Text::ASCIITable
   Text::FormatTable: participant=Text::FormatTable
   Text::MarkdownTable: participant=Text::MarkdownTable
   Text::Table: participant=Text::Table
   Text::Table::Any: participant=Text::Table::Any
   Text::Table::CSV: participant=Text::Table::CSV
   Text::Table::HTML: participant=Text::Table::HTML
   Text::Table::HTML::DataTables: participant=Text::Table::HTML::DataTables
   Text::Table::Manifold: participant=Text::Table::Manifold
   Text::Table::More: participant=Text::Table::More
   Text::Table::Org: participant=Text::Table::Org
   Text::Table::Sprintf: participant=Text::Table::Sprintf
   Text::Table::Tiny: participant=Text::Table::Tiny
   Text::Table::TinyBorderStyle: participant=Text::Table::TinyBorderStyle
   Text::Table::TinyColor: participant=Text::Table::TinyColor
   Text::Table::TinyColorWide: participant=Text::Table::TinyColorWide
   Text::Table::TinyWide: participant=Text::Table::TinyWide
   Text::TabularDisplay: participant=Text::TabularDisplay
   Text::UnicodeBox::Table: participant=Text::UnicodeBox::Table

 #table5#
 {dataset=>"wide (30x5)"}
 +-------------------------------+-----------+------------+-----------------------+-----------------------+---------+---------+
 | participant                   | rate (/s) | time (ms)  | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------------------+-----------+------------+-----------------------+-----------------------+---------+---------+
 | Text::UnicodeBox::Table       |      52   | 19         |                 0.00% |             36652.35% | 5.6e-05 |      20 |
 | Text::ANSITable               |     130   |  7.5       |               156.73% |             14215.35% | 7.8e-06 |      20 |
 | Text::Table::More             |     320   |  3.1       |               515.18% |              5874.27% | 3.1e-06 |      20 |
 | Text::ASCIITable              |     640   |  1.6       |              1121.38% |              2909.08% | 4.3e-06 |      20 |
 | Text::FormatTable             |     870   |  1.1       |              1573.21% |              2096.52% | 1.5e-06 |      21 |
 | Text::Table::TinyColorWide    |    1060   |  0.94      |              1942.19% |              1699.65% | 8.5e-07 |      23 |
 | Text::Table                   |    1200   |  0.81      |              2261.95% |              1456.02% | 8.8e-07 |      21 |
 | Text::Table::TinyWide         |    1520   |  0.657     |              2821.15% |              1158.15% | 4.3e-07 |      20 |
 | Text::Table::Manifold         |    2050   |  0.488     |              3833.97% |               834.23% | 4.3e-07 |      20 |
 | Text::TabularDisplay          |    2000   |  0.4       |              4699.60% |               665.74% | 4.4e-06 |      20 |
 | Text::Table::Tiny             |    2510   |  0.398     |              4720.02% |               662.49% |   2e-07 |      23 |
 | Text::Table::TinyColor        |    3730   |  0.268     |              7067.11% |               412.79% | 2.1e-07 |      20 |
 | Text::MarkdownTable           |    4000   |  0.25      |              7624.86% |               375.77% |   2e-06 |      20 |
 | Text::Table::TinyBorderStyle  |    4070   |  0.246     |              7708.35% |               370.68% | 2.1e-07 |      20 |
 | Text::Table::HTML             |    4720   |  0.212     |              8967.44% |               305.32% | 5.3e-08 |      20 |
 | Text::Table::HTML::DataTables |    5900   |  0.17      |             11319.87% |               221.83% | 1.1e-06 |      20 |
 | Text::Table::Org              |   11100   |  0.0897    |             21310.00% |                71.66% |   8e-08 |      20 |
 | Text::Table::CSV              |   14500   |  0.0687    |             27840.51% |                31.54% | 2.6e-08 |      21 |
 | Text::Table::Any              |   17000   |  0.06      |             32095.31% |                14.15% |   8e-08 |      20 |
 | Text::Table::Sprintf          |   19136.5 |  0.0522561 |             36652.35% |                 0.00% |   0     |      27 |
 +-------------------------------+-----------+------------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                                      Rate  Text::UnicodeBox::Table  Text::ANSITable  Text::Table::More  Text::ASCIITable  Text::FormatTable  Text::Table::TinyColorWide  Text::Table  Text::Table::TinyWide  Text::Table::Manifold  Text::TabularDisplay  Text::Table::Tiny  Text::Table::TinyColor  Text::MarkdownTable  Text::Table::TinyBorderStyle  Text::Table::HTML  Text::Table::HTML::DataTables  Text::Table::Org  Text::Table::CSV  Text::Table::Any  Text::Table::Sprintf 
  Text::UnicodeBox::Table             52/s                       --             -60%               -83%              -91%               -94%                        -95%         -95%                   -96%                   -97%                  -97%               -97%                    -98%                 -98%                          -98%               -98%                           -99%              -99%              -99%              -99%                  -99% 
  Text::ANSITable                    130/s                     153%               --               -58%              -78%               -85%                        -87%         -89%                   -91%                   -93%                  -94%               -94%                    -96%                 -96%                          -96%               -97%                           -97%              -98%              -99%              -99%                  -99% 
  Text::Table::More                  320/s                     512%             141%                 --              -48%               -64%                        -69%         -73%                   -78%                   -84%                  -87%               -87%                    -91%                 -91%                          -92%               -93%                           -94%              -97%              -97%              -98%                  -98% 
  Text::ASCIITable                   640/s                    1087%             368%                93%                --               -31%                        -41%         -49%                   -58%                   -69%                  -75%               -75%                    -83%                 -84%                          -84%               -86%                           -89%              -94%              -95%              -96%                  -96% 
  Text::FormatTable                  870/s                    1627%             581%               181%               45%                 --                        -14%         -26%                   -40%                   -55%                  -63%               -63%                    -75%                 -77%                          -77%               -80%                           -84%              -91%              -93%              -94%                  -95% 
  Text::Table::TinyColorWide        1060/s                    1921%             697%               229%               70%                17%                          --         -13%                   -30%                   -48%                  -57%               -57%                    -71%                 -73%                          -73%               -77%                           -81%              -90%              -92%              -93%                  -94% 
  Text::Table                       1200/s                    2245%             825%               282%               97%                35%                         16%           --                   -18%                   -39%                  -50%               -50%                    -66%                 -69%                          -69%               -73%                           -79%              -88%              -91%              -92%                  -93% 
  Text::Table::TinyWide             1520/s                    2791%            1041%               371%              143%                67%                         43%          23%                     --                   -25%                  -39%               -39%                    -59%                 -61%                          -62%               -67%                           -74%              -86%              -89%              -90%                  -92% 
  Text::Table::Manifold             2050/s                    3793%            1436%               535%              227%               125%                         92%          65%                    34%                     --                  -18%               -18%                    -45%                 -48%                          -49%               -56%                           -65%              -81%              -85%              -87%                  -89% 
  Text::TabularDisplay              2000/s                    4650%            1775%               675%              300%               175%                        134%         102%                    64%                    21%                    --                 0%                    -32%                 -37%                          -38%               -47%                           -57%              -77%              -82%              -85%                  -86% 
  Text::Table::Tiny                 2510/s                    4673%            1784%               678%              302%               176%                        136%         103%                    65%                    22%                    0%                 --                    -32%                 -37%                          -38%               -46%                           -57%              -77%              -82%              -84%                  -86% 
  Text::Table::TinyColor            3730/s                    6989%            2698%              1056%              497%               310%                        250%         202%                   145%                    82%                   49%                48%                      --                  -6%                           -8%               -20%                           -36%              -66%              -74%              -77%                  -80% 
  Text::MarkdownTable               4000/s                    7500%            2900%              1140%              540%               340%                        276%         224%                   162%                    95%                   60%                59%                      7%                   --                           -1%               -15%                           -31%              -64%              -72%              -76%                  -79% 
  Text::Table::TinyBorderStyle      4070/s                    7623%            2948%              1160%              550%               347%                        282%         229%                   167%                    98%                   62%                61%                      8%                   1%                            --               -13%                           -30%              -63%              -72%              -75%                  -78% 
  Text::Table::HTML                 4720/s                    8862%            3437%              1362%              654%               418%                        343%         282%                   209%                   130%                   88%                87%                     26%                  17%                           16%                 --                           -19%              -57%              -67%              -71%                  -75% 
  Text::Table::HTML::DataTables     5900/s                   11076%            4311%              1723%              841%               547%                        452%         376%                   286%                   187%                  135%               134%                     57%                  47%                           44%                24%                             --              -47%              -59%              -64%                  -69% 
  Text::Table::Org                 11100/s                   21081%            8261%              3355%             1683%              1126%                        947%         803%                   632%                   444%                  345%               343%                    198%                 178%                          174%               136%                            89%                --              -23%              -33%                  -41% 
  Text::Table::CSV                 14500/s                   27556%           10817%              4412%             2228%              1501%                       1268%        1079%                   856%                   610%                  482%               479%                    290%                 263%                          258%               208%                           147%               30%                --              -12%                  -23% 
  Text::Table::Any                 17000/s                   31566%           12400%              5066%             2566%              1733%                       1466%        1250%                   995%                   713%                  566%               563%                    346%                 316%                          310%               253%                           183%               49%               14%                --                  -12% 
  Text::Table::Sprintf           19136.5/s                   36259%           14252%              5832%             2961%              2005%                       1698%        1450%                  1157%                   833%                  665%               661%                    412%                 378%                          370%               305%                           225%               71%               31%               14%                    -- 
 
 Legends:
   Text::ANSITable: participant=Text::ANSITable
   Text::ASCIITable: participant=Text::ASCIITable
   Text::FormatTable: participant=Text::FormatTable
   Text::MarkdownTable: participant=Text::MarkdownTable
   Text::Table: participant=Text::Table
   Text::Table::Any: participant=Text::Table::Any
   Text::Table::CSV: participant=Text::Table::CSV
   Text::Table::HTML: participant=Text::Table::HTML
   Text::Table::HTML::DataTables: participant=Text::Table::HTML::DataTables
   Text::Table::Manifold: participant=Text::Table::Manifold
   Text::Table::More: participant=Text::Table::More
   Text::Table::Org: participant=Text::Table::Org
   Text::Table::Sprintf: participant=Text::Table::Sprintf
   Text::Table::Tiny: participant=Text::Table::Tiny
   Text::Table::TinyBorderStyle: participant=Text::Table::TinyBorderStyle
   Text::Table::TinyColor: participant=Text::Table::TinyColor
   Text::Table::TinyColorWide: participant=Text::Table::TinyColorWide
   Text::Table::TinyWide: participant=Text::Table::TinyWide
   Text::TabularDisplay: participant=Text::TabularDisplay
   Text::UnicodeBox::Table: participant=Text::UnicodeBox::Table


Benchmark module startup overhead (C<< bencher --cpanmodules-module TextTable --module-startup >>):

 #table6#
 +-------------------------------+-----------+-------------------+-----------------------+-----------------------+-----------+---------+
 | participant                   | time (ms) | mod_overhead_time | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors   | samples |
 +-------------------------------+-----------+-------------------+-----------------------+-----------------------+-----------+---------+
 | Text::UnicodeBox::Table       |     130   |             124   |                 0.00% |              2214.37% |   0.0005  |      20 |
 | Text::Table::Manifold         |      70   |              64   |                87.10% |              1136.99% |   0.00018 |      20 |
 | Text::ANSITable               |      39   |              33   |               232.76% |               595.51% |   0.00021 |      20 |
 | Text::MarkdownTable           |      37   |              31   |               252.71% |               556.16% |   0.00015 |      20 |
 | Text::Table::TinyColorWide    |      29   |              23   |               346.19% |               418.70% |   0.00017 |      21 |
 | Text::Table::TinyWide         |      26   |              20   |               396.49% |               366.15% |   0.00021 |      20 |
 | Text::Table::More             |      21   |              15   |               530.66% |               266.98% |   0.00016 |      20 |
 | Text::Table                   |      20   |              14   |               552.74% |               254.56% |   0.00029 |      20 |
 | Text::Table::Tiny             |      20   |              14   |               727.14% |               179.80% |   0.00024 |      23 |
 | Text::ASCIITable              |      15   |               9   |               764.47% |               167.72% |   0.00014 |      20 |
 | Text::FormatTable             |      10   |               4   |               969.89% |               116.32% |   0.00012 |      20 |
 | Text::Table::TinyColor        |      10   |               4   |               985.12% |               113.28% |   0.00031 |      20 |
 | Text::TabularDisplay          |      10   |               4   |              1230.91% |                73.89% |   0.00022 |      20 |
 | Text::Table::TinyBorderStyle  |      10   |               4   |              1237.62% |                73.02% |   0.00025 |      21 |
 | Text::Table::HTML::DataTables |       9   |               3   |              1387.80% |                55.56% |   0.00012 |      20 |
 | Text::Table::HTML             |       8.6 |               2.6 |              1422.09% |                52.05% | 7.3e-05   |      20 |
 | Text::Table::Any              |       8   |               2   |              1598.36% |                36.27% | 8.7e-05   |      21 |
 | Text::Table::Sprintf          |       7   |               1   |              1838.08% |                19.42% |   0.00013 |      20 |
 | Text::Table::Org              |       7   |               1   |              1853.24% |                18.49% |   0.0001  |      20 |
 | Text::Table::CSV              |       6   |               0   |              1904.61% |                15.45% |   0.00014 |      21 |
 | perl -e1 (baseline)           |       6   |               0   |              2214.37% |                 0.00% |   0.00019 |      20 |
 +-------------------------------+-----------+-------------------+-----------------------+-----------------------+-----------+---------+


Formatted as L<Benchmark.pm|Benchmark> result:

                                  Rate  Text::UnicodeBox::Table  Text::Table::Manifold  Text::ANSITable  Text::MarkdownTable  Text::Table::TinyColorWide  Text::Table::TinyWide  Text::Table::More  Text::Table  Text::Table::Tiny  Text::ASCIITable  Text::FormatTable  Text::Table::TinyColor  Text::TabularDisplay  Text::Table::TinyBorderStyle  Text::Table::HTML::DataTables  Text::Table::HTML  Text::Table::Any  Text::Table::Sprintf  Text::Table::Org  Text::Table::CSV  perl -e1 (baseline) 
  Text::UnicodeBox::Table        0.0/s                       --                   -46%             -70%                 -71%                        -77%                   -80%               -83%         -84%               -84%              -88%               -92%                    -92%                  -92%                          -92%                           -93%               -93%              -93%                  -94%              -94%              -95%                 -95% 
  Text::Table::Manifold          0.0/s                      85%                     --             -44%                 -47%                        -58%                   -62%               -70%         -71%               -71%              -78%               -85%                    -85%                  -85%                          -85%                           -87%               -87%              -88%                  -90%              -90%              -91%                 -91% 
  Text::ANSITable                0.0/s                     233%                    79%               --                  -5%                        -25%                   -33%               -46%         -48%               -48%              -61%               -74%                    -74%                  -74%                          -74%                           -76%               -77%              -79%                  -82%              -82%              -84%                 -84% 
  Text::MarkdownTable            0.0/s                     251%                    89%               5%                   --                        -21%                   -29%               -43%         -45%               -45%              -59%               -72%                    -72%                  -72%                          -72%                           -75%               -76%              -78%                  -81%              -81%              -83%                 -83% 
  Text::Table::TinyColorWide     0.0/s                     348%                   141%              34%                  27%                          --                   -10%               -27%         -31%               -31%              -48%               -65%                    -65%                  -65%                          -65%                           -68%               -70%              -72%                  -75%              -75%              -79%                 -79% 
  Text::Table::TinyWide          0.0/s                     400%                   169%              50%                  42%                         11%                     --               -19%         -23%               -23%              -42%               -61%                    -61%                  -61%                          -61%                           -65%               -66%              -69%                  -73%              -73%              -76%                 -76% 
  Text::Table::More              0.0/s                     519%                   233%              85%                  76%                         38%                    23%                 --          -4%                -4%              -28%               -52%                    -52%                  -52%                          -52%                           -57%               -59%              -61%                  -66%              -66%              -71%                 -71% 
  Text::Table                    0.1/s                     550%                   250%              95%                  85%                         44%                    30%                 5%           --                 0%              -25%               -50%                    -50%                  -50%                          -50%                           -55%               -57%              -60%                  -65%              -65%              -70%                 -70% 
  Text::Table::Tiny              0.1/s                     550%                   250%              95%                  85%                         44%                    30%                 5%           0%                 --              -25%               -50%                    -50%                  -50%                          -50%                           -55%               -57%              -60%                  -65%              -65%              -70%                 -70% 
  Text::ASCIITable               0.1/s                     766%                   366%             160%                 146%                         93%                    73%                39%          33%                33%                --               -33%                    -33%                  -33%                          -33%                           -40%               -42%              -46%                  -53%              -53%              -60%                 -60% 
  Text::FormatTable              0.1/s                    1200%                   600%             290%                 270%                        190%                   160%               110%         100%               100%               50%                 --                      0%                    0%                            0%                            -9%               -14%              -19%                  -30%              -30%              -40%                 -40% 
  Text::Table::TinyColor         0.1/s                    1200%                   600%             290%                 270%                        190%                   160%               110%         100%               100%               50%                 0%                      --                    0%                            0%                            -9%               -14%              -19%                  -30%              -30%              -40%                 -40% 
  Text::TabularDisplay           0.1/s                    1200%                   600%             290%                 270%                        190%                   160%               110%         100%               100%               50%                 0%                      0%                    --                            0%                            -9%               -14%              -19%                  -30%              -30%              -40%                 -40% 
  Text::Table::TinyBorderStyle   0.1/s                    1200%                   600%             290%                 270%                        190%                   160%               110%         100%               100%               50%                 0%                      0%                    0%                            --                            -9%               -14%              -19%                  -30%              -30%              -40%                 -40% 
  Text::Table::HTML::DataTables  0.1/s                    1344%                   677%             333%                 311%                        222%                   188%               133%         122%               122%               66%                11%                     11%                   11%                           11%                             --                -4%              -11%                  -22%              -22%              -33%                 -33% 
  Text::Table::HTML              0.1/s                    1411%                   713%             353%                 330%                        237%                   202%               144%         132%               132%               74%                16%                     16%                   16%                           16%                             4%                 --               -6%                  -18%              -18%              -30%                 -30% 
  Text::Table::Any               0.1/s                    1525%                   775%             387%                 362%                        262%                   225%               162%         150%               150%               87%                25%                     25%                   25%                           25%                            12%                 7%                --                  -12%              -12%              -25%                 -25% 
  Text::Table::Sprintf           0.1/s                    1757%                   900%             457%                 428%                        314%                   271%               200%         185%               185%              114%                42%                     42%                   42%                           42%                            28%                22%               14%                    --                0%              -14%                 -14% 
  Text::Table::Org               0.1/s                    1757%                   900%             457%                 428%                        314%                   271%               200%         185%               185%              114%                42%                     42%                   42%                           42%                            28%                22%               14%                    0%                --              -14%                 -14% 
  Text::Table::CSV               0.2/s                    2066%                  1066%             550%                 516%                        383%                   333%               250%         233%               233%              150%                66%                     66%                   66%                           66%                            50%                43%               33%                   16%               16%                --                   0% 
  perl -e1 (baseline)            0.2/s                    2066%                  1066%             550%                 516%                        383%                   333%               250%         233%               233%              150%                66%                     66%                   66%                           66%                            50%                43%               33%                   16%               16%                0%                   -- 
 
 Legends:
   Text::ANSITable: mod_overhead_time=33 participant=Text::ANSITable
   Text::ASCIITable: mod_overhead_time=9 participant=Text::ASCIITable
   Text::FormatTable: mod_overhead_time=4 participant=Text::FormatTable
   Text::MarkdownTable: mod_overhead_time=31 participant=Text::MarkdownTable
   Text::Table: mod_overhead_time=14 participant=Text::Table
   Text::Table::Any: mod_overhead_time=2 participant=Text::Table::Any
   Text::Table::CSV: mod_overhead_time=0 participant=Text::Table::CSV
   Text::Table::HTML: mod_overhead_time=2.6 participant=Text::Table::HTML
   Text::Table::HTML::DataTables: mod_overhead_time=3 participant=Text::Table::HTML::DataTables
   Text::Table::Manifold: mod_overhead_time=64 participant=Text::Table::Manifold
   Text::Table::More: mod_overhead_time=15 participant=Text::Table::More
   Text::Table::Org: mod_overhead_time=1 participant=Text::Table::Org
   Text::Table::Sprintf: mod_overhead_time=1 participant=Text::Table::Sprintf
   Text::Table::Tiny: mod_overhead_time=14 participant=Text::Table::Tiny
   Text::Table::TinyBorderStyle: mod_overhead_time=4 participant=Text::Table::TinyBorderStyle
   Text::Table::TinyColor: mod_overhead_time=4 participant=Text::Table::TinyColor
   Text::Table::TinyColorWide: mod_overhead_time=23 participant=Text::Table::TinyColorWide
   Text::Table::TinyWide: mod_overhead_time=20 participant=Text::Table::TinyWide
   Text::TabularDisplay: mod_overhead_time=4 participant=Text::TabularDisplay
   Text::UnicodeBox::Table: mod_overhead_time=124 participant=Text::UnicodeBox::Table
   perl -e1 (baseline): mod_overhead_time=0 participant=perl -e1 (baseline)

To display as an interactive HTML table on a browser, you can add option C<--format html+datatables>.

=head1 SAMPLE OUTPUTS

This section shows what the output is like for (some of the) modules:

=over

=item * L</Text::Table::Any>

 +--------+--------+--------+
 | col1   | col2   | col3   |
 +--------+--------+--------+
 | row1.1 | row1.2 | row1.3 |
 | row2.1 | row2.2 | row2.3 |
 | row3.1 | row3.2 | row3.3 |
 | row4.1 | row4.2 | row4.3 |
 | row5.1 | row5.2 | row5.3 |
 +--------+--------+--------+

=item * L</Text::Table::Manifold>

 +--------+--------+--------+
 |  col1  |  col2  |  col3  |
 +--------+--------+--------+
 | row1.1 | row1.2 | row1.3 |
 | row2.1 | row2.2 | row2.3 |
 | row3.1 | row3.2 | row3.3 |
 | row4.1 | row4.2 | row4.3 |
 | row5.1 | row5.2 | row5.3 |
 +--------+--------+--------+

=item * L</Text::ANSITable>

 .--------+--------+--------.
 | col1   | col2   | col3   |
 +--------+--------+--------+
 | row1.1 | row1.2 | row1.3 |
 | row2.1 | row2.2 | row2.3 |
 | row3.1 | row3.2 | row3.3 |
 | row4.1 | row4.2 | row4.3 |
 | row5.1 | row5.2 | row5.3 |
 `--------+--------+--------'

=item * L</Text::ASCIITable>

 .--------------------------.
 | col1   | col2   | col3   |
 +--------+--------+--------+
 | row1.1 | row1.2 | row1.3 |
 | row2.1 | row2.2 | row2.3 |
 | row3.1 | row3.2 | row3.3 |
 | row4.1 | row4.2 | row4.3 |
 | row5.1 | row5.2 | row5.3 |
 '--------+--------+--------'

=item * L</Text::FormatTable>

 col1  |col2  |col3  
 row1.1|row1.2|row1.3
 row2.1|row2.2|row2.3
 row3.1|row3.2|row3.3
 row4.1|row4.2|row4.3
 row5.1|row5.2|row5.3

=item * L</Text::MarkdownTable>

 | col1   | col2   | col3   |
 |--------|--------|--------|
 | row1.1 | row1.2 | row1.3 |
 | row2.1 | row2.2 | row2.3 |
 | row3.1 | row3.2 | row3.3 |
 | row4.1 | row4.2 | row4.3 |
 | row5.1 | row5.2 | row5.3 |

=item * L</Text::Table>

 col1   col2   col3  
 row1.1 row1.2 row1.3
 row2.1 row2.2 row2.3
 row3.1 row3.2 row3.3
 row4.1 row4.2 row4.3
 row5.1 row5.2 row5.3

=item * L</Text::Table::Tiny>

 +--------+--------+--------+
 | col1   | col2   | col3   |
 +--------+--------+--------+
 | row1.1 | row1.2 | row1.3 |
 | row2.1 | row2.2 | row2.3 |
 | row3.1 | row3.2 | row3.3 |
 | row4.1 | row4.2 | row4.3 |
 | row5.1 | row5.2 | row5.3 |
 +--------+--------+--------+

=item * L</Text::Table::TinyBorderStyle>

 .--------+--------+--------.
 | col1   | col2   | col3   |
 +--------+--------+--------+
 | row1.1 | row1.2 | row1.3 |
 | row2.1 | row2.2 | row2.3 |
 | row3.1 | row3.2 | row3.3 |
 | row4.1 | row4.2 | row4.3 |
 | row5.1 | row5.2 | row5.3 |
 `--------+--------+--------'

=item * L</Text::Table::More>

 .--------+--------+--------.
 | col1   | col2   | col3   |
 +========+========+========+
 | row1.1 | row1.2 | row1.3 |
 | row2.1 | row2.2 | row2.3 |
 | row3.1 | row3.2 | row3.3 |
 | row4.1 | row4.2 | row4.3 |
 | row5.1 | row5.2 | row5.3 |
 `--------+--------+--------'

=item * L</Text::Table::Sprintf>

 +--------+--------+--------+
 | col1   | col2   | col3   |
 +--------+--------+--------+
 | row1.1 | row1.2 | row1.3 |
 | row2.1 | row2.2 | row2.3 |
 | row3.1 | row3.2 | row3.3 |
 | row4.1 | row4.2 | row4.3 |
 | row5.1 | row5.2 | row5.3 |
 +--------+--------+--------+

=item * L</Text::Table::TinyColor>

 +--------+--------+--------+
 | col1   | col2   | col3   |
 +--------+--------+--------+
 | row1.1 | row1.2 | row1.3 |
 | row2.1 | row2.2 | row2.3 |
 | row3.1 | row3.2 | row3.3 |
 | row4.1 | row4.2 | row4.3 |
 | row5.1 | row5.2 | row5.3 |
 +--------+--------+--------+

=item * L</Text::Table::TinyColorWide>

 +--------+--------+--------+
 | col1   | col2   | col3   |
 +--------+--------+--------+
 | row1.1 | row1.2 | row1.3 |
 | row2.1 | row2.2 | row2.3 |
 | row3.1 | row3.2 | row3.3 |
 | row4.1 | row4.2 | row4.3 |
 | row5.1 | row5.2 | row5.3 |
 +--------+--------+--------+

=item * L</Text::Table::TinyWide>

 +--------+--------+--------+
 | col1   | col2   | col3   |
 +--------+--------+--------+
 | row1.1 | row1.2 | row1.3 |
 | row2.1 | row2.2 | row2.3 |
 | row3.1 | row3.2 | row3.3 |
 | row4.1 | row4.2 | row4.3 |
 | row5.1 | row5.2 | row5.3 |
 +--------+--------+--------+

=item * L</Text::Table::Org>

 | col1   | col2   | col3   |
 |--------+--------+--------|
 | row1.1 | row1.2 | row1.3 |
 | row2.1 | row2.2 | row2.3 |
 | row3.1 | row3.2 | row3.3 |
 | row4.1 | row4.2 | row4.3 |
 | row5.1 | row5.2 | row5.3 |

=item * L</Text::Table::CSV>

 "col1","col2","col3"
 "row1.1","row1.2","row1.3"
 "row2.1","row2.2","row2.3"
 "row3.1","row3.2","row3.3"
 "row4.1","row4.2","row4.3"
 "row5.1","row5.2","row5.3"

=item * L</Text::Table::HTML>

 <table>
 <thead>
 <tr><th>col1</th><th>col2</th><th>col3</th></tr>
 </thead>
 <tbody>
 <tr><td>row1.1</td><td>row1.2</td><td>row1.3</td></tr>
 <tr><td>row2.1</td><td>row2.2</td><td>row2.3</td></tr>
 <tr><td>row3.1</td><td>row3.2</td><td>row3.3</td></tr>
 <tr><td>row4.1</td><td>row4.2</td><td>row4.3</td></tr>
 <tr><td>row5.1</td><td>row5.2</td><td>row5.3</td></tr>
 </tbody>
 </table>

=item * L</Text::Table::HTML::DataTables>

 <html>
 <head>
 <link rel="stylesheet" type="text/css" href="file:///home/s1/perl5/perlbrew/perls/perl-5.30.2/lib/site_perl/5.30.2/auto/share/dist/Text-Table-HTML-DataTables/datatables-1.10.22/datatables.css">
 <script src="file:///home/s1/perl5/perlbrew/perls/perl-5.30.2/lib/site_perl/5.30.2/auto/share/dist/Text-Table-HTML-DataTables/jquery-2.2.4/jquery-2.2.4.min.js"></script>
 <script src="file:///home/s1/perl5/perlbrew/perls/perl-5.30.2/lib/site_perl/5.30.2/auto/share/dist/Text-Table-HTML-DataTables/datatables-1.10.22/datatables.js"></script>
 <script>var dt_opts = {"buttons":["colvis","print"],"dom":"lQfrtip"}; $(document).ready(function() { $("table").DataTable(dt_opts); });</script>
 
 </head>
 
 <body>
 <table>
 <thead>
 <tr><th>col1</th><th>col2</th><th>col3</th></tr>
 </thead>
 <tbody>
 <tr><td>row1.1</td><td>row1.2</td><td>row1.3</td></tr>
 <tr><td>row2.1</td><td>row2.2</td><td>row2.3</td></tr>
 <tr><td>row3.1</td><td>row3.2</td><td>row3.3</td></tr>
 <tr><td>row4.1</td><td>row4.2</td><td>row4.3</td></tr>
 <tr><td>row5.1</td><td>row5.2</td><td>row5.3</td></tr>
 </tbody>
 </table>
 </body>
 
 </html>

=item * L</Text::TabularDisplay>

 +--------+--------+--------+
 | col1   | col2   | col3   |
 +--------+--------+--------+
 | row1.1 | row1.2 | row1.3 |
 | row2.1 | row2.2 | row2.3 |
 | row3.1 | row3.2 | row3.3 |
 | row4.1 | row4.2 | row4.3 |
 | row5.1 | row5.2 | row5.3 |
 +--------+--------+--------+

=back

=head1 FAQ

=head2 What is an Acme::CPANModules::* module?

An Acme::CPANModules::* module, like this module, contains just a list of module
names that share a common characteristics. It is a way to categorize modules and
document CPAN. See L<Acme::CPANModules> for more details.

=head2 What are ways to use this Acme::CPANModules module?

Aside from reading this Acme::CPANModules module's POD documentation, you can
install all the listed modules (entries) using L<cpanm-cpanmodules> script (from
L<App::cpanm::cpanmodules> distribution):

 % cpanm-cpanmodules -n TextTable

Alternatively you can use the L<cpanmodules> CLI (from L<App::cpanmodules>
distribution):

    % cpanmodules ls-entries TextTable | cpanm -n

or L<Acme::CM::Get>:

    % perl -MAcme::CM::Get=TextTable -E'say $_->{module} for @{ $LIST->{entries} }' | cpanm -n

or directly:

    % perl -MAcme::CPANModules::TextTable -E'say $_->{module} for @{ $Acme::CPANModules::TextTable::LIST->{entries} }' | cpanm -n

This Acme::CPANModules module contains benchmark instructions. You can run a
benchmark for some/all the modules listed in this Acme::CPANModules module using
the L<bencher> CLI (from L<Bencher> distribution):

    % bencher --cpanmodules-module TextTable

This Acme::CPANModules module also helps L<lcpan> produce a more meaningful
result for C<lcpan related-mods> command when it comes to finding related
modules for the modules listed in this Acme::CPANModules module.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Acme-CPANModules-TextTable>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Acme-CPANModules-TextTable>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Acme-CPANModules-TextTable>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 SEE ALSO

L<Acme::CPANModules::HTMLTable>

L<Acme::CPANModules> - about the Acme::CPANModules namespace

L<cpanmodules> - CLI tool to let you browse/view the lists

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2021, 2020, 2019 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
