package Net::DNS::SEC::Private;

#
# $Id: Private.pm 1287 2014-12-19 08:18:17Z willem $
#
use vars qw($VERSION);
$VERSION = (qw$LastChangedRevision: 1287 $)[1];


=head1 NAME

Net::DNS::SEC::Private - DNSSEC Private key object


=head1 SYNOPSIS

    use Net::DNS::SEC::Private;

    $private = new Net::DNS::SEC::Private($keypath);


=head1 DESCRIPTION

Class representing private keys as read from a keyfile generated by
dnssec-keygen. The class is written to be used only in the context of the
Net::DNS::RR::RRSIG create method. This class is not designed to interact
with any other system.

=cut


use strict;
use integer;
use warnings;
use Carp;
use File::Spec;
use FileHandle;

use base qw(Net::DNS::SEC);

my $debug = 0;


sub new {
	my $self = bless {}, shift;
	my $file = shift;

	my ( $vol, $dir, $keyname ) = File::Spec->splitpath($file);

	print "\nKeyname: $keyname\n" if $debug;

	# Format something like: /Kbla.foo.+001+60114.private'
	# assuming proper file name.
	# We determine the algorithm from the filename.
	croak "$keyname does not appear to be a private key\n"
			unless $keyname =~ /^K(.*\.)\+(\d+)\+(\d+)\.private$/;
	@{$self}{qw(signame algorithm keytag)} = ( $1, 0 + $2, $3 );

	my $handle = new FileHandle($file) or croak qq(open: "$file" $!);

	local $_;
	while (<$handle>) {
		s/\(.+\)//;
		my ( $name, $value ) = split;
		$name =~ tr/A-Za-z0-9\000-\377/a-za-z0-9/d;
		$self->{$name} = $value;
	}

	return $self;
}


use vars qw($AUTOLOAD);

sub DESTROY { }				## Avoid tickling AUTOLOAD (in cleanup)

sub AUTOLOAD {				## Default method
	my ($self) = @_;
	no strict q/refs/;

	my ($attribute) = $AUTOLOAD =~ m/::([^:]*)$/;
	$attribute =~ tr/A-Za-z0-9\000-\377/a-za-z0-9/d;

	return undef unless exists $self->{$attribute};

	# Build a method in the class
	*{$AUTOLOAD} = sub { shift->{$attribute} };

	# and jump to it
	goto &{$AUTOLOAD};
}


1;

__END__



=head1 METHODS

=head2 new

    $private = new Net::DNS::SEC::Private (
		'/home/foo/Kexample.com.+001+11567.private'
		);

Creator method. The argument is the full path to a private key
generated by the BIND dnssec-keygen tool. Note that the filename
contains information about the algorithm and keyid.


=head2 private_key_format

    $format = $private->private_key_format;

Returns a string which identifies the format of the private key file.


=head2 created, publish, activate

    $created  = $private->created;
    $publish  = $private->publish;
    $activate = $private->activate;

Returns a string which represents a date in the form 20141212123456.
Returns undefined value for key formats older than v1.3.


=head2 algorithm, keytag, signame
 
    $algorithm = $private->algorithm;
    $keytag    = $private->keytag;
    $signame   = $private->signame;

Returns the corresponding attribute determined from the filename.


=head2 Private key attributes

    $attribute = $private->attribute;

Returns the value as it appears in the private key file.
The attribute names correspond to the tag in the key file, modified
to form an acceptable Perl subroutine name.


=head1 COPYRIGHT

Copyright (c)2014 Dick Franks

All Rights Reserved

This program is free software; you may redistribute it and/or modify it
under the same terms as Perl itself.


=head1 SEE ALSO

L<perl>, L<Net::DNS>,
L<Net::DNS::RR::RRSIG>, L<Net::DNS::RR::SIG>

=cut

