package Net::DNS::RR::CERT;

#
# $Id: CERT.pm 1347 2015-05-19 10:24:36Z willem $
#
use vars qw($VERSION);
$VERSION = (qw$LastChangedRevision: 1347 $)[1];


use strict;
use base qw(Net::DNS::RR);

=head1 NAME

Net::DNS::RR::CERT - DNS CERT resource record

=cut


use integer;

use MIME::Base64;

my %formats = (
	PKIX	=> 1,						# X.509 as per PKIX
	SPKI	=> 2,						# SPKI certificate
	PGP	=> 3,						# OpenPGP packet
	IPKIX	=> 4,						# The URL of an X.509 data object
	ISPKI	=> 5,						# The URL of an SPKI certificate
	IPGP	=> 6,						# The fingerprint and URL of an OpenPGP packet
	ACPKIX	=> 7,						# Attribute Certificate
	IACPKIX => 8,						# The URL of an Attribute Certificate
	URI	=> 253,						# URI private
	OID	=> 254,						# OID private
	);

my %r_formats = reverse %formats;


my %algorithms = (						# RFC4034 except where noted
	RSAMD5	   => 1,
	DH	   => 2,
	DSA	   => 3,
	ECC	   => 4,
	RSASHA1	   => 5,
	RESERVE123 => 123,					# RFC6014
	RESERVE251 => 251,					# RFC6014
	INDIRECT   => 252,
	PRIVATEDNS => 253,
	PRIVATEOID => 254,
	);

my %r_algorithms = reverse %algorithms;


sub decode_rdata {			## decode rdata from wire-format octet string
	my $self = shift;
	my ( $data, $offset ) = @_;

	@{$self}{qw(format tag algorithm)} = unpack "\@$offset n2 C", $$data;
	$self->{certbin} = substr $$data, $offset + 5, $self->{rdlength} - 5;
}


sub encode_rdata {			## encode rdata as wire-format octet string
	my $self = shift;

	return '' unless $self->{certbin};
	pack "n2 C a*", @{$self}{qw(format tag algorithm certbin)};
}


sub format_rdata {			## format rdata portion of RR string.
	my $self = shift;

	my @base64 = split /\s+/, encode_base64( $self->{certbin} || return '' );
	my $algorithm = $r_algorithms{$self->{algorithm}} || $self->{algorithm};
	my $format    = $r_formats{$self->{format}}	  || $self->{format};
	my @rdata = $format, $self->{tag}, $algorithm, @base64;
}


sub parse_rdata {			## populate RR from rdata in argument list
	my $self = shift;

	$self->$_(shift) for qw(format tag algorithm);
	$self->cert(@_);
}


sub format {
	my $self = shift;

	return $self->{format} unless scalar @_;

	my $format = shift;
	$format = '<undef>' unless defined $format;
	$format = $formats{$format} || die "Unknown mnemonic: '$format'"
			if $format =~ /\D/;			# look up mnemonic
	$self->{format} = $format;
}


sub tag {
	my $self = shift;

	$self->{tag} = 0 + shift if scalar @_;
	return $self->{tag} || 0;
}


sub algorithm {
	my $self = shift;

	return $self->{algorithm} unless scalar @_;

	my $algorithm = shift;
	$algorithm = '<undef>' unless defined $algorithm;
	$algorithm = $algorithms{$algorithm} || die "Unknown mnemonic: '$algorithm'"
			if $algorithm =~ /\D/;			# look up mnemonic
	$self->{algorithm} = $algorithm;
}


sub cert {
	my $self = shift;

	$self->certbin( MIME::Base64::decode( join "", @_ ) ) if scalar @_;
	return MIME::Base64::encode( $self->certbin(), "" ) if defined wantarray;
}


sub certbin {
	my $self = shift;

	$self->{certbin} = shift if scalar @_;
	$self->{certbin} || "";
}


sub certificate { &certbin; }		## historical


1;
__END__


=head1 SYNOPSIS

    use Net::DNS;
    $rr = new Net::DNS::RR('name IN CERT format tag algorithm cert');

=head1 DESCRIPTION

Class for DNS Certificate (CERT) resource records.

=head1 METHODS

The available methods are those inherited from the base class augmented
by the type-specific methods defined in this package.

Use of undocumented package features or direct access to internal data
structures is discouraged and could result in program termination or
other unpredictable behaviour.


=head2 format

    $format =  $rr->format;

Returns the format code for the certificate (in numeric form)

=head2 tag

    $tag = $rr->tag;
    $rr->tag( $tag );

Returns the key tag for the public key in the certificate

=head2 algorithm

    $algorithm = $rr->algorithm;

Returns the algorithm used by the certificate (in numeric form).

=head2 cert

    $cert = $rr->cert;
    $rr->cert( $cert );

Base64 representation of the certificate.

=head2 certbin

    $certbin = $rr->certbin;
    $rr->certbin( $certbin );

Binary representation of the certificate.


=head1 COPYRIGHT

Copyright (c)2002 VeriSign, Mike Schiraldi

All rights reserved.

Package template (c)2009,2012 O.M.Kolkman and R.W.Franks.


=head1 LICENSE

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted, provided
that the above copyright notice appear in all copies and that both that
copyright notice and this permission notice appear in supporting
documentation, and that the name of the author not be used in advertising
or publicity pertaining to distribution of the software without specific
prior written permission.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.


=head1 SEE ALSO

L<perl>, L<Net::DNS>, L<Net::DNS::RR>, RFC4398

=cut
