#
#
#

package MY;

#
# the MakeMaker distributed with perl 5.8.x generates makefiles with
# a bug that causes object files to be created in the wrong directory.
# this override fixes that.
#

sub const_cccmd {
	my $inherited = shift->SUPER::const_cccmd (@_);
	use Config;
	# a more sophisticated match may be necessary, but this works for me.
	if ($Config{cc} eq "cl") {
		warn "you are using MSVC... my condolences.\n";
		$inherited .= ' /Fo$@';
	} else {
		$inherited .= ' -o $@';
	}
	$inherited;
}

package Glib::MakeHelper;

=head1 NAME

Glib::MakeHelper - Makefile.PL utilities for Glib-based extensions

=head1 SYNOPSIS

 eval "use Glib::MakeHelper; 1"
     or complain_that_glib_is_too_old_and_die();
 
 %xspod_files = Glib::MakeHelper->do_pod_files (@xs_files);

 package MY;
 sub postamble {
     return Glib::MakeHelper->postamble_clean ()
          . Glib::MakeHelper->postamble_docs (@main::xs_files)
          . Glib::MakeHelper->postamble_rpms (
                 MYLIB_RUN => $runtime_reqs{MyLib},
                 MYLIB     => $build_reqs{MyLib},
            );
 }

=head1 DESCRIPTION

The Makefile.PL for your typical Glib-based module is huge and hairy, thanks to
all the crazy hoops you have to jump through to get things right.  This module
wraps up some of the more intense and error-prone bits to reduce the amount of
copied code and potential for errors.

=cut

use strict;
use warnings;
use Carp;
use Cwd;

our @gend_pods = ();

=head1 METHODS

=over

=item HASH = Glib::MakeHelper->do_pod_files (@xs_files)

Scan the I<@xs_files> and return a hash describing the pod files that will
be created.  This is in the format wanted by WriteMakefile().

=cut
sub do_pod_files
{
	shift; # package name

	# try to get it from pwd first, then fall back to installed
	# this is so Glib will get associated copy, and everyone else
	# should use the installed glib copy
	eval { require 'ParseXSDoc.pm'; 1; } or require Glib::ParseXSDoc;
	$@ = undef;
	import Glib::ParseXSDoc;

	my %pod_files = ();

	open PARSE, '>build/doc.pl';
	select PARSE;
	my $pods = xsdocparse (@_);
	select STDOUT;
	@gend_pods = ();
	foreach (@$pods)
	{
		my $pod = $_;
		my $path = '$(INST_LIB)';
		$pod = File::Spec->catfile ($path, split (/::/, $_)) . ".pod";
		push @gend_pods, $pod;
		$pod_files{$pod} = '$(INST_MAN3DIR)/'.$_.'.$(MAN3EXT)';
	}
	$pod_files{'$(INST_LIB)/$(FULLEXT)/index.pod'} = '$(INST_MAN3DIR)/$(NAME)::index.$(MAN3EXT)';

	return %pod_files;
}

=item string = Glib::MakeHelper->postamble_clean (@files)

Create and return the text of a realclean rule that cleans up after much 
of the autogeneration performed by Glib-based modules.  Everything in @files
will be deleted, too (it may be empty).

The reasoning behind using this instead of just having you use the 'clean'
or 'realclean' keys is that this avoids you having to remember to put Glib's
stuff in your Makefile.PL's WriteMakefile arguments.

=cut
sub postamble_clean
{
	shift; # package name
"
realclean ::
	-\$(RM_RF) build perl-\$(DISTNAME).spec ".join(" ", @_)."
";
}

=item string = Glib::MakeHelper->postamble_docs (@xs_files)

Create and return the text of Makefile rules to build documentation from
the XS files with Glib::ParseXSDoc and Glib::GenPod.

Use this in your MY::postamble to enable autogeneration of POD.

This updates dependencies with the list of pod names generated by an earlier
run of C<do_pod_files>.

=cut
sub postamble_docs
{
	shift; # package name
	my @xs_files = @_;
"
# documentation stuff
build/doc.pl :: Makefile @xs_files 
	$^X -I \$(INST_LIB) -I \$(INST_ARCHLIB) -MGlib::ParseXSDoc \\
		-e 'xsdocparse (".join(", ",map {"\"$_\""} @xs_files).")' > \$@

build/xsapi.pod :: build/doc.pl apidoc.pl xsapi.pod.head xsapi.pod.foot
	$^X apidoc.pl xsapi.pod.head xsapi.pod.foot build/doc.pl > \$@

# passing all of these files through the single podindex file, which is 
# created at the same time, prevents problems with -j4 where xsdoc2pod would 
# have multiple instances
@gend_pods :: build/podindex

build/podindex :: Makefile build/doc.pl \$(INST_LIB)/\$(FULLEXT).pm \$(INST_DYNAMIC)
	$^X -I \$(INST_LIB) -I \$(INST_ARCHLIB) -MGlib::GenPod -M\$(NAME) \\
		-e \"xsdoc2pod('build/doc.pl', '\$(INST_LIB)', 'build/podindex')\"

\$(INST_LIB)/\$(FULLEXT)/index.pod :: build/podindex
	$^X -e 'print \"\\n=head1 NAME\\n\\n\$(NAME) API Reference Pod Index\\n\\n=head1 PAGES\\n\\n=over\\n\\n\"' \\
		> \$(INST_LIB)/\$(FULLEXT)/index.pod
	$^X -nae 'print \"=item L<\$\$F[1]>\\n\\n\";' < build/podindex >> \$(INST_LIB)/\$(FULLEXT)/index.pod
	$^X -e 'print \"=back\\n\\n\";' >> \$(INST_LIB)/\$(FULLEXT)/index.pod
"
}

=item string = Glib::MakeHelper->postamble_rpms (HASH)

Create and return the text of Makefile rules to manage building RPMs.
You'd put this in your Makefile.PL's MY::postamble.

I<HASH> is a set of search and replace keys for the spec file.  All 
occurences of @I<key>@ in the spec file template perl-$(DISTNAME).spec.in
will be replaced with I<value>.  'VERSION' and 'SOURCE' are supplied for
you.  For example:

 Glib::MakeHelper->postamble_rpms (
        MYLIB     => 2.0.0, # we can work with anything from this up
        MYLIB_RUN => 2.3.1, # we are actually compiled against this one
        PERL_GLIB => 1.01,  # you must have this version of Glib
 );

will replace @MYLIB@, @MYLIB_RUN@, and @PERL_GLIB@ in spec file.  See
the build setups for Glib and Gtk2 for examples.

Note: This function just returns an empty string on Win32.

=cut
sub postamble_rpms
{
	shift; # package name

	return '' if $^O eq 'MSWin32';
	
	my @dirs = qw{rpms rpms/BUILD rpms/RPMS rpms/SOURCES
		      rpms/SPECS rpms/SRPMS};
	my $cwd = getcwd();
	
	my %subs = (
		'VERSION' => '$(VERSION)',
		'SOURCE' => '$(DISTNAME)-$(VERSION).tar.gz',
		@_,
	);
	
	my $substitute = '$(PERL) -npe \''.join('; ', map {
			"s/\\\@$_\\\@/$subs{$_}/g";
		} keys %subs).'\'';

"
rpms/:
	-mkdir @dirs

SUBSTITUTE=$substitute

perl-\$(DISTNAME).spec :: perl-\$(DISTNAME).spec.in \$(VERSION_FROM) Makefile
	\$(SUBSTITUTE) \$< > \$@

dist-rpms :: Makefile dist perl-\$(DISTNAME).spec rpms/
	cp \$(DISTNAME)-\$(VERSION).tar.gz rpms/SOURCES/
	rpmbuild -ba --define \"_topdir $cwd/rpms\" perl-\$(DISTNAME).spec
";
}

1;

=back

=head1 AUTHOR

Ross McFarland E<lt>rwmcfa1 at neces dot comE<gt>

hacked up and documented by muppet.

=cut
