package App::bif::drop::project;
use strict;
use warnings;
use Bif::Mo;
use DBIx::ThinSQL qw/sq/;

our $VERSION = '0.1.5_6';
extends 'App::bif';

sub run {
    my $self = shift;
    my $opts = $self->opts;
    my $dbw  = $self->dbw;
    my $info = $self->get_project( $opts->{path} );

    if ( !$opts->{force} ) {
        print "Nothing dropped (missing --force, -f)\n";
        return $self->ok('DropNoForce');
    }

    $dbw->txn(
        sub {
            if ( $info->{default_hub_id} ) {
                $self->start_work(
                    start         => time,
                    start_comment => "drop project $info->{path} (shallow)",
                    billable      => 0,
                    save          => 1,
                );

                $self->drop_shallow($info);

                print "Project dropped: $info->{path} (shallow)\n";
            }
            else {
                $self->start_work(
                    start         => time,
                    start_comment => "drop project $info->{path}",
                    billable      => 0,
                    save          => 1,
                );

                $dbw->xdo(
                    delete_from => 'projects',
                    where       => { id => $info->{id} },
                );

                print "Project dropped: $info->{path})\n";
            }

            $self->stop_work(
                stop    => time,
                restore => 1,
            );
        }
    );

    return $self->ok('DropProjectShallow') if $info->{default_hub_id};
    return $self->ok('DropProject');
}

sub drop_shallow {
    my $self = shift;
    my $info = shift;
    my $dbw  = $self->dbw;

    # Drop topics that are not part of any other project
    my $res = $dbw->xdo(
        with => 'x',
        as   => sq(
            select => [ 't.link_topic_id', 'COUNT(t.link_topic_id) AS total' ],
            from   => 'topics t',
            where    => { 't.project_id' => $info->{id} },
            group_by => 't.id',
            having   => 'total = 1',
        ),
        delete_from => 'link_topics',
        where       => [
            'id IN ',
            sq(
                select => 'x.id',
                from   => 'x',
            )
        ],
    );

 # Delete project node entities, except those which are
 # our own identities, project entities, or are also
 # other project node entities.
 #    $res += $dbw->xdo(
 #        delete_from => 'entities',
 #        where       => [
 #            'id IN ',
 #            sq(
 #                select        => 'pte.entity_id',
 #                from          => 'project_node_entities pte',
 #                where         => { 'pte.project_id' => $info->{id} },
 #                except_select => 'x.entity_id',
 #                from          => sq(
 #                    select           => 'bif.identity_id AS entity_id',
 #                    from             => 'bifkv bif',
 #                    where            => { 'bif.key' => 'self' },
 #                    union_all_select => 'pe.entity_id',
 #                    from             => 'project_entities pe',
 #                    where            => { 'pe.project_id' => $info->{id} },
 #                    union_all_select => 'pte.entity_id',
 #                    from             => 'project_node_entities pte',
 #                    where            => { 'pte.project_id !' => $info->{id} },
 #                )->as('x'),
 #            ),
 #        ],
 #    );

    $res += $dbw->xdo(
        update => 'projects',
        set    => { local => 0 },
        where  => { id => $info->{id} },
    );
    return $res;
}

1;
__END__

=head1 NAME

=for bif-doc #delete

bif-drop-project - remove an project from the repository

=head1 VERSION

0.1.5_6 (2015-10-20)

=head1 SYNOPSIS

    bif drop project PATH [OPTIONS...]

=head1 DESCRIPTION

The bif-drop-project command removes a project from the repository.

=head1 ARGUMENTS

=over

=item PATH

A project path or ID.

=back

=head1 OPTIONS

=over

=item --force, -f

Actually do the drop. This option is required as a safety measure to
stop you shooting yourself in the foot.

=back

=head1 SEE ALSO

L<bif>(1)

=head1 AUTHOR

Mark Lawrence E<lt>nomad@null.netE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2013-2015 Mark Lawrence <nomad@null.net>

This program is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 3 of the License, or (at your
option) any later version.

