package App::bif::show::issue;
use strict;
use warnings;
use Bif::Mo;
use DBIx::ThinSQL qw/bv sum case coalesce concat qv sq/;

our $VERSION = '0.1.5_2';
extends 'App::bif::show';

sub run {
    my $self = shift;
    my $opts = $self->opts;
    my $db   = $self->db;

    $opts->{id} = $self->uuid2id( $opts->{id} );

    my $info = $self->get_topic( $opts->{id}, 'issue' );
    my $now = $self->now;
    my @data;

    my $ref = $db->xhashref(
        select => [
            'SUBSTR(t.uuid,1,8) as uuid',
            'i.title',
            't.ctime',
            't.ctimetz',
            't.ctimetzhm AS ctimetzhm',
            "$now - t.ctime AS ctime_age",
            't.mtime',
            't.mtimetz',
            't.mtimetzhm AS mtimetzhm',
            "$now - t.mtime AS mtime_age",
            'e1.name as creator',
            'e2.name as updator',
        ],
        from       => 'topics t',
        inner_join => 'changes c1',
        on         => 'c1.id = t.first_change_id',
        inner_join => 'entities e1',
        on         => 'e2.id = c2.identity_id',
        inner_join => 'changes c2',
        on         => 'c2.id = t.last_change_id',
        inner_join => 'entities e2',
        on         => 'e1.id = c1.identity_id',
        inner_join => 'issues i',
        on         => 'i.id = t.id',
        where      => { 't.id' => $info->{id} },
    );

    push( @data, $self->header( '  UUID', $ref->{uuid} ), );
    push( @data,
        $self->header( '  Created',    $self->ctime_ago($ref) ),
        $self->header( '  Created-By', $ref->{creator} ),
    );

    my @refs = $db->xhashrefs(
        with => 'ids',
        as   => sq(

            # self
            select => bv( $info->{id}, DBI::SQL_INTEGER )->as('id'),

            # parent
            union_select => 'i.src_id',
            from         => 'issues i',
            where        => { 'i.id' => $info->{id} },

            # siblings
            union_select => 'i2.id',
            from         => 'issues i',
            inner_join   => 'issues i2',
            on           => 'i2.src_id = i.src_id',
            where        => { 'i.id' => $info->{id} },

            # children
            union_select => 'i.id',
            from         => 'issues i',
            where        => { 'i.src_id' => $info->{id} },
        ),
        select => [
            'i.id AS id',
            'p.path',
            'ist.status',
            'c.mtime AS mtime',
            'c.mtimetz AS mtimetz',
            'c.mtimetzhm AS mtimetzhm',
            "$now - c.mtime AS mtime_age",
            'p.hub_id',
        ],
        from       => 'ids',
        inner_join => 'issues i',
        on         => 'i.id = ids.id',
        inner_join => 'projects p',
        on         => 'p.id = i.project_id',
        left_join  => 'hubs h',
        on         => 'h.id = p.hub_id',
        inner_join => 'issue_status ist',
        on         => 'ist.id = i.issue_status_id',
        inner_join => 'topics t',
        on         => 't.id = i.id',
        inner_join => 'changes c',
        on         => 'c.id = t.first_change_id',
        order_by   => 'p.path',
    );

    my %seen;
    my $count = @refs;
    my $i     = 1;
    foreach my $ref (@refs) {
        my @ago = $self->mtime_ago($ref);

        push(
            @data,
            $self->header(
                '  Status',
                "$ref->{path}/$ref->{id}: $ref->{status} (" . $ago[0] . ')',
                $ago[1]
            ),
        );
    }

    push( @data,
        $self->header( '  Updated',    $self->mtime_ago($ref) ),
        $self->header( '  Updated-By', $ref->{updator} ),
    ) unless $ref->{mtime} == $ref->{ctime};

    $self->start_pager;
    print $self->render_table( 'l  l', [ 'Issue', $ref->{title} ], \@data, 1 );

    print "\n";
    $self->dispatch( 'App::bif::log::issue',
        { opts => { id => $opts->{id} } } );

    $self->ok( 'ShowIssue', \@data );
}

1;
__END__

=head1 NAME

=for bif-doc #show

bif-show-issue - display an issue's current status

=head1 VERSION

0.1.5_2 (2015-06-26)

=head1 SYNOPSIS

    bif show issue ID [OPTIONS...]

=head1 DESCRIPTION

The B<bif-show-issue> command displays the characteristics of an issue.

=head1 ARGUMENTS & OPTIONS

=over

=item ID

An issue ID. Required.

=item --full, -f

Display a more verbose version of the current status.

=item --uuid, -U

Lookup the topic using ID as a UUID string instead of a topic integer.

=back

=head1 SEE ALSO

L<bif>(1)

=head1 AUTHOR

Mark Lawrence E<lt>nomad@null.netE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2014-2015 Mark Lawrence <nomad@null.net>

This program is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 3 of the License, or (at your
option) any later version.

