package Bio::Palantir::Refiner;
# ABSTRACT: front-end class for Bio::Palantir::Refiner module, wich handles the refinement of NRPS/PKS BGC annotations
$Bio::Palantir::Refiner::VERSION = '0.201670';
use Moose;
use namespace::autoclean;

use Data::UUID;

use aliased 'Bio::Palantir::Parser';
use aliased 'Bio::Palantir::Refiner::ClusterPlus';
extends 'Bio::FastParsers::Base';


# ATTRIBUTES



has 'module_delineation' => (
    is      => 'ro',
    isa     => 'Str',
    default => 'substrate-selection',
);


has 'clusters' => (
    traits   => ['Array'],
    is       => 'ro',
    isa      => 'ArrayRef[Bio::Palantir::Refiner::ClusterPlus]',
    init_arg => undef,
    lazy     => 1,
    builder  => '_build_clusters',
    handles  => {
         count_clusters => 'count',
           all_clusters => 'elements',
           get_cluster  => 'get',
          next_cluster  => 'shift',        
    },
);

## no critic (ProhibitUnusedPrivateSubroutines)

sub _build_clusters {
    my $self = shift;
   
    my $report = Parser->new( file => $self->file );
    my $root = $report->root;

    my @cluster_plus;
    push @cluster_plus, ClusterPlus->new(
        _cluster => $_,
        module_delineation => $self->module_delineation
        ) for $root->all_clusters
    ;

    return \@cluster_plus;

}

## use critic


__PACKAGE__->meta->make_immutable;
1;

__END__

=pod

=head1 NAME

Bio::Palantir::Refiner - front-end class for Bio::Palantir::Refiner module, wich handles the refinement of NRPS/PKS BGC annotations

=head1 VERSION

version 0.201670

=head1 DESCRIPTION

This module implements classes and their methods for B<improving the antisMASH
annotation of NRPS/PKS BGCs>. 

The refined B<Biosynthetic Gene Cluster (BGC) information> is hierarchically
organized as follows:

C<ClusterPlus.pm>: contains attributes and methods for the BGC B<Cluster> level,
including an array of GenePlus objects 

C<GenePlus.pm>:    contains attributes and methods for the BGC B<Gene> level,
including an array of DomainPlus objects (if NRPS/PKS BGCs)

C<ModulePlus.pm>:  contains attributes and methods for the BGC B<Module> level
(generated by Palantir), including an array of DomainPlus objects (this class is
parallel to Genes, as module can be overlapping 2 genes)

C<DomainPlus.pm>:  contains attributes and methods for the BGC B<Domain> level

=head1 ATTRIBUTES

=head2 file

Path to biosynML.xml or regions.js antiSMASH report file to be parsed.

=head2 file

Path to a biosynML.xml or regions.js file

=head2 module_delineation

Module delineation method: generates modules from condensation or selection domains.

=head2 clusters

ArrayRef of L<Bio::Palantir::Refiner::ClusterPlus>

=head1 METHODS

=head2 count_clusters

Returns the number of Clusters of the Root.

    # $root is a Bio::Palantir::Refiner::RootPlus
    my $count = $root->count_clusters;

This method does not accept any arguments.

=head2 all_clusters

Returns all the Clusters of the Root (not an array reference).

    # $root is a Bio::Palantir::Refiner::RootPlus
    my @clusters = $root->all_clusters;

This method does not accept any arguments.

=head2 get_cluster

Returns one Cluster of the Root by its index. You can also use
negative index numbers, just as with Perl's core array handling. If the
specified Cluster does not exist, this method will return C<undef>.

    # $root is a Bio::Palantir::Refiner::RootPlus
    my $cluster = $root->get_cluster($index);
    croak "Cluster $index not found!" unless defined $cluster;

This method accepts just one argument (and not an array slice).

=head2 next_cluster

Shifts the first Cluster of the array off and returns it, shortening the
array by 1 and moving everything down. If there are no more Clusters in
the array, returns C<undef>.

    # $root is a Bio::Palantir::Refiner::RootPlus
    while (my $cluster = $root->next_cluster) {
        # process $cluster
        # ...
    }

This method does not accept any arguments.

=head1 AUTHOR

Loic MEUNIER <lmeunier@uliege.be>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2019 by University of Liege / Unit of Eukaryotic Phylogenomics / Loic MEUNIER and Denis BAURAIN.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
