package Quiq::PlotlyJs::Reference;
use base qw/Quiq::Hash/;

use v5.10;
use strict;
use warnings;

our $VERSION = '1.169';

use Quiq::Hash;
use HTML::TreeBuilder ();
use Quiq::Html::List;
use Quiq::Html::Page;

# -----------------------------------------------------------------------------

=encoding utf8

=head1 NAME

Quiq::PlotlyJs::Reference - Erzeuge Plotly.js Reference Manual

=head1 BASE CLASS

L<Quiq::Hash>

=head1 SYNOPSIS

  use Quiq::PlotlyJs::Reference;
  use Quiq::Html::Producer;
  
  $root = Quiq::PlotlyJs::Reference->new;
  $h = Quiq::Html::Producer->new;
  $html = $root->asHtml($h);

=head1 DESCRIPTION

Die L<Dokumentation zu Plotly.js|https://plot.ly/javascript/> ist umfangreich. Ein wichtiger
Teil davon, die L<Full Reference|https://plot.ly/javascript/reference/> mit der Beschreibung aller Attribute
ist allerdings umständlich zu handhaben. Diese Klasse erzeugt eine
übersichtlichere Fassung und enthält diese auch gleichzeitig. Siehe
Abschnitt L<Plotly.js Reference|"Plotly.js Reference">.

=head1 EXAMPLE

=head2 Dokument an der Kommandozeile erzeugen

  $ perl -MQuiq::Html::Producer -MQuiq::PlotlyJs::Reference -E '$h = Quiq::Html::Producer->new; print Quiq::PlotlyJs::Reference->new->asHtml($h)'

produziert auf stdout

  <details>
    <summary>
      0. Layout
    </summary>
    <div style="margin-left: 22px">
      <p>
        <details>
          <summary>
            angularaxis
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>[0, 1]</dd>
            </dl>
            <p>
              Legacy polar charts are deprecated! Please switch to "polar" subplots. Defines the start and end point of this angular axis.
            </p>
          </div>
        </details>
  ... und mehr als 11.000 weitere Zeilen ...

=head1 METHODS

=head2 Konstruktor

=head3 new() - Instantiiere Objekt

=head4 Synopsis

  $root = $class->new;

=head4 Returns

Wurzelknoten des Dokument-Baums (Object)

=head4 Description

Parse das Originaldokument, überführe es in einen Dokumentbaum
und liefere eine Referenz auf den Wurzelknoten dieses Baums zurück.

=cut

# -----------------------------------------------------------------------------

sub new {
    my $class = shift;

    # my $file = 'Blob/doc-content/plotlyjs-reference-orig.html';
    # my $hRoot = HTML::TreeBuilder->new_from_file($file)->elementify;

    my $url = 'https://plot.ly/javascript/reference/';
    my $hRoot = HTML::TreeBuilder->new_from_url($url)->elementify;

    # Parse die Attributliste eines Knotens

    my $attributesSub; # muss wg. rekursivem Aufruf vorher deklariert werden
    $attributesSub = sub {
        my $h = shift;

        my @attributes;

        my $ul = $h->look_down(_tag=>'ul');
        if ($ul) {
            for my $li ($ul->content_list) {
                # Name

                my $name = $li->look_down(class=>'attribute-name')->as_text;
                $name =~ s/^\s+//;
                $name =~ s/\s+$//;

                my $html = $li->as_HTML;

                # Parent

                my ($parent) = $html =~ m|Parent:.*?<code>(.*?)</code>|;
                if (!defined $parent) {
                    # Angabe Parent: erwarten wir immer
                    $class->throw;
                }

                # Type

                my $type;
                if ($html =~ /Type:/) {
                    ($type) = $html =~ m{Type:</em>(.*?)(<br|<p>|<ul>|$)}s;
                    if (!defined $type) {
                        # Wenn Angabe Type: vorkommt, müssen wir sie
                        # extrahieren können
                        $class->throw(
                             'PLOTYJS-00001: Can\'t extract Type:',
                             Html => $html,
                        );
                    }
                    $type =~ s/^\s+//;
                    $type =~ s/\s+$//;
                    $type =~ s|</?code>||g;
                    $type =~ s|&quot;|"|g;
                }

                # Default

                my $default;
                if ($html =~ /Default:/) {
                    ($default) = $html =~ m|Default:.*?<code>(.*?)</code>|;
                    if (!defined $default) {
                        # Wenn Angabe Default: vorkommt, müssen wir sie
                        # extrahieren können
                        $class->throw(
                             'PLOTYJS-00001: Can\'t extract Default:',
                             Html => $html,
                        );
                    }
                    $default =~ s|</?code>||g;
                    $default =~ s|&quot;|"|g;
                }

                # Description

                my $descr;
                my $p = $li->look_down(_tag=>'p');
                if ($p) {
                    $descr = $p->as_text;
                    $descr =~ s|M~|\\M~|g;
                }

                push @attributes,Quiq::Hash->new(
                    name => $name,
                    parent => $parent,
                    type => $type,
                    default => $default,
                    description => $descr,
                    attributeA => $attributesSub->($li),
                );
            }
        }

        return \@attributes;
    };

    # Parse Dokument

    my $i = 0;
    my @sections;
    for my $hSec ($hRoot->look_down(_tag=>'div',class=>'row')) {
        if (!$i++) {
            # Die Einleitung des Reference-Dokuments übergehen wir
            next;
        }

        my $title = ucfirst $hSec->look_down(_tag=>'h4')->as_text;
        $title =~ s/^\s+//;
        $title =~ s/\s+$//;

        my $descr;
        my $e = $hSec->look_down(_tag=>'div',class=>'description');
        if ($e) {
            $descr = $e->as_text;
        }

        push @sections,Quiq::Hash->new(
            title => $title,
            description => $descr,
            attributeA => $attributesSub->($hSec),
        );
    }

    # Abschnitt Layout an den Anfang
    unshift @sections,pop @sections;

    return $class->SUPER::new(
        title => 'Plotly.js Reference',
        sectionA => \@sections,
    );
}

# -----------------------------------------------------------------------------

=head2 HTML-Repräsentation

=head3 asHtml() - Erzeuge HTML-Repräsentation

=head4 Synopsis

  $html = $obj->asHtml($h);

=head4 Arguments

=over 4

=item $h

Quiq::Html::Tag Objekt.

=back

=head4 Options

=over 4

=item -document => $bool (Default: 0)

Erzeuge ein vollständiges HTML-Dokument.

=item -indent => $n (Default: 22)

Rücke die Ebenen um $n Pixel ein.

=back

=head4 Returns

HTML-Code (String)

=head4 Description

Liefere die plotly.js Dokumentation in HTML.

=cut

# -----------------------------------------------------------------------------

sub asHtml {
    my ($self,$h) = splice @_,0,2;

    # Optionen

    my $document = 0;
    my $indent = 22;

    $self->parameters(\@_,
        document => \$document,
        indent => \$indent,
    );

    # Generiere Attributliste 

    my $attributesAsHtmlSub; # muss wg. rekursivem Aufruf deklariert werden
    $attributesAsHtmlSub = sub {
        my ($node,$h,$indent) = @_;

        my $html = '';

        my @attributes = sort {$a->get('name') cmp $b->get('name')}
            @{$node->get('attributeA')};

        for my $att (@attributes) {
            $html .= $h->tag('details',
                '-',
                $h->tag('summary',
                    $att->get('name')
                ),
                $h->tag('div',
                    style => "margin-left: ${indent}px",
                    '-',
                    Quiq::Html::List->html($h,
                        type => 'description',
                        isText => 1,
                        items => [
                            'Parent:' => $att->get('parent'),
                            'Type:' => $att->get('type'),
                            'Default:' => $att->get('default'),
                        ],
                    ),
                    $h->tag('p',
                        -text => 1,
                        $att->get('description')
                    ),
                ),
            );
            $html .= $h->tag('div',
                -ignoreIfNull => 1,
                style => "margin-left: ${indent}px",
                $attributesAsHtmlSub->($att,$h,$indent)
            );
        }

        return $html;
    };

    # Generiere Dokument

    my $html = '';
    my $i = 0;
    for my $sec (@{$self->get('sectionA')}) {
        $html .= $h->tag('details',
            '-',
            $h->tag('summary',
                "$i. ".$sec->get('title')
            ),
            $h->tag('div',
                style => "margin-left: ${indent}px",
                '-',
                $h->tag('p',
                    -text => 1,
                    $sec->get('description')
                ),
                $h->tag('p',
                    $attributesAsHtmlSub->($sec,$h,$indent)
                ),
            ),
        );
        $i++;
    }

    if ($document) {
        my $title = $self->get('title');
        $html = Quiq::Html::Page->html($h,
            title => $title,
            styleSheet => qq~
                body {
                    font-family: sans-serif;
                    font-size: 11pt;
                }
            ~,
            body => $h->cat(
                $h->tag('h1',$title),
                $html,
            ),
        );
    }

    return $html;
}

# -----------------------------------------------------------------------------

=head1 DETAILS

=head2 Plotly.js Reference

(Der Inhalt dieses Abschnitts erscheint nur in HTML. Original
siehe L<Full Reference|https://plot.ly/javascript/reference/>)

=begin html

<details>
  <summary>
    0. Layout
  </summary>
  <div style="margin-left: 22px">
    <p>
      <details>
        <summary>
          angularaxis
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>object containing one or more of the keys listed below.</dd>
            <dt>Default:</dt>
            <dd>[0, 1]</dd>
          </dl>
          <p>
            Legacy polar charts are deprecated! Please switch to "polar" subplots. Defines the start and end point of this angular axis.
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            domain
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.angularaxis</dd>
              <dt>Type:</dt>
              <dd>array</dd>
              <dt>Default:</dt>
              <dd>[0, 1]</dd>
            </dl>
            <p>
              Polar chart subplots are not supported yet. This key has currently no effect.
            </p>
          </div>
        </details>
        <details>
          <summary>
            endpadding
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.angularaxis</dd>
              <dt>Type:</dt>
              <dd>number</dd>
            </dl>
            <p>
              Legacy polar charts are deprecated! Please switch to "polar" subplots.
            </p>
          </div>
        </details>
        <details>
          <summary>
            range
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.angularaxis</dd>
              <dt>Type:</dt>
              <dd>array</dd>
            </dl>
            <p>
              Legacy polar charts are deprecated! Please switch to "polar" subplots. Defines the start and end point of this angular axis.
            </p>
          </div>
        </details>
        <details>
          <summary>
            showline
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.angularaxis</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Legacy polar charts are deprecated! Please switch to "polar" subplots. Determines whether or not the line bounding this angular axis will be shown on the figure.
            </p>
          </div>
        </details>
        <details>
          <summary>
            showticklabels
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.angularaxis</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Legacy polar charts are deprecated! Please switch to "polar" subplots. Determines whether or not the angular axis ticks will feature tick labels.
            </p>
          </div>
        </details>
        <details>
          <summary>
            tickcolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.angularaxis</dd>
              <dt>Type:</dt>
              <dd>color</dd>
            </dl>
            <p>
              Legacy polar charts are deprecated! Please switch to "polar" subplots. Sets the color of the tick lines on this angular axis.
            </p>
          </div>
        </details>
        <details>
          <summary>
            ticklen
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.angularaxis</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
            </dl>
            <p>
              Legacy polar charts are deprecated! Please switch to "polar" subplots. Sets the length of the tick lines on this angular axis.
            </p>
          </div>
        </details>
        <details>
          <summary>
            tickorientation
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.angularaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "horizontal" | "vertical" )</dd>
            </dl>
            <p>
              Legacy polar charts are deprecated! Please switch to "polar" subplots. Sets the orientation (from the paper perspective) of the angular axis tick labels.
            </p>
          </div>
        </details>
        <details>
          <summary>
            ticksuffix
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.angularaxis</dd>
              <dt>Type:</dt>
              <dd>string</dd>
            </dl>
            <p>
              Legacy polar charts are deprecated! Please switch to "polar" subplots. Sets the length of the tick lines on this angular axis.
            </p>
          </div>
        </details>
        <details>
          <summary>
            visible
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.angularaxis</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Legacy polar charts are deprecated! Please switch to "polar" subplots. Determines whether or not this axis will be visible.
            </p>
          </div>
        </details>
      </div>
      <details>
        <summary>
          annotations
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>array of object where each object has one or more of the keys listed below.</dd>
            <dt>Default:</dt>
            <dd>true</dd>
          </dl>
          <p>
            Determines whether or not this annotation is visible.
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            align
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "left" | "center" | "right" )</dd>
              <dt>Default:</dt>
              <dd>"center"</dd>
            </dl>
            <p>
              Sets the horizontal alignment of the `text` within the box. Has an effect only if `text` spans more two or more lines (i.e. `text` contains one or more &lt;br&gt; HTML tags) or if an explicit width is set to override the text width.
            </p>
          </div>
        </details>
        <details>
          <summary>
            arrowcolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>color</dd>
            </dl>
            <p>
              Sets the color of the annotation arrow.
            </p>
          </div>
        </details>
        <details>
          <summary>
            arrowhead
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>integer between or equal to 0 and 8</dd>
              <dt>Default:</dt>
              <dd>1</dd>
            </dl>
            <p>
              Sets the end annotation arrow head style.
            </p>
          </div>
        </details>
        <details>
          <summary>
            arrowside
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>flaglist string. Any combination of "end", "start" joined with a "+" OR "none".</dd>
              <dt>Default:</dt>
              <dd>"end"</dd>
            </dl>
            <p>
              Sets the annotation arrow head position.
            </p>
          </div>
        </details>
        <details>
          <summary>
            arrowsize
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0.3</dd>
              <dt>Default:</dt>
              <dd>1</dd>
            </dl>
            <p>
              Sets the size of the end annotation arrow head, relative to `arrowwidth`. A value of 1 (default) gives a head about 3x as wide as the line.
            </p>
          </div>
        </details>
        <details>
          <summary>
            arrowwidth
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0.1</dd>
            </dl>
            <p>
              Sets the width (in px) of annotation arrow line.
            </p>
          </div>
        </details>
        <details>
          <summary>
            ax
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>number or categorical coordinate string</dd>
            </dl>
            <p>
              Sets the x component of the arrow tail about the arrow head. If `axref` is `pixel`, a positive (negative) component corresponds to an arrow pointing from right to left (left to right). If `axref` is an axis, this is an absolute value on that axis, like `x`, NOT a relative value.
            </p>
          </div>
        </details>
        <details>
          <summary>
            axref
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "pixel" | "/^x([2-9]|[1-9][0-9]+)?$/" )</dd>
              <dt>Default:</dt>
              <dd>"pixel"</dd>
            </dl>
            <p>
              Indicates in what terms the tail of the annotation (ax,ay) is specified. If `pixel`, `ax` is a relative offset in pixels from `x`. If set to an x axis id (e.g. "x" or "x2"), `ax` is specified in the same terms as that axis. This is useful for trendline annotations which should continue to indicate the correct trend when zoomed.
            </p>
          </div>
        </details>
        <details>
          <summary>
            ay
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>number or categorical coordinate string</dd>
            </dl>
            <p>
              Sets the y component of the arrow tail about the arrow head. If `ayref` is `pixel`, a positive (negative) component corresponds to an arrow pointing from bottom to top (top to bottom). If `ayref` is an axis, this is an absolute value on that axis, like `y`, NOT a relative value.
            </p>
          </div>
        </details>
        <details>
          <summary>
            ayref
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "pixel" | "/^y([2-9]|[1-9][0-9]+)?$/" )</dd>
              <dt>Default:</dt>
              <dd>"pixel"</dd>
            </dl>
            <p>
              Indicates in what terms the tail of the annotation (ax,ay) is specified. If `pixel`, `ay` is a relative offset in pixels from `y`. If set to a y axis id (e.g. "y" or "y2"), `ay` is specified in the same terms as that axis. This is useful for trendline annotations which should continue to indicate the correct trend when zoomed.
            </p>
          </div>
        </details>
        <details>
          <summary>
            bgcolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>color</dd>
              <dt>Default:</dt>
              <dd>"rgba(0,0,0,0)"</dd>
            </dl>
            <p>
              Sets the background color of the annotation.
            </p>
          </div>
        </details>
        <details>
          <summary>
            bordercolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>color</dd>
              <dt>Default:</dt>
              <dd>"rgba(0,0,0,0)"</dd>
            </dl>
            <p>
              Sets the color of the border enclosing the annotation `text`.
            </p>
          </div>
        </details>
        <details>
          <summary>
            borderpad
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>1</dd>
            </dl>
            <p>
              Sets the padding (in px) between the `text` and the enclosing border.
            </p>
          </div>
        </details>
        <details>
          <summary>
            borderwidth
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>1</dd>
            </dl>
            <p>
              Sets the width (in px) of the border enclosing the annotation `text`.
            </p>
          </div>
        </details>
        <details>
          <summary>
            captureevents
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Determines whether the annotation text box captures mouse move and click events, or allows those events to pass through to data points in the plot that may be behind the annotation. By default `captureevents` is "false" unless `hovertext` is provided. If you use the event `plotly_clickannotation` without `hovertext` you must explicitly enable `captureevents`.
            </p>
          </div>
        </details>
        <details>
          <summary>
            clicktoshow
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( false | "onoff" | "onout" )</dd>
            </dl>
            <p>
              Makes this annotation respond to clicks on the plot. If you click a data point that exactly matches the `x` and `y` values of this annotation, and it is hidden (visible: false), it will appear. In "onoff" mode, you must click the same point again to make it disappear, so if you click multiple points, you can show multiple annotations. In "onout" mode, a click anywhere else in the plot (on another data point or not) will hide this annotation. If you need to show/hide this annotation in response to different `x` or `y` values, you can set `xclick` and/or `yclick`. This is useful for example to label the side of a bar. To label markers though, `standoff` is preferred over `xclick` and `yclick`.
            </p>
          </div>
        </details>
        <details>
          <summary>
            font
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
            </dl>
            <p>
              Sets the annotation text font.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              color
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.annotations[].font</dd>
                <dt>Type:</dt>
                <dd>color</dd>
              </dl>
            </div>
          </details>
          <details>
            <summary>
              family
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.annotations[].font</dd>
                <dt>Type:</dt>
                <dd>string</dd>
              </dl>
              <p>
                HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
              </p>
            </div>
          </details>
          <details>
            <summary>
              size
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.annotations[].font</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 1</dd>
              </dl>
            </div>
          </details>
        </div>
        <details>
          <summary>
            height
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 1</dd>
            </dl>
            <p>
              Sets an explicit height for the text box. null (default) lets the text set the box height. Taller text will be clipped.
            </p>
          </div>
        </details>
        <details>
          <summary>
            hoverlabel
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
            </dl>
            <p>
              Sets the background color of the hover label. By default uses the annotation's `bgcolor` made opaque, or white if it was transparent.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              bgcolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.annotations[].hoverlabel</dd>
                <dt>Type:</dt>
                <dd>color</dd>
              </dl>
              <p>
                Sets the background color of the hover label. By default uses the annotation's `bgcolor` made opaque, or white if it was transparent.
              </p>
            </div>
          </details>
          <details>
            <summary>
              bordercolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.annotations[].hoverlabel</dd>
                <dt>Type:</dt>
                <dd>color</dd>
              </dl>
              <p>
                Sets the border color of the hover label. By default uses either dark grey or white, for maximum contrast with `hoverlabel.bgcolor`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              font
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.annotations[].hoverlabel</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
              </dl>
              <p>
                Sets the hover label text font. By default uses the global hover font and size, with color from `hoverlabel.bordercolor`.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                color
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.annotations[].hoverlabel.font</dd>
                  <dt>Type:</dt>
                  <dd>color</dd>
                </dl>
              </div>
            </details>
            <details>
              <summary>
                family
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.annotations[].hoverlabel.font</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
                </p>
              </div>
            </details>
            <details>
              <summary>
                size
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.annotations[].hoverlabel.font</dd>
                  <dt>Type:</dt>
                  <dd>number greater than or equal to 1</dd>
                </dl>
              </div>
            </details>
          </div>
        </div>
        <details>
          <summary>
            hovertext
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>string</dd>
            </dl>
            <p>
              Sets text to appear when hovering over this annotation. If omitted or blank, no hover label will appear.
            </p>
          </div>
        </details>
        <details>
          <summary>
            name
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>string</dd>
            </dl>
            <p>
              When used in a template, named items are created in the output figure in addition to any items the figure already has in this array. You can modify these items in the output figure by making your own item with `templateitemname` matching this `name` alongside your modifications (including `visible: false` or `enabled: false` to hide it). Has no effect outside of a template.
            </p>
          </div>
        </details>
        <details>
          <summary>
            opacity
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>number between or equal to 0 and 1</dd>
              <dt>Default:</dt>
              <dd>1</dd>
            </dl>
            <p>
              Sets the opacity of the annotation (text + arrow).
            </p>
          </div>
        </details>
        <details>
          <summary>
            showarrow
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
              <dt>Default:</dt>
              <dd>true</dd>
            </dl>
            <p>
              Determines whether or not the annotation is drawn with an arrow. If "true", `text` is placed near the arrow's tail. If "false", `text` lines up with the `x` and `y` provided.
            </p>
          </div>
        </details>
        <details>
          <summary>
            standoff
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
            <p>
              Sets a distance, in pixels, to move the end arrowhead away from the position it is pointing at, for example to point at the edge of a marker independent of zoom. Note that this shortens the arrow from the `ax` / `ay` vector, in contrast to `xshift` / `yshift` which moves everything by this amount.
            </p>
          </div>
        </details>
        <details>
          <summary>
            startarrowhead
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>integer between or equal to 0 and 8</dd>
              <dt>Default:</dt>
              <dd>1</dd>
            </dl>
            <p>
              Sets the start annotation arrow head style.
            </p>
          </div>
        </details>
        <details>
          <summary>
            startarrowsize
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0.3</dd>
              <dt>Default:</dt>
              <dd>1</dd>
            </dl>
            <p>
              Sets the size of the start annotation arrow head, relative to `arrowwidth`. A value of 1 (default) gives a head about 3x as wide as the line.
            </p>
          </div>
        </details>
        <details>
          <summary>
            startstandoff
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
            <p>
              Sets a distance, in pixels, to move the start arrowhead away from the position it is pointing at, for example to point at the edge of a marker independent of zoom. Note that this shortens the arrow from the `ax` / `ay` vector, in contrast to `xshift` / `yshift` which moves everything by this amount.
            </p>
          </div>
        </details>
        <details>
          <summary>
            templateitemname
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>string</dd>
            </dl>
            <p>
              Used to refer to a named item in this array in the template. Named items from the template will be created even without a matching item in the input figure, but you can modify one by making an item with `templateitemname` matching its `name`, alongside your modifications (including `visible: false` or `enabled: false` to hide it). If there is no template or no matching item, this item will be hidden unless you explicitly show it with `visible: true`.
            </p>
          </div>
        </details>
        <details>
          <summary>
            text
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>string</dd>
            </dl>
            <p>
              Sets the text associated with this annotation. Plotly uses a subset of HTML tags to do things like newline (&lt;br&gt;), bold (&lt;b&gt;&lt;/b&gt;), italics (&lt;i&gt;&lt;/i&gt;), hyperlinks (&lt;a href='...'&gt;&lt;/a&gt;). Tags &lt;em&gt;, &lt;sup&gt;, &lt;sub&gt; &lt;span&gt; are also supported.
            </p>
          </div>
        </details>
        <details>
          <summary>
            textangle
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>angle</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
            <p>
              Sets the angle at which the `text` is drawn with respect to the horizontal.
            </p>
          </div>
        </details>
        <details>
          <summary>
            valign
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "top" | "middle" | "bottom" )</dd>
              <dt>Default:</dt>
              <dd>"middle"</dd>
            </dl>
            <p>
              Sets the vertical alignment of the `text` within the box. Has an effect only if an explicit height is set to override the text height.
            </p>
          </div>
        </details>
        <details>
          <summary>
            visible
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
              <dt>Default:</dt>
              <dd>true</dd>
            </dl>
            <p>
              Determines whether or not this annotation is visible.
            </p>
          </div>
        </details>
        <details>
          <summary>
            width
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 1</dd>
            </dl>
            <p>
              Sets an explicit width for the text box. null (default) lets the text set the box width. Wider text will be clipped. There is no automatic wrapping; use &lt;br&gt; to start a new line.
            </p>
          </div>
        </details>
        <details>
          <summary>
            x
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>number or categorical coordinate string</dd>
            </dl>
            <p>
              Sets the annotation's x position. If the axis `type` is "log", then you must take the log of your desired range. If the axis `type` is "date", it should be date strings, like date data, though Date objects and unix milliseconds will be accepted and converted to strings. If the axis `type` is "category", it should be numbers, using the scale where each category is assigned a serial number from zero in the order it appears.
            </p>
          </div>
        </details>
        <details>
          <summary>
            xanchor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "auto" | "left" | "center" | "right" )</dd>
              <dt>Default:</dt>
              <dd>"auto"</dd>
            </dl>
            <p>
              Sets the text box's horizontal position anchor This anchor binds the `x` position to the "left", "center" or "right" of the annotation. For example, if `x` is set to 1, `xref` to "paper" and `xanchor` to "right" then the right-most portion of the annotation lines up with the right-most edge of the plotting area. If "auto", the anchor is equivalent to "center" for data-referenced annotations or if there is an arrow, whereas for paper-referenced with no arrow, the anchor picked corresponds to the closest side.
            </p>
          </div>
        </details>
        <details>
          <summary>
            xclick
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>number or categorical coordinate string</dd>
            </dl>
            <p>
              Toggle this annotation when clicking a data point whose `x` value is `xclick` rather than the annotation's `x` value.
            </p>
          </div>
        </details>
        <details>
          <summary>
            xref
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "paper" | "/^x([2-9]|[1-9][0-9]+)?$/" )</dd>
            </dl>
            <p>
              Sets the annotation's x coordinate axis. If set to an x axis id (e.g. "x" or "x2"), the `x` position refers to an x coordinate If set to "paper", the `x` position refers to the distance from the left side of the plotting area in normalized coordinates where 0 (1) corresponds to the left (right) side.
            </p>
          </div>
        </details>
        <details>
          <summary>
            xshift
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>number</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
            <p>
              Shifts the position of the whole annotation and arrow to the right (positive) or left (negative) by this many pixels.
            </p>
          </div>
        </details>
        <details>
          <summary>
            y
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>number or categorical coordinate string</dd>
            </dl>
            <p>
              Sets the annotation's y position. If the axis `type` is "log", then you must take the log of your desired range. If the axis `type` is "date", it should be date strings, like date data, though Date objects and unix milliseconds will be accepted and converted to strings. If the axis `type` is "category", it should be numbers, using the scale where each category is assigned a serial number from zero in the order it appears.
            </p>
          </div>
        </details>
        <details>
          <summary>
            yanchor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "auto" | "top" | "middle" | "bottom" )</dd>
              <dt>Default:</dt>
              <dd>"auto"</dd>
            </dl>
            <p>
              Sets the text box's vertical position anchor This anchor binds the `y` position to the "top", "middle" or "bottom" of the annotation. For example, if `y` is set to 1, `yref` to "paper" and `yanchor` to "top" then the top-most portion of the annotation lines up with the top-most edge of the plotting area. If "auto", the anchor is equivalent to "middle" for data-referenced annotations or if there is an arrow, whereas for paper-referenced with no arrow, the anchor picked corresponds to the closest side.
            </p>
          </div>
        </details>
        <details>
          <summary>
            yclick
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>number or categorical coordinate string</dd>
            </dl>
            <p>
              Toggle this annotation when clicking a data point whose `y` value is `yclick` rather than the annotation's `y` value.
            </p>
          </div>
        </details>
        <details>
          <summary>
            yref
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "paper" | "/^y([2-9]|[1-9][0-9]+)?$/" )</dd>
            </dl>
            <p>
              Sets the annotation's y coordinate axis. If set to an y axis id (e.g. "y" or "y2"), the `y` position refers to an y coordinate If set to "paper", the `y` position refers to the distance from the bottom of the plotting area in normalized coordinates where 0 (1) corresponds to the bottom (top).
            </p>
          </div>
        </details>
        <details>
          <summary>
            yshift
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.annotations[]</dd>
              <dt>Type:</dt>
              <dd>number</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
            <p>
              Shifts the position of the whole annotation and arrow up (positive) or down (negative) by this many pixels.
            </p>
          </div>
        </details>
      </div>
      <details>
        <summary>
          autosize
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>boolean</dd>
          </dl>
          <p>
            Determines whether or not a layout width or height that has been left undefined by the user is initialized on each relayout. Note that, regardless of this attribute, an undefined layout width or height is always initialized on the first call to plot.
          </p>
        </div>
      </details>
      <details>
        <summary>
          calendar
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>enumerated , one of ( "gregorian" | "chinese" | "coptic" | "discworld" | "ethiopian" | "hebrew" | "islamic" | "julian" | "mayan" | "nanakshahi" | "nepali" | "persian" | "jalali" | "taiwan" | "thai" | "ummalqura" )</dd>
            <dt>Default:</dt>
            <dd>"gregorian"</dd>
          </dl>
          <p>
            Sets the default calendar system to use for interpreting and displaying dates throughout the plot.
          </p>
        </div>
      </details>
      <details>
        <summary>
          clickmode
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>flaglist string. Any combination of "event", "select" joined with a "+" OR "none".</dd>
            <dt>Default:</dt>
            <dd>"event"</dd>
          </dl>
          <p>
            Determines the mode of single click interactions. "event" is the default value and emits the `plotly_click` event. In addition this mode emits the `plotly_selected` event in drag modes "lasso" and "select", but with no event data attached (kept for compatibility reasons). The "select" flag enables selecting single data points via click. This mode also supports persistent selections, meaning that pressing Shift while clicking, adds to / subtracts from an existing selection. "select" with `hovermode`: "x" can be confusing, consider explicitly setting `hovermode`: "closest" when using this feature. Selection events are sent accordingly as long as "event" flag is set as well. When the "event" flag is missing, `plotly_click` and `plotly_selected` events are not fired.
          </p>
        </div>
      </details>
      <details>
        <summary>
          coloraxis
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>object containing one or more of the keys listed below.</dd>
            <dt>Default:</dt>
            <dd>true</dd>
          </dl>
          <p>
            Determines whether or not the color domain is computed with respect to the input data (here corresponding trace color array(s)) or the bounds set in `cmin` and `cmax` Defaults to `false` when `cmin` and `cmax` are set by the user.
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            autocolorscale
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.coloraxis</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
              <dt>Default:</dt>
              <dd>true</dd>
            </dl>
            <p>
              Determines whether the colorscale is a default palette (`autocolorscale: true`) or the palette determined by `colorscale`. In case `colorscale` is unspecified or `autocolorscale` is true, the default palette will be chosen according to whether numbers in the `color` array are all positive, all negative or mixed.
            </p>
          </div>
        </details>
        <details>
          <summary>
            cauto
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.coloraxis</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
              <dt>Default:</dt>
              <dd>true</dd>
            </dl>
            <p>
              Determines whether or not the color domain is computed with respect to the input data (here corresponding trace color array(s)) or the bounds set in `cmin` and `cmax` Defaults to `false` when `cmin` and `cmax` are set by the user.
            </p>
          </div>
        </details>
        <details>
          <summary>
            cmax
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.coloraxis</dd>
              <dt>Type:</dt>
              <dd>number</dd>
            </dl>
            <p>
              Sets the upper bound of the color domain. Value should have the same units as corresponding trace color array(s) and if set, `cmin` must be set as well.
            </p>
          </div>
        </details>
        <details>
          <summary>
            cmid
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.coloraxis</dd>
              <dt>Type:</dt>
              <dd>number</dd>
            </dl>
            <p>
              Sets the mid-point of the color domain by scaling `cmin` and/or `cmax` to be equidistant to this point. Value should have the same units as corresponding trace color array(s). Has no effect when `cauto` is `false`.
            </p>
          </div>
        </details>
        <details>
          <summary>
            cmin
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.coloraxis</dd>
              <dt>Type:</dt>
              <dd>number</dd>
            </dl>
            <p>
              Sets the lower bound of the color domain. Value should have the same units as corresponding trace color array(s) and if set, `cmax` must be set as well.
            </p>
          </div>
        </details>
        <details>
          <summary>
            colorbar
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.coloraxis</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>"pixels"</dd>
            </dl>
            <p>
              Determines whether this color bar's thickness (i.e. the measure in the constant color direction) is set in units of plot "fraction" or in "pixels". Use `thickness` to set the value.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              bgcolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"rgba(0,0,0,0)"</dd>
              </dl>
              <p>
                Sets the color of padded area.
              </p>
            </div>
          </details>
          <details>
            <summary>
              bordercolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets the axis line color.
              </p>
            </div>
          </details>
          <details>
            <summary>
              borderwidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                Sets the width (in px) or the border enclosing this color bar.
              </p>
            </div>
          </details>
          <details>
            <summary>
              dtick
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>number or categorical coordinate string</dd>
              </dl>
              <p>
                Sets the step in-between ticks on this axis. Use with `tick0`. Must be a positive number, or special strings available to "log" and "date" axes. If the axis `type` is "log", then ticks are set every 10^(n"dtick) where n is the tick number. For example, to set a tick mark at 1, 10, 100, 1000, ... set dtick to 1. To set tick marks at 1, 100, 10000, ... set dtick to 2. To set tick marks at 1, 5, 25, 125, 625, 3125, ... set dtick to log_10(5), or 0.69897000433. "log" has several special values; "L&lt;f&gt;", where `f` is a positive number, gives ticks linearly spaced in value (but not position). For example `tick0` = 0.1, `dtick` = "L0.5" will put ticks at 0.1, 0.6, 1.1, 1.6 etc. To show powers of 10 plus small digits between, use "D1" (all digits) or "D2" (only 2 and 5). `tick0` is ignored for "D1" and "D2". If the axis `type` is "date", then you must convert the time to milliseconds. For example, to set the interval between ticks to one day, set `dtick` to 86400000.0. "date" also has special values "M&lt;n&gt;" gives ticks spaced by a number of months. `n` must be a positive integer. To set ticks on the 15th of every third month, set `tick0` to "2000-01-15" and `dtick` to "M3". To set ticks every 4 years, set `dtick` to "M48"
              </p>
            </div>
          </details>
          <details>
            <summary>
              exponentformat
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "none" | "e" | "E" | "power" | "SI" | "B" )</dd>
                <dt>Default:</dt>
                <dd>"B"</dd>
              </dl>
              <p>
                Determines a formatting rule for the tick exponents. For example, consider the number 1,000,000,000. If "none", it appears as 1,000,000,000. If "e", 1e+9. If "E", 1E+9. If "power", 1x10^9 (with 9 in a super script). If "SI", 1G. If "B", 1B.
              </p>
            </div>
          </details>
          <details>
            <summary>
              len
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the length of the color bar This measure excludes the padding of both ends. That is, the color bar length is this length minus the padding on both ends.
              </p>
            </div>
          </details>
          <details>
            <summary>
              lenmode
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "fraction" | "pixels" )</dd>
                <dt>Default:</dt>
                <dd>"fraction"</dd>
              </dl>
              <p>
                Determines whether this color bar's length (i.e. the measure in the color variation direction) is set in units of plot "fraction" or in "pixels. Use `len` to set the value.
              </p>
            </div>
          </details>
          <details>
            <summary>
              nticks
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>integer greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                Specifies the maximum number of ticks for the particular axis. The actual number of ticks will be chosen automatically to be less than or equal to `nticks`. Has an effect only if `tickmode` is set to "auto".
              </p>
            </div>
          </details>
          <details>
            <summary>
              outlinecolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets the axis line color.
              </p>
            </div>
          </details>
          <details>
            <summary>
              outlinewidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the width (in px) of the axis line.
              </p>
            </div>
          </details>
          <details>
            <summary>
              separatethousands
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
              </dl>
              <p>
                If "true", even 4-digit integers are separated
              </p>
            </div>
          </details>
          <details>
            <summary>
              showexponent
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
                <dt>Default:</dt>
                <dd>"all"</dd>
              </dl>
              <p>
                If "all", all exponents are shown besides their significands. If "first", only the exponent of the first tick is shown. If "last", only the exponent of the last tick is shown. If "none", no exponents appear.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showticklabels
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not the tick labels are drawn.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showtickprefix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
                <dt>Default:</dt>
                <dd>"all"</dd>
              </dl>
              <p>
                If "all", all tick labels are displayed with a prefix. If "first", only the first tick is displayed with a prefix. If "last", only the last tick is displayed with a suffix. If "none", tick prefixes are hidden.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showticksuffix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
                <dt>Default:</dt>
                <dd>"all"</dd>
              </dl>
              <p>
                Same as `showtickprefix` but for tick suffixes.
              </p>
            </div>
          </details>
          <details>
            <summary>
              thickness
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>30</dd>
              </dl>
              <p>
                Sets the thickness of the color bar This measure excludes the size of the padding, ticks and labels.
              </p>
            </div>
          </details>
          <details>
            <summary>
              thicknessmode
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "fraction" | "pixels" )</dd>
                <dt>Default:</dt>
                <dd>"pixels"</dd>
              </dl>
              <p>
                Determines whether this color bar's thickness (i.e. the measure in the constant color direction) is set in units of plot "fraction" or in "pixels". Use `thickness` to set the value.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tick0
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>number or categorical coordinate string</dd>
              </dl>
              <p>
                Sets the placement of the first tick on this axis. Use with `dtick`. If the axis `type` is "log", then you must take the log of your starting tick (e.g. to set the starting tick to 100, set the `tick0` to 2) except when `dtick`="L&lt;f&gt;" (see `dtick` for more info). If the axis `type` is "date", it should be a date string, like date data. If the axis `type` is "category", it should be a number, using the scale where each category is assigned a serial number from zero in the order it appears.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickangle
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>angle</dd>
                <dt>Default:</dt>
                <dd>"auto"</dd>
              </dl>
              <p>
                Sets the angle of the tick labels with respect to the horizontal. For example, a `tickangle` of -90 draws the tick labels vertically.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickcolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets the tick color.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickfont
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
              </dl>
              <p>
                Sets the color bar's tick label font
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                color
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.coloraxis.colorbar.tickfont</dd>
                  <dt>Type:</dt>
                  <dd>color</dd>
                </dl>
              </div>
            </details>
            <details>
              <summary>
                family
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.coloraxis.colorbar.tickfont</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
                </p>
              </div>
            </details>
            <details>
              <summary>
                size
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.coloraxis.colorbar.tickfont</dd>
                  <dt>Type:</dt>
                  <dd>number greater than or equal to 1</dd>
                </dl>
              </div>
            </details>
          </div>
          <details>
            <summary>
              tickformat
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets the tick label formatting rule using d3 formatting mini-languages which are very similar to those in Python. For numbers, see: https://github.com/d3/d3-3.x-api-reference/blob/master/Formatting.md#d3_format And for dates see: https://github.com/d3/d3-3.x-api-reference/blob/master/Time-Formatting.md#format We add one item to d3's date formatter: "%{n}f" for fractional seconds with n digits. For example, "2016-10-13 09:15:23.456" with tickformat "%H~%\M~%S.%2f" would display "09~15~23.46"
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickformatstops
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>array of object where each object has one or more of the keys listed below.</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not this stop is used. If `false`, this stop is ignored even within its `dtickrange`.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                dtickrange
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.coloraxis.colorbar.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>array</dd>
                </dl>
                <p>
                  range ["min", "max"], where "min", "max" - dtick values which describe some zoom level, it is possible to omit "min" or "max" value by passing "null"
                </p>
              </div>
            </details>
            <details>
              <summary>
                enabled
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.coloraxis.colorbar.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>boolean</dd>
                  <dt>Default:</dt>
                  <dd>true</dd>
                </dl>
                <p>
                  Determines whether or not this stop is used. If `false`, this stop is ignored even within its `dtickrange`.
                </p>
              </div>
            </details>
            <details>
              <summary>
                name
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.coloraxis.colorbar.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  When used in a template, named items are created in the output figure in addition to any items the figure already has in this array. You can modify these items in the output figure by making your own item with `templateitemname` matching this `name` alongside your modifications (including `visible: false` or `enabled: false` to hide it). Has no effect outside of a template.
                </p>
              </div>
            </details>
            <details>
              <summary>
                templateitemname
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.coloraxis.colorbar.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  Used to refer to a named item in this array in the template. Named items from the template will be created even without a matching item in the input figure, but you can modify one by making an item with `templateitemname` matching its `name`, alongside your modifications (including `visible: false` or `enabled: false` to hide it). If there is no template or no matching item, this item will be hidden unless you explicitly show it with `visible: true`.
                </p>
              </div>
            </details>
            <details>
              <summary>
                value
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.coloraxis.colorbar.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                  <dt>Default:</dt>
                  <dd>""</dd>
                </dl>
                <p>
                  string - dtickformat for described zoom level, the same as "tickformat"
                </p>
              </div>
            </details>
          </div>
          <details>
            <summary>
              ticklen
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>5</dd>
              </dl>
              <p>
                Sets the tick length (in px).
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickmode
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "auto" | "linear" | "array" )</dd>
              </dl>
              <p>
                Sets the tick mode for this axis. If "auto", the number of ticks is set via `nticks`. If "linear", the placement of the ticks is determined by a starting position `tick0` and a tick step `dtick` ("linear" is the default value if `tick0` and `dtick` are provided). If "array", the placement of the ticks is set via `tickvals` and the tick text is `ticktext`. ("array" is the default value if `tickvals` is provided).
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickprefix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets a tick label prefix.
              </p>
            </div>
          </details>
          <details>
            <summary>
              ticks
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "outside" | "inside" | "" )</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Determines whether ticks are drawn or not. If "", this axis' ticks are not drawn. If "outside" ("inside"), this axis' are drawn outside (inside) the axis lines.
              </p>
            </div>
          </details>
          <details>
            <summary>
              ticksuffix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets a tick label suffix.
              </p>
            </div>
          </details>
          <details>
            <summary>
              ticktext
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>data array</dd>
              </dl>
              <p>
                Sets the text displayed at the ticks position via `tickvals`. Only has an effect if `tickmode` is set to "array". Used with `tickvals`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickvals
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>data array</dd>
              </dl>
              <p>
                Sets the values at which ticks on this axis appear. Only has an effect if `tickmode` is set to "array". Used with `ticktext`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickwidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the tick width (in px).
              </p>
            </div>
          </details>
          <details>
            <summary>
              title
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
                <dt>Default:</dt>
                <dd>"top"</dd>
              </dl>
              <p>
                Sets the title of the color bar. Note that before the existence of `title.text`, the title's contents used to be defined as the `title` attribute itself. This behavior has been deprecated.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                font
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.coloraxis.colorbar.title</dd>
                  <dt>Type:</dt>
                  <dd>object containing one or more of the keys listed below.</dd>
                </dl>
                <p>
                  Sets this color bar's title font. Note that the title's font used to be set by the now deprecated `titlefont` attribute.
                </p>
              </div>
            </details>
            <div style="margin-left: 22px">
              <details>
                <summary>
                  color
                </summary>
                <div style="margin-left: 22px">
                  <dl>
                    <dt>Parent:</dt>
                    <dd>layout.coloraxis.colorbar.title.font</dd>
                    <dt>Type:</dt>
                    <dd>color</dd>
                  </dl>
                </div>
              </details>
              <details>
                <summary>
                  family
                </summary>
                <div style="margin-left: 22px">
                  <dl>
                    <dt>Parent:</dt>
                    <dd>layout.coloraxis.colorbar.title.font</dd>
                    <dt>Type:</dt>
                    <dd>string</dd>
                  </dl>
                  <p>
                    HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
                  </p>
                </div>
              </details>
              <details>
                <summary>
                  size
                </summary>
                <div style="margin-left: 22px">
                  <dl>
                    <dt>Parent:</dt>
                    <dd>layout.coloraxis.colorbar.title.font</dd>
                    <dt>Type:</dt>
                    <dd>number greater than or equal to 1</dd>
                  </dl>
                </div>
              </details>
            </div>
            <details>
              <summary>
                side
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.coloraxis.colorbar.title</dd>
                  <dt>Type:</dt>
                  <dd>enumerated , one of ( "right" | "top" | "bottom" )</dd>
                  <dt>Default:</dt>
                  <dd>"top"</dd>
                </dl>
                <p>
                  Determines the location of color bar's title with respect to the color bar. Note that the title's location used to be set by the now deprecated `titleside` attribute.
                </p>
              </div>
            </details>
            <details>
              <summary>
                text
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.coloraxis.colorbar.title</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  Sets the title of the color bar. Note that before the existence of `title.text`, the title's contents used to be defined as the `title` attribute itself. This behavior has been deprecated.
                </p>
              </div>
            </details>
          </div>
          <details>
            <summary>
              x
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>number between or equal to -2 and 3</dd>
                <dt>Default:</dt>
                <dd>1.02</dd>
              </dl>
              <p>
                Sets the x position of the color bar (in plot fraction).
              </p>
            </div>
          </details>
          <details>
            <summary>
              xanchor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "left" | "center" | "right" )</dd>
                <dt>Default:</dt>
                <dd>"left"</dd>
              </dl>
              <p>
                Sets this color bar's horizontal position anchor. This anchor binds the `x` position to the "left", "center" or "right" of the color bar.
              </p>
            </div>
          </details>
          <details>
            <summary>
              xpad
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>10</dd>
              </dl>
              <p>
                Sets the amount of padding (in px) along the x direction.
              </p>
            </div>
          </details>
          <details>
            <summary>
              y
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>number between or equal to -2 and 3</dd>
                <dt>Default:</dt>
                <dd>0.5</dd>
              </dl>
              <p>
                Sets the y position of the color bar (in plot fraction).
              </p>
            </div>
          </details>
          <details>
            <summary>
              yanchor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "top" | "middle" | "bottom" )</dd>
                <dt>Default:</dt>
                <dd>"middle"</dd>
              </dl>
              <p>
                Sets this color bar's vertical position anchor This anchor binds the `y` position to the "top", "middle" or "bottom" of the color bar.
              </p>
            </div>
          </details>
          <details>
            <summary>
              ypad
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.coloraxis.colorbar</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>10</dd>
              </dl>
              <p>
                Sets the amount of padding (in px) along the y direction.
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            colorscale
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.coloraxis</dd>
              <dt>Type:</dt>
              <dd>colorscale</dd>
            </dl>
            <p>
              Sets the colorscale. The colorscale must be an array containing arrays mapping a normalized value to an rgb, rgba, hex, hsl, hsv, or named color string. At minimum, a mapping for the lowest (0) and highest (1) values are required. For example, `[[0, 'rgb(0,0,255)'], [1, 'rgb(255,0,0)']]`. To control the bounds of the colorscale in color space, use`cmin` and `cmax`. Alternatively, `colorscale` may be a palette name string of the following list: Greys,YlGnBu,Greens,YlOrRd,Bluered,RdBu,Reds,Blues,Picnic,Rainbow,Portland,Jet,Hot,Blackbody,Earth,Electric,Viridis,Cividis.
            </p>
          </div>
        </details>
        <details>
          <summary>
            reversescale
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.coloraxis</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Reverses the color mapping if true. If true, `cmin` will correspond to the last color in the array and `cmax` will correspond to the first color.
            </p>
          </div>
        </details>
        <details>
          <summary>
            showscale
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.coloraxis</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
              <dt>Default:</dt>
              <dd>true</dd>
            </dl>
            <p>
              Determines whether or not a colorbar is displayed for this trace.
            </p>
          </div>
        </details>
      </div>
      <details>
        <summary>
          colorscale
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>object containing one or more of the keys listed below.</dd>
            <dt>Default:</dt>
            <dd>[[0, rgb(220,220,220)], [0.2, rgb(245,195,157)], [0.4, rgb(245,160,105)], [1, rgb(178,10,28)], ]</dd>
          </dl>
          <p>
            Sets the default sequential colorscale for positive values. Note that `autocolorscale` must be true for this attribute to work.
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            diverging
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.colorscale</dd>
              <dt>Type:</dt>
              <dd>colorscale</dd>
              <dt>Default:</dt>
              <dd>[[0, rgb(5,10,172)], [0.35, rgb(106,137,247)], [0.5, rgb(190,190,190)], [0.6, rgb(220,170,132)], [0.7, rgb(230,145,90)], [1, rgb(178,10,28)], ]</dd>
            </dl>
            <p>
              Sets the default diverging colorscale. Note that `autocolorscale` must be true for this attribute to work.
            </p>
          </div>
        </details>
        <details>
          <summary>
            sequential
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.colorscale</dd>
              <dt>Type:</dt>
              <dd>colorscale</dd>
              <dt>Default:</dt>
              <dd>[[0, rgb(220,220,220)], [0.2, rgb(245,195,157)], [0.4, rgb(245,160,105)], [1, rgb(178,10,28)], ]</dd>
            </dl>
            <p>
              Sets the default sequential colorscale for positive values. Note that `autocolorscale` must be true for this attribute to work.
            </p>
          </div>
        </details>
        <details>
          <summary>
            sequentialminus
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.colorscale</dd>
              <dt>Type:</dt>
              <dd>colorscale</dd>
              <dt>Default:</dt>
              <dd>[[0, rgb(5,10,172)], [0.35, rgb(40,60,190)], [0.5, rgb(70,100,245)], [0.6, rgb(90,120,245)], [0.7, rgb(106,137,247)], [1, rgb(220,220,220)], ]</dd>
            </dl>
            <p>
              Sets the default sequential colorscale for negative values. Note that `autocolorscale` must be true for this attribute to work.
            </p>
          </div>
        </details>
      </div>
      <details>
        <summary>
          colorway
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>colorlist</dd>
            <dt>Default:</dt>
            <dd>[#1f77b4, #ff7f0e, #2ca02c, #d62728, #9467bd, #8c564b, #e377c2, #7f7f7f, #bcbd22, #17becf]</dd>
          </dl>
          <p>
            Sets the default trace colors.
          </p>
        </div>
      </details>
      <details>
        <summary>
          datarevision
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>number or categorical coordinate string</dd>
          </dl>
          <p>
            If provided, a changed value tells `Plotly.react` that one or more data arrays has changed. This way you can modify arrays in-place rather than making a complete new copy for an incremental change. If NOT provided, `Plotly.react` assumes that data arrays are being treated as immutable, thus any data array with a different identity from its predecessor contains new data.
          </p>
        </div>
      </details>
      <details>
        <summary>
          direction
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>enumerated , one of ( "clockwise" | "counterclockwise" )</dd>
          </dl>
          <p>
            Legacy polar charts are deprecated! Please switch to "polar" subplots. Sets the direction corresponding to positive angles in legacy polar charts.
          </p>
        </div>
      </details>
      <details>
        <summary>
          dragmode
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>enumerated , one of ( "zoom" | "pan" | "select" | "lasso" | "orbit" | "turntable" | false )</dd>
            <dt>Default:</dt>
            <dd>"zoom"</dd>
          </dl>
          <p>
            Determines the mode of drag interactions. "select" and "lasso" apply only to scatter traces with markers or text. "orbit" and "turntable" apply only to 3D scenes.
          </p>
        </div>
      </details>
      <details>
        <summary>
          editrevision
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>number or categorical coordinate string</dd>
          </dl>
          <p>
            Controls persistence of user-driven changes in `editable: true` configuration, other than trace names and axis titles. Defaults to `layout.uirevision`.
          </p>
        </div>
      </details>
      <details>
        <summary>
          font
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>object containing one or more of the keys listed below.</dd>
            <dt>Default:</dt>
            <dd>""Open Sans", verdana, arial, sans-serif"</dd>
          </dl>
          <p>
            Sets the global font. Note that fonts used in traces and other layout components inherit from the global font.
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            color
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.font</dd>
              <dt>Type:</dt>
              <dd>color</dd>
              <dt>Default:</dt>
              <dd>"#444"</dd>
            </dl>
          </div>
        </details>
        <details>
          <summary>
            family
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.font</dd>
              <dt>Type:</dt>
              <dd>string</dd>
              <dt>Default:</dt>
              <dd>""Open Sans", verdana, arial, sans-serif"</dd>
            </dl>
            <p>
              HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
            </p>
          </div>
        </details>
        <details>
          <summary>
            size
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.font</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 1</dd>
              <dt>Default:</dt>
              <dd>12</dd>
            </dl>
          </div>
        </details>
      </div>
      <details>
        <summary>
          geo
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>object containing one or more of the keys listed below.</dd>
            <dt>Default:</dt>
            <dd>[0, 1]</dd>
          </dl>
          <p>
            Sets the horizontal domain of this geo subplot (in plot fraction). Note that geo subplots are constrained by domain. In general, when `projection.scale` is set to 1. a map will fit either its x or y domain, but not both.
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            bgcolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.geo</dd>
              <dt>Type:</dt>
              <dd>color</dd>
              <dt>Default:</dt>
              <dd>"#fff"</dd>
            </dl>
            <p>
              Set the background color of the map
            </p>
          </div>
        </details>
        <details>
          <summary>
            center
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.geo</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
            </dl>
            <p>
              Sets the longitude of the map's center. By default, the map's longitude center lies at the middle of the longitude range for scoped projection and above `projection.rotation.lon` otherwise.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              lat
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.geo.center</dd>
                <dt>Type:</dt>
                <dd>number</dd>
              </dl>
              <p>
                Sets the latitude of the map's center. For all projection types, the map's latitude center lies at the middle of the latitude range by default.
              </p>
            </div>
          </details>
          <details>
            <summary>
              lon
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.geo.center</dd>
                <dt>Type:</dt>
                <dd>number</dd>
              </dl>
              <p>
                Sets the longitude of the map's center. By default, the map's longitude center lies at the middle of the longitude range for scoped projection and above `projection.rotation.lon` otherwise.
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            coastlinecolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.geo</dd>
              <dt>Type:</dt>
              <dd>color</dd>
              <dt>Default:</dt>
              <dd>"#444"</dd>
            </dl>
            <p>
              Sets the coastline color.
            </p>
          </div>
        </details>
        <details>
          <summary>
            coastlinewidth
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.geo</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>1</dd>
            </dl>
            <p>
              Sets the coastline stroke width (in px).
            </p>
          </div>
        </details>
        <details>
          <summary>
            countrycolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.geo</dd>
              <dt>Type:</dt>
              <dd>color</dd>
              <dt>Default:</dt>
              <dd>"#444"</dd>
            </dl>
            <p>
              Sets line color of the country boundaries.
            </p>
          </div>
        </details>
        <details>
          <summary>
            countrywidth
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.geo</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>1</dd>
            </dl>
            <p>
              Sets line width (in px) of the country boundaries.
            </p>
          </div>
        </details>
        <details>
          <summary>
            domain
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.geo</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>[0, 1]</dd>
            </dl>
            <p>
              Sets the horizontal domain of this geo subplot (in plot fraction). Note that geo subplots are constrained by domain. In general, when `projection.scale` is set to 1. a map will fit either its x or y domain, but not both.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              column
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.geo.domain</dd>
                <dt>Type:</dt>
                <dd>integer greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                If there is a layout grid, use the domain for this column in the grid for this geo subplot . Note that geo subplots are constrained by domain. In general, when `projection.scale` is set to 1. a map will fit either its x or y domain, but not both.
              </p>
            </div>
          </details>
          <details>
            <summary>
              row
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.geo.domain</dd>
                <dt>Type:</dt>
                <dd>integer greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                If there is a layout grid, use the domain for this row in the grid for this geo subplot . Note that geo subplots are constrained by domain. In general, when `projection.scale` is set to 1. a map will fit either its x or y domain, but not both.
              </p>
            </div>
          </details>
          <details>
            <summary>
              x
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.geo.domain</dd>
                <dt>Type:</dt>
                <dd>array</dd>
                <dt>Default:</dt>
                <dd>[0, 1]</dd>
              </dl>
              <p>
                Sets the horizontal domain of this geo subplot (in plot fraction). Note that geo subplots are constrained by domain. In general, when `projection.scale` is set to 1. a map will fit either its x or y domain, but not both.
              </p>
            </div>
          </details>
          <details>
            <summary>
              y
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.geo.domain</dd>
                <dt>Type:</dt>
                <dd>array</dd>
                <dt>Default:</dt>
                <dd>[0, 1]</dd>
              </dl>
              <p>
                Sets the vertical domain of this geo subplot (in plot fraction). Note that geo subplots are constrained by domain. In general, when `projection.scale` is set to 1. a map will fit either its x or y domain, but not both.
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            framecolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.geo</dd>
              <dt>Type:</dt>
              <dd>color</dd>
              <dt>Default:</dt>
              <dd>"#444"</dd>
            </dl>
            <p>
              Sets the color the frame.
            </p>
          </div>
        </details>
        <details>
          <summary>
            framewidth
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.geo</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>1</dd>
            </dl>
            <p>
              Sets the stroke width (in px) of the frame.
            </p>
          </div>
        </details>
        <details>
          <summary>
            lakecolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.geo</dd>
              <dt>Type:</dt>
              <dd>color</dd>
              <dt>Default:</dt>
              <dd>"#3399FF"</dd>
            </dl>
            <p>
              Sets the color of the lakes.
            </p>
          </div>
        </details>
        <details>
          <summary>
            landcolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.geo</dd>
              <dt>Type:</dt>
              <dd>color</dd>
              <dt>Default:</dt>
              <dd>"#F0DC82"</dd>
            </dl>
            <p>
              Sets the land mass color.
            </p>
          </div>
        </details>
        <details>
          <summary>
            lataxis
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.geo</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
            <p>
              Sets the range of this axis (in degrees), sets the map's clipped coordinates.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              dtick
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.geo.lataxis</dd>
                <dt>Type:</dt>
                <dd>number</dd>
              </dl>
              <p>
                Sets the graticule's longitude/latitude tick step.
              </p>
            </div>
          </details>
          <details>
            <summary>
              gridcolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.geo.lataxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#eee"</dd>
              </dl>
              <p>
                Sets the graticule's stroke color.
              </p>
            </div>
          </details>
          <details>
            <summary>
              gridwidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.geo.lataxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the graticule's stroke width (in px).
              </p>
            </div>
          </details>
          <details>
            <summary>
              range
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.geo.lataxis</dd>
                <dt>Type:</dt>
                <dd>array</dd>
              </dl>
              <p>
                Sets the range of this axis (in degrees), sets the map's clipped coordinates.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showgrid
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.geo.lataxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
              </dl>
              <p>
                Sets whether or not graticule are shown on the map.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tick0
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.geo.lataxis</dd>
                <dt>Type:</dt>
                <dd>number</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                Sets the graticule's starting tick longitude/latitude.
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            lonaxis
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.geo</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
            <p>
              Sets the range of this axis (in degrees), sets the map's clipped coordinates.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              dtick
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.geo.lonaxis</dd>
                <dt>Type:</dt>
                <dd>number</dd>
              </dl>
              <p>
                Sets the graticule's longitude/latitude tick step.
              </p>
            </div>
          </details>
          <details>
            <summary>
              gridcolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.geo.lonaxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#eee"</dd>
              </dl>
              <p>
                Sets the graticule's stroke color.
              </p>
            </div>
          </details>
          <details>
            <summary>
              gridwidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.geo.lonaxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the graticule's stroke width (in px).
              </p>
            </div>
          </details>
          <details>
            <summary>
              range
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.geo.lonaxis</dd>
                <dt>Type:</dt>
                <dd>array</dd>
              </dl>
              <p>
                Sets the range of this axis (in degrees), sets the map's clipped coordinates.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showgrid
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.geo.lonaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
              </dl>
              <p>
                Sets whether or not graticule are shown on the map.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tick0
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.geo.lonaxis</dd>
                <dt>Type:</dt>
                <dd>number</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                Sets the graticule's starting tick longitude/latitude.
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            oceancolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.geo</dd>
              <dt>Type:</dt>
              <dd>color</dd>
              <dt>Default:</dt>
              <dd>"#3399FF"</dd>
            </dl>
            <p>
              Sets the ocean color
            </p>
          </div>
        </details>
        <details>
          <summary>
            projection
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.geo</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>1</dd>
            </dl>
            <p>
              Sets the projection type.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              parallels
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.geo.projection</dd>
                <dt>Type:</dt>
                <dd>array</dd>
              </dl>
              <p>
                For conic projection types only. Sets the parallels (tangent, secant) where the cone intersects the sphere.
              </p>
            </div>
          </details>
          <details>
            <summary>
              rotation
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.geo.projection</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
              </dl>
              <p>
                Rotates the map along parallels (in degrees East). Defaults to the center of the `lonaxis.range` values.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                lat
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.geo.projection.rotation</dd>
                  <dt>Type:</dt>
                  <dd>number</dd>
                </dl>
                <p>
                  Rotates the map along meridians (in degrees North).
                </p>
              </div>
            </details>
            <details>
              <summary>
                lon
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.geo.projection.rotation</dd>
                  <dt>Type:</dt>
                  <dd>number</dd>
                </dl>
                <p>
                  Rotates the map along parallels (in degrees East). Defaults to the center of the `lonaxis.range` values.
                </p>
              </div>
            </details>
            <details>
              <summary>
                roll
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.geo.projection.rotation</dd>
                  <dt>Type:</dt>
                  <dd>number</dd>
                </dl>
                <p>
                  Roll the map (in degrees) For example, a roll of "180" makes the map appear upside down.
                </p>
              </div>
            </details>
          </div>
          <details>
            <summary>
              scale
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.geo.projection</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Zooms in or out on the map view. A scale of "1" corresponds to the largest zoom level that fits the map's lon and lat ranges.
              </p>
            </div>
          </details>
          <details>
            <summary>
              type
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.geo.projection</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "equirectangular" | "mercator" | "orthographic" | "natural earth" | "kavrayskiy7" | "miller" | "robinson" | "eckert4" | "azimuthal equal area" | "azimuthal equidistant" | "conic equal area" | "conic conformal" | "conic equidistant" | "gnomonic" | "stereographic" | "mollweide" | "hammer" | "transverse mercator" | "albers usa" | "winkel tripel" | "aitoff" | "sinusoidal" )</dd>
              </dl>
              <p>
                Sets the projection type.
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            resolution
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.geo</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "110" | "50" )</dd>
              <dt>Default:</dt>
              <dd>"110"</dd>
            </dl>
            <p>
              Sets the resolution of the base layers. The values have units of km/mm e.g. 110 corresponds to a scale ratio of 1:110,000,000.
            </p>
          </div>
        </details>
        <details>
          <summary>
            rivercolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.geo</dd>
              <dt>Type:</dt>
              <dd>color</dd>
              <dt>Default:</dt>
              <dd>"#3399FF"</dd>
            </dl>
            <p>
              Sets color of the rivers.
            </p>
          </div>
        </details>
        <details>
          <summary>
            riverwidth
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.geo</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>1</dd>
            </dl>
            <p>
              Sets the stroke width (in px) of the rivers.
            </p>
          </div>
        </details>
        <details>
          <summary>
            scope
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.geo</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "world" | "usa" | "europe" | "asia" | "africa" | "north america" | "south america" )</dd>
              <dt>Default:</dt>
              <dd>"world"</dd>
            </dl>
            <p>
              Set the scope of the map.
            </p>
          </div>
        </details>
        <details>
          <summary>
            showcoastlines
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.geo</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Sets whether or not the coastlines are drawn.
            </p>
          </div>
        </details>
        <details>
          <summary>
            showcountries
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.geo</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Sets whether or not country boundaries are drawn.
            </p>
          </div>
        </details>
        <details>
          <summary>
            showframe
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.geo</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Sets whether or not a frame is drawn around the map.
            </p>
          </div>
        </details>
        <details>
          <summary>
            showlakes
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.geo</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Sets whether or not lakes are drawn.
            </p>
          </div>
        </details>
        <details>
          <summary>
            showland
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.geo</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Sets whether or not land masses are filled in color.
            </p>
          </div>
        </details>
        <details>
          <summary>
            showocean
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.geo</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Sets whether or not oceans are filled in color.
            </p>
          </div>
        </details>
        <details>
          <summary>
            showrivers
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.geo</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Sets whether or not rivers are drawn.
            </p>
          </div>
        </details>
        <details>
          <summary>
            showsubunits
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.geo</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Sets whether or not boundaries of subunits within countries (e.g. states, provinces) are drawn.
            </p>
          </div>
        </details>
        <details>
          <summary>
            subunitcolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.geo</dd>
              <dt>Type:</dt>
              <dd>color</dd>
              <dt>Default:</dt>
              <dd>"#444"</dd>
            </dl>
            <p>
              Sets the color of the subunits boundaries.
            </p>
          </div>
        </details>
        <details>
          <summary>
            subunitwidth
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.geo</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>1</dd>
            </dl>
            <p>
              Sets the stroke width (in px) of the subunits boundaries.
            </p>
          </div>
        </details>
        <details>
          <summary>
            uirevision
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.geo</dd>
              <dt>Type:</dt>
              <dd>number or categorical coordinate string</dd>
            </dl>
            <p>
              Controls persistence of user-driven changes in the view (projection and center). Defaults to `layout.uirevision`.
            </p>
          </div>
        </details>
      </div>
      <details>
        <summary>
          grid
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>object containing one or more of the keys listed below.</dd>
            <dt>Default:</dt>
            <dd>"top to bottom"</dd>
          </dl>
          <p>
            The number of rows in the grid. If you provide a 2D `subplots` array or a `yaxes` array, its length is used as the default. But it's also possible to have a different length, if you want to leave a row at the end for non-cartesian subplots.
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            columns
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.grid</dd>
              <dt>Type:</dt>
              <dd>integer greater than or equal to 1</dd>
            </dl>
            <p>
              The number of columns in the grid. If you provide a 2D `subplots` array, the length of its longest row is used as the default. If you give an `xaxes` array, its length is used as the default. But it's also possible to have a different length, if you want to leave a row at the end for non-cartesian subplots.
            </p>
          </div>
        </details>
        <details>
          <summary>
            domain
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.grid</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>[0, 1]</dd>
            </dl>
            <p>
              Sets the horizontal domain of this grid subplot (in plot fraction). The first and last cells end exactly at the domain edges, with no grout around the edges.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              x
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.grid.domain</dd>
                <dt>Type:</dt>
                <dd>array</dd>
                <dt>Default:</dt>
                <dd>[0, 1]</dd>
              </dl>
              <p>
                Sets the horizontal domain of this grid subplot (in plot fraction). The first and last cells end exactly at the domain edges, with no grout around the edges.
              </p>
            </div>
          </details>
          <details>
            <summary>
              y
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.grid.domain</dd>
                <dt>Type:</dt>
                <dd>array</dd>
                <dt>Default:</dt>
                <dd>[0, 1]</dd>
              </dl>
              <p>
                Sets the vertical domain of this grid subplot (in plot fraction). The first and last cells end exactly at the domain edges, with no grout around the edges.
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            pattern
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.grid</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "independent" | "coupled" )</dd>
              <dt>Default:</dt>
              <dd>"coupled"</dd>
            </dl>
            <p>
              If no `subplots`, `xaxes`, or `yaxes` are given but we do have `rows` and `columns`, we can generate defaults using consecutive axis IDs, in two ways: "coupled" gives one x axis per column and one y axis per row. "independent" uses a new xy pair for each cell, left-to-right across each row then iterating rows according to `roworder`.
            </p>
          </div>
        </details>
        <details>
          <summary>
            roworder
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.grid</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "top to bottom" | "bottom to top" )</dd>
              <dt>Default:</dt>
              <dd>"top to bottom"</dd>
            </dl>
            <p>
              Is the first row the top or the bottom? Note that columns are always enumerated from left to right.
            </p>
          </div>
        </details>
        <details>
          <summary>
            rows
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.grid</dd>
              <dt>Type:</dt>
              <dd>integer greater than or equal to 1</dd>
            </dl>
            <p>
              The number of rows in the grid. If you provide a 2D `subplots` array or a `yaxes` array, its length is used as the default. But it's also possible to have a different length, if you want to leave a row at the end for non-cartesian subplots.
            </p>
          </div>
        </details>
        <details>
          <summary>
            subplots
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.grid</dd>
              <dt>Type:</dt>
              <dd>array</dd>
            </dl>
            <p>
              Used for freeform grids, where some axes may be shared across subplots but others are not. Each entry should be a cartesian subplot id, like "xy" or "x3y2", or "" to leave that cell empty. You may reuse x axes within the same column, and y axes within the same row. Non-cartesian subplots and traces that support `domain` can place themselves in this grid separately using the `gridcell` attribute.
            </p>
          </div>
        </details>
        <details>
          <summary>
            xaxes
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.grid</dd>
              <dt>Type:</dt>
              <dd>array</dd>
            </dl>
            <p>
              Used with `yaxes` when the x and y axes are shared across columns and rows. Each entry should be an x axis id like "x", "x2", etc., or "" to not put an x axis in that column. Entries other than "" must be unique. Ignored if `subplots` is present. If missing but `yaxes` is present, will generate consecutive IDs.
            </p>
          </div>
        </details>
        <details>
          <summary>
            xgap
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.grid</dd>
              <dt>Type:</dt>
              <dd>number between or equal to 0 and 1</dd>
            </dl>
            <p>
              Horizontal space between grid cells, expressed as a fraction of the total width available to one cell. Defaults to 0.1 for coupled-axes grids and 0.2 for independent grids.
            </p>
          </div>
        </details>
        <details>
          <summary>
            xside
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.grid</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "bottom" | "bottom plot" | "top plot" | "top" )</dd>
              <dt>Default:</dt>
              <dd>"bottom plot"</dd>
            </dl>
            <p>
              Sets where the x axis labels and titles go. "bottom" means the very bottom of the grid. "bottom plot" is the lowest plot that each x axis is used in. "top" and "top plot" are similar.
            </p>
          </div>
        </details>
        <details>
          <summary>
            yaxes
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.grid</dd>
              <dt>Type:</dt>
              <dd>array</dd>
            </dl>
            <p>
              Used with `yaxes` when the x and y axes are shared across columns and rows. Each entry should be an y axis id like "y", "y2", etc., or "" to not put a y axis in that row. Entries other than "" must be unique. Ignored if `subplots` is present. If missing but `xaxes` is present, will generate consecutive IDs.
            </p>
          </div>
        </details>
        <details>
          <summary>
            ygap
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.grid</dd>
              <dt>Type:</dt>
              <dd>number between or equal to 0 and 1</dd>
            </dl>
            <p>
              Vertical space between grid cells, expressed as a fraction of the total height available to one cell. Defaults to 0.1 for coupled-axes grids and 0.3 for independent grids.
            </p>
          </div>
        </details>
        <details>
          <summary>
            yside
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.grid</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "left" | "left plot" | "right plot" | "right" )</dd>
              <dt>Default:</dt>
              <dd>"left plot"</dd>
            </dl>
            <p>
              Sets where the y axis labels and titles go. "left" means the very left edge of the grid. "left plot" is the leftmost plot that each y axis is used in. "right" and "right plot" are similar.
            </p>
          </div>
        </details>
      </div>
      <details>
        <summary>
          height
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>number greater than or equal to 10</dd>
            <dt>Default:</dt>
            <dd>450</dd>
          </dl>
          <p>
            Sets the plot's height (in px).
          </p>
        </div>
      </details>
      <details>
        <summary>
          hidesources
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>boolean</dd>
          </dl>
          <p>
            Determines whether or not a text link citing the data source is placed at the bottom-right cored of the figure. Has only an effect only on graphs that have been generated via forked graphs from the plotly service (at https://plot.ly or on-premise).
          </p>
        </div>
      </details>
      <details>
        <summary>
          hoverdistance
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>integer greater than or equal to -1</dd>
            <dt>Default:</dt>
            <dd>20</dd>
          </dl>
          <p>
            Sets the default distance (in pixels) to look for data to add hover labels (-1 means no cutoff, 0 means no looking for data). This is only a real distance for hovering on point-like objects, like scatter points. For area-like objects (bars, scatter fills, etc) hovering is on inside the area and off outside, but these objects will not supersede hover on point-like objects in case of conflict.
          </p>
        </div>
      </details>
      <details>
        <summary>
          hoverlabel
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>object containing one or more of the keys listed below.</dd>
            <dt>Default:</dt>
            <dd>"Arial, sans-serif"</dd>
          </dl>
          <p>
            Sets the background color of all hover labels on graph
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            align
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.hoverlabel</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "left" | "right" | "auto" )</dd>
              <dt>Default:</dt>
              <dd>"auto"</dd>
            </dl>
            <p>
              Sets the horizontal alignment of the text content within hover label box. Has an effect only if the hover label text spans more two or more lines
            </p>
          </div>
        </details>
        <details>
          <summary>
            bgcolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.hoverlabel</dd>
              <dt>Type:</dt>
              <dd>color</dd>
            </dl>
            <p>
              Sets the background color of all hover labels on graph
            </p>
          </div>
        </details>
        <details>
          <summary>
            bordercolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.hoverlabel</dd>
              <dt>Type:</dt>
              <dd>color</dd>
            </dl>
            <p>
              Sets the border color of all hover labels on graph.
            </p>
          </div>
        </details>
        <details>
          <summary>
            font
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.hoverlabel</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>"Arial, sans-serif"</dd>
            </dl>
            <p>
              Sets the default hover label font used by all traces on the graph.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              color
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.hoverlabel.font</dd>
                <dt>Type:</dt>
                <dd>color</dd>
              </dl>
            </div>
          </details>
          <details>
            <summary>
              family
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.hoverlabel.font</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>"Arial, sans-serif"</dd>
              </dl>
              <p>
                HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
              </p>
            </div>
          </details>
          <details>
            <summary>
              size
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.hoverlabel.font</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 1</dd>
                <dt>Default:</dt>
                <dd>13</dd>
              </dl>
            </div>
          </details>
        </div>
        <details>
          <summary>
            namelength
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.hoverlabel</dd>
              <dt>Type:</dt>
              <dd>integer greater than or equal to -1</dd>
              <dt>Default:</dt>
              <dd>15</dd>
            </dl>
            <p>
              Sets the default length (in number of characters) of the trace name in the hover labels for all traces. -1 shows the whole name regardless of length. 0-3 shows the first 0-3 characters, and an integer &gt;3 will show the whole name if it is less than that many characters, but if it is longer, will truncate to `namelength - 3` characters and add an ellipsis.
            </p>
          </div>
        </details>
      </div>
      <details>
        <summary>
          hovermode
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>enumerated , one of ( "x" | "y" | "closest" | false )</dd>
          </dl>
          <p>
            Determines the mode of hover interactions. If `clickmode` includes the "select" flag, `hovermode` defaults to "closest". If `clickmode` lacks the "select" flag, it defaults to "x" or "y" (depending on the trace's `orientation` value) for plots based on cartesian coordinates. For anything else the default value is "closest".
          </p>
        </div>
      </details>
      <details>
        <summary>
          images
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>array of object where each object has one or more of the keys listed below.</dd>
            <dt>Default:</dt>
            <dd>true</dd>
          </dl>
          <p>
            Determines whether or not this image is visible.
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            layer
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.images[]</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "below" | "above" )</dd>
              <dt>Default:</dt>
              <dd>"above"</dd>
            </dl>
            <p>
              Specifies whether images are drawn below or above traces. When `xref` and `yref` are both set to `paper`, image is drawn below the entire plot area.
            </p>
          </div>
        </details>
        <details>
          <summary>
            name
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.images[]</dd>
              <dt>Type:</dt>
              <dd>string</dd>
            </dl>
            <p>
              When used in a template, named items are created in the output figure in addition to any items the figure already has in this array. You can modify these items in the output figure by making your own item with `templateitemname` matching this `name` alongside your modifications (including `visible: false` or `enabled: false` to hide it). Has no effect outside of a template.
            </p>
          </div>
        </details>
        <details>
          <summary>
            opacity
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.images[]</dd>
              <dt>Type:</dt>
              <dd>number between or equal to 0 and 1</dd>
              <dt>Default:</dt>
              <dd>1</dd>
            </dl>
            <p>
              Sets the opacity of the image.
            </p>
          </div>
        </details>
        <details>
          <summary>
            sizex
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.images[]</dd>
              <dt>Type:</dt>
              <dd>number</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
            <p>
              Sets the image container size horizontally. The image will be sized based on the `position` value. When `xref` is set to `paper`, units are sized relative to the plot width.
            </p>
          </div>
        </details>
        <details>
          <summary>
            sizey
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.images[]</dd>
              <dt>Type:</dt>
              <dd>number</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
            <p>
              Sets the image container size vertically. The image will be sized based on the `position` value. When `yref` is set to `paper`, units are sized relative to the plot height.
            </p>
          </div>
        </details>
        <details>
          <summary>
            sizing
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.images[]</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "fill" | "contain" | "stretch" )</dd>
              <dt>Default:</dt>
              <dd>"contain"</dd>
            </dl>
            <p>
              Specifies which dimension of the image to constrain.
            </p>
          </div>
        </details>
        <details>
          <summary>
            source
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.images[]</dd>
              <dt>Type:</dt>
              <dd>string</dd>
            </dl>
            <p>
              Specifies the URL of the image to be used. The URL must be accessible from the domain where the plot code is run, and can be either relative or absolute.
            </p>
          </div>
        </details>
        <details>
          <summary>
            templateitemname
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.images[]</dd>
              <dt>Type:</dt>
              <dd>string</dd>
            </dl>
            <p>
              Used to refer to a named item in this array in the template. Named items from the template will be created even without a matching item in the input figure, but you can modify one by making an item with `templateitemname` matching its `name`, alongside your modifications (including `visible: false` or `enabled: false` to hide it). If there is no template or no matching item, this item will be hidden unless you explicitly show it with `visible: true`.
            </p>
          </div>
        </details>
        <details>
          <summary>
            visible
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.images[]</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
              <dt>Default:</dt>
              <dd>true</dd>
            </dl>
            <p>
              Determines whether or not this image is visible.
            </p>
          </div>
        </details>
        <details>
          <summary>
            x
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.images[]</dd>
              <dt>Type:</dt>
              <dd>number or categorical coordinate string</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
            <p>
              Sets the image's x position. When `xref` is set to `paper`, units are sized relative to the plot height. See `xref` for more info
            </p>
          </div>
        </details>
        <details>
          <summary>
            xanchor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.images[]</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "left" | "center" | "right" )</dd>
              <dt>Default:</dt>
              <dd>"left"</dd>
            </dl>
            <p>
              Sets the anchor for the x position
            </p>
          </div>
        </details>
        <details>
          <summary>
            xref
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.images[]</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "paper" | "/^x([2-9]|[1-9][0-9]+)?$/" )</dd>
              <dt>Default:</dt>
              <dd>"paper"</dd>
            </dl>
            <p>
              Sets the images's x coordinate axis. If set to a x axis id (e.g. "x" or "x2"), the `x` position refers to an x data coordinate If set to "paper", the `x` position refers to the distance from the left of plot in normalized coordinates where "0" ("1") corresponds to the left (right).
            </p>
          </div>
        </details>
        <details>
          <summary>
            y
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.images[]</dd>
              <dt>Type:</dt>
              <dd>number or categorical coordinate string</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
            <p>
              Sets the image's y position. When `yref` is set to `paper`, units are sized relative to the plot height. See `yref` for more info
            </p>
          </div>
        </details>
        <details>
          <summary>
            yanchor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.images[]</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "top" | "middle" | "bottom" )</dd>
              <dt>Default:</dt>
              <dd>"top"</dd>
            </dl>
            <p>
              Sets the anchor for the y position.
            </p>
          </div>
        </details>
        <details>
          <summary>
            yref
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.images[]</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "paper" | "/^y([2-9]|[1-9][0-9]+)?$/" )</dd>
              <dt>Default:</dt>
              <dd>"paper"</dd>
            </dl>
            <p>
              Sets the images's y coordinate axis. If set to a y axis id (e.g. "y" or "y2"), the `y` position refers to a y data coordinate. If set to "paper", the `y` position refers to the distance from the bottom of the plot in normalized coordinates where "0" ("1") corresponds to the bottom (top).
            </p>
          </div>
        </details>
      </div>
      <details>
        <summary>
          legend
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>object containing one or more of the keys listed below.</dd>
            <dt>Default:</dt>
            <dd>"#444"</dd>
          </dl>
          <p>
            Sets the legend background color.
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            bgcolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.legend</dd>
              <dt>Type:</dt>
              <dd>color</dd>
            </dl>
            <p>
              Sets the legend background color.
            </p>
          </div>
        </details>
        <details>
          <summary>
            bordercolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.legend</dd>
              <dt>Type:</dt>
              <dd>color</dd>
              <dt>Default:</dt>
              <dd>"#444"</dd>
            </dl>
            <p>
              Sets the color of the border enclosing the legend.
            </p>
          </div>
        </details>
        <details>
          <summary>
            borderwidth
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.legend</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
            <p>
              Sets the width (in px) of the border enclosing the legend.
            </p>
          </div>
        </details>
        <details>
          <summary>
            font
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.legend</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
            </dl>
            <p>
              Sets the font used to text the legend items.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              color
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.legend.font</dd>
                <dt>Type:</dt>
                <dd>color</dd>
              </dl>
            </div>
          </details>
          <details>
            <summary>
              family
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.legend.font</dd>
                <dt>Type:</dt>
                <dd>string</dd>
              </dl>
              <p>
                HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
              </p>
            </div>
          </details>
          <details>
            <summary>
              size
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.legend.font</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 1</dd>
              </dl>
            </div>
          </details>
        </div>
        <details>
          <summary>
            itemclick
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.legend</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "toggle" | "toggleothers" | false )</dd>
              <dt>Default:</dt>
              <dd>"toggle"</dd>
            </dl>
            <p>
              Determines the behavior on legend item click. "toggle" toggles the visibility of the item clicked on the graph. "toggleothers" makes the clicked item the sole visible item on the graph. "false" disable legend item click interactions.
            </p>
          </div>
        </details>
        <details>
          <summary>
            itemdoubleclick
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.legend</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "toggle" | "toggleothers" | false )</dd>
              <dt>Default:</dt>
              <dd>"toggleothers"</dd>
            </dl>
            <p>
              Determines the behavior on legend item double-click. "toggle" toggles the visibility of the item clicked on the graph. "toggleothers" makes the clicked item the sole visible item on the graph. "false" disable legend item double-click interactions.
            </p>
          </div>
        </details>
        <details>
          <summary>
            itemsizing
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.legend</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "trace" | "constant" )</dd>
              <dt>Default:</dt>
              <dd>"trace"</dd>
            </dl>
            <p>
              Determines if the legend items symbols scale with their corresponding "trace" attributes or remain "constant" independent of the symbol size on the graph.
            </p>
          </div>
        </details>
        <details>
          <summary>
            orientation
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.legend</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "v" | "h" )</dd>
              <dt>Default:</dt>
              <dd>"v"</dd>
            </dl>
            <p>
              Sets the orientation of the legend.
            </p>
          </div>
        </details>
        <details>
          <summary>
            tracegroupgap
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.legend</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>10</dd>
            </dl>
            <p>
              Sets the amount of vertical space (in px) between legend groups.
            </p>
          </div>
        </details>
        <details>
          <summary>
            traceorder
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.legend</dd>
              <dt>Type:</dt>
              <dd>flaglist string. Any combination of "reversed", "grouped" joined with a "+" OR "normal".</dd>
            </dl>
            <p>
              Determines the order at which the legend items are displayed. If "normal", the items are displayed top-to-bottom in the same order as the input data. If "reversed", the items are displayed in the opposite order as "normal". If "grouped", the items are displayed in groups (when a trace `legendgroup` is provided). if "grouped+reversed", the items are displayed in the opposite order as "grouped".
            </p>
          </div>
        </details>
        <details>
          <summary>
            uirevision
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.legend</dd>
              <dt>Type:</dt>
              <dd>number or categorical coordinate string</dd>
            </dl>
            <p>
              Controls persistence of legend-driven changes in trace and pie label visibility. Defaults to `layout.uirevision`.
            </p>
          </div>
        </details>
        <details>
          <summary>
            valign
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.legend</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "top" | "middle" | "bottom" )</dd>
              <dt>Default:</dt>
              <dd>"middle"</dd>
            </dl>
            <p>
              Sets the vertical alignment of the symbols with respect to their associated text.
            </p>
          </div>
        </details>
        <details>
          <summary>
            x
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.legend</dd>
              <dt>Type:</dt>
              <dd>number between or equal to -2 and 3</dd>
            </dl>
            <p>
              Sets the x position (in normalized coordinates) of the legend. Defaults to "1.02" for vertical legends and defaults to "0" for horizontal legends.
            </p>
          </div>
        </details>
        <details>
          <summary>
            xanchor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.legend</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "auto" | "left" | "center" | "right" )</dd>
              <dt>Default:</dt>
              <dd>"left"</dd>
            </dl>
            <p>
              Sets the legend's horizontal position anchor. This anchor binds the `x` position to the "left", "center" or "right" of the legend. Value "auto" anchors legends to the right for `x` values greater than or equal to 2/3, anchors legends to the left for `x` values less than or equal to 1/3 and anchors legends with respect to their center otherwise.
            </p>
          </div>
        </details>
        <details>
          <summary>
            y
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.legend</dd>
              <dt>Type:</dt>
              <dd>number between or equal to -2 and 3</dd>
            </dl>
            <p>
              Sets the y position (in normalized coordinates) of the legend. Defaults to "1" for vertical legends, defaults to "-0.1" for horizontal legends on graphs w/o range sliders and defaults to "1.1" for horizontal legends on graph with one or multiple range sliders.
            </p>
          </div>
        </details>
        <details>
          <summary>
            yanchor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.legend</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "auto" | "top" | "middle" | "bottom" )</dd>
            </dl>
            <p>
              Sets the legend's vertical position anchor This anchor binds the `y` position to the "top", "middle" or "bottom" of the legend. Value "auto" anchors legends at their bottom for `y` values less than or equal to 1/3, anchors legends to at their top for `y` values greater than or equal to 2/3 and anchors legends with respect to their middle otherwise.
            </p>
          </div>
        </details>
      </div>
      <details>
        <summary>
          mapbox
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>object containing one or more of the keys listed below.</dd>
            <dt>Default:</dt>
            <dd>[0, 1]</dd>
          </dl>
          <p>
            Sets the horizontal domain of this mapbox subplot (in plot fraction).
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            accesstoken
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.mapbox</dd>
              <dt>Type:</dt>
              <dd>string</dd>
            </dl>
            <p>
              Sets the mapbox access token to be used for this mapbox map. Alternatively, the mapbox access token can be set in the configuration options under `mapboxAccessToken`. Note that accessToken are only required when `style` (e.g with values : basic, streets, outdoors, light, dark, satellite, satellite-streets ) and/or a layout layer references the Mapbox server.
            </p>
          </div>
        </details>
        <details>
          <summary>
            bearing
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.mapbox</dd>
              <dt>Type:</dt>
              <dd>number</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
            <p>
              Sets the bearing angle of the map in degrees counter-clockwise from North (mapbox.bearing).
            </p>
          </div>
        </details>
        <details>
          <summary>
            center
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.mapbox</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
            <p>
              Sets the longitude of the center of the map (in degrees East).
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              lat
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.mapbox.center</dd>
                <dt>Type:</dt>
                <dd>number</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                Sets the latitude of the center of the map (in degrees North).
              </p>
            </div>
          </details>
          <details>
            <summary>
              lon
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.mapbox.center</dd>
                <dt>Type:</dt>
                <dd>number</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                Sets the longitude of the center of the map (in degrees East).
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            domain
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.mapbox</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>[0, 1]</dd>
            </dl>
            <p>
              Sets the horizontal domain of this mapbox subplot (in plot fraction).
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              column
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.mapbox.domain</dd>
                <dt>Type:</dt>
                <dd>integer greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                If there is a layout grid, use the domain for this column in the grid for this mapbox subplot .
              </p>
            </div>
          </details>
          <details>
            <summary>
              row
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.mapbox.domain</dd>
                <dt>Type:</dt>
                <dd>integer greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                If there is a layout grid, use the domain for this row in the grid for this mapbox subplot .
              </p>
            </div>
          </details>
          <details>
            <summary>
              x
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.mapbox.domain</dd>
                <dt>Type:</dt>
                <dd>array</dd>
                <dt>Default:</dt>
                <dd>[0, 1]</dd>
              </dl>
              <p>
                Sets the horizontal domain of this mapbox subplot (in plot fraction).
              </p>
            </div>
          </details>
          <details>
            <summary>
              y
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.mapbox.domain</dd>
                <dt>Type:</dt>
                <dd>array</dd>
                <dt>Default:</dt>
                <dd>[0, 1]</dd>
              </dl>
              <p>
                Sets the vertical domain of this mapbox subplot (in plot fraction).
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            layers
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.mapbox</dd>
              <dt>Type:</dt>
              <dd>array of object where each object has one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>true</dd>
            </dl>
            <p>
              Determines whether this layer is displayed
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              below
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.mapbox.layers[]</dd>
                <dt>Type:</dt>
                <dd>string</dd>
              </dl>
              <p>
                Determines if the layer will be inserted before the layer with the specified ID. If omitted or set to '', the layer will be inserted above every existing layer.
              </p>
            </div>
          </details>
          <details>
            <summary>
              circle
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.mapbox.layers[]</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
                <dt>Default:</dt>
                <dd>15</dd>
              </dl>
              <p>
                Sets the circle radius (mapbox.layer.paint.circle-radius). Has an effect only when `type` is set to "circle".
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                radius
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.mapbox.layers[].circle</dd>
                  <dt>Type:</dt>
                  <dd>number</dd>
                  <dt>Default:</dt>
                  <dd>15</dd>
                </dl>
                <p>
                  Sets the circle radius (mapbox.layer.paint.circle-radius). Has an effect only when `type` is set to "circle".
                </p>
              </div>
            </details>
          </div>
          <details>
            <summary>
              color
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.mapbox.layers[]</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets the primary layer color. If `type` is "circle", color corresponds to the circle color (mapbox.layer.paint.circle-color) If `type` is "line", color corresponds to the line color (mapbox.layer.paint.line-color) If `type` is "fill", color corresponds to the fill color (mapbox.layer.paint.fill-color) If `type` is "symbol", color corresponds to the icon color (mapbox.layer.paint.icon-color)
              </p>
            </div>
          </details>
          <details>
            <summary>
              coordinates
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.mapbox.layers[]</dd>
                <dt>Type:</dt>
                <dd>number or categorical coordinate string</dd>
              </dl>
              <p>
                Sets the coordinates array contains [longitude, latitude] pairs for the image corners listed in clockwise order: top left, top right, bottom right, bottom left. Only has an effect for "image" `sourcetype`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              fill
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.mapbox.layers[]</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets the fill outline color (mapbox.layer.paint.fill-outline-color). Has an effect only when `type` is set to "fill".
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                outlinecolor
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.mapbox.layers[].fill</dd>
                  <dt>Type:</dt>
                  <dd>color</dd>
                  <dt>Default:</dt>
                  <dd>"#444"</dd>
                </dl>
                <p>
                  Sets the fill outline color (mapbox.layer.paint.fill-outline-color). Has an effect only when `type` is set to "fill".
                </p>
              </div>
            </details>
          </div>
          <details>
            <summary>
              line
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.mapbox.layers[]</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
                <dt>Default:</dt>
                <dd>2</dd>
              </dl>
              <p>
                Sets the line width (mapbox.layer.paint.line-width). Has an effect only when `type` is set to "line".
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                dash
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.mapbox.layers[].line</dd>
                  <dt>Type:</dt>
                  <dd>data array</dd>
                </dl>
                <p>
                  Sets the length of dashes and gaps (mapbox.layer.paint.line-dasharray). Has an effect only when `type` is set to "line".
                </p>
              </div>
            </details>
            <details>
              <summary>
                width
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.mapbox.layers[].line</dd>
                  <dt>Type:</dt>
                  <dd>number</dd>
                  <dt>Default:</dt>
                  <dd>2</dd>
                </dl>
                <p>
                  Sets the line width (mapbox.layer.paint.line-width). Has an effect only when `type` is set to "line".
                </p>
              </div>
            </details>
          </div>
          <details>
            <summary>
              maxzoom
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.mapbox.layers[]</dd>
                <dt>Type:</dt>
                <dd>number between or equal to 0 and 24</dd>
                <dt>Default:</dt>
                <dd>24</dd>
              </dl>
              <p>
                Sets the maximum zoom level (mapbox.layer.maxzoom). At zoom levels equal to or greater than the maxzoom, the layer will be hidden.
              </p>
            </div>
          </details>
          <details>
            <summary>
              minzoom
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.mapbox.layers[]</dd>
                <dt>Type:</dt>
                <dd>number between or equal to 0 and 24</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                Sets the minimum zoom level (mapbox.layer.minzoom). At zoom levels less than the minzoom, the layer will be hidden.
              </p>
            </div>
          </details>
          <details>
            <summary>
              name
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.mapbox.layers[]</dd>
                <dt>Type:</dt>
                <dd>string</dd>
              </dl>
              <p>
                When used in a template, named items are created in the output figure in addition to any items the figure already has in this array. You can modify these items in the output figure by making your own item with `templateitemname` matching this `name` alongside your modifications (including `visible: false` or `enabled: false` to hide it). Has no effect outside of a template.
              </p>
            </div>
          </details>
          <details>
            <summary>
              opacity
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.mapbox.layers[]</dd>
                <dt>Type:</dt>
                <dd>number between or equal to 0 and 1</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the opacity of the layer. If `type` is "circle", opacity corresponds to the circle opacity (mapbox.layer.paint.circle-opacity) If `type` is "line", opacity corresponds to the line opacity (mapbox.layer.paint.line-opacity) If `type` is "fill", opacity corresponds to the fill opacity (mapbox.layer.paint.fill-opacity) If `type` is "symbol", opacity corresponds to the icon/text opacity (mapbox.layer.paint.text-opacity)
              </p>
            </div>
          </details>
          <details>
            <summary>
              source
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.mapbox.layers[]</dd>
                <dt>Type:</dt>
                <dd>number or categorical coordinate string</dd>
              </dl>
              <p>
                Sets the source data for this layer (mapbox.layer.source). When `sourcetype` is set to "geojson", `source` can be a URL to a GeoJSON or a GeoJSON object. When `sourcetype` is set to "vector" or "raster", `source` can be a URL or an array of tile URLs. When `sourcetype` is set to "image", `source` can be a URL to an image.
              </p>
            </div>
          </details>
          <details>
            <summary>
              sourceattribution
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.mapbox.layers[]</dd>
                <dt>Type:</dt>
                <dd>string</dd>
              </dl>
              <p>
                Sets the attribution for this source.
              </p>
            </div>
          </details>
          <details>
            <summary>
              sourcelayer
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.mapbox.layers[]</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Specifies the layer to use from a vector tile source (mapbox.layer.source-layer). Required for "vector" source type that supports multiple layers.
              </p>
            </div>
          </details>
          <details>
            <summary>
              sourcetype
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.mapbox.layers[]</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "geojson" | "vector" | "raster" | "image" )</dd>
                <dt>Default:</dt>
                <dd>"geojson"</dd>
              </dl>
              <p>
                Sets the source type for this layer, that is the type of the layer data.
              </p>
            </div>
          </details>
          <details>
            <summary>
              symbol
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.mapbox.layers[]</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
                <dt>Default:</dt>
                <dd>"marker"</dd>
              </dl>
              <p>
                Sets the symbol icon image (mapbox.layer.layout.icon-image). Full list: https://www.mapbox.com/maki-icons/
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                icon
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.mapbox.layers[].symbol</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                  <dt>Default:</dt>
                  <dd>"marker"</dd>
                </dl>
                <p>
                  Sets the symbol icon image (mapbox.layer.layout.icon-image). Full list: https://www.mapbox.com/maki-icons/
                </p>
              </div>
            </details>
            <details>
              <summary>
                iconsize
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.mapbox.layers[].symbol</dd>
                  <dt>Type:</dt>
                  <dd>number</dd>
                  <dt>Default:</dt>
                  <dd>10</dd>
                </dl>
                <p>
                  Sets the symbol icon size (mapbox.layer.layout.icon-size). Has an effect only when `type` is set to "symbol".
                </p>
              </div>
            </details>
            <details>
              <summary>
                placement
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.mapbox.layers[].symbol</dd>
                  <dt>Type:</dt>
                  <dd>enumerated , one of ( "point" | "line" | "line-center" )</dd>
                  <dt>Default:</dt>
                  <dd>"point"</dd>
                </dl>
                <p>
                  Sets the symbol and/or text placement (mapbox.layer.layout.symbol-placement). If `placement` is "point", the label is placed where the geometry is located If `placement` is "line", the label is placed along the line of the geometry If `placement` is "line-center", the label is placed on the center of the geometry
                </p>
              </div>
            </details>
            <details>
              <summary>
                text
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.mapbox.layers[].symbol</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                  <dt>Default:</dt>
                  <dd>""</dd>
                </dl>
                <p>
                  Sets the symbol text (mapbox.layer.layout.text-field).
                </p>
              </div>
            </details>
            <details>
              <summary>
                textfont
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.mapbox.layers[].symbol</dd>
                  <dt>Type:</dt>
                  <dd>object containing one or more of the keys listed below.</dd>
                  <dt>Default:</dt>
                  <dd>"Open Sans Regular, Arial Unicode MS Regular"</dd>
                </dl>
                <p>
                  Sets the icon text font (color=mapbox.layer.paint.text-color, size=mapbox.layer.layout.text-size). Has an effect only when `type` is set to "symbol".
                </p>
              </div>
            </details>
            <div style="margin-left: 22px">
              <details>
                <summary>
                  color
                </summary>
                <div style="margin-left: 22px">
                  <dl>
                    <dt>Parent:</dt>
                    <dd>layout.mapbox.layers[].symbol.textfont</dd>
                    <dt>Type:</dt>
                    <dd>color</dd>
                  </dl>
                </div>
              </details>
              <details>
                <summary>
                  family
                </summary>
                <div style="margin-left: 22px">
                  <dl>
                    <dt>Parent:</dt>
                    <dd>layout.mapbox.layers[].symbol.textfont</dd>
                    <dt>Type:</dt>
                    <dd>string</dd>
                    <dt>Default:</dt>
                    <dd>"Open Sans Regular, Arial Unicode MS Regular"</dd>
                  </dl>
                  <p>
                    HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
                  </p>
                </div>
              </details>
              <details>
                <summary>
                  size
                </summary>
                <div style="margin-left: 22px">
                  <dl>
                    <dt>Parent:</dt>
                    <dd>layout.mapbox.layers[].symbol.textfont</dd>
                    <dt>Type:</dt>
                    <dd>number greater than or equal to 1</dd>
                  </dl>
                </div>
              </details>
            </div>
            <details>
              <summary>
                textposition
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.mapbox.layers[].symbol</dd>
                  <dt>Type:</dt>
                  <dd>enumerated , one of ( "top left" | "top center" | "top right" | "middle left" | "middle center" | "middle right" | "bottom left" | "bottom center" | "bottom right" )</dd>
                  <dt>Default:</dt>
                  <dd>"middle center"</dd>
                </dl>
                <p>
                  Sets the positions of the `text` elements with respects to the (x,y) coordinates.
                </p>
              </div>
            </details>
          </div>
          <details>
            <summary>
              templateitemname
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.mapbox.layers[]</dd>
                <dt>Type:</dt>
                <dd>string</dd>
              </dl>
              <p>
                Used to refer to a named item in this array in the template. Named items from the template will be created even without a matching item in the input figure, but you can modify one by making an item with `templateitemname` matching its `name`, alongside your modifications (including `visible: false` or `enabled: false` to hide it). If there is no template or no matching item, this item will be hidden unless you explicitly show it with `visible: true`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              type
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.mapbox.layers[]</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "circle" | "line" | "fill" | "symbol" | "raster" )</dd>
                <dt>Default:</dt>
                <dd>"circle"</dd>
              </dl>
              <p>
                Sets the layer type, that is the how the layer data set in `source` will be rendered With `sourcetype` set to "geojson", the following values are allowed: "circle", "line", "fill" and "symbol". but note that "line" and "fill" are not compatible with Point GeoJSON geometries. With `sourcetype` set to "vector", the following values are allowed: "circle", "line", "fill" and "symbol". With `sourcetype` set to "raster" or `"image"`, only the "raster" value is allowed.
              </p>
            </div>
          </details>
          <details>
            <summary>
              visible
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.mapbox.layers[]</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether this layer is displayed
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            pitch
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.mapbox</dd>
              <dt>Type:</dt>
              <dd>number</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
            <p>
              Sets the pitch angle of the map (in degrees, where "0" means perpendicular to the surface of the map) (mapbox.pitch).
            </p>
          </div>
        </details>
        <details>
          <summary>
            style
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.mapbox</dd>
              <dt>Type:</dt>
              <dd>number or categorical coordinate string</dd>
              <dt>Default:</dt>
              <dd>basic</dd>
            </dl>
            <p>
              Defines the map layers that are rendered by default below the trace layers defined in `data`, which are themselves by default rendered below the layers defined in `layout.mapbox.layers`. These layers can be defined either explicitly as a Mapbox Style object which can contain multiple layer definitions that load data from any public or private Tile Map Service (TMS or XYZ) or Web Map Service (WMS) or implicitly by using one of the built-in style objects which use WMSes which do not require any access tokens, or by using a default Mapbox style or custom Mapbox style URL, both of which require a Mapbox access token Note that Mapbox access token can be set in the `accesstoken` attribute or in the `mapboxAccessToken` config option. Mapbox Style objects are of the form described in the Mapbox GL JS documentation available at https://docs.mapbox.com/mapbox-gl-js/style-spec The built-in plotly.js styles objects are: open-street-map, white-bg, carto-positron, carto-darkmatter, stamen-terrain, stamen-toner, stamen-watercolor The built-in Mapbox styles are: basic, streets, outdoors, light, dark, satellite, satellite-streets Mapbox style URLs are of the form: mapbox://mapbox.mapbox-&lt;name&gt;-&lt;version&gt;
            </p>
          </div>
        </details>
        <details>
          <summary>
            uirevision
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.mapbox</dd>
              <dt>Type:</dt>
              <dd>number or categorical coordinate string</dd>
            </dl>
            <p>
              Controls persistence of user-driven changes in the view: `center`, `zoom`, `bearing`, `pitch`. Defaults to `layout.uirevision`.
            </p>
          </div>
        </details>
        <details>
          <summary>
            zoom
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.mapbox</dd>
              <dt>Type:</dt>
              <dd>number</dd>
              <dt>Default:</dt>
              <dd>1</dd>
            </dl>
            <p>
              Sets the zoom level of the map (mapbox.zoom).
            </p>
          </div>
        </details>
      </div>
      <details>
        <summary>
          margin
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>object containing one or more of the keys listed below.</dd>
            <dt>Default:</dt>
            <dd>80</dd>
          </dl>
          <p>
            Sets the left margin (in px).
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            autoexpand
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.margin</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
              <dt>Default:</dt>
              <dd>true</dd>
            </dl>
            <p>
              Turns on/off margin expansion computations. Legends, colorbars, updatemenus, sliders, axis rangeselector and rangeslider are allowed to push the margins by defaults.
            </p>
          </div>
        </details>
        <details>
          <summary>
            b
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.margin</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>80</dd>
            </dl>
            <p>
              Sets the bottom margin (in px).
            </p>
          </div>
        </details>
        <details>
          <summary>
            l
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.margin</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>80</dd>
            </dl>
            <p>
              Sets the left margin (in px).
            </p>
          </div>
        </details>
        <details>
          <summary>
            pad
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.margin</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
            <p>
              Sets the amount of padding (in px) between the plotting area and the axis lines
            </p>
          </div>
        </details>
        <details>
          <summary>
            t
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.margin</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>100</dd>
            </dl>
            <p>
              Sets the top margin (in px).
            </p>
          </div>
        </details>
      </div>
      <details>
        <summary>
          meta
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>number or categorical coordinate string</dd>
          </dl>
          <p>
            Assigns extra meta information that can be used in various `text` attributes. Attributes such as the graph, axis and colorbar `title.text`, annotation `text` `trace.name` in legend items, `rangeselector`, `updatemenus` and `sliders` `label` text all support `meta`. One can access `meta` fields using template strings: `%{meta[i]}` where `i` is the index of the `meta` item in question. `meta` can also be an object for example `{key: value}` which can be accessed %{meta[key]}.
          </p>
        </div>
      </details>
      <details>
        <summary>
          modebar
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>object containing one or more of the keys listed below.</dd>
            <dt>Default:</dt>
            <dd>"h"</dd>
          </dl>
          <p>
            Sets the orientation of the modebar.
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            activecolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.modebar</dd>
              <dt>Type:</dt>
              <dd>color</dd>
            </dl>
            <p>
              Sets the color of the active or hovered on icons in the modebar.
            </p>
          </div>
        </details>
        <details>
          <summary>
            bgcolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.modebar</dd>
              <dt>Type:</dt>
              <dd>color</dd>
            </dl>
            <p>
              Sets the background color of the modebar.
            </p>
          </div>
        </details>
        <details>
          <summary>
            color
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.modebar</dd>
              <dt>Type:</dt>
              <dd>color</dd>
            </dl>
            <p>
              Sets the color of the icons in the modebar.
            </p>
          </div>
        </details>
        <details>
          <summary>
            orientation
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.modebar</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "v" | "h" )</dd>
              <dt>Default:</dt>
              <dd>"h"</dd>
            </dl>
            <p>
              Sets the orientation of the modebar.
            </p>
          </div>
        </details>
        <details>
          <summary>
            uirevision
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.modebar</dd>
              <dt>Type:</dt>
              <dd>number or categorical coordinate string</dd>
            </dl>
            <p>
              Controls persistence of user-driven changes related to the modebar, including `hovermode`, `dragmode`, and `showspikes` at both the root level and inside subplots. Defaults to `layout.uirevision`.
            </p>
          </div>
        </details>
      </div>
      <details>
        <summary>
          orientation
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>angle</dd>
          </dl>
          <p>
            Legacy polar charts are deprecated! Please switch to "polar" subplots. Rotates the entire polar by the given angle in legacy polar charts.
          </p>
        </div>
      </details>
      <details>
        <summary>
          paper_bgcolor
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>color</dd>
            <dt>Default:</dt>
            <dd>"#fff"</dd>
          </dl>
          <p>
            Sets the color of paper where the graph is drawn.
          </p>
        </div>
      </details>
      <details>
        <summary>
          plot_bgcolor
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>color</dd>
            <dt>Default:</dt>
            <dd>"#fff"</dd>
          </dl>
          <p>
            Sets the color of plotting area in-between x and y axes.
          </p>
        </div>
      </details>
      <details>
        <summary>
          polar
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>object containing one or more of the keys listed below.</dd>
            <dt>Default:</dt>
            <dd>[0, 1]</dd>
          </dl>
          <p>
            Sets the horizontal domain of this polar subplot (in plot fraction).
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            angularaxis
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.polar</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>true</dd>
            </dl>
            <p>
              A single toggle to hide the axis while preserving interaction like dragging. Default is true when a cheater plot is present on the axis, otherwise false
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              categoryarray
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>data array</dd>
              </dl>
              <p>
                Sets the order in which categories on this axis appear. Only has an effect if `categoryorder` is set to "array". Used with `categoryorder`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              categoryorder
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "trace" | "category ascending" | "category descending" | "array" | "total ascending" | "total descending" | "min ascending" | "min descending" | "max ascending" | "max descending" | "sum ascending" | "sum descending" | "mean ascending" | "mean descending" | "median ascending" | "median descending" )</dd>
                <dt>Default:</dt>
                <dd>"trace"</dd>
              </dl>
              <p>
                Specifies the ordering logic for the case of categorical variables. By default, plotly uses "trace", which specifies the order that is present in the data supplied. Set `categoryorder` to "category ascending" or "category descending" if order should be determined by the alphanumerical order of the category names. Set `categoryorder` to "array" to derive the ordering from the attribute `categoryarray`. If a category is not found in the `categoryarray` array, the sorting behavior for that attribute will be identical to the "trace" mode. The unspecified categories will follow the categories in `categoryarray`. Set `categoryorder` to "total ascending" or "total descending" if order should be determined by the numerical order of the values. Similarly, the order can be determined by the min, max, sum, mean or median of all the values.
              </p>
            </div>
          </details>
          <details>
            <summary>
              color
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets default for all colors associated with this axis all at once: line, font, tick, and grid colors. Grid color is lightened by blending this with the plot background Individual pieces can override this.
              </p>
            </div>
          </details>
          <details>
            <summary>
              direction
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "counterclockwise" | "clockwise" )</dd>
                <dt>Default:</dt>
                <dd>"counterclockwise"</dd>
              </dl>
              <p>
                Sets the direction corresponding to positive angles.
              </p>
            </div>
          </details>
          <details>
            <summary>
              dtick
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>number or categorical coordinate string</dd>
              </dl>
              <p>
                Sets the step in-between ticks on this axis. Use with `tick0`. Must be a positive number, or special strings available to "log" and "date" axes. If the axis `type` is "log", then ticks are set every 10^(n"dtick) where n is the tick number. For example, to set a tick mark at 1, 10, 100, 1000, ... set dtick to 1. To set tick marks at 1, 100, 10000, ... set dtick to 2. To set tick marks at 1, 5, 25, 125, 625, 3125, ... set dtick to log_10(5), or 0.69897000433. "log" has several special values; "L&lt;f&gt;", where `f` is a positive number, gives ticks linearly spaced in value (but not position). For example `tick0` = 0.1, `dtick` = "L0.5" will put ticks at 0.1, 0.6, 1.1, 1.6 etc. To show powers of 10 plus small digits between, use "D1" (all digits) or "D2" (only 2 and 5). `tick0` is ignored for "D1" and "D2". If the axis `type` is "date", then you must convert the time to milliseconds. For example, to set the interval between ticks to one day, set `dtick` to 86400000.0. "date" also has special values "M&lt;n&gt;" gives ticks spaced by a number of months. `n` must be a positive integer. To set ticks on the 15th of every third month, set `tick0` to "2000-01-15" and `dtick` to "M3". To set ticks every 4 years, set `dtick` to "M48"
              </p>
            </div>
          </details>
          <details>
            <summary>
              exponentformat
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "none" | "e" | "E" | "power" | "SI" | "B" )</dd>
                <dt>Default:</dt>
                <dd>"B"</dd>
              </dl>
              <p>
                Determines a formatting rule for the tick exponents. For example, consider the number 1,000,000,000. If "none", it appears as 1,000,000,000. If "e", 1e+9. If "E", 1E+9. If "power", 1x10^9 (with 9 in a super script). If "SI", 1G. If "B", 1B.
              </p>
            </div>
          </details>
          <details>
            <summary>
              gridcolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#eee"</dd>
              </dl>
              <p>
                Sets the color of the grid lines.
              </p>
            </div>
          </details>
          <details>
            <summary>
              gridwidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the width (in px) of the grid lines.
              </p>
            </div>
          </details>
          <details>
            <summary>
              hoverformat
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets the hover text formatting rule using d3 formatting mini-languages which are very similar to those in Python. For numbers, see: https://github.com/d3/d3-3.x-api-reference/blob/master/Formatting.md#d3_format And for dates see: https://github.com/d3/d3-3.x-api-reference/blob/master/Time-Formatting.md#format We add one item to d3's date formatter: "%{n}f" for fractional seconds with n digits. For example, "2016-10-13 09:15:23.456" with tickformat "%H~%\M~%S.%2f" would display "09~15~23.46"
              </p>
            </div>
          </details>
          <details>
            <summary>
              layer
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "above traces" | "below traces" )</dd>
                <dt>Default:</dt>
                <dd>"above traces"</dd>
              </dl>
              <p>
                Sets the layer on which this axis is displayed. If "above traces", this axis is displayed above all the subplot's traces If "below traces", this axis is displayed below all the subplot's traces, but above the grid lines. Useful when used together with scatter-like traces with `cliponaxis` set to "false" to show markers and/or text nodes above this axis.
              </p>
            </div>
          </details>
          <details>
            <summary>
              linecolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets the axis line color.
              </p>
            </div>
          </details>
          <details>
            <summary>
              linewidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the width (in px) of the axis line.
              </p>
            </div>
          </details>
          <details>
            <summary>
              nticks
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>integer greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                Specifies the maximum number of ticks for the particular axis. The actual number of ticks will be chosen automatically to be less than or equal to `nticks`. Has an effect only if `tickmode` is set to "auto".
              </p>
            </div>
          </details>
          <details>
            <summary>
              period
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
              </dl>
              <p>
                Set the angular period. Has an effect only when `angularaxis.type` is "category".
              </p>
            </div>
          </details>
          <details>
            <summary>
              rotation
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>angle</dd>
              </dl>
              <p>
                Sets that start position (in degrees) of the angular axis By default, polar subplots with `direction` set to "counterclockwise" get a `rotation` of "0" which corresponds to due East (like what mathematicians prefer). In turn, polar with `direction` set to "clockwise" get a rotation of "90" which corresponds to due North (like on a compass),
              </p>
            </div>
          </details>
          <details>
            <summary>
              separatethousands
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
              </dl>
              <p>
                If "true", even 4-digit integers are separated
              </p>
            </div>
          </details>
          <details>
            <summary>
              showexponent
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
                <dt>Default:</dt>
                <dd>"all"</dd>
              </dl>
              <p>
                If "all", all exponents are shown besides their significands. If "first", only the exponent of the first tick is shown. If "last", only the exponent of the last tick is shown. If "none", no exponents appear.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showgrid
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not grid lines are drawn. If "true", the grid lines are drawn at every tick mark.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showline
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not a line bounding this axis is drawn.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showticklabels
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not the tick labels are drawn.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showtickprefix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
                <dt>Default:</dt>
                <dd>"all"</dd>
              </dl>
              <p>
                If "all", all tick labels are displayed with a prefix. If "first", only the first tick is displayed with a prefix. If "last", only the last tick is displayed with a suffix. If "none", tick prefixes are hidden.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showticksuffix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
                <dt>Default:</dt>
                <dd>"all"</dd>
              </dl>
              <p>
                Same as `showtickprefix` but for tick suffixes.
              </p>
            </div>
          </details>
          <details>
            <summary>
              thetaunit
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "radians" | "degrees" )</dd>
                <dt>Default:</dt>
                <dd>"degrees"</dd>
              </dl>
              <p>
                Sets the format unit of the formatted "theta" values. Has an effect only when `angularaxis.type` is "linear".
              </p>
            </div>
          </details>
          <details>
            <summary>
              tick0
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>number or categorical coordinate string</dd>
              </dl>
              <p>
                Sets the placement of the first tick on this axis. Use with `dtick`. If the axis `type` is "log", then you must take the log of your starting tick (e.g. to set the starting tick to 100, set the `tick0` to 2) except when `dtick`="L&lt;f&gt;" (see `dtick` for more info). If the axis `type` is "date", it should be a date string, like date data. If the axis `type` is "category", it should be a number, using the scale where each category is assigned a serial number from zero in the order it appears.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickangle
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>angle</dd>
                <dt>Default:</dt>
                <dd>"auto"</dd>
              </dl>
              <p>
                Sets the angle of the tick labels with respect to the horizontal. For example, a `tickangle` of -90 draws the tick labels vertically.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickcolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets the tick color.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickfont
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
              </dl>
              <p>
                Sets the tick font.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                color
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.polar.angularaxis.tickfont</dd>
                  <dt>Type:</dt>
                  <dd>color</dd>
                </dl>
              </div>
            </details>
            <details>
              <summary>
                family
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.polar.angularaxis.tickfont</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
                </p>
              </div>
            </details>
            <details>
              <summary>
                size
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.polar.angularaxis.tickfont</dd>
                  <dt>Type:</dt>
                  <dd>number greater than or equal to 1</dd>
                </dl>
              </div>
            </details>
          </div>
          <details>
            <summary>
              tickformat
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets the tick label formatting rule using d3 formatting mini-languages which are very similar to those in Python. For numbers, see: https://github.com/d3/d3-3.x-api-reference/blob/master/Formatting.md#d3_format And for dates see: https://github.com/d3/d3-3.x-api-reference/blob/master/Time-Formatting.md#format We add one item to d3's date formatter: "%{n}f" for fractional seconds with n digits. For example, "2016-10-13 09:15:23.456" with tickformat "%H~%\M~%S.%2f" would display "09~15~23.46"
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickformatstops
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>array of object where each object has one or more of the keys listed below.</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not this stop is used. If `false`, this stop is ignored even within its `dtickrange`.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                dtickrange
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.polar.angularaxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>array</dd>
                </dl>
                <p>
                  range ["min", "max"], where "min", "max" - dtick values which describe some zoom level, it is possible to omit "min" or "max" value by passing "null"
                </p>
              </div>
            </details>
            <details>
              <summary>
                enabled
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.polar.angularaxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>boolean</dd>
                  <dt>Default:</dt>
                  <dd>true</dd>
                </dl>
                <p>
                  Determines whether or not this stop is used. If `false`, this stop is ignored even within its `dtickrange`.
                </p>
              </div>
            </details>
            <details>
              <summary>
                name
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.polar.angularaxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  When used in a template, named items are created in the output figure in addition to any items the figure already has in this array. You can modify these items in the output figure by making your own item with `templateitemname` matching this `name` alongside your modifications (including `visible: false` or `enabled: false` to hide it). Has no effect outside of a template.
                </p>
              </div>
            </details>
            <details>
              <summary>
                templateitemname
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.polar.angularaxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  Used to refer to a named item in this array in the template. Named items from the template will be created even without a matching item in the input figure, but you can modify one by making an item with `templateitemname` matching its `name`, alongside your modifications (including `visible: false` or `enabled: false` to hide it). If there is no template or no matching item, this item will be hidden unless you explicitly show it with `visible: true`.
                </p>
              </div>
            </details>
            <details>
              <summary>
                value
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.polar.angularaxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                  <dt>Default:</dt>
                  <dd>""</dd>
                </dl>
                <p>
                  string - dtickformat for described zoom level, the same as "tickformat"
                </p>
              </div>
            </details>
          </div>
          <details>
            <summary>
              ticklen
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>5</dd>
              </dl>
              <p>
                Sets the tick length (in px).
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickmode
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "auto" | "linear" | "array" )</dd>
              </dl>
              <p>
                Sets the tick mode for this axis. If "auto", the number of ticks is set via `nticks`. If "linear", the placement of the ticks is determined by a starting position `tick0` and a tick step `dtick` ("linear" is the default value if `tick0` and `dtick` are provided). If "array", the placement of the ticks is set via `tickvals` and the tick text is `ticktext`. ("array" is the default value if `tickvals` is provided).
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickprefix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets a tick label prefix.
              </p>
            </div>
          </details>
          <details>
            <summary>
              ticks
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "outside" | "inside" | "" )</dd>
              </dl>
              <p>
                Determines whether ticks are drawn or not. If "", this axis' ticks are not drawn. If "outside" ("inside"), this axis' are drawn outside (inside) the axis lines.
              </p>
            </div>
          </details>
          <details>
            <summary>
              ticksuffix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets a tick label suffix.
              </p>
            </div>
          </details>
          <details>
            <summary>
              ticktext
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>data array</dd>
              </dl>
              <p>
                Sets the text displayed at the ticks position via `tickvals`. Only has an effect if `tickmode` is set to "array". Used with `tickvals`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickvals
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>data array</dd>
              </dl>
              <p>
                Sets the values at which ticks on this axis appear. Only has an effect if `tickmode` is set to "array". Used with `ticktext`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickwidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the tick width (in px).
              </p>
            </div>
          </details>
          <details>
            <summary>
              type
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "-" | "linear" | "category" )</dd>
                <dt>Default:</dt>
                <dd>"-"</dd>
              </dl>
              <p>
                Sets the angular axis type. If "linear", set `thetaunit` to determine the unit in which axis value are shown. If "category, use `period` to set the number of integer coordinates around polar axis.
              </p>
            </div>
          </details>
          <details>
            <summary>
              uirevision
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>number or categorical coordinate string</dd>
              </dl>
              <p>
                Controls persistence of user-driven changes in axis `rotation`. Defaults to `polar&lt;N&gt;.uirevision`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              visible
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.angularaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                A single toggle to hide the axis while preserving interaction like dragging. Default is true when a cheater plot is present on the axis, otherwise false
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            bgcolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.polar</dd>
              <dt>Type:</dt>
              <dd>color</dd>
              <dt>Default:</dt>
              <dd>"#fff"</dd>
            </dl>
            <p>
              Set the background color of the subplot
            </p>
          </div>
        </details>
        <details>
          <summary>
            domain
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.polar</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>[0, 1]</dd>
            </dl>
            <p>
              Sets the horizontal domain of this polar subplot (in plot fraction).
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              column
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.domain</dd>
                <dt>Type:</dt>
                <dd>integer greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                If there is a layout grid, use the domain for this column in the grid for this polar subplot .
              </p>
            </div>
          </details>
          <details>
            <summary>
              row
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.domain</dd>
                <dt>Type:</dt>
                <dd>integer greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                If there is a layout grid, use the domain for this row in the grid for this polar subplot .
              </p>
            </div>
          </details>
          <details>
            <summary>
              x
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.domain</dd>
                <dt>Type:</dt>
                <dd>array</dd>
                <dt>Default:</dt>
                <dd>[0, 1]</dd>
              </dl>
              <p>
                Sets the horizontal domain of this polar subplot (in plot fraction).
              </p>
            </div>
          </details>
          <details>
            <summary>
              y
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.domain</dd>
                <dt>Type:</dt>
                <dd>array</dd>
                <dt>Default:</dt>
                <dd>[0, 1]</dd>
              </dl>
              <p>
                Sets the vertical domain of this polar subplot (in plot fraction).
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            gridshape
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.polar</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "circular" | "linear" )</dd>
              <dt>Default:</dt>
              <dd>"circular"</dd>
            </dl>
            <p>
              Determines if the radial axis grid lines and angular axis line are drawn as "circular" sectors or as "linear" (polygon) sectors. Has an effect only when the angular axis has `type` "category". Note that `radialaxis.angle` is snapped to the angle of the closest vertex when `gridshape` is "circular" (so that radial axis scale is the same as the data scale).
            </p>
          </div>
        </details>
        <details>
          <summary>
            hole
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.polar</dd>
              <dt>Type:</dt>
              <dd>number between or equal to 0 and 1</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
            <p>
              Sets the fraction of the radius to cut out of the polar subplot.
            </p>
          </div>
        </details>
        <details>
          <summary>
            radialaxis
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.polar</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>true</dd>
            </dl>
            <p>
              A single toggle to hide the axis while preserving interaction like dragging. Default is true when a cheater plot is present on the axis, otherwise false
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              angle
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>angle</dd>
              </dl>
              <p>
                Sets the angle (in degrees) from which the radial axis is drawn. Note that by default, radial axis line on the theta=0 line corresponds to a line pointing right (like what mathematicians prefer). Defaults to the first `polar.sector` angle.
              </p>
            </div>
          </details>
          <details>
            <summary>
              autorange
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( true | false | "reversed" )</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not the range of this axis is computed in relation to the input data. See `rangemode` for more info. If `range` is provided, then `autorange` is set to "false".
              </p>
            </div>
          </details>
          <details>
            <summary>
              calendar
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "gregorian" | "chinese" | "coptic" | "discworld" | "ethiopian" | "hebrew" | "islamic" | "julian" | "mayan" | "nanakshahi" | "nepali" | "persian" | "jalali" | "taiwan" | "thai" | "ummalqura" )</dd>
                <dt>Default:</dt>
                <dd>"gregorian"</dd>
              </dl>
              <p>
                Sets the calendar system to use for `range` and `tick0` if this is a date axis. This does not set the calendar for interpreting data on this axis, that's specified in the trace or via the global `layout.calendar`
              </p>
            </div>
          </details>
          <details>
            <summary>
              categoryarray
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>data array</dd>
              </dl>
              <p>
                Sets the order in which categories on this axis appear. Only has an effect if `categoryorder` is set to "array". Used with `categoryorder`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              categoryorder
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "trace" | "category ascending" | "category descending" | "array" | "total ascending" | "total descending" | "min ascending" | "min descending" | "max ascending" | "max descending" | "sum ascending" | "sum descending" | "mean ascending" | "mean descending" | "median ascending" | "median descending" )</dd>
                <dt>Default:</dt>
                <dd>"trace"</dd>
              </dl>
              <p>
                Specifies the ordering logic for the case of categorical variables. By default, plotly uses "trace", which specifies the order that is present in the data supplied. Set `categoryorder` to "category ascending" or "category descending" if order should be determined by the alphanumerical order of the category names. Set `categoryorder` to "array" to derive the ordering from the attribute `categoryarray`. If a category is not found in the `categoryarray` array, the sorting behavior for that attribute will be identical to the "trace" mode. The unspecified categories will follow the categories in `categoryarray`. Set `categoryorder` to "total ascending" or "total descending" if order should be determined by the numerical order of the values. Similarly, the order can be determined by the min, max, sum, mean or median of all the values.
              </p>
            </div>
          </details>
          <details>
            <summary>
              color
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets default for all colors associated with this axis all at once: line, font, tick, and grid colors. Grid color is lightened by blending this with the plot background Individual pieces can override this.
              </p>
            </div>
          </details>
          <details>
            <summary>
              dtick
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>number or categorical coordinate string</dd>
              </dl>
              <p>
                Sets the step in-between ticks on this axis. Use with `tick0`. Must be a positive number, or special strings available to "log" and "date" axes. If the axis `type` is "log", then ticks are set every 10^(n"dtick) where n is the tick number. For example, to set a tick mark at 1, 10, 100, 1000, ... set dtick to 1. To set tick marks at 1, 100, 10000, ... set dtick to 2. To set tick marks at 1, 5, 25, 125, 625, 3125, ... set dtick to log_10(5), or 0.69897000433. "log" has several special values; "L&lt;f&gt;", where `f` is a positive number, gives ticks linearly spaced in value (but not position). For example `tick0` = 0.1, `dtick` = "L0.5" will put ticks at 0.1, 0.6, 1.1, 1.6 etc. To show powers of 10 plus small digits between, use "D1" (all digits) or "D2" (only 2 and 5). `tick0` is ignored for "D1" and "D2". If the axis `type` is "date", then you must convert the time to milliseconds. For example, to set the interval between ticks to one day, set `dtick` to 86400000.0. "date" also has special values "M&lt;n&gt;" gives ticks spaced by a number of months. `n` must be a positive integer. To set ticks on the 15th of every third month, set `tick0` to "2000-01-15" and `dtick` to "M3". To set ticks every 4 years, set `dtick` to "M48"
              </p>
            </div>
          </details>
          <details>
            <summary>
              exponentformat
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "none" | "e" | "E" | "power" | "SI" | "B" )</dd>
                <dt>Default:</dt>
                <dd>"B"</dd>
              </dl>
              <p>
                Determines a formatting rule for the tick exponents. For example, consider the number 1,000,000,000. If "none", it appears as 1,000,000,000. If "e", 1e+9. If "E", 1E+9. If "power", 1x10^9 (with 9 in a super script). If "SI", 1G. If "B", 1B.
              </p>
            </div>
          </details>
          <details>
            <summary>
              gridcolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#eee"</dd>
              </dl>
              <p>
                Sets the color of the grid lines.
              </p>
            </div>
          </details>
          <details>
            <summary>
              gridwidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the width (in px) of the grid lines.
              </p>
            </div>
          </details>
          <details>
            <summary>
              hoverformat
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets the hover text formatting rule using d3 formatting mini-languages which are very similar to those in Python. For numbers, see: https://github.com/d3/d3-3.x-api-reference/blob/master/Formatting.md#d3_format And for dates see: https://github.com/d3/d3-3.x-api-reference/blob/master/Time-Formatting.md#format We add one item to d3's date formatter: "%{n}f" for fractional seconds with n digits. For example, "2016-10-13 09:15:23.456" with tickformat "%H~%\M~%S.%2f" would display "09~15~23.46"
              </p>
            </div>
          </details>
          <details>
            <summary>
              layer
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "above traces" | "below traces" )</dd>
                <dt>Default:</dt>
                <dd>"above traces"</dd>
              </dl>
              <p>
                Sets the layer on which this axis is displayed. If "above traces", this axis is displayed above all the subplot's traces If "below traces", this axis is displayed below all the subplot's traces, but above the grid lines. Useful when used together with scatter-like traces with `cliponaxis` set to "false" to show markers and/or text nodes above this axis.
              </p>
            </div>
          </details>
          <details>
            <summary>
              linecolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets the axis line color.
              </p>
            </div>
          </details>
          <details>
            <summary>
              linewidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the width (in px) of the axis line.
              </p>
            </div>
          </details>
          <details>
            <summary>
              nticks
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>integer greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                Specifies the maximum number of ticks for the particular axis. The actual number of ticks will be chosen automatically to be less than or equal to `nticks`. Has an effect only if `tickmode` is set to "auto".
              </p>
            </div>
          </details>
          <details>
            <summary>
              range
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>array</dd>
              </dl>
              <p>
                Sets the range of this axis. If the axis `type` is "log", then you must take the log of your desired range (e.g. to set the range from 1 to 100, set the range from 0 to 2). If the axis `type` is "date", it should be date strings, like date data, though Date objects and unix milliseconds will be accepted and converted to strings. If the axis `type` is "category", it should be numbers, using the scale where each category is assigned a serial number from zero in the order it appears.
              </p>
            </div>
          </details>
          <details>
            <summary>
              rangemode
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "tozero" | "nonnegative" | "normal" )</dd>
                <dt>Default:</dt>
                <dd>"tozero"</dd>
              </dl>
              <p>
                If "tozero"`, the range extends to 0, regardless of the input data If "nonnegative", the range is non-negative, regardless of the input data. If "normal", the range is computed in relation to the extrema of the input data (same behavior as for cartesian axes).
              </p>
            </div>
          </details>
          <details>
            <summary>
              separatethousands
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
              </dl>
              <p>
                If "true", even 4-digit integers are separated
              </p>
            </div>
          </details>
          <details>
            <summary>
              showexponent
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
                <dt>Default:</dt>
                <dd>"all"</dd>
              </dl>
              <p>
                If "all", all exponents are shown besides their significands. If "first", only the exponent of the first tick is shown. If "last", only the exponent of the last tick is shown. If "none", no exponents appear.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showgrid
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not grid lines are drawn. If "true", the grid lines are drawn at every tick mark.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showline
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not a line bounding this axis is drawn.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showticklabels
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not the tick labels are drawn.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showtickprefix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
                <dt>Default:</dt>
                <dd>"all"</dd>
              </dl>
              <p>
                If "all", all tick labels are displayed with a prefix. If "first", only the first tick is displayed with a prefix. If "last", only the last tick is displayed with a suffix. If "none", tick prefixes are hidden.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showticksuffix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
                <dt>Default:</dt>
                <dd>"all"</dd>
              </dl>
              <p>
                Same as `showtickprefix` but for tick suffixes.
              </p>
            </div>
          </details>
          <details>
            <summary>
              side
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "clockwise" | "counterclockwise" )</dd>
                <dt>Default:</dt>
                <dd>"clockwise"</dd>
              </dl>
              <p>
                Determines on which side of radial axis line the tick and tick labels appear.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tick0
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>number or categorical coordinate string</dd>
              </dl>
              <p>
                Sets the placement of the first tick on this axis. Use with `dtick`. If the axis `type` is "log", then you must take the log of your starting tick (e.g. to set the starting tick to 100, set the `tick0` to 2) except when `dtick`="L&lt;f&gt;" (see `dtick` for more info). If the axis `type` is "date", it should be a date string, like date data. If the axis `type` is "category", it should be a number, using the scale where each category is assigned a serial number from zero in the order it appears.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickangle
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>angle</dd>
                <dt>Default:</dt>
                <dd>"auto"</dd>
              </dl>
              <p>
                Sets the angle of the tick labels with respect to the horizontal. For example, a `tickangle` of -90 draws the tick labels vertically.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickcolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets the tick color.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickfont
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
              </dl>
              <p>
                Sets the tick font.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                color
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.polar.radialaxis.tickfont</dd>
                  <dt>Type:</dt>
                  <dd>color</dd>
                </dl>
              </div>
            </details>
            <details>
              <summary>
                family
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.polar.radialaxis.tickfont</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
                </p>
              </div>
            </details>
            <details>
              <summary>
                size
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.polar.radialaxis.tickfont</dd>
                  <dt>Type:</dt>
                  <dd>number greater than or equal to 1</dd>
                </dl>
              </div>
            </details>
          </div>
          <details>
            <summary>
              tickformat
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets the tick label formatting rule using d3 formatting mini-languages which are very similar to those in Python. For numbers, see: https://github.com/d3/d3-3.x-api-reference/blob/master/Formatting.md#d3_format And for dates see: https://github.com/d3/d3-3.x-api-reference/blob/master/Time-Formatting.md#format We add one item to d3's date formatter: "%{n}f" for fractional seconds with n digits. For example, "2016-10-13 09:15:23.456" with tickformat "%H~%\M~%S.%2f" would display "09~15~23.46"
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickformatstops
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>array of object where each object has one or more of the keys listed below.</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not this stop is used. If `false`, this stop is ignored even within its `dtickrange`.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                dtickrange
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.polar.radialaxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>array</dd>
                </dl>
                <p>
                  range ["min", "max"], where "min", "max" - dtick values which describe some zoom level, it is possible to omit "min" or "max" value by passing "null"
                </p>
              </div>
            </details>
            <details>
              <summary>
                enabled
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.polar.radialaxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>boolean</dd>
                  <dt>Default:</dt>
                  <dd>true</dd>
                </dl>
                <p>
                  Determines whether or not this stop is used. If `false`, this stop is ignored even within its `dtickrange`.
                </p>
              </div>
            </details>
            <details>
              <summary>
                name
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.polar.radialaxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  When used in a template, named items are created in the output figure in addition to any items the figure already has in this array. You can modify these items in the output figure by making your own item with `templateitemname` matching this `name` alongside your modifications (including `visible: false` or `enabled: false` to hide it). Has no effect outside of a template.
                </p>
              </div>
            </details>
            <details>
              <summary>
                templateitemname
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.polar.radialaxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  Used to refer to a named item in this array in the template. Named items from the template will be created even without a matching item in the input figure, but you can modify one by making an item with `templateitemname` matching its `name`, alongside your modifications (including `visible: false` or `enabled: false` to hide it). If there is no template or no matching item, this item will be hidden unless you explicitly show it with `visible: true`.
                </p>
              </div>
            </details>
            <details>
              <summary>
                value
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.polar.radialaxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                  <dt>Default:</dt>
                  <dd>""</dd>
                </dl>
                <p>
                  string - dtickformat for described zoom level, the same as "tickformat"
                </p>
              </div>
            </details>
          </div>
          <details>
            <summary>
              ticklen
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>5</dd>
              </dl>
              <p>
                Sets the tick length (in px).
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickmode
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "auto" | "linear" | "array" )</dd>
              </dl>
              <p>
                Sets the tick mode for this axis. If "auto", the number of ticks is set via `nticks`. If "linear", the placement of the ticks is determined by a starting position `tick0` and a tick step `dtick` ("linear" is the default value if `tick0` and `dtick` are provided). If "array", the placement of the ticks is set via `tickvals` and the tick text is `ticktext`. ("array" is the default value if `tickvals` is provided).
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickprefix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets a tick label prefix.
              </p>
            </div>
          </details>
          <details>
            <summary>
              ticks
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "outside" | "inside" | "" )</dd>
              </dl>
              <p>
                Determines whether ticks are drawn or not. If "", this axis' ticks are not drawn. If "outside" ("inside"), this axis' are drawn outside (inside) the axis lines.
              </p>
            </div>
          </details>
          <details>
            <summary>
              ticksuffix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets a tick label suffix.
              </p>
            </div>
          </details>
          <details>
            <summary>
              ticktext
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>data array</dd>
              </dl>
              <p>
                Sets the text displayed at the ticks position via `tickvals`. Only has an effect if `tickmode` is set to "array". Used with `tickvals`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickvals
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>data array</dd>
              </dl>
              <p>
                Sets the values at which ticks on this axis appear. Only has an effect if `tickmode` is set to "array". Used with `ticktext`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickwidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the tick width (in px).
              </p>
            </div>
          </details>
          <details>
            <summary>
              title
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets the title of this axis. Note that before the existence of `title.text`, the title's contents used to be defined as the `title` attribute itself. This behavior has been deprecated.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                font
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.polar.radialaxis.title</dd>
                  <dt>Type:</dt>
                  <dd>object containing one or more of the keys listed below.</dd>
                </dl>
                <p>
                  Sets this axis' title font. Note that the title's font used to be customized by the now deprecated `titlefont` attribute.
                </p>
              </div>
            </details>
            <div style="margin-left: 22px">
              <details>
                <summary>
                  color
                </summary>
                <div style="margin-left: 22px">
                  <dl>
                    <dt>Parent:</dt>
                    <dd>layout.polar.radialaxis.title.font</dd>
                    <dt>Type:</dt>
                    <dd>color</dd>
                  </dl>
                </div>
              </details>
              <details>
                <summary>
                  family
                </summary>
                <div style="margin-left: 22px">
                  <dl>
                    <dt>Parent:</dt>
                    <dd>layout.polar.radialaxis.title.font</dd>
                    <dt>Type:</dt>
                    <dd>string</dd>
                  </dl>
                  <p>
                    HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
                  </p>
                </div>
              </details>
              <details>
                <summary>
                  size
                </summary>
                <div style="margin-left: 22px">
                  <dl>
                    <dt>Parent:</dt>
                    <dd>layout.polar.radialaxis.title.font</dd>
                    <dt>Type:</dt>
                    <dd>number greater than or equal to 1</dd>
                  </dl>
                </div>
              </details>
            </div>
            <details>
              <summary>
                text
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.polar.radialaxis.title</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                  <dt>Default:</dt>
                  <dd>""</dd>
                </dl>
                <p>
                  Sets the title of this axis. Note that before the existence of `title.text`, the title's contents used to be defined as the `title` attribute itself. This behavior has been deprecated.
                </p>
              </div>
            </details>
          </div>
          <details>
            <summary>
              type
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "-" | "linear" | "log" | "date" | "category" )</dd>
                <dt>Default:</dt>
                <dd>"-"</dd>
              </dl>
              <p>
                Sets the axis type. By default, plotly attempts to determined the axis type by looking into the data of the traces that referenced the axis in question.
              </p>
            </div>
          </details>
          <details>
            <summary>
              uirevision
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>number or categorical coordinate string</dd>
              </dl>
              <p>
                Controls persistence of user-driven changes in axis `range`, `autorange`, `angle`, and `title` if in `editable: true` configuration. Defaults to `polar&lt;N&gt;.uirevision`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              visible
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.polar.radialaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                A single toggle to hide the axis while preserving interaction like dragging. Default is true when a cheater plot is present on the axis, otherwise false
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            sector
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.polar</dd>
              <dt>Type:</dt>
              <dd>array</dd>
              <dt>Default:</dt>
              <dd>[0, 360]</dd>
            </dl>
            <p>
              Sets angular span of this polar subplot with two angles (in degrees). Sector are assumed to be spanned in the counterclockwise direction with "0" corresponding to rightmost limit of the polar subplot.
            </p>
          </div>
        </details>
        <details>
          <summary>
            uirevision
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.polar</dd>
              <dt>Type:</dt>
              <dd>number or categorical coordinate string</dd>
            </dl>
            <p>
              Controls persistence of user-driven changes in axis attributes, if not overridden in the individual axes. Defaults to `layout.uirevision`.
            </p>
          </div>
        </details>
      </div>
      <details>
        <summary>
          radialaxis
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>object containing one or more of the keys listed below.</dd>
            <dt>Default:</dt>
            <dd>[0, 1]</dd>
          </dl>
          <p>
            Legacy polar charts are deprecated! Please switch to "polar" subplots. Defines the start and end point of this radial axis.
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            domain
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.radialaxis</dd>
              <dt>Type:</dt>
              <dd>array</dd>
              <dt>Default:</dt>
              <dd>[0, 1]</dd>
            </dl>
            <p>
              Polar chart subplots are not supported yet. This key has currently no effect.
            </p>
          </div>
        </details>
        <details>
          <summary>
            endpadding
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.radialaxis</dd>
              <dt>Type:</dt>
              <dd>number</dd>
            </dl>
            <p>
              Legacy polar charts are deprecated! Please switch to "polar" subplots.
            </p>
          </div>
        </details>
        <details>
          <summary>
            orientation
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.radialaxis</dd>
              <dt>Type:</dt>
              <dd>number</dd>
            </dl>
            <p>
              Legacy polar charts are deprecated! Please switch to "polar" subplots. Sets the orientation (an angle with respect to the origin) of the radial axis.
            </p>
          </div>
        </details>
        <details>
          <summary>
            range
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.radialaxis</dd>
              <dt>Type:</dt>
              <dd>array</dd>
            </dl>
            <p>
              Legacy polar charts are deprecated! Please switch to "polar" subplots. Defines the start and end point of this radial axis.
            </p>
          </div>
        </details>
        <details>
          <summary>
            showline
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.radialaxis</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Legacy polar charts are deprecated! Please switch to "polar" subplots. Determines whether or not the line bounding this radial axis will be shown on the figure.
            </p>
          </div>
        </details>
        <details>
          <summary>
            showticklabels
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.radialaxis</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Legacy polar charts are deprecated! Please switch to "polar" subplots. Determines whether or not the radial axis ticks will feature tick labels.
            </p>
          </div>
        </details>
        <details>
          <summary>
            tickcolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.radialaxis</dd>
              <dt>Type:</dt>
              <dd>color</dd>
            </dl>
            <p>
              Legacy polar charts are deprecated! Please switch to "polar" subplots. Sets the color of the tick lines on this radial axis.
            </p>
          </div>
        </details>
        <details>
          <summary>
            ticklen
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.radialaxis</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
            </dl>
            <p>
              Legacy polar charts are deprecated! Please switch to "polar" subplots. Sets the length of the tick lines on this radial axis.
            </p>
          </div>
        </details>
        <details>
          <summary>
            tickorientation
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.radialaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "horizontal" | "vertical" )</dd>
            </dl>
            <p>
              Legacy polar charts are deprecated! Please switch to "polar" subplots. Sets the orientation (from the paper perspective) of the radial axis tick labels.
            </p>
          </div>
        </details>
        <details>
          <summary>
            ticksuffix
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.radialaxis</dd>
              <dt>Type:</dt>
              <dd>string</dd>
            </dl>
            <p>
              Legacy polar charts are deprecated! Please switch to "polar" subplots. Sets the length of the tick lines on this radial axis.
            </p>
          </div>
        </details>
        <details>
          <summary>
            visible
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.radialaxis</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Legacy polar charts are deprecated! Please switch to "polar" subplots. Determines whether or not this axis will be visible.
            </p>
          </div>
        </details>
      </div>
      <details>
        <summary>
          scene
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>object containing one or more of the keys listed below.</dd>
            <dt>Default:</dt>
            <dd>"rgba(0,0,0,0)"</dd>
          </dl>
          <p>
            Sets the (x,y,z) components of the 'up' camera vector. This vector determines the up direction of this scene with respect to the page. The default is "{x: 0, y: 0, z: 1}" which means that the z axis points up.
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            annotations
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.scene</dd>
              <dt>Type:</dt>
              <dd>array of object where each object has one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>true</dd>
            </dl>
            <p>
              Determines whether or not this annotation is visible.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              align
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.annotations[]</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "left" | "center" | "right" )</dd>
                <dt>Default:</dt>
                <dd>"center"</dd>
              </dl>
              <p>
                Sets the horizontal alignment of the `text` within the box. Has an effect only if `text` spans more two or more lines (i.e. `text` contains one or more &lt;br&gt; HTML tags) or if an explicit width is set to override the text width.
              </p>
            </div>
          </details>
          <details>
            <summary>
              arrowcolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.annotations[]</dd>
                <dt>Type:</dt>
                <dd>color</dd>
              </dl>
              <p>
                Sets the color of the annotation arrow.
              </p>
            </div>
          </details>
          <details>
            <summary>
              arrowhead
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.annotations[]</dd>
                <dt>Type:</dt>
                <dd>integer between or equal to 0 and 8</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the end annotation arrow head style.
              </p>
            </div>
          </details>
          <details>
            <summary>
              arrowside
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.annotations[]</dd>
                <dt>Type:</dt>
                <dd>flaglist string. Any combination of "end", "start" joined with a "+" OR "none".</dd>
                <dt>Default:</dt>
                <dd>"end"</dd>
              </dl>
              <p>
                Sets the annotation arrow head position.
              </p>
            </div>
          </details>
          <details>
            <summary>
              arrowsize
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.annotations[]</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0.3</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the size of the end annotation arrow head, relative to `arrowwidth`. A value of 1 (default) gives a head about 3x as wide as the line.
              </p>
            </div>
          </details>
          <details>
            <summary>
              arrowwidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.annotations[]</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0.1</dd>
              </dl>
              <p>
                Sets the width (in px) of annotation arrow line.
              </p>
            </div>
          </details>
          <details>
            <summary>
              ax
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.annotations[]</dd>
                <dt>Type:</dt>
                <dd>number</dd>
              </dl>
              <p>
                Sets the x component of the arrow tail about the arrow head (in pixels).
              </p>
            </div>
          </details>
          <details>
            <summary>
              ay
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.annotations[]</dd>
                <dt>Type:</dt>
                <dd>number</dd>
              </dl>
              <p>
                Sets the y component of the arrow tail about the arrow head (in pixels).
              </p>
            </div>
          </details>
          <details>
            <summary>
              bgcolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.annotations[]</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"rgba(0,0,0,0)"</dd>
              </dl>
              <p>
                Sets the background color of the annotation.
              </p>
            </div>
          </details>
          <details>
            <summary>
              bordercolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.annotations[]</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"rgba(0,0,0,0)"</dd>
              </dl>
              <p>
                Sets the color of the border enclosing the annotation `text`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              borderpad
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.annotations[]</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the padding (in px) between the `text` and the enclosing border.
              </p>
            </div>
          </details>
          <details>
            <summary>
              borderwidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.annotations[]</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the width (in px) of the border enclosing the annotation `text`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              captureevents
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.annotations[]</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
              </dl>
              <p>
                Determines whether the annotation text box captures mouse move and click events, or allows those events to pass through to data points in the plot that may be behind the annotation. By default `captureevents` is "false" unless `hovertext` is provided. If you use the event `plotly_clickannotation` without `hovertext` you must explicitly enable `captureevents`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              font
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.annotations[]</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
              </dl>
              <p>
                Sets the annotation text font.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                color
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.annotations[].font</dd>
                  <dt>Type:</dt>
                  <dd>color</dd>
                </dl>
              </div>
            </details>
            <details>
              <summary>
                family
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.annotations[].font</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
                </p>
              </div>
            </details>
            <details>
              <summary>
                size
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.annotations[].font</dd>
                  <dt>Type:</dt>
                  <dd>number greater than or equal to 1</dd>
                </dl>
              </div>
            </details>
          </div>
          <details>
            <summary>
              height
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.annotations[]</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 1</dd>
              </dl>
              <p>
                Sets an explicit height for the text box. null (default) lets the text set the box height. Taller text will be clipped.
              </p>
            </div>
          </details>
          <details>
            <summary>
              hoverlabel
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.annotations[]</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
              </dl>
              <p>
                Sets the background color of the hover label. By default uses the annotation's `bgcolor` made opaque, or white if it was transparent.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                bgcolor
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.annotations[].hoverlabel</dd>
                  <dt>Type:</dt>
                  <dd>color</dd>
                </dl>
                <p>
                  Sets the background color of the hover label. By default uses the annotation's `bgcolor` made opaque, or white if it was transparent.
                </p>
              </div>
            </details>
            <details>
              <summary>
                bordercolor
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.annotations[].hoverlabel</dd>
                  <dt>Type:</dt>
                  <dd>color</dd>
                </dl>
                <p>
                  Sets the border color of the hover label. By default uses either dark grey or white, for maximum contrast with `hoverlabel.bgcolor`.
                </p>
              </div>
            </details>
            <details>
              <summary>
                font
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.annotations[].hoverlabel</dd>
                  <dt>Type:</dt>
                  <dd>object containing one or more of the keys listed below.</dd>
                </dl>
                <p>
                  Sets the hover label text font. By default uses the global hover font and size, with color from `hoverlabel.bordercolor`.
                </p>
              </div>
            </details>
            <div style="margin-left: 22px">
              <details>
                <summary>
                  color
                </summary>
                <div style="margin-left: 22px">
                  <dl>
                    <dt>Parent:</dt>
                    <dd>layout.scene.annotations[].hoverlabel.font</dd>
                    <dt>Type:</dt>
                    <dd>color</dd>
                  </dl>
                </div>
              </details>
              <details>
                <summary>
                  family
                </summary>
                <div style="margin-left: 22px">
                  <dl>
                    <dt>Parent:</dt>
                    <dd>layout.scene.annotations[].hoverlabel.font</dd>
                    <dt>Type:</dt>
                    <dd>string</dd>
                  </dl>
                  <p>
                    HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
                  </p>
                </div>
              </details>
              <details>
                <summary>
                  size
                </summary>
                <div style="margin-left: 22px">
                  <dl>
                    <dt>Parent:</dt>
                    <dd>layout.scene.annotations[].hoverlabel.font</dd>
                    <dt>Type:</dt>
                    <dd>number greater than or equal to 1</dd>
                  </dl>
                </div>
              </details>
            </div>
          </div>
          <details>
            <summary>
              hovertext
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.annotations[]</dd>
                <dt>Type:</dt>
                <dd>string</dd>
              </dl>
              <p>
                Sets text to appear when hovering over this annotation. If omitted or blank, no hover label will appear.
              </p>
            </div>
          </details>
          <details>
            <summary>
              name
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.annotations[]</dd>
                <dt>Type:</dt>
                <dd>string</dd>
              </dl>
              <p>
                When used in a template, named items are created in the output figure in addition to any items the figure already has in this array. You can modify these items in the output figure by making your own item with `templateitemname` matching this `name` alongside your modifications (including `visible: false` or `enabled: false` to hide it). Has no effect outside of a template.
              </p>
            </div>
          </details>
          <details>
            <summary>
              opacity
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.annotations[]</dd>
                <dt>Type:</dt>
                <dd>number between or equal to 0 and 1</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the opacity of the annotation (text + arrow).
              </p>
            </div>
          </details>
          <details>
            <summary>
              showarrow
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.annotations[]</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not the annotation is drawn with an arrow. If "true", `text` is placed near the arrow's tail. If "false", `text` lines up with the `x` and `y` provided.
              </p>
            </div>
          </details>
          <details>
            <summary>
              standoff
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.annotations[]</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                Sets a distance, in pixels, to move the end arrowhead away from the position it is pointing at, for example to point at the edge of a marker independent of zoom. Note that this shortens the arrow from the `ax` / `ay` vector, in contrast to `xshift` / `yshift` which moves everything by this amount.
              </p>
            </div>
          </details>
          <details>
            <summary>
              startarrowhead
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.annotations[]</dd>
                <dt>Type:</dt>
                <dd>integer between or equal to 0 and 8</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the start annotation arrow head style.
              </p>
            </div>
          </details>
          <details>
            <summary>
              startarrowsize
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.annotations[]</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0.3</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the size of the start annotation arrow head, relative to `arrowwidth`. A value of 1 (default) gives a head about 3x as wide as the line.
              </p>
            </div>
          </details>
          <details>
            <summary>
              startstandoff
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.annotations[]</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                Sets a distance, in pixels, to move the start arrowhead away from the position it is pointing at, for example to point at the edge of a marker independent of zoom. Note that this shortens the arrow from the `ax` / `ay` vector, in contrast to `xshift` / `yshift` which moves everything by this amount.
              </p>
            </div>
          </details>
          <details>
            <summary>
              templateitemname
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.annotations[]</dd>
                <dt>Type:</dt>
                <dd>string</dd>
              </dl>
              <p>
                Used to refer to a named item in this array in the template. Named items from the template will be created even without a matching item in the input figure, but you can modify one by making an item with `templateitemname` matching its `name`, alongside your modifications (including `visible: false` or `enabled: false` to hide it). If there is no template or no matching item, this item will be hidden unless you explicitly show it with `visible: true`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              text
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.annotations[]</dd>
                <dt>Type:</dt>
                <dd>string</dd>
              </dl>
              <p>
                Sets the text associated with this annotation. Plotly uses a subset of HTML tags to do things like newline (&lt;br&gt;), bold (&lt;b&gt;&lt;/b&gt;), italics (&lt;i&gt;&lt;/i&gt;), hyperlinks (&lt;a href='...'&gt;&lt;/a&gt;). Tags &lt;em&gt;, &lt;sup&gt;, &lt;sub&gt; &lt;span&gt; are also supported.
              </p>
            </div>
          </details>
          <details>
            <summary>
              textangle
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.annotations[]</dd>
                <dt>Type:</dt>
                <dd>angle</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                Sets the angle at which the `text` is drawn with respect to the horizontal.
              </p>
            </div>
          </details>
          <details>
            <summary>
              valign
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.annotations[]</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "top" | "middle" | "bottom" )</dd>
                <dt>Default:</dt>
                <dd>"middle"</dd>
              </dl>
              <p>
                Sets the vertical alignment of the `text` within the box. Has an effect only if an explicit height is set to override the text height.
              </p>
            </div>
          </details>
          <details>
            <summary>
              visible
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.annotations[]</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not this annotation is visible.
              </p>
            </div>
          </details>
          <details>
            <summary>
              width
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.annotations[]</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 1</dd>
              </dl>
              <p>
                Sets an explicit width for the text box. null (default) lets the text set the box width. Wider text will be clipped. There is no automatic wrapping; use &lt;br&gt; to start a new line.
              </p>
            </div>
          </details>
          <details>
            <summary>
              x
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.annotations[]</dd>
                <dt>Type:</dt>
                <dd>number or categorical coordinate string</dd>
              </dl>
              <p>
                Sets the annotation's x position.
              </p>
            </div>
          </details>
          <details>
            <summary>
              xanchor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.annotations[]</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "auto" | "left" | "center" | "right" )</dd>
                <dt>Default:</dt>
                <dd>"auto"</dd>
              </dl>
              <p>
                Sets the text box's horizontal position anchor This anchor binds the `x` position to the "left", "center" or "right" of the annotation. For example, if `x` is set to 1, `xref` to "paper" and `xanchor` to "right" then the right-most portion of the annotation lines up with the right-most edge of the plotting area. If "auto", the anchor is equivalent to "center" for data-referenced annotations or if there is an arrow, whereas for paper-referenced with no arrow, the anchor picked corresponds to the closest side.
              </p>
            </div>
          </details>
          <details>
            <summary>
              xshift
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.annotations[]</dd>
                <dt>Type:</dt>
                <dd>number</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                Shifts the position of the whole annotation and arrow to the right (positive) or left (negative) by this many pixels.
              </p>
            </div>
          </details>
          <details>
            <summary>
              y
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.annotations[]</dd>
                <dt>Type:</dt>
                <dd>number or categorical coordinate string</dd>
              </dl>
              <p>
                Sets the annotation's y position.
              </p>
            </div>
          </details>
          <details>
            <summary>
              yanchor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.annotations[]</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "auto" | "top" | "middle" | "bottom" )</dd>
                <dt>Default:</dt>
                <dd>"auto"</dd>
              </dl>
              <p>
                Sets the text box's vertical position anchor This anchor binds the `y` position to the "top", "middle" or "bottom" of the annotation. For example, if `y` is set to 1, `yref` to "paper" and `yanchor` to "top" then the top-most portion of the annotation lines up with the top-most edge of the plotting area. If "auto", the anchor is equivalent to "middle" for data-referenced annotations or if there is an arrow, whereas for paper-referenced with no arrow, the anchor picked corresponds to the closest side.
              </p>
            </div>
          </details>
          <details>
            <summary>
              yshift
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.annotations[]</dd>
                <dt>Type:</dt>
                <dd>number</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                Shifts the position of the whole annotation and arrow up (positive) or down (negative) by this many pixels.
              </p>
            </div>
          </details>
          <details>
            <summary>
              z
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.annotations[]</dd>
                <dt>Type:</dt>
                <dd>number or categorical coordinate string</dd>
              </dl>
              <p>
                Sets the annotation's z position.
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            aspectmode
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.scene</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "auto" | "cube" | "data" | "manual" )</dd>
              <dt>Default:</dt>
              <dd>"auto"</dd>
            </dl>
            <p>
              If "cube", this scene's axes are drawn as a cube, regardless of the axes' ranges. If "data", this scene's axes are drawn in proportion with the axes' ranges. If "manual", this scene's axes are drawn in proportion with the input of "aspectratio" (the default behavior if "aspectratio" is provided). If "auto", this scene's axes are drawn using the results of "data" except when one axis is more than four times the size of the two others, where in that case the results of "cube" are used.
            </p>
          </div>
        </details>
        <details>
          <summary>
            aspectratio
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.scene</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
            </dl>
            <p>
              Sets this scene's axis aspectratio.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              x
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.aspectratio</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
              </dl>
            </div>
          </details>
          <details>
            <summary>
              y
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.aspectratio</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
              </dl>
            </div>
          </details>
          <details>
            <summary>
              z
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.aspectratio</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
              </dl>
            </div>
          </details>
        </div>
        <details>
          <summary>
            bgcolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.scene</dd>
              <dt>Type:</dt>
              <dd>color</dd>
              <dt>Default:</dt>
              <dd>"rgba(0,0,0,0)"</dd>
            </dl>
          </div>
        </details>
        <details>
          <summary>
            camera
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.scene</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
            <p>
              Sets the (x,y,z) components of the 'up' camera vector. This vector determines the up direction of this scene with respect to the page. The default is "{x: 0, y: 0, z: 1}" which means that the z axis points up.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              center
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.camera</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                Sets the (x,y,z) components of the 'center' camera vector This vector determines the translation (x,y,z) space about the center of this scene. By default, there is no such translation.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                x
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.camera.center</dd>
                  <dt>Type:</dt>
                  <dd>number</dd>
                  <dt>Default:</dt>
                  <dd>0</dd>
                </dl>
              </div>
            </details>
            <details>
              <summary>
                y
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.camera.center</dd>
                  <dt>Type:</dt>
                  <dd>number</dd>
                  <dt>Default:</dt>
                  <dd>0</dd>
                </dl>
              </div>
            </details>
            <details>
              <summary>
                z
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.camera.center</dd>
                  <dt>Type:</dt>
                  <dd>number</dd>
                  <dt>Default:</dt>
                  <dd>0</dd>
                </dl>
              </div>
            </details>
          </div>
          <details>
            <summary>
              eye
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.camera</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
                <dt>Default:</dt>
                <dd>1.25</dd>
              </dl>
              <p>
                Sets the (x,y,z) components of the 'eye' camera vector. This vector determines the view point about the origin of this scene.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                x
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.camera.eye</dd>
                  <dt>Type:</dt>
                  <dd>number</dd>
                  <dt>Default:</dt>
                  <dd>1.25</dd>
                </dl>
              </div>
            </details>
            <details>
              <summary>
                y
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.camera.eye</dd>
                  <dt>Type:</dt>
                  <dd>number</dd>
                  <dt>Default:</dt>
                  <dd>1.25</dd>
                </dl>
              </div>
            </details>
            <details>
              <summary>
                z
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.camera.eye</dd>
                  <dt>Type:</dt>
                  <dd>number</dd>
                  <dt>Default:</dt>
                  <dd>1.25</dd>
                </dl>
              </div>
            </details>
          </div>
          <details>
            <summary>
              projection
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.camera</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
                <dt>Default:</dt>
                <dd>"perspective"</dd>
              </dl>
              <p>
                Sets the projection type. The projection type could be either "perspective" or "orthographic". The default is "perspective".
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                type
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.camera.projection</dd>
                  <dt>Type:</dt>
                  <dd>enumerated , one of ( "perspective" | "orthographic" )</dd>
                  <dt>Default:</dt>
                  <dd>"perspective"</dd>
                </dl>
                <p>
                  Sets the projection type. The projection type could be either "perspective" or "orthographic". The default is "perspective".
                </p>
              </div>
            </details>
          </div>
          <details>
            <summary>
              up
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.camera</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                Sets the (x,y,z) components of the 'up' camera vector. This vector determines the up direction of this scene with respect to the page. The default is "{x: 0, y: 0, z: 1}" which means that the z axis points up.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                x
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.camera.up</dd>
                  <dt>Type:</dt>
                  <dd>number</dd>
                  <dt>Default:</dt>
                  <dd>0</dd>
                </dl>
              </div>
            </details>
            <details>
              <summary>
                y
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.camera.up</dd>
                  <dt>Type:</dt>
                  <dd>number</dd>
                  <dt>Default:</dt>
                  <dd>0</dd>
                </dl>
              </div>
            </details>
            <details>
              <summary>
                z
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.camera.up</dd>
                  <dt>Type:</dt>
                  <dd>number</dd>
                  <dt>Default:</dt>
                  <dd>1</dd>
                </dl>
              </div>
            </details>
          </div>
        </div>
        <details>
          <summary>
            domain
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.scene</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>[0, 1]</dd>
            </dl>
            <p>
              Sets the horizontal domain of this scene subplot (in plot fraction).
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              column
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.domain</dd>
                <dt>Type:</dt>
                <dd>integer greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                If there is a layout grid, use the domain for this column in the grid for this scene subplot .
              </p>
            </div>
          </details>
          <details>
            <summary>
              row
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.domain</dd>
                <dt>Type:</dt>
                <dd>integer greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                If there is a layout grid, use the domain for this row in the grid for this scene subplot .
              </p>
            </div>
          </details>
          <details>
            <summary>
              x
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.domain</dd>
                <dt>Type:</dt>
                <dd>array</dd>
                <dt>Default:</dt>
                <dd>[0, 1]</dd>
              </dl>
              <p>
                Sets the horizontal domain of this scene subplot (in plot fraction).
              </p>
            </div>
          </details>
          <details>
            <summary>
              y
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.domain</dd>
                <dt>Type:</dt>
                <dd>array</dd>
                <dt>Default:</dt>
                <dd>[0, 1]</dd>
              </dl>
              <p>
                Sets the vertical domain of this scene subplot (in plot fraction).
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            dragmode
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.scene</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "orbit" | "turntable" | "zoom" | "pan" | false )</dd>
            </dl>
            <p>
              Determines the mode of drag interactions for this scene.
            </p>
          </div>
        </details>
        <details>
          <summary>
            hovermode
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.scene</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "closest" | false )</dd>
              <dt>Default:</dt>
              <dd>"closest"</dd>
            </dl>
            <p>
              Determines the mode of hover interactions for this scene.
            </p>
          </div>
        </details>
        <details>
          <summary>
            uirevision
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.scene</dd>
              <dt>Type:</dt>
              <dd>number or categorical coordinate string</dd>
            </dl>
            <p>
              Controls persistence of user-driven changes in camera attributes. Defaults to `layout.uirevision`.
            </p>
          </div>
        </details>
        <details>
          <summary>
            xaxis
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.scene</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>true</dd>
            </dl>
            <p>
              A single toggle to hide the axis while preserving interaction like dragging. Default is true when a cheater plot is present on the axis, otherwise false
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              autorange
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( true | false | "reversed" )</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not the range of this axis is computed in relation to the input data. See `rangemode` for more info. If `range` is provided, then `autorange` is set to "false".
              </p>
            </div>
          </details>
          <details>
            <summary>
              backgroundcolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"rgba(204, 204, 204, 0.5)"</dd>
              </dl>
              <p>
                Sets the background color of this axis' wall.
              </p>
            </div>
          </details>
          <details>
            <summary>
              calendar
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "gregorian" | "chinese" | "coptic" | "discworld" | "ethiopian" | "hebrew" | "islamic" | "julian" | "mayan" | "nanakshahi" | "nepali" | "persian" | "jalali" | "taiwan" | "thai" | "ummalqura" )</dd>
                <dt>Default:</dt>
                <dd>"gregorian"</dd>
              </dl>
              <p>
                Sets the calendar system to use for `range` and `tick0` if this is a date axis. This does not set the calendar for interpreting data on this axis, that's specified in the trace or via the global `layout.calendar`
              </p>
            </div>
          </details>
          <details>
            <summary>
              categoryarray
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>data array</dd>
              </dl>
              <p>
                Sets the order in which categories on this axis appear. Only has an effect if `categoryorder` is set to "array". Used with `categoryorder`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              categoryorder
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "trace" | "category ascending" | "category descending" | "array" | "total ascending" | "total descending" | "min ascending" | "min descending" | "max ascending" | "max descending" | "sum ascending" | "sum descending" | "mean ascending" | "mean descending" | "median ascending" | "median descending" )</dd>
                <dt>Default:</dt>
                <dd>"trace"</dd>
              </dl>
              <p>
                Specifies the ordering logic for the case of categorical variables. By default, plotly uses "trace", which specifies the order that is present in the data supplied. Set `categoryorder` to "category ascending" or "category descending" if order should be determined by the alphanumerical order of the category names. Set `categoryorder` to "array" to derive the ordering from the attribute `categoryarray`. If a category is not found in the `categoryarray` array, the sorting behavior for that attribute will be identical to the "trace" mode. The unspecified categories will follow the categories in `categoryarray`. Set `categoryorder` to "total ascending" or "total descending" if order should be determined by the numerical order of the values. Similarly, the order can be determined by the min, max, sum, mean or median of all the values.
              </p>
            </div>
          </details>
          <details>
            <summary>
              color
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets default for all colors associated with this axis all at once: line, font, tick, and grid colors. Grid color is lightened by blending this with the plot background Individual pieces can override this.
              </p>
            </div>
          </details>
          <details>
            <summary>
              dtick
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>number or categorical coordinate string</dd>
              </dl>
              <p>
                Sets the step in-between ticks on this axis. Use with `tick0`. Must be a positive number, or special strings available to "log" and "date" axes. If the axis `type` is "log", then ticks are set every 10^(n"dtick) where n is the tick number. For example, to set a tick mark at 1, 10, 100, 1000, ... set dtick to 1. To set tick marks at 1, 100, 10000, ... set dtick to 2. To set tick marks at 1, 5, 25, 125, 625, 3125, ... set dtick to log_10(5), or 0.69897000433. "log" has several special values; "L&lt;f&gt;", where `f` is a positive number, gives ticks linearly spaced in value (but not position). For example `tick0` = 0.1, `dtick` = "L0.5" will put ticks at 0.1, 0.6, 1.1, 1.6 etc. To show powers of 10 plus small digits between, use "D1" (all digits) or "D2" (only 2 and 5). `tick0` is ignored for "D1" and "D2". If the axis `type` is "date", then you must convert the time to milliseconds. For example, to set the interval between ticks to one day, set `dtick` to 86400000.0. "date" also has special values "M&lt;n&gt;" gives ticks spaced by a number of months. `n` must be a positive integer. To set ticks on the 15th of every third month, set `tick0` to "2000-01-15" and `dtick` to "M3". To set ticks every 4 years, set `dtick` to "M48"
              </p>
            </div>
          </details>
          <details>
            <summary>
              exponentformat
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "none" | "e" | "E" | "power" | "SI" | "B" )</dd>
                <dt>Default:</dt>
                <dd>"B"</dd>
              </dl>
              <p>
                Determines a formatting rule for the tick exponents. For example, consider the number 1,000,000,000. If "none", it appears as 1,000,000,000. If "e", 1e+9. If "E", 1E+9. If "power", 1x10^9 (with 9 in a super script). If "SI", 1G. If "B", 1B.
              </p>
            </div>
          </details>
          <details>
            <summary>
              gridcolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"rgb(204, 204, 204)"</dd>
              </dl>
              <p>
                Sets the color of the grid lines.
              </p>
            </div>
          </details>
          <details>
            <summary>
              gridwidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the width (in px) of the grid lines.
              </p>
            </div>
          </details>
          <details>
            <summary>
              hoverformat
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets the hover text formatting rule using d3 formatting mini-languages which are very similar to those in Python. For numbers, see: https://github.com/d3/d3-3.x-api-reference/blob/master/Formatting.md#d3_format And for dates see: https://github.com/d3/d3-3.x-api-reference/blob/master/Time-Formatting.md#format We add one item to d3's date formatter: "%{n}f" for fractional seconds with n digits. For example, "2016-10-13 09:15:23.456" with tickformat "%H~%\M~%S.%2f" would display "09~15~23.46"
              </p>
            </div>
          </details>
          <details>
            <summary>
              linecolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets the axis line color.
              </p>
            </div>
          </details>
          <details>
            <summary>
              linewidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the width (in px) of the axis line.
              </p>
            </div>
          </details>
          <details>
            <summary>
              mirror
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( true | "ticks" | false | "all" | "allticks" )</dd>
              </dl>
              <p>
                Determines if the axis lines or/and ticks are mirrored to the opposite side of the plotting area. If "true", the axis lines are mirrored. If "ticks", the axis lines and ticks are mirrored. If "false", mirroring is disable. If "all", axis lines are mirrored on all shared-axes subplots. If "allticks", axis lines and ticks are mirrored on all shared-axes subplots.
              </p>
            </div>
          </details>
          <details>
            <summary>
              nticks
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>integer greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                Specifies the maximum number of ticks for the particular axis. The actual number of ticks will be chosen automatically to be less than or equal to `nticks`. Has an effect only if `tickmode` is set to "auto".
              </p>
            </div>
          </details>
          <details>
            <summary>
              range
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>array</dd>
              </dl>
              <p>
                Sets the range of this axis. If the axis `type` is "log", then you must take the log of your desired range (e.g. to set the range from 1 to 100, set the range from 0 to 2). If the axis `type` is "date", it should be date strings, like date data, though Date objects and unix milliseconds will be accepted and converted to strings. If the axis `type` is "category", it should be numbers, using the scale where each category is assigned a serial number from zero in the order it appears.
              </p>
            </div>
          </details>
          <details>
            <summary>
              rangemode
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "normal" | "tozero" | "nonnegative" )</dd>
                <dt>Default:</dt>
                <dd>"normal"</dd>
              </dl>
              <p>
                If "normal", the range is computed in relation to the extrema of the input data. If "tozero"`, the range extends to 0, regardless of the input data If "nonnegative", the range is non-negative, regardless of the input data. Applies only to linear axes.
              </p>
            </div>
          </details>
          <details>
            <summary>
              separatethousands
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
              </dl>
              <p>
                If "true", even 4-digit integers are separated
              </p>
            </div>
          </details>
          <details>
            <summary>
              showaxeslabels
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Sets whether or not this axis is labeled
              </p>
            </div>
          </details>
          <details>
            <summary>
              showbackground
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
              </dl>
              <p>
                Sets whether or not this axis' wall has a background color.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showexponent
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
                <dt>Default:</dt>
                <dd>"all"</dd>
              </dl>
              <p>
                If "all", all exponents are shown besides their significands. If "first", only the exponent of the first tick is shown. If "last", only the exponent of the last tick is shown. If "none", no exponents appear.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showgrid
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
              </dl>
              <p>
                Determines whether or not grid lines are drawn. If "true", the grid lines are drawn at every tick mark.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showline
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
              </dl>
              <p>
                Determines whether or not a line bounding this axis is drawn.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showspikes
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Sets whether or not spikes starting from data points to this axis' wall are shown on hover.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showticklabels
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not the tick labels are drawn.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showtickprefix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
                <dt>Default:</dt>
                <dd>"all"</dd>
              </dl>
              <p>
                If "all", all tick labels are displayed with a prefix. If "first", only the first tick is displayed with a prefix. If "last", only the last tick is displayed with a suffix. If "none", tick prefixes are hidden.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showticksuffix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
                <dt>Default:</dt>
                <dd>"all"</dd>
              </dl>
              <p>
                Same as `showtickprefix` but for tick suffixes.
              </p>
            </div>
          </details>
          <details>
            <summary>
              spikecolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets the color of the spikes.
              </p>
            </div>
          </details>
          <details>
            <summary>
              spikesides
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Sets whether or not spikes extending from the projection data points to this axis' wall boundaries are shown on hover.
              </p>
            </div>
          </details>
          <details>
            <summary>
              spikethickness
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>2</dd>
              </dl>
              <p>
                Sets the thickness (in px) of the spikes.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tick0
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>number or categorical coordinate string</dd>
              </dl>
              <p>
                Sets the placement of the first tick on this axis. Use with `dtick`. If the axis `type` is "log", then you must take the log of your starting tick (e.g. to set the starting tick to 100, set the `tick0` to 2) except when `dtick`="L&lt;f&gt;" (see `dtick` for more info). If the axis `type` is "date", it should be a date string, like date data. If the axis `type` is "category", it should be a number, using the scale where each category is assigned a serial number from zero in the order it appears.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickangle
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>angle</dd>
                <dt>Default:</dt>
                <dd>"auto"</dd>
              </dl>
              <p>
                Sets the angle of the tick labels with respect to the horizontal. For example, a `tickangle` of -90 draws the tick labels vertically.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickcolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets the tick color.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickfont
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
              </dl>
              <p>
                Sets the tick font.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                color
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.xaxis.tickfont</dd>
                  <dt>Type:</dt>
                  <dd>color</dd>
                </dl>
              </div>
            </details>
            <details>
              <summary>
                family
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.xaxis.tickfont</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
                </p>
              </div>
            </details>
            <details>
              <summary>
                size
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.xaxis.tickfont</dd>
                  <dt>Type:</dt>
                  <dd>number greater than or equal to 1</dd>
                </dl>
              </div>
            </details>
          </div>
          <details>
            <summary>
              tickformat
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets the tick label formatting rule using d3 formatting mini-languages which are very similar to those in Python. For numbers, see: https://github.com/d3/d3-3.x-api-reference/blob/master/Formatting.md#d3_format And for dates see: https://github.com/d3/d3-3.x-api-reference/blob/master/Time-Formatting.md#format We add one item to d3's date formatter: "%{n}f" for fractional seconds with n digits. For example, "2016-10-13 09:15:23.456" with tickformat "%H~%\M~%S.%2f" would display "09~15~23.46"
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickformatstops
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>array of object where each object has one or more of the keys listed below.</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not this stop is used. If `false`, this stop is ignored even within its `dtickrange`.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                dtickrange
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.xaxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>array</dd>
                </dl>
                <p>
                  range ["min", "max"], where "min", "max" - dtick values which describe some zoom level, it is possible to omit "min" or "max" value by passing "null"
                </p>
              </div>
            </details>
            <details>
              <summary>
                enabled
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.xaxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>boolean</dd>
                  <dt>Default:</dt>
                  <dd>true</dd>
                </dl>
                <p>
                  Determines whether or not this stop is used. If `false`, this stop is ignored even within its `dtickrange`.
                </p>
              </div>
            </details>
            <details>
              <summary>
                name
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.xaxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  When used in a template, named items are created in the output figure in addition to any items the figure already has in this array. You can modify these items in the output figure by making your own item with `templateitemname` matching this `name` alongside your modifications (including `visible: false` or `enabled: false` to hide it). Has no effect outside of a template.
                </p>
              </div>
            </details>
            <details>
              <summary>
                templateitemname
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.xaxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  Used to refer to a named item in this array in the template. Named items from the template will be created even without a matching item in the input figure, but you can modify one by making an item with `templateitemname` matching its `name`, alongside your modifications (including `visible: false` or `enabled: false` to hide it). If there is no template or no matching item, this item will be hidden unless you explicitly show it with `visible: true`.
                </p>
              </div>
            </details>
            <details>
              <summary>
                value
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.xaxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                  <dt>Default:</dt>
                  <dd>""</dd>
                </dl>
                <p>
                  string - dtickformat for described zoom level, the same as "tickformat"
                </p>
              </div>
            </details>
          </div>
          <details>
            <summary>
              ticklen
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>5</dd>
              </dl>
              <p>
                Sets the tick length (in px).
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickmode
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "auto" | "linear" | "array" )</dd>
              </dl>
              <p>
                Sets the tick mode for this axis. If "auto", the number of ticks is set via `nticks`. If "linear", the placement of the ticks is determined by a starting position `tick0` and a tick step `dtick` ("linear" is the default value if `tick0` and `dtick` are provided). If "array", the placement of the ticks is set via `tickvals` and the tick text is `ticktext`. ("array" is the default value if `tickvals` is provided).
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickprefix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets a tick label prefix.
              </p>
            </div>
          </details>
          <details>
            <summary>
              ticks
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "outside" | "inside" | "" )</dd>
              </dl>
              <p>
                Determines whether ticks are drawn or not. If "", this axis' ticks are not drawn. If "outside" ("inside"), this axis' are drawn outside (inside) the axis lines.
              </p>
            </div>
          </details>
          <details>
            <summary>
              ticksuffix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets a tick label suffix.
              </p>
            </div>
          </details>
          <details>
            <summary>
              ticktext
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>data array</dd>
              </dl>
              <p>
                Sets the text displayed at the ticks position via `tickvals`. Only has an effect if `tickmode` is set to "array". Used with `tickvals`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickvals
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>data array</dd>
              </dl>
              <p>
                Sets the values at which ticks on this axis appear. Only has an effect if `tickmode` is set to "array". Used with `ticktext`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickwidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the tick width (in px).
              </p>
            </div>
          </details>
          <details>
            <summary>
              title
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
              </dl>
              <p>
                Sets the title of this axis. Note that before the existence of `title.text`, the title's contents used to be defined as the `title` attribute itself. This behavior has been deprecated.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                font
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.xaxis.title</dd>
                  <dt>Type:</dt>
                  <dd>object containing one or more of the keys listed below.</dd>
                </dl>
                <p>
                  Sets this axis' title font. Note that the title's font used to be customized by the now deprecated `titlefont` attribute.
                </p>
              </div>
            </details>
            <div style="margin-left: 22px">
              <details>
                <summary>
                  color
                </summary>
                <div style="margin-left: 22px">
                  <dl>
                    <dt>Parent:</dt>
                    <dd>layout.scene.xaxis.title.font</dd>
                    <dt>Type:</dt>
                    <dd>color</dd>
                  </dl>
                </div>
              </details>
              <details>
                <summary>
                  family
                </summary>
                <div style="margin-left: 22px">
                  <dl>
                    <dt>Parent:</dt>
                    <dd>layout.scene.xaxis.title.font</dd>
                    <dt>Type:</dt>
                    <dd>string</dd>
                  </dl>
                  <p>
                    HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
                  </p>
                </div>
              </details>
              <details>
                <summary>
                  size
                </summary>
                <div style="margin-left: 22px">
                  <dl>
                    <dt>Parent:</dt>
                    <dd>layout.scene.xaxis.title.font</dd>
                    <dt>Type:</dt>
                    <dd>number greater than or equal to 1</dd>
                  </dl>
                </div>
              </details>
            </div>
            <details>
              <summary>
                text
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.xaxis.title</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  Sets the title of this axis. Note that before the existence of `title.text`, the title's contents used to be defined as the `title` attribute itself. This behavior has been deprecated.
                </p>
              </div>
            </details>
          </div>
          <details>
            <summary>
              type
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "-" | "linear" | "log" | "date" | "category" )</dd>
                <dt>Default:</dt>
                <dd>"-"</dd>
              </dl>
              <p>
                Sets the axis type. By default, plotly attempts to determined the axis type by looking into the data of the traces that referenced the axis in question.
              </p>
            </div>
          </details>
          <details>
            <summary>
              visible
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
              </dl>
              <p>
                A single toggle to hide the axis while preserving interaction like dragging. Default is true when a cheater plot is present on the axis, otherwise false
              </p>
            </div>
          </details>
          <details>
            <summary>
              zeroline
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
              </dl>
              <p>
                Determines whether or not a line is drawn at along the 0 value of this axis. If "true", the zero line is drawn on top of the grid lines.
              </p>
            </div>
          </details>
          <details>
            <summary>
              zerolinecolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets the line color of the zero line.
              </p>
            </div>
          </details>
          <details>
            <summary>
              zerolinewidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.xaxis</dd>
                <dt>Type:</dt>
                <dd>number</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the width (in px) of the zero line.
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            yaxis
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.scene</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>true</dd>
            </dl>
            <p>
              A single toggle to hide the axis while preserving interaction like dragging. Default is true when a cheater plot is present on the axis, otherwise false
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              autorange
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( true | false | "reversed" )</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not the range of this axis is computed in relation to the input data. See `rangemode` for more info. If `range` is provided, then `autorange` is set to "false".
              </p>
            </div>
          </details>
          <details>
            <summary>
              backgroundcolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"rgba(204, 204, 204, 0.5)"</dd>
              </dl>
              <p>
                Sets the background color of this axis' wall.
              </p>
            </div>
          </details>
          <details>
            <summary>
              calendar
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "gregorian" | "chinese" | "coptic" | "discworld" | "ethiopian" | "hebrew" | "islamic" | "julian" | "mayan" | "nanakshahi" | "nepali" | "persian" | "jalali" | "taiwan" | "thai" | "ummalqura" )</dd>
                <dt>Default:</dt>
                <dd>"gregorian"</dd>
              </dl>
              <p>
                Sets the calendar system to use for `range` and `tick0` if this is a date axis. This does not set the calendar for interpreting data on this axis, that's specified in the trace or via the global `layout.calendar`
              </p>
            </div>
          </details>
          <details>
            <summary>
              categoryarray
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>data array</dd>
              </dl>
              <p>
                Sets the order in which categories on this axis appear. Only has an effect if `categoryorder` is set to "array". Used with `categoryorder`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              categoryorder
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "trace" | "category ascending" | "category descending" | "array" | "total ascending" | "total descending" | "min ascending" | "min descending" | "max ascending" | "max descending" | "sum ascending" | "sum descending" | "mean ascending" | "mean descending" | "median ascending" | "median descending" )</dd>
                <dt>Default:</dt>
                <dd>"trace"</dd>
              </dl>
              <p>
                Specifies the ordering logic for the case of categorical variables. By default, plotly uses "trace", which specifies the order that is present in the data supplied. Set `categoryorder` to "category ascending" or "category descending" if order should be determined by the alphanumerical order of the category names. Set `categoryorder` to "array" to derive the ordering from the attribute `categoryarray`. If a category is not found in the `categoryarray` array, the sorting behavior for that attribute will be identical to the "trace" mode. The unspecified categories will follow the categories in `categoryarray`. Set `categoryorder` to "total ascending" or "total descending" if order should be determined by the numerical order of the values. Similarly, the order can be determined by the min, max, sum, mean or median of all the values.
              </p>
            </div>
          </details>
          <details>
            <summary>
              color
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets default for all colors associated with this axis all at once: line, font, tick, and grid colors. Grid color is lightened by blending this with the plot background Individual pieces can override this.
              </p>
            </div>
          </details>
          <details>
            <summary>
              dtick
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>number or categorical coordinate string</dd>
              </dl>
              <p>
                Sets the step in-between ticks on this axis. Use with `tick0`. Must be a positive number, or special strings available to "log" and "date" axes. If the axis `type` is "log", then ticks are set every 10^(n"dtick) where n is the tick number. For example, to set a tick mark at 1, 10, 100, 1000, ... set dtick to 1. To set tick marks at 1, 100, 10000, ... set dtick to 2. To set tick marks at 1, 5, 25, 125, 625, 3125, ... set dtick to log_10(5), or 0.69897000433. "log" has several special values; "L&lt;f&gt;", where `f` is a positive number, gives ticks linearly spaced in value (but not position). For example `tick0` = 0.1, `dtick` = "L0.5" will put ticks at 0.1, 0.6, 1.1, 1.6 etc. To show powers of 10 plus small digits between, use "D1" (all digits) or "D2" (only 2 and 5). `tick0` is ignored for "D1" and "D2". If the axis `type` is "date", then you must convert the time to milliseconds. For example, to set the interval between ticks to one day, set `dtick` to 86400000.0. "date" also has special values "M&lt;n&gt;" gives ticks spaced by a number of months. `n` must be a positive integer. To set ticks on the 15th of every third month, set `tick0` to "2000-01-15" and `dtick` to "M3". To set ticks every 4 years, set `dtick` to "M48"
              </p>
            </div>
          </details>
          <details>
            <summary>
              exponentformat
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "none" | "e" | "E" | "power" | "SI" | "B" )</dd>
                <dt>Default:</dt>
                <dd>"B"</dd>
              </dl>
              <p>
                Determines a formatting rule for the tick exponents. For example, consider the number 1,000,000,000. If "none", it appears as 1,000,000,000. If "e", 1e+9. If "E", 1E+9. If "power", 1x10^9 (with 9 in a super script). If "SI", 1G. If "B", 1B.
              </p>
            </div>
          </details>
          <details>
            <summary>
              gridcolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"rgb(204, 204, 204)"</dd>
              </dl>
              <p>
                Sets the color of the grid lines.
              </p>
            </div>
          </details>
          <details>
            <summary>
              gridwidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the width (in px) of the grid lines.
              </p>
            </div>
          </details>
          <details>
            <summary>
              hoverformat
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets the hover text formatting rule using d3 formatting mini-languages which are very similar to those in Python. For numbers, see: https://github.com/d3/d3-3.x-api-reference/blob/master/Formatting.md#d3_format And for dates see: https://github.com/d3/d3-3.x-api-reference/blob/master/Time-Formatting.md#format We add one item to d3's date formatter: "%{n}f" for fractional seconds with n digits. For example, "2016-10-13 09:15:23.456" with tickformat "%H~%\M~%S.%2f" would display "09~15~23.46"
              </p>
            </div>
          </details>
          <details>
            <summary>
              linecolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets the axis line color.
              </p>
            </div>
          </details>
          <details>
            <summary>
              linewidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the width (in px) of the axis line.
              </p>
            </div>
          </details>
          <details>
            <summary>
              mirror
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( true | "ticks" | false | "all" | "allticks" )</dd>
              </dl>
              <p>
                Determines if the axis lines or/and ticks are mirrored to the opposite side of the plotting area. If "true", the axis lines are mirrored. If "ticks", the axis lines and ticks are mirrored. If "false", mirroring is disable. If "all", axis lines are mirrored on all shared-axes subplots. If "allticks", axis lines and ticks are mirrored on all shared-axes subplots.
              </p>
            </div>
          </details>
          <details>
            <summary>
              nticks
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>integer greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                Specifies the maximum number of ticks for the particular axis. The actual number of ticks will be chosen automatically to be less than or equal to `nticks`. Has an effect only if `tickmode` is set to "auto".
              </p>
            </div>
          </details>
          <details>
            <summary>
              range
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>array</dd>
              </dl>
              <p>
                Sets the range of this axis. If the axis `type` is "log", then you must take the log of your desired range (e.g. to set the range from 1 to 100, set the range from 0 to 2). If the axis `type` is "date", it should be date strings, like date data, though Date objects and unix milliseconds will be accepted and converted to strings. If the axis `type` is "category", it should be numbers, using the scale where each category is assigned a serial number from zero in the order it appears.
              </p>
            </div>
          </details>
          <details>
            <summary>
              rangemode
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "normal" | "tozero" | "nonnegative" )</dd>
                <dt>Default:</dt>
                <dd>"normal"</dd>
              </dl>
              <p>
                If "normal", the range is computed in relation to the extrema of the input data. If "tozero"`, the range extends to 0, regardless of the input data If "nonnegative", the range is non-negative, regardless of the input data. Applies only to linear axes.
              </p>
            </div>
          </details>
          <details>
            <summary>
              separatethousands
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
              </dl>
              <p>
                If "true", even 4-digit integers are separated
              </p>
            </div>
          </details>
          <details>
            <summary>
              showaxeslabels
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Sets whether or not this axis is labeled
              </p>
            </div>
          </details>
          <details>
            <summary>
              showbackground
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
              </dl>
              <p>
                Sets whether or not this axis' wall has a background color.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showexponent
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
                <dt>Default:</dt>
                <dd>"all"</dd>
              </dl>
              <p>
                If "all", all exponents are shown besides their significands. If "first", only the exponent of the first tick is shown. If "last", only the exponent of the last tick is shown. If "none", no exponents appear.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showgrid
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
              </dl>
              <p>
                Determines whether or not grid lines are drawn. If "true", the grid lines are drawn at every tick mark.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showline
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
              </dl>
              <p>
                Determines whether or not a line bounding this axis is drawn.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showspikes
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Sets whether or not spikes starting from data points to this axis' wall are shown on hover.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showticklabels
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not the tick labels are drawn.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showtickprefix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
                <dt>Default:</dt>
                <dd>"all"</dd>
              </dl>
              <p>
                If "all", all tick labels are displayed with a prefix. If "first", only the first tick is displayed with a prefix. If "last", only the last tick is displayed with a suffix. If "none", tick prefixes are hidden.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showticksuffix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
                <dt>Default:</dt>
                <dd>"all"</dd>
              </dl>
              <p>
                Same as `showtickprefix` but for tick suffixes.
              </p>
            </div>
          </details>
          <details>
            <summary>
              spikecolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets the color of the spikes.
              </p>
            </div>
          </details>
          <details>
            <summary>
              spikesides
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Sets whether or not spikes extending from the projection data points to this axis' wall boundaries are shown on hover.
              </p>
            </div>
          </details>
          <details>
            <summary>
              spikethickness
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>2</dd>
              </dl>
              <p>
                Sets the thickness (in px) of the spikes.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tick0
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>number or categorical coordinate string</dd>
              </dl>
              <p>
                Sets the placement of the first tick on this axis. Use with `dtick`. If the axis `type` is "log", then you must take the log of your starting tick (e.g. to set the starting tick to 100, set the `tick0` to 2) except when `dtick`="L&lt;f&gt;" (see `dtick` for more info). If the axis `type` is "date", it should be a date string, like date data. If the axis `type` is "category", it should be a number, using the scale where each category is assigned a serial number from zero in the order it appears.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickangle
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>angle</dd>
                <dt>Default:</dt>
                <dd>"auto"</dd>
              </dl>
              <p>
                Sets the angle of the tick labels with respect to the horizontal. For example, a `tickangle` of -90 draws the tick labels vertically.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickcolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets the tick color.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickfont
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
              </dl>
              <p>
                Sets the tick font.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                color
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.yaxis.tickfont</dd>
                  <dt>Type:</dt>
                  <dd>color</dd>
                </dl>
              </div>
            </details>
            <details>
              <summary>
                family
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.yaxis.tickfont</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
                </p>
              </div>
            </details>
            <details>
              <summary>
                size
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.yaxis.tickfont</dd>
                  <dt>Type:</dt>
                  <dd>number greater than or equal to 1</dd>
                </dl>
              </div>
            </details>
          </div>
          <details>
            <summary>
              tickformat
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets the tick label formatting rule using d3 formatting mini-languages which are very similar to those in Python. For numbers, see: https://github.com/d3/d3-3.x-api-reference/blob/master/Formatting.md#d3_format And for dates see: https://github.com/d3/d3-3.x-api-reference/blob/master/Time-Formatting.md#format We add one item to d3's date formatter: "%{n}f" for fractional seconds with n digits. For example, "2016-10-13 09:15:23.456" with tickformat "%H~%\M~%S.%2f" would display "09~15~23.46"
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickformatstops
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>array of object where each object has one or more of the keys listed below.</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not this stop is used. If `false`, this stop is ignored even within its `dtickrange`.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                dtickrange
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.yaxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>array</dd>
                </dl>
                <p>
                  range ["min", "max"], where "min", "max" - dtick values which describe some zoom level, it is possible to omit "min" or "max" value by passing "null"
                </p>
              </div>
            </details>
            <details>
              <summary>
                enabled
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.yaxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>boolean</dd>
                  <dt>Default:</dt>
                  <dd>true</dd>
                </dl>
                <p>
                  Determines whether or not this stop is used. If `false`, this stop is ignored even within its `dtickrange`.
                </p>
              </div>
            </details>
            <details>
              <summary>
                name
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.yaxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  When used in a template, named items are created in the output figure in addition to any items the figure already has in this array. You can modify these items in the output figure by making your own item with `templateitemname` matching this `name` alongside your modifications (including `visible: false` or `enabled: false` to hide it). Has no effect outside of a template.
                </p>
              </div>
            </details>
            <details>
              <summary>
                templateitemname
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.yaxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  Used to refer to a named item in this array in the template. Named items from the template will be created even without a matching item in the input figure, but you can modify one by making an item with `templateitemname` matching its `name`, alongside your modifications (including `visible: false` or `enabled: false` to hide it). If there is no template or no matching item, this item will be hidden unless you explicitly show it with `visible: true`.
                </p>
              </div>
            </details>
            <details>
              <summary>
                value
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.yaxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                  <dt>Default:</dt>
                  <dd>""</dd>
                </dl>
                <p>
                  string - dtickformat for described zoom level, the same as "tickformat"
                </p>
              </div>
            </details>
          </div>
          <details>
            <summary>
              ticklen
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>5</dd>
              </dl>
              <p>
                Sets the tick length (in px).
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickmode
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "auto" | "linear" | "array" )</dd>
              </dl>
              <p>
                Sets the tick mode for this axis. If "auto", the number of ticks is set via `nticks`. If "linear", the placement of the ticks is determined by a starting position `tick0` and a tick step `dtick` ("linear" is the default value if `tick0` and `dtick` are provided). If "array", the placement of the ticks is set via `tickvals` and the tick text is `ticktext`. ("array" is the default value if `tickvals` is provided).
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickprefix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets a tick label prefix.
              </p>
            </div>
          </details>
          <details>
            <summary>
              ticks
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "outside" | "inside" | "" )</dd>
              </dl>
              <p>
                Determines whether ticks are drawn or not. If "", this axis' ticks are not drawn. If "outside" ("inside"), this axis' are drawn outside (inside) the axis lines.
              </p>
            </div>
          </details>
          <details>
            <summary>
              ticksuffix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets a tick label suffix.
              </p>
            </div>
          </details>
          <details>
            <summary>
              ticktext
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>data array</dd>
              </dl>
              <p>
                Sets the text displayed at the ticks position via `tickvals`. Only has an effect if `tickmode` is set to "array". Used with `tickvals`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickvals
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>data array</dd>
              </dl>
              <p>
                Sets the values at which ticks on this axis appear. Only has an effect if `tickmode` is set to "array". Used with `ticktext`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickwidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the tick width (in px).
              </p>
            </div>
          </details>
          <details>
            <summary>
              title
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
              </dl>
              <p>
                Sets the title of this axis. Note that before the existence of `title.text`, the title's contents used to be defined as the `title` attribute itself. This behavior has been deprecated.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                font
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.yaxis.title</dd>
                  <dt>Type:</dt>
                  <dd>object containing one or more of the keys listed below.</dd>
                </dl>
                <p>
                  Sets this axis' title font. Note that the title's font used to be customized by the now deprecated `titlefont` attribute.
                </p>
              </div>
            </details>
            <div style="margin-left: 22px">
              <details>
                <summary>
                  color
                </summary>
                <div style="margin-left: 22px">
                  <dl>
                    <dt>Parent:</dt>
                    <dd>layout.scene.yaxis.title.font</dd>
                    <dt>Type:</dt>
                    <dd>color</dd>
                  </dl>
                </div>
              </details>
              <details>
                <summary>
                  family
                </summary>
                <div style="margin-left: 22px">
                  <dl>
                    <dt>Parent:</dt>
                    <dd>layout.scene.yaxis.title.font</dd>
                    <dt>Type:</dt>
                    <dd>string</dd>
                  </dl>
                  <p>
                    HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
                  </p>
                </div>
              </details>
              <details>
                <summary>
                  size
                </summary>
                <div style="margin-left: 22px">
                  <dl>
                    <dt>Parent:</dt>
                    <dd>layout.scene.yaxis.title.font</dd>
                    <dt>Type:</dt>
                    <dd>number greater than or equal to 1</dd>
                  </dl>
                </div>
              </details>
            </div>
            <details>
              <summary>
                text
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.yaxis.title</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  Sets the title of this axis. Note that before the existence of `title.text`, the title's contents used to be defined as the `title` attribute itself. This behavior has been deprecated.
                </p>
              </div>
            </details>
          </div>
          <details>
            <summary>
              type
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "-" | "linear" | "log" | "date" | "category" )</dd>
                <dt>Default:</dt>
                <dd>"-"</dd>
              </dl>
              <p>
                Sets the axis type. By default, plotly attempts to determined the axis type by looking into the data of the traces that referenced the axis in question.
              </p>
            </div>
          </details>
          <details>
            <summary>
              visible
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
              </dl>
              <p>
                A single toggle to hide the axis while preserving interaction like dragging. Default is true when a cheater plot is present on the axis, otherwise false
              </p>
            </div>
          </details>
          <details>
            <summary>
              zeroline
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
              </dl>
              <p>
                Determines whether or not a line is drawn at along the 0 value of this axis. If "true", the zero line is drawn on top of the grid lines.
              </p>
            </div>
          </details>
          <details>
            <summary>
              zerolinecolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets the line color of the zero line.
              </p>
            </div>
          </details>
          <details>
            <summary>
              zerolinewidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.yaxis</dd>
                <dt>Type:</dt>
                <dd>number</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the width (in px) of the zero line.
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            zaxis
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.scene</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>true</dd>
            </dl>
            <p>
              A single toggle to hide the axis while preserving interaction like dragging. Default is true when a cheater plot is present on the axis, otherwise false
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              autorange
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( true | false | "reversed" )</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not the range of this axis is computed in relation to the input data. See `rangemode` for more info. If `range` is provided, then `autorange` is set to "false".
              </p>
            </div>
          </details>
          <details>
            <summary>
              backgroundcolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"rgba(204, 204, 204, 0.5)"</dd>
              </dl>
              <p>
                Sets the background color of this axis' wall.
              </p>
            </div>
          </details>
          <details>
            <summary>
              calendar
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "gregorian" | "chinese" | "coptic" | "discworld" | "ethiopian" | "hebrew" | "islamic" | "julian" | "mayan" | "nanakshahi" | "nepali" | "persian" | "jalali" | "taiwan" | "thai" | "ummalqura" )</dd>
                <dt>Default:</dt>
                <dd>"gregorian"</dd>
              </dl>
              <p>
                Sets the calendar system to use for `range` and `tick0` if this is a date axis. This does not set the calendar for interpreting data on this axis, that's specified in the trace or via the global `layout.calendar`
              </p>
            </div>
          </details>
          <details>
            <summary>
              categoryarray
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>data array</dd>
              </dl>
              <p>
                Sets the order in which categories on this axis appear. Only has an effect if `categoryorder` is set to "array". Used with `categoryorder`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              categoryorder
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "trace" | "category ascending" | "category descending" | "array" | "total ascending" | "total descending" | "min ascending" | "min descending" | "max ascending" | "max descending" | "sum ascending" | "sum descending" | "mean ascending" | "mean descending" | "median ascending" | "median descending" )</dd>
                <dt>Default:</dt>
                <dd>"trace"</dd>
              </dl>
              <p>
                Specifies the ordering logic for the case of categorical variables. By default, plotly uses "trace", which specifies the order that is present in the data supplied. Set `categoryorder` to "category ascending" or "category descending" if order should be determined by the alphanumerical order of the category names. Set `categoryorder` to "array" to derive the ordering from the attribute `categoryarray`. If a category is not found in the `categoryarray` array, the sorting behavior for that attribute will be identical to the "trace" mode. The unspecified categories will follow the categories in `categoryarray`. Set `categoryorder` to "total ascending" or "total descending" if order should be determined by the numerical order of the values. Similarly, the order can be determined by the min, max, sum, mean or median of all the values.
              </p>
            </div>
          </details>
          <details>
            <summary>
              color
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets default for all colors associated with this axis all at once: line, font, tick, and grid colors. Grid color is lightened by blending this with the plot background Individual pieces can override this.
              </p>
            </div>
          </details>
          <details>
            <summary>
              dtick
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>number or categorical coordinate string</dd>
              </dl>
              <p>
                Sets the step in-between ticks on this axis. Use with `tick0`. Must be a positive number, or special strings available to "log" and "date" axes. If the axis `type` is "log", then ticks are set every 10^(n"dtick) where n is the tick number. For example, to set a tick mark at 1, 10, 100, 1000, ... set dtick to 1. To set tick marks at 1, 100, 10000, ... set dtick to 2. To set tick marks at 1, 5, 25, 125, 625, 3125, ... set dtick to log_10(5), or 0.69897000433. "log" has several special values; "L&lt;f&gt;", where `f` is a positive number, gives ticks linearly spaced in value (but not position). For example `tick0` = 0.1, `dtick` = "L0.5" will put ticks at 0.1, 0.6, 1.1, 1.6 etc. To show powers of 10 plus small digits between, use "D1" (all digits) or "D2" (only 2 and 5). `tick0` is ignored for "D1" and "D2". If the axis `type` is "date", then you must convert the time to milliseconds. For example, to set the interval between ticks to one day, set `dtick` to 86400000.0. "date" also has special values "M&lt;n&gt;" gives ticks spaced by a number of months. `n` must be a positive integer. To set ticks on the 15th of every third month, set `tick0` to "2000-01-15" and `dtick` to "M3". To set ticks every 4 years, set `dtick` to "M48"
              </p>
            </div>
          </details>
          <details>
            <summary>
              exponentformat
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "none" | "e" | "E" | "power" | "SI" | "B" )</dd>
                <dt>Default:</dt>
                <dd>"B"</dd>
              </dl>
              <p>
                Determines a formatting rule for the tick exponents. For example, consider the number 1,000,000,000. If "none", it appears as 1,000,000,000. If "e", 1e+9. If "E", 1E+9. If "power", 1x10^9 (with 9 in a super script). If "SI", 1G. If "B", 1B.
              </p>
            </div>
          </details>
          <details>
            <summary>
              gridcolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"rgb(204, 204, 204)"</dd>
              </dl>
              <p>
                Sets the color of the grid lines.
              </p>
            </div>
          </details>
          <details>
            <summary>
              gridwidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the width (in px) of the grid lines.
              </p>
            </div>
          </details>
          <details>
            <summary>
              hoverformat
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets the hover text formatting rule using d3 formatting mini-languages which are very similar to those in Python. For numbers, see: https://github.com/d3/d3-3.x-api-reference/blob/master/Formatting.md#d3_format And for dates see: https://github.com/d3/d3-3.x-api-reference/blob/master/Time-Formatting.md#format We add one item to d3's date formatter: "%{n}f" for fractional seconds with n digits. For example, "2016-10-13 09:15:23.456" with tickformat "%H~%\M~%S.%2f" would display "09~15~23.46"
              </p>
            </div>
          </details>
          <details>
            <summary>
              linecolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets the axis line color.
              </p>
            </div>
          </details>
          <details>
            <summary>
              linewidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the width (in px) of the axis line.
              </p>
            </div>
          </details>
          <details>
            <summary>
              mirror
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( true | "ticks" | false | "all" | "allticks" )</dd>
              </dl>
              <p>
                Determines if the axis lines or/and ticks are mirrored to the opposite side of the plotting area. If "true", the axis lines are mirrored. If "ticks", the axis lines and ticks are mirrored. If "false", mirroring is disable. If "all", axis lines are mirrored on all shared-axes subplots. If "allticks", axis lines and ticks are mirrored on all shared-axes subplots.
              </p>
            </div>
          </details>
          <details>
            <summary>
              nticks
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>integer greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                Specifies the maximum number of ticks for the particular axis. The actual number of ticks will be chosen automatically to be less than or equal to `nticks`. Has an effect only if `tickmode` is set to "auto".
              </p>
            </div>
          </details>
          <details>
            <summary>
              range
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>array</dd>
              </dl>
              <p>
                Sets the range of this axis. If the axis `type` is "log", then you must take the log of your desired range (e.g. to set the range from 1 to 100, set the range from 0 to 2). If the axis `type` is "date", it should be date strings, like date data, though Date objects and unix milliseconds will be accepted and converted to strings. If the axis `type` is "category", it should be numbers, using the scale where each category is assigned a serial number from zero in the order it appears.
              </p>
            </div>
          </details>
          <details>
            <summary>
              rangemode
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "normal" | "tozero" | "nonnegative" )</dd>
                <dt>Default:</dt>
                <dd>"normal"</dd>
              </dl>
              <p>
                If "normal", the range is computed in relation to the extrema of the input data. If "tozero"`, the range extends to 0, regardless of the input data If "nonnegative", the range is non-negative, regardless of the input data. Applies only to linear axes.
              </p>
            </div>
          </details>
          <details>
            <summary>
              separatethousands
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
              </dl>
              <p>
                If "true", even 4-digit integers are separated
              </p>
            </div>
          </details>
          <details>
            <summary>
              showaxeslabels
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Sets whether or not this axis is labeled
              </p>
            </div>
          </details>
          <details>
            <summary>
              showbackground
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
              </dl>
              <p>
                Sets whether or not this axis' wall has a background color.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showexponent
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
                <dt>Default:</dt>
                <dd>"all"</dd>
              </dl>
              <p>
                If "all", all exponents are shown besides their significands. If "first", only the exponent of the first tick is shown. If "last", only the exponent of the last tick is shown. If "none", no exponents appear.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showgrid
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
              </dl>
              <p>
                Determines whether or not grid lines are drawn. If "true", the grid lines are drawn at every tick mark.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showline
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
              </dl>
              <p>
                Determines whether or not a line bounding this axis is drawn.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showspikes
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Sets whether or not spikes starting from data points to this axis' wall are shown on hover.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showticklabels
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not the tick labels are drawn.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showtickprefix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
                <dt>Default:</dt>
                <dd>"all"</dd>
              </dl>
              <p>
                If "all", all tick labels are displayed with a prefix. If "first", only the first tick is displayed with a prefix. If "last", only the last tick is displayed with a suffix. If "none", tick prefixes are hidden.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showticksuffix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
                <dt>Default:</dt>
                <dd>"all"</dd>
              </dl>
              <p>
                Same as `showtickprefix` but for tick suffixes.
              </p>
            </div>
          </details>
          <details>
            <summary>
              spikecolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets the color of the spikes.
              </p>
            </div>
          </details>
          <details>
            <summary>
              spikesides
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Sets whether or not spikes extending from the projection data points to this axis' wall boundaries are shown on hover.
              </p>
            </div>
          </details>
          <details>
            <summary>
              spikethickness
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>2</dd>
              </dl>
              <p>
                Sets the thickness (in px) of the spikes.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tick0
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>number or categorical coordinate string</dd>
              </dl>
              <p>
                Sets the placement of the first tick on this axis. Use with `dtick`. If the axis `type` is "log", then you must take the log of your starting tick (e.g. to set the starting tick to 100, set the `tick0` to 2) except when `dtick`="L&lt;f&gt;" (see `dtick` for more info). If the axis `type` is "date", it should be a date string, like date data. If the axis `type` is "category", it should be a number, using the scale where each category is assigned a serial number from zero in the order it appears.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickangle
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>angle</dd>
                <dt>Default:</dt>
                <dd>"auto"</dd>
              </dl>
              <p>
                Sets the angle of the tick labels with respect to the horizontal. For example, a `tickangle` of -90 draws the tick labels vertically.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickcolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets the tick color.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickfont
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
              </dl>
              <p>
                Sets the tick font.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                color
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.zaxis.tickfont</dd>
                  <dt>Type:</dt>
                  <dd>color</dd>
                </dl>
              </div>
            </details>
            <details>
              <summary>
                family
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.zaxis.tickfont</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
                </p>
              </div>
            </details>
            <details>
              <summary>
                size
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.zaxis.tickfont</dd>
                  <dt>Type:</dt>
                  <dd>number greater than or equal to 1</dd>
                </dl>
              </div>
            </details>
          </div>
          <details>
            <summary>
              tickformat
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets the tick label formatting rule using d3 formatting mini-languages which are very similar to those in Python. For numbers, see: https://github.com/d3/d3-3.x-api-reference/blob/master/Formatting.md#d3_format And for dates see: https://github.com/d3/d3-3.x-api-reference/blob/master/Time-Formatting.md#format We add one item to d3's date formatter: "%{n}f" for fractional seconds with n digits. For example, "2016-10-13 09:15:23.456" with tickformat "%H~%\M~%S.%2f" would display "09~15~23.46"
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickformatstops
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>array of object where each object has one or more of the keys listed below.</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not this stop is used. If `false`, this stop is ignored even within its `dtickrange`.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                dtickrange
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.zaxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>array</dd>
                </dl>
                <p>
                  range ["min", "max"], where "min", "max" - dtick values which describe some zoom level, it is possible to omit "min" or "max" value by passing "null"
                </p>
              </div>
            </details>
            <details>
              <summary>
                enabled
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.zaxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>boolean</dd>
                  <dt>Default:</dt>
                  <dd>true</dd>
                </dl>
                <p>
                  Determines whether or not this stop is used. If `false`, this stop is ignored even within its `dtickrange`.
                </p>
              </div>
            </details>
            <details>
              <summary>
                name
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.zaxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  When used in a template, named items are created in the output figure in addition to any items the figure already has in this array. You can modify these items in the output figure by making your own item with `templateitemname` matching this `name` alongside your modifications (including `visible: false` or `enabled: false` to hide it). Has no effect outside of a template.
                </p>
              </div>
            </details>
            <details>
              <summary>
                templateitemname
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.zaxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  Used to refer to a named item in this array in the template. Named items from the template will be created even without a matching item in the input figure, but you can modify one by making an item with `templateitemname` matching its `name`, alongside your modifications (including `visible: false` or `enabled: false` to hide it). If there is no template or no matching item, this item will be hidden unless you explicitly show it with `visible: true`.
                </p>
              </div>
            </details>
            <details>
              <summary>
                value
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.zaxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                  <dt>Default:</dt>
                  <dd>""</dd>
                </dl>
                <p>
                  string - dtickformat for described zoom level, the same as "tickformat"
                </p>
              </div>
            </details>
          </div>
          <details>
            <summary>
              ticklen
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>5</dd>
              </dl>
              <p>
                Sets the tick length (in px).
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickmode
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "auto" | "linear" | "array" )</dd>
              </dl>
              <p>
                Sets the tick mode for this axis. If "auto", the number of ticks is set via `nticks`. If "linear", the placement of the ticks is determined by a starting position `tick0` and a tick step `dtick` ("linear" is the default value if `tick0` and `dtick` are provided). If "array", the placement of the ticks is set via `tickvals` and the tick text is `ticktext`. ("array" is the default value if `tickvals` is provided).
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickprefix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets a tick label prefix.
              </p>
            </div>
          </details>
          <details>
            <summary>
              ticks
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "outside" | "inside" | "" )</dd>
              </dl>
              <p>
                Determines whether ticks are drawn or not. If "", this axis' ticks are not drawn. If "outside" ("inside"), this axis' are drawn outside (inside) the axis lines.
              </p>
            </div>
          </details>
          <details>
            <summary>
              ticksuffix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets a tick label suffix.
              </p>
            </div>
          </details>
          <details>
            <summary>
              ticktext
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>data array</dd>
              </dl>
              <p>
                Sets the text displayed at the ticks position via `tickvals`. Only has an effect if `tickmode` is set to "array". Used with `tickvals`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickvals
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>data array</dd>
              </dl>
              <p>
                Sets the values at which ticks on this axis appear. Only has an effect if `tickmode` is set to "array". Used with `ticktext`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickwidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the tick width (in px).
              </p>
            </div>
          </details>
          <details>
            <summary>
              title
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
              </dl>
              <p>
                Sets the title of this axis. Note that before the existence of `title.text`, the title's contents used to be defined as the `title` attribute itself. This behavior has been deprecated.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                font
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.zaxis.title</dd>
                  <dt>Type:</dt>
                  <dd>object containing one or more of the keys listed below.</dd>
                </dl>
                <p>
                  Sets this axis' title font. Note that the title's font used to be customized by the now deprecated `titlefont` attribute.
                </p>
              </div>
            </details>
            <div style="margin-left: 22px">
              <details>
                <summary>
                  color
                </summary>
                <div style="margin-left: 22px">
                  <dl>
                    <dt>Parent:</dt>
                    <dd>layout.scene.zaxis.title.font</dd>
                    <dt>Type:</dt>
                    <dd>color</dd>
                  </dl>
                </div>
              </details>
              <details>
                <summary>
                  family
                </summary>
                <div style="margin-left: 22px">
                  <dl>
                    <dt>Parent:</dt>
                    <dd>layout.scene.zaxis.title.font</dd>
                    <dt>Type:</dt>
                    <dd>string</dd>
                  </dl>
                  <p>
                    HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
                  </p>
                </div>
              </details>
              <details>
                <summary>
                  size
                </summary>
                <div style="margin-left: 22px">
                  <dl>
                    <dt>Parent:</dt>
                    <dd>layout.scene.zaxis.title.font</dd>
                    <dt>Type:</dt>
                    <dd>number greater than or equal to 1</dd>
                  </dl>
                </div>
              </details>
            </div>
            <details>
              <summary>
                text
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.scene.zaxis.title</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  Sets the title of this axis. Note that before the existence of `title.text`, the title's contents used to be defined as the `title` attribute itself. This behavior has been deprecated.
                </p>
              </div>
            </details>
          </div>
          <details>
            <summary>
              type
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "-" | "linear" | "log" | "date" | "category" )</dd>
                <dt>Default:</dt>
                <dd>"-"</dd>
              </dl>
              <p>
                Sets the axis type. By default, plotly attempts to determined the axis type by looking into the data of the traces that referenced the axis in question.
              </p>
            </div>
          </details>
          <details>
            <summary>
              visible
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
              </dl>
              <p>
                A single toggle to hide the axis while preserving interaction like dragging. Default is true when a cheater plot is present on the axis, otherwise false
              </p>
            </div>
          </details>
          <details>
            <summary>
              zeroline
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
              </dl>
              <p>
                Determines whether or not a line is drawn at along the 0 value of this axis. If "true", the zero line is drawn on top of the grid lines.
              </p>
            </div>
          </details>
          <details>
            <summary>
              zerolinecolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets the line color of the zero line.
              </p>
            </div>
          </details>
          <details>
            <summary>
              zerolinewidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.scene.zaxis</dd>
                <dt>Type:</dt>
                <dd>number</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the width (in px) of the zero line.
              </p>
            </div>
          </details>
        </div>
      </div>
      <details>
        <summary>
          selectdirection
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>enumerated , one of ( "h" | "v" | "d" | "any" )</dd>
            <dt>Default:</dt>
            <dd>"any"</dd>
          </dl>
          <p>
            When "dragmode" is set to "select", this limits the selection of the drag to horizontal, vertical or diagonal. "h" only allows horizontal selection, "v" only vertical, "d" only diagonal and "any" sets no limit.
          </p>
        </div>
      </details>
      <details>
        <summary>
          selectionrevision
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>number or categorical coordinate string</dd>
          </dl>
          <p>
            Controls persistence of user-driven changes in selected points from all traces.
          </p>
        </div>
      </details>
      <details>
        <summary>
          separators
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>string</dd>
          </dl>
          <p>
            Sets the decimal and thousand separators. For example, ". " puts a '.' before decimals and a space between thousands. In English locales, dflt is ".," but other locales may alter this default.
          </p>
        </div>
      </details>
      <details>
        <summary>
          shapes
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>array of object where each object has one or more of the keys listed below.</dd>
            <dt>Default:</dt>
            <dd>true</dd>
          </dl>
          <p>
            Determines whether or not this shape is visible.
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            fillcolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.shapes[]</dd>
              <dt>Type:</dt>
              <dd>color</dd>
              <dt>Default:</dt>
              <dd>"rgba(0,0,0,0)"</dd>
            </dl>
            <p>
              Sets the color filling the shape's interior.
            </p>
          </div>
        </details>
        <details>
          <summary>
            layer
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.shapes[]</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "below" | "above" )</dd>
              <dt>Default:</dt>
              <dd>"above"</dd>
            </dl>
            <p>
              Specifies whether shapes are drawn below or above traces.
            </p>
          </div>
        </details>
        <details>
          <summary>
            line
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.shapes[]</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>2</dd>
            </dl>
            <p>
              Sets the line color.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              color
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.shapes[].line</dd>
                <dt>Type:</dt>
                <dd>color</dd>
              </dl>
              <p>
                Sets the line color.
              </p>
            </div>
          </details>
          <details>
            <summary>
              dash
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.shapes[].line</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>"solid"</dd>
              </dl>
              <p>
                Sets the dash style of lines. Set to a dash type string ("solid", "dot", "dash", "longdash", "dashdot", or "longdashdot") or a dash length list in px (eg "5px,10px,2px,2px").
              </p>
            </div>
          </details>
          <details>
            <summary>
              width
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.shapes[].line</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>2</dd>
              </dl>
              <p>
                Sets the line width (in px).
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            name
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.shapes[]</dd>
              <dt>Type:</dt>
              <dd>string</dd>
            </dl>
            <p>
              When used in a template, named items are created in the output figure in addition to any items the figure already has in this array. You can modify these items in the output figure by making your own item with `templateitemname` matching this `name` alongside your modifications (including `visible: false` or `enabled: false` to hide it). Has no effect outside of a template.
            </p>
          </div>
        </details>
        <details>
          <summary>
            opacity
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.shapes[]</dd>
              <dt>Type:</dt>
              <dd>number between or equal to 0 and 1</dd>
              <dt>Default:</dt>
              <dd>1</dd>
            </dl>
            <p>
              Sets the opacity of the shape.
            </p>
          </div>
        </details>
        <details>
          <summary>
            path
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.shapes[]</dd>
              <dt>Type:</dt>
              <dd>string</dd>
            </dl>
            <p>
              For `type` "path" - a valid SVG path with the pixel values replaced by data values in `xsizemode`/`ysizemode` being "scaled" and taken unmodified as pixels relative to `xanchor` and `yanchor` in case of "pixel" size mode. There are a few restrictions / quirks only absolute instructions, not relative. So the allowed segments are: M, L, H, V, Q, C, T, S, and Z arcs (A) are not allowed because radius rx and ry are relative. In the future we could consider supporting relative commands, but we would have to decide on how to handle date and log axes. Note that even as is, Q and C Bezier paths that are smooth on linear axes may not be smooth on log, and vice versa. no chained "polybezier" commands - specify the segment type for each one. On category axes, values are numbers scaled to the serial numbers of categories because using the categories themselves there would be no way to describe fractional positions On data axes: because space and T are both normal components of path strings, we can't use either to separate date from time parts. Therefore we'll use underscore for this purpose: 2015-02-21_13:45:56.789
            </p>
          </div>
        </details>
        <details>
          <summary>
            templateitemname
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.shapes[]</dd>
              <dt>Type:</dt>
              <dd>string</dd>
            </dl>
            <p>
              Used to refer to a named item in this array in the template. Named items from the template will be created even without a matching item in the input figure, but you can modify one by making an item with `templateitemname` matching its `name`, alongside your modifications (including `visible: false` or `enabled: false` to hide it). If there is no template or no matching item, this item will be hidden unless you explicitly show it with `visible: true`.
            </p>
          </div>
        </details>
        <details>
          <summary>
            type
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.shapes[]</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "circle" | "rect" | "path" | "line" )</dd>
            </dl>
            <p>
              Specifies the shape type to be drawn. If "line", a line is drawn from (`x0`,`y0`) to (`x1`,`y1`) with respect to the axes' sizing mode. If "circle", a circle is drawn from ((`x0`+`x1`)/2, (`y0`+`y1`)/2)) with radius (|(`x0`+`x1`)/2 - `x0`|, |(`y0`+`y1`)/2 -`y0`)|) with respect to the axes' sizing mode. If "rect", a rectangle is drawn linking (`x0`,`y0`), (`x1`,`y0`), (`x1`,`y1`), (`x0`,`y1`), (`x0`,`y0`) with respect to the axes' sizing mode. If "path", draw a custom SVG path using `path`. with respect to the axes' sizing mode.
            </p>
          </div>
        </details>
        <details>
          <summary>
            visible
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.shapes[]</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
              <dt>Default:</dt>
              <dd>true</dd>
            </dl>
            <p>
              Determines whether or not this shape is visible.
            </p>
          </div>
        </details>
        <details>
          <summary>
            x0
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.shapes[]</dd>
              <dt>Type:</dt>
              <dd>number or categorical coordinate string</dd>
            </dl>
            <p>
              Sets the shape's starting x position. See `type` and `xsizemode` for more info.
            </p>
          </div>
        </details>
        <details>
          <summary>
            x1
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.shapes[]</dd>
              <dt>Type:</dt>
              <dd>number or categorical coordinate string</dd>
            </dl>
            <p>
              Sets the shape's end x position. See `type` and `xsizemode` for more info.
            </p>
          </div>
        </details>
        <details>
          <summary>
            xanchor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.shapes[]</dd>
              <dt>Type:</dt>
              <dd>number or categorical coordinate string</dd>
            </dl>
            <p>
              Only relevant in conjunction with `xsizemode` set to "pixel". Specifies the anchor point on the x axis to which `x0`, `x1` and x coordinates within `path` are relative to. E.g. useful to attach a pixel sized shape to a certain data value. No effect when `xsizemode` not set to "pixel".
            </p>
          </div>
        </details>
        <details>
          <summary>
            xref
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.shapes[]</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "paper" | "/^x([2-9]|[1-9][0-9]+)?$/" )</dd>
            </dl>
            <p>
              Sets the shape's x coordinate axis. If set to an x axis id (e.g. "x" or "x2"), the `x` position refers to an x coordinate. If set to "paper", the `x` position refers to the distance from the left side of the plotting area in normalized coordinates where "0" ("1") corresponds to the left (right) side. If the axis `type` is "log", then you must take the log of your desired range. If the axis `type` is "date", then you must convert the date to unix time in milliseconds.
            </p>
          </div>
        </details>
        <details>
          <summary>
            xsizemode
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.shapes[]</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "scaled" | "pixel" )</dd>
              <dt>Default:</dt>
              <dd>"scaled"</dd>
            </dl>
            <p>
              Sets the shapes's sizing mode along the x axis. If set to "scaled", `x0`, `x1` and x coordinates within `path` refer to data values on the x axis or a fraction of the plot area's width (`xref` set to "paper"). If set to "pixel", `xanchor` specifies the x position in terms of data or plot fraction but `x0`, `x1` and x coordinates within `path` are pixels relative to `xanchor`. This way, the shape can have a fixed width while maintaining a position relative to data or plot fraction.
            </p>
          </div>
        </details>
        <details>
          <summary>
            y0
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.shapes[]</dd>
              <dt>Type:</dt>
              <dd>number or categorical coordinate string</dd>
            </dl>
            <p>
              Sets the shape's starting y position. See `type` and `ysizemode` for more info.
            </p>
          </div>
        </details>
        <details>
          <summary>
            y1
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.shapes[]</dd>
              <dt>Type:</dt>
              <dd>number or categorical coordinate string</dd>
            </dl>
            <p>
              Sets the shape's end y position. See `type` and `ysizemode` for more info.
            </p>
          </div>
        </details>
        <details>
          <summary>
            yanchor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.shapes[]</dd>
              <dt>Type:</dt>
              <dd>number or categorical coordinate string</dd>
            </dl>
            <p>
              Only relevant in conjunction with `ysizemode` set to "pixel". Specifies the anchor point on the y axis to which `y0`, `y1` and y coordinates within `path` are relative to. E.g. useful to attach a pixel sized shape to a certain data value. No effect when `ysizemode` not set to "pixel".
            </p>
          </div>
        </details>
        <details>
          <summary>
            yref
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.shapes[]</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "paper" | "/^y([2-9]|[1-9][0-9]+)?$/" )</dd>
            </dl>
            <p>
              Sets the annotation's y coordinate axis. If set to an y axis id (e.g. "y" or "y2"), the `y` position refers to an y coordinate If set to "paper", the `y` position refers to the distance from the bottom of the plotting area in normalized coordinates where "0" ("1") corresponds to the bottom (top).
            </p>
          </div>
        </details>
        <details>
          <summary>
            ysizemode
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.shapes[]</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "scaled" | "pixel" )</dd>
              <dt>Default:</dt>
              <dd>"scaled"</dd>
            </dl>
            <p>
              Sets the shapes's sizing mode along the y axis. If set to "scaled", `y0`, `y1` and y coordinates within `path` refer to data values on the y axis or a fraction of the plot area's height (`yref` set to "paper"). If set to "pixel", `yanchor` specifies the y position in terms of data or plot fraction but `y0`, `y1` and y coordinates within `path` are pixels relative to `yanchor`. This way, the shape can have a fixed height while maintaining a position relative to data or plot fraction.
            </p>
          </div>
        </details>
      </div>
      <details>
        <summary>
          showlegend
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>boolean</dd>
          </dl>
          <p>
            Determines whether or not a legend is drawn. Default is `true` if there is a trace to show and any of these: a) Two or more traces would by default be shown in the legend. b) One pie trace is shown in the legend. c) One trace is explicitly given with `showlegend: true`.
          </p>
        </div>
      </details>
      <details>
        <summary>
          sliders
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>array of object where each object has one or more of the keys listed below.</dd>
            <dt>Default:</dt>
            <dd>true</dd>
          </dl>
          <p>
            Determines whether or not the slider is visible.
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            active
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.sliders[]</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
            <p>
              Determines which button (by index starting from 0) is considered active.
            </p>
          </div>
        </details>
        <details>
          <summary>
            activebgcolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.sliders[]</dd>
              <dt>Type:</dt>
              <dd>color</dd>
              <dt>Default:</dt>
              <dd>"#dbdde0"</dd>
            </dl>
            <p>
              Sets the background color of the slider grip while dragging.
            </p>
          </div>
        </details>
        <details>
          <summary>
            bgcolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.sliders[]</dd>
              <dt>Type:</dt>
              <dd>color</dd>
              <dt>Default:</dt>
              <dd>"#f8fafc"</dd>
            </dl>
            <p>
              Sets the background color of the slider.
            </p>
          </div>
        </details>
        <details>
          <summary>
            bordercolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.sliders[]</dd>
              <dt>Type:</dt>
              <dd>color</dd>
              <dt>Default:</dt>
              <dd>"#bec8d9"</dd>
            </dl>
            <p>
              Sets the color of the border enclosing the slider.
            </p>
          </div>
        </details>
        <details>
          <summary>
            borderwidth
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.sliders[]</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>1</dd>
            </dl>
            <p>
              Sets the width (in px) of the border enclosing the slider.
            </p>
          </div>
        </details>
        <details>
          <summary>
            currentvalue
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.sliders[]</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>true</dd>
            </dl>
            <p>
              Shows the currently-selected value above the slider.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              font
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.sliders[].currentvalue</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
              </dl>
              <p>
                Sets the font of the current value label text.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                color
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.sliders[].currentvalue.font</dd>
                  <dt>Type:</dt>
                  <dd>color</dd>
                </dl>
              </div>
            </details>
            <details>
              <summary>
                family
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.sliders[].currentvalue.font</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
                </p>
              </div>
            </details>
            <details>
              <summary>
                size
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.sliders[].currentvalue.font</dd>
                  <dt>Type:</dt>
                  <dd>number greater than or equal to 1</dd>
                </dl>
              </div>
            </details>
          </div>
          <details>
            <summary>
              offset
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.sliders[].currentvalue</dd>
                <dt>Type:</dt>
                <dd>number</dd>
                <dt>Default:</dt>
                <dd>10</dd>
              </dl>
              <p>
                The amount of space, in pixels, between the current value label and the slider.
              </p>
            </div>
          </details>
          <details>
            <summary>
              prefix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.sliders[].currentvalue</dd>
                <dt>Type:</dt>
                <dd>string</dd>
              </dl>
              <p>
                When currentvalue.visible is true, this sets the prefix of the label.
              </p>
            </div>
          </details>
          <details>
            <summary>
              suffix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.sliders[].currentvalue</dd>
                <dt>Type:</dt>
                <dd>string</dd>
              </dl>
              <p>
                When currentvalue.visible is true, this sets the suffix of the label.
              </p>
            </div>
          </details>
          <details>
            <summary>
              visible
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.sliders[].currentvalue</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Shows the currently-selected value above the slider.
              </p>
            </div>
          </details>
          <details>
            <summary>
              xanchor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.sliders[].currentvalue</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "left" | "center" | "right" )</dd>
                <dt>Default:</dt>
                <dd>"left"</dd>
              </dl>
              <p>
                The alignment of the value readout relative to the length of the slider.
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            font
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.sliders[]</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
            </dl>
            <p>
              Sets the font of the slider step labels.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              color
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.sliders[].font</dd>
                <dt>Type:</dt>
                <dd>color</dd>
              </dl>
            </div>
          </details>
          <details>
            <summary>
              family
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.sliders[].font</dd>
                <dt>Type:</dt>
                <dd>string</dd>
              </dl>
              <p>
                HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
              </p>
            </div>
          </details>
          <details>
            <summary>
              size
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.sliders[].font</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 1</dd>
              </dl>
            </div>
          </details>
        </div>
        <details>
          <summary>
            len
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.sliders[]</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>1</dd>
            </dl>
            <p>
              Sets the length of the slider This measure excludes the padding of both ends. That is, the slider's length is this length minus the padding on both ends.
            </p>
          </div>
        </details>
        <details>
          <summary>
            lenmode
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.sliders[]</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "fraction" | "pixels" )</dd>
              <dt>Default:</dt>
              <dd>"fraction"</dd>
            </dl>
            <p>
              Determines whether this slider length is set in units of plot "fraction" or in "pixels. Use `len` to set the value.
            </p>
          </div>
        </details>
        <details>
          <summary>
            minorticklen
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.sliders[]</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>4</dd>
            </dl>
            <p>
              Sets the length in pixels of minor step tick marks
            </p>
          </div>
        </details>
        <details>
          <summary>
            name
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.sliders[]</dd>
              <dt>Type:</dt>
              <dd>string</dd>
            </dl>
            <p>
              When used in a template, named items are created in the output figure in addition to any items the figure already has in this array. You can modify these items in the output figure by making your own item with `templateitemname` matching this `name` alongside your modifications (including `visible: false` or `enabled: false` to hide it). Has no effect outside of a template.
            </p>
          </div>
        </details>
        <details>
          <summary>
            pad
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.sliders[]</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>20</dd>
            </dl>
            <p>
              Set the padding of the slider component along each side.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              b
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.sliders[].pad</dd>
                <dt>Type:</dt>
                <dd>number</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                The amount of padding (in px) along the bottom of the component.
              </p>
            </div>
          </details>
          <details>
            <summary>
              l
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.sliders[].pad</dd>
                <dt>Type:</dt>
                <dd>number</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                The amount of padding (in px) on the left side of the component.
              </p>
            </div>
          </details>
          <details>
            <summary>
              t
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.sliders[].pad</dd>
                <dt>Type:</dt>
                <dd>number</dd>
                <dt>Default:</dt>
                <dd>20</dd>
              </dl>
              <p>
                The amount of padding (in px) along the top of the component.
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            steps
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.sliders[]</dd>
              <dt>Type:</dt>
              <dd>array of object where each object has one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>true</dd>
            </dl>
            <p>
              Determines whether or not this step is included in the slider.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              args
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.sliders[].steps[]</dd>
                <dt>Type:</dt>
                <dd>array</dd>
              </dl>
              <p>
                Sets the arguments values to be passed to the Plotly method set in `method` on slide.
              </p>
            </div>
          </details>
          <details>
            <summary>
              execute
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.sliders[].steps[]</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                When true, the API method is executed. When false, all other behaviors are the same and command execution is skipped. This may be useful when hooking into, for example, the `plotly_sliderchange` method and executing the API command manually without losing the benefit of the slider automatically binding to the state of the plot through the specification of `method` and `args`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              label
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.sliders[].steps[]</dd>
                <dt>Type:</dt>
                <dd>string</dd>
              </dl>
              <p>
                Sets the text label to appear on the slider
              </p>
            </div>
          </details>
          <details>
            <summary>
              method
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.sliders[].steps[]</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "restyle" | "relayout" | "animate" | "update" | "skip" )</dd>
                <dt>Default:</dt>
                <dd>"restyle"</dd>
              </dl>
              <p>
                Sets the Plotly method to be called when the slider value is changed. If the `skip` method is used, the API slider will function as normal but will perform no API calls and will not bind automatically to state updates. This may be used to create a component interface and attach to slider events manually via JavaScript.
              </p>
            </div>
          </details>
          <details>
            <summary>
              name
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.sliders[].steps[]</dd>
                <dt>Type:</dt>
                <dd>string</dd>
              </dl>
              <p>
                When used in a template, named items are created in the output figure in addition to any items the figure already has in this array. You can modify these items in the output figure by making your own item with `templateitemname` matching this `name` alongside your modifications (including `visible: false` or `enabled: false` to hide it). Has no effect outside of a template.
              </p>
            </div>
          </details>
          <details>
            <summary>
              templateitemname
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.sliders[].steps[]</dd>
                <dt>Type:</dt>
                <dd>string</dd>
              </dl>
              <p>
                Used to refer to a named item in this array in the template. Named items from the template will be created even without a matching item in the input figure, but you can modify one by making an item with `templateitemname` matching its `name`, alongside your modifications (including `visible: false` or `enabled: false` to hide it). If there is no template or no matching item, this item will be hidden unless you explicitly show it with `visible: true`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              value
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.sliders[].steps[]</dd>
                <dt>Type:</dt>
                <dd>string</dd>
              </dl>
              <p>
                Sets the value of the slider step, used to refer to the step programatically. Defaults to the slider label if not provided.
              </p>
            </div>
          </details>
          <details>
            <summary>
              visible
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.sliders[].steps[]</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not this step is included in the slider.
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            templateitemname
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.sliders[]</dd>
              <dt>Type:</dt>
              <dd>string</dd>
            </dl>
            <p>
              Used to refer to a named item in this array in the template. Named items from the template will be created even without a matching item in the input figure, but you can modify one by making an item with `templateitemname` matching its `name`, alongside your modifications (including `visible: false` or `enabled: false` to hide it). If there is no template or no matching item, this item will be hidden unless you explicitly show it with `visible: true`.
            </p>
          </div>
        </details>
        <details>
          <summary>
            tickcolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.sliders[]</dd>
              <dt>Type:</dt>
              <dd>color</dd>
              <dt>Default:</dt>
              <dd>"#333"</dd>
            </dl>
            <p>
              Sets the color of the border enclosing the slider.
            </p>
          </div>
        </details>
        <details>
          <summary>
            ticklen
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.sliders[]</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>7</dd>
            </dl>
            <p>
              Sets the length in pixels of step tick marks
            </p>
          </div>
        </details>
        <details>
          <summary>
            tickwidth
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.sliders[]</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>1</dd>
            </dl>
            <p>
              Sets the tick width (in px).
            </p>
          </div>
        </details>
        <details>
          <summary>
            transition
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.sliders[]</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>150</dd>
            </dl>
            <p>
              Sets the duration of the slider transition
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              duration
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.sliders[].transition</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>150</dd>
              </dl>
              <p>
                Sets the duration of the slider transition
              </p>
            </div>
          </details>
          <details>
            <summary>
              easing
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.sliders[].transition</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "linear" | "quad" | "cubic" | "sin" | "exp" | "circle" | "elastic" | "back" | "bounce" | "linear-in" | "quad-in" | "cubic-in" | "sin-in" | "exp-in" | "circle-in" | "elastic-in" | "back-in" | "bounce-in" | "linear-out" | "quad-out" | "cubic-out" | "sin-out" | "exp-out" | "circle-out" | "elastic-out" | "back-out" | "bounce-out" | "linear-in-out" | "quad-in-out" | "cubic-in-out" | "sin-in-out" | "exp-in-out" | "circle-in-out" | "elastic-in-out" | "back-in-out" | "bounce-in-out" )</dd>
                <dt>Default:</dt>
                <dd>"cubic-in-out"</dd>
              </dl>
              <p>
                Sets the easing function of the slider transition
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            visible
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.sliders[]</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
              <dt>Default:</dt>
              <dd>true</dd>
            </dl>
            <p>
              Determines whether or not the slider is visible.
            </p>
          </div>
        </details>
        <details>
          <summary>
            x
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.sliders[]</dd>
              <dt>Type:</dt>
              <dd>number between or equal to -2 and 3</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
            <p>
              Sets the x position (in normalized coordinates) of the slider.
            </p>
          </div>
        </details>
        <details>
          <summary>
            xanchor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.sliders[]</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "auto" | "left" | "center" | "right" )</dd>
              <dt>Default:</dt>
              <dd>"left"</dd>
            </dl>
            <p>
              Sets the slider's horizontal position anchor. This anchor binds the `x` position to the "left", "center" or "right" of the range selector.
            </p>
          </div>
        </details>
        <details>
          <summary>
            y
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.sliders[]</dd>
              <dt>Type:</dt>
              <dd>number between or equal to -2 and 3</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
            <p>
              Sets the y position (in normalized coordinates) of the slider.
            </p>
          </div>
        </details>
        <details>
          <summary>
            yanchor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.sliders[]</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "auto" | "top" | "middle" | "bottom" )</dd>
              <dt>Default:</dt>
              <dd>"top"</dd>
            </dl>
            <p>
              Sets the slider's vertical position anchor This anchor binds the `y` position to the "top", "middle" or "bottom" of the range selector.
            </p>
          </div>
        </details>
      </div>
      <details>
        <summary>
          spikedistance
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>integer greater than or equal to -1</dd>
            <dt>Default:</dt>
            <dd>20</dd>
          </dl>
          <p>
            Sets the default distance (in pixels) to look for data to draw spikelines to (-1 means no cutoff, 0 means no looking for data). As with hoverdistance, distance does not apply to area-like objects. In addition, some objects can be hovered on but will not generate spikelines, such as scatter fills.
          </p>
        </div>
      </details>
      <details>
        <summary>
          template
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>number or categorical coordinate string</dd>
          </dl>
          <p>
            Default attributes to be applied to the plot. Templates can be created from existing plots using `Plotly.makeTemplate`, or created manually. They should be objects with format: `{layout: layoutTemplate, data: {[type]: [traceTemplate, ...]}, ...}` `layoutTemplate` and `traceTemplate` are objects matching the attribute structure of `layout` and a data trace. Trace templates are applied cyclically to traces of each type. Container arrays (eg `annotations`) have special handling: An object ending in `defaults` (eg `annotationdefaults`) is applied to each array item. But if an item has a `templateitemname` key we look in the template array for an item with matching `name` and apply that instead. If no matching `name` is found we mark the item invisible. Any named template item not referenced is appended to the end of the array, so you can use this for a watermark annotation or a logo image, for example. To omit one of these items on the plot, make an item with matching `templateitemname` and `visible: false`.
          </p>
        </div>
      </details>
      <details>
        <summary>
          ternary
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>object containing one or more of the keys listed below.</dd>
            <dt>Default:</dt>
            <dd>[0, 1]</dd>
          </dl>
          <p>
            Sets the horizontal domain of this ternary subplot (in plot fraction).
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            aaxis
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.ternary</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>"#444"</dd>
            </dl>
            <p>
              Sets the title of this axis. Note that before the existence of `title.text`, the title's contents used to be defined as the `title` attribute itself. This behavior has been deprecated.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              color
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.aaxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets default for all colors associated with this axis all at once: line, font, tick, and grid colors. Grid color is lightened by blending this with the plot background Individual pieces can override this.
              </p>
            </div>
          </details>
          <details>
            <summary>
              dtick
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.aaxis</dd>
                <dt>Type:</dt>
                <dd>number or categorical coordinate string</dd>
              </dl>
              <p>
                Sets the step in-between ticks on this axis. Use with `tick0`. Must be a positive number, or special strings available to "log" and "date" axes. If the axis `type` is "log", then ticks are set every 10^(n"dtick) where n is the tick number. For example, to set a tick mark at 1, 10, 100, 1000, ... set dtick to 1. To set tick marks at 1, 100, 10000, ... set dtick to 2. To set tick marks at 1, 5, 25, 125, 625, 3125, ... set dtick to log_10(5), or 0.69897000433. "log" has several special values; "L&lt;f&gt;", where `f` is a positive number, gives ticks linearly spaced in value (but not position). For example `tick0` = 0.1, `dtick` = "L0.5" will put ticks at 0.1, 0.6, 1.1, 1.6 etc. To show powers of 10 plus small digits between, use "D1" (all digits) or "D2" (only 2 and 5). `tick0` is ignored for "D1" and "D2". If the axis `type` is "date", then you must convert the time to milliseconds. For example, to set the interval between ticks to one day, set `dtick` to 86400000.0. "date" also has special values "M&lt;n&gt;" gives ticks spaced by a number of months. `n` must be a positive integer. To set ticks on the 15th of every third month, set `tick0` to "2000-01-15" and `dtick` to "M3". To set ticks every 4 years, set `dtick` to "M48"
              </p>
            </div>
          </details>
          <details>
            <summary>
              exponentformat
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.aaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "none" | "e" | "E" | "power" | "SI" | "B" )</dd>
                <dt>Default:</dt>
                <dd>"B"</dd>
              </dl>
              <p>
                Determines a formatting rule for the tick exponents. For example, consider the number 1,000,000,000. If "none", it appears as 1,000,000,000. If "e", 1e+9. If "E", 1E+9. If "power", 1x10^9 (with 9 in a super script). If "SI", 1G. If "B", 1B.
              </p>
            </div>
          </details>
          <details>
            <summary>
              gridcolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.aaxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#eee"</dd>
              </dl>
              <p>
                Sets the color of the grid lines.
              </p>
            </div>
          </details>
          <details>
            <summary>
              gridwidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.aaxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the width (in px) of the grid lines.
              </p>
            </div>
          </details>
          <details>
            <summary>
              hoverformat
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.aaxis</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets the hover text formatting rule using d3 formatting mini-languages which are very similar to those in Python. For numbers, see: https://github.com/d3/d3-3.x-api-reference/blob/master/Formatting.md#d3_format And for dates see: https://github.com/d3/d3-3.x-api-reference/blob/master/Time-Formatting.md#format We add one item to d3's date formatter: "%{n}f" for fractional seconds with n digits. For example, "2016-10-13 09:15:23.456" with tickformat "%H~%\M~%S.%2f" would display "09~15~23.46"
              </p>
            </div>
          </details>
          <details>
            <summary>
              layer
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.aaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "above traces" | "below traces" )</dd>
                <dt>Default:</dt>
                <dd>"above traces"</dd>
              </dl>
              <p>
                Sets the layer on which this axis is displayed. If "above traces", this axis is displayed above all the subplot's traces If "below traces", this axis is displayed below all the subplot's traces, but above the grid lines. Useful when used together with scatter-like traces with `cliponaxis` set to "false" to show markers and/or text nodes above this axis.
              </p>
            </div>
          </details>
          <details>
            <summary>
              linecolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.aaxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets the axis line color.
              </p>
            </div>
          </details>
          <details>
            <summary>
              linewidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.aaxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the width (in px) of the axis line.
              </p>
            </div>
          </details>
          <details>
            <summary>
              min
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.aaxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                The minimum value visible on this axis. The maximum is determined by the sum minus the minimum values of the other two axes. The full view corresponds to all the minima set to zero.
              </p>
            </div>
          </details>
          <details>
            <summary>
              nticks
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.aaxis</dd>
                <dt>Type:</dt>
                <dd>integer greater than or equal to 1</dd>
                <dt>Default:</dt>
                <dd>6</dd>
              </dl>
              <p>
                Specifies the maximum number of ticks for the particular axis. The actual number of ticks will be chosen automatically to be less than or equal to `nticks`. Has an effect only if `tickmode` is set to "auto".
              </p>
            </div>
          </details>
          <details>
            <summary>
              separatethousands
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.aaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
              </dl>
              <p>
                If "true", even 4-digit integers are separated
              </p>
            </div>
          </details>
          <details>
            <summary>
              showexponent
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.aaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
                <dt>Default:</dt>
                <dd>"all"</dd>
              </dl>
              <p>
                If "all", all exponents are shown besides their significands. If "first", only the exponent of the first tick is shown. If "last", only the exponent of the last tick is shown. If "none", no exponents appear.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showgrid
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.aaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not grid lines are drawn. If "true", the grid lines are drawn at every tick mark.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showline
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.aaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not a line bounding this axis is drawn.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showticklabels
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.aaxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not the tick labels are drawn.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showtickprefix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.aaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
                <dt>Default:</dt>
                <dd>"all"</dd>
              </dl>
              <p>
                If "all", all tick labels are displayed with a prefix. If "first", only the first tick is displayed with a prefix. If "last", only the last tick is displayed with a suffix. If "none", tick prefixes are hidden.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showticksuffix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.aaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
                <dt>Default:</dt>
                <dd>"all"</dd>
              </dl>
              <p>
                Same as `showtickprefix` but for tick suffixes.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tick0
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.aaxis</dd>
                <dt>Type:</dt>
                <dd>number or categorical coordinate string</dd>
              </dl>
              <p>
                Sets the placement of the first tick on this axis. Use with `dtick`. If the axis `type` is "log", then you must take the log of your starting tick (e.g. to set the starting tick to 100, set the `tick0` to 2) except when `dtick`="L&lt;f&gt;" (see `dtick` for more info). If the axis `type` is "date", it should be a date string, like date data. If the axis `type` is "category", it should be a number, using the scale where each category is assigned a serial number from zero in the order it appears.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickangle
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.aaxis</dd>
                <dt>Type:</dt>
                <dd>angle</dd>
                <dt>Default:</dt>
                <dd>"auto"</dd>
              </dl>
              <p>
                Sets the angle of the tick labels with respect to the horizontal. For example, a `tickangle` of -90 draws the tick labels vertically.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickcolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.aaxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets the tick color.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickfont
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.aaxis</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
              </dl>
              <p>
                Sets the tick font.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                color
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.ternary.aaxis.tickfont</dd>
                  <dt>Type:</dt>
                  <dd>color</dd>
                </dl>
              </div>
            </details>
            <details>
              <summary>
                family
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.ternary.aaxis.tickfont</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
                </p>
              </div>
            </details>
            <details>
              <summary>
                size
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.ternary.aaxis.tickfont</dd>
                  <dt>Type:</dt>
                  <dd>number greater than or equal to 1</dd>
                </dl>
              </div>
            </details>
          </div>
          <details>
            <summary>
              tickformat
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.aaxis</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets the tick label formatting rule using d3 formatting mini-languages which are very similar to those in Python. For numbers, see: https://github.com/d3/d3-3.x-api-reference/blob/master/Formatting.md#d3_format And for dates see: https://github.com/d3/d3-3.x-api-reference/blob/master/Time-Formatting.md#format We add one item to d3's date formatter: "%{n}f" for fractional seconds with n digits. For example, "2016-10-13 09:15:23.456" with tickformat "%H~%\M~%S.%2f" would display "09~15~23.46"
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickformatstops
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.aaxis</dd>
                <dt>Type:</dt>
                <dd>array of object where each object has one or more of the keys listed below.</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not this stop is used. If `false`, this stop is ignored even within its `dtickrange`.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                dtickrange
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.ternary.aaxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>array</dd>
                </dl>
                <p>
                  range ["min", "max"], where "min", "max" - dtick values which describe some zoom level, it is possible to omit "min" or "max" value by passing "null"
                </p>
              </div>
            </details>
            <details>
              <summary>
                enabled
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.ternary.aaxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>boolean</dd>
                  <dt>Default:</dt>
                  <dd>true</dd>
                </dl>
                <p>
                  Determines whether or not this stop is used. If `false`, this stop is ignored even within its `dtickrange`.
                </p>
              </div>
            </details>
            <details>
              <summary>
                name
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.ternary.aaxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  When used in a template, named items are created in the output figure in addition to any items the figure already has in this array. You can modify these items in the output figure by making your own item with `templateitemname` matching this `name` alongside your modifications (including `visible: false` or `enabled: false` to hide it). Has no effect outside of a template.
                </p>
              </div>
            </details>
            <details>
              <summary>
                templateitemname
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.ternary.aaxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  Used to refer to a named item in this array in the template. Named items from the template will be created even without a matching item in the input figure, but you can modify one by making an item with `templateitemname` matching its `name`, alongside your modifications (including `visible: false` or `enabled: false` to hide it). If there is no template or no matching item, this item will be hidden unless you explicitly show it with `visible: true`.
                </p>
              </div>
            </details>
            <details>
              <summary>
                value
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.ternary.aaxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                  <dt>Default:</dt>
                  <dd>""</dd>
                </dl>
                <p>
                  string - dtickformat for described zoom level, the same as "tickformat"
                </p>
              </div>
            </details>
          </div>
          <details>
            <summary>
              ticklen
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.aaxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>5</dd>
              </dl>
              <p>
                Sets the tick length (in px).
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickmode
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.aaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "auto" | "linear" | "array" )</dd>
              </dl>
              <p>
                Sets the tick mode for this axis. If "auto", the number of ticks is set via `nticks`. If "linear", the placement of the ticks is determined by a starting position `tick0` and a tick step `dtick` ("linear" is the default value if `tick0` and `dtick` are provided). If "array", the placement of the ticks is set via `tickvals` and the tick text is `ticktext`. ("array" is the default value if `tickvals` is provided).
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickprefix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.aaxis</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets a tick label prefix.
              </p>
            </div>
          </details>
          <details>
            <summary>
              ticks
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.aaxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "outside" | "inside" | "" )</dd>
              </dl>
              <p>
                Determines whether ticks are drawn or not. If "", this axis' ticks are not drawn. If "outside" ("inside"), this axis' are drawn outside (inside) the axis lines.
              </p>
            </div>
          </details>
          <details>
            <summary>
              ticksuffix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.aaxis</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets a tick label suffix.
              </p>
            </div>
          </details>
          <details>
            <summary>
              ticktext
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.aaxis</dd>
                <dt>Type:</dt>
                <dd>data array</dd>
              </dl>
              <p>
                Sets the text displayed at the ticks position via `tickvals`. Only has an effect if `tickmode` is set to "array". Used with `tickvals`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickvals
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.aaxis</dd>
                <dt>Type:</dt>
                <dd>data array</dd>
              </dl>
              <p>
                Sets the values at which ticks on this axis appear. Only has an effect if `tickmode` is set to "array". Used with `ticktext`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickwidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.aaxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the tick width (in px).
              </p>
            </div>
          </details>
          <details>
            <summary>
              title
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.aaxis</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
              </dl>
              <p>
                Sets the title of this axis. Note that before the existence of `title.text`, the title's contents used to be defined as the `title` attribute itself. This behavior has been deprecated.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                font
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.ternary.aaxis.title</dd>
                  <dt>Type:</dt>
                  <dd>object containing one or more of the keys listed below.</dd>
                </dl>
                <p>
                  Sets this axis' title font. Note that the title's font used to be customized by the now deprecated `titlefont` attribute.
                </p>
              </div>
            </details>
            <div style="margin-left: 22px">
              <details>
                <summary>
                  color
                </summary>
                <div style="margin-left: 22px">
                  <dl>
                    <dt>Parent:</dt>
                    <dd>layout.ternary.aaxis.title.font</dd>
                    <dt>Type:</dt>
                    <dd>color</dd>
                  </dl>
                </div>
              </details>
              <details>
                <summary>
                  family
                </summary>
                <div style="margin-left: 22px">
                  <dl>
                    <dt>Parent:</dt>
                    <dd>layout.ternary.aaxis.title.font</dd>
                    <dt>Type:</dt>
                    <dd>string</dd>
                  </dl>
                  <p>
                    HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
                  </p>
                </div>
              </details>
              <details>
                <summary>
                  size
                </summary>
                <div style="margin-left: 22px">
                  <dl>
                    <dt>Parent:</dt>
                    <dd>layout.ternary.aaxis.title.font</dd>
                    <dt>Type:</dt>
                    <dd>number greater than or equal to 1</dd>
                  </dl>
                </div>
              </details>
            </div>
            <details>
              <summary>
                text
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.ternary.aaxis.title</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  Sets the title of this axis. Note that before the existence of `title.text`, the title's contents used to be defined as the `title` attribute itself. This behavior has been deprecated.
                </p>
              </div>
            </details>
          </div>
          <details>
            <summary>
              uirevision
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.aaxis</dd>
                <dt>Type:</dt>
                <dd>number or categorical coordinate string</dd>
              </dl>
              <p>
                Controls persistence of user-driven changes in axis `min`, and `title` if in `editable: true` configuration. Defaults to `ternary&lt;N&gt;.uirevision`.
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            baxis
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.ternary</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>"#444"</dd>
            </dl>
            <p>
              Sets the title of this axis. Note that before the existence of `title.text`, the title's contents used to be defined as the `title` attribute itself. This behavior has been deprecated.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              color
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.baxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets default for all colors associated with this axis all at once: line, font, tick, and grid colors. Grid color is lightened by blending this with the plot background Individual pieces can override this.
              </p>
            </div>
          </details>
          <details>
            <summary>
              dtick
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.baxis</dd>
                <dt>Type:</dt>
                <dd>number or categorical coordinate string</dd>
              </dl>
              <p>
                Sets the step in-between ticks on this axis. Use with `tick0`. Must be a positive number, or special strings available to "log" and "date" axes. If the axis `type` is "log", then ticks are set every 10^(n"dtick) where n is the tick number. For example, to set a tick mark at 1, 10, 100, 1000, ... set dtick to 1. To set tick marks at 1, 100, 10000, ... set dtick to 2. To set tick marks at 1, 5, 25, 125, 625, 3125, ... set dtick to log_10(5), or 0.69897000433. "log" has several special values; "L&lt;f&gt;", where `f` is a positive number, gives ticks linearly spaced in value (but not position). For example `tick0` = 0.1, `dtick` = "L0.5" will put ticks at 0.1, 0.6, 1.1, 1.6 etc. To show powers of 10 plus small digits between, use "D1" (all digits) or "D2" (only 2 and 5). `tick0` is ignored for "D1" and "D2". If the axis `type` is "date", then you must convert the time to milliseconds. For example, to set the interval between ticks to one day, set `dtick` to 86400000.0. "date" also has special values "M&lt;n&gt;" gives ticks spaced by a number of months. `n` must be a positive integer. To set ticks on the 15th of every third month, set `tick0` to "2000-01-15" and `dtick` to "M3". To set ticks every 4 years, set `dtick` to "M48"
              </p>
            </div>
          </details>
          <details>
            <summary>
              exponentformat
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.baxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "none" | "e" | "E" | "power" | "SI" | "B" )</dd>
                <dt>Default:</dt>
                <dd>"B"</dd>
              </dl>
              <p>
                Determines a formatting rule for the tick exponents. For example, consider the number 1,000,000,000. If "none", it appears as 1,000,000,000. If "e", 1e+9. If "E", 1E+9. If "power", 1x10^9 (with 9 in a super script). If "SI", 1G. If "B", 1B.
              </p>
            </div>
          </details>
          <details>
            <summary>
              gridcolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.baxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#eee"</dd>
              </dl>
              <p>
                Sets the color of the grid lines.
              </p>
            </div>
          </details>
          <details>
            <summary>
              gridwidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.baxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the width (in px) of the grid lines.
              </p>
            </div>
          </details>
          <details>
            <summary>
              hoverformat
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.baxis</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets the hover text formatting rule using d3 formatting mini-languages which are very similar to those in Python. For numbers, see: https://github.com/d3/d3-3.x-api-reference/blob/master/Formatting.md#d3_format And for dates see: https://github.com/d3/d3-3.x-api-reference/blob/master/Time-Formatting.md#format We add one item to d3's date formatter: "%{n}f" for fractional seconds with n digits. For example, "2016-10-13 09:15:23.456" with tickformat "%H~%\M~%S.%2f" would display "09~15~23.46"
              </p>
            </div>
          </details>
          <details>
            <summary>
              layer
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.baxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "above traces" | "below traces" )</dd>
                <dt>Default:</dt>
                <dd>"above traces"</dd>
              </dl>
              <p>
                Sets the layer on which this axis is displayed. If "above traces", this axis is displayed above all the subplot's traces If "below traces", this axis is displayed below all the subplot's traces, but above the grid lines. Useful when used together with scatter-like traces with `cliponaxis` set to "false" to show markers and/or text nodes above this axis.
              </p>
            </div>
          </details>
          <details>
            <summary>
              linecolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.baxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets the axis line color.
              </p>
            </div>
          </details>
          <details>
            <summary>
              linewidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.baxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the width (in px) of the axis line.
              </p>
            </div>
          </details>
          <details>
            <summary>
              min
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.baxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                The minimum value visible on this axis. The maximum is determined by the sum minus the minimum values of the other two axes. The full view corresponds to all the minima set to zero.
              </p>
            </div>
          </details>
          <details>
            <summary>
              nticks
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.baxis</dd>
                <dt>Type:</dt>
                <dd>integer greater than or equal to 1</dd>
                <dt>Default:</dt>
                <dd>6</dd>
              </dl>
              <p>
                Specifies the maximum number of ticks for the particular axis. The actual number of ticks will be chosen automatically to be less than or equal to `nticks`. Has an effect only if `tickmode` is set to "auto".
              </p>
            </div>
          </details>
          <details>
            <summary>
              separatethousands
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.baxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
              </dl>
              <p>
                If "true", even 4-digit integers are separated
              </p>
            </div>
          </details>
          <details>
            <summary>
              showexponent
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.baxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
                <dt>Default:</dt>
                <dd>"all"</dd>
              </dl>
              <p>
                If "all", all exponents are shown besides their significands. If "first", only the exponent of the first tick is shown. If "last", only the exponent of the last tick is shown. If "none", no exponents appear.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showgrid
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.baxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not grid lines are drawn. If "true", the grid lines are drawn at every tick mark.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showline
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.baxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not a line bounding this axis is drawn.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showticklabels
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.baxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not the tick labels are drawn.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showtickprefix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.baxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
                <dt>Default:</dt>
                <dd>"all"</dd>
              </dl>
              <p>
                If "all", all tick labels are displayed with a prefix. If "first", only the first tick is displayed with a prefix. If "last", only the last tick is displayed with a suffix. If "none", tick prefixes are hidden.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showticksuffix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.baxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
                <dt>Default:</dt>
                <dd>"all"</dd>
              </dl>
              <p>
                Same as `showtickprefix` but for tick suffixes.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tick0
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.baxis</dd>
                <dt>Type:</dt>
                <dd>number or categorical coordinate string</dd>
              </dl>
              <p>
                Sets the placement of the first tick on this axis. Use with `dtick`. If the axis `type` is "log", then you must take the log of your starting tick (e.g. to set the starting tick to 100, set the `tick0` to 2) except when `dtick`="L&lt;f&gt;" (see `dtick` for more info). If the axis `type` is "date", it should be a date string, like date data. If the axis `type` is "category", it should be a number, using the scale where each category is assigned a serial number from zero in the order it appears.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickangle
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.baxis</dd>
                <dt>Type:</dt>
                <dd>angle</dd>
                <dt>Default:</dt>
                <dd>"auto"</dd>
              </dl>
              <p>
                Sets the angle of the tick labels with respect to the horizontal. For example, a `tickangle` of -90 draws the tick labels vertically.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickcolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.baxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets the tick color.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickfont
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.baxis</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
              </dl>
              <p>
                Sets the tick font.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                color
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.ternary.baxis.tickfont</dd>
                  <dt>Type:</dt>
                  <dd>color</dd>
                </dl>
              </div>
            </details>
            <details>
              <summary>
                family
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.ternary.baxis.tickfont</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
                </p>
              </div>
            </details>
            <details>
              <summary>
                size
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.ternary.baxis.tickfont</dd>
                  <dt>Type:</dt>
                  <dd>number greater than or equal to 1</dd>
                </dl>
              </div>
            </details>
          </div>
          <details>
            <summary>
              tickformat
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.baxis</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets the tick label formatting rule using d3 formatting mini-languages which are very similar to those in Python. For numbers, see: https://github.com/d3/d3-3.x-api-reference/blob/master/Formatting.md#d3_format And for dates see: https://github.com/d3/d3-3.x-api-reference/blob/master/Time-Formatting.md#format We add one item to d3's date formatter: "%{n}f" for fractional seconds with n digits. For example, "2016-10-13 09:15:23.456" with tickformat "%H~%\M~%S.%2f" would display "09~15~23.46"
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickformatstops
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.baxis</dd>
                <dt>Type:</dt>
                <dd>array of object where each object has one or more of the keys listed below.</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not this stop is used. If `false`, this stop is ignored even within its `dtickrange`.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                dtickrange
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.ternary.baxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>array</dd>
                </dl>
                <p>
                  range ["min", "max"], where "min", "max" - dtick values which describe some zoom level, it is possible to omit "min" or "max" value by passing "null"
                </p>
              </div>
            </details>
            <details>
              <summary>
                enabled
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.ternary.baxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>boolean</dd>
                  <dt>Default:</dt>
                  <dd>true</dd>
                </dl>
                <p>
                  Determines whether or not this stop is used. If `false`, this stop is ignored even within its `dtickrange`.
                </p>
              </div>
            </details>
            <details>
              <summary>
                name
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.ternary.baxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  When used in a template, named items are created in the output figure in addition to any items the figure already has in this array. You can modify these items in the output figure by making your own item with `templateitemname` matching this `name` alongside your modifications (including `visible: false` or `enabled: false` to hide it). Has no effect outside of a template.
                </p>
              </div>
            </details>
            <details>
              <summary>
                templateitemname
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.ternary.baxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  Used to refer to a named item in this array in the template. Named items from the template will be created even without a matching item in the input figure, but you can modify one by making an item with `templateitemname` matching its `name`, alongside your modifications (including `visible: false` or `enabled: false` to hide it). If there is no template or no matching item, this item will be hidden unless you explicitly show it with `visible: true`.
                </p>
              </div>
            </details>
            <details>
              <summary>
                value
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.ternary.baxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                  <dt>Default:</dt>
                  <dd>""</dd>
                </dl>
                <p>
                  string - dtickformat for described zoom level, the same as "tickformat"
                </p>
              </div>
            </details>
          </div>
          <details>
            <summary>
              ticklen
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.baxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>5</dd>
              </dl>
              <p>
                Sets the tick length (in px).
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickmode
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.baxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "auto" | "linear" | "array" )</dd>
              </dl>
              <p>
                Sets the tick mode for this axis. If "auto", the number of ticks is set via `nticks`. If "linear", the placement of the ticks is determined by a starting position `tick0` and a tick step `dtick` ("linear" is the default value if `tick0` and `dtick` are provided). If "array", the placement of the ticks is set via `tickvals` and the tick text is `ticktext`. ("array" is the default value if `tickvals` is provided).
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickprefix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.baxis</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets a tick label prefix.
              </p>
            </div>
          </details>
          <details>
            <summary>
              ticks
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.baxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "outside" | "inside" | "" )</dd>
              </dl>
              <p>
                Determines whether ticks are drawn or not. If "", this axis' ticks are not drawn. If "outside" ("inside"), this axis' are drawn outside (inside) the axis lines.
              </p>
            </div>
          </details>
          <details>
            <summary>
              ticksuffix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.baxis</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets a tick label suffix.
              </p>
            </div>
          </details>
          <details>
            <summary>
              ticktext
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.baxis</dd>
                <dt>Type:</dt>
                <dd>data array</dd>
              </dl>
              <p>
                Sets the text displayed at the ticks position via `tickvals`. Only has an effect if `tickmode` is set to "array". Used with `tickvals`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickvals
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.baxis</dd>
                <dt>Type:</dt>
                <dd>data array</dd>
              </dl>
              <p>
                Sets the values at which ticks on this axis appear. Only has an effect if `tickmode` is set to "array". Used with `ticktext`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickwidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.baxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the tick width (in px).
              </p>
            </div>
          </details>
          <details>
            <summary>
              title
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.baxis</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
              </dl>
              <p>
                Sets the title of this axis. Note that before the existence of `title.text`, the title's contents used to be defined as the `title` attribute itself. This behavior has been deprecated.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                font
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.ternary.baxis.title</dd>
                  <dt>Type:</dt>
                  <dd>object containing one or more of the keys listed below.</dd>
                </dl>
                <p>
                  Sets this axis' title font. Note that the title's font used to be customized by the now deprecated `titlefont` attribute.
                </p>
              </div>
            </details>
            <div style="margin-left: 22px">
              <details>
                <summary>
                  color
                </summary>
                <div style="margin-left: 22px">
                  <dl>
                    <dt>Parent:</dt>
                    <dd>layout.ternary.baxis.title.font</dd>
                    <dt>Type:</dt>
                    <dd>color</dd>
                  </dl>
                </div>
              </details>
              <details>
                <summary>
                  family
                </summary>
                <div style="margin-left: 22px">
                  <dl>
                    <dt>Parent:</dt>
                    <dd>layout.ternary.baxis.title.font</dd>
                    <dt>Type:</dt>
                    <dd>string</dd>
                  </dl>
                  <p>
                    HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
                  </p>
                </div>
              </details>
              <details>
                <summary>
                  size
                </summary>
                <div style="margin-left: 22px">
                  <dl>
                    <dt>Parent:</dt>
                    <dd>layout.ternary.baxis.title.font</dd>
                    <dt>Type:</dt>
                    <dd>number greater than or equal to 1</dd>
                  </dl>
                </div>
              </details>
            </div>
            <details>
              <summary>
                text
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.ternary.baxis.title</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  Sets the title of this axis. Note that before the existence of `title.text`, the title's contents used to be defined as the `title` attribute itself. This behavior has been deprecated.
                </p>
              </div>
            </details>
          </div>
          <details>
            <summary>
              uirevision
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.baxis</dd>
                <dt>Type:</dt>
                <dd>number or categorical coordinate string</dd>
              </dl>
              <p>
                Controls persistence of user-driven changes in axis `min`, and `title` if in `editable: true` configuration. Defaults to `ternary&lt;N&gt;.uirevision`.
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            bgcolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.ternary</dd>
              <dt>Type:</dt>
              <dd>color</dd>
              <dt>Default:</dt>
              <dd>"#fff"</dd>
            </dl>
            <p>
              Set the background color of the subplot
            </p>
          </div>
        </details>
        <details>
          <summary>
            caxis
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.ternary</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>"#444"</dd>
            </dl>
            <p>
              Sets the title of this axis. Note that before the existence of `title.text`, the title's contents used to be defined as the `title` attribute itself. This behavior has been deprecated.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              color
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.caxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets default for all colors associated with this axis all at once: line, font, tick, and grid colors. Grid color is lightened by blending this with the plot background Individual pieces can override this.
              </p>
            </div>
          </details>
          <details>
            <summary>
              dtick
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.caxis</dd>
                <dt>Type:</dt>
                <dd>number or categorical coordinate string</dd>
              </dl>
              <p>
                Sets the step in-between ticks on this axis. Use with `tick0`. Must be a positive number, or special strings available to "log" and "date" axes. If the axis `type` is "log", then ticks are set every 10^(n"dtick) where n is the tick number. For example, to set a tick mark at 1, 10, 100, 1000, ... set dtick to 1. To set tick marks at 1, 100, 10000, ... set dtick to 2. To set tick marks at 1, 5, 25, 125, 625, 3125, ... set dtick to log_10(5), or 0.69897000433. "log" has several special values; "L&lt;f&gt;", where `f` is a positive number, gives ticks linearly spaced in value (but not position). For example `tick0` = 0.1, `dtick` = "L0.5" will put ticks at 0.1, 0.6, 1.1, 1.6 etc. To show powers of 10 plus small digits between, use "D1" (all digits) or "D2" (only 2 and 5). `tick0` is ignored for "D1" and "D2". If the axis `type` is "date", then you must convert the time to milliseconds. For example, to set the interval between ticks to one day, set `dtick` to 86400000.0. "date" also has special values "M&lt;n&gt;" gives ticks spaced by a number of months. `n` must be a positive integer. To set ticks on the 15th of every third month, set `tick0` to "2000-01-15" and `dtick` to "M3". To set ticks every 4 years, set `dtick` to "M48"
              </p>
            </div>
          </details>
          <details>
            <summary>
              exponentformat
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.caxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "none" | "e" | "E" | "power" | "SI" | "B" )</dd>
                <dt>Default:</dt>
                <dd>"B"</dd>
              </dl>
              <p>
                Determines a formatting rule for the tick exponents. For example, consider the number 1,000,000,000. If "none", it appears as 1,000,000,000. If "e", 1e+9. If "E", 1E+9. If "power", 1x10^9 (with 9 in a super script). If "SI", 1G. If "B", 1B.
              </p>
            </div>
          </details>
          <details>
            <summary>
              gridcolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.caxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#eee"</dd>
              </dl>
              <p>
                Sets the color of the grid lines.
              </p>
            </div>
          </details>
          <details>
            <summary>
              gridwidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.caxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the width (in px) of the grid lines.
              </p>
            </div>
          </details>
          <details>
            <summary>
              hoverformat
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.caxis</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets the hover text formatting rule using d3 formatting mini-languages which are very similar to those in Python. For numbers, see: https://github.com/d3/d3-3.x-api-reference/blob/master/Formatting.md#d3_format And for dates see: https://github.com/d3/d3-3.x-api-reference/blob/master/Time-Formatting.md#format We add one item to d3's date formatter: "%{n}f" for fractional seconds with n digits. For example, "2016-10-13 09:15:23.456" with tickformat "%H~%\M~%S.%2f" would display "09~15~23.46"
              </p>
            </div>
          </details>
          <details>
            <summary>
              layer
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.caxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "above traces" | "below traces" )</dd>
                <dt>Default:</dt>
                <dd>"above traces"</dd>
              </dl>
              <p>
                Sets the layer on which this axis is displayed. If "above traces", this axis is displayed above all the subplot's traces If "below traces", this axis is displayed below all the subplot's traces, but above the grid lines. Useful when used together with scatter-like traces with `cliponaxis` set to "false" to show markers and/or text nodes above this axis.
              </p>
            </div>
          </details>
          <details>
            <summary>
              linecolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.caxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets the axis line color.
              </p>
            </div>
          </details>
          <details>
            <summary>
              linewidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.caxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the width (in px) of the axis line.
              </p>
            </div>
          </details>
          <details>
            <summary>
              min
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.caxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                The minimum value visible on this axis. The maximum is determined by the sum minus the minimum values of the other two axes. The full view corresponds to all the minima set to zero.
              </p>
            </div>
          </details>
          <details>
            <summary>
              nticks
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.caxis</dd>
                <dt>Type:</dt>
                <dd>integer greater than or equal to 1</dd>
                <dt>Default:</dt>
                <dd>6</dd>
              </dl>
              <p>
                Specifies the maximum number of ticks for the particular axis. The actual number of ticks will be chosen automatically to be less than or equal to `nticks`. Has an effect only if `tickmode` is set to "auto".
              </p>
            </div>
          </details>
          <details>
            <summary>
              separatethousands
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.caxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
              </dl>
              <p>
                If "true", even 4-digit integers are separated
              </p>
            </div>
          </details>
          <details>
            <summary>
              showexponent
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.caxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
                <dt>Default:</dt>
                <dd>"all"</dd>
              </dl>
              <p>
                If "all", all exponents are shown besides their significands. If "first", only the exponent of the first tick is shown. If "last", only the exponent of the last tick is shown. If "none", no exponents appear.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showgrid
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.caxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not grid lines are drawn. If "true", the grid lines are drawn at every tick mark.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showline
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.caxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not a line bounding this axis is drawn.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showticklabels
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.caxis</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not the tick labels are drawn.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showtickprefix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.caxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
                <dt>Default:</dt>
                <dd>"all"</dd>
              </dl>
              <p>
                If "all", all tick labels are displayed with a prefix. If "first", only the first tick is displayed with a prefix. If "last", only the last tick is displayed with a suffix. If "none", tick prefixes are hidden.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showticksuffix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.caxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
                <dt>Default:</dt>
                <dd>"all"</dd>
              </dl>
              <p>
                Same as `showtickprefix` but for tick suffixes.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tick0
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.caxis</dd>
                <dt>Type:</dt>
                <dd>number or categorical coordinate string</dd>
              </dl>
              <p>
                Sets the placement of the first tick on this axis. Use with `dtick`. If the axis `type` is "log", then you must take the log of your starting tick (e.g. to set the starting tick to 100, set the `tick0` to 2) except when `dtick`="L&lt;f&gt;" (see `dtick` for more info). If the axis `type` is "date", it should be a date string, like date data. If the axis `type` is "category", it should be a number, using the scale where each category is assigned a serial number from zero in the order it appears.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickangle
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.caxis</dd>
                <dt>Type:</dt>
                <dd>angle</dd>
                <dt>Default:</dt>
                <dd>"auto"</dd>
              </dl>
              <p>
                Sets the angle of the tick labels with respect to the horizontal. For example, a `tickangle` of -90 draws the tick labels vertically.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickcolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.caxis</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets the tick color.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickfont
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.caxis</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
              </dl>
              <p>
                Sets the tick font.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                color
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.ternary.caxis.tickfont</dd>
                  <dt>Type:</dt>
                  <dd>color</dd>
                </dl>
              </div>
            </details>
            <details>
              <summary>
                family
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.ternary.caxis.tickfont</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
                </p>
              </div>
            </details>
            <details>
              <summary>
                size
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.ternary.caxis.tickfont</dd>
                  <dt>Type:</dt>
                  <dd>number greater than or equal to 1</dd>
                </dl>
              </div>
            </details>
          </div>
          <details>
            <summary>
              tickformat
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.caxis</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets the tick label formatting rule using d3 formatting mini-languages which are very similar to those in Python. For numbers, see: https://github.com/d3/d3-3.x-api-reference/blob/master/Formatting.md#d3_format And for dates see: https://github.com/d3/d3-3.x-api-reference/blob/master/Time-Formatting.md#format We add one item to d3's date formatter: "%{n}f" for fractional seconds with n digits. For example, "2016-10-13 09:15:23.456" with tickformat "%H~%\M~%S.%2f" would display "09~15~23.46"
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickformatstops
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.caxis</dd>
                <dt>Type:</dt>
                <dd>array of object where each object has one or more of the keys listed below.</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not this stop is used. If `false`, this stop is ignored even within its `dtickrange`.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                dtickrange
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.ternary.caxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>array</dd>
                </dl>
                <p>
                  range ["min", "max"], where "min", "max" - dtick values which describe some zoom level, it is possible to omit "min" or "max" value by passing "null"
                </p>
              </div>
            </details>
            <details>
              <summary>
                enabled
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.ternary.caxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>boolean</dd>
                  <dt>Default:</dt>
                  <dd>true</dd>
                </dl>
                <p>
                  Determines whether or not this stop is used. If `false`, this stop is ignored even within its `dtickrange`.
                </p>
              </div>
            </details>
            <details>
              <summary>
                name
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.ternary.caxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  When used in a template, named items are created in the output figure in addition to any items the figure already has in this array. You can modify these items in the output figure by making your own item with `templateitemname` matching this `name` alongside your modifications (including `visible: false` or `enabled: false` to hide it). Has no effect outside of a template.
                </p>
              </div>
            </details>
            <details>
              <summary>
                templateitemname
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.ternary.caxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  Used to refer to a named item in this array in the template. Named items from the template will be created even without a matching item in the input figure, but you can modify one by making an item with `templateitemname` matching its `name`, alongside your modifications (including `visible: false` or `enabled: false` to hide it). If there is no template or no matching item, this item will be hidden unless you explicitly show it with `visible: true`.
                </p>
              </div>
            </details>
            <details>
              <summary>
                value
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.ternary.caxis.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                  <dt>Default:</dt>
                  <dd>""</dd>
                </dl>
                <p>
                  string - dtickformat for described zoom level, the same as "tickformat"
                </p>
              </div>
            </details>
          </div>
          <details>
            <summary>
              ticklen
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.caxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>5</dd>
              </dl>
              <p>
                Sets the tick length (in px).
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickmode
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.caxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "auto" | "linear" | "array" )</dd>
              </dl>
              <p>
                Sets the tick mode for this axis. If "auto", the number of ticks is set via `nticks`. If "linear", the placement of the ticks is determined by a starting position `tick0` and a tick step `dtick` ("linear" is the default value if `tick0` and `dtick` are provided). If "array", the placement of the ticks is set via `tickvals` and the tick text is `ticktext`. ("array" is the default value if `tickvals` is provided).
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickprefix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.caxis</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets a tick label prefix.
              </p>
            </div>
          </details>
          <details>
            <summary>
              ticks
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.caxis</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "outside" | "inside" | "" )</dd>
              </dl>
              <p>
                Determines whether ticks are drawn or not. If "", this axis' ticks are not drawn. If "outside" ("inside"), this axis' are drawn outside (inside) the axis lines.
              </p>
            </div>
          </details>
          <details>
            <summary>
              ticksuffix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.caxis</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets a tick label suffix.
              </p>
            </div>
          </details>
          <details>
            <summary>
              ticktext
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.caxis</dd>
                <dt>Type:</dt>
                <dd>data array</dd>
              </dl>
              <p>
                Sets the text displayed at the ticks position via `tickvals`. Only has an effect if `tickmode` is set to "array". Used with `tickvals`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickvals
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.caxis</dd>
                <dt>Type:</dt>
                <dd>data array</dd>
              </dl>
              <p>
                Sets the values at which ticks on this axis appear. Only has an effect if `tickmode` is set to "array". Used with `ticktext`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickwidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.caxis</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the tick width (in px).
              </p>
            </div>
          </details>
          <details>
            <summary>
              title
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.caxis</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
              </dl>
              <p>
                Sets the title of this axis. Note that before the existence of `title.text`, the title's contents used to be defined as the `title` attribute itself. This behavior has been deprecated.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                font
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.ternary.caxis.title</dd>
                  <dt>Type:</dt>
                  <dd>object containing one or more of the keys listed below.</dd>
                </dl>
                <p>
                  Sets this axis' title font. Note that the title's font used to be customized by the now deprecated `titlefont` attribute.
                </p>
              </div>
            </details>
            <div style="margin-left: 22px">
              <details>
                <summary>
                  color
                </summary>
                <div style="margin-left: 22px">
                  <dl>
                    <dt>Parent:</dt>
                    <dd>layout.ternary.caxis.title.font</dd>
                    <dt>Type:</dt>
                    <dd>color</dd>
                  </dl>
                </div>
              </details>
              <details>
                <summary>
                  family
                </summary>
                <div style="margin-left: 22px">
                  <dl>
                    <dt>Parent:</dt>
                    <dd>layout.ternary.caxis.title.font</dd>
                    <dt>Type:</dt>
                    <dd>string</dd>
                  </dl>
                  <p>
                    HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
                  </p>
                </div>
              </details>
              <details>
                <summary>
                  size
                </summary>
                <div style="margin-left: 22px">
                  <dl>
                    <dt>Parent:</dt>
                    <dd>layout.ternary.caxis.title.font</dd>
                    <dt>Type:</dt>
                    <dd>number greater than or equal to 1</dd>
                  </dl>
                </div>
              </details>
            </div>
            <details>
              <summary>
                text
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.ternary.caxis.title</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  Sets the title of this axis. Note that before the existence of `title.text`, the title's contents used to be defined as the `title` attribute itself. This behavior has been deprecated.
                </p>
              </div>
            </details>
          </div>
          <details>
            <summary>
              uirevision
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.caxis</dd>
                <dt>Type:</dt>
                <dd>number or categorical coordinate string</dd>
              </dl>
              <p>
                Controls persistence of user-driven changes in axis `min`, and `title` if in `editable: true` configuration. Defaults to `ternary&lt;N&gt;.uirevision`.
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            domain
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.ternary</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>[0, 1]</dd>
            </dl>
            <p>
              Sets the horizontal domain of this ternary subplot (in plot fraction).
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              column
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.domain</dd>
                <dt>Type:</dt>
                <dd>integer greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                If there is a layout grid, use the domain for this column in the grid for this ternary subplot .
              </p>
            </div>
          </details>
          <details>
            <summary>
              row
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.domain</dd>
                <dt>Type:</dt>
                <dd>integer greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                If there is a layout grid, use the domain for this row in the grid for this ternary subplot .
              </p>
            </div>
          </details>
          <details>
            <summary>
              x
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.domain</dd>
                <dt>Type:</dt>
                <dd>array</dd>
                <dt>Default:</dt>
                <dd>[0, 1]</dd>
              </dl>
              <p>
                Sets the horizontal domain of this ternary subplot (in plot fraction).
              </p>
            </div>
          </details>
          <details>
            <summary>
              y
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.ternary.domain</dd>
                <dt>Type:</dt>
                <dd>array</dd>
                <dt>Default:</dt>
                <dd>[0, 1]</dd>
              </dl>
              <p>
                Sets the vertical domain of this ternary subplot (in plot fraction).
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            sum
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.ternary</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>1</dd>
            </dl>
            <p>
              The number each triplet should sum to, and the maximum range of each axis
            </p>
          </div>
        </details>
        <details>
          <summary>
            uirevision
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.ternary</dd>
              <dt>Type:</dt>
              <dd>number or categorical coordinate string</dd>
            </dl>
            <p>
              Controls persistence of user-driven changes in axis `min` and `title`, if not overridden in the individual axes. Defaults to `layout.uirevision`.
            </p>
          </div>
        </details>
      </div>
      <details>
        <summary>
          title
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>object containing one or more of the keys listed below.</dd>
            <dt>Default:</dt>
            <dd>"container"</dd>
          </dl>
          <p>
            Sets the plot's title. Note that before the existence of `title.text`, the title's contents used to be defined as the `title` attribute itself. This behavior has been deprecated.
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            font
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.title</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
            </dl>
            <p>
              Sets the title font. Note that the title's font used to be customized by the now deprecated `titlefont` attribute.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              color
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.title.font</dd>
                <dt>Type:</dt>
                <dd>color</dd>
              </dl>
            </div>
          </details>
          <details>
            <summary>
              family
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.title.font</dd>
                <dt>Type:</dt>
                <dd>string</dd>
              </dl>
              <p>
                HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
              </p>
            </div>
          </details>
          <details>
            <summary>
              size
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.title.font</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 1</dd>
              </dl>
            </div>
          </details>
        </div>
        <details>
          <summary>
            pad
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.title</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
            <p>
              Sets the padding of the title. Each padding value only applies when the corresponding `xanchor`/`yanchor` value is set accordingly. E.g. for left padding to take effect, `xanchor` must be set to "left". The same rule applies if `xanchor`/`yanchor` is determined automatically. Padding is muted if the respective anchor value is "middle"/"center".
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              b
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.title.pad</dd>
                <dt>Type:</dt>
                <dd>number</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                The amount of padding (in px) along the bottom of the component.
              </p>
            </div>
          </details>
          <details>
            <summary>
              l
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.title.pad</dd>
                <dt>Type:</dt>
                <dd>number</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                The amount of padding (in px) on the left side of the component.
              </p>
            </div>
          </details>
          <details>
            <summary>
              t
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.title.pad</dd>
                <dt>Type:</dt>
                <dd>number</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                The amount of padding (in px) along the top of the component.
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            text
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.title</dd>
              <dt>Type:</dt>
              <dd>string</dd>
            </dl>
            <p>
              Sets the plot's title. Note that before the existence of `title.text`, the title's contents used to be defined as the `title` attribute itself. This behavior has been deprecated.
            </p>
          </div>
        </details>
        <details>
          <summary>
            x
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.title</dd>
              <dt>Type:</dt>
              <dd>number between or equal to 0 and 1</dd>
              <dt>Default:</dt>
              <dd>0.5</dd>
            </dl>
            <p>
              Sets the x position with respect to `xref` in normalized coordinates from "0" (left) to "1" (right).
            </p>
          </div>
        </details>
        <details>
          <summary>
            xanchor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.title</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "auto" | "left" | "center" | "right" )</dd>
              <dt>Default:</dt>
              <dd>"auto"</dd>
            </dl>
            <p>
              Sets the title's horizontal alignment with respect to its x position. "left" means that the title starts at x, "right" means that the title ends at x and "center" means that the title's center is at x. "auto" divides `xref` by three and calculates the `xanchor` value automatically based on the value of `x`.
            </p>
          </div>
        </details>
        <details>
          <summary>
            xref
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.title</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "container" | "paper" )</dd>
              <dt>Default:</dt>
              <dd>"container"</dd>
            </dl>
            <p>
              Sets the container `x` refers to. "container" spans the entire `width` of the plot. "paper" refers to the width of the plotting area only.
            </p>
          </div>
        </details>
        <details>
          <summary>
            y
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.title</dd>
              <dt>Type:</dt>
              <dd>number between or equal to 0 and 1</dd>
              <dt>Default:</dt>
              <dd>"auto"</dd>
            </dl>
            <p>
              Sets the y position with respect to `yref` in normalized coordinates from "0" (bottom) to "1" (top). "auto" places the baseline of the title onto the vertical center of the top margin.
            </p>
          </div>
        </details>
        <details>
          <summary>
            yanchor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.title</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "auto" | "top" | "middle" | "bottom" )</dd>
              <dt>Default:</dt>
              <dd>"auto"</dd>
            </dl>
            <p>
              Sets the title's vertical alignment with respect to its y position. "top" means that the title's cap line is at y, "bottom" means that the title's baseline is at y and "middle" means that the title's midline is at y. "auto" divides `yref` by three and calculates the `yanchor` value automatically based on the value of `y`.
            </p>
          </div>
        </details>
        <details>
          <summary>
            yref
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.title</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "container" | "paper" )</dd>
              <dt>Default:</dt>
              <dd>"container"</dd>
            </dl>
            <p>
              Sets the container `y` refers to. "container" spans the entire `height` of the plot. "paper" refers to the height of the plotting area only.
            </p>
          </div>
        </details>
      </div>
      <details>
        <summary>
          transition
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>object containing one or more of the keys listed below.</dd>
            <dt>Default:</dt>
            <dd>500</dd>
          </dl>
          <p>
            Sets transition options used during Plotly.react updates.
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            duration
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.transition</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>500</dd>
            </dl>
            <p>
              The duration of the transition, in milliseconds. If equal to zero, updates are synchronous.
            </p>
          </div>
        </details>
        <details>
          <summary>
            easing
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.transition</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "linear" | "quad" | "cubic" | "sin" | "exp" | "circle" | "elastic" | "back" | "bounce" | "linear-in" | "quad-in" | "cubic-in" | "sin-in" | "exp-in" | "circle-in" | "elastic-in" | "back-in" | "bounce-in" | "linear-out" | "quad-out" | "cubic-out" | "sin-out" | "exp-out" | "circle-out" | "elastic-out" | "back-out" | "bounce-out" | "linear-in-out" | "quad-in-out" | "cubic-in-out" | "sin-in-out" | "exp-in-out" | "circle-in-out" | "elastic-in-out" | "back-in-out" | "bounce-in-out" )</dd>
              <dt>Default:</dt>
              <dd>"cubic-in-out"</dd>
            </dl>
            <p>
              The easing function used for the transition
            </p>
          </div>
        </details>
        <details>
          <summary>
            ordering
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.transition</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "layout first" | "traces first" )</dd>
              <dt>Default:</dt>
              <dd>"layout first"</dd>
            </dl>
            <p>
              Determines whether the figure's layout or traces smoothly transitions during updates that make both traces and layout change.
            </p>
          </div>
        </details>
      </div>
      <details>
        <summary>
          uirevision
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>number or categorical coordinate string</dd>
          </dl>
          <p>
            Used to allow user interactions with the plot to persist after `Plotly.react` calls that are unaware of these interactions. If `uirevision` is omitted, or if it is given and it changed from the previous `Plotly.react` call, the exact new figure is used. If `uirevision` is truthy and did NOT change, any attribute that has been affected by user interactions and did not receive a different value in the new figure will keep the interaction value. `layout.uirevision` attribute serves as the default for `uirevision` attributes in various sub-containers. For finer control you can set these sub-attributes directly. For example, if your app separately controls the data on the x and y axes you might set `xaxis.uirevision="time"` and `yaxis.uirevision="cost"`. Then if only the y data is changed, you can update `yaxis.uirevision="quantity"` and the y axis range will reset but the x axis range will retain any user-driven zoom.
          </p>
        </div>
      </details>
      <details>
        <summary>
          updatemenus
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>array of object where each object has one or more of the keys listed below.</dd>
            <dt>Default:</dt>
            <dd>"dropdown"</dd>
          </dl>
          <p>
            Determines whether or not the update menu is visible.
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            active
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.updatemenus[]</dd>
              <dt>Type:</dt>
              <dd>integer greater than or equal to -1</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
            <p>
              Determines which button (by index starting from 0) is considered active.
            </p>
          </div>
        </details>
        <details>
          <summary>
            bgcolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.updatemenus[]</dd>
              <dt>Type:</dt>
              <dd>color</dd>
            </dl>
            <p>
              Sets the background color of the update menu buttons.
            </p>
          </div>
        </details>
        <details>
          <summary>
            bordercolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.updatemenus[]</dd>
              <dt>Type:</dt>
              <dd>color</dd>
              <dt>Default:</dt>
              <dd>"#BEC8D9"</dd>
            </dl>
            <p>
              Sets the color of the border enclosing the update menu.
            </p>
          </div>
        </details>
        <details>
          <summary>
            borderwidth
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.updatemenus[]</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>1</dd>
            </dl>
            <p>
              Sets the width (in px) of the border enclosing the update menu.
            </p>
          </div>
        </details>
        <details>
          <summary>
            buttons
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.updatemenus[]</dd>
              <dt>Type:</dt>
              <dd>array of object where each object has one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>"restyle"</dd>
            </dl>
            <p>
              Determines whether or not this button is visible.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              args
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.updatemenus[].buttons[]</dd>
                <dt>Type:</dt>
                <dd>array</dd>
              </dl>
              <p>
                Sets the arguments values to be passed to the Plotly method set in `method` on click.
              </p>
            </div>
          </details>
          <details>
            <summary>
              args2
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.updatemenus[].buttons[]</dd>
                <dt>Type:</dt>
                <dd>array</dd>
              </dl>
              <p>
                Sets a 2nd set of `args`, these arguments values are passed to the Plotly method set in `method` when clicking this button while in the active state. Use this to create toggle buttons.
              </p>
            </div>
          </details>
          <details>
            <summary>
              execute
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.updatemenus[].buttons[]</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                When true, the API method is executed. When false, all other behaviors are the same and command execution is skipped. This may be useful when hooking into, for example, the `plotly_buttonclicked` method and executing the API command manually without losing the benefit of the updatemenu automatically binding to the state of the plot through the specification of `method` and `args`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              label
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.updatemenus[].buttons[]</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets the text label to appear on the button.
              </p>
            </div>
          </details>
          <details>
            <summary>
              method
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.updatemenus[].buttons[]</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "restyle" | "relayout" | "animate" | "update" | "skip" )</dd>
                <dt>Default:</dt>
                <dd>"restyle"</dd>
              </dl>
              <p>
                Sets the Plotly method to be called on click. If the `skip` method is used, the API updatemenu will function as normal but will perform no API calls and will not bind automatically to state updates. This may be used to create a component interface and attach to updatemenu events manually via JavaScript.
              </p>
            </div>
          </details>
          <details>
            <summary>
              name
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.updatemenus[].buttons[]</dd>
                <dt>Type:</dt>
                <dd>string</dd>
              </dl>
              <p>
                When used in a template, named items are created in the output figure in addition to any items the figure already has in this array. You can modify these items in the output figure by making your own item with `templateitemname` matching this `name` alongside your modifications (including `visible: false` or `enabled: false` to hide it). Has no effect outside of a template.
              </p>
            </div>
          </details>
          <details>
            <summary>
              templateitemname
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.updatemenus[].buttons[]</dd>
                <dt>Type:</dt>
                <dd>string</dd>
              </dl>
              <p>
                Used to refer to a named item in this array in the template. Named items from the template will be created even without a matching item in the input figure, but you can modify one by making an item with `templateitemname` matching its `name`, alongside your modifications (including `visible: false` or `enabled: false` to hide it). If there is no template or no matching item, this item will be hidden unless you explicitly show it with `visible: true`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              visible
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.updatemenus[].buttons[]</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
              </dl>
              <p>
                Determines whether or not this button is visible.
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            direction
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.updatemenus[]</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "left" | "right" | "up" | "down" )</dd>
              <dt>Default:</dt>
              <dd>"down"</dd>
            </dl>
            <p>
              Determines the direction in which the buttons are laid out, whether in a dropdown menu or a row/column of buttons. For `left` and `up`, the buttons will still appear in left-to-right or top-to-bottom order respectively.
            </p>
          </div>
        </details>
        <details>
          <summary>
            font
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.updatemenus[]</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
            </dl>
            <p>
              Sets the font of the update menu button text.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              color
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.updatemenus[].font</dd>
                <dt>Type:</dt>
                <dd>color</dd>
              </dl>
            </div>
          </details>
          <details>
            <summary>
              family
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.updatemenus[].font</dd>
                <dt>Type:</dt>
                <dd>string</dd>
              </dl>
              <p>
                HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
              </p>
            </div>
          </details>
          <details>
            <summary>
              size
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.updatemenus[].font</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 1</dd>
              </dl>
            </div>
          </details>
        </div>
        <details>
          <summary>
            name
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.updatemenus[]</dd>
              <dt>Type:</dt>
              <dd>string</dd>
            </dl>
            <p>
              When used in a template, named items are created in the output figure in addition to any items the figure already has in this array. You can modify these items in the output figure by making your own item with `templateitemname` matching this `name` alongside your modifications (including `visible: false` or `enabled: false` to hide it). Has no effect outside of a template.
            </p>
          </div>
        </details>
        <details>
          <summary>
            pad
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.updatemenus[]</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
            <p>
              Sets the padding around the buttons or dropdown menu.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              b
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.updatemenus[].pad</dd>
                <dt>Type:</dt>
                <dd>number</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                The amount of padding (in px) along the bottom of the component.
              </p>
            </div>
          </details>
          <details>
            <summary>
              l
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.updatemenus[].pad</dd>
                <dt>Type:</dt>
                <dd>number</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                The amount of padding (in px) on the left side of the component.
              </p>
            </div>
          </details>
          <details>
            <summary>
              t
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.updatemenus[].pad</dd>
                <dt>Type:</dt>
                <dd>number</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                The amount of padding (in px) along the top of the component.
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            showactive
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.updatemenus[]</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
              <dt>Default:</dt>
              <dd>true</dd>
            </dl>
            <p>
              Highlights active dropdown item or active button if true.
            </p>
          </div>
        </details>
        <details>
          <summary>
            templateitemname
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.updatemenus[]</dd>
              <dt>Type:</dt>
              <dd>string</dd>
            </dl>
            <p>
              Used to refer to a named item in this array in the template. Named items from the template will be created even without a matching item in the input figure, but you can modify one by making an item with `templateitemname` matching its `name`, alongside your modifications (including `visible: false` or `enabled: false` to hide it). If there is no template or no matching item, this item will be hidden unless you explicitly show it with `visible: true`.
            </p>
          </div>
        </details>
        <details>
          <summary>
            type
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.updatemenus[]</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "dropdown" | "buttons" )</dd>
              <dt>Default:</dt>
              <dd>"dropdown"</dd>
            </dl>
            <p>
              Determines whether the buttons are accessible via a dropdown menu or whether the buttons are stacked horizontally or vertically
            </p>
          </div>
        </details>
        <details>
          <summary>
            visible
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.updatemenus[]</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Determines whether or not the update menu is visible.
            </p>
          </div>
        </details>
        <details>
          <summary>
            x
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.updatemenus[]</dd>
              <dt>Type:</dt>
              <dd>number between or equal to -2 and 3</dd>
              <dt>Default:</dt>
              <dd>-0.05</dd>
            </dl>
            <p>
              Sets the x position (in normalized coordinates) of the update menu.
            </p>
          </div>
        </details>
        <details>
          <summary>
            xanchor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.updatemenus[]</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "auto" | "left" | "center" | "right" )</dd>
              <dt>Default:</dt>
              <dd>"right"</dd>
            </dl>
            <p>
              Sets the update menu's horizontal position anchor. This anchor binds the `x` position to the "left", "center" or "right" of the range selector.
            </p>
          </div>
        </details>
        <details>
          <summary>
            y
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.updatemenus[]</dd>
              <dt>Type:</dt>
              <dd>number between or equal to -2 and 3</dd>
              <dt>Default:</dt>
              <dd>1</dd>
            </dl>
            <p>
              Sets the y position (in normalized coordinates) of the update menu.
            </p>
          </div>
        </details>
        <details>
          <summary>
            yanchor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.updatemenus[]</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "auto" | "top" | "middle" | "bottom" )</dd>
              <dt>Default:</dt>
              <dd>"top"</dd>
            </dl>
            <p>
              Sets the update menu's vertical position anchor This anchor binds the `y` position to the "top", "middle" or "bottom" of the range selector.
            </p>
          </div>
        </details>
      </div>
      <details>
        <summary>
          width
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>number greater than or equal to 10</dd>
            <dt>Default:</dt>
            <dd>700</dd>
          </dl>
          <p>
            Sets the plot's width (in px).
          </p>
        </div>
      </details>
      <details>
        <summary>
          xaxis
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>object containing one or more of the keys listed below.</dd>
            <dt>Default:</dt>
            <dd>"#444"</dd>
          </dl>
          <p>
            A single toggle to hide the axis while preserving interaction like dragging. Default is true when a cheater plot is present on the axis, otherwise false
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            anchor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "free" | "/^x([2-9]|[1-9][0-9]+)?$/" | "/^y([2-9]|[1-9][0-9]+)?$/" )</dd>
            </dl>
            <p>
              If set to an opposite-letter axis id (e.g. `x2`, `y`), this axis is bound to the corresponding opposite-letter axis. If set to "free", this axis' position is determined by `position`.
            </p>
          </div>
        </details>
        <details>
          <summary>
            automargin
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Determines whether long tick labels automatically grow the figure margins.
            </p>
          </div>
        </details>
        <details>
          <summary>
            autorange
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( true | false | "reversed" )</dd>
              <dt>Default:</dt>
              <dd>true</dd>
            </dl>
            <p>
              Determines whether or not the range of this axis is computed in relation to the input data. See `rangemode` for more info. If `range` is provided, then `autorange` is set to "false".
            </p>
          </div>
        </details>
        <details>
          <summary>
            calendar
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "gregorian" | "chinese" | "coptic" | "discworld" | "ethiopian" | "hebrew" | "islamic" | "julian" | "mayan" | "nanakshahi" | "nepali" | "persian" | "jalali" | "taiwan" | "thai" | "ummalqura" )</dd>
              <dt>Default:</dt>
              <dd>"gregorian"</dd>
            </dl>
            <p>
              Sets the calendar system to use for `range` and `tick0` if this is a date axis. This does not set the calendar for interpreting data on this axis, that's specified in the trace or via the global `layout.calendar`
            </p>
          </div>
        </details>
        <details>
          <summary>
            categoryarray
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>data array</dd>
            </dl>
            <p>
              Sets the order in which categories on this axis appear. Only has an effect if `categoryorder` is set to "array". Used with `categoryorder`.
            </p>
          </div>
        </details>
        <details>
          <summary>
            categoryorder
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "trace" | "category ascending" | "category descending" | "array" | "total ascending" | "total descending" | "min ascending" | "min descending" | "max ascending" | "max descending" | "sum ascending" | "sum descending" | "mean ascending" | "mean descending" | "median ascending" | "median descending" )</dd>
              <dt>Default:</dt>
              <dd>"trace"</dd>
            </dl>
            <p>
              Specifies the ordering logic for the case of categorical variables. By default, plotly uses "trace", which specifies the order that is present in the data supplied. Set `categoryorder` to "category ascending" or "category descending" if order should be determined by the alphanumerical order of the category names. Set `categoryorder` to "array" to derive the ordering from the attribute `categoryarray`. If a category is not found in the `categoryarray` array, the sorting behavior for that attribute will be identical to the "trace" mode. The unspecified categories will follow the categories in `categoryarray`. Set `categoryorder` to "total ascending" or "total descending" if order should be determined by the numerical order of the values. Similarly, the order can be determined by the min, max, sum, mean or median of all the values.
            </p>
          </div>
        </details>
        <details>
          <summary>
            color
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>color</dd>
              <dt>Default:</dt>
              <dd>"#444"</dd>
            </dl>
            <p>
              Sets default for all colors associated with this axis all at once: line, font, tick, and grid colors. Grid color is lightened by blending this with the plot background Individual pieces can override this.
            </p>
          </div>
        </details>
        <details>
          <summary>
            constrain
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "range" | "domain" )</dd>
              <dt>Default:</dt>
              <dd>"range"</dd>
            </dl>
            <p>
              If this axis needs to be compressed (either due to its own `scaleanchor` and `scaleratio` or those of the other axis), determines how that happens: by increasing the "range" (default), or by decreasing the "domain".
            </p>
          </div>
        </details>
        <details>
          <summary>
            constraintoward
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "left" | "center" | "right" | "top" | "middle" | "bottom" )</dd>
            </dl>
            <p>
              If this axis needs to be compressed (either due to its own `scaleanchor` and `scaleratio` or those of the other axis), determines which direction we push the originally specified plot area. Options are "left", "center" (default), and "right" for x axes, and "top", "middle" (default), and "bottom" for y axes.
            </p>
          </div>
        </details>
        <details>
          <summary>
            dividercolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>color</dd>
              <dt>Default:</dt>
              <dd>"#444"</dd>
            </dl>
            <p>
              Sets the color of the dividers Only has an effect on "multicategory" axes.
            </p>
          </div>
        </details>
        <details>
          <summary>
            dividerwidth
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>number</dd>
              <dt>Default:</dt>
              <dd>1</dd>
            </dl>
            <p>
              Sets the width (in px) of the dividers Only has an effect on "multicategory" axes.
            </p>
          </div>
        </details>
        <details>
          <summary>
            domain
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>array</dd>
              <dt>Default:</dt>
              <dd>[0, 1]</dd>
            </dl>
            <p>
              Sets the domain of this axis (in plot fraction).
            </p>
          </div>
        </details>
        <details>
          <summary>
            dtick
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>number or categorical coordinate string</dd>
            </dl>
            <p>
              Sets the step in-between ticks on this axis. Use with `tick0`. Must be a positive number, or special strings available to "log" and "date" axes. If the axis `type` is "log", then ticks are set every 10^(n"dtick) where n is the tick number. For example, to set a tick mark at 1, 10, 100, 1000, ... set dtick to 1. To set tick marks at 1, 100, 10000, ... set dtick to 2. To set tick marks at 1, 5, 25, 125, 625, 3125, ... set dtick to log_10(5), or 0.69897000433. "log" has several special values; "L&lt;f&gt;", where `f` is a positive number, gives ticks linearly spaced in value (but not position). For example `tick0` = 0.1, `dtick` = "L0.5" will put ticks at 0.1, 0.6, 1.1, 1.6 etc. To show powers of 10 plus small digits between, use "D1" (all digits) or "D2" (only 2 and 5). `tick0` is ignored for "D1" and "D2". If the axis `type` is "date", then you must convert the time to milliseconds. For example, to set the interval between ticks to one day, set `dtick` to 86400000.0. "date" also has special values "M&lt;n&gt;" gives ticks spaced by a number of months. `n` must be a positive integer. To set ticks on the 15th of every third month, set `tick0` to "2000-01-15" and `dtick` to "M3". To set ticks every 4 years, set `dtick` to "M48"
            </p>
          </div>
        </details>
        <details>
          <summary>
            exponentformat
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "none" | "e" | "E" | "power" | "SI" | "B" )</dd>
              <dt>Default:</dt>
              <dd>"B"</dd>
            </dl>
            <p>
              Determines a formatting rule for the tick exponents. For example, consider the number 1,000,000,000. If "none", it appears as 1,000,000,000. If "e", 1e+9. If "E", 1E+9. If "power", 1x10^9 (with 9 in a super script). If "SI", 1G. If "B", 1B.
            </p>
          </div>
        </details>
        <details>
          <summary>
            fixedrange
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Determines whether or not this axis is zoom-able. If true, then zoom is disabled.
            </p>
          </div>
        </details>
        <details>
          <summary>
            gridcolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>color</dd>
              <dt>Default:</dt>
              <dd>"#eee"</dd>
            </dl>
            <p>
              Sets the color of the grid lines.
            </p>
          </div>
        </details>
        <details>
          <summary>
            gridwidth
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>1</dd>
            </dl>
            <p>
              Sets the width (in px) of the grid lines.
            </p>
          </div>
        </details>
        <details>
          <summary>
            hoverformat
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>string</dd>
              <dt>Default:</dt>
              <dd>""</dd>
            </dl>
            <p>
              Sets the hover text formatting rule using d3 formatting mini-languages which are very similar to those in Python. For numbers, see: https://github.com/d3/d3-3.x-api-reference/blob/master/Formatting.md#d3_format And for dates see: https://github.com/d3/d3-3.x-api-reference/blob/master/Time-Formatting.md#format We add one item to d3's date formatter: "%{n}f" for fractional seconds with n digits. For example, "2016-10-13 09:15:23.456" with tickformat "%H~%\M~%S.%2f" would display "09~15~23.46"
            </p>
          </div>
        </details>
        <details>
          <summary>
            layer
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "above traces" | "below traces" )</dd>
              <dt>Default:</dt>
              <dd>"above traces"</dd>
            </dl>
            <p>
              Sets the layer on which this axis is displayed. If "above traces", this axis is displayed above all the subplot's traces If "below traces", this axis is displayed below all the subplot's traces, but above the grid lines. Useful when used together with scatter-like traces with `cliponaxis` set to "false" to show markers and/or text nodes above this axis.
            </p>
          </div>
        </details>
        <details>
          <summary>
            linecolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>color</dd>
              <dt>Default:</dt>
              <dd>"#444"</dd>
            </dl>
            <p>
              Sets the axis line color.
            </p>
          </div>
        </details>
        <details>
          <summary>
            linewidth
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>1</dd>
            </dl>
            <p>
              Sets the width (in px) of the axis line.
            </p>
          </div>
        </details>
        <details>
          <summary>
            matches
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "/^x([2-9]|[1-9][0-9]+)?$/" | "/^y([2-9]|[1-9][0-9]+)?$/" )</dd>
            </dl>
            <p>
              If set to another axis id (e.g. `x2`, `y`), the range of this axis will match the range of the corresponding axis in data-coordinates space. Moreover, matching axes share auto-range values, category lists and histogram auto-bins. Note that setting axes simultaneously in both a `scaleanchor` and a `matches` constraint is currently forbidden. Moreover, note that matching axes must have the same `type`.
            </p>
          </div>
        </details>
        <details>
          <summary>
            mirror
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( true | "ticks" | false | "all" | "allticks" )</dd>
            </dl>
            <p>
              Determines if the axis lines or/and ticks are mirrored to the opposite side of the plotting area. If "true", the axis lines are mirrored. If "ticks", the axis lines and ticks are mirrored. If "false", mirroring is disable. If "all", axis lines are mirrored on all shared-axes subplots. If "allticks", axis lines and ticks are mirrored on all shared-axes subplots.
            </p>
          </div>
        </details>
        <details>
          <summary>
            nticks
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>integer greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
            <p>
              Specifies the maximum number of ticks for the particular axis. The actual number of ticks will be chosen automatically to be less than or equal to `nticks`. Has an effect only if `tickmode` is set to "auto".
            </p>
          </div>
        </details>
        <details>
          <summary>
            overlaying
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "free" | "/^x([2-9]|[1-9][0-9]+)?$/" | "/^y([2-9]|[1-9][0-9]+)?$/" )</dd>
            </dl>
            <p>
              If set a same-letter axis id, this axis is overlaid on top of the corresponding same-letter axis, with traces and axes visible for both axes. If "false", this axis does not overlay any same-letter axes. In this case, for axes with overlapping domains only the highest-numbered axis will be visible.
            </p>
          </div>
        </details>
        <details>
          <summary>
            position
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>number between or equal to 0 and 1</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
            <p>
              Sets the position of this axis in the plotting space (in normalized coordinates). Only has an effect if `anchor` is set to "free".
            </p>
          </div>
        </details>
        <details>
          <summary>
            range
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>array</dd>
            </dl>
            <p>
              Sets the range of this axis. If the axis `type` is "log", then you must take the log of your desired range (e.g. to set the range from 1 to 100, set the range from 0 to 2). If the axis `type` is "date", it should be date strings, like date data, though Date objects and unix milliseconds will be accepted and converted to strings. If the axis `type` is "category", it should be numbers, using the scale where each category is assigned a serial number from zero in the order it appears.
            </p>
          </div>
        </details>
        <details>
          <summary>
            rangemode
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "normal" | "tozero" | "nonnegative" )</dd>
              <dt>Default:</dt>
              <dd>"normal"</dd>
            </dl>
            <p>
              If "normal", the range is computed in relation to the extrema of the input data. If "tozero"`, the range extends to 0, regardless of the input data If "nonnegative", the range is non-negative, regardless of the input data. Applies only to linear axes.
            </p>
          </div>
        </details>
        <details>
          <summary>
            rangeselector
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>true</dd>
            </dl>
            <p>
              Determines whether or not this range selector is visible. Note that range selectors are only available for x axes of `type` set to or auto-typed to "date".
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              activecolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.xaxis.rangeselector</dd>
                <dt>Type:</dt>
                <dd>color</dd>
              </dl>
              <p>
                Sets the background color of the active range selector button.
              </p>
            </div>
          </details>
          <details>
            <summary>
              bgcolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.xaxis.rangeselector</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#eee"</dd>
              </dl>
              <p>
                Sets the background color of the range selector buttons.
              </p>
            </div>
          </details>
          <details>
            <summary>
              bordercolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.xaxis.rangeselector</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets the color of the border enclosing the range selector.
              </p>
            </div>
          </details>
          <details>
            <summary>
              borderwidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.xaxis.rangeselector</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                Sets the width (in px) of the border enclosing the range selector.
              </p>
            </div>
          </details>
          <details>
            <summary>
              buttons
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.xaxis.rangeselector</dd>
                <dt>Type:</dt>
                <dd>array of object where each object has one or more of the keys listed below.</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not this button is visible.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                count
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.xaxis.rangeselector.buttons[]</dd>
                  <dt>Type:</dt>
                  <dd>number greater than or equal to 0</dd>
                  <dt>Default:</dt>
                  <dd>1</dd>
                </dl>
                <p>
                  Sets the number of steps to take to update the range. Use with `step` to specify the update interval.
                </p>
              </div>
            </details>
            <details>
              <summary>
                label
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.xaxis.rangeselector.buttons[]</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  Sets the text label to appear on the button.
                </p>
              </div>
            </details>
            <details>
              <summary>
                name
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.xaxis.rangeselector.buttons[]</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  When used in a template, named items are created in the output figure in addition to any items the figure already has in this array. You can modify these items in the output figure by making your own item with `templateitemname` matching this `name` alongside your modifications (including `visible: false` or `enabled: false` to hide it). Has no effect outside of a template.
                </p>
              </div>
            </details>
            <details>
              <summary>
                step
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.xaxis.rangeselector.buttons[]</dd>
                  <dt>Type:</dt>
                  <dd>enumerated , one of ( "month" | "year" | "day" | "hour" | "minute" | "second" | "all" )</dd>
                  <dt>Default:</dt>
                  <dd>"month"</dd>
                </dl>
                <p>
                  The unit of measurement that the `count` value will set the range by.
                </p>
              </div>
            </details>
            <details>
              <summary>
                stepmode
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.xaxis.rangeselector.buttons[]</dd>
                  <dt>Type:</dt>
                  <dd>enumerated , one of ( "backward" | "todate" )</dd>
                  <dt>Default:</dt>
                  <dd>"backward"</dd>
                </dl>
                <p>
                  Sets the range update mode. If "backward", the range update shifts the start of range back "count" times "step" milliseconds. If "todate", the range update shifts the start of range back to the first timestamp from "count" times "step" milliseconds back. For example, with `step` set to "year" and `count` set to "1" the range update shifts the start of the range back to January 01 of the current year. Month and year "todate" are currently available only for the built-in (Gregorian) calendar.
                </p>
              </div>
            </details>
            <details>
              <summary>
                templateitemname
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.xaxis.rangeselector.buttons[]</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  Used to refer to a named item in this array in the template. Named items from the template will be created even without a matching item in the input figure, but you can modify one by making an item with `templateitemname` matching its `name`, alongside your modifications (including `visible: false` or `enabled: false` to hide it). If there is no template or no matching item, this item will be hidden unless you explicitly show it with `visible: true`.
                </p>
              </div>
            </details>
            <details>
              <summary>
                visible
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.xaxis.rangeselector.buttons[]</dd>
                  <dt>Type:</dt>
                  <dd>boolean</dd>
                  <dt>Default:</dt>
                  <dd>true</dd>
                </dl>
                <p>
                  Determines whether or not this button is visible.
                </p>
              </div>
            </details>
          </div>
          <details>
            <summary>
              font
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.xaxis.rangeselector</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
              </dl>
              <p>
                Sets the font of the range selector button text.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                color
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.xaxis.rangeselector.font</dd>
                  <dt>Type:</dt>
                  <dd>color</dd>
                </dl>
              </div>
            </details>
            <details>
              <summary>
                family
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.xaxis.rangeselector.font</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
                </p>
              </div>
            </details>
            <details>
              <summary>
                size
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.xaxis.rangeselector.font</dd>
                  <dt>Type:</dt>
                  <dd>number greater than or equal to 1</dd>
                </dl>
              </div>
            </details>
          </div>
          <details>
            <summary>
              visible
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.xaxis.rangeselector</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
              </dl>
              <p>
                Determines whether or not this range selector is visible. Note that range selectors are only available for x axes of `type` set to or auto-typed to "date".
              </p>
            </div>
          </details>
          <details>
            <summary>
              x
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.xaxis.rangeselector</dd>
                <dt>Type:</dt>
                <dd>number between or equal to -2 and 3</dd>
              </dl>
              <p>
                Sets the x position (in normalized coordinates) of the range selector.
              </p>
            </div>
          </details>
          <details>
            <summary>
              xanchor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.xaxis.rangeselector</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "auto" | "left" | "center" | "right" )</dd>
                <dt>Default:</dt>
                <dd>"left"</dd>
              </dl>
              <p>
                Sets the range selector's horizontal position anchor. This anchor binds the `x` position to the "left", "center" or "right" of the range selector.
              </p>
            </div>
          </details>
          <details>
            <summary>
              y
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.xaxis.rangeselector</dd>
                <dt>Type:</dt>
                <dd>number between or equal to -2 and 3</dd>
              </dl>
              <p>
                Sets the y position (in normalized coordinates) of the range selector.
              </p>
            </div>
          </details>
          <details>
            <summary>
              yanchor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.xaxis.rangeselector</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "auto" | "top" | "middle" | "bottom" )</dd>
                <dt>Default:</dt>
                <dd>"bottom"</dd>
              </dl>
              <p>
                Sets the range selector's vertical position anchor This anchor binds the `y` position to the "top", "middle" or "bottom" of the range selector.
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            rangeslider
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>"#fff"</dd>
            </dl>
            <p>
              Sets the background color of the range slider.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              autorange
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.xaxis.rangeslider</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not the range slider range is computed in relation to the input data. If `range` is provided, then `autorange` is set to "false".
              </p>
            </div>
          </details>
          <details>
            <summary>
              bgcolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.xaxis.rangeslider</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#fff"</dd>
              </dl>
              <p>
                Sets the background color of the range slider.
              </p>
            </div>
          </details>
          <details>
            <summary>
              bordercolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.xaxis.rangeslider</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets the border color of the range slider.
              </p>
            </div>
          </details>
          <details>
            <summary>
              borderwidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.xaxis.rangeslider</dd>
                <dt>Type:</dt>
                <dd>integer greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                Sets the border width of the range slider.
              </p>
            </div>
          </details>
          <details>
            <summary>
              range
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.xaxis.rangeslider</dd>
                <dt>Type:</dt>
                <dd>array</dd>
              </dl>
              <p>
                Sets the range of the range slider. If not set, defaults to the full xaxis range. If the axis `type` is "log", then you must take the log of your desired range. If the axis `type` is "date", it should be date strings, like date data, though Date objects and unix milliseconds will be accepted and converted to strings. If the axis `type` is "category", it should be numbers, using the scale where each category is assigned a serial number from zero in the order it appears.
              </p>
            </div>
          </details>
          <details>
            <summary>
              thickness
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.xaxis.rangeslider</dd>
                <dt>Type:</dt>
                <dd>number between or equal to 0 and 1</dd>
                <dt>Default:</dt>
                <dd>0.15</dd>
              </dl>
              <p>
                The height of the range slider as a fraction of the total plot area height.
              </p>
            </div>
          </details>
          <details>
            <summary>
              visible
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.xaxis.rangeslider</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not the range slider will be visible. If visible, perpendicular axes will be set to `fixedrange`
              </p>
            </div>
          </details>
          <details>
            <summary>
              yaxis
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.xaxis.rangeslider</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
                <dt>Default:</dt>
                <dd>"match"</dd>
              </dl>
              <p>
                Determines whether or not the range of this axis in the rangeslider use the same value than in the main plot when zooming in/out. If "auto", the autorange will be used. If "fixed", the `range` is used. If "match", the current range of the corresponding y-axis on the main subplot is used.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                range
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.xaxis.rangeslider.yaxis</dd>
                  <dt>Type:</dt>
                  <dd>array</dd>
                </dl>
                <p>
                  Sets the range of this axis for the rangeslider.
                </p>
              </div>
            </details>
            <details>
              <summary>
                rangemode
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.xaxis.rangeslider.yaxis</dd>
                  <dt>Type:</dt>
                  <dd>enumerated , one of ( "auto" | "fixed" | "match" )</dd>
                  <dt>Default:</dt>
                  <dd>"match"</dd>
                </dl>
                <p>
                  Determines whether or not the range of this axis in the rangeslider use the same value than in the main plot when zooming in/out. If "auto", the autorange will be used. If "fixed", the `range` is used. If "match", the current range of the corresponding y-axis on the main subplot is used.
                </p>
              </div>
            </details>
          </div>
        </div>
        <details>
          <summary>
            scaleanchor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "/^x([2-9]|[1-9][0-9]+)?$/" | "/^y([2-9]|[1-9][0-9]+)?$/" )</dd>
            </dl>
            <p>
              If set to another axis id (e.g. `x2`, `y`), the range of this axis changes together with the range of the corresponding axis such that the scale of pixels per unit is in a constant ratio. Both axes are still zoomable, but when you zoom one, the other will zoom the same amount, keeping a fixed midpoint. `constrain` and `constraintoward` determine how we enforce the constraint. You can chain these, ie `yaxis: {scaleanchor: "x"}, xaxis2: {scaleanchor: "y"}` but you can only link axes of the same `type`. The linked axis can have the opposite letter (to constrain the aspect ratio) or the same letter (to match scales across subplots). Loops (`yaxis: {scaleanchor: "x"}, xaxis: {scaleanchor: "y"}` or longer) are redundant and the last constraint encountered will be ignored to avoid possible inconsistent constraints via `scaleratio`. Note that setting axes simultaneously in both a `scaleanchor` and a `matches` constraint is currently forbidden.
            </p>
          </div>
        </details>
        <details>
          <summary>
            scaleratio
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>1</dd>
            </dl>
            <p>
              If this axis is linked to another by `scaleanchor`, this determines the pixel to unit scale ratio. For example, if this value is 10, then every unit on this axis spans 10 times the number of pixels as a unit on the linked axis. Use this for example to create an elevation profile where the vertical scale is exaggerated a fixed amount with respect to the horizontal.
            </p>
          </div>
        </details>
        <details>
          <summary>
            separatethousands
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              If "true", even 4-digit integers are separated
            </p>
          </div>
        </details>
        <details>
          <summary>
            showdividers
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
              <dt>Default:</dt>
              <dd>true</dd>
            </dl>
            <p>
              Determines whether or not a dividers are drawn between the category levels of this axis. Only has an effect on "multicategory" axes.
            </p>
          </div>
        </details>
        <details>
          <summary>
            showexponent
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
              <dt>Default:</dt>
              <dd>"all"</dd>
            </dl>
            <p>
              If "all", all exponents are shown besides their significands. If "first", only the exponent of the first tick is shown. If "last", only the exponent of the last tick is shown. If "none", no exponents appear.
            </p>
          </div>
        </details>
        <details>
          <summary>
            showgrid
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Determines whether or not grid lines are drawn. If "true", the grid lines are drawn at every tick mark.
            </p>
          </div>
        </details>
        <details>
          <summary>
            showline
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Determines whether or not a line bounding this axis is drawn.
            </p>
          </div>
        </details>
        <details>
          <summary>
            showspikes
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Determines whether or not spikes (aka droplines) are drawn for this axis. Note: This only takes affect when hovermode = closest
            </p>
          </div>
        </details>
        <details>
          <summary>
            showticklabels
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
              <dt>Default:</dt>
              <dd>true</dd>
            </dl>
            <p>
              Determines whether or not the tick labels are drawn.
            </p>
          </div>
        </details>
        <details>
          <summary>
            showtickprefix
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
              <dt>Default:</dt>
              <dd>"all"</dd>
            </dl>
            <p>
              If "all", all tick labels are displayed with a prefix. If "first", only the first tick is displayed with a prefix. If "last", only the last tick is displayed with a suffix. If "none", tick prefixes are hidden.
            </p>
          </div>
        </details>
        <details>
          <summary>
            showticksuffix
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
              <dt>Default:</dt>
              <dd>"all"</dd>
            </dl>
            <p>
              Same as `showtickprefix` but for tick suffixes.
            </p>
          </div>
        </details>
        <details>
          <summary>
            side
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "top" | "bottom" | "left" | "right" )</dd>
            </dl>
            <p>
              Determines whether a x (y) axis is positioned at the "bottom" ("left") or "top" ("right") of the plotting area.
            </p>
          </div>
        </details>
        <details>
          <summary>
            spikecolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>color</dd>
            </dl>
            <p>
              Sets the spike color. If undefined, will use the series color
            </p>
          </div>
        </details>
        <details>
          <summary>
            spikedash
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>string</dd>
              <dt>Default:</dt>
              <dd>"dash"</dd>
            </dl>
            <p>
              Sets the dash style of lines. Set to a dash type string ("solid", "dot", "dash", "longdash", "dashdot", or "longdashdot") or a dash length list in px (eg "5px,10px,2px,2px").
            </p>
          </div>
        </details>
        <details>
          <summary>
            spikemode
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>flaglist string. Any combination of "toaxis", "across", "marker" joined with a "+"</dd>
              <dt>Default:</dt>
              <dd>"toaxis"</dd>
            </dl>
            <p>
              Determines the drawing mode for the spike line If "toaxis", the line is drawn from the data point to the axis the series is plotted on. If "across", the line is drawn across the entire plot area, and supercedes "toaxis". If "marker", then a marker dot is drawn on the axis the series is plotted on
            </p>
          </div>
        </details>
        <details>
          <summary>
            spikesnap
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "data" | "cursor" )</dd>
              <dt>Default:</dt>
              <dd>"data"</dd>
            </dl>
            <p>
              Determines whether spikelines are stuck to the cursor or to the closest datapoints.
            </p>
          </div>
        </details>
        <details>
          <summary>
            spikethickness
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>number</dd>
              <dt>Default:</dt>
              <dd>3</dd>
            </dl>
            <p>
              Sets the width (in px) of the zero line.
            </p>
          </div>
        </details>
        <details>
          <summary>
            tick0
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>number or categorical coordinate string</dd>
            </dl>
            <p>
              Sets the placement of the first tick on this axis. Use with `dtick`. If the axis `type` is "log", then you must take the log of your starting tick (e.g. to set the starting tick to 100, set the `tick0` to 2) except when `dtick`="L&lt;f&gt;" (see `dtick` for more info). If the axis `type` is "date", it should be a date string, like date data. If the axis `type` is "category", it should be a number, using the scale where each category is assigned a serial number from zero in the order it appears.
            </p>
          </div>
        </details>
        <details>
          <summary>
            tickangle
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>angle</dd>
              <dt>Default:</dt>
              <dd>"auto"</dd>
            </dl>
            <p>
              Sets the angle of the tick labels with respect to the horizontal. For example, a `tickangle` of -90 draws the tick labels vertically.
            </p>
          </div>
        </details>
        <details>
          <summary>
            tickcolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>color</dd>
              <dt>Default:</dt>
              <dd>"#444"</dd>
            </dl>
            <p>
              Sets the tick color.
            </p>
          </div>
        </details>
        <details>
          <summary>
            tickfont
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
            </dl>
            <p>
              Sets the tick font.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              color
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.xaxis.tickfont</dd>
                <dt>Type:</dt>
                <dd>color</dd>
              </dl>
            </div>
          </details>
          <details>
            <summary>
              family
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.xaxis.tickfont</dd>
                <dt>Type:</dt>
                <dd>string</dd>
              </dl>
              <p>
                HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
              </p>
            </div>
          </details>
          <details>
            <summary>
              size
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.xaxis.tickfont</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 1</dd>
              </dl>
            </div>
          </details>
        </div>
        <details>
          <summary>
            tickformat
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>string</dd>
              <dt>Default:</dt>
              <dd>""</dd>
            </dl>
            <p>
              Sets the tick label formatting rule using d3 formatting mini-languages which are very similar to those in Python. For numbers, see: https://github.com/d3/d3-3.x-api-reference/blob/master/Formatting.md#d3_format And for dates see: https://github.com/d3/d3-3.x-api-reference/blob/master/Time-Formatting.md#format We add one item to d3's date formatter: "%{n}f" for fractional seconds with n digits. For example, "2016-10-13 09:15:23.456" with tickformat "%H~%\M~%S.%2f" would display "09~15~23.46"
            </p>
          </div>
        </details>
        <details>
          <summary>
            tickformatstops
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>array of object where each object has one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>true</dd>
            </dl>
            <p>
              Determines whether or not this stop is used. If `false`, this stop is ignored even within its `dtickrange`.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              dtickrange
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.xaxis.tickformatstops[]</dd>
                <dt>Type:</dt>
                <dd>array</dd>
              </dl>
              <p>
                range ["min", "max"], where "min", "max" - dtick values which describe some zoom level, it is possible to omit "min" or "max" value by passing "null"
              </p>
            </div>
          </details>
          <details>
            <summary>
              enabled
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.xaxis.tickformatstops[]</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not this stop is used. If `false`, this stop is ignored even within its `dtickrange`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              name
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.xaxis.tickformatstops[]</dd>
                <dt>Type:</dt>
                <dd>string</dd>
              </dl>
              <p>
                When used in a template, named items are created in the output figure in addition to any items the figure already has in this array. You can modify these items in the output figure by making your own item with `templateitemname` matching this `name` alongside your modifications (including `visible: false` or `enabled: false` to hide it). Has no effect outside of a template.
              </p>
            </div>
          </details>
          <details>
            <summary>
              templateitemname
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.xaxis.tickformatstops[]</dd>
                <dt>Type:</dt>
                <dd>string</dd>
              </dl>
              <p>
                Used to refer to a named item in this array in the template. Named items from the template will be created even without a matching item in the input figure, but you can modify one by making an item with `templateitemname` matching its `name`, alongside your modifications (including `visible: false` or `enabled: false` to hide it). If there is no template or no matching item, this item will be hidden unless you explicitly show it with `visible: true`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              value
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.xaxis.tickformatstops[]</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                string - dtickformat for described zoom level, the same as "tickformat"
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            ticklen
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>5</dd>
            </dl>
            <p>
              Sets the tick length (in px).
            </p>
          </div>
        </details>
        <details>
          <summary>
            tickmode
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "auto" | "linear" | "array" )</dd>
            </dl>
            <p>
              Sets the tick mode for this axis. If "auto", the number of ticks is set via `nticks`. If "linear", the placement of the ticks is determined by a starting position `tick0` and a tick step `dtick` ("linear" is the default value if `tick0` and `dtick` are provided). If "array", the placement of the ticks is set via `tickvals` and the tick text is `ticktext`. ("array" is the default value if `tickvals` is provided).
            </p>
          </div>
        </details>
        <details>
          <summary>
            tickprefix
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>string</dd>
              <dt>Default:</dt>
              <dd>""</dd>
            </dl>
            <p>
              Sets a tick label prefix.
            </p>
          </div>
        </details>
        <details>
          <summary>
            ticks
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "outside" | "inside" | "" )</dd>
            </dl>
            <p>
              Determines whether ticks are drawn or not. If "", this axis' ticks are not drawn. If "outside" ("inside"), this axis' are drawn outside (inside) the axis lines.
            </p>
          </div>
        </details>
        <details>
          <summary>
            tickson
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "labels" | "boundaries" )</dd>
              <dt>Default:</dt>
              <dd>"labels"</dd>
            </dl>
            <p>
              Determines where ticks and grid lines are drawn with respect to their corresponding tick labels. Only has an effect for axes of `type` "category" or "multicategory". When set to "boundaries", ticks and grid lines are drawn half a category to the left/bottom of labels.
            </p>
          </div>
        </details>
        <details>
          <summary>
            ticksuffix
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>string</dd>
              <dt>Default:</dt>
              <dd>""</dd>
            </dl>
            <p>
              Sets a tick label suffix.
            </p>
          </div>
        </details>
        <details>
          <summary>
            ticktext
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>data array</dd>
            </dl>
            <p>
              Sets the text displayed at the ticks position via `tickvals`. Only has an effect if `tickmode` is set to "array". Used with `tickvals`.
            </p>
          </div>
        </details>
        <details>
          <summary>
            tickvals
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>data array</dd>
            </dl>
            <p>
              Sets the values at which ticks on this axis appear. Only has an effect if `tickmode` is set to "array". Used with `ticktext`.
            </p>
          </div>
        </details>
        <details>
          <summary>
            tickwidth
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>1</dd>
            </dl>
            <p>
              Sets the tick width (in px).
            </p>
          </div>
        </details>
        <details>
          <summary>
            title
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
            </dl>
            <p>
              Sets the title of this axis. Note that before the existence of `title.text`, the title's contents used to be defined as the `title` attribute itself. This behavior has been deprecated.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              font
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.xaxis.title</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
              </dl>
              <p>
                Sets this axis' title font. Note that the title's font used to be customized by the now deprecated `titlefont` attribute.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                color
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.xaxis.title.font</dd>
                  <dt>Type:</dt>
                  <dd>color</dd>
                </dl>
              </div>
            </details>
            <details>
              <summary>
                family
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.xaxis.title.font</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
                </p>
              </div>
            </details>
            <details>
              <summary>
                size
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.xaxis.title.font</dd>
                  <dt>Type:</dt>
                  <dd>number greater than or equal to 1</dd>
                </dl>
              </div>
            </details>
          </div>
          <details>
            <summary>
              standoff
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.xaxis.title</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
              </dl>
              <p>
                Sets the standoff distance (in px) between the axis labels and the title text The default value is a function of the axis tick labels, the title `font.size` and the axis `linewidth`. Note that the axis title position is always constrained within the margins, so the actual standoff distance is always less than the set or default value. By setting `standoff` and turning on `automargin`, plotly.js will push the margins to fit the axis title at given standoff distance.
              </p>
            </div>
          </details>
          <details>
            <summary>
              text
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.xaxis.title</dd>
                <dt>Type:</dt>
                <dd>string</dd>
              </dl>
              <p>
                Sets the title of this axis. Note that before the existence of `title.text`, the title's contents used to be defined as the `title` attribute itself. This behavior has been deprecated.
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            type
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "-" | "linear" | "log" | "date" | "category" | "multicategory" )</dd>
              <dt>Default:</dt>
              <dd>"-"</dd>
            </dl>
            <p>
              Sets the axis type. By default, plotly attempts to determined the axis type by looking into the data of the traces that referenced the axis in question.
            </p>
          </div>
        </details>
        <details>
          <summary>
            uirevision
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>number or categorical coordinate string</dd>
            </dl>
            <p>
              Controls persistence of user-driven changes in axis `range`, `autorange`, and `title` if in `editable: true` configuration. Defaults to `layout.uirevision`.
            </p>
          </div>
        </details>
        <details>
          <summary>
            visible
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              A single toggle to hide the axis while preserving interaction like dragging. Default is true when a cheater plot is present on the axis, otherwise false
            </p>
          </div>
        </details>
        <details>
          <summary>
            zeroline
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Determines whether or not a line is drawn at along the 0 value of this axis. If "true", the zero line is drawn on top of the grid lines.
            </p>
          </div>
        </details>
        <details>
          <summary>
            zerolinecolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>color</dd>
              <dt>Default:</dt>
              <dd>"#444"</dd>
            </dl>
            <p>
              Sets the line color of the zero line.
            </p>
          </div>
        </details>
        <details>
          <summary>
            zerolinewidth
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.xaxis</dd>
              <dt>Type:</dt>
              <dd>number</dd>
              <dt>Default:</dt>
              <dd>1</dd>
            </dl>
            <p>
              Sets the width (in px) of the zero line.
            </p>
          </div>
        </details>
      </div>
      <details>
        <summary>
          yaxis
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>layout</dd>
            <dt>Type:</dt>
            <dd>object containing one or more of the keys listed below.</dd>
            <dt>Default:</dt>
            <dd>"#444"</dd>
          </dl>
          <p>
            A single toggle to hide the axis while preserving interaction like dragging. Default is true when a cheater plot is present on the axis, otherwise false
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            anchor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "free" | "/^x([2-9]|[1-9][0-9]+)?$/" | "/^y([2-9]|[1-9][0-9]+)?$/" )</dd>
            </dl>
            <p>
              If set to an opposite-letter axis id (e.g. `x2`, `y`), this axis is bound to the corresponding opposite-letter axis. If set to "free", this axis' position is determined by `position`.
            </p>
          </div>
        </details>
        <details>
          <summary>
            automargin
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Determines whether long tick labels automatically grow the figure margins.
            </p>
          </div>
        </details>
        <details>
          <summary>
            autorange
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( true | false | "reversed" )</dd>
              <dt>Default:</dt>
              <dd>true</dd>
            </dl>
            <p>
              Determines whether or not the range of this axis is computed in relation to the input data. See `rangemode` for more info. If `range` is provided, then `autorange` is set to "false".
            </p>
          </div>
        </details>
        <details>
          <summary>
            calendar
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "gregorian" | "chinese" | "coptic" | "discworld" | "ethiopian" | "hebrew" | "islamic" | "julian" | "mayan" | "nanakshahi" | "nepali" | "persian" | "jalali" | "taiwan" | "thai" | "ummalqura" )</dd>
              <dt>Default:</dt>
              <dd>"gregorian"</dd>
            </dl>
            <p>
              Sets the calendar system to use for `range` and `tick0` if this is a date axis. This does not set the calendar for interpreting data on this axis, that's specified in the trace or via the global `layout.calendar`
            </p>
          </div>
        </details>
        <details>
          <summary>
            categoryarray
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>data array</dd>
            </dl>
            <p>
              Sets the order in which categories on this axis appear. Only has an effect if `categoryorder` is set to "array". Used with `categoryorder`.
            </p>
          </div>
        </details>
        <details>
          <summary>
            categoryorder
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "trace" | "category ascending" | "category descending" | "array" | "total ascending" | "total descending" | "min ascending" | "min descending" | "max ascending" | "max descending" | "sum ascending" | "sum descending" | "mean ascending" | "mean descending" | "median ascending" | "median descending" )</dd>
              <dt>Default:</dt>
              <dd>"trace"</dd>
            </dl>
            <p>
              Specifies the ordering logic for the case of categorical variables. By default, plotly uses "trace", which specifies the order that is present in the data supplied. Set `categoryorder` to "category ascending" or "category descending" if order should be determined by the alphanumerical order of the category names. Set `categoryorder` to "array" to derive the ordering from the attribute `categoryarray`. If a category is not found in the `categoryarray` array, the sorting behavior for that attribute will be identical to the "trace" mode. The unspecified categories will follow the categories in `categoryarray`. Set `categoryorder` to "total ascending" or "total descending" if order should be determined by the numerical order of the values. Similarly, the order can be determined by the min, max, sum, mean or median of all the values.
            </p>
          </div>
        </details>
        <details>
          <summary>
            color
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>color</dd>
              <dt>Default:</dt>
              <dd>"#444"</dd>
            </dl>
            <p>
              Sets default for all colors associated with this axis all at once: line, font, tick, and grid colors. Grid color is lightened by blending this with the plot background Individual pieces can override this.
            </p>
          </div>
        </details>
        <details>
          <summary>
            constrain
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "range" | "domain" )</dd>
              <dt>Default:</dt>
              <dd>"range"</dd>
            </dl>
            <p>
              If this axis needs to be compressed (either due to its own `scaleanchor` and `scaleratio` or those of the other axis), determines how that happens: by increasing the "range" (default), or by decreasing the "domain".
            </p>
          </div>
        </details>
        <details>
          <summary>
            constraintoward
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "left" | "center" | "right" | "top" | "middle" | "bottom" )</dd>
            </dl>
            <p>
              If this axis needs to be compressed (either due to its own `scaleanchor` and `scaleratio` or those of the other axis), determines which direction we push the originally specified plot area. Options are "left", "center" (default), and "right" for x axes, and "top", "middle" (default), and "bottom" for y axes.
            </p>
          </div>
        </details>
        <details>
          <summary>
            dividercolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>color</dd>
              <dt>Default:</dt>
              <dd>"#444"</dd>
            </dl>
            <p>
              Sets the color of the dividers Only has an effect on "multicategory" axes.
            </p>
          </div>
        </details>
        <details>
          <summary>
            dividerwidth
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>number</dd>
              <dt>Default:</dt>
              <dd>1</dd>
            </dl>
            <p>
              Sets the width (in px) of the dividers Only has an effect on "multicategory" axes.
            </p>
          </div>
        </details>
        <details>
          <summary>
            domain
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>array</dd>
              <dt>Default:</dt>
              <dd>[0, 1]</dd>
            </dl>
            <p>
              Sets the domain of this axis (in plot fraction).
            </p>
          </div>
        </details>
        <details>
          <summary>
            dtick
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>number or categorical coordinate string</dd>
            </dl>
            <p>
              Sets the step in-between ticks on this axis. Use with `tick0`. Must be a positive number, or special strings available to "log" and "date" axes. If the axis `type` is "log", then ticks are set every 10^(n"dtick) where n is the tick number. For example, to set a tick mark at 1, 10, 100, 1000, ... set dtick to 1. To set tick marks at 1, 100, 10000, ... set dtick to 2. To set tick marks at 1, 5, 25, 125, 625, 3125, ... set dtick to log_10(5), or 0.69897000433. "log" has several special values; "L&lt;f&gt;", where `f` is a positive number, gives ticks linearly spaced in value (but not position). For example `tick0` = 0.1, `dtick` = "L0.5" will put ticks at 0.1, 0.6, 1.1, 1.6 etc. To show powers of 10 plus small digits between, use "D1" (all digits) or "D2" (only 2 and 5). `tick0` is ignored for "D1" and "D2". If the axis `type` is "date", then you must convert the time to milliseconds. For example, to set the interval between ticks to one day, set `dtick` to 86400000.0. "date" also has special values "M&lt;n&gt;" gives ticks spaced by a number of months. `n` must be a positive integer. To set ticks on the 15th of every third month, set `tick0` to "2000-01-15" and `dtick` to "M3". To set ticks every 4 years, set `dtick` to "M48"
            </p>
          </div>
        </details>
        <details>
          <summary>
            exponentformat
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "none" | "e" | "E" | "power" | "SI" | "B" )</dd>
              <dt>Default:</dt>
              <dd>"B"</dd>
            </dl>
            <p>
              Determines a formatting rule for the tick exponents. For example, consider the number 1,000,000,000. If "none", it appears as 1,000,000,000. If "e", 1e+9. If "E", 1E+9. If "power", 1x10^9 (with 9 in a super script). If "SI", 1G. If "B", 1B.
            </p>
          </div>
        </details>
        <details>
          <summary>
            fixedrange
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Determines whether or not this axis is zoom-able. If true, then zoom is disabled.
            </p>
          </div>
        </details>
        <details>
          <summary>
            gridcolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>color</dd>
              <dt>Default:</dt>
              <dd>"#eee"</dd>
            </dl>
            <p>
              Sets the color of the grid lines.
            </p>
          </div>
        </details>
        <details>
          <summary>
            gridwidth
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>1</dd>
            </dl>
            <p>
              Sets the width (in px) of the grid lines.
            </p>
          </div>
        </details>
        <details>
          <summary>
            hoverformat
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>string</dd>
              <dt>Default:</dt>
              <dd>""</dd>
            </dl>
            <p>
              Sets the hover text formatting rule using d3 formatting mini-languages which are very similar to those in Python. For numbers, see: https://github.com/d3/d3-3.x-api-reference/blob/master/Formatting.md#d3_format And for dates see: https://github.com/d3/d3-3.x-api-reference/blob/master/Time-Formatting.md#format We add one item to d3's date formatter: "%{n}f" for fractional seconds with n digits. For example, "2016-10-13 09:15:23.456" with tickformat "%H~%\M~%S.%2f" would display "09~15~23.46"
            </p>
          </div>
        </details>
        <details>
          <summary>
            layer
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "above traces" | "below traces" )</dd>
              <dt>Default:</dt>
              <dd>"above traces"</dd>
            </dl>
            <p>
              Sets the layer on which this axis is displayed. If "above traces", this axis is displayed above all the subplot's traces If "below traces", this axis is displayed below all the subplot's traces, but above the grid lines. Useful when used together with scatter-like traces with `cliponaxis` set to "false" to show markers and/or text nodes above this axis.
            </p>
          </div>
        </details>
        <details>
          <summary>
            linecolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>color</dd>
              <dt>Default:</dt>
              <dd>"#444"</dd>
            </dl>
            <p>
              Sets the axis line color.
            </p>
          </div>
        </details>
        <details>
          <summary>
            linewidth
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>1</dd>
            </dl>
            <p>
              Sets the width (in px) of the axis line.
            </p>
          </div>
        </details>
        <details>
          <summary>
            matches
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "/^x([2-9]|[1-9][0-9]+)?$/" | "/^y([2-9]|[1-9][0-9]+)?$/" )</dd>
            </dl>
            <p>
              If set to another axis id (e.g. `x2`, `y`), the range of this axis will match the range of the corresponding axis in data-coordinates space. Moreover, matching axes share auto-range values, category lists and histogram auto-bins. Note that setting axes simultaneously in both a `scaleanchor` and a `matches` constraint is currently forbidden. Moreover, note that matching axes must have the same `type`.
            </p>
          </div>
        </details>
        <details>
          <summary>
            mirror
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( true | "ticks" | false | "all" | "allticks" )</dd>
            </dl>
            <p>
              Determines if the axis lines or/and ticks are mirrored to the opposite side of the plotting area. If "true", the axis lines are mirrored. If "ticks", the axis lines and ticks are mirrored. If "false", mirroring is disable. If "all", axis lines are mirrored on all shared-axes subplots. If "allticks", axis lines and ticks are mirrored on all shared-axes subplots.
            </p>
          </div>
        </details>
        <details>
          <summary>
            nticks
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>integer greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
            <p>
              Specifies the maximum number of ticks for the particular axis. The actual number of ticks will be chosen automatically to be less than or equal to `nticks`. Has an effect only if `tickmode` is set to "auto".
            </p>
          </div>
        </details>
        <details>
          <summary>
            overlaying
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "free" | "/^x([2-9]|[1-9][0-9]+)?$/" | "/^y([2-9]|[1-9][0-9]+)?$/" )</dd>
            </dl>
            <p>
              If set a same-letter axis id, this axis is overlaid on top of the corresponding same-letter axis, with traces and axes visible for both axes. If "false", this axis does not overlay any same-letter axes. In this case, for axes with overlapping domains only the highest-numbered axis will be visible.
            </p>
          </div>
        </details>
        <details>
          <summary>
            position
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>number between or equal to 0 and 1</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
            <p>
              Sets the position of this axis in the plotting space (in normalized coordinates). Only has an effect if `anchor` is set to "free".
            </p>
          </div>
        </details>
        <details>
          <summary>
            range
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>array</dd>
            </dl>
            <p>
              Sets the range of this axis. If the axis `type` is "log", then you must take the log of your desired range (e.g. to set the range from 1 to 100, set the range from 0 to 2). If the axis `type` is "date", it should be date strings, like date data, though Date objects and unix milliseconds will be accepted and converted to strings. If the axis `type` is "category", it should be numbers, using the scale where each category is assigned a serial number from zero in the order it appears.
            </p>
          </div>
        </details>
        <details>
          <summary>
            rangemode
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "normal" | "tozero" | "nonnegative" )</dd>
              <dt>Default:</dt>
              <dd>"normal"</dd>
            </dl>
            <p>
              If "normal", the range is computed in relation to the extrema of the input data. If "tozero"`, the range extends to 0, regardless of the input data If "nonnegative", the range is non-negative, regardless of the input data. Applies only to linear axes.
            </p>
          </div>
        </details>
        <details>
          <summary>
            scaleanchor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "/^x([2-9]|[1-9][0-9]+)?$/" | "/^y([2-9]|[1-9][0-9]+)?$/" )</dd>
            </dl>
            <p>
              If set to another axis id (e.g. `x2`, `y`), the range of this axis changes together with the range of the corresponding axis such that the scale of pixels per unit is in a constant ratio. Both axes are still zoomable, but when you zoom one, the other will zoom the same amount, keeping a fixed midpoint. `constrain` and `constraintoward` determine how we enforce the constraint. You can chain these, ie `yaxis: {scaleanchor: "x"}, xaxis2: {scaleanchor: "y"}` but you can only link axes of the same `type`. The linked axis can have the opposite letter (to constrain the aspect ratio) or the same letter (to match scales across subplots). Loops (`yaxis: {scaleanchor: "x"}, xaxis: {scaleanchor: "y"}` or longer) are redundant and the last constraint encountered will be ignored to avoid possible inconsistent constraints via `scaleratio`. Note that setting axes simultaneously in both a `scaleanchor` and a `matches` constraint is currently forbidden.
            </p>
          </div>
        </details>
        <details>
          <summary>
            scaleratio
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>1</dd>
            </dl>
            <p>
              If this axis is linked to another by `scaleanchor`, this determines the pixel to unit scale ratio. For example, if this value is 10, then every unit on this axis spans 10 times the number of pixels as a unit on the linked axis. Use this for example to create an elevation profile where the vertical scale is exaggerated a fixed amount with respect to the horizontal.
            </p>
          </div>
        </details>
        <details>
          <summary>
            separatethousands
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              If "true", even 4-digit integers are separated
            </p>
          </div>
        </details>
        <details>
          <summary>
            showdividers
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
              <dt>Default:</dt>
              <dd>true</dd>
            </dl>
            <p>
              Determines whether or not a dividers are drawn between the category levels of this axis. Only has an effect on "multicategory" axes.
            </p>
          </div>
        </details>
        <details>
          <summary>
            showexponent
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
              <dt>Default:</dt>
              <dd>"all"</dd>
            </dl>
            <p>
              If "all", all exponents are shown besides their significands. If "first", only the exponent of the first tick is shown. If "last", only the exponent of the last tick is shown. If "none", no exponents appear.
            </p>
          </div>
        </details>
        <details>
          <summary>
            showgrid
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Determines whether or not grid lines are drawn. If "true", the grid lines are drawn at every tick mark.
            </p>
          </div>
        </details>
        <details>
          <summary>
            showline
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Determines whether or not a line bounding this axis is drawn.
            </p>
          </div>
        </details>
        <details>
          <summary>
            showspikes
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Determines whether or not spikes (aka droplines) are drawn for this axis. Note: This only takes affect when hovermode = closest
            </p>
          </div>
        </details>
        <details>
          <summary>
            showticklabels
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
              <dt>Default:</dt>
              <dd>true</dd>
            </dl>
            <p>
              Determines whether or not the tick labels are drawn.
            </p>
          </div>
        </details>
        <details>
          <summary>
            showtickprefix
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
              <dt>Default:</dt>
              <dd>"all"</dd>
            </dl>
            <p>
              If "all", all tick labels are displayed with a prefix. If "first", only the first tick is displayed with a prefix. If "last", only the last tick is displayed with a suffix. If "none", tick prefixes are hidden.
            </p>
          </div>
        </details>
        <details>
          <summary>
            showticksuffix
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
              <dt>Default:</dt>
              <dd>"all"</dd>
            </dl>
            <p>
              Same as `showtickprefix` but for tick suffixes.
            </p>
          </div>
        </details>
        <details>
          <summary>
            side
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "top" | "bottom" | "left" | "right" )</dd>
            </dl>
            <p>
              Determines whether a x (y) axis is positioned at the "bottom" ("left") or "top" ("right") of the plotting area.
            </p>
          </div>
        </details>
        <details>
          <summary>
            spikecolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>color</dd>
            </dl>
            <p>
              Sets the spike color. If undefined, will use the series color
            </p>
          </div>
        </details>
        <details>
          <summary>
            spikedash
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>string</dd>
              <dt>Default:</dt>
              <dd>"dash"</dd>
            </dl>
            <p>
              Sets the dash style of lines. Set to a dash type string ("solid", "dot", "dash", "longdash", "dashdot", or "longdashdot") or a dash length list in px (eg "5px,10px,2px,2px").
            </p>
          </div>
        </details>
        <details>
          <summary>
            spikemode
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>flaglist string. Any combination of "toaxis", "across", "marker" joined with a "+"</dd>
              <dt>Default:</dt>
              <dd>"toaxis"</dd>
            </dl>
            <p>
              Determines the drawing mode for the spike line If "toaxis", the line is drawn from the data point to the axis the series is plotted on. If "across", the line is drawn across the entire plot area, and supercedes "toaxis". If "marker", then a marker dot is drawn on the axis the series is plotted on
            </p>
          </div>
        </details>
        <details>
          <summary>
            spikesnap
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "data" | "cursor" )</dd>
              <dt>Default:</dt>
              <dd>"data"</dd>
            </dl>
            <p>
              Determines whether spikelines are stuck to the cursor or to the closest datapoints.
            </p>
          </div>
        </details>
        <details>
          <summary>
            spikethickness
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>number</dd>
              <dt>Default:</dt>
              <dd>3</dd>
            </dl>
            <p>
              Sets the width (in px) of the zero line.
            </p>
          </div>
        </details>
        <details>
          <summary>
            tick0
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>number or categorical coordinate string</dd>
            </dl>
            <p>
              Sets the placement of the first tick on this axis. Use with `dtick`. If the axis `type` is "log", then you must take the log of your starting tick (e.g. to set the starting tick to 100, set the `tick0` to 2) except when `dtick`="L&lt;f&gt;" (see `dtick` for more info). If the axis `type` is "date", it should be a date string, like date data. If the axis `type` is "category", it should be a number, using the scale where each category is assigned a serial number from zero in the order it appears.
            </p>
          </div>
        </details>
        <details>
          <summary>
            tickangle
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>angle</dd>
              <dt>Default:</dt>
              <dd>"auto"</dd>
            </dl>
            <p>
              Sets the angle of the tick labels with respect to the horizontal. For example, a `tickangle` of -90 draws the tick labels vertically.
            </p>
          </div>
        </details>
        <details>
          <summary>
            tickcolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>color</dd>
              <dt>Default:</dt>
              <dd>"#444"</dd>
            </dl>
            <p>
              Sets the tick color.
            </p>
          </div>
        </details>
        <details>
          <summary>
            tickfont
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
            </dl>
            <p>
              Sets the tick font.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              color
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.yaxis.tickfont</dd>
                <dt>Type:</dt>
                <dd>color</dd>
              </dl>
            </div>
          </details>
          <details>
            <summary>
              family
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.yaxis.tickfont</dd>
                <dt>Type:</dt>
                <dd>string</dd>
              </dl>
              <p>
                HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
              </p>
            </div>
          </details>
          <details>
            <summary>
              size
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.yaxis.tickfont</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 1</dd>
              </dl>
            </div>
          </details>
        </div>
        <details>
          <summary>
            tickformat
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>string</dd>
              <dt>Default:</dt>
              <dd>""</dd>
            </dl>
            <p>
              Sets the tick label formatting rule using d3 formatting mini-languages which are very similar to those in Python. For numbers, see: https://github.com/d3/d3-3.x-api-reference/blob/master/Formatting.md#d3_format And for dates see: https://github.com/d3/d3-3.x-api-reference/blob/master/Time-Formatting.md#format We add one item to d3's date formatter: "%{n}f" for fractional seconds with n digits. For example, "2016-10-13 09:15:23.456" with tickformat "%H~%\M~%S.%2f" would display "09~15~23.46"
            </p>
          </div>
        </details>
        <details>
          <summary>
            tickformatstops
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>array of object where each object has one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>true</dd>
            </dl>
            <p>
              Determines whether or not this stop is used. If `false`, this stop is ignored even within its `dtickrange`.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              dtickrange
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.yaxis.tickformatstops[]</dd>
                <dt>Type:</dt>
                <dd>array</dd>
              </dl>
              <p>
                range ["min", "max"], where "min", "max" - dtick values which describe some zoom level, it is possible to omit "min" or "max" value by passing "null"
              </p>
            </div>
          </details>
          <details>
            <summary>
              enabled
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.yaxis.tickformatstops[]</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not this stop is used. If `false`, this stop is ignored even within its `dtickrange`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              name
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.yaxis.tickformatstops[]</dd>
                <dt>Type:</dt>
                <dd>string</dd>
              </dl>
              <p>
                When used in a template, named items are created in the output figure in addition to any items the figure already has in this array. You can modify these items in the output figure by making your own item with `templateitemname` matching this `name` alongside your modifications (including `visible: false` or `enabled: false` to hide it). Has no effect outside of a template.
              </p>
            </div>
          </details>
          <details>
            <summary>
              templateitemname
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.yaxis.tickformatstops[]</dd>
                <dt>Type:</dt>
                <dd>string</dd>
              </dl>
              <p>
                Used to refer to a named item in this array in the template. Named items from the template will be created even without a matching item in the input figure, but you can modify one by making an item with `templateitemname` matching its `name`, alongside your modifications (including `visible: false` or `enabled: false` to hide it). If there is no template or no matching item, this item will be hidden unless you explicitly show it with `visible: true`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              value
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.yaxis.tickformatstops[]</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                string - dtickformat for described zoom level, the same as "tickformat"
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            ticklen
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>5</dd>
            </dl>
            <p>
              Sets the tick length (in px).
            </p>
          </div>
        </details>
        <details>
          <summary>
            tickmode
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "auto" | "linear" | "array" )</dd>
            </dl>
            <p>
              Sets the tick mode for this axis. If "auto", the number of ticks is set via `nticks`. If "linear", the placement of the ticks is determined by a starting position `tick0` and a tick step `dtick` ("linear" is the default value if `tick0` and `dtick` are provided). If "array", the placement of the ticks is set via `tickvals` and the tick text is `ticktext`. ("array" is the default value if `tickvals` is provided).
            </p>
          </div>
        </details>
        <details>
          <summary>
            tickprefix
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>string</dd>
              <dt>Default:</dt>
              <dd>""</dd>
            </dl>
            <p>
              Sets a tick label prefix.
            </p>
          </div>
        </details>
        <details>
          <summary>
            ticks
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "outside" | "inside" | "" )</dd>
            </dl>
            <p>
              Determines whether ticks are drawn or not. If "", this axis' ticks are not drawn. If "outside" ("inside"), this axis' are drawn outside (inside) the axis lines.
            </p>
          </div>
        </details>
        <details>
          <summary>
            tickson
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "labels" | "boundaries" )</dd>
              <dt>Default:</dt>
              <dd>"labels"</dd>
            </dl>
            <p>
              Determines where ticks and grid lines are drawn with respect to their corresponding tick labels. Only has an effect for axes of `type` "category" or "multicategory". When set to "boundaries", ticks and grid lines are drawn half a category to the left/bottom of labels.
            </p>
          </div>
        </details>
        <details>
          <summary>
            ticksuffix
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>string</dd>
              <dt>Default:</dt>
              <dd>""</dd>
            </dl>
            <p>
              Sets a tick label suffix.
            </p>
          </div>
        </details>
        <details>
          <summary>
            ticktext
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>data array</dd>
            </dl>
            <p>
              Sets the text displayed at the ticks position via `tickvals`. Only has an effect if `tickmode` is set to "array". Used with `tickvals`.
            </p>
          </div>
        </details>
        <details>
          <summary>
            tickvals
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>data array</dd>
            </dl>
            <p>
              Sets the values at which ticks on this axis appear. Only has an effect if `tickmode` is set to "array". Used with `ticktext`.
            </p>
          </div>
        </details>
        <details>
          <summary>
            tickwidth
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>1</dd>
            </dl>
            <p>
              Sets the tick width (in px).
            </p>
          </div>
        </details>
        <details>
          <summary>
            title
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
            </dl>
            <p>
              Sets the title of this axis. Note that before the existence of `title.text`, the title's contents used to be defined as the `title` attribute itself. This behavior has been deprecated.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              font
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.yaxis.title</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
              </dl>
              <p>
                Sets this axis' title font. Note that the title's font used to be customized by the now deprecated `titlefont` attribute.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                color
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.yaxis.title.font</dd>
                  <dt>Type:</dt>
                  <dd>color</dd>
                </dl>
              </div>
            </details>
            <details>
              <summary>
                family
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.yaxis.title.font</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
                </p>
              </div>
            </details>
            <details>
              <summary>
                size
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>layout.yaxis.title.font</dd>
                  <dt>Type:</dt>
                  <dd>number greater than or equal to 1</dd>
                </dl>
              </div>
            </details>
          </div>
          <details>
            <summary>
              standoff
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.yaxis.title</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
              </dl>
              <p>
                Sets the standoff distance (in px) between the axis labels and the title text The default value is a function of the axis tick labels, the title `font.size` and the axis `linewidth`. Note that the axis title position is always constrained within the margins, so the actual standoff distance is always less than the set or default value. By setting `standoff` and turning on `automargin`, plotly.js will push the margins to fit the axis title at given standoff distance.
              </p>
            </div>
          </details>
          <details>
            <summary>
              text
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>layout.yaxis.title</dd>
                <dt>Type:</dt>
                <dd>string</dd>
              </dl>
              <p>
                Sets the title of this axis. Note that before the existence of `title.text`, the title's contents used to be defined as the `title` attribute itself. This behavior has been deprecated.
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            type
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "-" | "linear" | "log" | "date" | "category" | "multicategory" )</dd>
              <dt>Default:</dt>
              <dd>"-"</dd>
            </dl>
            <p>
              Sets the axis type. By default, plotly attempts to determined the axis type by looking into the data of the traces that referenced the axis in question.
            </p>
          </div>
        </details>
        <details>
          <summary>
            uirevision
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>number or categorical coordinate string</dd>
            </dl>
            <p>
              Controls persistence of user-driven changes in axis `range`, `autorange`, and `title` if in `editable: true` configuration. Defaults to `layout.uirevision`.
            </p>
          </div>
        </details>
        <details>
          <summary>
            visible
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              A single toggle to hide the axis while preserving interaction like dragging. Default is true when a cheater plot is present on the axis, otherwise false
            </p>
          </div>
        </details>
        <details>
          <summary>
            zeroline
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Determines whether or not a line is drawn at along the 0 value of this axis. If "true", the zero line is drawn on top of the grid lines.
            </p>
          </div>
        </details>
        <details>
          <summary>
            zerolinecolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>color</dd>
              <dt>Default:</dt>
              <dd>"#444"</dd>
            </dl>
            <p>
              Sets the line color of the zero line.
            </p>
          </div>
        </details>
        <details>
          <summary>
            zerolinewidth
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>layout.yaxis</dd>
              <dt>Type:</dt>
              <dd>number</dd>
              <dt>Default:</dt>
              <dd>1</dd>
            </dl>
            <p>
              Sets the width (in px) of the zero line.
            </p>
          </div>
        </details>
      </div>
    </p>
  </div>
</details>
<details>
  <summary>
    1. Scatter traces
  </summary>
  <div style="margin-left: 22px">
    <p>
      A scatter trace is an object with the key "type" equal to "scatter" (i.e. {"type": "scatter"}) and any of the keys listed below. The scatter trace type encompasses line charts, scatter charts, text charts, and bubble charts. The data visualized as scatter point or lines is set in `x` and `y`. Text (appearing either on the chart or on hover only) is via `text`. Bubble charts are achieved by setting `marker.size` and/or `marker.color` to numerical arrays.
    </p>
    <p>
      <details>
        <summary>
          cliponaxis
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>boolean</dd>
            <dt>Default:</dt>
            <dd>true</dd>
          </dl>
          <p>
            Determines whether or not markers and text nodes are clipped about the subplot axes. To show markers and text nodes above axis lines and tick labels, make sure to set `xaxis.layer` and `yaxis.layer` to "below traces".
          </p>
        </div>
      </details>
      <details>
        <summary>
          connectgaps
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>boolean</dd>
          </dl>
          <p>
            Determines whether or not gaps (i.e. {nan} or missing values) in the provided data arrays are connected.
          </p>
        </div>
      </details>
      <details>
        <summary>
          customdata
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>data array</dd>
          </dl>
          <p>
            Assigns extra data each datum. This may be useful when listening to hover, click and selection events. Note that, "scatter" traces also appends customdata items in the markers DOM elements
          </p>
        </div>
      </details>
      <details>
        <summary>
          dx
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>number</dd>
            <dt>Default:</dt>
            <dd>1</dd>
          </dl>
          <p>
            Sets the x coordinate step. See `x0` for more info.
          </p>
        </div>
      </details>
      <details>
        <summary>
          dy
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>number</dd>
            <dt>Default:</dt>
            <dd>1</dd>
          </dl>
          <p>
            Sets the y coordinate step. See `y0` for more info.
          </p>
        </div>
      </details>
      <details>
        <summary>
          error_x
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>object containing one or more of the keys listed below.</dd>
            <dt>Default:</dt>
            <dd>10</dd>
          </dl>
          <p>
            Determines whether or not this set of error bars is visible.
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            array
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].error_x</dd>
              <dt>Type:</dt>
              <dd>data array</dd>
            </dl>
            <p>
              Sets the data corresponding the length of each error bar. Values are plotted relative to the underlying data.
            </p>
          </div>
        </details>
        <details>
          <summary>
            arrayminus
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].error_x</dd>
              <dt>Type:</dt>
              <dd>data array</dd>
            </dl>
            <p>
              Sets the data corresponding the length of each error bar in the bottom (left) direction for vertical (horizontal) bars Values are plotted relative to the underlying data.
            </p>
          </div>
        </details>
        <details>
          <summary>
            color
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].error_x</dd>
              <dt>Type:</dt>
              <dd>color</dd>
            </dl>
            <p>
              Sets the stoke color of the error bars.
            </p>
          </div>
        </details>
        <details>
          <summary>
            copy_ystyle
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].error_x</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
          </div>
        </details>
        <details>
          <summary>
            symmetric
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].error_x</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Determines whether or not the error bars have the same length in both direction (top/bottom for vertical bars, left/right for horizontal bars.
            </p>
          </div>
        </details>
        <details>
          <summary>
            thickness
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].error_x</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>2</dd>
            </dl>
            <p>
              Sets the thickness (in px) of the error bars.
            </p>
          </div>
        </details>
        <details>
          <summary>
            traceref
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].error_x</dd>
              <dt>Type:</dt>
              <dd>integer greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
          </div>
        </details>
        <details>
          <summary>
            tracerefminus
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].error_x</dd>
              <dt>Type:</dt>
              <dd>integer greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
          </div>
        </details>
        <details>
          <summary>
            type
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].error_x</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "percent" | "constant" | "sqrt" | "data" )</dd>
            </dl>
            <p>
              Determines the rule used to generate the error bars. If "constant`, the bar lengths are of a constant value. Set this constant in `value`. If "percent", the bar lengths correspond to a percentage of underlying data. Set this percentage in `value`. If "sqrt", the bar lengths correspond to the sqaure of the underlying data. If "data", the bar lengths are set with data set `array`.
            </p>
          </div>
        </details>
        <details>
          <summary>
            value
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].error_x</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>10</dd>
            </dl>
            <p>
              Sets the value of either the percentage (if `type` is set to "percent") or the constant (if `type` is set to "constant") corresponding to the lengths of the error bars.
            </p>
          </div>
        </details>
        <details>
          <summary>
            valueminus
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].error_x</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>10</dd>
            </dl>
            <p>
              Sets the value of either the percentage (if `type` is set to "percent") or the constant (if `type` is set to "constant") corresponding to the lengths of the error bars in the bottom (left) direction for vertical (horizontal) bars
            </p>
          </div>
        </details>
        <details>
          <summary>
            visible
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].error_x</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Determines whether or not this set of error bars is visible.
            </p>
          </div>
        </details>
        <details>
          <summary>
            width
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].error_x</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
            </dl>
            <p>
              Sets the width (in px) of the cross-bar at both ends of the error bars.
            </p>
          </div>
        </details>
      </div>
      <details>
        <summary>
          error_y
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>object containing one or more of the keys listed below.</dd>
            <dt>Default:</dt>
            <dd>10</dd>
          </dl>
          <p>
            Determines whether or not this set of error bars is visible.
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            array
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].error_y</dd>
              <dt>Type:</dt>
              <dd>data array</dd>
            </dl>
            <p>
              Sets the data corresponding the length of each error bar. Values are plotted relative to the underlying data.
            </p>
          </div>
        </details>
        <details>
          <summary>
            arrayminus
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].error_y</dd>
              <dt>Type:</dt>
              <dd>data array</dd>
            </dl>
            <p>
              Sets the data corresponding the length of each error bar in the bottom (left) direction for vertical (horizontal) bars Values are plotted relative to the underlying data.
            </p>
          </div>
        </details>
        <details>
          <summary>
            color
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].error_y</dd>
              <dt>Type:</dt>
              <dd>color</dd>
            </dl>
            <p>
              Sets the stoke color of the error bars.
            </p>
          </div>
        </details>
        <details>
          <summary>
            symmetric
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].error_y</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Determines whether or not the error bars have the same length in both direction (top/bottom for vertical bars, left/right for horizontal bars.
            </p>
          </div>
        </details>
        <details>
          <summary>
            thickness
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].error_y</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>2</dd>
            </dl>
            <p>
              Sets the thickness (in px) of the error bars.
            </p>
          </div>
        </details>
        <details>
          <summary>
            traceref
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].error_y</dd>
              <dt>Type:</dt>
              <dd>integer greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
          </div>
        </details>
        <details>
          <summary>
            tracerefminus
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].error_y</dd>
              <dt>Type:</dt>
              <dd>integer greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
          </div>
        </details>
        <details>
          <summary>
            type
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].error_y</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "percent" | "constant" | "sqrt" | "data" )</dd>
            </dl>
            <p>
              Determines the rule used to generate the error bars. If "constant`, the bar lengths are of a constant value. Set this constant in `value`. If "percent", the bar lengths correspond to a percentage of underlying data. Set this percentage in `value`. If "sqrt", the bar lengths correspond to the sqaure of the underlying data. If "data", the bar lengths are set with data set `array`.
            </p>
          </div>
        </details>
        <details>
          <summary>
            value
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].error_y</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>10</dd>
            </dl>
            <p>
              Sets the value of either the percentage (if `type` is set to "percent") or the constant (if `type` is set to "constant") corresponding to the lengths of the error bars.
            </p>
          </div>
        </details>
        <details>
          <summary>
            valueminus
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].error_y</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>10</dd>
            </dl>
            <p>
              Sets the value of either the percentage (if `type` is set to "percent") or the constant (if `type` is set to "constant") corresponding to the lengths of the error bars in the bottom (left) direction for vertical (horizontal) bars
            </p>
          </div>
        </details>
        <details>
          <summary>
            visible
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].error_y</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Determines whether or not this set of error bars is visible.
            </p>
          </div>
        </details>
        <details>
          <summary>
            width
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].error_y</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
            </dl>
            <p>
              Sets the width (in px) of the cross-bar at both ends of the error bars.
            </p>
          </div>
        </details>
      </div>
      <details>
        <summary>
          fill
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>enumerated , one of ( "none" | "tozeroy" | "tozerox" | "tonexty" | "tonextx" | "toself" | "tonext" )</dd>
          </dl>
          <p>
            Sets the area to fill with a solid color. Defaults to "none" unless this trace is stacked, then it gets "tonexty" ("tonextx") if `orientation` is "v" ("h") Use with `fillcolor` if not "none". "tozerox" and "tozeroy" fill to x=0 and y=0 respectively. "tonextx" and "tonexty" fill between the endpoints of this trace and the endpoints of the trace before it, connecting those endpoints with straight lines (to make a stacked area graph); if there is no trace before it, they behave like "tozerox" and "tozeroy". "toself" connects the endpoints of the trace (or each segment of the trace if it has gaps) into a closed shape. "tonext" fills the space between two traces if one completely encloses the other (eg consecutive contour lines), and behaves like "toself" if there is no trace before it. "tonext" should not be used if one trace does not enclose the other. Traces in a `stackgroup` will only fill to (or be filled to) other traces in the same group. With multiple `stackgroup`s or some traces stacked and some not, if fill-linked traces are not already consecutive, the later ones will be pushed down in the drawing order.
          </p>
        </div>
      </details>
      <details>
        <summary>
          fillcolor
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>color</dd>
          </dl>
          <p>
            Sets the fill color. Defaults to a half-transparent variant of the line color, marker color, or marker line color, whichever is available.
          </p>
        </div>
      </details>
      <details>
        <summary>
          groupnorm
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>enumerated , one of ( "" | "fraction" | "percent" )</dd>
            <dt>Default:</dt>
            <dd>""</dd>
          </dl>
          <p>
            Only relevant when `stackgroup` is used, and only the first `groupnorm` found in the `stackgroup` will be used - including if `visible` is "legendonly" but not if it is `false`. Sets the normalization for the sum of this `stackgroup`. With "fraction", the value of each trace at each location is divided by the sum of all trace values at that location. "percent" is the same but multiplied by 100 to show percentages. If there are multiple subplots, or multiple `stackgroup`s on one subplot, each will be normalized within its own set.
          </p>
        </div>
      </details>
      <details>
        <summary>
          hoverinfo
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>flaglist string. Any combination of "x", "y", "z", "text", "name" joined with a "+" OR "all" or "none" or "skip".</dd>
            <dt>Default:</dt>
            <dd>"all"</dd>
          </dl>
          <p>
            Determines which trace information appear on hover. If `none` or `skip` are set, no information is displayed upon hovering. But, if `none` is set, click and hover events are still fired.
          </p>
        </div>
      </details>
      <details>
        <summary>
          hoverlabel
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>object containing one or more of the keys listed below.</dd>
            <dt>Default:</dt>
            <dd>"auto"</dd>
          </dl>
          <p>
            Sets the background color of the hover labels for this trace
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            align
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].hoverlabel</dd>
              <dt>Type:</dt>
              <dd>enumerated or array of enumerateds , one of ( "left" | "right" | "auto" )</dd>
              <dt>Default:</dt>
              <dd>"auto"</dd>
            </dl>
            <p>
              Sets the horizontal alignment of the text content within hover label box. Has an effect only if the hover label text spans more two or more lines
            </p>
          </div>
        </details>
        <details>
          <summary>
            bgcolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].hoverlabel</dd>
              <dt>Type:</dt>
              <dd>color or array of colors</dd>
            </dl>
            <p>
              Sets the background color of the hover labels for this trace
            </p>
          </div>
        </details>
        <details>
          <summary>
            bordercolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].hoverlabel</dd>
              <dt>Type:</dt>
              <dd>color or array of colors</dd>
            </dl>
            <p>
              Sets the border color of the hover labels for this trace.
            </p>
          </div>
        </details>
        <details>
          <summary>
            font
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].hoverlabel</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
            </dl>
            <p>
              Sets the font used in hover labels.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              color
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].hoverlabel.font</dd>
                <dt>Type:</dt>
                <dd>color or array of colors</dd>
              </dl>
            </div>
          </details>
          <details>
            <summary>
              family
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].hoverlabel.font</dd>
                <dt>Type:</dt>
                <dd>string or array of strings</dd>
              </dl>
              <p>
                HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
              </p>
            </div>
          </details>
          <details>
            <summary>
              size
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].hoverlabel.font</dd>
                <dt>Type:</dt>
                <dd>number or array of numbers greater than or equal to 1</dd>
              </dl>
            </div>
          </details>
        </div>
        <details>
          <summary>
            namelength
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].hoverlabel</dd>
              <dt>Type:</dt>
              <dd>integer or array of integers greater than or equal to -1</dd>
              <dt>Default:</dt>
              <dd>15</dd>
            </dl>
            <p>
              Sets the default length (in number of characters) of the trace name in the hover labels for all traces. -1 shows the whole name regardless of length. 0-3 shows the first 0-3 characters, and an integer &gt;3 will show the whole name if it is less than that many characters, but if it is longer, will truncate to `namelength - 3` characters and add an ellipsis.
            </p>
          </div>
        </details>
      </div>
      <details>
        <summary>
          hoveron
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>flaglist string. Any combination of "points", "fills" joined with a "+"</dd>
          </dl>
          <p>
            Do the hover effects highlight individual points (markers or line points) or do they highlight filled regions? If the fill is "toself" or "tonext" and there are no markers or text, then the default is "fills", otherwise it is "points".
          </p>
        </div>
      </details>
      <details>
        <summary>
          hovertemplate
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>string or array of strings</dd>
            <dt>Default:</dt>
            <dd>""</dd>
          </dl>
          <p>
            Template string used for rendering the information that appear on hover box. Note that this will override `hoverinfo`. Variables are inserted using %{variable}, for example "y: %{y}". Numbers are formatted using d3-format's syntax %{variable:d3-format}, for example "Price: %{y:$.2f}". https://github.com/d3/d3-3.x-api-reference/blob/master/Formatting.md#d3_format for details on the formatting syntax. Dates are formatted using d3-time-format's syntax %{variable|d3-time-format}, for example "Day: %{2019-01-01|%A}". https://github.com/d3/d3-3.x-api-reference/blob/master/Time-Formatting.md#format for details on the date formatting syntax. The variables available in `hovertemplate` are the ones emitted as event data described at this link https://plot.ly/javascript/plotlyjs-events/#event-data. Additionally, every attributes that can be specified per-point (the ones that are `arrayOk: true`) are available. Anything contained in tag `&lt;extra&gt;` is displayed in the secondary box, for example "&lt;extra&gt;{fullData.name}&lt;/extra&gt;". To hide the secondary box completely, use an empty tag `&lt;extra&gt;&lt;/extra&gt;`.
          </p>
        </div>
      </details>
      <details>
        <summary>
          hovertext
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>string or array of strings</dd>
            <dt>Default:</dt>
            <dd>""</dd>
          </dl>
          <p>
            Sets hover text elements associated with each (x,y) pair. If a single string, the same string appears over all the data points. If an array of string, the items are mapped in order to the this trace's (x,y) coordinates. To be seen, trace `hoverinfo` must contain a "text" flag.
          </p>
        </div>
      </details>
      <details>
        <summary>
          ids
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>data array</dd>
          </dl>
          <p>
            Assigns id labels to each datum. These ids for object constancy of data points during animation. Should be an array of strings, not numbers or any other type.
          </p>
        </div>
      </details>
      <details>
        <summary>
          legendgroup
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>string</dd>
            <dt>Default:</dt>
            <dd>""</dd>
          </dl>
          <p>
            Sets the legend group for this trace. Traces part of the same legend group hide/show at the same time when toggling legend items.
          </p>
        </div>
      </details>
      <details>
        <summary>
          line
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>object containing one or more of the keys listed below.</dd>
            <dt>Default:</dt>
            <dd>2</dd>
          </dl>
          <p>
            Sets the line color.
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            color
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].line</dd>
              <dt>Type:</dt>
              <dd>color</dd>
            </dl>
            <p>
              Sets the line color.
            </p>
          </div>
        </details>
        <details>
          <summary>
            dash
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].line</dd>
              <dt>Type:</dt>
              <dd>string</dd>
              <dt>Default:</dt>
              <dd>"solid"</dd>
            </dl>
            <p>
              Sets the dash style of lines. Set to a dash type string ("solid", "dot", "dash", "longdash", "dashdot", or "longdashdot") or a dash length list in px (eg "5px,10px,2px,2px").
            </p>
          </div>
        </details>
        <details>
          <summary>
            shape
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].line</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "linear" | "spline" | "hv" | "vh" | "hvh" | "vhv" )</dd>
              <dt>Default:</dt>
              <dd>"linear"</dd>
            </dl>
            <p>
              Determines the line shape. With "spline" the lines are drawn using spline interpolation. The other available values correspond to step-wise line shapes.
            </p>
          </div>
        </details>
        <details>
          <summary>
            simplify
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].line</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
              <dt>Default:</dt>
              <dd>true</dd>
            </dl>
            <p>
              Simplifies lines by removing nearly-collinear points. When transitioning lines, it may be desirable to disable this so that the number of points along the resulting SVG path is unaffected.
            </p>
          </div>
        </details>
        <details>
          <summary>
            smoothing
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].line</dd>
              <dt>Type:</dt>
              <dd>number between or equal to 0 and 1.3</dd>
              <dt>Default:</dt>
              <dd>1</dd>
            </dl>
            <p>
              Has an effect only if `shape` is set to "spline" Sets the amount of smoothing. "0" corresponds to no smoothing (equivalent to a "linear" shape).
            </p>
          </div>
        </details>
        <details>
          <summary>
            width
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].line</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>2</dd>
            </dl>
            <p>
              Sets the line width (in px).
            </p>
          </div>
        </details>
      </div>
      <details>
        <summary>
          marker
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>object containing one or more of the keys listed below.</dd>
            <dt>Default:</dt>
            <dd>"circle"</dd>
          </dl>
          <p>
            Sets the marker symbol type. Adding 100 is equivalent to appending "-open" to a symbol name. Adding 200 is equivalent to appending "-dot" to a symbol name. Adding 300 is equivalent to appending "-open-dot" or "dot-open" to a symbol name.
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            autocolorscale
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].marker</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
              <dt>Default:</dt>
              <dd>true</dd>
            </dl>
            <p>
              Determines whether the colorscale is a default palette (`autocolorscale: true`) or the palette determined by `marker.colorscale`. Has an effect only if in `marker.color`is set to a numerical array. In case `colorscale` is unspecified or `autocolorscale` is true, the default palette will be chosen according to whether numbers in the `color` array are all positive, all negative or mixed.
            </p>
          </div>
        </details>
        <details>
          <summary>
            cauto
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].marker</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
              <dt>Default:</dt>
              <dd>true</dd>
            </dl>
            <p>
              Determines whether or not the color domain is computed with respect to the input data (here in `marker.color`) or the bounds set in `marker.cmin` and `marker.cmax` Has an effect only if in `marker.color`is set to a numerical array. Defaults to `false` when `marker.cmin` and `marker.cmax` are set by the user.
            </p>
          </div>
        </details>
        <details>
          <summary>
            cmax
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].marker</dd>
              <dt>Type:</dt>
              <dd>number</dd>
            </dl>
            <p>
              Sets the upper bound of the color domain. Has an effect only if in `marker.color`is set to a numerical array. Value should have the same units as in `marker.color` and if set, `marker.cmin` must be set as well.
            </p>
          </div>
        </details>
        <details>
          <summary>
            cmid
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].marker</dd>
              <dt>Type:</dt>
              <dd>number</dd>
            </dl>
            <p>
              Sets the mid-point of the color domain by scaling `marker.cmin` and/or `marker.cmax` to be equidistant to this point. Has an effect only if in `marker.color`is set to a numerical array. Value should have the same units as in `marker.color`. Has no effect when `marker.cauto` is `false`.
            </p>
          </div>
        </details>
        <details>
          <summary>
            cmin
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].marker</dd>
              <dt>Type:</dt>
              <dd>number</dd>
            </dl>
            <p>
              Sets the lower bound of the color domain. Has an effect only if in `marker.color`is set to a numerical array. Value should have the same units as in `marker.color` and if set, `marker.cmax` must be set as well.
            </p>
          </div>
        </details>
        <details>
          <summary>
            color
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].marker</dd>
              <dt>Type:</dt>
              <dd>color or array of colors</dd>
            </dl>
            <p>
              Sets themarkercolor. It accepts either a specific color or an array of numbers that are mapped to the colorscale relative to the max and min values of the array or relative to `marker.cmin` and `marker.cmax` if set.
            </p>
          </div>
        </details>
        <details>
          <summary>
            coloraxis
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].marker</dd>
              <dt>Type:</dt>
              <dd>subplotid</dd>
            </dl>
            <p>
              Sets a reference to a shared color axis. References to these shared color axes are "coloraxis", "coloraxis2", "coloraxis3", etc. Settings for these shared color axes are set in the layout, under `layout.coloraxis`, `layout.coloraxis2`, etc. Note that multiple color scales can be linked to the same color axis.
            </p>
          </div>
        </details>
        <details>
          <summary>
            colorbar
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].marker</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>"pixels"</dd>
            </dl>
            <p>
              Determines whether this color bar's thickness (i.e. the measure in the constant color direction) is set in units of plot "fraction" or in "pixels". Use `thickness` to set the value.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              bgcolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"rgba(0,0,0,0)"</dd>
              </dl>
              <p>
                Sets the color of padded area.
              </p>
            </div>
          </details>
          <details>
            <summary>
              bordercolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets the axis line color.
              </p>
            </div>
          </details>
          <details>
            <summary>
              borderwidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                Sets the width (in px) or the border enclosing this color bar.
              </p>
            </div>
          </details>
          <details>
            <summary>
              dtick
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>number or categorical coordinate string</dd>
              </dl>
              <p>
                Sets the step in-between ticks on this axis. Use with `tick0`. Must be a positive number, or special strings available to "log" and "date" axes. If the axis `type` is "log", then ticks are set every 10^(n"dtick) where n is the tick number. For example, to set a tick mark at 1, 10, 100, 1000, ... set dtick to 1. To set tick marks at 1, 100, 10000, ... set dtick to 2. To set tick marks at 1, 5, 25, 125, 625, 3125, ... set dtick to log_10(5), or 0.69897000433. "log" has several special values; "L&lt;f&gt;", where `f` is a positive number, gives ticks linearly spaced in value (but not position). For example `tick0` = 0.1, `dtick` = "L0.5" will put ticks at 0.1, 0.6, 1.1, 1.6 etc. To show powers of 10 plus small digits between, use "D1" (all digits) or "D2" (only 2 and 5). `tick0` is ignored for "D1" and "D2". If the axis `type` is "date", then you must convert the time to milliseconds. For example, to set the interval between ticks to one day, set `dtick` to 86400000.0. "date" also has special values "M&lt;n&gt;" gives ticks spaced by a number of months. `n` must be a positive integer. To set ticks on the 15th of every third month, set `tick0` to "2000-01-15" and `dtick` to "M3". To set ticks every 4 years, set `dtick` to "M48"
              </p>
            </div>
          </details>
          <details>
            <summary>
              exponentformat
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "none" | "e" | "E" | "power" | "SI" | "B" )</dd>
                <dt>Default:</dt>
                <dd>"B"</dd>
              </dl>
              <p>
                Determines a formatting rule for the tick exponents. For example, consider the number 1,000,000,000. If "none", it appears as 1,000,000,000. If "e", 1e+9. If "E", 1E+9. If "power", 1x10^9 (with 9 in a super script). If "SI", 1G. If "B", 1B.
              </p>
            </div>
          </details>
          <details>
            <summary>
              len
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the length of the color bar This measure excludes the padding of both ends. That is, the color bar length is this length minus the padding on both ends.
              </p>
            </div>
          </details>
          <details>
            <summary>
              lenmode
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "fraction" | "pixels" )</dd>
                <dt>Default:</dt>
                <dd>"fraction"</dd>
              </dl>
              <p>
                Determines whether this color bar's length (i.e. the measure in the color variation direction) is set in units of plot "fraction" or in "pixels. Use `len` to set the value.
              </p>
            </div>
          </details>
          <details>
            <summary>
              nticks
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>integer greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                Specifies the maximum number of ticks for the particular axis. The actual number of ticks will be chosen automatically to be less than or equal to `nticks`. Has an effect only if `tickmode` is set to "auto".
              </p>
            </div>
          </details>
          <details>
            <summary>
              outlinecolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets the axis line color.
              </p>
            </div>
          </details>
          <details>
            <summary>
              outlinewidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the width (in px) of the axis line.
              </p>
            </div>
          </details>
          <details>
            <summary>
              separatethousands
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
              </dl>
              <p>
                If "true", even 4-digit integers are separated
              </p>
            </div>
          </details>
          <details>
            <summary>
              showexponent
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
                <dt>Default:</dt>
                <dd>"all"</dd>
              </dl>
              <p>
                If "all", all exponents are shown besides their significands. If "first", only the exponent of the first tick is shown. If "last", only the exponent of the last tick is shown. If "none", no exponents appear.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showticklabels
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not the tick labels are drawn.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showtickprefix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
                <dt>Default:</dt>
                <dd>"all"</dd>
              </dl>
              <p>
                If "all", all tick labels are displayed with a prefix. If "first", only the first tick is displayed with a prefix. If "last", only the last tick is displayed with a suffix. If "none", tick prefixes are hidden.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showticksuffix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
                <dt>Default:</dt>
                <dd>"all"</dd>
              </dl>
              <p>
                Same as `showtickprefix` but for tick suffixes.
              </p>
            </div>
          </details>
          <details>
            <summary>
              thickness
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>30</dd>
              </dl>
              <p>
                Sets the thickness of the color bar This measure excludes the size of the padding, ticks and labels.
              </p>
            </div>
          </details>
          <details>
            <summary>
              thicknessmode
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "fraction" | "pixels" )</dd>
                <dt>Default:</dt>
                <dd>"pixels"</dd>
              </dl>
              <p>
                Determines whether this color bar's thickness (i.e. the measure in the constant color direction) is set in units of plot "fraction" or in "pixels". Use `thickness` to set the value.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tick0
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>number or categorical coordinate string</dd>
              </dl>
              <p>
                Sets the placement of the first tick on this axis. Use with `dtick`. If the axis `type` is "log", then you must take the log of your starting tick (e.g. to set the starting tick to 100, set the `tick0` to 2) except when `dtick`="L&lt;f&gt;" (see `dtick` for more info). If the axis `type` is "date", it should be a date string, like date data. If the axis `type` is "category", it should be a number, using the scale where each category is assigned a serial number from zero in the order it appears.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickangle
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>angle</dd>
                <dt>Default:</dt>
                <dd>"auto"</dd>
              </dl>
              <p>
                Sets the angle of the tick labels with respect to the horizontal. For example, a `tickangle` of -90 draws the tick labels vertically.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickcolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets the tick color.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickfont
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
              </dl>
              <p>
                Sets the color bar's tick label font
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                color
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>data[type=scatter].marker.colorbar.tickfont</dd>
                  <dt>Type:</dt>
                  <dd>color</dd>
                </dl>
              </div>
            </details>
            <details>
              <summary>
                family
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>data[type=scatter].marker.colorbar.tickfont</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
                </p>
              </div>
            </details>
            <details>
              <summary>
                size
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>data[type=scatter].marker.colorbar.tickfont</dd>
                  <dt>Type:</dt>
                  <dd>number greater than or equal to 1</dd>
                </dl>
              </div>
            </details>
          </div>
          <details>
            <summary>
              tickformat
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets the tick label formatting rule using d3 formatting mini-languages which are very similar to those in Python. For numbers, see: https://github.com/d3/d3-3.x-api-reference/blob/master/Formatting.md#d3_format And for dates see: https://github.com/d3/d3-3.x-api-reference/blob/master/Time-Formatting.md#format We add one item to d3's date formatter: "%{n}f" for fractional seconds with n digits. For example, "2016-10-13 09:15:23.456" with tickformat "%H~%\M~%S.%2f" would display "09~15~23.46"
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickformatstops
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>array of object where each object has one or more of the keys listed below.</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not this stop is used. If `false`, this stop is ignored even within its `dtickrange`.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                dtickrange
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>data[type=scatter].marker.colorbar.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>array</dd>
                </dl>
                <p>
                  range ["min", "max"], where "min", "max" - dtick values which describe some zoom level, it is possible to omit "min" or "max" value by passing "null"
                </p>
              </div>
            </details>
            <details>
              <summary>
                enabled
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>data[type=scatter].marker.colorbar.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>boolean</dd>
                  <dt>Default:</dt>
                  <dd>true</dd>
                </dl>
                <p>
                  Determines whether or not this stop is used. If `false`, this stop is ignored even within its `dtickrange`.
                </p>
              </div>
            </details>
            <details>
              <summary>
                name
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>data[type=scatter].marker.colorbar.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  When used in a template, named items are created in the output figure in addition to any items the figure already has in this array. You can modify these items in the output figure by making your own item with `templateitemname` matching this `name` alongside your modifications (including `visible: false` or `enabled: false` to hide it). Has no effect outside of a template.
                </p>
              </div>
            </details>
            <details>
              <summary>
                templateitemname
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>data[type=scatter].marker.colorbar.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  Used to refer to a named item in this array in the template. Named items from the template will be created even without a matching item in the input figure, but you can modify one by making an item with `templateitemname` matching its `name`, alongside your modifications (including `visible: false` or `enabled: false` to hide it). If there is no template or no matching item, this item will be hidden unless you explicitly show it with `visible: true`.
                </p>
              </div>
            </details>
            <details>
              <summary>
                value
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>data[type=scatter].marker.colorbar.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                  <dt>Default:</dt>
                  <dd>""</dd>
                </dl>
                <p>
                  string - dtickformat for described zoom level, the same as "tickformat"
                </p>
              </div>
            </details>
          </div>
          <details>
            <summary>
              ticklen
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>5</dd>
              </dl>
              <p>
                Sets the tick length (in px).
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickmode
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "auto" | "linear" | "array" )</dd>
              </dl>
              <p>
                Sets the tick mode for this axis. If "auto", the number of ticks is set via `nticks`. If "linear", the placement of the ticks is determined by a starting position `tick0` and a tick step `dtick` ("linear" is the default value if `tick0` and `dtick` are provided). If "array", the placement of the ticks is set via `tickvals` and the tick text is `ticktext`. ("array" is the default value if `tickvals` is provided).
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickprefix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets a tick label prefix.
              </p>
            </div>
          </details>
          <details>
            <summary>
              ticks
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "outside" | "inside" | "" )</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Determines whether ticks are drawn or not. If "", this axis' ticks are not drawn. If "outside" ("inside"), this axis' are drawn outside (inside) the axis lines.
              </p>
            </div>
          </details>
          <details>
            <summary>
              ticksuffix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets a tick label suffix.
              </p>
            </div>
          </details>
          <details>
            <summary>
              ticktext
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>data array</dd>
              </dl>
              <p>
                Sets the text displayed at the ticks position via `tickvals`. Only has an effect if `tickmode` is set to "array". Used with `tickvals`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickvals
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>data array</dd>
              </dl>
              <p>
                Sets the values at which ticks on this axis appear. Only has an effect if `tickmode` is set to "array". Used with `ticktext`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickwidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the tick width (in px).
              </p>
            </div>
          </details>
          <details>
            <summary>
              title
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
                <dt>Default:</dt>
                <dd>"top"</dd>
              </dl>
              <p>
                Sets the title of the color bar. Note that before the existence of `title.text`, the title's contents used to be defined as the `title` attribute itself. This behavior has been deprecated.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                font
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>data[type=scatter].marker.colorbar.title</dd>
                  <dt>Type:</dt>
                  <dd>object containing one or more of the keys listed below.</dd>
                </dl>
                <p>
                  Sets this color bar's title font. Note that the title's font used to be set by the now deprecated `titlefont` attribute.
                </p>
              </div>
            </details>
            <div style="margin-left: 22px">
              <details>
                <summary>
                  color
                </summary>
                <div style="margin-left: 22px">
                  <dl>
                    <dt>Parent:</dt>
                    <dd>data[type=scatter].marker.colorbar.title.font</dd>
                    <dt>Type:</dt>
                    <dd>color</dd>
                  </dl>
                </div>
              </details>
              <details>
                <summary>
                  family
                </summary>
                <div style="margin-left: 22px">
                  <dl>
                    <dt>Parent:</dt>
                    <dd>data[type=scatter].marker.colorbar.title.font</dd>
                    <dt>Type:</dt>
                    <dd>string</dd>
                  </dl>
                  <p>
                    HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
                  </p>
                </div>
              </details>
              <details>
                <summary>
                  size
                </summary>
                <div style="margin-left: 22px">
                  <dl>
                    <dt>Parent:</dt>
                    <dd>data[type=scatter].marker.colorbar.title.font</dd>
                    <dt>Type:</dt>
                    <dd>number greater than or equal to 1</dd>
                  </dl>
                </div>
              </details>
            </div>
            <details>
              <summary>
                side
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>data[type=scatter].marker.colorbar.title</dd>
                  <dt>Type:</dt>
                  <dd>enumerated , one of ( "right" | "top" | "bottom" )</dd>
                  <dt>Default:</dt>
                  <dd>"top"</dd>
                </dl>
                <p>
                  Determines the location of color bar's title with respect to the color bar. Note that the title's location used to be set by the now deprecated `titleside` attribute.
                </p>
              </div>
            </details>
            <details>
              <summary>
                text
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>data[type=scatter].marker.colorbar.title</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  Sets the title of the color bar. Note that before the existence of `title.text`, the title's contents used to be defined as the `title` attribute itself. This behavior has been deprecated.
                </p>
              </div>
            </details>
          </div>
          <details>
            <summary>
              x
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>number between or equal to -2 and 3</dd>
                <dt>Default:</dt>
                <dd>1.02</dd>
              </dl>
              <p>
                Sets the x position of the color bar (in plot fraction).
              </p>
            </div>
          </details>
          <details>
            <summary>
              xanchor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "left" | "center" | "right" )</dd>
                <dt>Default:</dt>
                <dd>"left"</dd>
              </dl>
              <p>
                Sets this color bar's horizontal position anchor. This anchor binds the `x` position to the "left", "center" or "right" of the color bar.
              </p>
            </div>
          </details>
          <details>
            <summary>
              xpad
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>10</dd>
              </dl>
              <p>
                Sets the amount of padding (in px) along the x direction.
              </p>
            </div>
          </details>
          <details>
            <summary>
              y
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>number between or equal to -2 and 3</dd>
                <dt>Default:</dt>
                <dd>0.5</dd>
              </dl>
              <p>
                Sets the y position of the color bar (in plot fraction).
              </p>
            </div>
          </details>
          <details>
            <summary>
              yanchor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "top" | "middle" | "bottom" )</dd>
                <dt>Default:</dt>
                <dd>"middle"</dd>
              </dl>
              <p>
                Sets this color bar's vertical position anchor This anchor binds the `y` position to the "top", "middle" or "bottom" of the color bar.
              </p>
            </div>
          </details>
          <details>
            <summary>
              ypad
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>10</dd>
              </dl>
              <p>
                Sets the amount of padding (in px) along the y direction.
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            colorscale
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].marker</dd>
              <dt>Type:</dt>
              <dd>colorscale</dd>
            </dl>
            <p>
              Sets the colorscale. Has an effect only if in `marker.color`is set to a numerical array. The colorscale must be an array containing arrays mapping a normalized value to an rgb, rgba, hex, hsl, hsv, or named color string. At minimum, a mapping for the lowest (0) and highest (1) values are required. For example, `[[0, 'rgb(0,0,255)'], [1, 'rgb(255,0,0)']]`. To control the bounds of the colorscale in color space, use`marker.cmin` and `marker.cmax`. Alternatively, `colorscale` may be a palette name string of the following list: Greys,YlGnBu,Greens,YlOrRd,Bluered,RdBu,Reds,Blues,Picnic,Rainbow,Portland,Jet,Hot,Blackbody,Earth,Electric,Viridis,Cividis.
            </p>
          </div>
        </details>
        <details>
          <summary>
            gradient
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].marker</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>"none"</dd>
            </dl>
            <p>
              Sets the type of gradient used to fill the markers
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              color
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.gradient</dd>
                <dt>Type:</dt>
                <dd>color or array of colors</dd>
              </dl>
              <p>
                Sets the final color of the gradient fill: the center color for radial, the right for horizontal, or the bottom for vertical.
              </p>
            </div>
          </details>
          <details>
            <summary>
              type
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.gradient</dd>
                <dt>Type:</dt>
                <dd>enumerated or array of enumerateds , one of ( "radial" | "horizontal" | "vertical" | "none" )</dd>
                <dt>Default:</dt>
                <dd>"none"</dd>
              </dl>
              <p>
                Sets the type of gradient used to fill the markers
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            line
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].marker</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>true</dd>
            </dl>
            <p>
              Sets the width (in px) of the lines bounding the marker points.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              autocolorscale
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.line</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether the colorscale is a default palette (`autocolorscale: true`) or the palette determined by `marker.line.colorscale`. Has an effect only if in `marker.line.color`is set to a numerical array. In case `colorscale` is unspecified or `autocolorscale` is true, the default palette will be chosen according to whether numbers in the `color` array are all positive, all negative or mixed.
              </p>
            </div>
          </details>
          <details>
            <summary>
              cauto
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.line</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not the color domain is computed with respect to the input data (here in `marker.line.color`) or the bounds set in `marker.line.cmin` and `marker.line.cmax` Has an effect only if in `marker.line.color`is set to a numerical array. Defaults to `false` when `marker.line.cmin` and `marker.line.cmax` are set by the user.
              </p>
            </div>
          </details>
          <details>
            <summary>
              cmax
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.line</dd>
                <dt>Type:</dt>
                <dd>number</dd>
              </dl>
              <p>
                Sets the upper bound of the color domain. Has an effect only if in `marker.line.color`is set to a numerical array. Value should have the same units as in `marker.line.color` and if set, `marker.line.cmin` must be set as well.
              </p>
            </div>
          </details>
          <details>
            <summary>
              cmid
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.line</dd>
                <dt>Type:</dt>
                <dd>number</dd>
              </dl>
              <p>
                Sets the mid-point of the color domain by scaling `marker.line.cmin` and/or `marker.line.cmax` to be equidistant to this point. Has an effect only if in `marker.line.color`is set to a numerical array. Value should have the same units as in `marker.line.color`. Has no effect when `marker.line.cauto` is `false`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              cmin
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.line</dd>
                <dt>Type:</dt>
                <dd>number</dd>
              </dl>
              <p>
                Sets the lower bound of the color domain. Has an effect only if in `marker.line.color`is set to a numerical array. Value should have the same units as in `marker.line.color` and if set, `marker.line.cmax` must be set as well.
              </p>
            </div>
          </details>
          <details>
            <summary>
              color
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.line</dd>
                <dt>Type:</dt>
                <dd>color or array of colors</dd>
              </dl>
              <p>
                Sets themarker.linecolor. It accepts either a specific color or an array of numbers that are mapped to the colorscale relative to the max and min values of the array or relative to `marker.line.cmin` and `marker.line.cmax` if set.
              </p>
            </div>
          </details>
          <details>
            <summary>
              coloraxis
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.line</dd>
                <dt>Type:</dt>
                <dd>subplotid</dd>
              </dl>
              <p>
                Sets a reference to a shared color axis. References to these shared color axes are "coloraxis", "coloraxis2", "coloraxis3", etc. Settings for these shared color axes are set in the layout, under `layout.coloraxis`, `layout.coloraxis2`, etc. Note that multiple color scales can be linked to the same color axis.
              </p>
            </div>
          </details>
          <details>
            <summary>
              colorscale
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.line</dd>
                <dt>Type:</dt>
                <dd>colorscale</dd>
              </dl>
              <p>
                Sets the colorscale. Has an effect only if in `marker.line.color`is set to a numerical array. The colorscale must be an array containing arrays mapping a normalized value to an rgb, rgba, hex, hsl, hsv, or named color string. At minimum, a mapping for the lowest (0) and highest (1) values are required. For example, `[[0, 'rgb(0,0,255)'], [1, 'rgb(255,0,0)']]`. To control the bounds of the colorscale in color space, use`marker.line.cmin` and `marker.line.cmax`. Alternatively, `colorscale` may be a palette name string of the following list: Greys,YlGnBu,Greens,YlOrRd,Bluered,RdBu,Reds,Blues,Picnic,Rainbow,Portland,Jet,Hot,Blackbody,Earth,Electric,Viridis,Cividis.
              </p>
            </div>
          </details>
          <details>
            <summary>
              reversescale
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.line</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
              </dl>
              <p>
                Reverses the color mapping if true. Has an effect only if in `marker.line.color`is set to a numerical array. If true, `marker.line.cmin` will correspond to the last color in the array and `marker.line.cmax` will correspond to the first color.
              </p>
            </div>
          </details>
          <details>
            <summary>
              width
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].marker.line</dd>
                <dt>Type:</dt>
                <dd>number or array of numbers greater than or equal to 0</dd>
              </dl>
              <p>
                Sets the width (in px) of the lines bounding the marker points.
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            maxdisplayed
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].marker</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
            <p>
              Sets a maximum number of points to be drawn on the graph. "0" corresponds to no limit.
            </p>
          </div>
        </details>
        <details>
          <summary>
            opacity
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].marker</dd>
              <dt>Type:</dt>
              <dd>number or array of numbers between or equal to 0 and 1</dd>
            </dl>
            <p>
              Sets the marker opacity.
            </p>
          </div>
        </details>
        <details>
          <summary>
            reversescale
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].marker</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Reverses the color mapping if true. Has an effect only if in `marker.color`is set to a numerical array. If true, `marker.cmin` will correspond to the last color in the array and `marker.cmax` will correspond to the first color.
            </p>
          </div>
        </details>
        <details>
          <summary>
            showscale
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].marker</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Determines whether or not a colorbar is displayed for this trace. Has an effect only if in `marker.color`is set to a numerical array.
            </p>
          </div>
        </details>
        <details>
          <summary>
            size
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].marker</dd>
              <dt>Type:</dt>
              <dd>number or array of numbers greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>6</dd>
            </dl>
            <p>
              Sets the marker size (in px).
            </p>
          </div>
        </details>
        <details>
          <summary>
            sizemin
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].marker</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
            <p>
              Has an effect only if `marker.size` is set to a numerical array. Sets the minimum size (in px) of the rendered marker points.
            </p>
          </div>
        </details>
        <details>
          <summary>
            sizemode
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].marker</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "diameter" | "area" )</dd>
              <dt>Default:</dt>
              <dd>"diameter"</dd>
            </dl>
            <p>
              Has an effect only if `marker.size` is set to a numerical array. Sets the rule for which the data in `size` is converted to pixels.
            </p>
          </div>
        </details>
        <details>
          <summary>
            sizeref
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].marker</dd>
              <dt>Type:</dt>
              <dd>number</dd>
              <dt>Default:</dt>
              <dd>1</dd>
            </dl>
            <p>
              Has an effect only if `marker.size` is set to a numerical array. Sets the scale factor used to determine the rendered size of marker points. Use with `sizemin` and `sizemode`.
            </p>
          </div>
        </details>
        <details>
          <summary>
            symbol
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].marker</dd>
              <dt>Type:</dt>
              <dd>enumerated or array of enumerateds , one of ( "0" | "circle" | "100" | "circle-open" | "200" | "circle-dot" | "300" | "circle-open-dot" | "1" | "square" | "101" | "square-open" | "201" | "square-dot" | "301" | "square-open-dot" | "2" | "diamond" | "102" | "diamond-open" | "202" | "diamond-dot" | "302" | "diamond-open-dot" | "3" | "cross" | "103" | "cross-open" | "203" | "cross-dot" | "303" | "cross-open-dot" | "4" | "x" | "104" | "x-open" | "204" | "x-dot" | "304" | "x-open-dot" | "5" | "triangle-up" | "105" | "triangle-up-open" | "205" | "triangle-up-dot" | "305" | "triangle-up-open-dot" | "6" | "triangle-down" | "106" | "triangle-down-open" | "206" | "triangle-down-dot" | "306" | "triangle-down-open-dot" | "7" | "triangle-left" | "107" | "triangle-left-open" | "207" | "triangle-left-dot" | "307" | "triangle-left-open-dot" | "8" | "triangle-right" | "108" | "triangle-right-open" | "208" | "triangle-right-dot" | "308" | "triangle-right-open-dot" | "9" | "triangle-ne" | "109" | "triangle-ne-open" | "209" | "triangle-ne-dot" | "309" | "triangle-ne-open-dot" | "10" | "triangle-se" | "110" | "triangle-se-open" | "210" | "triangle-se-dot" | "310" | "triangle-se-open-dot" | "11" | "triangle-sw" | "111" | "triangle-sw-open" | "211" | "triangle-sw-dot" | "311" | "triangle-sw-open-dot" | "12" | "triangle-nw" | "112" | "triangle-nw-open" | "212" | "triangle-nw-dot" | "312" | "triangle-nw-open-dot" | "13" | "pentagon" | "113" | "pentagon-open" | "213" | "pentagon-dot" | "313" | "pentagon-open-dot" | "14" | "hexagon" | "114" | "hexagon-open" | "214" | "hexagon-dot" | "314" | "hexagon-open-dot" | "15" | "hexagon2" | "115" | "hexagon2-open" | "215" | "hexagon2-dot" | "315" | "hexagon2-open-dot" | "16" | "octagon" | "116" | "octagon-open" | "216" | "octagon-dot" | "316" | "octagon-open-dot" | "17" | "star" | "117" | "star-open" | "217" | "star-dot" | "317" | "star-open-dot" | "18" | "hexagram" | "118" | "hexagram-open" | "218" | "hexagram-dot" | "318" | "hexagram-open-dot" | "19" | "star-triangle-up" | "119" | "star-triangle-up-open" | "219" | "star-triangle-up-dot" | "319" | "star-triangle-up-open-dot" | "20" | "star-triangle-down" | "120" | "star-triangle-down-open" | "220" | "star-triangle-down-dot" | "320" | "star-triangle-down-open-dot" | "21" | "star-square" | "121" | "star-square-open" | "221" | "star-square-dot" | "321" | "star-square-open-dot" | "22" | "star-diamond" | "122" | "star-diamond-open" | "222" | "star-diamond-dot" | "322" | "star-diamond-open-dot" | "23" | "diamond-tall" | "123" | "diamond-tall-open" | "223" | "diamond-tall-dot" | "323" | "diamond-tall-open-dot" | "24" | "diamond-wide" | "124" | "diamond-wide-open" | "224" | "diamond-wide-dot" | "324" | "diamond-wide-open-dot" | "25" | "hourglass" | "125" | "hourglass-open" | "26" | "bowtie" | "126" | "bowtie-open" | "27" | "circle-cross" | "127" | "circle-cross-open" | "28" | "circle-x" | "128" | "circle-x-open" | "29" | "square-cross" | "129" | "square-cross-open" | "30" | "square-x" | "130" | "square-x-open" | "31" | "diamond-cross" | "131" | "diamond-cross-open" | "32" | "diamond-x" | "132" | "diamond-x-open" | "33" | "cross-thin" | "133" | "cross-thin-open" | "34" | "x-thin" | "134" | "x-thin-open" | "35" | "asterisk" | "135" | "asterisk-open" | "36" | "hash" | "136" | "hash-open" | "236" | "hash-dot" | "336" | "hash-open-dot" | "37" | "y-up" | "137" | "y-up-open" | "38" | "y-down" | "138" | "y-down-open" | "39" | "y-left" | "139" | "y-left-open" | "40" | "y-right" | "140" | "y-right-open" | "41" | "line-ew" | "141" | "line-ew-open" | "42" | "line-ns" | "142" | "line-ns-open" | "43" | "line-ne" | "143" | "line-ne-open" | "44" | "line-nw" | "144" | "line-nw-open" )</dd>
              <dt>Default:</dt>
              <dd>"circle"</dd>
            </dl>
            <p>
              Sets the marker symbol type. Adding 100 is equivalent to appending "-open" to a symbol name. Adding 200 is equivalent to appending "-dot" to a symbol name. Adding 300 is equivalent to appending "-open-dot" or "dot-open" to a symbol name.
            </p>
          </div>
        </details>
      </div>
      <details>
        <summary>
          meta
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>number or categorical coordinate string</dd>
          </dl>
          <p>
            Assigns extra meta information associated with this trace that can be used in various text attributes. Attributes such as trace `name`, graph, axis and colorbar `title.text`, annotation `text` `rangeselector`, `updatemenues` and `sliders` `label` text all support `meta`. To access the trace `meta` values in an attribute in the same trace, simply use `%{meta[i]}` where `i` is the index or key of the `meta` item in question. To access trace `meta` in layout attributes, use `%{data[n[.meta[i]}` where `i` is the index or key of the `meta` and `n` is the trace index.
          </p>
        </div>
      </details>
      <details>
        <summary>
          mode
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>flaglist string. Any combination of "lines", "markers", "text" joined with a "+" OR "none".</dd>
          </dl>
          <p>
            Determines the drawing mode for this scatter trace. If the provided `mode` includes "text" then the `text` elements appear at the coordinates. Otherwise, the `text` elements appear on hover. If there are less than 20 points and the trace is not stacked then the default is "lines+markers". Otherwise, "lines".
          </p>
        </div>
      </details>
      <details>
        <summary>
          name
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>string</dd>
          </dl>
          <p>
            Sets the trace name. The trace name appear as the legend item and on hover.
          </p>
        </div>
      </details>
      <details>
        <summary>
          opacity
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>number between or equal to 0 and 1</dd>
            <dt>Default:</dt>
            <dd>1</dd>
          </dl>
          <p>
            Sets the opacity of the trace.
          </p>
        </div>
      </details>
      <details>
        <summary>
          orientation
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>enumerated , one of ( "v" | "h" )</dd>
          </dl>
          <p>
            Only relevant when `stackgroup` is used, and only the first `orientation` found in the `stackgroup` will be used - including if `visible` is "legendonly" but not if it is `false`. Sets the stacking direction. With "v" ("h"), the y (x) values of subsequent traces are added. Also affects the default value of `fill`.
          </p>
        </div>
      </details>
      <details>
        <summary>
          selected
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>object containing one or more of the keys listed below.</dd>
          </dl>
          <p>
            Sets the marker opacity of selected points.
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            marker
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].selected</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
            </dl>
            <p>
              Sets the marker opacity of selected points.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              color
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].selected.marker</dd>
                <dt>Type:</dt>
                <dd>color</dd>
              </dl>
              <p>
                Sets the marker color of selected points.
              </p>
            </div>
          </details>
          <details>
            <summary>
              opacity
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].selected.marker</dd>
                <dt>Type:</dt>
                <dd>number between or equal to 0 and 1</dd>
              </dl>
              <p>
                Sets the marker opacity of selected points.
              </p>
            </div>
          </details>
          <details>
            <summary>
              size
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].selected.marker</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
              </dl>
              <p>
                Sets the marker size of selected points.
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            textfont
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].selected</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
            </dl>
            <p>
              Sets the text font color of selected points.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              color
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].selected.textfont</dd>
                <dt>Type:</dt>
                <dd>color</dd>
              </dl>
              <p>
                Sets the text font color of selected points.
              </p>
            </div>
          </details>
        </div>
      </div>
      <details>
        <summary>
          selectedpoints
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>number or categorical coordinate string</dd>
          </dl>
          <p>
            Array containing integer indices of selected points. Has an effect only for traces that support selections. Note that an empty array means an empty selection where the `unselected` are turned on for all points, whereas, any other non-array values means no selection all where the `selected` and `unselected` styles have no effect.
          </p>
        </div>
      </details>
      <details>
        <summary>
          showlegend
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>boolean</dd>
            <dt>Default:</dt>
            <dd>true</dd>
          </dl>
          <p>
            Determines whether or not an item corresponding to this trace is shown in the legend.
          </p>
        </div>
      </details>
      <details>
        <summary>
          stackgaps
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>enumerated , one of ( "infer zero" | "interpolate" )</dd>
            <dt>Default:</dt>
            <dd>"infer zero"</dd>
          </dl>
          <p>
            Only relevant when `stackgroup` is used, and only the first `stackgaps` found in the `stackgroup` will be used - including if `visible` is "legendonly" but not if it is `false`. Determines how we handle locations at which other traces in this group have data but this one does not. With "infer zero" we insert a zero at these locations. With "interpolate" we linearly interpolate between existing values, and extrapolate a constant beyond the existing values.
          </p>
        </div>
      </details>
      <details>
        <summary>
          stackgroup
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>string</dd>
            <dt>Default:</dt>
            <dd>""</dd>
          </dl>
          <p>
            Set several scatter traces (on the same subplot) to the same stackgroup in order to add their y values (or their x values if `orientation` is "h"). If blank or omitted this trace will not be stacked. Stacking also turns `fill` on by default, using "tonexty" ("tonextx") if `orientation` is "h" ("v") and sets the default `mode` to "lines" irrespective of point count. You can only stack on a numeric (linear or log) axis. Traces in a `stackgroup` will only fill to (or be filled to) other traces in the same group. With multiple `stackgroup`s or some traces stacked and some not, if fill-linked traces are not already consecutive, the later ones will be pushed down in the drawing order.
          </p>
        </div>
      </details>
      <details>
        <summary>
          text
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>string or array of strings</dd>
            <dt>Default:</dt>
            <dd>""</dd>
          </dl>
          <p>
            Sets text elements associated with each (x,y) pair. If a single string, the same string appears over all the data points. If an array of string, the items are mapped in order to the this trace's (x,y) coordinates. If trace `hoverinfo` contains a "text" flag and "hovertext" is not set, these elements will be seen in the hover labels.
          </p>
        </div>
      </details>
      <details>
        <summary>
          textfont
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>object containing one or more of the keys listed below.</dd>
          </dl>
          <p>
            Sets the text font.
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            color
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].textfont</dd>
              <dt>Type:</dt>
              <dd>color or array of colors</dd>
            </dl>
          </div>
        </details>
        <details>
          <summary>
            family
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].textfont</dd>
              <dt>Type:</dt>
              <dd>string or array of strings</dd>
            </dl>
            <p>
              HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
            </p>
          </div>
        </details>
        <details>
          <summary>
            size
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].textfont</dd>
              <dt>Type:</dt>
              <dd>number or array of numbers greater than or equal to 1</dd>
            </dl>
          </div>
        </details>
      </div>
      <details>
        <summary>
          textposition
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>enumerated or array of enumerateds , one of ( "top left" | "top center" | "top right" | "middle left" | "middle center" | "middle right" | "bottom left" | "bottom center" | "bottom right" )</dd>
            <dt>Default:</dt>
            <dd>"middle center"</dd>
          </dl>
          <p>
            Sets the positions of the `text` elements with respects to the (x,y) coordinates.
          </p>
        </div>
      </details>
      <details>
        <summary>
          texttemplate
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>string or array of strings</dd>
            <dt>Default:</dt>
            <dd>""</dd>
          </dl>
          <p>
            Template string used for rendering the information text that appear on points. Note that this will override `textinfo`. Variables are inserted using %{variable}, for example "y: %{y}". Numbers are formatted using d3-format's syntax %{variable:d3-format}, for example "Price: %{y:$.2f}". https://github.com/d3/d3-3.x-api-reference/blob/master/Formatting.md#d3_format for details on the formatting syntax. Dates are formatted using d3-time-format's syntax %{variable|d3-time-format}, for example "Day: %{2019-01-01|%A}". https://github.com/d3/d3-3.x-api-reference/blob/master/Time-Formatting.md#format for details on the date formatting syntax. Every attributes that can be specified per-point (the ones that are `arrayOk: true`) are available.
          </p>
        </div>
      </details>
      <details>
        <summary>
          type
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>"scatter"</dd>
          </dl>
        </div>
      </details>
      <details>
        <summary>
          uirevision
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>number or categorical coordinate string</dd>
          </dl>
          <p>
            Controls persistence of some user-driven changes to the trace: `constraintrange` in `parcoords` traces, as well as some `editable: true` modifications such as `name` and `colorbar.title`. Defaults to `layout.uirevision`. Note that other user-driven trace attribute changes are controlled by `layout` attributes: `trace.visible` is controlled by `layout.legend.uirevision`, `selectedpoints` is controlled by `layout.selectionrevision`, and `colorbar.(x|y)` (accessible with `config: {editable: true}`) is controlled by `layout.editrevision`. Trace changes are tracked by `uid`, which only falls back on trace index if no `uid` is provided. So if your app can add/remove traces before the end of the `data` array, such that the same trace has a different index, you can still preserve user-driven changes if you give each trace a `uid` that stays with it as it moves.
          </p>
        </div>
      </details>
      <details>
        <summary>
          unselected
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>object containing one or more of the keys listed below.</dd>
          </dl>
          <p>
            Sets the marker opacity of unselected points, applied only when a selection exists.
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            marker
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].unselected</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
            </dl>
            <p>
              Sets the marker opacity of unselected points, applied only when a selection exists.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              color
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].unselected.marker</dd>
                <dt>Type:</dt>
                <dd>color</dd>
              </dl>
              <p>
                Sets the marker color of unselected points, applied only when a selection exists.
              </p>
            </div>
          </details>
          <details>
            <summary>
              opacity
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].unselected.marker</dd>
                <dt>Type:</dt>
                <dd>number between or equal to 0 and 1</dd>
              </dl>
              <p>
                Sets the marker opacity of unselected points, applied only when a selection exists.
              </p>
            </div>
          </details>
          <details>
            <summary>
              size
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].unselected.marker</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
              </dl>
              <p>
                Sets the marker size of unselected points, applied only when a selection exists.
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            textfont
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scatter].unselected</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
            </dl>
            <p>
              Sets the text font color of unselected points, applied only when a selection exists.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              color
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scatter].unselected.textfont</dd>
                <dt>Type:</dt>
                <dd>color</dd>
              </dl>
              <p>
                Sets the text font color of unselected points, applied only when a selection exists.
              </p>
            </div>
          </details>
        </div>
      </div>
      <details>
        <summary>
          visible
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>enumerated , one of ( true | false | "legendonly" )</dd>
            <dt>Default:</dt>
            <dd>true</dd>
          </dl>
          <p>
            Determines whether or not this trace is visible. If "legendonly", the trace is not drawn, but can appear as a legend item (provided that the legend itself is visible).
          </p>
        </div>
      </details>
      <details>
        <summary>
          x
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>data array</dd>
          </dl>
          <p>
            Sets the x coordinates.
          </p>
        </div>
      </details>
      <details>
        <summary>
          x0
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>number or categorical coordinate string</dd>
            <dt>Default:</dt>
            <dd>0</dd>
          </dl>
          <p>
            Alternate to `x`. Builds a linear space of x coordinates. Use with `dx` where `x0` is the starting coordinate and `dx` the step.
          </p>
        </div>
      </details>
      <details>
        <summary>
          xaxis
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>subplotid</dd>
            <dt>Default:</dt>
            <dd>x</dd>
          </dl>
          <p>
            Sets a reference between this trace's x coordinates and a 2D cartesian x axis. If "x" (the default value), the x coordinates refer to `layout.xaxis`. If "x2", the x coordinates refer to `layout.xaxis2`, and so on.
          </p>
        </div>
      </details>
      <details>
        <summary>
          xcalendar
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>enumerated , one of ( "gregorian" | "chinese" | "coptic" | "discworld" | "ethiopian" | "hebrew" | "islamic" | "julian" | "mayan" | "nanakshahi" | "nepali" | "persian" | "jalali" | "taiwan" | "thai" | "ummalqura" )</dd>
            <dt>Default:</dt>
            <dd>"gregorian"</dd>
          </dl>
          <p>
            Sets the calendar system to use with `x` date data.
          </p>
        </div>
      </details>
      <details>
        <summary>
          y
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>data array</dd>
          </dl>
          <p>
            Sets the y coordinates.
          </p>
        </div>
      </details>
      <details>
        <summary>
          y0
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>number or categorical coordinate string</dd>
            <dt>Default:</dt>
            <dd>0</dd>
          </dl>
          <p>
            Alternate to `y`. Builds a linear space of y coordinates. Use with `dy` where `y0` is the starting coordinate and `dy` the step.
          </p>
        </div>
      </details>
      <details>
        <summary>
          yaxis
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>subplotid</dd>
            <dt>Default:</dt>
            <dd>y</dd>
          </dl>
          <p>
            Sets a reference between this trace's y coordinates and a 2D cartesian y axis. If "y" (the default value), the y coordinates refer to `layout.yaxis`. If "y2", the y coordinates refer to `layout.yaxis2`, and so on.
          </p>
        </div>
      </details>
      <details>
        <summary>
          ycalendar
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scatter]</dd>
            <dt>Type:</dt>
            <dd>enumerated , one of ( "gregorian" | "chinese" | "coptic" | "discworld" | "ethiopian" | "hebrew" | "islamic" | "julian" | "mayan" | "nanakshahi" | "nepali" | "persian" | "jalali" | "taiwan" | "thai" | "ummalqura" )</dd>
            <dt>Default:</dt>
            <dd>"gregorian"</dd>
          </dl>
          <p>
            Sets the calendar system to use with `y` date data.
          </p>
        </div>
      </details>
    </p>
  </div>
</details>
<details>
  <summary>
    2. Scattergl traces
  </summary>
  <div style="margin-left: 22px">
    <p>
      A scattergl trace is an object with the key "type" equal to "scattergl" (i.e. {"type": "scattergl"}) and any of the keys listed below. The data visualized as scatter point or lines is set in `x` and `y` using the WebGL plotting engine. Bubble charts are achieved by setting `marker.size` and/or `marker.color` to a numerical arrays.
    </p>
    <p>
      <details>
        <summary>
          connectgaps
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scattergl]</dd>
            <dt>Type:</dt>
            <dd>boolean</dd>
          </dl>
          <p>
            Determines whether or not gaps (i.e. {nan} or missing values) in the provided data arrays are connected.
          </p>
        </div>
      </details>
      <details>
        <summary>
          customdata
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scattergl]</dd>
            <dt>Type:</dt>
            <dd>data array</dd>
          </dl>
          <p>
            Assigns extra data each datum. This may be useful when listening to hover, click and selection events. Note that, "scatter" traces also appends customdata items in the markers DOM elements
          </p>
        </div>
      </details>
      <details>
        <summary>
          dx
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scattergl]</dd>
            <dt>Type:</dt>
            <dd>number</dd>
            <dt>Default:</dt>
            <dd>1</dd>
          </dl>
          <p>
            Sets the x coordinate step. See `x0` for more info.
          </p>
        </div>
      </details>
      <details>
        <summary>
          dy
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scattergl]</dd>
            <dt>Type:</dt>
            <dd>number</dd>
            <dt>Default:</dt>
            <dd>1</dd>
          </dl>
          <p>
            Sets the y coordinate step. See `y0` for more info.
          </p>
        </div>
      </details>
      <details>
        <summary>
          error_x
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scattergl]</dd>
            <dt>Type:</dt>
            <dd>object containing one or more of the keys listed below.</dd>
            <dt>Default:</dt>
            <dd>10</dd>
          </dl>
          <p>
            Determines whether or not this set of error bars is visible.
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            array
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].error_x</dd>
              <dt>Type:</dt>
              <dd>data array</dd>
            </dl>
            <p>
              Sets the data corresponding the length of each error bar. Values are plotted relative to the underlying data.
            </p>
          </div>
        </details>
        <details>
          <summary>
            arrayminus
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].error_x</dd>
              <dt>Type:</dt>
              <dd>data array</dd>
            </dl>
            <p>
              Sets the data corresponding the length of each error bar in the bottom (left) direction for vertical (horizontal) bars Values are plotted relative to the underlying data.
            </p>
          </div>
        </details>
        <details>
          <summary>
            color
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].error_x</dd>
              <dt>Type:</dt>
              <dd>color</dd>
            </dl>
            <p>
              Sets the stoke color of the error bars.
            </p>
          </div>
        </details>
        <details>
          <summary>
            copy_ystyle
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].error_x</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
          </div>
        </details>
        <details>
          <summary>
            symmetric
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].error_x</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Determines whether or not the error bars have the same length in both direction (top/bottom for vertical bars, left/right for horizontal bars.
            </p>
          </div>
        </details>
        <details>
          <summary>
            thickness
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].error_x</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>2</dd>
            </dl>
            <p>
              Sets the thickness (in px) of the error bars.
            </p>
          </div>
        </details>
        <details>
          <summary>
            traceref
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].error_x</dd>
              <dt>Type:</dt>
              <dd>integer greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
          </div>
        </details>
        <details>
          <summary>
            tracerefminus
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].error_x</dd>
              <dt>Type:</dt>
              <dd>integer greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
          </div>
        </details>
        <details>
          <summary>
            type
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].error_x</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "percent" | "constant" | "sqrt" | "data" )</dd>
            </dl>
            <p>
              Determines the rule used to generate the error bars. If "constant`, the bar lengths are of a constant value. Set this constant in `value`. If "percent", the bar lengths correspond to a percentage of underlying data. Set this percentage in `value`. If "sqrt", the bar lengths correspond to the sqaure of the underlying data. If "data", the bar lengths are set with data set `array`.
            </p>
          </div>
        </details>
        <details>
          <summary>
            value
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].error_x</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>10</dd>
            </dl>
            <p>
              Sets the value of either the percentage (if `type` is set to "percent") or the constant (if `type` is set to "constant") corresponding to the lengths of the error bars.
            </p>
          </div>
        </details>
        <details>
          <summary>
            valueminus
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].error_x</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>10</dd>
            </dl>
            <p>
              Sets the value of either the percentage (if `type` is set to "percent") or the constant (if `type` is set to "constant") corresponding to the lengths of the error bars in the bottom (left) direction for vertical (horizontal) bars
            </p>
          </div>
        </details>
        <details>
          <summary>
            visible
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].error_x</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Determines whether or not this set of error bars is visible.
            </p>
          </div>
        </details>
        <details>
          <summary>
            width
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].error_x</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
            </dl>
            <p>
              Sets the width (in px) of the cross-bar at both ends of the error bars.
            </p>
          </div>
        </details>
      </div>
      <details>
        <summary>
          error_y
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scattergl]</dd>
            <dt>Type:</dt>
            <dd>object containing one or more of the keys listed below.</dd>
            <dt>Default:</dt>
            <dd>10</dd>
          </dl>
          <p>
            Determines whether or not this set of error bars is visible.
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            array
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].error_y</dd>
              <dt>Type:</dt>
              <dd>data array</dd>
            </dl>
            <p>
              Sets the data corresponding the length of each error bar. Values are plotted relative to the underlying data.
            </p>
          </div>
        </details>
        <details>
          <summary>
            arrayminus
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].error_y</dd>
              <dt>Type:</dt>
              <dd>data array</dd>
            </dl>
            <p>
              Sets the data corresponding the length of each error bar in the bottom (left) direction for vertical (horizontal) bars Values are plotted relative to the underlying data.
            </p>
          </div>
        </details>
        <details>
          <summary>
            color
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].error_y</dd>
              <dt>Type:</dt>
              <dd>color</dd>
            </dl>
            <p>
              Sets the stoke color of the error bars.
            </p>
          </div>
        </details>
        <details>
          <summary>
            symmetric
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].error_y</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Determines whether or not the error bars have the same length in both direction (top/bottom for vertical bars, left/right for horizontal bars.
            </p>
          </div>
        </details>
        <details>
          <summary>
            thickness
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].error_y</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>2</dd>
            </dl>
            <p>
              Sets the thickness (in px) of the error bars.
            </p>
          </div>
        </details>
        <details>
          <summary>
            traceref
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].error_y</dd>
              <dt>Type:</dt>
              <dd>integer greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
          </div>
        </details>
        <details>
          <summary>
            tracerefminus
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].error_y</dd>
              <dt>Type:</dt>
              <dd>integer greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
          </div>
        </details>
        <details>
          <summary>
            type
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].error_y</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "percent" | "constant" | "sqrt" | "data" )</dd>
            </dl>
            <p>
              Determines the rule used to generate the error bars. If "constant`, the bar lengths are of a constant value. Set this constant in `value`. If "percent", the bar lengths correspond to a percentage of underlying data. Set this percentage in `value`. If "sqrt", the bar lengths correspond to the sqaure of the underlying data. If "data", the bar lengths are set with data set `array`.
            </p>
          </div>
        </details>
        <details>
          <summary>
            value
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].error_y</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>10</dd>
            </dl>
            <p>
              Sets the value of either the percentage (if `type` is set to "percent") or the constant (if `type` is set to "constant") corresponding to the lengths of the error bars.
            </p>
          </div>
        </details>
        <details>
          <summary>
            valueminus
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].error_y</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>10</dd>
            </dl>
            <p>
              Sets the value of either the percentage (if `type` is set to "percent") or the constant (if `type` is set to "constant") corresponding to the lengths of the error bars in the bottom (left) direction for vertical (horizontal) bars
            </p>
          </div>
        </details>
        <details>
          <summary>
            visible
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].error_y</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Determines whether or not this set of error bars is visible.
            </p>
          </div>
        </details>
        <details>
          <summary>
            width
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].error_y</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
            </dl>
            <p>
              Sets the width (in px) of the cross-bar at both ends of the error bars.
            </p>
          </div>
        </details>
      </div>
      <details>
        <summary>
          fill
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scattergl]</dd>
            <dt>Type:</dt>
            <dd>enumerated , one of ( "none" | "tozeroy" | "tozerox" | "tonexty" | "tonextx" | "toself" | "tonext" )</dd>
            <dt>Default:</dt>
            <dd>"none"</dd>
          </dl>
          <p>
            Sets the area to fill with a solid color. Defaults to "none" unless this trace is stacked, then it gets "tonexty" ("tonextx") if `orientation` is "v" ("h") Use with `fillcolor` if not "none". "tozerox" and "tozeroy" fill to x=0 and y=0 respectively. "tonextx" and "tonexty" fill between the endpoints of this trace and the endpoints of the trace before it, connecting those endpoints with straight lines (to make a stacked area graph); if there is no trace before it, they behave like "tozerox" and "tozeroy". "toself" connects the endpoints of the trace (or each segment of the trace if it has gaps) into a closed shape. "tonext" fills the space between two traces if one completely encloses the other (eg consecutive contour lines), and behaves like "toself" if there is no trace before it. "tonext" should not be used if one trace does not enclose the other. Traces in a `stackgroup` will only fill to (or be filled to) other traces in the same group. With multiple `stackgroup`s or some traces stacked and some not, if fill-linked traces are not already consecutive, the later ones will be pushed down in the drawing order.
          </p>
        </div>
      </details>
      <details>
        <summary>
          fillcolor
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scattergl]</dd>
            <dt>Type:</dt>
            <dd>color</dd>
          </dl>
          <p>
            Sets the fill color. Defaults to a half-transparent variant of the line color, marker color, or marker line color, whichever is available.
          </p>
        </div>
      </details>
      <details>
        <summary>
          hoverinfo
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scattergl]</dd>
            <dt>Type:</dt>
            <dd>flaglist string. Any combination of "x", "y", "z", "text", "name" joined with a "+" OR "all" or "none" or "skip".</dd>
            <dt>Default:</dt>
            <dd>"all"</dd>
          </dl>
          <p>
            Determines which trace information appear on hover. If `none` or `skip` are set, no information is displayed upon hovering. But, if `none` is set, click and hover events are still fired.
          </p>
        </div>
      </details>
      <details>
        <summary>
          hoverlabel
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scattergl]</dd>
            <dt>Type:</dt>
            <dd>object containing one or more of the keys listed below.</dd>
            <dt>Default:</dt>
            <dd>"auto"</dd>
          </dl>
          <p>
            Sets the background color of the hover labels for this trace
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            align
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].hoverlabel</dd>
              <dt>Type:</dt>
              <dd>enumerated or array of enumerateds , one of ( "left" | "right" | "auto" )</dd>
              <dt>Default:</dt>
              <dd>"auto"</dd>
            </dl>
            <p>
              Sets the horizontal alignment of the text content within hover label box. Has an effect only if the hover label text spans more two or more lines
            </p>
          </div>
        </details>
        <details>
          <summary>
            bgcolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].hoverlabel</dd>
              <dt>Type:</dt>
              <dd>color or array of colors</dd>
            </dl>
            <p>
              Sets the background color of the hover labels for this trace
            </p>
          </div>
        </details>
        <details>
          <summary>
            bordercolor
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].hoverlabel</dd>
              <dt>Type:</dt>
              <dd>color or array of colors</dd>
            </dl>
            <p>
              Sets the border color of the hover labels for this trace.
            </p>
          </div>
        </details>
        <details>
          <summary>
            font
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].hoverlabel</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
            </dl>
            <p>
              Sets the font used in hover labels.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              color
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].hoverlabel.font</dd>
                <dt>Type:</dt>
                <dd>color or array of colors</dd>
              </dl>
            </div>
          </details>
          <details>
            <summary>
              family
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].hoverlabel.font</dd>
                <dt>Type:</dt>
                <dd>string or array of strings</dd>
              </dl>
              <p>
                HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
              </p>
            </div>
          </details>
          <details>
            <summary>
              size
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].hoverlabel.font</dd>
                <dt>Type:</dt>
                <dd>number or array of numbers greater than or equal to 1</dd>
              </dl>
            </div>
          </details>
        </div>
        <details>
          <summary>
            namelength
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].hoverlabel</dd>
              <dt>Type:</dt>
              <dd>integer or array of integers greater than or equal to -1</dd>
              <dt>Default:</dt>
              <dd>15</dd>
            </dl>
            <p>
              Sets the default length (in number of characters) of the trace name in the hover labels for all traces. -1 shows the whole name regardless of length. 0-3 shows the first 0-3 characters, and an integer &gt;3 will show the whole name if it is less than that many characters, but if it is longer, will truncate to `namelength - 3` characters and add an ellipsis.
            </p>
          </div>
        </details>
      </div>
      <details>
        <summary>
          hovertemplate
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scattergl]</dd>
            <dt>Type:</dt>
            <dd>string or array of strings</dd>
            <dt>Default:</dt>
            <dd>""</dd>
          </dl>
          <p>
            Template string used for rendering the information that appear on hover box. Note that this will override `hoverinfo`. Variables are inserted using %{variable}, for example "y: %{y}". Numbers are formatted using d3-format's syntax %{variable:d3-format}, for example "Price: %{y:$.2f}". https://github.com/d3/d3-3.x-api-reference/blob/master/Formatting.md#d3_format for details on the formatting syntax. Dates are formatted using d3-time-format's syntax %{variable|d3-time-format}, for example "Day: %{2019-01-01|%A}". https://github.com/d3/d3-3.x-api-reference/blob/master/Time-Formatting.md#format for details on the date formatting syntax. The variables available in `hovertemplate` are the ones emitted as event data described at this link https://plot.ly/javascript/plotlyjs-events/#event-data. Additionally, every attributes that can be specified per-point (the ones that are `arrayOk: true`) are available. Anything contained in tag `&lt;extra&gt;` is displayed in the secondary box, for example "&lt;extra&gt;{fullData.name}&lt;/extra&gt;". To hide the secondary box completely, use an empty tag `&lt;extra&gt;&lt;/extra&gt;`.
          </p>
        </div>
      </details>
      <details>
        <summary>
          hovertext
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scattergl]</dd>
            <dt>Type:</dt>
            <dd>string or array of strings</dd>
            <dt>Default:</dt>
            <dd>""</dd>
          </dl>
          <p>
            Sets hover text elements associated with each (x,y) pair. If a single string, the same string appears over all the data points. If an array of string, the items are mapped in order to the this trace's (x,y) coordinates. To be seen, trace `hoverinfo` must contain a "text" flag.
          </p>
        </div>
      </details>
      <details>
        <summary>
          ids
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scattergl]</dd>
            <dt>Type:</dt>
            <dd>data array</dd>
          </dl>
          <p>
            Assigns id labels to each datum. These ids for object constancy of data points during animation. Should be an array of strings, not numbers or any other type.
          </p>
        </div>
      </details>
      <details>
        <summary>
          legendgroup
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scattergl]</dd>
            <dt>Type:</dt>
            <dd>string</dd>
            <dt>Default:</dt>
            <dd>""</dd>
          </dl>
          <p>
            Sets the legend group for this trace. Traces part of the same legend group hide/show at the same time when toggling legend items.
          </p>
        </div>
      </details>
      <details>
        <summary>
          line
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scattergl]</dd>
            <dt>Type:</dt>
            <dd>object containing one or more of the keys listed below.</dd>
            <dt>Default:</dt>
            <dd>2</dd>
          </dl>
          <p>
            Sets the line color.
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            color
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].line</dd>
              <dt>Type:</dt>
              <dd>color</dd>
            </dl>
            <p>
              Sets the line color.
            </p>
          </div>
        </details>
        <details>
          <summary>
            dash
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].line</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "solid" | "dot" | "dash" | "longdash" | "dashdot" | "longdashdot" )</dd>
              <dt>Default:</dt>
              <dd>"solid"</dd>
            </dl>
            <p>
              Sets the style of the lines.
            </p>
          </div>
        </details>
        <details>
          <summary>
            shape
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].line</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "linear" | "hv" | "vh" | "hvh" | "vhv" )</dd>
              <dt>Default:</dt>
              <dd>"linear"</dd>
            </dl>
            <p>
              Determines the line shape. The values correspond to step-wise line shapes.
            </p>
          </div>
        </details>
        <details>
          <summary>
            width
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].line</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>2</dd>
            </dl>
            <p>
              Sets the line width (in px).
            </p>
          </div>
        </details>
      </div>
      <details>
        <summary>
          marker
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scattergl]</dd>
            <dt>Type:</dt>
            <dd>object containing one or more of the keys listed below.</dd>
            <dt>Default:</dt>
            <dd>true</dd>
          </dl>
          <p>
            Sets themarkercolor. It accepts either a specific color or an array of numbers that are mapped to the colorscale relative to the max and min values of the array or relative to `marker.cmin` and `marker.cmax` if set.
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            autocolorscale
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].marker</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
              <dt>Default:</dt>
              <dd>true</dd>
            </dl>
            <p>
              Determines whether the colorscale is a default palette (`autocolorscale: true`) or the palette determined by `marker.colorscale`. Has an effect only if in `marker.color`is set to a numerical array. In case `colorscale` is unspecified or `autocolorscale` is true, the default palette will be chosen according to whether numbers in the `color` array are all positive, all negative or mixed.
            </p>
          </div>
        </details>
        <details>
          <summary>
            cauto
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].marker</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
              <dt>Default:</dt>
              <dd>true</dd>
            </dl>
            <p>
              Determines whether or not the color domain is computed with respect to the input data (here in `marker.color`) or the bounds set in `marker.cmin` and `marker.cmax` Has an effect only if in `marker.color`is set to a numerical array. Defaults to `false` when `marker.cmin` and `marker.cmax` are set by the user.
            </p>
          </div>
        </details>
        <details>
          <summary>
            cmax
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].marker</dd>
              <dt>Type:</dt>
              <dd>number</dd>
            </dl>
            <p>
              Sets the upper bound of the color domain. Has an effect only if in `marker.color`is set to a numerical array. Value should have the same units as in `marker.color` and if set, `marker.cmin` must be set as well.
            </p>
          </div>
        </details>
        <details>
          <summary>
            cmid
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].marker</dd>
              <dt>Type:</dt>
              <dd>number</dd>
            </dl>
            <p>
              Sets the mid-point of the color domain by scaling `marker.cmin` and/or `marker.cmax` to be equidistant to this point. Has an effect only if in `marker.color`is set to a numerical array. Value should have the same units as in `marker.color`. Has no effect when `marker.cauto` is `false`.
            </p>
          </div>
        </details>
        <details>
          <summary>
            cmin
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].marker</dd>
              <dt>Type:</dt>
              <dd>number</dd>
            </dl>
            <p>
              Sets the lower bound of the color domain. Has an effect only if in `marker.color`is set to a numerical array. Value should have the same units as in `marker.color` and if set, `marker.cmax` must be set as well.
            </p>
          </div>
        </details>
        <details>
          <summary>
            color
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].marker</dd>
              <dt>Type:</dt>
              <dd>color or array of colors</dd>
            </dl>
            <p>
              Sets themarkercolor. It accepts either a specific color or an array of numbers that are mapped to the colorscale relative to the max and min values of the array or relative to `marker.cmin` and `marker.cmax` if set.
            </p>
          </div>
        </details>
        <details>
          <summary>
            coloraxis
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].marker</dd>
              <dt>Type:</dt>
              <dd>subplotid</dd>
            </dl>
            <p>
              Sets a reference to a shared color axis. References to these shared color axes are "coloraxis", "coloraxis2", "coloraxis3", etc. Settings for these shared color axes are set in the layout, under `layout.coloraxis`, `layout.coloraxis2`, etc. Note that multiple color scales can be linked to the same color axis.
            </p>
          </div>
        </details>
        <details>
          <summary>
            colorbar
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].marker</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>"pixels"</dd>
            </dl>
            <p>
              Determines whether this color bar's thickness (i.e. the measure in the constant color direction) is set in units of plot "fraction" or in "pixels". Use `thickness` to set the value.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              bgcolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"rgba(0,0,0,0)"</dd>
              </dl>
              <p>
                Sets the color of padded area.
              </p>
            </div>
          </details>
          <details>
            <summary>
              bordercolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets the axis line color.
              </p>
            </div>
          </details>
          <details>
            <summary>
              borderwidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                Sets the width (in px) or the border enclosing this color bar.
              </p>
            </div>
          </details>
          <details>
            <summary>
              dtick
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>number or categorical coordinate string</dd>
              </dl>
              <p>
                Sets the step in-between ticks on this axis. Use with `tick0`. Must be a positive number, or special strings available to "log" and "date" axes. If the axis `type` is "log", then ticks are set every 10^(n"dtick) where n is the tick number. For example, to set a tick mark at 1, 10, 100, 1000, ... set dtick to 1. To set tick marks at 1, 100, 10000, ... set dtick to 2. To set tick marks at 1, 5, 25, 125, 625, 3125, ... set dtick to log_10(5), or 0.69897000433. "log" has several special values; "L&lt;f&gt;", where `f` is a positive number, gives ticks linearly spaced in value (but not position). For example `tick0` = 0.1, `dtick` = "L0.5" will put ticks at 0.1, 0.6, 1.1, 1.6 etc. To show powers of 10 plus small digits between, use "D1" (all digits) or "D2" (only 2 and 5). `tick0` is ignored for "D1" and "D2". If the axis `type` is "date", then you must convert the time to milliseconds. For example, to set the interval between ticks to one day, set `dtick` to 86400000.0. "date" also has special values "M&lt;n&gt;" gives ticks spaced by a number of months. `n` must be a positive integer. To set ticks on the 15th of every third month, set `tick0` to "2000-01-15" and `dtick` to "M3". To set ticks every 4 years, set `dtick` to "M48"
              </p>
            </div>
          </details>
          <details>
            <summary>
              exponentformat
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "none" | "e" | "E" | "power" | "SI" | "B" )</dd>
                <dt>Default:</dt>
                <dd>"B"</dd>
              </dl>
              <p>
                Determines a formatting rule for the tick exponents. For example, consider the number 1,000,000,000. If "none", it appears as 1,000,000,000. If "e", 1e+9. If "E", 1E+9. If "power", 1x10^9 (with 9 in a super script). If "SI", 1G. If "B", 1B.
              </p>
            </div>
          </details>
          <details>
            <summary>
              len
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the length of the color bar This measure excludes the padding of both ends. That is, the color bar length is this length minus the padding on both ends.
              </p>
            </div>
          </details>
          <details>
            <summary>
              lenmode
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "fraction" | "pixels" )</dd>
                <dt>Default:</dt>
                <dd>"fraction"</dd>
              </dl>
              <p>
                Determines whether this color bar's length (i.e. the measure in the color variation direction) is set in units of plot "fraction" or in "pixels. Use `len` to set the value.
              </p>
            </div>
          </details>
          <details>
            <summary>
              nticks
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>integer greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>0</dd>
              </dl>
              <p>
                Specifies the maximum number of ticks for the particular axis. The actual number of ticks will be chosen automatically to be less than or equal to `nticks`. Has an effect only if `tickmode` is set to "auto".
              </p>
            </div>
          </details>
          <details>
            <summary>
              outlinecolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets the axis line color.
              </p>
            </div>
          </details>
          <details>
            <summary>
              outlinewidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the width (in px) of the axis line.
              </p>
            </div>
          </details>
          <details>
            <summary>
              separatethousands
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
              </dl>
              <p>
                If "true", even 4-digit integers are separated
              </p>
            </div>
          </details>
          <details>
            <summary>
              showexponent
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
                <dt>Default:</dt>
                <dd>"all"</dd>
              </dl>
              <p>
                If "all", all exponents are shown besides their significands. If "first", only the exponent of the first tick is shown. If "last", only the exponent of the last tick is shown. If "none", no exponents appear.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showticklabels
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not the tick labels are drawn.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showtickprefix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
                <dt>Default:</dt>
                <dd>"all"</dd>
              </dl>
              <p>
                If "all", all tick labels are displayed with a prefix. If "first", only the first tick is displayed with a prefix. If "last", only the last tick is displayed with a suffix. If "none", tick prefixes are hidden.
              </p>
            </div>
          </details>
          <details>
            <summary>
              showticksuffix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "all" | "first" | "last" | "none" )</dd>
                <dt>Default:</dt>
                <dd>"all"</dd>
              </dl>
              <p>
                Same as `showtickprefix` but for tick suffixes.
              </p>
            </div>
          </details>
          <details>
            <summary>
              thickness
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>30</dd>
              </dl>
              <p>
                Sets the thickness of the color bar This measure excludes the size of the padding, ticks and labels.
              </p>
            </div>
          </details>
          <details>
            <summary>
              thicknessmode
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "fraction" | "pixels" )</dd>
                <dt>Default:</dt>
                <dd>"pixels"</dd>
              </dl>
              <p>
                Determines whether this color bar's thickness (i.e. the measure in the constant color direction) is set in units of plot "fraction" or in "pixels". Use `thickness` to set the value.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tick0
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>number or categorical coordinate string</dd>
              </dl>
              <p>
                Sets the placement of the first tick on this axis. Use with `dtick`. If the axis `type` is "log", then you must take the log of your starting tick (e.g. to set the starting tick to 100, set the `tick0` to 2) except when `dtick`="L&lt;f&gt;" (see `dtick` for more info). If the axis `type` is "date", it should be a date string, like date data. If the axis `type` is "category", it should be a number, using the scale where each category is assigned a serial number from zero in the order it appears.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickangle
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>angle</dd>
                <dt>Default:</dt>
                <dd>"auto"</dd>
              </dl>
              <p>
                Sets the angle of the tick labels with respect to the horizontal. For example, a `tickangle` of -90 draws the tick labels vertically.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickcolor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>color</dd>
                <dt>Default:</dt>
                <dd>"#444"</dd>
              </dl>
              <p>
                Sets the tick color.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickfont
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
              </dl>
              <p>
                Sets the color bar's tick label font
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                color
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>data[type=scattergl].marker.colorbar.tickfont</dd>
                  <dt>Type:</dt>
                  <dd>color</dd>
                </dl>
              </div>
            </details>
            <details>
              <summary>
                family
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>data[type=scattergl].marker.colorbar.tickfont</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
                </p>
              </div>
            </details>
            <details>
              <summary>
                size
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>data[type=scattergl].marker.colorbar.tickfont</dd>
                  <dt>Type:</dt>
                  <dd>number greater than or equal to 1</dd>
                </dl>
              </div>
            </details>
          </div>
          <details>
            <summary>
              tickformat
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets the tick label formatting rule using d3 formatting mini-languages which are very similar to those in Python. For numbers, see: https://github.com/d3/d3-3.x-api-reference/blob/master/Formatting.md#d3_format And for dates see: https://github.com/d3/d3-3.x-api-reference/blob/master/Time-Formatting.md#format We add one item to d3's date formatter: "%{n}f" for fractional seconds with n digits. For example, "2016-10-13 09:15:23.456" with tickformat "%H~%\M~%S.%2f" would display "09~15~23.46"
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickformatstops
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>array of object where each object has one or more of the keys listed below.</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not this stop is used. If `false`, this stop is ignored even within its `dtickrange`.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                dtickrange
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>data[type=scattergl].marker.colorbar.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>array</dd>
                </dl>
                <p>
                  range ["min", "max"], where "min", "max" - dtick values which describe some zoom level, it is possible to omit "min" or "max" value by passing "null"
                </p>
              </div>
            </details>
            <details>
              <summary>
                enabled
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>data[type=scattergl].marker.colorbar.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>boolean</dd>
                  <dt>Default:</dt>
                  <dd>true</dd>
                </dl>
                <p>
                  Determines whether or not this stop is used. If `false`, this stop is ignored even within its `dtickrange`.
                </p>
              </div>
            </details>
            <details>
              <summary>
                name
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>data[type=scattergl].marker.colorbar.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  When used in a template, named items are created in the output figure in addition to any items the figure already has in this array. You can modify these items in the output figure by making your own item with `templateitemname` matching this `name` alongside your modifications (including `visible: false` or `enabled: false` to hide it). Has no effect outside of a template.
                </p>
              </div>
            </details>
            <details>
              <summary>
                templateitemname
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>data[type=scattergl].marker.colorbar.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  Used to refer to a named item in this array in the template. Named items from the template will be created even without a matching item in the input figure, but you can modify one by making an item with `templateitemname` matching its `name`, alongside your modifications (including `visible: false` or `enabled: false` to hide it). If there is no template or no matching item, this item will be hidden unless you explicitly show it with `visible: true`.
                </p>
              </div>
            </details>
            <details>
              <summary>
                value
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>data[type=scattergl].marker.colorbar.tickformatstops[]</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                  <dt>Default:</dt>
                  <dd>""</dd>
                </dl>
                <p>
                  string - dtickformat for described zoom level, the same as "tickformat"
                </p>
              </div>
            </details>
          </div>
          <details>
            <summary>
              ticklen
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>5</dd>
              </dl>
              <p>
                Sets the tick length (in px).
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickmode
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "auto" | "linear" | "array" )</dd>
              </dl>
              <p>
                Sets the tick mode for this axis. If "auto", the number of ticks is set via `nticks`. If "linear", the placement of the ticks is determined by a starting position `tick0` and a tick step `dtick` ("linear" is the default value if `tick0` and `dtick` are provided). If "array", the placement of the ticks is set via `tickvals` and the tick text is `ticktext`. ("array" is the default value if `tickvals` is provided).
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickprefix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets a tick label prefix.
              </p>
            </div>
          </details>
          <details>
            <summary>
              ticks
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "outside" | "inside" | "" )</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Determines whether ticks are drawn or not. If "", this axis' ticks are not drawn. If "outside" ("inside"), this axis' are drawn outside (inside) the axis lines.
              </p>
            </div>
          </details>
          <details>
            <summary>
              ticksuffix
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>string</dd>
                <dt>Default:</dt>
                <dd>""</dd>
              </dl>
              <p>
                Sets a tick label suffix.
              </p>
            </div>
          </details>
          <details>
            <summary>
              ticktext
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>data array</dd>
              </dl>
              <p>
                Sets the text displayed at the ticks position via `tickvals`. Only has an effect if `tickmode` is set to "array". Used with `tickvals`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickvals
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>data array</dd>
              </dl>
              <p>
                Sets the values at which ticks on this axis appear. Only has an effect if `tickmode` is set to "array". Used with `ticktext`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              tickwidth
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>1</dd>
              </dl>
              <p>
                Sets the tick width (in px).
              </p>
            </div>
          </details>
          <details>
            <summary>
              title
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>object containing one or more of the keys listed below.</dd>
                <dt>Default:</dt>
                <dd>"top"</dd>
              </dl>
              <p>
                Sets the title of the color bar. Note that before the existence of `title.text`, the title's contents used to be defined as the `title` attribute itself. This behavior has been deprecated.
              </p>
            </div>
          </details>
          <div style="margin-left: 22px">
            <details>
              <summary>
                font
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>data[type=scattergl].marker.colorbar.title</dd>
                  <dt>Type:</dt>
                  <dd>object containing one or more of the keys listed below.</dd>
                </dl>
                <p>
                  Sets this color bar's title font. Note that the title's font used to be set by the now deprecated `titlefont` attribute.
                </p>
              </div>
            </details>
            <div style="margin-left: 22px">
              <details>
                <summary>
                  color
                </summary>
                <div style="margin-left: 22px">
                  <dl>
                    <dt>Parent:</dt>
                    <dd>data[type=scattergl].marker.colorbar.title.font</dd>
                    <dt>Type:</dt>
                    <dd>color</dd>
                  </dl>
                </div>
              </details>
              <details>
                <summary>
                  family
                </summary>
                <div style="margin-left: 22px">
                  <dl>
                    <dt>Parent:</dt>
                    <dd>data[type=scattergl].marker.colorbar.title.font</dd>
                    <dt>Type:</dt>
                    <dd>string</dd>
                  </dl>
                  <p>
                    HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
                  </p>
                </div>
              </details>
              <details>
                <summary>
                  size
                </summary>
                <div style="margin-left: 22px">
                  <dl>
                    <dt>Parent:</dt>
                    <dd>data[type=scattergl].marker.colorbar.title.font</dd>
                    <dt>Type:</dt>
                    <dd>number greater than or equal to 1</dd>
                  </dl>
                </div>
              </details>
            </div>
            <details>
              <summary>
                side
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>data[type=scattergl].marker.colorbar.title</dd>
                  <dt>Type:</dt>
                  <dd>enumerated , one of ( "right" | "top" | "bottom" )</dd>
                  <dt>Default:</dt>
                  <dd>"top"</dd>
                </dl>
                <p>
                  Determines the location of color bar's title with respect to the color bar. Note that the title's location used to be set by the now deprecated `titleside` attribute.
                </p>
              </div>
            </details>
            <details>
              <summary>
                text
              </summary>
              <div style="margin-left: 22px">
                <dl>
                  <dt>Parent:</dt>
                  <dd>data[type=scattergl].marker.colorbar.title</dd>
                  <dt>Type:</dt>
                  <dd>string</dd>
                </dl>
                <p>
                  Sets the title of the color bar. Note that before the existence of `title.text`, the title's contents used to be defined as the `title` attribute itself. This behavior has been deprecated.
                </p>
              </div>
            </details>
          </div>
          <details>
            <summary>
              x
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>number between or equal to -2 and 3</dd>
                <dt>Default:</dt>
                <dd>1.02</dd>
              </dl>
              <p>
                Sets the x position of the color bar (in plot fraction).
              </p>
            </div>
          </details>
          <details>
            <summary>
              xanchor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "left" | "center" | "right" )</dd>
                <dt>Default:</dt>
                <dd>"left"</dd>
              </dl>
              <p>
                Sets this color bar's horizontal position anchor. This anchor binds the `x` position to the "left", "center" or "right" of the color bar.
              </p>
            </div>
          </details>
          <details>
            <summary>
              xpad
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>10</dd>
              </dl>
              <p>
                Sets the amount of padding (in px) along the x direction.
              </p>
            </div>
          </details>
          <details>
            <summary>
              y
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>number between or equal to -2 and 3</dd>
                <dt>Default:</dt>
                <dd>0.5</dd>
              </dl>
              <p>
                Sets the y position of the color bar (in plot fraction).
              </p>
            </div>
          </details>
          <details>
            <summary>
              yanchor
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>enumerated , one of ( "top" | "middle" | "bottom" )</dd>
                <dt>Default:</dt>
                <dd>"middle"</dd>
              </dl>
              <p>
                Sets this color bar's vertical position anchor This anchor binds the `y` position to the "top", "middle" or "bottom" of the color bar.
              </p>
            </div>
          </details>
          <details>
            <summary>
              ypad
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.colorbar</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
                <dt>Default:</dt>
                <dd>10</dd>
              </dl>
              <p>
                Sets the amount of padding (in px) along the y direction.
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            colorscale
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].marker</dd>
              <dt>Type:</dt>
              <dd>colorscale</dd>
            </dl>
            <p>
              Sets the colorscale. Has an effect only if in `marker.color`is set to a numerical array. The colorscale must be an array containing arrays mapping a normalized value to an rgb, rgba, hex, hsl, hsv, or named color string. At minimum, a mapping for the lowest (0) and highest (1) values are required. For example, `[[0, 'rgb(0,0,255)'], [1, 'rgb(255,0,0)']]`. To control the bounds of the colorscale in color space, use`marker.cmin` and `marker.cmax`. Alternatively, `colorscale` may be a palette name string of the following list: Greys,YlGnBu,Greens,YlOrRd,Bluered,RdBu,Reds,Blues,Picnic,Rainbow,Portland,Jet,Hot,Blackbody,Earth,Electric,Viridis,Cividis.
            </p>
          </div>
        </details>
        <details>
          <summary>
            line
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].marker</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
              <dt>Default:</dt>
              <dd>true</dd>
            </dl>
            <p>
              Sets themarker.linecolor. It accepts either a specific color or an array of numbers that are mapped to the colorscale relative to the max and min values of the array or relative to `marker.line.cmin` and `marker.line.cmax` if set.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              autocolorscale
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.line</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether the colorscale is a default palette (`autocolorscale: true`) or the palette determined by `marker.line.colorscale`. Has an effect only if in `marker.line.color`is set to a numerical array. In case `colorscale` is unspecified or `autocolorscale` is true, the default palette will be chosen according to whether numbers in the `color` array are all positive, all negative or mixed.
              </p>
            </div>
          </details>
          <details>
            <summary>
              cauto
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.line</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
                <dt>Default:</dt>
                <dd>true</dd>
              </dl>
              <p>
                Determines whether or not the color domain is computed with respect to the input data (here in `marker.line.color`) or the bounds set in `marker.line.cmin` and `marker.line.cmax` Has an effect only if in `marker.line.color`is set to a numerical array. Defaults to `false` when `marker.line.cmin` and `marker.line.cmax` are set by the user.
              </p>
            </div>
          </details>
          <details>
            <summary>
              cmax
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.line</dd>
                <dt>Type:</dt>
                <dd>number</dd>
              </dl>
              <p>
                Sets the upper bound of the color domain. Has an effect only if in `marker.line.color`is set to a numerical array. Value should have the same units as in `marker.line.color` and if set, `marker.line.cmin` must be set as well.
              </p>
            </div>
          </details>
          <details>
            <summary>
              cmid
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.line</dd>
                <dt>Type:</dt>
                <dd>number</dd>
              </dl>
              <p>
                Sets the mid-point of the color domain by scaling `marker.line.cmin` and/or `marker.line.cmax` to be equidistant to this point. Has an effect only if in `marker.line.color`is set to a numerical array. Value should have the same units as in `marker.line.color`. Has no effect when `marker.line.cauto` is `false`.
              </p>
            </div>
          </details>
          <details>
            <summary>
              cmin
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.line</dd>
                <dt>Type:</dt>
                <dd>number</dd>
              </dl>
              <p>
                Sets the lower bound of the color domain. Has an effect only if in `marker.line.color`is set to a numerical array. Value should have the same units as in `marker.line.color` and if set, `marker.line.cmax` must be set as well.
              </p>
            </div>
          </details>
          <details>
            <summary>
              color
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.line</dd>
                <dt>Type:</dt>
                <dd>color or array of colors</dd>
              </dl>
              <p>
                Sets themarker.linecolor. It accepts either a specific color or an array of numbers that are mapped to the colorscale relative to the max and min values of the array or relative to `marker.line.cmin` and `marker.line.cmax` if set.
              </p>
            </div>
          </details>
          <details>
            <summary>
              coloraxis
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.line</dd>
                <dt>Type:</dt>
                <dd>subplotid</dd>
              </dl>
              <p>
                Sets a reference to a shared color axis. References to these shared color axes are "coloraxis", "coloraxis2", "coloraxis3", etc. Settings for these shared color axes are set in the layout, under `layout.coloraxis`, `layout.coloraxis2`, etc. Note that multiple color scales can be linked to the same color axis.
              </p>
            </div>
          </details>
          <details>
            <summary>
              colorscale
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.line</dd>
                <dt>Type:</dt>
                <dd>colorscale</dd>
              </dl>
              <p>
                Sets the colorscale. Has an effect only if in `marker.line.color`is set to a numerical array. The colorscale must be an array containing arrays mapping a normalized value to an rgb, rgba, hex, hsl, hsv, or named color string. At minimum, a mapping for the lowest (0) and highest (1) values are required. For example, `[[0, 'rgb(0,0,255)'], [1, 'rgb(255,0,0)']]`. To control the bounds of the colorscale in color space, use`marker.line.cmin` and `marker.line.cmax`. Alternatively, `colorscale` may be a palette name string of the following list: Greys,YlGnBu,Greens,YlOrRd,Bluered,RdBu,Reds,Blues,Picnic,Rainbow,Portland,Jet,Hot,Blackbody,Earth,Electric,Viridis,Cividis.
              </p>
            </div>
          </details>
          <details>
            <summary>
              reversescale
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.line</dd>
                <dt>Type:</dt>
                <dd>boolean</dd>
              </dl>
              <p>
                Reverses the color mapping if true. Has an effect only if in `marker.line.color`is set to a numerical array. If true, `marker.line.cmin` will correspond to the last color in the array and `marker.line.cmax` will correspond to the first color.
              </p>
            </div>
          </details>
          <details>
            <summary>
              width
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].marker.line</dd>
                <dt>Type:</dt>
                <dd>number or array of numbers greater than or equal to 0</dd>
              </dl>
              <p>
                Sets the width (in px) of the lines bounding the marker points.
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            opacity
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].marker</dd>
              <dt>Type:</dt>
              <dd>number or array of numbers between or equal to 0 and 1</dd>
            </dl>
            <p>
              Sets the marker opacity.
            </p>
          </div>
        </details>
        <details>
          <summary>
            reversescale
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].marker</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Reverses the color mapping if true. Has an effect only if in `marker.color`is set to a numerical array. If true, `marker.cmin` will correspond to the last color in the array and `marker.cmax` will correspond to the first color.
            </p>
          </div>
        </details>
        <details>
          <summary>
            showscale
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].marker</dd>
              <dt>Type:</dt>
              <dd>boolean</dd>
            </dl>
            <p>
              Determines whether or not a colorbar is displayed for this trace. Has an effect only if in `marker.color`is set to a numerical array.
            </p>
          </div>
        </details>
        <details>
          <summary>
            size
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].marker</dd>
              <dt>Type:</dt>
              <dd>number or array of numbers greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>6</dd>
            </dl>
            <p>
              Sets the marker size (in px).
            </p>
          </div>
        </details>
        <details>
          <summary>
            sizemin
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].marker</dd>
              <dt>Type:</dt>
              <dd>number greater than or equal to 0</dd>
              <dt>Default:</dt>
              <dd>0</dd>
            </dl>
            <p>
              Has an effect only if `marker.size` is set to a numerical array. Sets the minimum size (in px) of the rendered marker points.
            </p>
          </div>
        </details>
        <details>
          <summary>
            sizemode
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].marker</dd>
              <dt>Type:</dt>
              <dd>enumerated , one of ( "diameter" | "area" )</dd>
              <dt>Default:</dt>
              <dd>"diameter"</dd>
            </dl>
            <p>
              Has an effect only if `marker.size` is set to a numerical array. Sets the rule for which the data in `size` is converted to pixels.
            </p>
          </div>
        </details>
        <details>
          <summary>
            sizeref
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].marker</dd>
              <dt>Type:</dt>
              <dd>number</dd>
              <dt>Default:</dt>
              <dd>1</dd>
            </dl>
            <p>
              Has an effect only if `marker.size` is set to a numerical array. Sets the scale factor used to determine the rendered size of marker points. Use with `sizemin` and `sizemode`.
            </p>
          </div>
        </details>
        <details>
          <summary>
            symbol
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].marker</dd>
              <dt>Type:</dt>
              <dd>enumerated or array of enumerateds , one of ( "0" | "circle" | "100" | "circle-open" | "200" | "circle-dot" | "300" | "circle-open-dot" | "1" | "square" | "101" | "square-open" | "201" | "square-dot" | "301" | "square-open-dot" | "2" | "diamond" | "102" | "diamond-open" | "202" | "diamond-dot" | "302" | "diamond-open-dot" | "3" | "cross" | "103" | "cross-open" | "203" | "cross-dot" | "303" | "cross-open-dot" | "4" | "x" | "104" | "x-open" | "204" | "x-dot" | "304" | "x-open-dot" | "5" | "triangle-up" | "105" | "triangle-up-open" | "205" | "triangle-up-dot" | "305" | "triangle-up-open-dot" | "6" | "triangle-down" | "106" | "triangle-down-open" | "206" | "triangle-down-dot" | "306" | "triangle-down-open-dot" | "7" | "triangle-left" | "107" | "triangle-left-open" | "207" | "triangle-left-dot" | "307" | "triangle-left-open-dot" | "8" | "triangle-right" | "108" | "triangle-right-open" | "208" | "triangle-right-dot" | "308" | "triangle-right-open-dot" | "9" | "triangle-ne" | "109" | "triangle-ne-open" | "209" | "triangle-ne-dot" | "309" | "triangle-ne-open-dot" | "10" | "triangle-se" | "110" | "triangle-se-open" | "210" | "triangle-se-dot" | "310" | "triangle-se-open-dot" | "11" | "triangle-sw" | "111" | "triangle-sw-open" | "211" | "triangle-sw-dot" | "311" | "triangle-sw-open-dot" | "12" | "triangle-nw" | "112" | "triangle-nw-open" | "212" | "triangle-nw-dot" | "312" | "triangle-nw-open-dot" | "13" | "pentagon" | "113" | "pentagon-open" | "213" | "pentagon-dot" | "313" | "pentagon-open-dot" | "14" | "hexagon" | "114" | "hexagon-open" | "214" | "hexagon-dot" | "314" | "hexagon-open-dot" | "15" | "hexagon2" | "115" | "hexagon2-open" | "215" | "hexagon2-dot" | "315" | "hexagon2-open-dot" | "16" | "octagon" | "116" | "octagon-open" | "216" | "octagon-dot" | "316" | "octagon-open-dot" | "17" | "star" | "117" | "star-open" | "217" | "star-dot" | "317" | "star-open-dot" | "18" | "hexagram" | "118" | "hexagram-open" | "218" | "hexagram-dot" | "318" | "hexagram-open-dot" | "19" | "star-triangle-up" | "119" | "star-triangle-up-open" | "219" | "star-triangle-up-dot" | "319" | "star-triangle-up-open-dot" | "20" | "star-triangle-down" | "120" | "star-triangle-down-open" | "220" | "star-triangle-down-dot" | "320" | "star-triangle-down-open-dot" | "21" | "star-square" | "121" | "star-square-open" | "221" | "star-square-dot" | "321" | "star-square-open-dot" | "22" | "star-diamond" | "122" | "star-diamond-open" | "222" | "star-diamond-dot" | "322" | "star-diamond-open-dot" | "23" | "diamond-tall" | "123" | "diamond-tall-open" | "223" | "diamond-tall-dot" | "323" | "diamond-tall-open-dot" | "24" | "diamond-wide" | "124" | "diamond-wide-open" | "224" | "diamond-wide-dot" | "324" | "diamond-wide-open-dot" | "25" | "hourglass" | "125" | "hourglass-open" | "26" | "bowtie" | "126" | "bowtie-open" | "27" | "circle-cross" | "127" | "circle-cross-open" | "28" | "circle-x" | "128" | "circle-x-open" | "29" | "square-cross" | "129" | "square-cross-open" | "30" | "square-x" | "130" | "square-x-open" | "31" | "diamond-cross" | "131" | "diamond-cross-open" | "32" | "diamond-x" | "132" | "diamond-x-open" | "33" | "cross-thin" | "133" | "cross-thin-open" | "34" | "x-thin" | "134" | "x-thin-open" | "35" | "asterisk" | "135" | "asterisk-open" | "36" | "hash" | "136" | "hash-open" | "236" | "hash-dot" | "336" | "hash-open-dot" | "37" | "y-up" | "137" | "y-up-open" | "38" | "y-down" | "138" | "y-down-open" | "39" | "y-left" | "139" | "y-left-open" | "40" | "y-right" | "140" | "y-right-open" | "41" | "line-ew" | "141" | "line-ew-open" | "42" | "line-ns" | "142" | "line-ns-open" | "43" | "line-ne" | "143" | "line-ne-open" | "44" | "line-nw" | "144" | "line-nw-open" )</dd>
              <dt>Default:</dt>
              <dd>"circle"</dd>
            </dl>
            <p>
              Sets the marker symbol type. Adding 100 is equivalent to appending "-open" to a symbol name. Adding 200 is equivalent to appending "-dot" to a symbol name. Adding 300 is equivalent to appending "-open-dot" or "dot-open" to a symbol name.
            </p>
          </div>
        </details>
      </div>
      <details>
        <summary>
          meta
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scattergl]</dd>
            <dt>Type:</dt>
            <dd>number or categorical coordinate string</dd>
          </dl>
          <p>
            Assigns extra meta information associated with this trace that can be used in various text attributes. Attributes such as trace `name`, graph, axis and colorbar `title.text`, annotation `text` `rangeselector`, `updatemenues` and `sliders` `label` text all support `meta`. To access the trace `meta` values in an attribute in the same trace, simply use `%{meta[i]}` where `i` is the index or key of the `meta` item in question. To access trace `meta` in layout attributes, use `%{data[n[.meta[i]}` where `i` is the index or key of the `meta` and `n` is the trace index.
          </p>
        </div>
      </details>
      <details>
        <summary>
          mode
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scattergl]</dd>
            <dt>Type:</dt>
            <dd>flaglist string. Any combination of "lines", "markers", "text" joined with a "+" OR "none".</dd>
          </dl>
          <p>
            Determines the drawing mode for this scatter trace.
          </p>
        </div>
      </details>
      <details>
        <summary>
          name
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scattergl]</dd>
            <dt>Type:</dt>
            <dd>string</dd>
          </dl>
          <p>
            Sets the trace name. The trace name appear as the legend item and on hover.
          </p>
        </div>
      </details>
      <details>
        <summary>
          opacity
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scattergl]</dd>
            <dt>Type:</dt>
            <dd>number between or equal to 0 and 1</dd>
            <dt>Default:</dt>
            <dd>1</dd>
          </dl>
          <p>
            Sets the opacity of the trace.
          </p>
        </div>
      </details>
      <details>
        <summary>
          selected
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scattergl]</dd>
            <dt>Type:</dt>
            <dd>object containing one or more of the keys listed below.</dd>
          </dl>
          <p>
            Sets the marker opacity of selected points.
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            marker
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].selected</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
            </dl>
            <p>
              Sets the marker opacity of selected points.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              color
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].selected.marker</dd>
                <dt>Type:</dt>
                <dd>color</dd>
              </dl>
              <p>
                Sets the marker color of selected points.
              </p>
            </div>
          </details>
          <details>
            <summary>
              opacity
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].selected.marker</dd>
                <dt>Type:</dt>
                <dd>number between or equal to 0 and 1</dd>
              </dl>
              <p>
                Sets the marker opacity of selected points.
              </p>
            </div>
          </details>
          <details>
            <summary>
              size
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].selected.marker</dd>
                <dt>Type:</dt>
                <dd>number greater than or equal to 0</dd>
              </dl>
              <p>
                Sets the marker size of selected points.
              </p>
            </div>
          </details>
        </div>
        <details>
          <summary>
            textfont
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].selected</dd>
              <dt>Type:</dt>
              <dd>object containing one or more of the keys listed below.</dd>
            </dl>
            <p>
              Sets the text font color of selected points.
            </p>
          </div>
        </details>
        <div style="margin-left: 22px">
          <details>
            <summary>
              color
            </summary>
            <div style="margin-left: 22px">
              <dl>
                <dt>Parent:</dt>
                <dd>data[type=scattergl].selected.textfont</dd>
                <dt>Type:</dt>
                <dd>color</dd>
              </dl>
              <p>
                Sets the text font color of selected points.
              </p>
            </div>
          </details>
        </div>
      </div>
      <details>
        <summary>
          selectedpoints
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scattergl]</dd>
            <dt>Type:</dt>
            <dd>number or categorical coordinate string</dd>
          </dl>
          <p>
            Array containing integer indices of selected points. Has an effect only for traces that support selections. Note that an empty array means an empty selection where the `unselected` are turned on for all points, whereas, any other non-array values means no selection all where the `selected` and `unselected` styles have no effect.
          </p>
        </div>
      </details>
      <details>
        <summary>
          showlegend
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scattergl]</dd>
            <dt>Type:</dt>
            <dd>boolean</dd>
            <dt>Default:</dt>
            <dd>true</dd>
          </dl>
          <p>
            Determines whether or not an item corresponding to this trace is shown in the legend.
          </p>
        </div>
      </details>
      <details>
        <summary>
          text
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scattergl]</dd>
            <dt>Type:</dt>
            <dd>string or array of strings</dd>
            <dt>Default:</dt>
            <dd>""</dd>
          </dl>
          <p>
            Sets text elements associated with each (x,y) pair. If a single string, the same string appears over all the data points. If an array of string, the items are mapped in order to the this trace's (x,y) coordinates. If trace `hoverinfo` contains a "text" flag and "hovertext" is not set, these elements will be seen in the hover labels.
          </p>
        </div>
      </details>
      <details>
        <summary>
          textfont
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scattergl]</dd>
            <dt>Type:</dt>
            <dd>object containing one or more of the keys listed below.</dd>
          </dl>
          <p>
            Sets the text font.
          </p>
        </div>
      </details>
      <div style="margin-left: 22px">
        <details>
          <summary>
            color
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].textfont</dd>
              <dt>Type:</dt>
              <dd>color or array of colors</dd>
            </dl>
          </div>
        </details>
        <details>
          <summary>
            family
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].textfont</dd>
              <dt>Type:</dt>
              <dd>string or array of strings</dd>
            </dl>
            <p>
              HTML font family - the typeface that will be applied by the web browser. The web browser will only be able to apply a font if it is available on the system which it operates. Provide multiple font families, separated by commas, to indicate the preference in which to apply fonts if they aren't available on the system. The plotly service (at https://plot.ly or on-premise) generates images on a server, where only a select number of fonts are installed and supported. These include "Arial", "Balto", "Courier New", "Droid Sans",, "Droid Serif", "Droid Sans Mono", "Gravitas One", "Old Standard TT", "Open Sans", "Overpass", "PT Sans Narrow", "Raleway", "Times New Roman".
            </p>
          </div>
        </details>
        <details>
          <summary>
            size
          </summary>
          <div style="margin-left: 22px">
            <dl>
              <dt>Parent:</dt>
              <dd>data[type=scattergl].textfont</dd>
              <dt>Type:</dt>
              <dd>number or array of numbers greater than or equal to 1</dd>
            </dl>
          </div>
        </details>
      </div>
      <details>
        <summary>
          textposition
        </summary>
        <div style="margin-left: 22px">
          <dl>
            <dt>Parent:</dt>
            <dd>data[type=scattergl]</dd>
        