######################################################################
#
# IBPerl -- a Perl 5 module for SQL RDBMS programming
#   with InterBase client library.
#
# IBPerl.pm
#
# Copyright (c) 1996-1999 Bill Karwin
#
# This is unsupported, free software.  Neither Bill Karwin,
# InterBase Software Corporation, nor Inprise Corporation are
# responsible for damages resulting from the use or misuse of
# this software.  Test early, test well, test often.
#
# You may distribute under the terms of either the GNU General
# Public License or the Artistics License, as specified in the
# Perl README file.
#
######################################################################

{
    package IBPerl;
    require 5.004;
    use strict;
    use vars qw($VERSION @ISA);
    $VERSION = '0.7';
    use DynaLoader;
    @ISA = qw(DynaLoader);
    bootstrap IBPerl;
}

########################################
#
# IBPerl::Connection
#   Methods:
#   ::initialize
#   ::new
#   ::create
#   ::disconnect
#   ::DESTROY
#
########################################
{   package IBPerl::Connection;
    use IBPerl;
    @ISA = qw(IBPerl);

    # virtual method initialize
    #  Valid arguments:
    #    'Server'
    #    'Path'
    #    'User'
    #    'Password'
    #    'Role'
    #    'Protocol'
    #    'Cache'
    #    'Page_Size'
    #    'Charset'
    sub initialize {
	my $self = shift;


	#
	# Take a hash as the arguments.
	#
	%$self = @_;


	if (!defined($self->{Server}) or $self->{Server} eq '')
	{
	    # this is okay, we'll access the database locally.
	    $self->{Protocol} = 'local';
	    $self->{Connect_String} = $self->{Path};
	}

    	elsif (!defined($self->{Protocol}) or
	    $self->{Protocol} =~ /tcp[\/-]ip/i)
	{
	    $self->{Protocol} = 'TCP/IP';
	    $self->{Connect_String} = $self->{Server}.':'.$self->{Path};
    	}

	elsif ($self->{Protocol} =~ /netbeui/i)
	{
	    $self->{Protocol} = 'NetBEUI';
	    $self->{Connect_String} = '//' . $self->{Server} . '/' .
		$self->{Path};
	}

	elsif ($self->{Protocol} =~ /([si]px\/)?[si]px/i)
	{
	    $self->{Protocol} = 'IPX/SPX';
	    $self->{Connect_String} = $self->{Server} . '@' .
		$self->{Path};
	}

	else
	{
	    $self->{Protocol} = 'TCP/IP';
	    $self->{Connect_String} = $self->{Server}.':'.$self->{Path};
	}

	if (!defined($self->{User}) or $self->{User} eq '')
	{
	    $self->{User} = $ENV{ISC_USER};
	}
	if (!defined($self->{User}) or $self->{User} eq '')
	{
	    $self->{Error} = "Not enough information to make a connection.\nMissing username.";
	    return -1;
	}

	if (!defined($self->{Password}) or $self->{Password} eq '')
	{
	    $self->{Password} = $ENV{ISC_PASSWORD};
	}
	if (!defined($self->{Password}) or $self->{Password} eq '')
	{
	    $self->{Error} = "Not enough information to make a connection.\nMissing password.";
	    return -1;
	}

	if (!defined($self->{Path}) or $self->{Path} eq '')
	{
	    $self->{Error} = "Not enough information to make a connection.\nMissing path to database";
	    return -1;
	}

	elsif (!(!defined($self->{Server}) or $self->{Server} eq '')
	    and !gethostbyname($self->{Server}))
	{
	    $self->{Error} = "I don't know how to reach host \"$self->{Server}\".";
	    return -1;
	}

	if (!defined($self->{Cache}) or ($self->{Cache}+=0) <= 0)
	{
	    # Cache value of zero is special to IB_connect()
	    # and prevents specification of cache size.
	    # The effect is that the server uses the default,
	    # which is 75 for V4 and 256 for V5.
	    $self->{Cache} = 0;
	}

	if (!defined($self->{Page_Size}) or ($self->{Page_Size}+=0) <= 0)
	{
	    # 0 means default
	    $self->{Page_Size} = 0;
	}

	if (!defined($self->{Charset}) or $self->{Charset} eq '')
	{
	    $self->{Charset} = 'NONE';
	}

	if (defined($self->{Role}) and $self->{Role} !~ m/^[\w\d]+$/o)
	{
	    $self->{Error} = "Improper Role name format \"$self->{Role}\".";
	    return -1;
	}

	$self->{Error} = '';
	return 0;
    }

    # static method new
    sub new {
	my $class = shift;
	my $self = {};
	bless $self, $class;
	if ($self->initialize(@_))
	{
	    $self->{Handle} = -1;
	    return $self;
	}
	if (IB_connect($self) < 0)
	{
	    $self->{Handle} = -1;
	}
	return $self;
    }

    # static method create
    sub create {
	my $class = shift;
	my $self = {};
	bless $self, $class;
	if ($self->initialize(@_))
	{
	    $self->{Handle} = -1;
	    return $self;
	}
	if (IB_create($self) < 0)
	{
	    $self->{Handle} = -1;
	}
	return $self;
    }

    # virtual method disconnect
    sub disconnect {
	my $self = shift;
	return $self->{Handle} > 0 ? IB_disconnect($self) : -1;
    }

    # virtual method DESTROY
    sub DESTROY {
	my $self = shift;
	$self->disconnect;
    }
}

########################################
#
# IBPerl::Transaction
#   Methods:
#   ::initialize
#   ::new
#   ::commit
#   ::rollback
#   ::DESTROY
#
########################################
{   package IBPerl::Transaction;
    use IBPerl;
    @ISA = qw(IBPerl);

    # virtual method initialize
    sub initialize {
    	my $self = shift;

	%$self = @_;

	if (!defined($self->{Database}))
	{
	    $self->{Error} = "Not enough information to make a transaction.\nMissing database.";
	    return -1;
	}

	$self->{Mode} = 'READ WRITE' unless defined($self->{Mode});

	$self->{Resolution} = 'WAIT' unless defined($self->{Resolution});

	$self->{Isolation} = 'SNAPSHOT' unless defined($self->{Isolation});

	# $self->{Reserving} reserved for future implementation

	$self->{Error} = '';
	return 0;
    }

    # static method new
    sub new {
	my $class = shift;
	my $self = {};
	bless $self, $class;
	if ($self->initialize(@_))
	{
	    return $self;
	}
	if (IB_start_transaction($self) < 0)
	{
	    $self->{Handle} = -1;
	}
	return $self;
    }

    # virtual method commit
    sub commit {
	my $self = shift;
	return IB_commit_transaction($self);
    }

    # virtual method commit
    sub rollback {
	my $self = shift;
	return IB_rollback_transaction($self);
    }

    # virtual method DESTROY
    sub DESTROY {
	my $self = shift;
	$self->commit;
    }
}

########################################
#
# IBPerl::Statement
#   Methods:
#   ::initialize
#   ::new
#   ::execute
#   ::open
#   ::fetch
#   ::update
#   ::delete
#   ::close
#   ::DESTROY
#
########################################
{   package IBPerl::Statement;
    use IBPerl;
    use Time::tm;
    @ISA = qw(IBPerl);

    # virtual method initialize
    sub initialize {
    	my $self = shift;

	%$self = @_;

    	if (!defined($self->{Transaction}))
	{
	    $self->{Error} = "Not enough information.\nMissing transaction.";
	    return -1;
    	}

    	if (!defined($self->{Stmt}))
	{
	    $self->{Error} ="Not enough information.\nMissing query string.";
	    return -1;
	}

	$self->{Sep} = ';' unless defined($self->{Sep});
	$self->{DateFormat} = '%C' unless defined($self->{DateFormat});
	$self->{SQLDialect} = 1 unless defined($self->{SQLDialect});

	$self->{Prepared} = 0;
	$self->{Opened} = 0;

	if (IB_prepare($self) < 0)
	{
	    $self->{Error} = "Could not prepare statement."
		unless ($self->{Error});
	    return -1;
	}
	$self->{Prepared}++;

	# $self->{Stmt_type_no} is set by IB_prepare().
	$self->{Type} = (
	    'Unsupported', 'SELECT', 'INSERT', 'UPDATE', 'DELETE',
	    'DDL', 'GET_SEGMENT', 'PUT_SEGMENT', 'EXEC_PROCEDURE',
	    'START_TRANS', 'COMMIT', 'ROLLBACK', 'SELECT_FOR_UPD' )
	    [ $self->{Stmt_type_no} ];
	$self->{Error} = '';
	return 0;
    }

    # static method new
    sub new {
	my $class = shift;
	my $self = {};
	bless $self, $class;
	if ($self->initialize(@_) < 0)
	{
	    $self->{Handle} = -1;
	}
	return $self;
    }

    # virtual method execute
    sub execute {
	my $self = shift;
	return -1 if ($self->{Type} eq 'SELECT');
	$self->{Parms} = \@_;
	return IB_execute($self);
    }

    # virtual method open
    sub open {
	my $self = shift;

	return 0 if ($self->{Opened} > 0);
	$self->{Parms} = \@_;
	return -1 if (IB_open($self) < 0);
	$self->{Opened}++;
	$self->{Values} = [];
	$self->{Columns} = [];
	$self->{Nulls} = [];
	$self->{Lengths} = [];
	return 0;
    }

    # virtual method fetch
    sub fetch {
	my $self = shift;
	my $ref = shift;
	my $ret;

	if (!(defined($self->{Type}) and
	    ($self->{Type} eq 'SELECT')
	     or $self->{Type} eq 'EXEC_PROCEDURE'
	     or $self->{Type} eq 'SELECT_FOR_UPD'))
	{
	    $self->{Error} = "Cannot fetch from a $self->{Type} statement";
	    return -1;
	}
	if ($self->{Opened} == 0)
	{
	    if ($self->open < 0)
	    {
		$self->{Error} = 'Cannot open statement';
		return -1;
	    }
	}

	$ret = IB_fetch($self);
	return $ret if ($ret < 0 or $ret == 100);

	# The simplest case is when no parameter was passed to the
	# fetch() method, and we just return in this case.
	# Pass back the values if we're in an array context
	# (trying to maintain backward compatibility with older
	# versions of IBPerl) or else the return value of IB_fetch().

	return wantarray ?
		@{ $self->{Values} } :
		$ret
	    unless (defined($ref));

	# We may return a hash, keyed by the column names in the query...
	if (ref $ref eq "HASH")
	{
	    my %h;

	    foreach $i (0..$#{ $self->{Values} })
	    {
		$h{ ${ $self->{Columns} }[$i] } = ${ $self->{Values} }[$i];
	    }
	    %$ref = %h;
	    return 0;
	}

	# ...or a scalar formed by joining the elements of the array...
	if (ref $ref eq "SCALAR")
	{
	    $$ref = join($self->{Sep},
		map { defined($_) ? $_ : '' } @{ $self->{Values} });

	    return 0;
	}

	# ...or simply an array.
	if (ref $ref eq "ARRAY")
	{
	    @$ref = @{ $self->{Values} };
	    return 0;
	}

	return -1;
    }

    # virtual method for positioned update
    sub update {
	my $self = shift;

	# Rely on the user to pass COLUMN=>VALUE pairs
	# from which to build the hash list member Changes.

	while (my $key = shift)
	{
	    $self->{Changes}{$key} = shift;
	}

	if (!defined($self->{Changes}))
	{
	    $self->{Error} = "You specified no changes.";
	    return -1;
	}

	return IB_update_current($self);
    }

    # virtual method for positioned delete
    sub delete {
	my $self = shift;
	return IB_delete_current($self);
    }

    # virtual method execute
    sub close {
	my $self = shift;

	return -1 unless ($self->{Handle} > 0
	    and ($self->{Type} eq 'SELECT' or
	         $self->{Type} eq 'SELECT_FOR_UPD')
	    and $self->{Opened});

	my $ret = IB_close($self);
	$self->{Opened} = 0;
	return $ret;
    }

    # virtual method DESTROY
    sub DESTROY {
	my $self = shift;
	$self->close;
	IB_destroy_ST($self) if ($self->{'Handle'} > 0);
    }
}

1;
__END__

######################################################################
#
# IBPerl Revision History:
#
# Version 0.1  - 6/10/96
#	Initial implementation, with Perl 5.001m
#	Fixed problems with DSQL and VARCHAR types
#
# Version 0.2  - 7/18/96
#	Upgraded to Perl 5.003
#	Changed fetch() to take a parameter by reference
#	Changed Connection::initialize to take hashed arguments
#
# Version 0.4  - 4/98
#       Upgraded to Perl 5.004
#       Implemented multiple databases, transactions, statements
#       Redefined return values of methods; see pod
#	Redesigned all XS functions to manipulate a ref to hash directly
#       Redesigned error handling
#	Added BLOb query capability
#
# Version 0.5 - 5/98
#	Fixed bugs in error text reporting
#	Made error text retrieval thread-safe
#	Tightened error conditions
#	Support for Data Definition statements: note that they
#	  are _not_ Autocommitted like in isql.
#	Fixed bugs to support concurrent statements
#	Fixed Makefile.PL to generate proper linking rule on Linux
#
# Version 0.6 - 9/98
#       Fixed bug in Page_size
#       Added connection property for SQL Role support
#       Added connection property for international character set
#       Tested Makefile.PL rule for IB 5.1.1 on Linux
#       Tested Makefile.PL rule for IB 5.5 on Solaris
#       Added parameterized queries
#       Added Blob insert/update (using parameterized queries)
#       Added positioned update and delete
#
# Version 0.61 - 10/98
#	Fixed a bug in IB_destroy_ST that caused core drop
#
# Version 0.62 - 11/98
#       Fixed a bug in IB_free_DB that caused core drop on Linux
#
# Version 0.7 - 3/99
#	Switched primary development platform to Linux
#	Fixed require Perl 5.005 to require Perl 5.004
#	Fixed Makefile.PL for Borland C++ compilation on Win32
#	Added statement property for record counts (but not for select)
#	Added statement property for Lengths of fields
#	Fixed bug in handling undefined DateFormat property
#	Added SQLDialect property, for use with InterBase 6.0
#	Improved checking that hash elements are valid before they're used
#       Distributed precompiled IBPerl.DLL for Win32
#
######################################################################
