package Catan::Game;
$Catan::Game::VERSION = '0.03';
use strict;
use 5.20.0;
use experimental qw/postderef signatures/;
use warnings;
no warnings 'experimental';
use Catan::Event::Monopoly;
use Catan::Event::RoadBuilding;
use Catan::Event::Robber;
use Catan::Event::YearOfPlenty;
use Catan::Game::Bank;
use Catan::Game::Player;
use Catan::Game::Trade;
use Catan::Map;
use Data::Dumper;
use Data::UUID;
use JSON::XS 'encode_json';
use List::Util qw/all/;

#ABSTRACT: a class for managing games of Catan

# valid phases
our @phases = qw/Setup Deployment Play End/;

our %actions = (
  BC => \&build_city,
  BD => \&build_dc,
  BR => \&build_road,
  BS => \&build_settlement,
  CH => \&chat,
  CR => \&concede_resources,
  DR => \&dice_roll,
  MD => \&map_define,
  MO => \&monopoly,
  PA => \&player_add,
  PD => \&play_dc,
  PE => \&phase_end,
  RM => \&robber_move,
  RP => \&resource_production,
  RR => \&robber_rob,
  TA => \&trade_accept,
  TB => \&trade_bank,
  TC => \&trade_cancel,
  TE => \&turn_end,
  TO => \&trade_offer,
  TS => \&turn_start,
  YP => \&year_of_plenty,
# these are called automatically:
#  GO => \&game_over,
#  LA => \&largest_army,
#  LR => \&longest_road,
#  PS => \&phase_start,
#  RA => \&robber_activate,
#  RD => \&robber_deactivate,
#  RE => \&round_end,
#  RS => \&round_start,
);

sub new ($class, $args = {})
{
  my $self = bless {
    bank                => Catan::Game::Bank->new,
    log                 => exists $args->{log} && ref $args->{log} eq 'GLOB' ? $args->{log} : undef,
    map                 => undef,
    max_players         => 4,
    max_victory_points  => 10,
    monopoly            => undef,
    phase_index         => -1,
    players             => [],
    road_building       => undef,
    round               => 0,
    trades              => {},
    turn_index          => -1,
    year_of_plenty      => undef,
  }, $class;

  # begin setup phase (and print to log if present)
  $self->_log($self->phase_start);

  return $self;
}

sub action ($self, $action_code, $args = {})
{
  my @action_history = ();

  if (!$action_code)
  {
    push @action_history, $self->chat({player => 'A', msg => "action requires an action code\n"});
  }
  elsif (exists $actions{$action_code})
  {
    my $results = eval { $actions{$action_code}->($self, $args) };
    push @$results, $self->chat({player => 'A', msg => $@}) if $@;
    push @action_history, @$results;
  }
  else
  {
    push @action_history, $self->chat({player => 'A', msg => "action type $action_code is unknown\n"});
  }
  $self->_log(\@action_history);
  return \@action_history;
}

sub build_road ($self, $args)
{
  my $player_number = $args->{player};
  my $location = $args->{path};

  die "deploy road requires player and path arguments!\n"
    unless $player_number && $location;

  die "It is not player $player_number\'s turn!\n"
   unless $self->is_players_turn($player_number);

  die "You can only build during the Deployment and Play phases\n"
    unless $self->phase =~ /^(?:Deployment|Play)$/;

  die "Player $player_number has already deployed a road this turn!\n"
    if $self->player->has_built_road && $self->phase eq 'Deployment';

  die "Player $player_number hasn't rolled the dice yet"
    unless $self->phase eq 'Deployment' || $self->player->has_rolled_dice
      || (defined $self->{road_building} && $self->{road_building}->can_build_road);

  die "Player $player_number must move the robber first\n"
    if $self->robber->active;

  die "Invalid path\n" unless my $path = $self->map->find_path($location);

  for my $player (@{$self->players})
  {
    die "That path is occupied\n"
      if grep($path->is_colliding($_->location), @{$player->roads});
  }

  my $free = ($self->phase eq 'Deployment' ||
    (defined $self->{road_building} && $self->{road_building}->can_build_road())) ? 1 : 0;
  $self->player->road_build($path, $self->bank, $free);

  my @actions = ();
  push @actions, { BR => { player => $player_number, path => $location }},
    @{$self->longest_road};
  return \@actions;
}

sub longest_road ($self)
{
  my @actions = ();

  my @properties = map { @{$_->properties} } @{$self->players};
  my $players_by_road_length = {};
  for (@{$self->players})
  {
    my $length = scalar @{$_->max_road_calculate(\@properties)};
    push @{ $players_by_road_length->{ $length } }, $_;
  }

  my $have_tested_leading_road;
  for my $road_length (sort {$b <=> $a} keys %$players_by_road_length)
  {
    my @players = @{$players_by_road_length->{$road_length}};

    for my $player (@players)
    {
      if ($have_tested_leading_road)
      {
        $player->longest_road_toggle if $player->longest_road;
      }
      elsif (@players == 1 && $road_length >= 6 && !$player->longest_road)
      {
        $player->longest_road_toggle;
        push @actions, { LR => {player => $player->number, length => $road_length}};
      }
    }
    $have_tested_leading_road = 1;
  }
  # check for player victory from longest road
  if (my $game_over = $self->game_over)
  {
    push @actions, @$game_over;
  }
  return \@actions;
}

sub game_over ($self)
{
  if (my $player = $self->player_victory_check)
  {
    return [{ GO => { player => $player->number } }, @{$self->phase_end}];
  }
}

sub build_settlement ($self, $args)
{
  my $player_number = $args->{player};
  my $location = $args->{intersection};

  die "You can only build during the Deployment and Play phases\n"
    unless $self->phase =~ /^(?:Deployment|Play)$/;

  die "deploy settlement requires player and intersection arguments!\n"
    unless $player_number && $location;

  die "It is not $player_number\'s turn!\n"
   unless $self->is_players_turn($player_number);

  die "Player $player_number hasn't rolled the dice yet\n"
    unless $self->phase eq 'Deployment' || $self->player->has_rolled_dice;

  die "Player $player_number has already deployed a settlement this turn!\n"
    if $self->player->has_built_settlement && $self->phase eq 'Deployment';

  die "Player $player_number must move the robber first\n"
    if $self->robber->active;

  die "Invalid intersection\n" unless my $intersection = $self->map->find_intersection($location);

  for my $player (@{$self->players})
  {
    die "That intersection is occupied or too close to another property\n"
      unless 0 == grep(($intersection->is_colliding($_->location)
            || $intersection->is_adjacent($_->location)), @{$player->properties});
  }

  my $free = $self->phase eq 'Deployment' ? 1 : 0;
  my $settlement = $self->player->settlement_build($intersection, $self->bank, $free);
  my @actions = ({ BS => { player => $player_number, intersection => $location } });
  $self->player->update_ratios($self->map); # in case they built next to a harbor

  # special resource production on deploying 2nd settlement
  if ($self->phase eq 'Deployment' && $self->round == 2)
  {
    push @actions, @{$self->resource_production_deployment($settlement->location)};
  }
  push @actions, @{$self->longest_road};
  return \@actions;
}

sub build_city ($self, $args)
{
  my $player_number = $args->{player};
  my $location = $args->{intersection};

  die "You can only build during the Deployment and Play phases\n"
    unless $self->phase =~ /^(?:Deployment|Play)$/;

  die "build settlement requires player and intersection arguments!\n"
    unless $player_number && $location;

  die "It is not Player $player_number\'s turn!\n"
   unless $self->is_players_turn($player_number);

  die "Player $player_number hasn't rolled the dice yet\n"
    unless $self->player->has_rolled_dice;

  die "Player $player_number must move the robber first\n"
    if $self->robber->active;

  die "Invalid intersection\n" unless my $intersection = $self->map->find_intersection($location);

  $self->player->city_build($intersection, $self->bank);
  return [{ BC => { player => $player_number, intersection => $location } }];
}

sub build_dc ($self, $args)
{
  my $player_number = $args->{player};
  my $type = $args->{type};

  die "You can only build during the Deployment and Play phases\n"
    unless $self->phase =~ /^(?:Deployment|Play)$/;

  die "deploy settlement requires a player argument!\n"
    unless $player_number;

  die "It is not $player_number\'s turn!\n"
   unless $self->player->number == $player_number;

  die "Player $player_number hasn't rolled the dice yet\n"
    unless $self->player->has_rolled_dice;

  die "Player $player_number must move the robber first\n"
    if $self->robber->active;

  my $card = $self->player->development_card_build($self->bank->deck_draw($type), $self->bank);
  return [{ BD => { player => $player_number, type => $card->type } }];
}

sub play_dc ($self, $args)
{
  my $player_number = $args->{player};
  my $type = $args->{type};

  die "play development card requires player and development card type arguments!\n"
    unless $player_number && $type;

  die "You can only play development cards during the Play phase\n"
    unless $self->phase eq 'Play';

  die "It is not Player $player_number\'s turn!\n"
   unless $self->is_players_turn($player_number);

  die "Player $player_number has already played a development card this turn!\n"
    if $self->player->has_played_dc;

  die "Player $player_number must move the robber first\n"
    if $self->robber->active;

  my $card = $self->player->development_card_play($type);

  my @actions = ();
  push @actions, { PD => { player => $player_number, type => $card->type } };

  if ($card->type eq 'KN')
  {
    $self->player->add_knight;
    push @actions, @{$self->robber_activate({from_7 => undef})},
      @{$self->largest_army};
  }
  elsif ($card->type eq 'YP')
  {
    $self->{year_of_plenty} = Catan::Event::YearOfPlenty->new;
  }
  elsif ($card->type eq 'MO')
  {
    $self->{monopoly} = Catan::Event::Monopoly->new();
  }
  elsif ($card->type eq 'RB')
  {
    $self->{road_building} = Catan::Event::RoadBuilding->new($self->player);
  }
  else
  {
    die sprintf "Invalid development card type %s\n", $card->type;
  }

  return \@actions;
}

sub largest_army
{
  my $self = shift;

  # largest army is the player who has played at least 3 knights
  # and more knights than any other player
  my %players_by_knights;
  for (@{$self->players})
  {
    push @{$players_by_knights{ $_->knights }}, $_;
  }

  my @actions = ();
  my $have_tested_leaders;
  for my $knights_played (sort {$b <=> $a} keys %players_by_knights)
  {
    my @players = @{$players_by_knights{$knights_played}};

    for my $player (@players)
    {
      if ($have_tested_leaders)
      {
        $player->largest_army->toggle if $player->largest_army;
      }
      elsif (@players == 1 && $knights_played >= 3 && !$player->largest_army)
      {
        $player->largest_army_toggle;
        push @actions, {LA => {player => $player->number, strength => $knights_played}};
      }
    }
    $have_tested_leaders = 1;
  }

  # check for player victory
  if (my $game_over = $self->game_over)
  {
    push @actions, @$game_over;
  }
  return \@actions;
}

sub monopoly ($self, $args)
{
  my $player_number = $args->{player};
  my $code = $args->{resource_code};

  die "Monopoly requires player and resource code arguments!\n"
    unless $player_number && $code;

  die "It is not Player $player_number\'s turn!\n"
   unless $self->is_players_turn($player_number);

  die "Player $player_number does not have a monopoly!\n"
    unless defined $self->{monopoly};

  die "Player $player_number must move the robber first\n"
    if $self->robber->active;

  my $resources = $self->{monopoly}->calculate($player_number, $code, $self->players, $self->bank);
  undef $self->{monopoly};

  return [{MO => {resources => $resources} }];
}

sub year_of_plenty ($self, $args)
{
  my $player_number = $args->{player};
  my $resources = $args->{resources};

  die "Year of Plenty requires player and resource arguments!\n"
    unless $player_number && $resources && ref $resources eq 'HASH';

  die "It is not Player $player_number\'s turn!\n"
   unless $self->is_players_turn($player_number);

  die "Player $player_number does not have year of plenty!\n"
    unless defined $self->{year_of_plenty};

  my $trade = Catan::Game::Trade->new($self->bank, $self->players, $resources, 1);

  $self->{year_of_plenty}->validate($trade);
  $trade->execute;
  undef $self->{year_of_plenty};

  return [{YP => {resources => $trade->as_hashref} }];
}

sub phase_start ($self, $args = {})
{
  die "The end phase is the last phase!\n" if $self->{phase_index} == $#phases;

  my $new_phase = $phases[ ++$self->{phase_index} ];

  # reset the round and turns counters
  $self->{round} = 0;
  $self->{turn_index} = -1;

  my @actions = ({PS => {phase => $new_phase}});

  # deploy the robber, start deployment round 1
  if ($new_phase eq 'Deployment')
  {
    push @actions, @{$self->robber_setup}, @{$self->round_start};
  }
  elsif ($new_phase eq 'Play')
  {
    push @actions, @{$self->round_start};
  }
  return \@actions;
}

sub concede_resources ($self, $args)
{
  my $player_number = $args->{player};
  my $resources     = $args->{resources};
  my $player        = $self->player_by_number($player_number);

  die "Concede resources requires player and resource argument for 1 player!\n"
    unless $player && $resources && ref $resources eq 'HASH'
      && (1 == keys %$resources)
      && (exists $resources->{$player_number})
      && (all { $_ < 0 } values %{$resources->{$player_number}})
      && (1 == grep($player_number == $_->{player}->number, @{$self->robber->check_players_to_concede}));

  my @actions = ();

  my $trade = Catan::Game::Trade->new($self->bank, $self->players, $resources, 1);
  my $summary = $trade->execute;

  push @actions, {CR => { player => $player_number, resources => $summary }};

  my $msg;
  for (@{$self->robber->check_players_to_concede})
  {
    $msg .= sprintf "Player %d must concede %d resources. ",
      $_->{player}->number, ($_->{player}->resource_total - $_->{target_total});
  }
  push(@actions, $self->chat({player => 'A', msg => $msg})) if $msg;

  return \@actions;
}

sub chat ($self,$args)
{
  my $player_number = $args->{player};
  my $msg           = $args->{msg};

  die "Chat requires player and msg arguments\n"
    unless ($player_number eq 'A' || $self->is_player_number($player_number)) && $msg;

  return { CH => {player => $player_number, msg => $msg} };
}

sub robber_setup ($self, $args = {})
{
  $self->{robber} = Catan::Event::Robber->new({map => $self->map});
  return [ { RM => $self->robber->location->uuid } ];
}

sub robber_activate ($self, $args = {})
{
  my @actions = ({ RA => undef });

  my $players = $args->{from_7} ? $self->players : [];

  my $msg;
  for (@{$self->robber->activate($players)})
  {
    $msg .= sprintf "Player %d must concede %d resources. ",
      $_->{player}->number, ($_->{player}->resource_total - $_->{target_total});
  }
  push(@actions, $self->chat({player => 'A', msg => $msg})) if $msg;
  return \@actions;
}

sub robber_deactivate ($self, $args = {})
{
  $self->robber->deactivate;
  return [ { RD => undef } ];
}

# steal and deactivate
sub robber_rob ($self, $args)
{
  my $player = $self->player_by_number($args->{player});
  my $target_player = $self->player_by_number($args->{target_player});
  my $code          = $args->{code};

  die "robber steal requires a player and target player arguments\n"
    unless $player && $target_player;

  die "It is not $player\'s turn\n" unless $self->is_players_turn($player->number);

  $code ||= $self->robber->steal($target_player);
  my $trade         = Catan::Game::Trade->new($self->bank, $self->players, {
      $target_player->number=> {$code =>-1},
      $player->number       => {$code => 1},
  });
  $trade->execute;
  $self->robber->deactivate;
  return [
    { RR => $trade->as_hashref },
    { RD => undef },
  ];
}

sub robber_move ($self, $args)
{
  my $player_number = $args->{player};
  my $location = $args->{tile};

  die "build settlement requires player and tile arguments!\n"
    unless $player_number && $location;

  die "It is not player $player_number\'s turn!"
    unless $player_number == $self->player->number;

  my $tile = $self->map->find_tile($location);

  $self->robber->move($tile, $self->players);
  my @actions;
  push @actions, { RM => { tile => $tile->uuid} };

  # robber will deactivate if there are no eligible players
  # to steal from
  push @actions, { RD => undef } unless $self->robber->active;
  return \@actions;
}

sub phase_end ($self, $args = {})
{
  # check a map is defined, and there are 3+ players
  if ($self->phase eq 'Setup')
  {
    die "The Setup phase cannot end until a map has been defined (MD)\n"
      unless defined $self->map;

    die "The Setup phase cannot end until there are at least 3 players\n"
      unless @{$self->players} >= 3;
  }
  elsif ($self->phase eq 'Deployment')
  {
    my $player = $self->player;
    die "The Deployment phase cannot end until all players have gone $player\n"
      unless $self->round == 2 && $self->player->number eq "1";
  }
  elsif ($self->phase eq 'Play')
  {
    die "The Play phase cannot end until a player has won the game\n"
      unless defined $self->player_victory_check;
  }

  return [ { PE => {phase => $self->phase} }, @{$self->phase_start} ];
}

sub round_start ($self, $args = {})
{
  return [ { RS => {round => ++$self->{round}} }, @{$self->turn_start} ];
}

sub round_end ($self, $args = {})
{
  $_->actions_clear for (@{$self->players});

  # 2 round limit on deployment phase
  if ($self->phase eq 'Deployment' && $self->round == 2)
  {
    return [ { RE => {round => $self->round}}, @{$self->phase_end}  ];
  }
  return [ { RE => {round => $self->round}}, @{$self->round_start} ];
}

sub turn_start ($self, $args = {})
{
  die "cannot start turn during setup\n" if $self->phase eq 'Setup';

  if ($self->turn == 0)
  {
    $self->{turn_index}++;
  }
  # if its the last players turn in deployment
  # & they haven't deployed 2 settlements
  # it's their turn again
  elsif ($self->phase eq 'Deployment'
         && $self->turn == @{$self->players}
         && @{$self->player->settlements} == 1)
  {
    $self->{turn_index} = $self->{turn_index};
  }
  # if its deployment and the current player
  # has deployed 2 settlements
  # switch to the previous player
  elsif ($self->phase eq 'Deployment'
         && @{$self->player->settlements} == 2)
  {
    $self->{turn_index}--;
  }
  elsif ($self->phase eq 'Play'
         && $self->{turn_index} + 1 == @{$self->players})
  {
    $self->{turn_index} = 0;
  }
  else
  {
    $self->{turn_index}++;
  }

  my @actions = ({ TS => {player => $self->player->number}});
  return \@actions;
}

sub turn_end ($self, $args = {})
{
  my $player_number = $args->{player};

  die "It is not Player $player_number\'s turn!\n"
   unless $self->is_players_turn($player_number);

  die "Player $player_number hasn't rolled the dice yet\n"
    unless $self->phase eq 'Deployment' || $self->player->has_rolled_dice;

  die "Player has not built a road and a settlement\n"
    if $self->phase eq 'Deployment'
      && !($self->player->has_built_road && $self->player->has_built_settlement);

  my @actions = ();

  if ($self->robber->active)
  {
    $self->robber->deactivate;
    push @actions, { RD => undef };
  }

  # clear outstanding events, open trades
  $self->trades_clear;
  undef $self->{monopoly};
  undef $self->{road_building};
  undef $self->{year_of_plenty};

  push @actions, { TE => {player => $self->player->number }};

  # if it's the last players turn
  # or its development phase & the current player has two properties
  # and it's player #1
  if (($self->turn == @{$self->players}
      && $self->phase eq 'Play')
      || ($self->phase eq 'Deployment'
          && @{$self->player->properties} == 2
          && $self->turn == 1)
      || ($self->phase eq 'Deployment'
          && @{$self->player->properties} == 1
          && $self->turn == @{$self->players}))
  {
    push @actions, @{$self->round_end};
  }
  # else move to the next player
  else
  {
    push @actions, @{$self->turn_start};
  }
  return \@actions;
}

sub player_victory_check ($self, $args = {})
{
  my @players_by_vps = sort {
    $b->victory_points_count <=> $a->victory_points_count } @{$self->players};

  return $players_by_vps[0] if $players_by_vps[0]->victory_points_count >= 10;
}

sub player_add ($self, $args = {})
{
  die "cannot add player outside of setup phase\n" unless $self->phase eq 'Setup';
  die "cannot add player as max players has been reached\n"
    unless $self->{max_players} > @{$self->players};

  my $number = @{$self->players} + 1;
  my $player = Catan::Game::Player->new({number => "$number"});
  push @{$self->players}, $player;
  return [{PA => {player => $player->number} }];
}

sub dice_roll ($self, $args)
{
  my $player_number = $args->{player};

  die "It is not Player $player_number\'s turn\n"
    unless $self->is_players_turn($player_number);

  my $dice_roll = $self->player->roll_dice($args->{result});
  my @actions = ({ DR => {player => $self->player->number, result => $dice_roll} });

  # trigger robber action
  if ($dice_roll == 7)
  {
    push @actions, @{$self->robber_activate({from_7 => 1})};
  }
  else
  {
    push @actions, @{$self->resource_production({resource_number => $dice_roll})};
  }
  return \@actions;
}

sub resource_production ($self, $args)
{
  my %resources = ();
  my $tiles = $self->map->tiles_by_resource_number($args->{resource_number});

  for my $tile (@$tiles)
  {
    # the robber prevents resource production
    next if $self->robber->location->uuid eq $tile->uuid;

    for my $player (@{$self->players})
    {
      for my $property (@{$player->properties})
      {
        if ($property->location->is_adjacent($tile))
        {
          # update the player and bank resource amounts
          my $amount = $property->isa('Catan::Asset::Settlement') ? 1 : 2;
          next unless my $resource = $tile->yields($amount);
          $resources{$player->number}{$resource->code} += $amount;
        }
      }
    }
  }
  for my $player (@{$self->players})
  {
    if (exists $resources{$player->number})
    {
      my $trade = Catan::Game::Trade->new(
        $self->bank, $self->players, {$player->number => $resources{$player->number}}, 1);
      my $results = $trade->execute;
      for (keys %{$results->{$self->bank->number}})
      {
        $resources{$self->bank->number}{$_} += $results->{$self->bank->number}{$_};
      }
    }
  }
  return [{ RP => {resources => \%resources} }];
}

# during deployment each player collects adjacent resources for their 2nd property
sub resource_production_deployment ($self, $intersection)
{
  my %resources = ();

  for my $tile (@{$self->map->tiles_by_intersection($intersection)})
  {
    # update the player and bank resource amounts
    my $resource = $tile->yields(1);
    # sea tiles don't give out!
    next unless $resource;

    $resources{$self->player->number}{$resource->code} += 1;
  }
  my $trade = Catan::Game::Trade->new($self->bank, $self->players, \%resources, 1);
  $trade->execute;
  return [{ RP => {resources => \%resources} }];
}

sub trade_offer ($self, $args)
{
  my $offering_player = $args->{player};
  my $details         = $args->{resources};
  my $uuid            = $args->{uuid} || Data::UUID->new->create_str;

  die "Offer is not for current player!\n"
    unless grep($self->player->number == $_, keys %$details);

  die "Offer does not include offering player!\n"
    unless grep($offering_player == $_, keys %$details);

  die "Player $offering_player hasn't rolled the dice yet\n"
    unless $self->player->has_rolled_dice;

  die "UUID $uuid is not unique!\n" if exists $self->trades->{$uuid};

  my $trade = Catan::Game::Trade->new($self->bank, $self->players, $details);
  my $trade_offer = {
    trade  => $trade,
    uuid   => $uuid,
    player => $offering_player,
  };
  $self->trade_add($trade_offer);
  return [{TO => { player => $offering_player, uuid => $uuid, resources => $details}}];
}

sub trade_bank ($self, $args)
{
  my $player_number = $args->{player};
  my $details       = $args->{resources};

  die "Offer is not for current player!\n"
    unless grep($self->player->number == $_, keys %$details)
      && $player_number == $self->player->number;

  my $trade = Catan::Game::Trade->new($self->bank, $self->players, $details);
  my $resources = $trade->execute;
  return [{TB => {resources => $resources} }];
}

sub trade_add ($self, $trade)
{
  $self->{trades}{$trade->{uuid}} = $trade;
}

sub trades_clear ($self)
{
  $self->{trades} = {};
}

sub trades { $_[0]->{trades} }

sub trade_accept ($self, $args)
{
  my $player_number = $args->{player};
  my $uuid = $args->{uuid} || '';

  die "Player $player_number does not have an active trade with uuid: $uuid\n"
    unless $player_number
      && $uuid && exists $self->trades->{$uuid}
      && $self->trades->{$uuid}{player} != $player_number
      && $self->trades->{$uuid}{trade}->resources($player_number);

  my @actions = ();
  push @actions, { TA => {player => $player_number, uuid => $uuid} };
  my $trade = $self->trades->{$uuid}{trade};
  my $details = $trade->execute;
  push @actions, { TR => { resources => $details }};
  delete $self->trades->{$uuid};
  return \@actions;
}

sub trade_cancel ($self, $args)
{
  my $player_number = $args->{player};
  my $uuid = $args->{uuid} || '';

  die "Player $player_number does not have an active trade with uuid: $uuid\n"
    unless $player_number
      && $uuid && exists $self->trades->{$uuid}
      && $self->trades->{$uuid}{player} == $player_number
      && $self->trades->{$uuid}{trade}->resources($player_number);

  delete $self->trades->{$uuid};
  return [{TC => {player => $player_number, uuid => $uuid}}];
}

sub starter_map ($self)
{
  $self->action('MD', Catan::Map->_starter);
}

sub random_map ($self)
{
  $self->action('MD', Catan::Map->_random);
}

sub map_define ($self, $map)
{
  die "map can only be defined during setup phase\n" unless $self->phase eq 'Setup';
  $self->{map} = Catan::Map->new({type => 'custom', map => $map});
  return [{MD => $map}];
}

sub robber { $_[0]->{robber} }
sub players{ $_[0]->{players} }
sub player { $_[0]->{players}[$_[0]->{turn_index}] }
sub bank   { $_[0]->{bank} }
sub map    { $_[0]->{map} }
sub phase  { $phases[$_[0]->{phase_index}] }
sub round  { $_[0]->{round} }
sub turn   { $_[0]->{turn_index} + 1 }

sub player_by_number ($self, $number)
{
  die "player_by_number requires a number argument!\n" unless $number;
  my @player = grep ($number == $_->number, @{$self->players});
  die "No players with nnumber $number found!\n" unless @player;
  return $player[0];
}

sub is_players_turn ($self, $number)
{
  die "is_players_turn requires a number argument!\n" unless $number;
  return $self->player->number == $number;
}

sub _log
{
  my ($self, $msgs) = @_;
  die "log requires an arrayref of msgs\n" unless $msgs && ref $msgs eq 'ARRAY';
  return unless my $fh = $self->{log};
  say $fh encode_json($_) for @$msgs;
}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

Catan::Game - a class for managing games of Catan

=head1 VERSION

version 0.03

=head1 SYNOPSIS

  use Catan::Game;

  open my $log_fh, '>>', \my $log;
  my $game = Catan::Game->new({log => $log_fh});

  # add 4 players
  $game->action('PA', {});
  $game->action('PA', {});
  $game->action('PA', {});
  $game->action('PA', {});

  # use the starter map
  $game->starter_map;

  # end setup, start deployment phase
  $game->action('PE', {});

  # log of Catan-Game-Notation
  print $log;

=head1 DESCRIPTION

C<Catan::Game> is a class for managing games of Catan. It implements v0.01 of L<Catan-Game-Notation|https://github.com/dnmfarrell/Catan-Game-Notation>, albeit the data structures should be decoded into Perl from JSON before calling C<action> (see L<#SYNOPSIS>). You should read and understand the Catan-Game-Notation docs before using this class. Have a look at the test script F<t/game.t> which runs through a complete game of Catan using this class.

=head1 AUTHOR

David Farrell <dfarrell@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2015 by David Farrell.

This is free software, licensed under:

  The (two-clause) FreeBSD License

=cut
