package Archive::Har::Entry::Header;

use warnings;
use strict;
use Carp();
use Archive::Har::Entry::Header();

our $VERSION = 0.09;

sub new {
	my ($class, $params) = @_;
	my $self = {};
	bless $self, $class;
	if (defined $params) {
		$self->name($params->{name});
		$self->value($params->{value});
		if (defined $params->{comment}) {
			$self->comment($params->{comment});
		}
		foreach my $key (sort { $a cmp $b } keys %{$params}) {
			if ($key =~ /^_[[:alnum:]]+$/smx) { # private fields
				$self->$key($params->{$key});
			}
		}
	}
	return $self;
}

sub name {
	my ($self, $new) = @_;
	my $old = $self->{name};
	if (@_ > 1) {
		$self->{name} = $new;
	}
	return $old;
}

sub value {
	my ($self, $new) = @_;
	my $old = $self->{value};
	if (@_ > 1) {
		$self->{value} = $new;
	}
	return $old;
}

sub comment {
	my ($self, $new) = @_;
	my $old = $self->{comment};
	if (@_ > 1) {
		$self->{comment} = $new;
	}
	return $old;
}

sub AUTOLOAD {
	my ($self, $new) = @_;

	my $name = $Archive::Har::Page::PageTimings::AUTOLOAD;
	$name =~ s/.*://smx;   # strip fully-qualified portion

	my $old;
	if ($name =~ /^_[[:alnum:]]+$/smx) { # private fields
		$old = $self->{$name};
		if (@_ > 1) {
			$self->{$name} = $new;
		}
	} else {
		Carp::croak("$name is not specified in the HAR 1.2 spec and does not start with an underscore");
	}
	return $old;
}

sub TO_JSON {
	my ($self) = @_;
	my $json = { };
	$json->{name} = $self->name();
	$json->{value} = $self->value();
	if (defined $self->comment()) {
		$json->{comment} = $self->comment();
	}
	foreach my $key (sort { $a cmp $b } keys %{$self}) {
		next if (!defined $self->{$key});
		if ($key =~ /^_[[:alnum:]]+$/smx) { # private fields
			$json->{$key} = $self->{$key};
		}
	}
	return $json;
}

1;
__END__
=head1 NAME

Archive::Har::Entry::Header - Represents a single http header object for a request or response inside the HTTP Archive

=head1 VERSION

Version 0.09

=head1 SYNOPSIS

    use Archive::Har();

    my $http_archive_string = '"log": { "version": "1.1", .... ';
    my $har = Archive::Har->new();
    $har->string($http_archive_string);
    foreach my $entry ($har->entries()) {
        my $request = $entry->request();
	foreach my $header ($request->headers()) {
		print "Name: " . $header->name() . "\n";
		print "Value: " . $header->value() . "\n";
		print "Comment: " . $header->comment() . "\n";
	}
    }

=head1 SUBROUTINES/METHODS

=head2 name

returns the name of the http header

=head2 value

returns the value of the http header

=head2 comment

returns the comment about the header

