package Archive::Har::Entry::Cache::Request;

use warnings;
use strict;
use Carp();

our $VERSION = 0.09;

sub new {
	my ($class, $params) = @_;
	my $self = {};
	bless $self, $class;
	if (defined $params) {
		if (defined $params->{expires}) {
			$self->expires($params->{expires});
		}
		$self->last_access($params->{lastAccess});
		$self->etag($params->{eTag});
		$self->hit_count($params->{hitCount});
		if (defined $params->{comment}) {
			$self->comment($params->{comment});
		}
		foreach my $key (sort { $a cmp $b } keys %{$params}) {
			if ($key =~ /^_[[:alnum:]]+$/smx) { # private fields
				$self->$key($params->{$key});
			}
		}
	}
	return $self;
}

sub expires {
	my ($self, $new) = @_;
	my $old = $self->{expires};
	if (@_ > 1) {
		$self->{expires} = $new;
	}
	return $old;
}

sub last_access {
	my ($self, $new) = @_;
	my $old = $self->{lastAccess};
	if (@_ > 1) {
		if (defined $new) {
			if ($new =~ /^\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}\.\d+(?:\+\d{2}:\d{2}|Z)$/smx) {
				$self->{lastAccess} = $new;
			} else {
				Carp::croak("last_access is not formatted correctly:$new");
			}
		} else {
			$self->{lastAccess} = undef;
		}
	}
	if (defined $old) {
		return $old;
	} else {
		return '0000-00-00T00-00-00';
	}
}

sub etag {
	my ($self, $new) = @_;
	my $old = $self->{eTag};
	if (@_ > 1) {
		$self->{eTag} = $new;
	}
	if (defined $old) {
		return $old;
	} else {
		return '';
	}
}

sub hit_count {
	my ($self, $new) = @_;
	my $old = $self->{hitCount};
	if (@_ > 1) {
		$self->{hitCount} = $new;
	}
	if (defined $old) {
		return $old;
	} else {
		return 0;
	}
}

sub comment {
	my ($self, $new) = @_;
	my $old = $self->{comment};
	if (@_ > 1) {
		$self->{comment} = $new;
	}
	return $old;
}

sub AUTOLOAD {
	my ($self, $new) = @_;

	my $name = $Archive::Har::Entry::Cache::Request::AUTOLOAD;
	$name =~ s/.*://smx;   # strip fully-qualified portion

	my $old;
	if ($name =~ /^_[[:alnum:]]+$/smx) { # private fields
		$old = $self->{$name};
		if (@_ > 1) {
			$self->{$name} = $new;
		}
	} else {
		Carp::croak("$name is not specified in the HAR 1.2 spec and does not start with an underscore");
	}
	return $old;
}

sub TO_JSON {
	my ($self) = @_;
	my $json = { };
	if (defined $self->expires()) {
		$json->{expires} = $self->expires();
	}
	$json->{lastAccess} = $self->last_access();
	$json->{eTag} = $self->etag();
	$json->{hitCount} = $self->hit_count();
	if (defined $self->comment()) {
		$json->{comment} = $self->comment();
	}
	foreach my $key (sort { $a cmp $b } keys %{$self}) {
		next if (!defined $self->{$key});
		if ($key =~ /^_[[:alnum:]]+$/smx) { # private fields
			$json->{$key} = $self->{$key};
		}
	}
	return $json;
}

1;
__END__
=head1 NAME

Archive::Har::Entry::Cache::Request - Represents a cache request for a cache inside the HTTP Archive

=head1 VERSION

Version 0.09

=head1 SYNOPSIS

    use Archive::Har();

    my $http_archive_string = '"log": { "version": "1.1", .... ';
    my $har = Archive::Har->new();
    $har->string($http_archive_string);
    foreach my $entry ($har->entries()) {
        my $cache = $entry->cache();
	my $before_request = $cache->before_request();
	print "Expires: " . $before_request->expires() . "\n";
	print "Last Access: " . $before_request->last_access() . "\n";
	print "eTag: " . $before_request->etag() . "\n";
	print "hitCount: " . $before_request->hit_count() . "\n";
	print "Comment: " . $before_request->comment() . "\n";
	my $after_request = $cache->after_request();
	print "Expires: " . $after_request->expires() . "\n";
	print "Last Access: " . $after_request->last_access() . "\n";
	print "eTag: " . $after_request->etag() . "\n";
	print "hitCount: " . $after_request->hit_count() . "\n";
	print "Comment: " . $after_request->comment() . "\n";
    }

=head1 SUBROUTINES/METHODS

=head2 new

returns a new Cache Request object

=head2 expires

returns the expiration time of the cache entry

=head2 last_access

returns the last time the cache was accessed

=head2 etag

returns the etag of the cache

=head2 comment

returns the comment about element in the Query String
