package OpenInteract::Handler::Box;

# $Id: Box.pm,v 1.18 2003/02/17 21:41:11 lachoy Exp $

use strict;
use SPOPS::Secure qw( :level );
use Data::Dumper qw ( Dumper );

@OpenInteract::Handler::Box::ISA     = ();
$OpenInteract::Handler::Box::VERSION = sprintf("%d.%02d", q$Revision: 1.18 $ =~ /(\d+)\.(\d+)/);

$OpenInteract::Handler::Box::author            = 'chris@cwinters.com';
$OpenInteract::Handler::Box::default_method    = '';
@OpenInteract::Handler::Box::forbidden_methods = ();
%OpenInteract::Handler::Box::security          = ( process_boxes  => SEC_LEVEL_NONE );

use constant DEFAULT_METHOD       => 'handler';
use constant DEFAULT_SEPARATOR    => "<br>\n";
use constant DEFAULT_TEMPLATE     => 'base_box::main_box_shell';
use constant DEFAULT_BOX_WEIGHT   => 5;
use constant MAX_BOX_WEIGHT       => 100;
use constant BLANK_SHELL_TEMPLATE => '_blank_';

my @BOX_KEYS = qw( class method title weight base_template image image_alt );

sub process_boxes {
    my ( $class ) = @_;

    my $R = OpenInteract::Request->instance;
    my $BOX_CONFIG = $R->CONFIG->{box};
    $R->DEBUG && $R->scrib( 2, "Box configuration: ", Dumper( $BOX_CONFIG ) );
    my $params = {};

    # First, do the system boxes -- this puts box information into the
    # box holding area. (We need to be sure we always have access to
    # the system box class...)

    my $system_box_class = $BOX_CONFIG->{system_box_handler};
    eval "require $system_box_class";

    my $system_box_method = $BOX_CONFIG->{system_box_method} || DEFAULT_METHOD;
    if ( $system_box_class and $system_box_method ) {
        $system_box_class->$system_box_method();
    }

    # If a website has boxes that it's adding on every page it can do
    # so in code rather than in a template. Note that this handler can
    # call other handlers as it deems necessary, so that the framework
    # doesn't care about the application-specific usage.

    my $custom_box_class = $BOX_CONFIG->{custom_box_handler};
    if ( $custom_box_class ) {
        eval "require $custom_box_class";
        if ( $@ ) {
            $R->scrib( 0, "FAILED: cannot require custom box class ",
                          "[$custom_box_class]: $@" );
        }
        else {
            my $custom_box_method = $BOX_CONFIG->{custom_box_method}
                                    || DEFAULT_METHOD;
            $R->DEBUG && $R->scrib( "Calling custom box handler:",
                                    "$custom_box_class\->$custom_box_method" );
            eval { $custom_box_class->$custom_box_method() };
            if ( $@ ) {
                $R->scrib( 0, "FAILED: cannot call custom box handler ",
                              "$custom_box_class\->$custom_box_method: $@" );
            }
        }
    }

    # Next, weed out boxes without at least a name and assign
    # class/method or template information as needed. Also be able to
    # remove a box from the boxes previously marked as good if someone
    # has asked us NOT to display it. For instance, if an app doesn't
    # want a login box it can do:
    #  push @{ $R->{boxes} }, { name => 'login_box', remove => 'yes' };

    my %good_boxes = ();
    my @to_remove  = ();

BOX:
    foreach my $box_info ( @{ $R->{boxes} } ) {

        # Check to see if they are just a name, in which case they're
        # probably an old (deprecated) call

        unless ( ref $box_info ) {
            $box_info = { name => $box_info, template => $box_info };
            $R->scrib( 0, "DEPRECATED: Found box with old scalar name-only call [$box_info->{name}]" );
        }

        # See if this is a 'remove' request

        if ( $box_info->{remove} ) {
            push @to_remove, $box_info->{name};
            next BOX;
        }

        unless ( $box_info->{name} ) {
            $R->scrib( 0, "Box put into the holding area without a name! Not processed.\n",
                          "Info:\n", Dumper( $box_info ) );
            next BOX;
        }

        # If a template, class or coderef are not defined, first try to
        # lookup the box in the action table

        unless ( $box_info->{template} or
                 $box_info->{class} or
                 ref $box_info->{code} eq 'CODE' ) {

            # Modify this to lookup ALL information, which can include a
            # title, weight, etc.

            my $lookup_box_info = $R->lookup_action( $box_info->{name},
                                                     { skip_default => 1,
                                                       return       => 'info' } );
            if ( ref $lookup_box_info eq 'HASH' ) {
                $R->DEBUG && $R->scrib( 1, "Found information in action table for ($box_info->{name})" );
                foreach my $box_key ( @BOX_KEYS ) {
                    $R->DEBUG && $R->scrib( 1, "Box information from action table:",
                                               "($box_key) = ($lookup_box_info->{ $box_key })" );
                    $box_info->{ $box_key }  ||= $lookup_box_info->{ $box_key };
                }
            }

            # If the box isn't in the action table, then assume that
            # the box-name == box-template-name (this might be
            # deprecated since all templates should be looked up by
            # package AND name)

            else {
                $box_info->{template} = $box_info->{name};
                $R->DEBUG && $R->scrib( 1, "Found box with old hash name-only call: $box_info->{name}" );
            }
        }

        # Assign default weight if not already there and if the weight
        # is too large skip the box entirely

        $box_info->{weight} ||= DEFAULT_BOX_WEIGHT;
        next BOX if ( $box_info->{weight} > MAX_BOX_WEIGHT );

        $R->DEBUG && $R->scrib( 1, "Putting box ($box_info->{name}) onto the",
                                   "stack with weight $box_info->{weight}" );
        $good_boxes{ $box_info->{name} } = $box_info;
    }

    # Remove the boxes as requested

    foreach my $remove_name ( @to_remove ) {
        $R->DEBUG && $R->scrib( 1, "Trying to remove box: $remove_name" );
        delete $good_boxes{ $remove_name } if ( $remove_name );
    }

    # Sort the boxes by weight then name

    my @sorted_boxes = sort { $a->{weight} <=> $b->{weight} ||
                              $a->{name} cmp $b->{name} }
                       values %good_boxes;

    # Grab the template that we'll plug the box content into

    my $box_template_name = $R->{theme}->property_value( 'box_template' )
                            || $BOX_CONFIG->{default_template}
                            || DEFAULT_TEMPLATE;
    my ( $box_template_package, $box_template_fullname );
    if ( $box_template_name =~ /^(.*)?::(.*)$/ ) {
        $box_template_fullname = $box_template_name;
        $box_template_package = $1;
        $box_template_name    = $2;
    }
    $box_template_package  ||= 'base_box';
    $box_template_fullname ||= "$box_template_package\:\:$box_template_name";

    $R->DEBUG && $R->scrib( 2, "Using box shell template $box_template_fullname" );

    # Generate content for each box

    my @content = ();
    $R->DEBUG && $R->scrib( 2, "Sorted boxes currently in the list:\n", Dumper( \@sorted_boxes ) );
    foreach my $box_info ( @sorted_boxes ) {
        my $shell_params = {};
        $box_info->{params} ||= {};

        # Treat the box as a component and get the html back

        my ( $html, $info ) = $R->component->handler( $box_info, 'return info' );
        my ( $box_content );

        # If the user has requested to keep this box 'naked', don't
        # wrap it in the shell

        if ( $box_info->{base_template} eq BLANK_SHELL_TEMPLATE() ) {
            $box_content = $html;
            $R->DEBUG && $R->scrib( 1, "No wrapper template used by request, ",
                                       "box is naked! (cover your eyes)" );
        }
        else {
            $shell_params->{content} = $html;
            $shell_params->{label} = $box_info->{title} ||
                                     $box_info->{params}{title} ||
                                     $info->{title};
            $shell_params->{label_image_src} = $box_info->{title_image_src} ||
                                               $box_info->{params}{title_img_src} ||
                                               $info->{title_img_src};
            $shell_params->{label_image_alt} = $box_info->{title_image_alt} ||
                                               $box_info->{params}{title_img_alt} ||
                                               $info->{title_img_alt};
            $R->DEBUG && $R->scrib( 3, "Passing parameters to fill in the box of: ",
                                       Dumper( $shell_params ) );
            $box_content = $R->template->handler( {}, $shell_params,
                                                  { name => $box_template_fullname } );
        }
        push @content, $box_content;
    }

    my $sep_string = $R->{theme}->property_value( 'box_separator' )
                     || $BOX_CONFIG->{default_separator}
                     || DEFAULT_SEPARATOR;
    return join( $sep_string, @content );
}

1;

__END__

=pod

=head1 NAME

OpenInteract::Handler::Box -- Handle input and output for independent "boxes"

=head1 SYNOPSIS

 # Deposit all boxes in the current location on the page:

 [% OI.comp( 'boxes' ) %]

 # Define global box information in your server.ini

 [box]
 handler            = MyWebsite::Handler::Box
 default_template   = base_box::main_box_shell
 default_separator  = <br>
 default_method     = run_box
 system_box_handler = MyWebsite::Handler::SystemBoxes
 system_box_method  =
 custom_box_handler =
 custom_box_method  =

 # Define an OI action (in conf/action.perl) to be used for a box with
 # a class and method:

 $action => {
   current_weather_box => {
      class    => 'OpenInteract::Handler::Weather',
      method   => 'box',
      weight   => 5,
      title    => 'Current Weather',
   },
 };

 # Add a box ('name' maps to the above OI action):

 my $zip = $R->{auth}{user}{zip_code};
 my $box = { name   => 'current_weather_box',
             weight => 2,
             title  => "Weather in Zip Code $zip",
             params => { zip_code => $zip };
 push @{ $R->{boxes} }, $box;

 # Add the same box from a template:

 [% user_zip = OI.login.zip_code;
    OI.box_add( 'current_weather_box',
                weight   = 2,
                title    = "Weather in Zip Code $user_zip",
                zip_code = $user_zip ) -%]

 # Define an OI action (in conf/action.perl) to be used for a
 # template-only box:

 $action => {
  'frequent_links_box' => {
    name       => 'frequent_links_box',
    template   => 'mypkg::box_frequent_links',
    weight     => 8,
    title      => 'Frequent Links',
    security   => 'no',
  },

 # Add a template-only box, overriding weight and title:

 my $box = { name   => 'frequent_links_box',
             weight => 2,
             title  => "Most visited sites" };
 push @{ $R->{boxes} }, $box;

 # Add the same box from a template, overriding title:

 [% OI.box_add( 'frequent_links_box',
                title  = 'Most visited sites' ) %]

 # Remove a box added in another part of the system

 push @{ $R->{boxes} }, { name => 'motd', remove => 'yes' };

 # Remove the same box from a template

 [% OI.box_add( 'motd', remove = 'yes' ) %]

=head1 DESCRIPTION

Boxes are standalone parcels of content that conform to a particular
format. Think of each box as an OpenInteract action: that action may
be a piece of code (method in a class) or it may simply be a template.

In either case, the action generates content and the box handler sorts
the boxes and places the content for each in a 'shell' so all the
boxes look the same. The standard box looks something like this:

 ------------------------- <-- 'shell'
 |      BOX TITLE        |
 -------------------------
 | Box content as        |
 | generated by an       |
 | action or a           |
 | template goes here    |
 -------------------------

But you can create your own shell by defining the key 'box_template'
in your theme to be a particular template (in the
'package::template_name' format).

=head1 CONFIGURATION

This module allows you to define default information in two separate
locations for a number of parameters.

=head2 Server Configuration

In the server configuration found in every OpenInteract website, you
can define certain information for your boxes under the 'box' key:

=over 4

=item *

B<handler> ($) (mandatory)

Define the class that will be used to process the boxes. Unless you
write your own class, this will B<always> be
C<OpenInteract::Handler:Box> and should not be changed.

=item *

B<separator> ($) (optional)

This is the string used to separate boxes. For instance, if you want
to put a short horizontal rule between each line, you could set this
to:

  separator = <hr width="50%" noshade/>

Or if you had a custom image you wanted to separate your boxes with:

  separator = <div align="center"><img src="/images/box_sep.gif" height="2" width="25"/></div>

This module defines the default separator as '<br>'. It will be used
only if there is no separator defined in the theme or in the server
configuration.

=item *

B<default_method> ($) (optional)

Define the method that will be used by boxes that do not specify
one. This module defines the default method as 'handler' and unless
you know what you are doing you should not change it.

=item *

B<default_template> ($) (optional)

This is the template into which the box content gets put. Normally
this is defined in the theme, but if for some reason someone blanked
the template out this will fill in.

The default template is C<base_box::main_box_shell>, which as the name
would indicate is installed with this package. 

=item *

B<system_box_handler> ($) (optional)

Defines what we should run on every request to display system
boxes. See
L<OpenInteract::Handler::SystemBoxes|OpenInteract::Handler::SystemBoxes>
for what this includes.

It is okay if you blank this out, you just will not get the 'login',
'templates_used' and other boxes on every page.

=item *

B<system_box_method> ($) (optional)

Method to call on the C<system_box_handler> defined above.

=item *

B<custom_box_handler> ($) (optional)

If you want to call a custom handler to run every time B<in addition
to> the system handler named above, list the class here.

=item *

B<custom_box_method> ($) (optional)

Method to call on the C<custom_box_handler> named above.

=back

=head2 Theme Properties

Two properties of the boxes can be defined on a per-theme basis.

=over 4

=item *

B<box_template> ($) (optional)

This is the template into which the box content gets put. OpenInteract
ships with one theme which has this property set to 'main_box_shell',
which is used if you do not specify anything. However, you can define
additional themes and change the look of a box entirely by modifying
this template.

=item *

B<box_separator> ($) (optional, but recommended)

See the discussion of B<separator> above in the L<Server
Configuration> section.

=back

=head2 Box Properties

An individual box also has a say as to how it will be rendered as well
as the content it will have.

The simplest case is a call:

 push @{ $R->{boxes} }, 'my_box_template';

Which simply uses the scalar passed in as the template name and the
box name, and uses all the defaults. However, you will likely get a
box with a title 'Generic Box', which is probably not what you want.

Another example:

 push @{ $R->{boxes} }, { name     => 'mybox',
                          template => 'mypkg::mybox',
                          weight   => 1,
                          title    => 'My First Box' };

Each box can define the following parameters:

=over 4

=item *

B<name> ($) (mandatory)

Just used to identify the box.

=item *

B<remove> ($) (optional)

If you use this parameter you are telling the box handler to remove a
box with name 'name'. This box does not have to be added by you or in
your package -- for instance, you might want to always get rid of the
'user_info' and 'login' boxes that come with OI:

 push @{ $R->{boxes} }, { name => 'user_info_box', remove => 'yes' },
                        { name => 'login_box', remove => 'yes' };

This does require you to know the name, but that should not be too
onerous a burden.

=item *

B<title> ($) (optional)

Display name of box used in the 'shell' wrapper, if you elect to use
that.

=item *

B<title_image_src> ($) (optional)

Display an image for the title to be used in the 'shell' wrapper.

=item *

B<title_image_alt> ($) (optional)

Text to put in the 'alt' tag if using an image in the title.

=item *

B<weight> ($)

Number between 1 (top) and 10 (bottom) indicating where you want the
box to be. If you do not specify the weight the constant from this
class DEFAULT_BOX_WEIGHT will be used. (Normally this is 5.)

=item *

B<box_template> ($) (optional)

If you specify the keyword '_blank_' then your box content will be
'naked' and not wrapped by anything else. If you leave this empty you
will use either the box_template property in your theme, the
'box_template' defined in your server configuration, or the
DEFAULT_TEMPLATE defined in this class.

=item *

B<params> (\%) (optional)

Whatever you pass here will passed through to the template or method
that is implementing the box.

=back

=head1 BUGS

None known.

=head1 TO DO

B<Cache base templates (wrappers)>

The base template wrapper should be cached in the handler so we do not
need to fetch it every time.

B<Flexible base_template handling>

Right now we allow you to use either the default base_template wrapper
(defined in either the theme or the server config) or none at all. We
need to allow each box to define its own wrapper.

=head1 SEE ALSO

L<OpenInteract::SiteTemplate|OpenInteract::SiteTemplate>,
L<OpenInteract::Theme|OpenInteract::Theme>

=head1 COPYRIGHT

Copyright (c) 2001-2002 Chris Winters. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
