package OpenInteract::Handler::PageDirectoryIndex;

# $Id: PageDirectoryIndex.pm,v 1.7 2002/01/17 23:38:38 lachoy Exp $

use strict;
use File::Basename ();
use SPOPS::Secure qw( :level );

@OpenInteract::Handler::PageDirectoryIndex::ISA     = qw();
$OpenInteract::Handler::PageDirectoryIndex::VERSION = sprintf("%d.%02d", q$Revision: 1.7 $ =~ /(\d+)\.(\d+)/);

use constant DEFAULT_IMAGE_SOURCE => '/images/icons/unknown.gif';
use constant DEFAULT_INDEX_FILE   => 'index.html';

my %MIME = ();

sub simple_index {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;

    unless ( keys %MIME ) { %MIME = $class->refresh_content_types() }

    my $apply_dir = $p->{directory} || $p->{page_directory}{directory};

    my $page_class = $R->page;

    my $security_level = SEC_LEVEL_WRITE;

    # If the pages are protected by security, bail if we don't have
    # access to this directory

    if ( $page_class->isa( 'SPOPS::Secure::Hierarchy' ) ) {
        $security_level = $page_class->check_security({ object_id => $apply_dir });
        $R->throw({ code => 303 }) if ( $security_level < SEC_LEVEL_READ );
    }

    # CHANGE? ('page' objects have directory w/o trailing slash...)

    $apply_dir =~ s|/$||;

    my $page_iter = eval { $page_class->fetch_iterator({ where => 'directory = ?',
                                                         value => [ $apply_dir ],
                                                         order => 'location' }) };
    if ( $@ ) {
        $R->{page}{title} = "Error creating index";
        return $R->template->handler({}, { page_directory => $p->{page_directory},
                                           directory      => $apply_dir,
                                           error_msg      => $@ },
                                     { name => 'base_page::directory_index_error' } );
    }

    # Unless we're at the root, create a parent

    my ( $parent );
    unless ( $apply_dir eq '/' ) {
        $parent = File::Basename::dirname( $apply_dir );
        $parent = "$parent/" if ( $parent and $parent ne '/' );
    }

    # Also find the actual subdirectories

    my ( $error_msg );
    my @dirs = ();

    my $html_dir = $R->CONFIG->get_dir( 'html' );
    $html_dir =~ s|/$||;
    my $fs_dir   = join( '/', $html_dir, $apply_dir );
    $R->DEBUG && $R->scrib( 1, "Trying to read dirs from ($fs_dir)" );
    eval { opendir( D, $fs_dir ) || die $! };
    if ( $@ ) {
        $error_msg = "Cannot read subdirectories from filesystem directory: $!";
    }
    else {
        @dirs = grep ! /^\./, grep { -d "$fs_dir/$_" } readdir( D );
        closedir( D );
    }

    my %params = ( iterator       => $page_iter,
                   this_dir       => $apply_dir,
                   dir_list       => \@dirs,
                   has_parent     => $parent,
                   default_iamge  => DEFAULT_IMAGE_SOURCE,
                   mime           => \%MIME,
                   error_msg      => $error_msg,
                   security_level => $security_level );
    $R->{page}{title} = "Directory index: $apply_dir";
    return $R->template->handler({}, \%params,
                                 { name => 'base_page::directory_index' } );
}

sub file_index {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;

    my $index_files = $R->CONFIG->{action}{fileindex}{index_files};
    unless( ref $index_files eq 'ARRAY' and scalar @{ $index_files } ) {
        $index_files = [ DEFAULT_INDEX_FILE ];
    }
    $R->DEBUG && $R->scrib( 1, "Using the following for index names: ",
                               join( ', ', @{ $index_files } ) );

    $p->{directory} =~ s|/$||;
    my @locations = map { join( '/', $p->{directory}, $_ ) } @{ $index_files };

    foreach my $location ( @locations ) {
        $R->DEBUG && $R->scrib( 1, "Trying to fetch location ($location)\n" );
        my $page = eval { $R->page->fetch( $location ) };
        if ( $@ ) {
            $R->scrib( 0, "Encountered error trying to retrieve ($location)\n",
                          "$SPOPS::Error::system_msg\n",
                          "Continuing with other locations..." );
        }
        if ( $page ) {
            $R->DEBUG && $R->scrib( 1, "Found matching location: ($page->{location})" );
            my ( $action_class, $action_method ) = $R->lookup_action( 'page' );
            return $action_class->show_displayable_page( $page, $p );
        }
    }

    # Location not found, return the appropriate message

    $R->throw({ code => 314, system_msg => $p->{directory} });
}


sub refresh_content_types {
    my ( $class ) = @_;
    my $R = OpenInteract::Request->instance;
    my $ct_iter = $R->content_type->fetch_iterator;
    my %types = ();
    while ( my $ct = $ct_iter->get_next ) {
        $types{ $ct->{mime_type} } = $ct;
    }
    return %types;
}

1;
