#!/usr/bin/perl

# $Id: static_page2base_page.pl,v 1.3 2001/11/27 13:21:03 lachoy Exp $

# Transition all your static_page objects to base_page. This assumes
# that you are using all-database storage and want to transition to
# all-filesystem storage.

use strict;
use OpenInteract::Startup;

{
    my ( $OPT_add_html, $OPT_as_file, $OPT_as_database );
    my %opts = ( 'add_html'    => \$OPT_add_html,
                 'as_file'     => \$OPT_as_file,
                 'as_database' => \$OPT_as_database );
    my $R = OpenInteract::Startup->setup_static_environment_options( '', \%opts );

    $OPT_as_file++ unless ( $OPT_as_file or $OPT_as_database );

    # First retrieve all the static page objects

    my $iter = $R->static_page->fetch_iterator({ skip_security => 1 });

    my $page_class = $R->page;
    my $count = 1;
    while ( my $static_page = $iter->get_next ) {
        print "$count: $static_page->{location}\n";

        # Modify the location as necessary

        my $location => $static_page->{location};
        if ( $OPT_add_html and $location !~ /\.\w+$/ ) {
            $location .= '.html';
        }

        # Set most page properties

        my $page = $page_class->new({
                         title         => $static_page->{title},
                         author        => $static_page->{author},
                         keywords      => $static_page->{keywords},
                         boxes         => $static_page->{boxes},
                         main_template => $static_page->{main_template},
                         active_on     => $static_page->{active_on},
                         expires_on    => $static_page->{expires_on},
                         is_active     => $static_page->{is_active},
                         notes         => $static_page->{notes},
                         location      => $location });

        # Set file status

        $page->{storage} = 'database' if ( $OPT_as_database );
        $page->{storage} = 'file'     if ( $OPT_as_file );

        # Set the content

        if ( $static_page->{script} ) {
            $page->{content} = join( "\n", $static_page->{pagetext},
                                           qq(<script language="JavaScript">\n<!--\n),
                                           $static_page->{script},
                                           qq(\n// -->\n</script>) );
        }
        else {
            $page->{content} = $static_page->{pagetext};
        }

        $page->{mime_type} = OpenInteract::Page->mime_type_content( $page->{content} ) ||
                             'text/html';
        $page->{size}      = length $page->{content};

        # Save the page object (automatically saves the content) and
        # display a status

        eval { $page->save({ skip_security => 1 }) };
        if ( $@ ) {
            print "   ERROR: $@\n";
        }
        else {
            print "   OK: $page->{location}\n";
        }

        $count++;
    }
}

__END__

=pod

=head1 NAME

static_page2base_page.pl - Migrate pages from the 'static_page' to the 'base_page' package

=head1 SYNOPSIS

 $ export OIWEBSITE=/home/httpd/mysite
 $ perl static_page2base_page.pl --add_html --as_file

=head1 DESCRIPTION

As of OpenInteract 1.35, all functionality for serving 'static' pages
and files is now in the 'base_page' package. (Previously it was in the
'static_page' package.)

The new package has been entirely rewritten and uses a different table
structure, so you will need to migrate the pages from the structure of
the earlier package to this one. This script should do the job.

=head1 OPTIONS

B<--add_html>

If pass we will add C<.html> to any file that does not have an
extension.

B<--as_file>

B<--as_database>

These two options are mutually exclusive. They determine where the
page content will be stored. If not specified, we use B<--as_file>.

=head1 BUGS

None known.

=head1 TO DO

Nothing known.

=head1 SEE ALSO

=head1 COPYRIGHT

Copyright (c) 2001 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
