package OpenInteract::FullTextIterator;

# $Id: FullTextIterator.pm,v 1.4 2001/10/29 14:29:22 lachoy Exp $

use strict;
use Data::Dumper     qw( Dumper );
use SPOPS::Iterator  qw( ITER_IS_DONE );

@OpenInteract::FullTextIterator::ISA     = qw( SPOPS::Iterator );
$OpenInteract::FullTextIterator::VERSION = sprintf("%d.%02d", q$Revision: 1.4 $ =~ /(\d+)\.(\d+)/);

sub initialize {
    my ( $self, $p ) = @_;
    $self->{_FT_RESULTS}       = $p->{results};
    $self->{_FT_SAVED_RESULTS} = $p->{saved_results};
    $self->{_FT_COUNT}         = 0;
    $self->{_FT_RAW_COUNT}     = 0;
    $self->{_FT_OFFSET}        = $p->{offset};
    $self->{_FT_MAX}           = $p->{max};
}


sub fetch_object {
    my ( $self ) = @_;

    # Get the info for retrieving the object

    my $current_count = $self->{_FT_RAW_COUNT};

    my ( $object_key, $fulltext_score );

    if ( ref $self->{_FT_RESULTS} ) {
        $object_key     = $self->{_FT_RESULTS}->[ $current_count ]->[0];
        $fulltext_score = $self->{_FT_RESULTS}->[ $current_count ]->[1];
    }

    elsif ( ref $self->{_FT_SAVED_RESULTS} ) {
        $object_key     = $self->{_FT_SAVED_RESULTS}->[ $current_count ]->{id};
        $fulltext_score = $self->{_FT_SAVED_RESULTS}->[ $current_count ]->{extra}->[0];
    }

    return ITER_IS_DONE unless ( $object_key );

    my $R = OpenInteract::Request->instance;
    my ( $object_class, $object_id ) =
                    OpenInteract::SPOPS->fetch_object_info_by_key(
                                             $object_key, { db => $R->db( 'main' ) } );

    # If there's no class/oid matching the key, go on to the next

    unless ( $object_class and $object_id ) {
        $self->{_FT_RAW_COUNT}++;
        return $self->fetch_object;
    }

    my $obj = eval { $object_class->fetch( $object_id,
                                           { skip_security => $self->{_SKIP_SECURITY} } ) };
    if ( $@ ) {
        $self->{_FT_RAW_COUNT}++;
        return $self->fetch_object;
    }

    unless ( $obj ) {
        $self->{_FT_RAW_COUNT}++;
        return $self->fetch_object;
    }

    # Using min/max and haven't reached it yet

    if ( $self->{_FT_OFFSET} and 
         ( $self->{_FT_COUNT} < $self->{_FT_OFFSET} ) ) {
        $self->{_FT_COUNT}++;
        $self->{_FT_RAW_COUNT}++;
        return $self->fetch_object;
    }

    if ( $self->{_FT_MAX} and
         ( $self->{_FT_COUNT} > $self->{_FT_MAX} ) ) {
        return ITER_IS_DONE;
    }

    # Ok, we've gone through all the necessary contortions -- we can
    # actually return the object. 

    $obj->{tmp_fulltext_score} = $fulltext_score;

    $self->{_FT_RAW_COUNT}++;
    $self->{_FT_COUNT}++;

    return ( $obj, $self->{_FT_COUNT} );
}

1;

__END__

=pod

=head1 NAME

OpenInteract::FullTextIterator - Iterator to scroll through search results that are objects of different classes.

=head1 SYNOPSIS

 my $iter = OpenInteract::FullText->search_fulltext_index({ 
                                        search_terms => [ 'mario', 'lemieux' ],
                                        search_type  => 'and',
                                        return       => 'iterator' });
 while ( my $obj = $iter->get_next ) {
     print "Object is a ", ref $obj, " with ID ", $obj->id, "\n";
 }

=head1 DESCRIPTION

This class implements L<SPOPS::Iterator> so we can scroll through
search results one at a time.

=head1 METHODS

B<initialize>

B<fetch_object>

=head1 BUGS

None yet!

=head1 TO DO

Nothing known.

=head1 SEE ALSO

L<SPOPS::Iterator>, L<OpenInteract::FullText>

=head1 COPYRIGHT

Copyright (c) 2001 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
