package OpenInteract::SQLInstall::User;

# $Id: User.pm,v 1.4 2001/08/12 14:45:42 lachoy Exp $

use strict;
use vars         qw( %HANDLERS );
use Data::Dumper qw( Dumper );
use SPOPS::Utility;

@OpenInteract::SQLInstall::User::ISA = qw( OpenInteract::SQLInstall );

use constant DEBUG => 0;

my %files = (
    tables   => [ 'user.sql' ],
    security => [ 'install_security.dat' ],
);

%HANDLERS = (
    create_structure => { '_default_' => [ 'create_structure',
                                           { table_file_list => $files{tables} } ] },
    install_data     => { '_default_' => \&create_admin_user },
    install_security => { '_default_' => [ 'install_data',
                                           { data_file_list => $files{security} } ] },
);


# Create the admin user and give him (or her) a default password -- be
# sure to let the user know what this password is!

sub create_admin_user {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $status = { action => 'data',
                   name   => 'create_admin_user',
                   ok     => 1 };
    my $CONFIG = $R->CONFIG;
    my $user = $R->user->new({ email      => $CONFIG->{admin_email},
                               login_name => 'superuser',
                               first_name => 'Super',
                               last_name  => 'User',
                               user_id    => $CONFIG->{default_objects}{superuser} });
    my $password = SPOPS::Utility->generate_random_code( 8 );
    if ( $CONFIG->{login}{crypt_password} ) {
        $user->{password} = $user->crypt_it( $password );
    }
    eval { $user->save({ is_add        => 1,
                         skip_security => 1,
                         skip_cache    => 1,
                         skip_log      => 1 }) };
    if ( $@ ) {
        my $ei = SPOPS::Error->get;
        $status->{ok}  = 0;
        $status->{msg} = "Cannot create admin user!\nError: $@\nVerbose: $ei->{system_msg}";
        DEBUG && warn "Cannot create admin user! Error: ", Dumper( $ei ), "\n";
    }
    else {
        my $achtung     = '**WRITE THIS PASSWORD DOWN!**';
        $status->{msg}  = "Created user (superuser)) with initial password: (($password)). $achtung";
        $status->{note} = "Administrator username: superuser\n" .
                          "Administrator password: $password\n" . $achtung;
    }
    return [ $status ];
}

1;

__END__

=pod

=head1 NAME

OpenInteract::SQLInstall::User - SQL installer for the base_user package

=head1 SYNOPSIS

 $ oi_manage install_sql --package=base_user

=head1 DESCRIPTION

This SQLInstall subclass is a little different than many of the other
handlers in OpenInteract -- you B<must> track information that it
generates during the SQL installation process.

We do not want to ship OpenInteract with either a blank or otherwise
known superuser password. And we do not want to force users to type it
in during installation -- doing as much as possible to allow automated
installs is a good thing.

So when we install the SQL we create the superuser with a random
password and echo the password back to you.

=head1 BUGS

None known.

=head1 TO DO

Nothing known.

=head1 COPYRIGHT

Copyright (c) 2001 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
