# ABSTRACT: Regexp Object Role for Perl 5
package Data::Object::Role::Regexp;

use 5.010;
use Data::Object::Role;

use Data::Object 'throw';

map with($_), our @ROLES = qw(
    Data::Object::Role::Defined
    Data::Object::Role::Detract
    Data::Object::Role::Output
    Data::Object::Role::Ref
    Data::Object::Role::Type
);

our $VERSION = '0.39'; # VERSION

sub search {
    my($self, $string, $flags) = @_;

    my $captures;
    my @matches;

    my $op   = '$string =~ m/$$self/';
    my $capt = '$captures = (' . $op . ($flags // '') . ')';
    my $mtch = '@matches  = ([@-], [@+], {%-})';
    my $expr = join ';', $capt, $mtch;

    my $error = do { local $@; eval $expr; $@ };
    throw $error if $error;

    return [$$self, $string, $captures, @matches, $string];
}

sub replace {
    my($self, $string, $replacement, $flags) = @_;

    my $captures;
    my @matches;

    my $op   = '$string =~ s/$$self/$replacement/';
    my $capt = '$captures = (' . $op . ($flags // '') . ')';
    my $mtch = '@matches  = ([@-], [@+], {%-})';
    my $expr = join ';', $capt, $mtch;

    my $initial = $string;

    my $error = do { local $@; eval $expr; $@ };
    throw $error if $error;

    return [$$self, $string, $captures, @matches, $initial];
}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

Data::Object::Role::Regexp - Regexp Object Role for Perl 5

=head1 VERSION

version 0.39

=head1 SYNOPSIS

    use Data::Object::Role::Regexp;

=head1 DESCRIPTION

Data::Object::Role::Regexp provides functions for operating on Perl 5 regular
expressions.

=head1 ROLES

This role is composed of the following roles.

=over 4

=item *

L<Data::Object::Role::Defined>

=item *

L<Data::Object::Role::Detract>

=item *

L<Data::Object::Role::Output>

=item *

L<Data::Object::Role::Ref>

=back

=head1 SEE ALSO

=over 4

=item *

L<Data::Object::Array>

=item *

L<Data::Object::Class>

=item *

L<Data::Object::Class::Syntax>

=item *

L<Data::Object::Code>

=item *

L<Data::Object::Float>

=item *

L<Data::Object::Hash>

=item *

L<Data::Object::Integer>

=item *

L<Data::Object::Number>

=item *

L<Data::Object::Role>

=item *

L<Data::Object::Role::Syntax>

=item *

L<Data::Object::Regexp>

=item *

L<Data::Object::Scalar>

=item *

L<Data::Object::String>

=item *

L<Data::Object::Undef>

=item *

L<Data::Object::Universal>

=item *

L<Data::Object::Autobox>

=item *

L<Data::Object::Library>

=item *

L<Data::Object::Signatures>

=back

=head1 AUTHOR

Al Newkirk <anewkirk@ana.io>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2014 by Al Newkirk.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
