package Lingua::Identifier::Feature::Alphabet;
$Lingua::Identifier::Feature::Alphabet::VERSION = '0.01_4';
use 5.006;
use strict;

use utf8;


sub features {
    my $a = shift;
    my ($c1,$c2,$c3,$c4,$c5,$c6,$c7,$c8,$c9,$ca);

    ## Cirílico
    $c1=($a =~ y[\x{0410}-\x{042F}\x{0430}-\x{044F}]
                [\x{0410}-\x{042F}\x{0430}-\x{044F}]);

    ## Latino
    $c2=($a =~ s/[a-zA-Z]//g);

    ## Japonês
    $c3=($a =~ y[\x{3040}-\x{30FF}]
                [\x{3040}-\x{30FF}]);

    ## Koreano
    $c4=($a =~ y[\x{AC00}-\x{D7AF}\x{1100}-\x{11FF}\x{3130}-\x{318F}\x{A960}-\x{A97F}\x{D7B0}-\x{D7FF}]
                [\x{AC00}-\x{D7AF}\x{1100}-\x{11FF}\x{3130}-\x{318F}\x{A960}-\x{A97F}\x{D7B0}-\x{D7FF}]);

    ## CJK (Kanji)
    $c5=($a =~ y[\x{4E00}-\x{9FAF}]
                [\x{4E00}-\x{9FAF}]);

    ## Tradicional Chinese
    $c6 = count_traditional_chinese($a);

    ## Simplified Chinese
    $c7 = count_simplified_chinese($a);

    ## Arabe
    $c8=($a =~ y[\x{0600}-\x{06FF}]
                [\x{0600}-\x{06FF}]);

    ## Thai
    $c9=($a =~ y[\x{0E00}-\x{0E7F}]
                [\x{0E00}-\x{0E7F}]);

    ## Greek
    $ca=($a =~ y[\x{0370}-\x{03FF}\x{1F00}-\x{1FFF}]
                [\x{0370}-\x{03FF}\x{1F00}-\x{1FFF}]);

    my $tot = length $a;

    my $res = {
      	_cir => $c1/$tot,
        _lat => $c2/$tot,
        _jap => $c3/$tot,
        _kor => $c4/$tot,
        _cjk => $c5/$tot,
        _ztw => $c6/$tot,
        _zcn => $c7/$tot,
        _arb => $c8/$tot,
        _tha => $c9/$tot,
        _gre => $ca/$tot,
    };

    $res = clean_other($res, '_cir') if $res->{_cir} > 0.20;
    $res = clean_other($res, '_lat') if $res->{_lat} > 0.40; # force a little more.

    $res = clean_other($res, '_jap') if $res->{_jap} > 0.20;
    $res = clean_other($res, '_kor') if $res->{_kor} > 0.20;
    $res = clean_other($res, '_ztw') if $res->{_cjk} > 0.30 and $res->{_ztw} > $res->{_zcn};
    $res = clean_other($res, '_zcn') if $res->{_cjk} > 0.30 and $res->{_ztw} < $res->{_zcn};

    $res = clean_other($res, '_arb') if $res->{_arb} > 0.20;
    $res = clean_other($res, '_tha') if $res->{_tha} > 0.20;
    $res = clean_other($res, '_gre') if $res->{_gre} > 0.20;

    return $res;
}

sub clean_other { 
	my ($res, $choose) = @_;
	for my $k (keys %$res) {
		$res->{$k} = $k eq $choose ? 1 : 0;
	}
	return $res;
}

sub count_traditional_chinese { my $a=shift;
  return ( $a =~ tr{
錒藹愛礙捱皚靄曖騃嬡璦鎄噯菴鵪闇銨諳骯奧襖嶴驁鼇媼罷壩鈀䰾擺敗唄辦頒闆絆鈑綁幫謗
鎊鮑報寶飽鉋鴇備輩貝狽盃憊揹鋇鵯錛繃鏰筆畢閉痺幣斃篳贔賁嗶鰏鉍詖蹕蓽編辯變邊辮貶
鯿籩緶標錶鑣飆颮鰾鏢別彆鼈鱉癟殯繽濱賓瀕鬢檳鑌髕擯儐臏並併餅稟蔔撥剝駁鵓餑鎛鮊鮁
缽鈸蘗鉑襏佈補鈽綵採財慘慚殘燦蠶參叄驂滄倉艙蒼傖測廁側冊厠惻層剎詫餷鍤鑔奼蠆儕釵
纏鏟產蟬剷攙闡饞驏懺鑱囅諂蕆禪嬋讒滻腸長償場暢嘗嚐廠閶萇倀鯧悵鈔徹車轍硨襯塵陳縝
諶齔磣讖櫬懲稱撐誠湞蟶赬棖檉騁呎遲熾馳恥齒喫癡鴟飭蟲衝沖寵銃籌醜綢疇幬讎儔躊儲處
礎廚鋤櫥觸雛齣芻躕絀傳釧牀創闖瘡戧愴錘搥鎚脣純蒓鶉輟綽齪詞辭賜薺聰從叢蔥囪蓯驄樅
湊輳麤竄鑹躥攛吋邨錯銼達韃薘墶貸帶膽誕擔單彈躭憚撣癉鉭簞殫鄲盪當檔黨擋蕩鐺襠噹壋
讜簹氹璫碭攩搗島導禱盜燾魛擣鍀燈鄧鐙滌締遞敵牴鏑糴覿軑諦詆澱點電墊顛癲巔鈿調弔釣
鵰彫銱鯛疊諜鰈釘頂訂錠丟銩棟凍動東鼕鶇腖鬥竇荳鈄讀獨賭牘鍍黷犢瀆篤闍櫝鍛斷緞對隊
兌懟鐓鈍噸頓躉燉墮奪飿綞嚲鐸惡額鱷鵝餓訛阨鶚噁堊軛顎鍔諤閼爾貳兒餌鉺邇鴯罰發髮閥
琺氾飯販煩範礬汎釩訪紡倣魴鈁鰟廢飛費緋鐨鯡誹憤糞紛墳奮鱝僨風鳳豐諷峯鋒楓縫瘋馮碸
灃渢復輻撫負彿膚縛婦輔賦複伕麩紱鳧頫韍訃鮒駙賻嘸紼鰒釓該蓋鈣賅榦桿幹趕稈鱤尷紺贛
槓綱岡鋼剛崗戇鋯誥縞擱閣個鴿閤鉻鎘箇給亙綆賡鯁鞏貢宮龔夠鈎構購溝鉤覯緱搆詬穀顧僱
轂鴣鯝蠱鶻餶鈷詁錮掛蝸颳鴰詿腡剮枴柺館貫觀慣關䲘鸛鰥摜廣獷櫃軌規貴詭龜歸閨溈鮭匱
劌媯匭劊鱖檜滾輥袞鯀緄鍋過國蟈餜幗堝槨摑錁鉿駭頦漢韓扞銲頇頷歛頏蠔號顥鎬灝鶴賀闔
齕閡訶頜紇龢恆橫黌絎鴴紅轟鬨鴻嗊葒閎訌後餱鱟壺鬍護滬戶滸沍餬鵠軤鶘話華嘩劃譁樺嫿
鏵驊懷壞煥歡換還緩瘓環喚渙奐繯鍰黃謊鍠鰉毀繪輝誨匯會賄迴穢揮燬彙諱翬燴褘薈翽詼譭
暉煇渾葷餛閽諢琿穫貨獲禍鑊鸌鍃鈥擊劑計緝雞繼跡擠極績記羈積濟輯際機級紀幾饑嘰飢鯽
譏蟣蹟霽薊覬璣魢鶺驥鱭齏躋賫磯茍傢夾頰駕價賈莢鉀郟蛺鋏鎵餄檟浹鑑艦殲監簡揀濺檢儉
漸減賤堅薦踐鑒鍵艱劍見間撿繭澗姦縑檻緘箋諫韉鐧譾餞梘鶼戔戩鰹襇筧瞼講將醬漿薑獎韁
槳蔣鱂殭絳驕攪僥轎繳澆膠矯餃腳嬌較絞勦嶠鮫撟鉸鷦誡節階潔屆結傑詰櫛頡癤訐勁緊進儘
謹盡僅晉錦燼藎饉觔贐縉璡覲驚荊頸鏡競淨徑鯨莖靜經痙逕脛涇靚剄烴舊糾鳩廄鷲鬮颶據懼
劇舉鋸跼駒侷齟屨櫸詎鋦窶鉅絹鵑捲錈鐫訣決絕覺玨譎鐝軍駿鈞鮶濬皸雋開凱塏愾鎧鐦剴闓
鍇愷龕闞摃鈧閌銬顆課殼剋騍緙軻懇墾齦硜鏗釦摳庫褲軲絝嚳誇塊澮鄶獪鱠膾噲儈寬髖況曠
礦纊誑鄺壙誆貺潰虧窺簣聵巋餽憒蕢饋騤崑綑睏褌錕鯤梱擴闊臘蠟鑞賴來癩萊棶賚徠睞崍瀨
淶錸籟爛攬纜濫攔懶欄欖蘭籃藍覽瀾讕嵐襴斕闌鑭襤瑯閬鋃勞撈嘮嶗鐒銠澇癆樂餎鰳類淚壘
鐳縲誄稜厲禮隸勵曆鯉歷裏籬離瀝麗釐琍貍裡靂浬慄礫壢癧櫟蘺嚦犛鋰蠣躒糲礪蒞鱧驪鸝縭
櫪邐儷鱺癘酈轢藶倆憐簾連煉練戀蓮鏈臉聯鐮斂鍊襝瀲鎌槤薟漣蘞璉殮鰱褳奩諒涼糧樑輛兩
両魎輬療遼繚鷯鷚釕鐐獵賃臨鄰躪痳鱗凜燐轔廩懍藺檁領鈴靈嶺齡鯪鴒綾櫺瀏餾劉鏐鉚飀鶹
騮綹鎦嚨朧籠龍聾攏矓壟隴壠曨礱瀧蘢櫳瓏摟樓婁簍耬蔞嶁螻髏瘺嘍鏤賂陸虜顱擄蘆錄魯爐
廬盧鹵輅壚祿瀘鱸轤臚轆籙擼鑥櫨淥艫鷺鸕櫓嚕巒攣亂灤孌欒鑾臠鸞孿淪輪倫論侖掄崙圇綸
羅蘿邏鑼騾駱籮絡囉濼氌犖欏玀鏍臝綠鋁驢慮侶濾褸屢呂縷漊僂閭櫚鋝螞罵馬嗎碼媽瑪痲獁
禡嘜嬤邁賣買麥脈勱蕒瞞滿蠻饅蟎顢鰻縵鏝謾貿貓錨氂沒黴鎇鎂鶥悶門們捫燜鍆懣夢矇濛錳
鸏謎瀰祕覓彌糸瞇獼冪羋謐綿麪緬麵澠廟緲鶓滅衊憫閩閔黽鰵緡鳴銘謬糢麽鏌驀謨饃歿謀繆
畝鉬納鈉鎿妳迺難饢腦鬧惱撓橈鐃吶訥餒內膩擬暱鷊鈮鯢唸撚攆輦鮎釀孃鳥嬝蔦裊聶櫱鎳囁
鑷隉嚙糱顳躡濘寧擰檸獰嚀甯聹鈕紐農濃膿噥穠儂駑煖諾儺釹瘧謔嘔鷗毆歐慪甌謳漚盤蹣槃
龐徬拋砲皰齙賠珮轡噴踫鵬疋闢鈹鮍鈈紕毘羆騙駢諞飄縹驃頻貧蘋顰嬪憑評鮃潑頗釙鉕醱錇
僕撲鋪譜樸舖鐠鏷臍啟齊豈悽氣棄棲騎鰭慼淒蠐頎磧錡騏蘄榿鯕綺旂訖遷簽錢牽淺鉛慳潛謙
鉗籤箝譴韆鈐僉鋟槧騫鵮塹釺繾牆搶槍強嗆蹌檣嬙鎗薔熗錆鏹羥鏘橋僑翹鍬喬竅蹺譙磽蕎誚
礄竊鍥愜篋欽親寢駸懃撳嶔傾輕頃請慶氫檾鯖窮瓊煢鞦巰坵鰍虯賕嶇軀驅趨區麴齲覷鴝闃詘
勸權綣銓鰁輇甽顴詮確鵲愨闕闋羣讓擾繞饒蟯遶蕘嬈熱認飪韌紉紝軔絨榮宂嶸蠑鎔銣顬縟軟
鋭銳叡潤閏篛薩灑颯賽鰓傘糝毿饊繖喪顙騷掃繅鰺澀嗇穡銫紗殺鯊廈鎩篩曬刪閃陝贍騸繕鱔
訕摻釤羶姍搧傷賞觴殤紹燒攝設捨賒懾厙灄紳審滲嬸腎瀋瘮諗詵聲聖勝繩昇賸陞適識詩實視
釋師飾蝕獅勢濕屍駛時試溼蝨諡塒鯷貰鯴軾蒔鉈弒鰣鳲鈰壽獸綬書樞輸屬數樹術贖豎朮攄紓
帥閂雙誰稅順説爍碩說鑠飼絲鍶鷥颸螄噝駟緦廝聳頌鬆訟誦慫崧蒐餿擻藪颼甦訴蘇肅囌謖穌
痠雖隨歲綏誶損孫筍猻蓀鎖瑣縮嗩簑撻闥闒獺躂鰨態颱臺駘鮐鈦檯貪歎攤灘癱談壇譚罈嘆曇
鐔錟嘽湯燙醣鏜钂儻鐋討濤綯絛韜鞀鋱騰籐謄體屜題銻緹鵜綈闐鍩靦條鰷齠糶貼鐵鋨聽廳鋌
頲統銅慟仝頭禿圖塗釷凃團摶糰頹蛻飩魨牠馱駝託橢脫鴕鼉籜襪窪媧膃咼彎頑萬灣輓壪鯇紈
綰網輞圍衞偉餵偽謂違為維葦衛緯蝟煒濰韋瑋幃潿䲁韙闈鮪諉鰃穩聞紋問溫鰛轀搵閿甕鶲窩
臥渦萵齷誣烏無誤蕪務嗚霧鵡吳塢汙摀鋙鄔鶩鎢廡嫵騖憮鵐戲細係犧襲習繫蓆錫谿釃屓餼璽
綌覡鬩嚇蝦俠峽狹轄陜芐硤餡銜鮮縣獻現啣鹹賢羨憲險纖閒線顯絃閑嫻蜆莧峴鷳癇獮躚銑嫺
獫秈鍁蘚嚮鑲廂詳餉響鄉項薌驤鯗緗饗銷曉嘯蕭瀟囂効簫傚梟驍嘵綃鴞協瀉諧寫挾謝脅攜蠍
鋣纈紲擷洩褻鋅釁興倖鈃滎餳陘騂兇洶鏽繡脩銹鵂饈虛須鬚緒續許敍敘卹噓勗詡緖頊選懸絢
癬軒䴉鉉諼學鱈勛遜訓馴訊尋詢燻塤潯蕁勳鱘訝鴨啞亞鴉壓軋埡氬婭巖嚴厭豔簷驗煙顏鹽菸
醃閻硯諺燄嚥厴兗贋釅鼴灩懨嵒贗讞魘艷饜閹儼彥養癢鴦揚樣陽楊颺瑒暘瘍煬遙搖藥鑰謠窯
餚瑤鰩鷂葯堯軺嶢銚業葉爺頁謁琊饁鄴燁靨曄藝譯蟻億義醫詣憶枱遺議誼異儀繹懌鎰貽艤餏
釔禕睪頤嶧迆軼驛縊瘞詒銥飴囈勩鐿銀蔭陰隱癮飲慇憖縯銦應瑩鷹螢鶯營嬰穎贏蠅櫻纓鸚熒
攖潁癭罌瓔瀠縈嚶鎣瀅塋喲湧傭詠踴擁鱅顒癰鯒鏞郵遊誘猶憂魷優祐銪鈾蕕鮋語鬱輿諭籲譽
獄娛癒漁預禦慾與嶼魚瘀瘉璵飫澦鵒鈺傴諛馭閾覦嫗歟紆俁蕷齬鷸踰畬願鴛淵園圓緣遠員轅
櫞鳶黿嶽閲躍粵約閱悅噦鉞醖勻暈韻紜雲運藴醞隕蘊韞溳氳贇惲慍鄖澐殞鄆篔薀雜紮偺災載
贊暫讚瓚攢趲鏨贓臟髒臢駔鑿棗譟繰蹧則澤責擇賾簀嘖幘賊譖贈繒閘詐柵搾鮓吒鍘債齋盞佔
顫嶄斬輾戰綻氈霑棧覘颭譫脹漲帳賬張趙詔釗櫂這鍺輒謫蟄鷓診陣針偵鎮賑貞楨軫鴆禎箏爭
掙鄭睜證猙証癥鋥幀崢錚諍鉦幟紙隻執摯織緻滯擲誌姪製職質躓梔鷙騭躑輊贄銍縶軹觶鑕種
鐘終鍾眾腫塚皺軸週晝鵃紂輈冑縐豬諸貯燭駐築囑矚鑄註硃櫧瀦紵櫫苧佇誅銖撾賺專轉磚顓
囀饌樁莊壯狀裝粧妝綴錐墜贅騅縋準諄濁鐲諑梲漬諮資齜玆緇輜茲鯔貲鎡錙眥鶿總綜蹤縱瘲
傯驟騶鄒諏謅鯫組詛鏃鑽躦纘檇鱒}{
錒藹愛礙捱皚靄曖騃嬡璦鎄噯菴鵪闇銨諳骯奧襖嶴驁鼇媼罷壩鈀䰾擺敗唄辦頒闆絆鈑綁幫謗
鎊鮑報寶飽鉋鴇備輩貝狽盃憊揹鋇鵯錛繃鏰筆畢閉痺幣斃篳贔賁嗶鰏鉍詖蹕蓽編辯變邊辮貶
鯿籩緶標錶鑣飆颮鰾鏢別彆鼈鱉癟殯繽濱賓瀕鬢檳鑌髕擯儐臏並併餅稟蔔撥剝駁鵓餑鎛鮊鮁
缽鈸蘗鉑襏佈補鈽綵採財慘慚殘燦蠶參叄驂滄倉艙蒼傖測廁側冊厠惻層剎詫餷鍤鑔奼蠆儕釵
纏鏟產蟬剷攙闡饞驏懺鑱囅諂蕆禪嬋讒滻腸長償場暢嘗嚐廠閶萇倀鯧悵鈔徹車轍硨襯塵陳縝
諶齔磣讖櫬懲稱撐誠湞蟶赬棖檉騁呎遲熾馳恥齒喫癡鴟飭蟲衝沖寵銃籌醜綢疇幬讎儔躊儲處
礎廚鋤櫥觸雛齣芻躕絀傳釧牀創闖瘡戧愴錘搥鎚脣純蒓鶉輟綽齪詞辭賜薺聰從叢蔥囪蓯驄樅
湊輳麤竄鑹躥攛吋邨錯銼達韃薘墶貸帶膽誕擔單彈躭憚撣癉鉭簞殫鄲盪當檔黨擋蕩鐺襠噹壋
讜簹氹璫碭攩搗島導禱盜燾魛擣鍀燈鄧鐙滌締遞敵牴鏑糴覿軑諦詆澱點電墊顛癲巔鈿調弔釣
鵰彫銱鯛疊諜鰈釘頂訂錠丟銩棟凍動東鼕鶇腖鬥竇荳鈄讀獨賭牘鍍黷犢瀆篤闍櫝鍛斷緞對隊
兌懟鐓鈍噸頓躉燉墮奪飿綞嚲鐸惡額鱷鵝餓訛阨鶚噁堊軛顎鍔諤閼爾貳兒餌鉺邇鴯罰發髮閥
琺氾飯販煩範礬汎釩訪紡倣魴鈁鰟廢飛費緋鐨鯡誹憤糞紛墳奮鱝僨風鳳豐諷峯鋒楓縫瘋馮碸
灃渢復輻撫負彿膚縛婦輔賦複伕麩紱鳧頫韍訃鮒駙賻嘸紼鰒釓該蓋鈣賅榦桿幹趕稈鱤尷紺贛
槓綱岡鋼剛崗戇鋯誥縞擱閣個鴿閤鉻鎘箇給亙綆賡鯁鞏貢宮龔夠鈎構購溝鉤覯緱搆詬穀顧僱
轂鴣鯝蠱鶻餶鈷詁錮掛蝸颳鴰詿腡剮枴柺館貫觀慣關䲘鸛鰥摜廣獷櫃軌規貴詭龜歸閨溈鮭匱
劌媯匭劊鱖檜滾輥袞鯀緄鍋過國蟈餜幗堝槨摑錁鉿駭頦漢韓扞銲頇頷歛頏蠔號顥鎬灝鶴賀闔
齕閡訶頜紇龢恆橫黌絎鴴紅轟鬨鴻嗊葒閎訌後餱鱟壺鬍護滬戶滸沍餬鵠軤鶘話華嘩劃譁樺嫿
鏵驊懷壞煥歡換還緩瘓環喚渙奐繯鍰黃謊鍠鰉毀繪輝誨匯會賄迴穢揮燬彙諱翬燴褘薈翽詼譭
暉煇渾葷餛閽諢琿穫貨獲禍鑊鸌鍃鈥擊劑計緝雞繼跡擠極績記羈積濟輯際機級紀幾饑嘰飢鯽
譏蟣蹟霽薊覬璣魢鶺驥鱭齏躋賫磯茍傢夾頰駕價賈莢鉀郟蛺鋏鎵餄檟浹鑑艦殲監簡揀濺檢儉
漸減賤堅薦踐鑒鍵艱劍見間撿繭澗姦縑檻緘箋諫韉鐧譾餞梘鶼戔戩鰹襇筧瞼講將醬漿薑獎韁
槳蔣鱂殭絳驕攪僥轎繳澆膠矯餃腳嬌較絞勦嶠鮫撟鉸鷦誡節階潔屆結傑詰櫛頡癤訐勁緊進儘
謹盡僅晉錦燼藎饉觔贐縉璡覲驚荊頸鏡競淨徑鯨莖靜經痙逕脛涇靚剄烴舊糾鳩廄鷲鬮颶據懼
劇舉鋸跼駒侷齟屨櫸詎鋦窶鉅絹鵑捲錈鐫訣決絕覺玨譎鐝軍駿鈞鮶濬皸雋開凱塏愾鎧鐦剴闓
鍇愷龕闞摃鈧閌銬顆課殼剋騍緙軻懇墾齦硜鏗釦摳庫褲軲絝嚳誇塊澮鄶獪鱠膾噲儈寬髖況曠
礦纊誑鄺壙誆貺潰虧窺簣聵巋餽憒蕢饋騤崑綑睏褌錕鯤梱擴闊臘蠟鑞賴來癩萊棶賚徠睞崍瀨
淶錸籟爛攬纜濫攔懶欄欖蘭籃藍覽瀾讕嵐襴斕闌鑭襤瑯閬鋃勞撈嘮嶗鐒銠澇癆樂餎鰳類淚壘
鐳縲誄稜厲禮隸勵曆鯉歷裏籬離瀝麗釐琍貍裡靂浬慄礫壢癧櫟蘺嚦犛鋰蠣躒糲礪蒞鱧驪鸝縭
櫪邐儷鱺癘酈轢藶倆憐簾連煉練戀蓮鏈臉聯鐮斂鍊襝瀲鎌槤薟漣蘞璉殮鰱褳奩諒涼糧樑輛兩
両魎輬療遼繚鷯鷚釕鐐獵賃臨鄰躪痳鱗凜燐轔廩懍藺檁領鈴靈嶺齡鯪鴒綾櫺瀏餾劉鏐鉚飀鶹
騮綹鎦嚨朧籠龍聾攏矓壟隴壠曨礱瀧蘢櫳瓏摟樓婁簍耬蔞嶁螻髏瘺嘍鏤賂陸虜顱擄蘆錄魯爐
廬盧鹵輅壚祿瀘鱸轤臚轆籙擼鑥櫨淥艫鷺鸕櫓嚕巒攣亂灤孌欒鑾臠鸞孿淪輪倫論侖掄崙圇綸
羅蘿邏鑼騾駱籮絡囉濼氌犖欏玀鏍臝綠鋁驢慮侶濾褸屢呂縷漊僂閭櫚鋝螞罵馬嗎碼媽瑪痲獁
禡嘜嬤邁賣買麥脈勱蕒瞞滿蠻饅蟎顢鰻縵鏝謾貿貓錨氂沒黴鎇鎂鶥悶門們捫燜鍆懣夢矇濛錳
鸏謎瀰祕覓彌糸瞇獼冪羋謐綿麪緬麵澠廟緲鶓滅衊憫閩閔黽鰵緡鳴銘謬糢麽鏌驀謨饃歿謀繆
畝鉬納鈉鎿妳迺難饢腦鬧惱撓橈鐃吶訥餒內膩擬暱鷊鈮鯢唸撚攆輦鮎釀孃鳥嬝蔦裊聶櫱鎳囁
鑷隉嚙糱顳躡濘寧擰檸獰嚀甯聹鈕紐農濃膿噥穠儂駑煖諾儺釹瘧謔嘔鷗毆歐慪甌謳漚盤蹣槃
龐徬拋砲皰齙賠珮轡噴踫鵬疋闢鈹鮍鈈紕毘羆騙駢諞飄縹驃頻貧蘋顰嬪憑評鮃潑頗釙鉕醱錇
僕撲鋪譜樸舖鐠鏷臍啟齊豈悽氣棄棲騎鰭慼淒蠐頎磧錡騏蘄榿鯕綺旂訖遷簽錢牽淺鉛慳潛謙
鉗籤箝譴韆鈐僉鋟槧騫鵮塹釺繾牆搶槍強嗆蹌檣嬙鎗薔熗錆鏹羥鏘橋僑翹鍬喬竅蹺譙磽蕎誚
礄竊鍥愜篋欽親寢駸懃撳嶔傾輕頃請慶氫檾鯖窮瓊煢鞦巰坵鰍虯賕嶇軀驅趨區麴齲覷鴝闃詘
勸權綣銓鰁輇甽顴詮確鵲愨闕闋羣讓擾繞饒蟯遶蕘嬈熱認飪韌紉紝軔絨榮宂嶸蠑鎔銣顬縟軟
鋭銳叡潤閏篛薩灑颯賽鰓傘糝毿饊繖喪顙騷掃繅鰺澀嗇穡銫紗殺鯊廈鎩篩曬刪閃陝贍騸繕鱔
訕摻釤羶姍搧傷賞觴殤紹燒攝設捨賒懾厙灄紳審滲嬸腎瀋瘮諗詵聲聖勝繩昇賸陞適識詩實視
釋師飾蝕獅勢濕屍駛時試溼蝨諡塒鯷貰鯴軾蒔鉈弒鰣鳲鈰壽獸綬書樞輸屬數樹術贖豎朮攄紓
帥閂雙誰稅順説爍碩說鑠飼絲鍶鷥颸螄噝駟緦廝聳頌鬆訟誦慫崧蒐餿擻藪颼甦訴蘇肅囌謖穌
痠雖隨歲綏誶損孫筍猻蓀鎖瑣縮嗩簑撻闥闒獺躂鰨態颱臺駘鮐鈦檯貪歎攤灘癱談壇譚罈嘆曇
鐔錟嘽湯燙醣鏜钂儻鐋討濤綯絛韜鞀鋱騰籐謄體屜題銻緹鵜綈闐鍩靦條鰷齠糶貼鐵鋨聽廳鋌
頲統銅慟仝頭禿圖塗釷凃團摶糰頹蛻飩魨牠馱駝託橢脫鴕鼉籜襪窪媧膃咼彎頑萬灣輓壪鯇紈
綰網輞圍衞偉餵偽謂違為維葦衛緯蝟煒濰韋瑋幃潿䲁韙闈鮪諉鰃穩聞紋問溫鰛轀搵閿甕鶲窩
臥渦萵齷誣烏無誤蕪務嗚霧鵡吳塢汙摀鋙鄔鶩鎢廡嫵騖憮鵐戲細係犧襲習繫蓆錫谿釃屓餼璽
綌覡鬩嚇蝦俠峽狹轄陜芐硤餡銜鮮縣獻現啣鹹賢羨憲險纖閒線顯絃閑嫻蜆莧峴鷳癇獮躚銑嫺
獫秈鍁蘚嚮鑲廂詳餉響鄉項薌驤鯗緗饗銷曉嘯蕭瀟囂効簫傚梟驍嘵綃鴞協瀉諧寫挾謝脅攜蠍
鋣纈紲擷洩褻鋅釁興倖鈃滎餳陘騂兇洶鏽繡脩銹鵂饈虛須鬚緒續許敍敘卹噓勗詡緖頊選懸絢
癬軒䴉鉉諼學鱈勛遜訓馴訊尋詢燻塤潯蕁勳鱘訝鴨啞亞鴉壓軋埡氬婭巖嚴厭豔簷驗煙顏鹽菸
醃閻硯諺燄嚥厴兗贋釅鼴灩懨嵒贗讞魘艷饜閹儼彥養癢鴦揚樣陽楊颺瑒暘瘍煬遙搖藥鑰謠窯
餚瑤鰩鷂葯堯軺嶢銚業葉爺頁謁琊饁鄴燁靨曄藝譯蟻億義醫詣憶枱遺議誼異儀繹懌鎰貽艤餏
釔禕睪頤嶧迆軼驛縊瘞詒銥飴囈勩鐿銀蔭陰隱癮飲慇憖縯銦應瑩鷹螢鶯營嬰穎贏蠅櫻纓鸚熒
攖潁癭罌瓔瀠縈嚶鎣瀅塋喲湧傭詠踴擁鱅顒癰鯒鏞郵遊誘猶憂魷優祐銪鈾蕕鮋語鬱輿諭籲譽
獄娛癒漁預禦慾與嶼魚瘀瘉璵飫澦鵒鈺傴諛馭閾覦嫗歟紆俁蕷齬鷸踰畬願鴛淵園圓緣遠員轅
櫞鳶黿嶽閲躍粵約閱悅噦鉞醖勻暈韻紜雲運藴醞隕蘊韞溳氳贇惲慍鄖澐殞鄆篔薀雜紮偺災載
贊暫讚瓚攢趲鏨贓臟髒臢駔鑿棗譟繰蹧則澤責擇賾簀嘖幘賊譖贈繒閘詐柵搾鮓吒鍘債齋盞佔
顫嶄斬輾戰綻氈霑棧覘颭譫脹漲帳賬張趙詔釗櫂這鍺輒謫蟄鷓診陣針偵鎮賑貞楨軫鴆禎箏爭
掙鄭睜證猙証癥鋥幀崢錚諍鉦幟紙隻執摯織緻滯擲誌姪製職質躓梔鷙騭躑輊贄銍縶軹觶鑕種
鐘終鍾眾腫塚皺軸週晝鵃紂輈冑縐豬諸貯燭駐築囑矚鑄註硃櫧瀦紵櫫苧佇誅銖撾賺專轉磚顓
囀饌樁莊壯狀裝粧妝綴錐墜贅騅縋準諄濁鐲諑梲漬諮資齜玆緇輜茲鯔貲鎡錙眥鶿總綜蹤縱瘲
傯驟騶鄒諏謅鯫組詛鏃鑽躦纘檇鱒});
}

sub count_simplified_chinese { my $a=shift;
  return ( $a =~ tr{
锕呆爱碍蔼皑嫒锿暧嗳瑷霭谙铵鹌肮袄奥媪岙鳌骜罢坝钯鲃败摆呗办绊颁钣绑帮谤镑宝报饱
鸨鲍备贝辈惫狈钡鹎锛绷镚毕闭币毙笔诐筚荜铋贲赑哔鲾跸辩编变辫边贬鳊笾缏标镳鳔飙飑
镖瘪宾滨缤濒鬓髌膑槟摈傧殡镔饼禀卜拨剥驳鲅鲌钹镈铂袯钵鹁饽补钸财惨蚕参灿惭残骖叁
苍舱仓沧伧册测侧恻层刹镲姹诧锸馇侪虿钗产缠馋阐蝉搀忏婵谗禅蒇镵浐骣谄冁畅厂长场肠
偿怅苌伥鲳阊钞车彻辙砗衬尘陈谌缜碜榇谶龀诚惩称撑柽枨浈蛏赪骋驰齿迟耻饬炽鸱宠铳绸
筹畴踌俦雠帱储厨处础触锄雏橱绌刍蹰传钏闯创疮戗怆纯莼鹑绰辍龊辞词赐荠聪丛从骢枞苁
凑辏窜蹿撺镩错锉达垯鞑荙贷带诞胆单弹担钽箪掸惮瘅殚郸档党裆铛垱谠珰凼砀筜导岛盗祷
焘鱽锝灯邓镫递敌缔涤镝籴觌诋谛轪颠垫电点淀钿癫巅钓调铞鲷叠谍鲽订钉顶锭丢铥冻栋东
动胨鸫钭窦赌独读牍镀笃椟阇渎犊黩断缎锻队对怼镦顿吨钝炖趸夺堕缍铎饳亸额饿鹅鳄讹垩
轭阏谔鹗锷颚儿饵尔贰铒鸸迩阀罚珐烦贩饭矾钒访纺鲂鳑钫费飞废镄鲱绯诽愤坟纷奋粪偾鲼
凤锋讽缝疯风枫冯砜沣沨负肤佛辅妇抚辐麸缚赋赙驸绂讣鳆凫呒韨绋鲋钆盖该钙赅赶秆赣鳡
尴绀岗刚纲杠冈钢戆诰锆缟阁搁鸽铬镉给亘鲠绠赓巩贡宫龚够沟购缑诟觏顾鲴鹘钴毂锢鸪诂
馉蛊蜗鸹脶诖剐馆关惯贯观掼鹳鳏鳤广犷龟轨规贵柜归诡闺沩鳜鲑匮刽妫刿桧匦滚辊绲鲧衮
锅过国锞埚帼馃椁掴蝈铪骇颏汉韩顸颔敛颃号颢镐蚝灏贺鹤颌阖龁纥诃阂横黉鸻绗红轰鸿荭
唝讧闳鲎糇护壶沪冱浒鹄鹕轷华话画桦铧骅婳怀还欢唤环换缓痪焕涣锾缳奂黄谎鳇锽会绘挥
贿讳诲秽翙烩诙袆翚荟晖浑荤阍诨珲馄货祸锪鹱镬钬鸡击剂绩极机纪际积级挤济记计继鲫叽
讥辑齑蓟赍骥玑缉苟羁矶虮鹡跻霁觊鱾鲚驾夹价贾颊荚钾浃饸镓槚郏蛱铗间茧舰渐键剑减坚
监捡艰拣践见简贱歼检俭碱溅涧戋缄饯鹣硷谫裥枧笺谏锏鲣鞯笕戬槛缣睑酱浆将桨奖讲缰蒋
鳉绛轿搅较骄娇缴脚浇绞胶饺矫侥铰鹪鲛峤挢结洁阶节届诫讦疖栉颉诘劲进晋锦紧仅谨琎荩
馑烬缙觐赆茎竞镜经净颈惊静鲸荆靓痉泾胫烃刭纠旧鸠阄厩鹫举锯惧剧驹锔龃屦讵飓榉窭绢
鹃镌锩觉绝决诀谲镢珏军骏钧隽皲鲪凯开锎忾闿垲铠剀锴恺阚龛闶钪铐壳颗课轲缂骒垦恳龈
铿硁抠裤库绔轱喾块鲙郐侩哙浍脍狯宽髋况旷矿纩诓贶邝圹诳亏窥溃骙愦聩蒉篑岿阃鲲锟裈
扩阔蜡腊镴来赖癞莱梾赉籁睐濑崃涞徕铼懒拦篮兰滥蓝烂栏览榄澜揽缆岚阑斓谰褴襕镧锒阆
涝捞劳唠铹铑痨崂乐鳓饹泪类垒缧镭诔棱丽厉礼励隶离砾篱雳鲤沥鳢坜栎疬疠苈莅鲡蓠缡骊
轹粝枥跞锂俪郦鹂砺呖牦蛎逦俩链莲怜脸练连恋联敛殓裣蔹梿奁莶裢涟潋鲢琏粮辆凉谅辌魉
辽疗缭鹩钌镣鹨猎邻麻临鳞凛躏赁廪辚懔蔺檩领岭龄灵铃棂绫鲮鸰刘馏浏镠绺鹠铆镏飗垄笼
聋拢龙咙茏砻陇珑昽垅栊泷搂楼篓娄瘘喽髅耧蝼镂蒌嵝虏炉鲁录芦陆卤颅卢庐赂箓辘栌渌舻
禄镥橹撸辂掳鲈噜鹭轳胪垆鸬泸乱峦滦孪栾娈脔銮挛鸾轮论抡伦沦囵纶萝箩锣骡罗骆络逻啰
荦氇椤泺猡镙屡驴虑滤绿侣吕铝缕闾溇褛榈偻锊骂蚂马妈码吗玛嬷祃犸唛卖迈麦买脉劢荬蛮
满馒瞒镘缦谩螨鳗颟贸猫锚牦么没镁镅鹛门闷们钔焖懑扪梦锰鹲眯谜丝觅芈幂谧猕绵缅渑庙
鹋缈灭悯闽黾闵鳘缗鸣铭谬么蓦镆殁谟馍谋缪亩钼纳钠镎你难馕挠闹脑恼铙桡呐讷内馁拟腻
铌鹝鲵昵撵辇鲇酿鸟茑聂糵陧镊镍蹑颞啮蘖嗫拧狞泞柠咛聍纽钮农浓脓秾侬哝驽诺傩钕疟谑
欧呕殴鸥讴怄瓯沤蹒仿庞抛疱龅赔辔喷鹏纰罴铍鲏钚骗骈谝飘骠缥贫苹频嫔颦凭评鲆泼颇钋
钷酦锫谱朴扑镨镤气齐启岂骑弃脐鳍栖骐蛴鲯讫蕲锜桤颀碛绮浅谦牵钱铅潜迁谴钤堑钎骞椠
鹐悭佥缱锓强抢墙呛锖锵镪蔷羟跄炝樯嫱侨乔桥锹翘跷窍硗诮荞谯硚窃箧锲惬亲寝钦揿嵚骎
顷倾请轻庆氢鲭苘穷琼茕巯鳅虬赇趋区驱躯岖诎鸲阒龋觑劝权圳铨鳈绻辁诠颧鹊却阙阕悫让
饶扰蛲荛娆热认纫韧轫纴饪绒荣嵘镕蝾颥缛铷软润闰箬萨飒赛鳃毵馓糁丧颡扫骚缫鲹涩铯穑
啬纱厦杀鲨铩筛陕删闪赡掺钐鳝骟姗膻讪缮赏伤殇觞烧绍摄设赊厍滠慑沈审渗肾婶什绅瘆谂
诜声绳圣识饰释蚀实狮师势时试驶诗适视铊谥铈鳀贳弑埘鲥轼鲺莳鸤寿兽绶树数竖输书属枢
赎纾摅帅闩双谁顺烁硕铄饲丝鸶蛳飔锶驷厮缌咝颂诵耸讼怂馊薮飕擞肃诉谡稣虽岁随谇绥笋
损孙荪狲锁缩琐唢跶闼挞鳎獭阘态鲐骀钛摊谈滩贪谭瘫啴镡昙锬烫汤铴镋镗傥讨涛绹鼗绦韬
铽腾誊题屉缇锑绨鹈锘阗腼条鲦龆粜铁贴锇听厅铤颋统铜恸头秃图钍抟颓蜕鲀饨驼它驮椭鸵
箨鼍袜洼娲腽呙湾顽弯万塆绾鲩纨网辋伟伪为围维违纬谓苇诿鳚闱玮韪鳂韦潍涠鲔帏炜纹稳
闻问鳁揾辒阌瓮鹟窝涡龌莴乌雾吴误务无呜诬鹉芜坞鹜鹀怃邬钨骛妩铻庑习牺袭锡戏细觋酾
饩绤屃玺阋陕虾吓狭峡侠辖硖苄显馅线献鲜纤县险现宪贤藓痫籼锨鹇蚬苋跹狝岘铣猃详项乡
响镶厢骧芗缃饷飨鲞销晓嚣箫萧啸绡鸮潇枭哓骁泻胁携协写谢蝎谐挟绁铘缬亵撷衅锌兴钘饧
荥骍陉汹绣馐鸺虚续许嘘诩勖顼选悬轩癣谖鹮铉绚学鳕训询寻讯逊驯埙浔荨鲟鸦亚哑轧鸭压
讶娅氩垭验颜盐厌严谚砚阎魇鼹厣餍谳俨兖滟阉彦酽恹样痒阳杨养鸯旸炀疡玚窑摇钥遥谣瑶
鳐轺峣铫尧鹞业叶页爷谒晔靥玡烨馌邺仪异遗亿义蚁医议谊忆艺译绎贻饻镒颐呓饴诣缢镱睾
舣钇祎轶驿勚铱怿诒瘗峄银饮阴隐瘾荫铟慭赢营樱应鹰蝇萤缨荧鹦颖婴莹莺璎罂瘿茔滢潆嘤
蓥颍萦撄哟咏拥踊痈镛鲬颙鳙邮诱犹优忧鱿铀铕鲉莸狱鱼娱预与誉屿语渔吁舆驭欤畲俣玙纡
谀鹆觎龉妪鹬钰阈滪饫谕伛馀蓣缘员圆远园辕鸳渊鼋橼鸢跃约粤哕钺韵晕匀运陨纭郓郧赟筼
殒沄恽蕰涢韫扎咱杂灾载暂攒錾瓒趱赃臜驵枣凿缲则择泽责啧箦赜帻贼谮赠缯闸诈铡咤鲊债
斋盏颤崭战斩毡栈绽谵觇飐辗涨张胀帐账赵棹诏钊这锗鹧谪辄蛰贞侦镇诊阵针赈桢轸祯鸩挣
郑争筝睁狰钲峥诤锃铮帧织质执职侄纸帜掷挚滞鸷贽絷骘锧踬觯铚轵踯轾栀肿种众终昼皱轴
纣绉辀鸼诸嘱驻猪着铸烛贮苎潴橥伫瞩槠诛铢纻挝赚砖专转馔颛啭装状壮桩赘坠锥缀缒骓谆
浊镯诼棁资渍眦缁鹚赀镃龇鲻辎锱踪总纵综疭偬骤诹邹驺鲰诌组诅镞钻缵躜槜鳟}{
锕呆爱碍蔼皑嫒锿暧嗳瑷霭谙铵鹌肮袄奥媪岙鳌骜罢坝钯鲃败摆呗办绊颁钣绑帮谤镑宝报饱
鸨鲍备贝辈惫狈钡鹎锛绷镚毕闭币毙笔诐筚荜铋贲赑哔鲾跸辩编变辫边贬鳊笾缏标镳鳔飙飑
镖瘪宾滨缤濒鬓髌膑槟摈傧殡镔饼禀卜拨剥驳鲅鲌钹镈铂袯钵鹁饽补钸财惨蚕参灿惭残骖叁
苍舱仓沧伧册测侧恻层刹镲姹诧锸馇侪虿钗产缠馋阐蝉搀忏婵谗禅蒇镵浐骣谄冁畅厂长场肠
偿怅苌伥鲳阊钞车彻辙砗衬尘陈谌缜碜榇谶龀诚惩称撑柽枨浈蛏赪骋驰齿迟耻饬炽鸱宠铳绸
筹畴踌俦雠帱储厨处础触锄雏橱绌刍蹰传钏闯创疮戗怆纯莼鹑绰辍龊辞词赐荠聪丛从骢枞苁
凑辏窜蹿撺镩错锉达垯鞑荙贷带诞胆单弹担钽箪掸惮瘅殚郸档党裆铛垱谠珰凼砀筜导岛盗祷
焘鱽锝灯邓镫递敌缔涤镝籴觌诋谛轪颠垫电点淀钿癫巅钓调铞鲷叠谍鲽订钉顶锭丢铥冻栋东
动胨鸫钭窦赌独读牍镀笃椟阇渎犊黩断缎锻队对怼镦顿吨钝炖趸夺堕缍铎饳亸额饿鹅鳄讹垩
轭阏谔鹗锷颚儿饵尔贰铒鸸迩阀罚珐烦贩饭矾钒访纺鲂鳑钫费飞废镄鲱绯诽愤坟纷奋粪偾鲼
凤锋讽缝疯风枫冯砜沣沨负肤佛辅妇抚辐麸缚赋赙驸绂讣鳆凫呒韨绋鲋钆盖该钙赅赶秆赣鳡
尴绀岗刚纲杠冈钢戆诰锆缟阁搁鸽铬镉给亘鲠绠赓巩贡宫龚够沟购缑诟觏顾鲴鹘钴毂锢鸪诂
馉蛊蜗鸹脶诖剐馆关惯贯观掼鹳鳏鳤广犷龟轨规贵柜归诡闺沩鳜鲑匮刽妫刿桧匦滚辊绲鲧衮
锅过国锞埚帼馃椁掴蝈铪骇颏汉韩顸颔敛颃号颢镐蚝灏贺鹤颌阖龁纥诃阂横黉鸻绗红轰鸿荭
唝讧闳鲎糇护壶沪冱浒鹄鹕轷华话画桦铧骅婳怀还欢唤环换缓痪焕涣锾缳奂黄谎鳇锽会绘挥
贿讳诲秽翙烩诙袆翚荟晖浑荤阍诨珲馄货祸锪鹱镬钬鸡击剂绩极机纪际积级挤济记计继鲫叽
讥辑齑蓟赍骥玑缉苟羁矶虮鹡跻霁觊鱾鲚驾夹价贾颊荚钾浃饸镓槚郏蛱铗间茧舰渐键剑减坚
监捡艰拣践见简贱歼检俭碱溅涧戋缄饯鹣硷谫裥枧笺谏锏鲣鞯笕戬槛缣睑酱浆将桨奖讲缰蒋
鳉绛轿搅较骄娇缴脚浇绞胶饺矫侥铰鹪鲛峤挢结洁阶节届诫讦疖栉颉诘劲进晋锦紧仅谨琎荩
馑烬缙觐赆茎竞镜经净颈惊静鲸荆靓痉泾胫烃刭纠旧鸠阄厩鹫举锯惧剧驹锔龃屦讵飓榉窭绢
鹃镌锩觉绝决诀谲镢珏军骏钧隽皲鲪凯开锎忾闿垲铠剀锴恺阚龛闶钪铐壳颗课轲缂骒垦恳龈
铿硁抠裤库绔轱喾块鲙郐侩哙浍脍狯宽髋况旷矿纩诓贶邝圹诳亏窥溃骙愦聩蒉篑岿阃鲲锟裈
扩阔蜡腊镴来赖癞莱梾赉籁睐濑崃涞徕铼懒拦篮兰滥蓝烂栏览榄澜揽缆岚阑斓谰褴襕镧锒阆
涝捞劳唠铹铑痨崂乐鳓饹泪类垒缧镭诔棱丽厉礼励隶离砾篱雳鲤沥鳢坜栎疬疠苈莅鲡蓠缡骊
轹粝枥跞锂俪郦鹂砺呖牦蛎逦俩链莲怜脸练连恋联敛殓裣蔹梿奁莶裢涟潋鲢琏粮辆凉谅辌魉
辽疗缭鹩钌镣鹨猎邻麻临鳞凛躏赁廪辚懔蔺檩领岭龄灵铃棂绫鲮鸰刘馏浏镠绺鹠铆镏飗垄笼
聋拢龙咙茏砻陇珑昽垅栊泷搂楼篓娄瘘喽髅耧蝼镂蒌嵝虏炉鲁录芦陆卤颅卢庐赂箓辘栌渌舻
禄镥橹撸辂掳鲈噜鹭轳胪垆鸬泸乱峦滦孪栾娈脔銮挛鸾轮论抡伦沦囵纶萝箩锣骡罗骆络逻啰
荦氇椤泺猡镙屡驴虑滤绿侣吕铝缕闾溇褛榈偻锊骂蚂马妈码吗玛嬷祃犸唛卖迈麦买脉劢荬蛮
满馒瞒镘缦谩螨鳗颟贸猫锚牦么没镁镅鹛门闷们钔焖懑扪梦锰鹲眯谜丝觅芈幂谧猕绵缅渑庙
鹋缈灭悯闽黾闵鳘缗鸣铭谬么蓦镆殁谟馍谋缪亩钼纳钠镎你难馕挠闹脑恼铙桡呐讷内馁拟腻
铌鹝鲵昵撵辇鲇酿鸟茑聂糵陧镊镍蹑颞啮蘖嗫拧狞泞柠咛聍纽钮农浓脓秾侬哝驽诺傩钕疟谑
欧呕殴鸥讴怄瓯沤蹒仿庞抛疱龅赔辔喷鹏纰罴铍鲏钚骗骈谝飘骠缥贫苹频嫔颦凭评鲆泼颇钋
钷酦锫谱朴扑镨镤气齐启岂骑弃脐鳍栖骐蛴鲯讫蕲锜桤颀碛绮浅谦牵钱铅潜迁谴钤堑钎骞椠
鹐悭佥缱锓强抢墙呛锖锵镪蔷羟跄炝樯嫱侨乔桥锹翘跷窍硗诮荞谯硚窃箧锲惬亲寝钦揿嵚骎
顷倾请轻庆氢鲭苘穷琼茕巯鳅虬赇趋区驱躯岖诎鸲阒龋觑劝权圳铨鳈绻辁诠颧鹊却阙阕悫让
饶扰蛲荛娆热认纫韧轫纴饪绒荣嵘镕蝾颥缛铷软润闰箬萨飒赛鳃毵馓糁丧颡扫骚缫鲹涩铯穑
啬纱厦杀鲨铩筛陕删闪赡掺钐鳝骟姗膻讪缮赏伤殇觞烧绍摄设赊厍滠慑沈审渗肾婶什绅瘆谂
诜声绳圣识饰释蚀实狮师势时试驶诗适视铊谥铈鳀贳弑埘鲥轼鲺莳鸤寿兽绶树数竖输书属枢
赎纾摅帅闩双谁顺烁硕铄饲丝鸶蛳飔锶驷厮缌咝颂诵耸讼怂馊薮飕擞肃诉谡稣虽岁随谇绥笋
损孙荪狲锁缩琐唢跶闼挞鳎獭阘态鲐骀钛摊谈滩贪谭瘫啴镡昙锬烫汤铴镋镗傥讨涛绹鼗绦韬
铽腾誊题屉缇锑绨鹈锘阗腼条鲦龆粜铁贴锇听厅铤颋统铜恸头秃图钍抟颓蜕鲀饨驼它驮椭鸵
箨鼍袜洼娲腽呙湾顽弯万塆绾鲩纨网辋伟伪为围维违纬谓苇诿鳚闱玮韪鳂韦潍涠鲔帏炜纹稳
闻问鳁揾辒阌瓮鹟窝涡龌莴乌雾吴误务无呜诬鹉芜坞鹜鹀怃邬钨骛妩铻庑习牺袭锡戏细觋酾
饩绤屃玺阋陕虾吓狭峡侠辖硖苄显馅线献鲜纤县险现宪贤藓痫籼锨鹇蚬苋跹狝岘铣猃详项乡
响镶厢骧芗缃饷飨鲞销晓嚣箫萧啸绡鸮潇枭哓骁泻胁携协写谢蝎谐挟绁铘缬亵撷衅锌兴钘饧
荥骍陉汹绣馐鸺虚续许嘘诩勖顼选悬轩癣谖鹮铉绚学鳕训询寻讯逊驯埙浔荨鲟鸦亚哑轧鸭压
讶娅氩垭验颜盐厌严谚砚阎魇鼹厣餍谳俨兖滟阉彦酽恹样痒阳杨养鸯旸炀疡玚窑摇钥遥谣瑶
鳐轺峣铫尧鹞业叶页爷谒晔靥玡烨馌邺仪异遗亿义蚁医议谊忆艺译绎贻饻镒颐呓饴诣缢镱睾
舣钇祎轶驿勚铱怿诒瘗峄银饮阴隐瘾荫铟慭赢营樱应鹰蝇萤缨荧鹦颖婴莹莺璎罂瘿茔滢潆嘤
蓥颍萦撄哟咏拥踊痈镛鲬颙鳙邮诱犹优忧鱿铀铕鲉莸狱鱼娱预与誉屿语渔吁舆驭欤畲俣玙纡
谀鹆觎龉妪鹬钰阈滪饫谕伛馀蓣缘员圆远园辕鸳渊鼋橼鸢跃约粤哕钺韵晕匀运陨纭郓郧赟筼
殒沄恽蕰涢韫扎咱杂灾载暂攒錾瓒趱赃臜驵枣凿缲则择泽责啧箦赜帻贼谮赠缯闸诈铡咤鲊债
斋盏颤崭战斩毡栈绽谵觇飐辗涨张胀帐账赵棹诏钊这锗鹧谪辄蛰贞侦镇诊阵针赈桢轸祯鸩挣
郑争筝睁狰钲峥诤锃铮帧织质执职侄纸帜掷挚滞鸷贽絷骘锧踬觯铚轵踯轾栀肿种众终昼皱轴
纣绉辀鸼诸嘱驻猪着铸烛贮苎潴橥伫瞩槠诛铢纻挝赚砖专转馔颛啭装状壮桩赘坠锥缀缒骓谆
浊镯诼棁资渍眦缁鹚赀镃龇鲻辎锱踪总纵综疭偬骤诹邹驺鲰诌组诅镞钻缵躜槜鳟});
}

=for Pod::Coverage clean_other count_simplified_chinese count_traditional_chinese features

=cut

1;
