package Promises6::Util;
use Evo::Base -strict;

use Scalar::Util;
use Exporter 'import';

my @CONST = qw(PENDING FULFILLED REJECTED PROGRESS);
our @EXPORT_OK = (@CONST, qw(is_promise is_thenable));
our %EXPORT_TAGS = (all => \@EXPORT_OK, const => \@CONST);

use constant PENDING   => -1;
use constant FULFILLED => 0;
use constant REJECTED  => 1;
use constant PROGRESS  => 2;

sub is_promise($val) {
  return unless $val && Scalar::Util::blessed($val);
  return $val->isa(Promises6::Builder->singleton->promise_class);
}

sub is_thenable($val) {
  return $val && Scalar::Util::blessed($val) && $val->can('then');
}

1;

# ABSTRACT: internal utils

__END__

=pod

=encoding UTF-8

=head1 NAME

Promises6::Util - internal utils

=head1 VERSION

version 0.004

=head1 SYNOPSIS

  use Evo::Base -strict;
  use Promises6::Util qw(is_promise is_thenable);

  my $object;
  my $is_promise  = is_promise($object);
  my $is_thenable = is_thenable($object);

=head1 METHODS

=head2 is_promise

  Returns true if argument is a promise object by checking if it's a
  subclass of L<Promises6::Promise>. Returns false otherwise

=head2 is_thenable

  Returns true if an argument is a blessed object with C<then> method.
  Returns false otherwise

=head1 CONST

Represent internal states

=over 4

=item *

PENDING

=item *

FULFILLED

=item

REJECTED

=item *

PROGRESS

=back

=head1 AUTHOR

alexbyk <alexbyk.com>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by alexbyk.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
