#!/usr/bin/perl -w

# Testing PITA::Guest

use strict;
use lib ();
use File::Spec::Functions ':ALL';
BEGIN {
	$| = 1;
	unless ( $ENV{HARNESS_ACTIVE} ) {
		require FindBin;
		$FindBin::Bin = $FindBin::Bin; # Avoid a warning
		chdir catdir( $FindBin::Bin, updir() );
		lib->import(
			catdir('blib', 'lib'),
			catdir('blib', 'arch'),
			'lib',
			);
	}
}

use Test::More tests => 40;

use PITA ();
use File::Remove 'remove';

# Find the test guest file
my $local_empty = catfile( 't', 'guests', 'local_empty.pita' );
ok( -f $local_empty, 'Found local_empty test file' );

# Set up the write test guest file
my $local_write = rel2abs(catfile( 't', 'guests', 'local_write.pita' ));
      if ( -f $local_write ) { remove( $local_write ) }
END { if ( -f $local_write ) { remove( $local_write ) } }
ok( ! -f $local_write, 'local_write.pita does not exist' );

# Find the test request file
my $simple_request = catfile( 't', 'requests', 'simple_request.pita' );
ok( -f $simple_request, 'Found simple request file' );
my $simple_module = catfile( 't', 'requests', 'PITA-Test-Dummy-Perl5-Make-1.01.tar.gz' );
ok( -f $simple_module, 'Found simple dist file' );





#####################################################################
# Tests for the Local case that is not discovered

# Load the raw PITA::XML::Guest directly
SCOPE: {
	my $xml = PITA::XML::Guest->read( $local_empty );
	isa_ok( $xml, 'PITA::XML::Guest' );
	is( $xml->driver, 'Local', '->driver is Local' );
	is( scalar($xml->platforms),    0,  '->platforms(scalar) returns 0' );
	is_deeply( [ $xml->platforms ], [], '->platforms(list) return ()'   );

	# Write the file for the write test
	ok( $xml->write( $local_write ), '->write returns true' );
	my $xml2 = PITA::XML::Guest->read( $local_write );
	isa_ok( $xml2, 'PITA::XML::Guest' );
	is_deeply( $xml2, $xml, 'local_empty matches local_write' );
}

# Load a PITA::Guest for the local_empty case
SCOPE: {
	my $guest = PITA::Guest->new( $local_empty );
	isa_ok( $guest, 'PITA::Guest' );
	is( $guest->file, $local_empty,
		'->file returns the original filename' );
	isa_ok( $guest->guestxml,  'PITA::XML::Guest'           );
	is( $guest->discovered, '', '->discovered returns false' );
	isa_ok( $guest->driver, 'PITA::Guest::Driver'        );
	isa_ok( $guest->driver, 'PITA::Guest::Driver::Local' );
	is( scalar($guest->guestxml->platforms),    0,  '->platforms(scalar) returns 0' );
	is_deeply( [ $guest->guestxml->platforms ], [], '->platforms(list) return ()'   ); 

	# Ping the Guest
	ok( $guest->ping, '->ping returned true' );

	# Discover the platforms
	ok( $guest->discover, '->discover returned true' );
	is( scalar($guest->guestxml->platforms),    1,  '->platforms(scalar) returns 1' );
	isa_ok( ($guest->guestxml->platforms)[0], 'PITA::XML::Platform' );
}

# Repeat, but this time discover and save it
SCOPE: {
	my $guest = PITA::Guest->new( $local_write );
	isa_ok( $guest, 'PITA::Guest' );
	is( $guest->discovered, '', '->discovered is false' );
	ok( $guest->discover, '->discover returns true' );
	is( scalar($guest->guestxml->platforms), 1, '->platforms(scalar) returns 1' );
	isa_ok( ($guest->guestxml->platforms)[0], 'PITA::XML::Platform' );
	ok( $guest->save, '->save returns true' );

	# Did we save ok?
	my $guest2 = PITA::Guest->new( $local_write );
	isa_ok( $guest2, 'PITA::Guest' );

	# Ignoring their tempdir values, do they match
	delete $guest->driver->{tempdir};
	delete $guest->driver->{workarea};
	delete $guest2->driver->{tempdir};
	delete $guest2->driver->{workarea};
	is_deeply( $guest2, $guest, 'PITA::Guest object saved ok' );
}

# Execute a simple test run
SCOPE: {
	my $guest = PITA::Guest->new( $local_write );
	isa_ok( $guest,           'PITA::Guest'      );
	isa_ok( $guest->guestxml, 'PITA::XML::Guest' );

	# Load the request object
	my $request = PITA::XML::Request->read( $simple_request );
	isa_ok( $request, 'PITA::XML::Request' );

	# Try to test it
	my $report = $guest->test( $simple_request );
	isa_ok( $report, 'PITA::XML::Report' );

	# Check that it actually did a test
	is( scalar($report->installs), 1, '->installs returns 1 object' );
	my $install = ($report->installs)[0];
	isa_ok( $install, 'PITA::XML::Install' );
	isa_ok( $install->request, 'PITA::XML::Request' );
	is_deeply( $install->request, $request, 'Request matched original' );
	is( scalar($install->commands), 3, 'Found three commands' );
}

exit(0);
