package    # hide from CPAN
  P v0.10.2;

use v5.20.0;
use utf8;
use strict;
use warnings ( qw[all], FATAL => qw[utf8], NONFATAL => qw[] );
no  if $^V ge 'v5.18', warnings => 'experimental';
use if $^V ge 'v5.10', feature  => ':all';
no  if $^V ge 'v5.16', feature  => 'array_base';
use if $^V ge 'v5.22', re       => 'strict';
use if $^V ge 'v5.10', mro      => 'c3';
no multidimensional;

# NOTE exporting indirect pragma cause random crashes under windows
# https://rt.cpan.org/Ticket/Display.html?id=102321
# no if $^O !~ /MSWin/sm, indirect => qw[fatal];

use namespace::clean();

use Encode();    ## no critic qw(Modules::ProhibitEvilModules)

# preload Moo
use Import::Into;
use Moo();          ## no critic qw(Modules::ProhibitEvilModules)
use Moo::Role();    ## no critic qw(Modules::ProhibitEvilModules)

# console related packages
use Term::ANSIColor();
use PerlIO::encoding();

use B::Hooks::AtRuntime();
use B::Hooks::EndOfScope::XS();

# AnyEvent
use EV;
use AnyEvent;

# define global variables
BEGIN {
    $P::CALLER           = caller;                    # namespace, from P was required first time
    $P::EMBEDDED         = 0;                         # P::Core used in embedded mode
    $P::IS_PAR           = $ENV{PAR_TEMP} ? 1 : 0;    # process runned from PAR
    $P::ENCODING_CONSOLE = 'UTF-8';                   # default console encoding, redefined later for windows
    $P::NO_ISA_ATTR      = 0;                         # do not check isa for class / role attributes

    # NOTE workaround for incompatibility with Moo lazy attributes
    # https://rt.cpan.org/Ticket/Display.html?id=102788
    eval {
        local $SIG{__DIE__} = undef;

        require Filter::Crypto::Decrypt if $P::IS_PAR;
    };

    # define %EXPORT_PRAGMAS
    %P::EXPORT_PRAGMAS = (
        config      => 0,    # mark package as perl config, used automatically during .perl config evaluation, do not use directly!!!
        embedded    => 0,    # run in embedded mode
        export      => 0,    # install standart import method
        no_isa_attr => 0,    # do not check isa for class / role attributes
        autoload    => 0,    # export AUTOLOAD
        cli         => 0,    # add package namespace to CLI processors stack
        no_clean    => 0,    # do not perform namespace autoclean
        class       => 0,    # package is a Moo class
        role        => 0,    # package is a Moo role
        types       => 0,    # export types
    );

    $P::UTIL = {
        bit      => 'P::Util::Bit',
        capture  => 'P::Util::Capture',
        cfg      => 'P::Util::Config',
        class    => 'P::Util::Class',
        data     => 'P::Util::Data',
        date     => 'P::Util::Date',
        digest   => 'P::Util::Digest',
        file     => 'P::Util::File',
        geoip    => 'P::Util::GeoIP',
        hash     => 'P::Util::Hash',
        list     => 'P::Util::List',
        mail     => 'P::Util::Mail',
        moo      => 'P::Util::Moo',
        pm       => 'P::Util::PM',
        progress => 'P::Util::Progress',
        prompt   => 'P::Util::Prompt',
        random   => 'P::Util::Random',
        res      => 'P::Util::Resources',
        scalar   => 'P::Util::Scalar',
        sys      => 'P::Util::Sys',
        text     => 'P::Util::Text',
        tmpl     => 'P::Util::Template',
        ua       => 'P::HTTP::UA',
        uri      => 'P::Util::URI',
        uuid     => 'P::Util::UUID',
    };
}

use P::Core::Exporter();

sub import {
    my $self = shift;

    # parse tags and pragmas
    my ( $tags, $pragma ) = P::Core::Exporter::Helper->parse_import( $self, @_ );

    # find caller
    my $caller = caller;

    # export perl pragmas
    utf8->import();
    strict->import();
    warnings->import( 'all', FATAL => qw[utf8], NONFATAL => qw[] );
    warnings->unimport('experimental') if $^V ge 'v5.18';
    feature->import(':all')            if $^V ge 'v5.10';
    feature->unimport('array_base')    if $^V ge 'v5.16';
    re->import('strict')               if $^V ge 'v5.22';
    mro::set_mro( $caller, 'c3' ) if $^V ge 'v5.10';
    multidimensional->unimport;

    # NOTE exporting indirect pragma cause random crashes under windows
    # https://rt.cpan.org/Ticket/Display.html?id=102321
    # indirect->unimport('fatal') if $^O !~ /MSWin/sm;

    # export P sub to avoid indirect calls
    {
        state $val;

        if ( !$val ) {
            $val = 'P';

            Internals::SvREADONLY( $val, 1 );
        }

        no strict qw[refs];

        my $symtab = \%{ $caller . q[::] };

        $symtab->{P} = \$val;

        # flush the cache exactly once if we make any direct symbol table changes
        mro::method_changed_in($caller);
    }

    # export P::Core
    P::Core::Constants->import( -caller => $caller, $tags->@* );
    P::Core::Dump->import( -caller => $caller, $tags->@* );
    P::Core::Exception->import( -caller => $caller, $tags->@* );
    P::Core::H->import( -caller => $caller, $tags->@* );
    P::Core::Log->import( -caller => $caller, $tags->@* );
    P::Core::I18N->import( -caller => $caller, $tags->@* );

    if ( !$pragma->{config} ) {

        # install run-time hook to caller package
        if ( $caller eq $P::CALLER ) {
            B::Hooks::AtRuntime::at_runtime(
                sub {
                    P->_CORE_RUN();
                }
            );
        }

        # install standart import method
        if ( $pragma->{export} ) {
            no strict qw[refs];

            push @{ $caller . '::ISA' }, 'P::Core::Exporter';
        }

        # autoload for non-Moo namespaces
        if ( $pragma->{autoload} && !$pragma->{class} && !$pragma->{role} ) {
            require P::Core::Autoload;

            P::Core::Autoload->import( -caller => $caller );
        }

        # store significant pragmas for use in run-time
        $P::EMBEDDED = 1 if $pragma->{embedded};

        $P::NO_ISA_ATTR = 1 if $pragma->{no_isa_attr};

        # CLI
        if ( $pragma->{cli} ) {
            push @P::Core::CLI::PACKAGES, $caller;
        }

        # re-export Moo
        if ( $pragma->{class} || $pragma->{role} ) {

            # install universal serializer methods
            B::Hooks::EndOfScope::XS::on_scope_end(
                sub {
                    no strict qw[refs];

                    if ( my $ref = $caller->can('TO_DATA') ) {
                        *{ $caller . '::TO_JSON' } = $ref unless $caller->can('TO_JSON');

                        *{ $caller . '::TO_CBOR' } = $ref unless $caller->can('TO_CBOR');
                    }

                    return;
                }
            );

            $pragma->{types} = 1;

            if ( $pragma->{class} ) {
                $self->_import_moo( caller => $caller, class => 1 );
            }
            elsif ( $pragma->{role} ) {
                $self->_import_moo( caller => $caller, role => 1 );
            }

            # reconfigure warnings, after Moo exported
            warnings->import( 'all', FATAL => qw[utf8], NONFATAL => qw[] );
            warnings->unimport('experimental');

            # apply default roles
            $self->_apply_roles( caller => $caller, roles => [qw[P::Core::Autoload::Role]] ) if $pragma->{autoload};
        }

        # export types
        $self->_import_types( caller => $caller ) if $pragma->{types};
    }

    # cleanup
    namespace::clean->import( -cleanee => $caller, -except => [qw[AUTOLOAD]] ) unless $pragma->{no_clean};

    return;
}

sub unimport {    ## no critic qw(Subroutines::ProhibitBuiltinHomonyms)
    my $self = shift;

    # parse pragmas and tags
    my ( $tags, $pragma ) = P::Core::Exporter::Helper->parse_import( $self, @_ );

    # find caller
    my $caller = caller;

    # try to unimport Moo keywords
    $self->_unimport_moo( caller => $caller );

    # unimport types
    $self->_unimport_types( caller => $caller );

    return;
}

sub _import_moo {
    my $self = shift;
    my %args = (
        caller => undef,
        class  => undef,
        role   => undef,
        @_,
    );

    if ( $args{class} ) {
        Moo->import::into( $args{caller} );
    }
    else {
        Moo::Role->import::into( $args{caller} );
    }

    # install "has" hook
    {
        no strict qw[refs];

        my $has = *{ $args{caller} . '::has' }{CODE};

        no warnings qw[redefine];    ## no critic qw(TestingAndDebugging::ProhibitNoWarnings)

        *{ $args{caller} . '::has' } = sub {
            my ( $name_proto, %spec ) = @_;

            # disable type checking
            delete $spec{isa} if $P::NO_ISA_ATTR;

            $has->( $name_proto, %spec );
        };
    }

    return;
}

sub _unimport_moo {
    my $self = shift;
    my %args = (
        caller => undef,
        @_,
    );

    if ( $Moo::MAKERS{ $args{caller} } && $Moo::MAKERS{ $args{caller} }->{is_class} ) {    # Moo class
        Moo->unimport::out_of( $args{caller} );
    }
    elsif ( Moo::Role->is_role( $args{caller} ) ) {                                        # Moo::Role
        Moo::Role->unimport::out_of( $args{caller} );
    }

    return;
}

sub _import_types {
    my $self = shift;
    my %args = (
        caller => undef,
        @_,
    );

    state $required;

    if ( !$required ) {
        $required = 1;

        local $ENV{PERL_TYPES_STANDARD_STRICTNUM} = 0;    # 0 - Num = LaxNum, 1 - Num = StrictNum

        require P::Core::Types;
        require Types::TypeTiny;                          ## no critic qw(Modules::ProhibitEvilModules)
        require Types::Standard;                          ## no critic qw(Modules::ProhibitEvilModules)
        require Types::Common::Numeric;                   ## no critic qw(Modules::ProhibitEvilModules)

        # require Types::Common::String;     ## no critic qw(Modules::ProhibitEvilModules)
        # require Types::Encodings();        ## no critic qw(Modules::ProhibitEvilModules)
        # require Types::XSD::Lite();        ## no critic qw(Modules::ProhibitEvilModules)
    }

    Types::TypeTiny->import( { into => $args{caller} }, qw[StringLike HashLike ArrayLike CodeLike TypeTiny] );

    Types::Standard->import( { into => $args{caller} }, ':types' );

    Types::Common::Numeric->import( { into => $args{caller} }, ':types' );

    P::Core::Types->import( { into => $args{caller} }, ':types' );

    return;
}

sub _unimport_types {
    my $self = shift;
    my %args = (
        caller => undef,
        @_,
    );

    P::Core::Types->unimport( { into => $args{caller} }, ':types' );

    Types::Common::Numeric->unimport( { into => $args{caller} }, ':types' );

    Types::Standard->unimport( { into => $args{caller} }, ':types' );

    Types::TypeTiny->unimport( { into => $args{caller} }, qw[StringLike HashLike ArrayLike CodeLike TypeTiny] );

    return;
}

sub _apply_roles {
    my $self = shift;
    my %args = (
        caller => undef,
        roles  => undef,
        @_,
    );

    Moo::Role->apply_roles_to_package( $args{caller}, $args{roles}->@* );

    if ( Moo::Role->is_role( $args{caller} ) ) {
        Moo::Role->_maybe_reset_handlemoose( $args{caller} );    ## no critic qw(Subroutines::ProtectPrivateSubs)
    }
    else {
        Moo->_maybe_reset_handlemoose( $args{caller} );          ## no critic qw(Subroutines::ProtectPrivateSubs)
    }

    return;
}

# CORE compilation phase
use P::Core::Constants qw[:CORE];
use P::Core::EV qw[:CORE];
use P::Core::Bootstrap qw[:CORE];
use P::Core::Dump qw[:CORE];
use P::Core::Exception qw[:CORE];
use P::Core::H qw[:CORE];
use P::Core::Log qw[:CORE];
use P::Core::I18N qw[:CORE];

use P::Core::CLI();

_CORE_INIT();

sub _CORE_INIT {

    # set default fallback mode for all further :encoding I/O layers
    $PerlIO::encoding::fallback = Encode::FB_CROAK | Encode::STOP_AT_PARTIAL;

    if ($MSWIN) {
        require Win32::API;
        require Win32::Console::ANSI;

        # detect windows active codepage
        $P::ENCODING_CONSOLE = 'cp' . Win32::API::More->new( 'kernel32', 'int GetACP()' )->Call;

        # check if we can properly decode STDIN under MSWIN
        eval {
            Encode::perlio_ok($P::ENCODING_CONSOLE) or die;
            1;
        } || do {
            say qq[FATAL: Console input encoding "$P::ENCODING_CONSOLE" isn't supported. Use chcp to change console codepage.];

            exit 1;
        };
    }

    # decode @ARGV
    for (@ARGV) {
        $_ = Encode::decode( $P::ENCODING_CONSOLE, $_, Encode::FB_CROAK );
    }

    P::Core::Bootstrap::CORE_INIT();    # load default configs

    _configure_console();

    P::Core::EV::CORE_INIT();           # create global event queue
    P::Core::Log::CORE_INIT();          # set default log pipes
    P::Core::Exception::CORE_INIT();    # set $SIG{__DIE__}, $SIG{__WARN__}, $SIG->{INT}, $SIG->{TERM} handlers
    P::Core::I18N::CORE_INIT();         # configure default I18N locations
    P::Core::CLI::CORE_INIT();          # register CORE::CLI event

    return;
}

sub _CORE_RUN {

    # EMBEDDED mode, if run not from INIT block or -embedded pragma specified:
    # CLI not parsed / processed;
    # process permissions not changed;
    # process will not daemonized;

    if ( !$P::EMBEDDED ) {

        # throw CORE#CLI event, CLI parsing and processing
        P->EV->throw('CORE#CLI');

        # throw CORE#RUN event to perform daemonize, depends on CLI param
        P->EV->throw('CORE#RUN');

        if ( !$MSWIN ) {

            # GID is inherited from UID by default
            if ( defined $PROC->{UID} && !defined $PROC->{GID} ) {
                my $uid = $PROC->{UID} =~ /\A\d+\z/sm ? $PROC->{UID} : getpwnam $PROC->{UID};

                die qq[Can't find uid "$PROC->{UID}"] if !defined $uid;

                $PROC->{GID} = [ getpwuid $uid ]->[2];
            }

            # change priv
            P->pm->change_priv( gid => $PROC->{GID}, uid => $PROC->{UID} );
        }
    }

    return;
}

# AUTOLOAD
sub AUTOLOAD ( $self, @ ) {    ## no critic qw(ClassHierarchies::ProhibitAutoloading)
    my $method = our $AUTOLOAD =~ s/\A.*:://smr;

    die qq[Unregistered P::Util "$method".] unless exists $P::UTIL->{$method};

    my $class = $P::UTIL->{$method};

    require $class =~ s[::][/]smgr . q[.pm];

    if ( $class->can('NEW') ) {
        no strict qw[refs];

        *{ $self . q[::] . $method } = \&{ $class . '::NEW' };

        goto &{ $self . q[::] . $method };
    }
    else {
        my $val = $P::UTIL->{$method};

        my $symtab;

        # get symbol table ref
        {
            no strict qw[refs];

            $symtab = \%{ $self . q[::] };
        }

        # install method
        $symtab->{$method} = sub : prototype() {$val};

        mro::method_changed_in($self);

        return $val;
    }
}

sub cv {
    state $cv = AE::cv;

    return $cv;
}

sub _configure_console {

    # clone handles
    open our $STDIN,  '<&STDIN'  or $STDIN  = *STDIN;     ## no critic qw(InputOutput::ProhibitBarewordFileHandles Variables::RequireLocalizedPunctuationVars)
    open our $STDOUT, '>&STDOUT' or $STDOUT = *STDOUT;    ## no critic qw(InputOutput::ProhibitBarewordFileHandles Variables::RequireLocalizedPunctuationVars)
    open our $STDERR, '>&STDERR' or $STDERR = *STDERR;    ## no critic qw(InputOutput::ProhibitBarewordFileHandles Variables::RequireLocalizedPunctuationVars)

    # setup default layers for STD* handles
    binmode STDIN,  q[:raw:encoding(UTF-8)] or die;
    binmode STDOUT, q[:raw:encoding(UTF-8)] or die;
    binmode STDERR, q[:raw:encoding(UTF-8)] or die;

    # setup default layers for STD* handles clones
    if ($MSWIN) {
        require PerlIO::via::Win32UnicodeConsole;

        binmode $STDIN,  qq[:raw:crlf:encoding($P::ENCODING_CONSOLE)] or die;
        binmode $STDOUT, q[:raw:via(Win32UnicodeConsole)]             or die;
        binmode $STDERR, q[:raw:via(Win32UnicodeConsole)]             or die;

        select $STDOUT;    ## no critic qw(InputOutput::ProhibitOneArgSelect)
    }
    else {
        binmode $STDIN,  q[:raw:encoding(UTF-8)] or die;
        binmode $STDOUT, q[:raw:encoding(UTF-8)] or die;
        binmode $STDERR, q[:raw:encoding(UTF-8)] or die;
    }

    # make STD* non-caching
    STDOUT->autoflush(1);
    STDERR->autoflush(1);

    $STDOUT->autoflush(1);
    $STDERR->autoflush(1);

    return;
}

1;
## -----SOURCE FILTER LOG BEGIN-----
##
## PerlCritic profile "pcore-script" policy violations:
## ┌──────┬──────────────────────┬────────────────────────────────────────────────────────────────────────────────────────────────────────────────┐
## │ Sev. │ Lines                │ Policy                                                                                                         │
## ╞══════╪══════════════════════╪════════════════════════════════════════════════════════════════════════════════════════════════════════════════╡
## │    3 │ 49                   │ ErrorHandling::RequireCheckingReturnValueOfEval - Return value of eval not tested                              │
## ├──────┼──────────────────────┼────────────────────────────────────────────────────────────────────────────────────────────────────────────────┤
## │    3 │ 100                  │ Subroutines::ProhibitExcessComplexity - Subroutine "import" with high complexity score (26)                    │
## ├──────┼──────────────────────┼────────────────────────────────────────────────────────────────────────────────────────────────────────────────┤
## │    3 │ 158                  │ Subroutines::ProtectPrivateSubs - Private subroutine/method used                                               │
## ├──────┼──────────────────────┼────────────────────────────────────────────────────────────────────────────────────────────────────────────────┤
## │    1 │ 496                  │ CodeLayout::ProhibitParensWithBuiltins - Builtin function called with parentheses                              │
## └──────┴──────────────────────┴────────────────────────────────────────────────────────────────────────────────────────────────────────────────┘
##
## -----SOURCE FILTER LOG END-----
__END__
=pod

=encoding utf8

=head1 NAME

P - pcore perl application framework

=head1 SYNOPSIS

=head1 DESCRIPTION

=head1 ENVIRONMENT

PCORE_DIST_LIB

PCORE_RESOURCES

PCORE_MOUNTED_RESOURCES

DZIL_CONFIRMRELEASE_DEFAULT

MODULE_SIGNATURE_AUTHOR = q["<author-email>"]

__PCORE_SCRIPT_DIR
__PCORE_SCRIPT_NAME

=cut
