package Pcore::Whois v0.5.0;

use Pcore -dist, -class, -const;
use Pcore::Util::Scalar qw[blessed];
use Pcore::Whois::Data qw[:CONST];
use Pcore::Whois::Server;
use Pcore::Whois::Request;
use Pcore::Whois::Response;

has max_threads => ( is => 'ro', isa => PositiveInt, default => 50 );    # number of total max. parallel threads allowed
has max_threads_server => ( is => 'lazy', isa => PositiveInt );                  # max. parallel threads / particular WHOIS server, if ip pool is used - can be setted to ip's number
has timeout            => ( is => 'ro',   isa => PositiveInt, default => 30 );
has retries            => ( is => 'ro',   isa => PositiveInt, default => 3 );    # max. retries

has ip_pool => ( is => 'ro', isa => Maybe [ InstanceOf ['Pcore::AE::Handle::IPPool'] ] );
has proxy   => ( is => 'ro', isa => Maybe [ InstanceOf ['Pcore::AE::Handle::ProxyPool'] ] );

has cache => ( is => 'ro', isa => Str );                                                   # path to SQLite database
has cache_timeout => ( is => 'ro', isa => PositiveOrZeroInt, default => 60 * 60 * 24 );    # 1 day, 0 - only put to the cache

has cache_dbh => ( is => 'lazy', isa => InstanceOf ['Pcore::Handle::SQLite'], init_arg => undef );

has server         => ( is => 'ro', isa => HashRef,  default => sub { {} }, init_arg => undef );    # server objects cache
has requests       => ( is => 'ro', isa => HashRef,  default => sub { {} }, init_arg => undef );    # active requests
has requests_queue => ( is => 'ro', isa => ArrayRef, default => sub { [] }, init_arg => undef );    # delayed requests queue
has threads => ( is => 'ro', isa => PositiveOrZeroInt, default => 0, init_arg => undef );           # total active threads num.

sub BUILDARGS ( $self, $args ) {
    $args->{ip_pool} = P->class->load('Pcore::AE::Handle::IPPool')->new( { ip => $args->{ip_pool} } ) if $args->{ip_pool} && !blessed $args->{ip_pool};

    return $args;
}

sub _build_max_threads_server ($self) {
    return $self->ip_pool ? $self->ip_pool->size : 1;
}

sub _build_cache_dbh ($self) {
    H->add(
        whois_cache  => 'SQLite',
        addr         => 'file:' . $self->cache,
        journal_mode => 'WAL',
        synchronous  => 'OFF',
        cache_size   => 20_000,
    );

    my $dbh = H->whois_cache;

    my $ddl = $dbh->ddl;

    $ddl->add_changeset(
        id  => 1,
        sql => <<'SQL'
                CREATE TABLE IF NOT EXISTS `whois` (
                    `id` INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL,
                    `domain` TEXT NOT NULL,         -- root domain
                    `check_ts` INTEGER NOT NULL,    -- check timestamp
                    `status` INTEGER NOT NULL,      -- check status
                    `is_found` INTEGER NOT NULL,    --
                    `expire` INTEGER                -- expire date, if available
                );

                CREATE UNIQUE INDEX IF NOT EXISTS `idx_uniq_domain` ON `whois` (`domain` ASC);
                -- CREATE INDEX IF NOT EXISTS `idx_url_status_depth` ON `url` (`status` ASC, `depth` ASC);
SQL
    );

    $ddl->upgrade;

    return $dbh;
}

sub whois ( $self, $domain, $cb ) {
    $domain = P->host($domain) if !ref $domain;

    # cut $domain->canon to the 3 labels max.;
    my @labels = split /[.]/sm, $domain->canon;

    my $query;

    if ( @labels == 1 ) {
        $query = $labels[-1];
    }
    elsif ( @labels == 2 ) {
        $query = "$labels[-2].$labels[-1]";
    }
    else {

        # TODO
        # compute.amazonaws.com    amazonaws.com is pub. suff.   query = amazonaws.com, NOTE now query will be compute.amazonaws.com, that is invalid for this whois server
        # aaa.kiev.ua              kiev.ua is pub. suff.         query = aaa.kiev.ua
        # aaa.ua                   ua is pub. suff.              query = aaa.ua
        # aaa.co.uk                co.uk is pub. suff.           query = aaa.co.uk

        $query = P->host("$labels[-3].$labels[-2].$labels[-1]");

        $query = $query->root_domain || $query->pub_suffix;
    }

    # if TLD is not valid - return immediately;
    if ( !$domain->tld_is_valid ) {
        my $res = Pcore::Whois::Response->new(
            {   query  => $query,
                raw    => undef,
                status => $WHOIS_STATUS_NO_SERVER,
                reason => 'TLD is not valid',
            }
        );

        $cb->( $domain, $res );

        return;
    }

    # return cached results immediately
    if ( my $res = $self->_check_cache($query) ) {
        $cb->( $domain, $res );

        return;
    }

    $self->_get_server_for_domain(
        $query,
        sub ($server) {
            if ( $server->status != 200 ) {

                # create and return empty response
                my $res = Pcore::Whois::Response->new(
                    {   query  => $query,
                        server => $server,
                        raw    => undef,
                        status => $WHOIS_STATUS_NO_SERVER,
                        reason => 'No WHOIS server available',
                    }
                );

                $cb->( $domain, $res );
            }
            else {
                $self->run_request(
                    $server, $query,
                    sub ($res) {

                        # cache only is_success requests
                        $self->_update_cache( $query, $res ) if $self->cache && $res->is_success;

                        $cb->( $domain, $res );

                        return;
                    }
                );
            }

            return;
        }
    );

    return;
}

sub _check_cache ( $self, $query ) {
    return if !$self->cache_timeout;

    return if !$self->cache;

    state $select_sql = $self->cache_dbh->query(q[SELECT * FROM whois WHERE domain = ? AND check_ts > ? LIMIT 1]);

    if ( my $row = $select_sql->selectrow( bind => [ $query, time - $self->cache_timeout ] ) ) {
        my $res = Pcore::Whois::Response->new(
            {   query  => $query,
                status => $row->{status},
                reason => q[],
                cached => 1,
            }
        );

        $res->{is_success} = 1;                  # we cache only success requests
        $res->{status}     = $row->{status};
        $res->{reason}     = 'OK';
        $res->{is_found}   = $row->{is_found};
        $res->{expire}     = $row->{expire};

        return $res;
    }

    return;
}

sub _update_cache ( $self, $query, $res ) {
    state $insert_sql = $self->cache_dbh->query(q[INSERT OR IGNORE INTO whois (domain, check_ts, status, is_found, expire) VALUES (?, ?, ?, ?, ?)]);

    state $update_sql = $self->cache_dbh->query(q[UPDATE whois SET domain = ?, check_ts = ?, status = ?, is_found = ?, expire = ? WHERE domain = ?]);

    if ( !$insert_sql->do( bind => [ $query, time, $res->status, $res->is_found, undef ] ) ) {
        $update_sql->do( bind => [ $query, time, $res->status, $res->is_found, $res->expire, $query ] );
    }

    return;
}

# REQUEST
sub run_request ( $self, $server, $query, $cb ) {
    my $id = $server->host . "-" . $query;

    if ( !exists $self->requests->{$id} ) {

        # create and cache new request
        my $req = Pcore::Whois::Request->new(
            {   id      => $id,
                whois   => $self,
                retries => $self->retries,
                server  => $server,
                query   => $query,
                request => [$cb],
                cb      => $cb,
            }
        );

        $self->requests->{$id} = $req;

        $req->run;
    }
    else {

        # update request
        my $req = $self->requests->{$id};

        push $req->request->@*, $cb;
    }

    return;
}

sub _get_server_for_domain ( $self, $query, $cb ) {
    my @labels = split /[.]/sm, $query;

    my $tld = $labels[-1];

    my $get_server = sub ($host) {
        $host //= q[];

        if ( exists $self->server->{$host} ) {
            $cb->( $self->server->{$host} );
        }
        else {
            my $server = Pcore::Whois::Server->new($host);

            if ( !$host ) {

                # create and return empty server object
                $self->server->{$host} //= $server;

                $cb->( $self->server->{$host} );
            }
            else {
                $server->validate(
                    $self, $tld,
                    sub ($server) {
                        $self->server->{$host} //= $server;

                        $cb->( $self->server->{$host} );

                        return;
                    }
                );
            }
        }

        return;
    };

    my $server_host;
    my $server_found;

    if ( @labels > 2 ) {
        if ( exists $Pcore::Whois::Data::TLD->{"$labels[-2].$labels[-1]"} ) {
            $server_found = 1;

            $server_host = $Pcore::Whois::Data::TLD->{"$labels[-2].$labels[-1]"};
        }
    }

    if ( !$server_found ) {
        if ( exists $Pcore::Whois::Data::TLD->{ $labels[-1] } ) {
            $server_found = 1;

            $server_host = $Pcore::Whois::Data::TLD->{ $labels[-1] };
        }
    }

    if ($server_found) {
        $get_server->($server_host);
    }
    else {

        # get server for the given domain TLD from IANA WHOIS service
        $self->_get_iana_server(
            sub ($iana_server) {
                $self->run_request(
                    $iana_server,
                    $tld,
                    sub ($res) {
                        my $host;

                        if ( $res->is_success && $res->raw->$* =~ /^whois:\s+(.+?)\n/sm ) {
                            $host = $1;
                        }

                        # store tld -> server info
                        $Pcore::Whois::Data::TLD->{$tld} = $host;

                        $get_server->($host);

                        return;
                    }
                );

                return;
            }
        );
    }

    return;
}

sub _get_iana_server ( $self, $cb ) {
    state $iana_server;

    if ( !$iana_server ) {
        my $_iana_server = Pcore::Whois::Server->new('whois.iana.org');

        $_iana_server->validate(
            $self,
            'f-u-c-k',
            sub ($self) {
                die q[IANA Whois server whois.iana.org is not available ] if $self->status != 200;

                $iana_server //= $_iana_server;

                $cb->($iana_server);

                return;
            }
        );
    }
    else {
        $cb->($iana_server);
    }

    return;
}

1;
## -----SOURCE FILTER LOG BEGIN-----
##
## PerlCritic profile "pcore-script" policy violations:
## ┌──────┬──────────────────────┬────────────────────────────────────────────────────────────────────────────────────────────────────────────────┐
## │ Sev. │ Lines                │ Policy                                                                                                         │
## ╞══════╪══════════════════════╪════════════════════════════════════════════════════════════════════════════════════════════════════════════════╡
## │    3 │ 203                  │ ValuesAndExpressions::ProhibitInterpolationOfLiterals - Useless interpolation of literal string                │
## ├──────┼──────────────────────┼────────────────────────────────────────────────────────────────────────────────────────────────────────────────┤
## │    2 │ 203                  │ ValuesAndExpressions::ProhibitNoisyQuotes - Quotes used with a noisy string                                    │
## └──────┴──────────────────────┴────────────────────────────────────────────────────────────────────────────────────────────────────────────────┘
##
## -----SOURCE FILTER LOG END-----
__END__
=pod

=encoding utf8

=head1 NAME

Pcore::Whois

=head1 SYNOPSIS

=head1 DESCRIPTION

=head1 SEE ALSO

L<WHOIS implementation|http://habrahabr.ru/post/165869/>

L<WHOIS Protocol Specification (RFC 3912) |https://tools.ietf.org/html/rfc3912>

=head1 AUTHOR

zdm <zdm@cpan.org>

=head1 CONTRIBUTORS

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by zdm.

=cut
