package Pcore::Whois v0.3.2;

use Pcore -class, -const;
use Pcore::Util::Scalar qw[blessed];
use Pcore::Whois::Data qw[:CONST];
use Pcore::Whois::Server;
use Pcore::Whois::Request;
use Pcore::Whois::Response;

has max_threads => ( is => 'ro', isa => PositiveInt, default => 50 );    # number of total max. parallel threads allowed
has max_threads_server => ( is => 'lazy', isa => PositiveInt );                  # max. parallel threads / particular WHOIS server, if ip pool is used - can be setted to ip's number
has timeout            => ( is => 'ro',   isa => PositiveInt, default => 30 );
has retries            => ( is => 'ro',   isa => PositiveInt, default => 3 );    # max. retries

has ip_pool => ( is => 'ro', isa => Maybe [ InstanceOf ['Pcore::AE::Handle::IPPool'] ] );
has proxy   => ( is => 'ro', isa => Maybe [ InstanceOf ['Pcore::AE::Handle::ProxyPool'] ] );

has cache         => ( is => 'ro', isa => Str );                                           # path to SQLite database
has cache_timeout => ( is => 'ro', isa => PositiveOrZeroInt, default => 60 * 60 * 24 );    # 1 day, 0 - only put to the cache
has log           => ( is => 'ro', isa => Str );                                           # path to SQLite database

has cache_dbh => ( is => 'lazy', isa => InstanceOf ['Pcore::Handle::SQLite'], init_arg => undef );
has log_dbh   => ( is => 'lazy', isa => InstanceOf ['Pcore::Handle::SQLite'], init_arg => undef );

has server         => ( is => 'ro', isa => HashRef,  default => sub { {} }, init_arg => undef );    # server objects cache
has requests       => ( is => 'ro', isa => HashRef,  default => sub { {} }, init_arg => undef );    # active requests
has requests_queue => ( is => 'ro', isa => ArrayRef, default => sub { [] }, init_arg => undef );    # delayed requests queue
has threads => ( is => 'ro', isa => PositiveOrZeroInt, default => 0, init_arg => undef );           # total active threads num.

sub BUILDARGS ( $self, $args ) {
    $args->{ip_pool} = P->class->load('Pcore::AE::Handle::IPPool')->new( { ip => $args->{ip_pool} } ) if $args->{ip_pool} && !blessed $args->{ip_pool};

    return $args;
}

sub _build_max_threads_server ($self) {
    return $self->ip_pool ? $self->ip_pool->size : 1;
}

sub _build_cache_dbh ($self) {
    H->add(
        whois_cache  => 'SQLite',
        addr         => 'file:' . $self->cache,
        journal_mode => 'WAL',
        synchronous  => 'OFF',
        cache_size   => 20_000,
    );

    my $dbh = H->whois_cache;

    my $ddl = $dbh->ddl;

    $ddl->add_changeset(
        id  => 1,
        sql => <<'SQL'
                CREATE TABLE IF NOT EXISTS `whois` (
                    `id` INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL,
                    `domain` TEXT NOT NULL,         -- root domain
                    `check_ts` INTEGER NOT NULL,    -- check timestamp
                    `status` INTEGER NOT NULL,      -- check status
                    `is_found` INTEGER NOT NULL,    --
                    `expire` INTEGER                -- expire date, if available
                );

                CREATE UNIQUE INDEX IF NOT EXISTS `idx_uniq_domain` ON `whois` (`domain` ASC);
                -- CREATE INDEX IF NOT EXISTS `idx_url_status_depth` ON `url` (`status` ASC, `depth` ASC);
SQL
    );

    $ddl->upgrade;

    return $dbh;
}

sub _build_log_dbh ($self) {
    my $dbh;

    H->add(
        whois_log    => 'SQLite',
        addr         => 'file:' . $self->log,
        journal_mode => 'WAL',
        synchronous  => 'OFF',
        cache_size   => 20_000,
    );

    $dbh = H->whois_log;

    my $ddl = $dbh->ddl;

    $ddl->add_changeset(
        id  => 1,
        sql => <<'SQL'
                CREATE TABLE IF NOT EXISTS `whois` (
                    `query` TEXT NOT NULL,          -- query domain
                    `root_domain` TEXT NOT NULL,    -- root domain
                    `tld` TEXT NOT NULL,            -- tld
                    `check_ts` INTEGER NOT NULL,    -- check timestamp
                    `server` TEXT,                  -- whois server, used to perform WHOIS request
                    `status` INTEGER NOT NULL,      -- check status
                    `reason` TEXT NOT NULL,         -- check reason
                    `is_success` INTEGER NOT NULL,  --
                    `is_found` INTEGER NOT NULL,    -- domain was found on whois server
                    `is_verisign` INTEGER NOT NULL, --
                    `expire` INTEGER,               -- expire date, if available
                    `raw` TEXT                      -- raw response data
                );

                -- CREATE INDEX IF NOT EXISTS `idx_url_status_depth` ON `url` (`status` ASC, `depth` ASC);
SQL
    );

    $ddl->upgrade;

    return $dbh;
}

# TODO maybe use $Pcore::Whois::Data::TLD_NOT_AVAIL for not available TLDs
sub whois ( $self, $domain, $cb ) {
    $domain = P->host($domain) if !ref $domain;

    # return cached results immediately
    if ( my $res = $self->_check_cache($domain) ) {
        $cb->( $domain, $res );

        return;
    }

    $self->_get_server_for_domain(
        $domain,
        sub ($server) {
            if ( $server->status != 200 ) {

                # create and return empty response
                my $res = Pcore::Whois::Response->new(
                    {   query => $domain->root_domain || $domain->tld,
                        server => $server,
                        raw    => undef,
                        status => $WHOIS_STATUS_NO_SERVER,
                        reason => 'No WHOIS server available',
                    }
                );

                $self->_log( $domain, $res );

                $cb->( $domain, $res );
            }
            else {
                $self->run_request(
                    $server, $domain,
                    sub ( $domain, $res ) {
                        $self->_log( $domain, $res );

                        $cb->( $domain, $res );

                        return;
                    }
                );
            }

            return;
        }
    );

    return;
}

sub _check_cache ( $self, $domain ) {
    return if !$self->cache_timeout;

    return if !$self->cache;

    state $select_sql = $self->cache_dbh->query(q[SELECT * FROM whois WHERE domain = ? AND check_ts > ? LIMIT 1]);

    if ( my $row = $select_sql->selectrow( bind => [ $domain->root_domain, time - $self->cache_timeout ] ) ) {
        my $res = Pcore::Whois::Response->new(
            {   query  => $domain->to_string,
                status => $row->{status},
                reason => q[],
                cached => 1,
            }
        );

        $res->{is_success} = 1;                  # we cache only success requests
        $res->{status}     = $row->{status};
        $res->{reason}     = 'OK';
        $res->{is_found}   = $row->{is_found};
        $res->{expire}     = $row->{expire};

        return $res;
    }

    return;
}

sub _log ( $self, $domain, $res ) {

    # cache only is_success requests
    if ( $self->cache && $res->is_success ) {
        state $insert_sql = $self->cache_dbh->query(q[INSERT OR IGNORE INTO whois (domain, check_ts, status, is_found, expire) VALUES (?, ?, ?, ?, ?)]);

        state $update_sql = $self->cache_dbh->query(q[UPDATE whois SET domain = ?, check_ts = ?, status = ?, is_found = ?, expire = ? WHERE domain = ?]);

        if ( !$insert_sql->do( bind => [ $domain->root_domain, time, $res->status, $res->is_found, undef ] ) ) {
            $update_sql->do( bind => [ $domain->root_domain, time, $res->status, $res->is_found, $res->expire, $domain->root_domain ] );
        }
    }

    if ( $self->log ) {
        state $insert_sql = $self->log_dbh->query(q[INSERT INTO whois (query, root_domain, tld, check_ts, server, status, reason, is_success, is_found, is_verisign, expire, raw) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)]);

        $insert_sql->do( bind => [ $domain->to_string, $domain->root_domain, $domain->tld, time, $res->server->host, $res->status, $res->reason, $res->is_success, $res->is_found, $res->is_verisign, $res->expire, $res->raw ? $res->raw->$* : q[] ] );

    }

    return;
}

# REQUEST
sub run_request ( $self, $server, $domain, $cb ) {
    $domain = P->host($domain) if !ref $domain;

    my $query = $domain->root_domain || $domain->tld;

    my $id = $server->host . "-" . $query;

    if ( !exists $self->requests->{$id} ) {

        # create and cache new request
        my $req = Pcore::Whois::Request->new(
            {   id      => $id,
                whois   => $self,
                retries => $self->retries,
                server  => $server,
                query   => $query,
                request => [ [ $domain, $cb ] ],
                cb      => $cb,
            }
        );

        $self->requests->{$id} = $req;

        $req->run;
    }
    else {

        # update request
        my $req = $self->requests->{$id};

        push $req->request->@*, [ $domain, $cb ];
    }

    return;
}

sub _get_server_for_domain ( $self, $domain, $cb ) {
    $domain = P->host($domain) if !ref $domain;

    my $get_server = sub ($host) {
        $host //= q[];

        if ( exists $self->server->{$host} ) {
            $cb->( $self->server->{$host} );
        }
        else {
            my $server = Pcore::Whois::Server->new($host);

            if ( !$host ) {

                # create and return empty server object
                $self->server->{$host} //= $server;

                $cb->( $self->server->{$host} );
            }
            else {
                $server->validate(
                    $self,
                    $domain->tld,
                    sub ($server) {
                        $self->server->{$host} //= $server;

                        $cb->( $self->server->{$host} );

                        return;
                    }
                );
            }
        }

        return;
    };

    if ( !$domain->tld_is_valid ) {
        $get_server->(undef);
    }
    elsif ( $domain->pub_suffix && $Pcore::Whois::Data::TLD->{ $domain->pub_suffix } ) {
        $get_server->( $Pcore::Whois::Data::TLD->{ $domain->pub_suffix } );
    }
    elsif ( exists $Pcore::Whois::Data::TLD->{ $domain->tld } ) {
        $get_server->( $Pcore::Whois::Data::TLD->{ $domain->tld } );
    }
    else {

        # get server for the given domain TLD from IANA
        $self->_get_iana_server(
            sub ($iana_server) {
                $self->run_request(
                    $iana_server,
                    $domain->tld,
                    sub ( $domain, $res ) {
                        my $host;

                        if ( $res->is_success && $res->raw->$* =~ /^whois:\s+(.+?)\n/sm ) {
                            $host = $1;
                        }

                        $Pcore::Whois::Data::TLD->{ $domain->tld } = $host;

                        $get_server->($host);

                        return;
                    }
                );

                return;
            }
        );
    }

    return;
}

sub _get_iana_server ( $self, $cb ) {
    state $iana_server;

    if ( !$iana_server ) {
        my $_iana_server = Pcore::Whois::Server->new('whois.iana.org');

        $_iana_server->validate(
            $self, 'fuck',
            sub ($self) {
                die q[IANA Whois server whois.iana.org is not available ] if $self->status != 200;

                $iana_server //= $_iana_server;

                $cb->($iana_server);

                return;
            }
        );
    }
    else {
        $cb->($iana_server);
    }

    return;
}

1;
## -----SOURCE FILTER LOG BEGIN-----
##
## PerlCritic profile "pcore-script" policy violations:
## ┌──────┬──────────────────────┬────────────────────────────────────────────────────────────────────────────────────────────────────────────────┐
## │ Sev. │ Lines                │ Policy                                                                                                         │
## ╞══════╪══════════════════════╪════════════════════════════════════════════════════════════════════════════════════════════════════════════════╡
## │    3 │ 224                  │ ValuesAndExpressions::ProhibitInterpolationOfLiterals - Useless interpolation of literal string                │
## ├──────┼──────────────────────┼────────────────────────────────────────────────────────────────────────────────────────────────────────────────┤
## │    2 │ 224                  │ ValuesAndExpressions::ProhibitNoisyQuotes - Quotes used with a noisy string                                    │
## └──────┴──────────────────────┴────────────────────────────────────────────────────────────────────────────────────────────────────────────────┘
##
## -----SOURCE FILTER LOG END-----
__END__
=pod

=encoding utf8

=head1 NAME

Pcore::Whois

=head1 SYNOPSIS

=head1 DESCRIPTION

=head1 SEE ALSO

L<WHOIS implementation|http://habrahabr.ru/post/165869/>

L<WHOIS Protocol Specification (RFC 3912) |https://tools.ietf.org/html/rfc3912>

=head1 AUTHOR

zdm <zdm@cpan.org>

=head1 CONTRIBUTORS

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by zdm.

=cut
