package Pcore::Whois::Server;

use Pcore -class;
use Pcore::Whois qw[:WHOIS_STATUS];
use Pcore::AE::Handle;
use Pcore::Whois::Server::Response;

has host       => ( is => 'ro', isa => Str, required => 1 );
has req_format => ( is => 'ro', isa => Str, default  => qq[<: QUERY :>$CRLF] );
has is_verisign => ( is => 'rwp', isa => Maybe [Bool], default => undef );
has not_found_md5 => ( is => 'rw', isa => Str );

sub request ( $self, $query, @ ) {
    my $cb = $_[-1];

    my %args = (
        timeout => 30,
        proxy   => undef,
        splice @_, 2, -1,
    );

    my $runtime_proxy;

    my $run = sub {
        Pcore::AE::Handle->new(
            connect                => [ $self->host, 43, 'whois' ],
            connect_timeout        => $args{timeout},
            timeout                => $args{timeout},
            persistent             => 0,
            proxy                  => $runtime_proxy,
            proxy_wait             => 1,
            proxy_ban_id           => $self->host,
            on_proxy_connect_error => sub {
                $self->request( $query, %args, $cb );

                return;
            },
            on_error => sub ( $h, $fatal, $message ) {
                $self->_process_response( $query, undef, $message, \%args, $h, $cb );

                return;
            },
            on_connect => sub ( $h, $host, $port, $retry ) {
                $h->push_write( $self->_get_request_content($query) );

                # read all dat until server close socket
                $h->read_eof(
                    sub ( $h1, $buf_ref, $total_bytes_readed, $error ) {
                        $self->_process_response( $query, $buf_ref, undef, \%args, $h, $cb );

                        return;
                    }
                );

                return;
            }
        );

        return;
    };

    if ( $args{proxy} ) {
        $args{proxy}->get_slot(
            [ $self->host, 43, 'whois' ],
            wait   => 0,
            ban_id => $self->host,
            sub ($proxy) {
                $runtime_proxy = $proxy if $proxy;

                $run->();

                return;
            }
        );
    }
    else {
        $run->();
    }

    return;
}

sub _process_response ( $self, $query, $content_ref, $error, $args, $h, $cb ) {
    if ($error) {
        my $res = Pcore::Whois::Server::Response->new(
            {   query  => $query,
                server => $self,
                status => $WHOIS_STATUS_NETWORK,
                reason => $error,
            }
        );

        $cb->($res);
    }
    else {

        # create whois response object
        my $res = Pcore::Whois::Server::Response->new(
            {   query  => $query,
                server => $self,
                status => $WHOIS_STATUS_OK,
                reason => 'OK',
                raw    => $content_ref,
            }
        );

        if ( !$res->is_success && $res->status == $WHOIS_STATUS_BANNED && $h->{proxy} ) {
            $h->{proxy}->ban( $self->host );

            # repeat request if has proxy and proxy is banned
            $self->request( $query, $args->%*, $cb );
        }

        if ( !defined $self->is_verisign ) {
            if ( $res->is_verisign ) {
                $self->_set_is_verisign(1);

                # repeat request to the verisign server in correct format
                $self->request( $query, $args->%*, $cb );

                return;
            }
            else {
                $self->_set_is_verisign(0);
            }
        }

        $cb->($res);
    }

    return;
}

sub _get_request_content ( $self, $query ) {
    return q[domain ] . $query . $CRLF if $self->is_verisign;

    return $query . q[/e] . $CRLF if $self->host =~ /[.]jp\z/smi;

    return $self->req_format =~ s/<: QUERY :>/$query/smgr;
}

1;
## -----SOURCE FILTER LOG BEGIN-----
##
## PerlCritic profile "pcore-script" policy violations:
## ┌──────┬──────────────────────┬────────────────────────────────────────────────────────────────────────────────────────────────────────────────┐
## │ Sev. │ Lines                │ Policy                                                                                                         │
## ╞══════╪══════════════════════╪════════════════════════════════════════════════════════════════════════════════════════════════════════════════╡
## │    3 │ 83                   │ Subroutines::ProhibitManyArgs - Too many arguments                                                             │
## ├──────┼──────────────────────┼────────────────────────────────────────────────────────────────────────────────────────────────────────────────┤
## │    3 │ 111, 119             │ References::ProhibitDoubleSigils - Double-sigil dereference                                                    │
## └──────┴──────────────────────┴────────────────────────────────────────────────────────────────────────────────────────────────────────────────┘
##
## -----SOURCE FILTER LOG END-----
__END__
=pod

=encoding utf8

=head1 NAME

Pcore::Whois::Server

=head1 SYNOPSIS

=head1 DESCRIPTION

=cut
