package Pcore::Whois v0.2.1;

use Pcore qw[-export -class -const];

has max_threads => ( is => 'ro', isa => PositiveInt, default => 50 );
has timeout     => ( is => 'ro', isa => PositiveInt, default => 30 );
has proxy => ( is => 'ro', isa => InstanceOf ['Pcore::AE::Handle::ProxyPool'] );

has threads => ( is => 'ro', default => 0, init_arg => undef );

no Pcore;

BEGIN {
    our %EXPORT_TAGS = (    #
        WHOIS_STATUS => [qw[$WHOIS_STATUS_OK $WHOIS_STATUS_NO_SERVER $WHOIS_STATUS_NETWORK $WHOIS_STATUS_BANNED]],
    );
}

const our $NOT_AVAIL_DOMAIN => 'cwxtwtqcuduvo78hr0dbrfxpuhv8gmtm';

const our $WHOIS_STATUS_OK        => 200;
const our $WHOIS_STATUS_NO_SERVER => 404;
const our $WHOIS_STATUS_NETWORK   => 500;
const our $WHOIS_STATUS_BANNED    => 501;

use Pcore::Whois::Data;
use Pcore::Whois::Server;
use Pcore::Whois::Server::Response;

sub whois ( $self, $domain, $cb ) {
    state $cb_cache = {};

    state $waiting_threads = [];

    state $on_finish = sub ( $domain, $res ) {
        $self->{threads}--;

        while ( my $cb = shift $cb_cache->{ $domain->name }->@* ) {
            $cb->($res);
        }

        delete $cb_cache->{ $domain->name };

        if ( my $next_domain = shift $waiting_threads->@* ) {
            $self->_run_thread( $next_domain, __SUB__ );
        }

        return;
    };

    $domain = P->host($domain) if !ref $domain;

    my $new_thread = exists $cb_cache->{ $domain->name } ? 0 : 1;

    # cache callback
    push $cb_cache->{ $domain->name }->@*, $cb;

    # can't start new thread
    if ( $new_thread && $self->{threads} >= $self->{max_threads} ) {
        push $waiting_threads->@*, $domain;

        return;
    }

    # thread already started
    return if scalar $cb_cache->{ $domain->name }->@* > 1;

    # start thread
    $self->{threads}++;

    $self->_run_thread( $domain, $on_finish );

    return;
}

sub _run_thread ( $self, $domain, $cb ) {
    $self->_get_server(
        $domain,
        sub ($server) {
            if ( !$server ) {
                my $res = Pcore::Whois::Server::Response->new(
                    {   query  => $domain->name,
                        status => $WHOIS_STATUS_NO_SERVER,
                        reason => 'WHOIS server is not available',
                    }
                );

                $cb->( $domain, $res );
            }
            else {
                $server->request(
                    $domain->name,
                    timeout => $self->timeout,
                    proxy   => $self->{proxy},
                    sub ($res) {
                        $cb->( $domain, $res );

                        return;
                    }
                );
            }

            return;
        }
    );

    return;
}

sub _get_server ( $self, $domain, $cb ) {
    state $cache = {};

    state $req_cache = {};

    $self->_get_server_host(
        $domain,
        sub ($host) {
            if ( !$host ) {
                $cb->(undef);
            }
            else {
                if ( !exists $cache->{$host} ) {

                    # create server object and put into local cache
                    my $args = {    #
                        host => $host,
                    };

                    P->hash->merge( $args, $Pcore::Whois::Data::SERVER->{$host} ) if exists $Pcore::Whois::Data::SERVER->{$host};

                    $cache->{$host} = Pcore::Whois::Server->new($args);
                }

                if ( $cache->{$host}->not_found_md5 ) {
                    $cb->( $cache->{$host} );
                }
                else {
                    my $query = $NOT_AVAIL_DOMAIN . q[.] . $domain->tld;

                    push $req_cache->{$query}->@*, $cb;

                    return if $req_cache->{$query}->@* > 1;

                    # perform query for not available domain
                    $cache->{$host}->request(
                        $query,
                        timeout => $self->timeout,
                        sub ($res) {
                            my $server;

                            if ( $res->is_success ) {
                                $cache->{$host}->not_found_md5( $res->md5 );

                                $server = $cache->{$host};
                            }

                            while ( my $cb = shift $req_cache->{$query}->@* ) {
                                $cb->($server);
                            }

                            delete $req_cache->{$query};

                            return;
                        }
                    );
                }
            }

            return;
        }
    );

    return;
}

sub _get_server_host ( $self, $domain, $cb ) {
    state $iana_cache = {};

    if ( !$domain->tld_is_valid ) {
        $cb->(undef);
    }
    elsif ( $domain->pub_suffix && $Pcore::Whois::Data::TLD->{ $domain->pub_suffix } ) {
        $cb->( $Pcore::Whois::Data::TLD->{ $domain->pub_suffix } );
    }
    elsif ( $Pcore::Whois::Data::TLD->{ $domain->tld } ) {
        $cb->( $Pcore::Whois::Data::TLD->{ $domain->tld } );
    }
    elsif ( exists $iana_cache->{ $domain->tld } ) {
        $cb->( $iana_cache->{ $domain->tld } );
    }
    else {
        state $iana_server = Pcore::Whois::Server->new( { host => 'whois.iana.org' } );

        state $iana_req_cache = {};

        push $iana_req_cache->{ $domain->tld }->@*, $cb;

        return if $iana_req_cache->{ $domain->tld }->@* > 1;

        $iana_server->request(
            $domain->tld,
            timeout => $self->timeout,
            sub ($res) {
                $iana_cache->{ $domain->tld } = undef;

                if ( $res->is_success && $res->raw->$* =~ /^whois:\s+(.+?)\n/sm ) {
                    $iana_cache->{ $domain->tld } = $1;
                }

                while ( my $cb = shift $iana_req_cache->{ $domain->tld }->@* ) {
                    $cb->( $iana_cache->{ $domain->tld } );
                }

                delete $iana_req_cache->{ $domain->tld };

                return;
            },
        );
    }

    return;
}

1;
__END__
=pod

=encoding utf8

=head1 NAME

Pcore::Whois

=head1 SYNOPSIS

=head1 DESCRIPTION

=head1 SEE ALSO

L<WHOIS implementation|http://habrahabr.ru/post/165869/>

L<WHOIS Protocol Specification (RFC 3912) |https://tools.ietf.org/html/rfc3912>

=head1 AUTHOR

zdm <zdm@cpan.org>

=head1 CONTRIBUTORS

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by zdm.

=cut
