# PDF-ISO_32000-p6

The [PDF 32000-1:2008 1.7 Specification](http://www.adobe.com/content/dam/Adobe/en/devnet/acrobat/pdfs/PDF32000_2008.pdf) contains around 380 tables, of which about 280 can be considered data or object definitions.

This module has been used to extract and generate the roles and classes from the PDF
specification for use by PDF::Class.

PDF::Class (at last count) implements roles and classes for about 100 of these objects, most of which
have been derived and/or checked against the roles generated by this module.

This library contains:

- XHTML tables that have been mined from the PDF-32000 1.7 specification.
- JSON data for the tables
- Perl 6 Roles for tables that represent PDF objects:

The roles are named ISO_32000::Table_NNN-Xxxx and contain method stubs and documentation for each entry in the role


## Tables

Data is available for all of the tables in the PDF-32000 1.7 specification:
```
use PDF::ISO_32000;
# Load data about the Document Information dictionary
my %info = PDF::ISO_32000.table: "Table_317-Entries_in_the_document_information_dictionary";
say %info<caption>;             # Table 317 – Entries in the document information dictionary
say %info<head>.join(" | ");    # Key | Type | Value
say %info<rows>[0].join(" | "); # Title | text string | (Optional; PDF 1.1) The document’s title.
```

The `table-index` method returns a list that maps table numbers to table names:

```
say PDF::ISO_32000.table-index[317] # Table 317 - Info_entries
```

The `appendix` method returns a hash index into the Appendix:

```
my $stream-ops = PDF::ISO_32000.appendix<A.1>;
say $stream-ops, # PDF_content_stream_operators
say PDF::ISO_32000.table($stream-ops)<caption>; #  Table A.1 – PDF content stream operators
```

## Roles

Roles are available for tables named `*_entries`, or `*_attributes`.

```
% p6doc ISO_320000:Info
% p6doc ISO_320000:Catalog
```

The roles also contain [method stubs](https://docs.perl6.org/language/objects#Stubs) for the entries that need to be implemented for the role. For example:

```
% cat << EOF > lib/Catalog.pm6
use ISO_32000::Catalog;
class Catalog does ISO_32000::Catalog {
}
EOF
% perl6 -I lib -M Catalog
===SORRY!=== Error while compiling /tmp/lib/Catalog.pm6 (Catalog)
Method 'SpiderInfo' must be implemented by Catalog because it is required by roles: ISO_32000::Catalog.
at lib/Catalog.pm6 (Catalog):1
```
This module contains:

- a copy of the [PDF-32000 specification](src/PDF32000_2008.pdf)
- [JSON Table](resources) extracted from the above
- [generated interface roles](gen/lib/ISO_32000) for building and validating PDF objects
- scripts and Makefiles for regenerating the XML tables and roles

todo: run-time introspection of resources and generated artefacts, e.g.:

```
use PDF::ISO_32000;
PDF::ISO_32000.table[28].xhtml;          # ?? access extracted xhtml by table number ??
PDF::ISO_32000.table[28].json;           # ?? access converted json by table number ??
PDF::ISO_32000.table<Catalog>.interface; # ?? access generated role by table name ??
```

## Scripts in this Distribution

##### `pdf-struct-dump.p6 --password=Xxxx --page=i --max-depth=j --search-tag=Ttt --select=k --obj-num=l --gen-num=m --render --atts --debug src/PDF32000_2008.pdf`

Dumps tagged PDF content as XML.

At the moment just does enough to semi-reliably extract content from the PDF ISO-32000 specification documents. Could evolve into a general purpose tool for mining elements from tagged PDF's. 


## ISO 3200 Roles

The following interface roles have been mined from the ISO-32000 specification

### Roles and Entries


ISO_32000 Reference|Entries
----|-----
[Table 5 – Entries common to all stream dictionaries](gen/lib/ISO_32000/Table_5-Entries_common_to_all_stream_dictionaries.pm6)|/Length /Filter /DecodeParms /F /FFilter /FDecodeParms /DL
[Table 8 – Optional parameters for LZWDecode and FlateDecode filters](gen/lib/ISO_32000/Table_8-Optional_parameters_for_LZWDecode_and_FlateDecode_filters.pm6)|/Predictor /Colors /BitsPerComponent /Columns /EarlyChange
[Table 11 – Optional parameters for the CCITTFaxDecode filter](gen/lib/ISO_32000/Table_11-Optional_parameters_for_the_CCITTFaxDecode_filter.pm6)|/K /EndOfLine /EncodedByteAlign /Columns /Rows /EndOfBlock /BlackIs1 /DamagedRowsBeforeError
[Table 12 – Optional parameter for the JBIG2Decode filter](gen/lib/ISO_32000/Table_12-Optional_parameter_for_the_JBIG2Decode_filter.pm6)|/JBIG2Globals
[Table 13 – Optional parameter for the DCTDecode filter](gen/lib/ISO_32000/Table_13-Optional_parameter_for_the_DCTDecode_filter.pm6)|/ColorTransform
[Table 14 – Optional parameters for Crypt filters](gen/lib/ISO_32000/Table_14-Optional_parameters_for_Crypt_filters.pm6)|/Type /Name
[Table 15 – Entries in the file trailer dictionary](gen/lib/ISO_32000/Table_15-Entries_in_the_file_trailer_dictionary.pm6)|/Size /Prev /Root /Encrypt /Info /ID
[Table 16 – Additional entries specific to an object stream dictionary](gen/lib/ISO_32000/Table_16-Additional_entries_specific_to_an_object_stream_dictionary.pm6)|/Type /N /First /Extends
[Table 17 – Additional entries specific to a cross-reference stream dictionary](gen/lib/ISO_32000/Table_17-Additional_entries_specific_to_a_cross-reference_stream_dictionary.pm6)|/Type /Size /Index /Prev /W
[Table 19 – Additional entries in a hybrid-reference file’s trailer dictionary](gen/lib/ISO_32000/Table_19-Additional_entries_in_a_hybrid-reference_files_trailer_dictionary.pm6)|/XRefStm
[Table 20 – Entries common to all encryption dictionaries](gen/lib/ISO_32000/Table_20-Entries_common_to_all_encryption_dictionaries.pm6)|/Filter /SubFilter /V /Length /CF /StmF /StrF /EFF
[Table 21 – Additional encryption dictionary entries for the standard security handler](gen/lib/ISO_32000/Table_21-Additional_encryption_dictionary_entries_for_the_standard_security_handler.pm6)|/R /O /U /P /EncryptMetadata
[Table 23 – Additional encryption dictionary entries for public-key security handlers](gen/lib/ISO_32000/Table_23-Additional_encryption_dictionary_entries_for_public-key_security_handlers.pm6)|/Recipients /P
[Table 25 – Entries common to all crypt filter dictionaries](gen/lib/ISO_32000/Table_25-Entries_common_to_all_crypt_filter_dictionaries.pm6)|/Type /CFM /AuthEvent /Length
[Table 27 – Additional crypt filter dictionary entries for public-key security handlers](gen/lib/ISO_32000/Table_27-Additional_crypt_filter_dictionary_entries_for_public-key_security_handlers.pm6)|/Recipients /EncryptMetadata
[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.pm6)|/Type /Version /Extensions /Pages /PageLabels /Names /Dests /ViewerPreferences /PageLayout /PageMode /Outlines /Threads /OpenAction /AA /URI /AcroForm /Metadata /StructTreeRoot /MarkInfo /Lang /SpiderInfo /OutputIntents /PieceInfo /OCProperties /Perms /Legal /Requirements /Collection /NeedsRendering
[Table 29 – Required entries in a page tree node](gen/lib/ISO_32000/Table_29-Required_entries_in_a_page_tree_node.pm6)|/Type /Parent /Kids /Count
[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.pm6)|/Type /Parent /LastModified /Resources /MediaBox /CropBox /BleedBox /TrimBox /ArtBox /BoxColorInfo /Contents /Rotate /Group /Thumb /B /Dur /Trans /Annots /AA /Metadata /PieceInfo /StructParents /ID /PZ /SeparationInfo /Tabs /TemplateInstantiated /PresSteps /UserUnit /VP
[Table 31 – Entries in the name dictionary](gen/lib/ISO_32000/Table_31-Entries_in_the_name_dictionary.pm6)|/Dests /AP /JavaScript /Pages /Templates /IDS /URLS /EmbeddedFiles /AlternatePresentations /Renditions
[Table 33 – Entries in a resource dictionary](gen/lib/ISO_32000/Table_33-Entries_in_a_resource_dictionary.pm6)|/ExtGState /ColorSpace /Pattern /Shading /XObject /Font /ProcSet /Properties
[Table 36 – Entries in a name tree node dictionary](gen/lib/ISO_32000/Table_36-Entries_in_a_name_tree_node_dictionary.pm6)|/Kids /Names /Limits
[Table 37 – Entries in a number tree node dictionary](gen/lib/ISO_32000/Table_37-Entries_in_a_number_tree_node_dictionary.pm6)|/Kids /Nums /Limits
[Table 38 – Entries common to all function dictionaries](gen/lib/ISO_32000/Table_38-Entries_common_to_all_function_dictionaries.pm6)|/FunctionType /Domain /Range
[Table 39 – Additional entries specific to a type 0 function dictionary](gen/lib/ISO_32000/Table_39-Additional_entries_specific_to_a_type_0_function_dictionary.pm6)|/Size /BitsPerSample /Order /Encode /Decode
[Table 40 – Additional entries specific to a type 2 function dictionary](gen/lib/ISO_32000/Table_40-Additional_entries_specific_to_a_type_2_function_dictionary.pm6)|/C0 /C1 /N
[Table 41 – Additional entries specific to a type 3 function dictionary](gen/lib/ISO_32000/Table_41-Additional_entries_specific_to_a_type_3_function_dictionary.pm6)|/Functions /Bounds /Encode
[Table 44 – Entries in a file specification dictionary](gen/lib/ISO_32000/Table_44-Entries_in_a_file_specification_dictionary.pm6)|/Type /FS /F /UF /DOS /Mac /Unix /ID /V /EF /RF /Desc /CI
[Table 45 – Additional entries in an embedded file stream dictionary](gen/lib/ISO_32000/Table_45-Additional_entries_in_an_embedded_file_stream_dictionary.pm6)|/Type /Subtype /Params
[Table 46 – Entries in an embedded file parameter dictionary](gen/lib/ISO_32000/Table_46-Entries_in_an_embedded_file_parameter_dictionary.pm6)|/Size /CreationDate /ModDate /Mac /CheckSum
[Table 47 – Entries in a Mac OS file information dictionary](gen/lib/ISO_32000/Table_47-Entries_in_a_Mac_OS_file_information_dictionary.pm6)|/Subtype /Creator /ResFork
[Table 48 – Entries in a collection item dictionary](gen/lib/ISO_32000/Table_48-Entries_in_a_collection_item_dictionary.pm6)|/Type
[Table 49 – Entries in a collection subitem dictionary](gen/lib/ISO_32000/Table_49-Entries_in_a_collection_subitem_dictionary.pm6)|/Type /D /P
[Table 50 – Entries in a developer extensions dictionary](gen/lib/ISO_32000/Table_50-Entries_in_a_developer_extensions_dictionary.pm6)|/Type /BaseVersion /ExtensionLevel
[Table 52 – Device-Independent Graphics State Parameters](gen/lib/ISO_32000/Table_52-Device-Independent_Graphics_State_Parameters.pm6)|/CTM /color
[Table 53 – Device-Dependent Graphics State Parameters](gen/lib/ISO_32000/Table_53-Device-Dependent_Graphics_State_Parameters.pm6)|/overprint /transfer /halftone /flatness /smoothness
[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.pm6)|/Type /LW /LC /LJ /ML /D /RI /OP /op /OPM /Font /BG /BG2 /UCR /UCR2 /TR /TR2 /HT /FL /SM /SA /BM /SMask /CA /ca /AIS /TK
[Table 63 – Entries in a CalGray Colour Space Dictionary](gen/lib/ISO_32000/Table_63-Entries_in_a_CalGray_Colour_Space_Dictionary.pm6)|/WhitePoint /BlackPoint /Gamma
[Table 64 – Entries in a CalRGB Colour Space Dictionary](gen/lib/ISO_32000/Table_64-Entries_in_a_CalRGB_Colour_Space_Dictionary.pm6)|/WhitePoint /BlackPoint /Gamma /Matrix
[Table 65 – Entries in a Lab Colour Space Dictionary](gen/lib/ISO_32000/Table_65-Entries_in_a_Lab_Colour_Space_Dictionary.pm6)|/WhitePoint /BlackPoint /Range
[Table 66 – Additional Entries Specific to an ICC Profile Stream Dictionary](gen/lib/ISO_32000/Table_66-Additional_Entries_Specific_to_an_ICC_Profile_Stream_Dictionary.pm6)|/N /Alternate /Range /Metadata
[Table 71 – Entries in a DeviceN Colour Space Attributes Dictionary](gen/lib/ISO_32000/Table_71-Entries_in_a_DeviceN_Colour_Space_Attributes_Dictionary.pm6)|/Subtype /Colorants /Process /MixingHints
[Table 72 – Entries in a DeviceN Process Dictionary](gen/lib/ISO_32000/Table_72-Entries_in_a_DeviceN_Process_Dictionary.pm6)|/ColorSpace /Components
[Table 73 – Entries in a DeviceN Mixing Hints Dictionary](gen/lib/ISO_32000/Table_73-Entries_in_a_DeviceN_Mixing_Hints_Dictionary.pm6)|/Solidities /PrintingOrder /DotGain
[Table 75 – Additional Entries Specific to a Type 1 Pattern Dictionary](gen/lib/ISO_32000/Table_75-Additional_Entries_Specific_to_a_Type_1_Pattern_Dictionary.pm6)|/Type /PatternType /PaintType /TilingType /BBox /XStep /YStep /Resources /Matrix
[Table 76 – Entries in a Type 2 Pattern Dictionary](gen/lib/ISO_32000/Table_76-Entries_in_a_Type_2_Pattern_Dictionary.pm6)|/Type /PatternType /Shading /Matrix /ExtGState
[Table 78 – Entries Common to All Shading Dictionaries](gen/lib/ISO_32000/Table_78-Entries_Common_to_All_Shading_Dictionaries.pm6)|/ShadingType /ColorSpace /Background /BBox /AntiAlias
[Table 79 – Additional Entries Specific to a Type 1 Shading Dictionary](gen/lib/ISO_32000/Table_79-Additional_Entries_Specific_to_a_Type_1_Shading_Dictionary.pm6)|/Domain /Matrix /Function
[Table 80 – Additional Entries Specific to a Type 2 Shading Dictionary](gen/lib/ISO_32000/Table_80-Additional_Entries_Specific_to_a_Type_2_Shading_Dictionary.pm6)|/Coords /Domain /Function /Extend
[Table 81 – Additional Entries Specific to a Type 3 Shading Dictionary](gen/lib/ISO_32000/Table_81-Additional_Entries_Specific_to_a_Type_3_Shading_Dictionary.pm6)|/Coords /Domain /Function /Extend
[Table 82 – Additional Entries Specific to a Type 4 Shading Dictionary](gen/lib/ISO_32000/Table_82-Additional_Entries_Specific_to_a_Type_4_Shading_Dictionary.pm6)|/BitsPerCoordinate /BitsPerComponent /BitsPerFlag /Decode /Function
[Table 83 – Additional Entries Specific to a Type 5 Shading Dictionary](gen/lib/ISO_32000/Table_83-Additional_Entries_Specific_to_a_Type_5_Shading_Dictionary.pm6)|/BitsPerCoordinate /BitsPerComponent /VerticesPerRow /Decode /Function
[Table 84 – Additional Entries Specific to a Type 6 Shading Dictionary](gen/lib/ISO_32000/Table_84-Additional_Entries_Specific_to_a_Type_6_Shading_Dictionary.pm6)|/BitsPerCoordinate /BitsPerComponent /BitsPerFlag /Decode /Function
[Table 88 – Additional Entries Specific to a PostScript XObject Dictionary](gen/lib/ISO_32000/Table_88-Additional_Entries_Specific_to_a_PostScript_XObject_Dictionary.pm6)|/Type /Subtype /Level1
[Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.pm6)|/Type /Subtype /Width /Height /ColorSpace /BitsPerComponent /Intent /ImageMask /Mask /Decode /Interpolate /Alternates /SMask /SMaskInData /Name /StructParent /ID /OPI /Metadata /OC
[Table 91 – Entries in an Alternate Image Dictionary](gen/lib/ISO_32000/Table_91-Entries_in_an_Alternate_Image_Dictionary.pm6)|/Image /DefaultForPrinting /OC
[Table 95 – Additional Entries Specific to a Type 1 Form Dictionary](gen/lib/ISO_32000/Table_95-Additional_Entries_Specific_to_a_Type_1_Form_Dictionary.pm6)|/Type /Subtype /FormType /BBox /Matrix /Resources /Group /Ref /Metadata /PieceInfo /LastModified /StructParent /StructParents /OPI /OC /Name
[Table 96 – Entries Common to all Group Attributes Dictionaries](gen/lib/ISO_32000/Table_96-Entries_Common_to_all_Group_Attributes_Dictionaries.pm6)|/Type /S
[Table 97 – Entries in a Reference Dictionary](gen/lib/ISO_32000/Table_97-Entries_in_a_Reference_Dictionary.pm6)|/F /Page /ID
[Table 98 – Entries in an Optional Content Group Dictionary](gen/lib/ISO_32000/Table_98-Entries_in_an_Optional_Content_Group_Dictionary.pm6)|/Type /Name /Intent /Usage
[Table 99 – Entries in an Optional Content Membership Dictionary](gen/lib/ISO_32000/Table_99-Entries_in_an_Optional_Content_Membership_Dictionary.pm6)|/Type /OCGs /P /VE
[Table 100 – Entries in the Optional Content Properties Dictionary](gen/lib/ISO_32000/Table_100-Entries_in_the_Optional_Content_Properties_Dictionary.pm6)|/OCGs /D /Configs
[Table 101 – Entries in an Optional Content Configuration Dictionary](gen/lib/ISO_32000/Table_101-Entries_in_an_Optional_Content_Configuration_Dictionary.pm6)|/Name /Creator /BaseState /ON /OFF /Intent /AS /Order /ListMode /RBGroups /Locked
[Table 102 – Entries in an Optional Content Usage Dictionary](gen/lib/ISO_32000/Table_102-Entries_in_an_Optional_Content_Usage_Dictionary.pm6)|/CreatorInfo /Language /Export /Zoom /Print /View /User /PageElement
[Table 103 – Entries in a Usage Application Dictionary](gen/lib/ISO_32000/Table_103-Entries_in_a_Usage_Application_Dictionary.pm6)|/Event /OCGs /Category
[Table 111 – Entries in a Type 1 font dictionary](gen/lib/ISO_32000/Table_111-Entries_in_a_Type_1_font_dictionary.pm6)|/Type /Subtype /Name /BaseFont /FirstChar /LastChar /Widths /FontDescriptor /Encoding /ToUnicode
[Table 112 – Entries in a Type 3 font dictionary](gen/lib/ISO_32000/Table_112-Entries_in_a_Type_3_font_dictionary.pm6)|/Type /Subtype /Name /FontBBox /FontMatrix /CharProcs /Encoding /FirstChar /LastChar /Widths /FontDescriptor /Resources /ToUnicode
[Table 114 – Entries in an encoding dictionary](gen/lib/ISO_32000/Table_114-Entries_in_an_encoding_dictionary.pm6)|/Type /BaseEncoding /Differences
[Table 116 – Entries in a CIDSystemInfo dictionary](gen/lib/ISO_32000/Table_116-Entries_in_a_CIDSystemInfo_dictionary.pm6)|/Registry /Ordering /Supplement
[Table 117 – Entries in a CIDFont dictionary](gen/lib/ISO_32000/Table_117-Entries_in_a_CIDFont_dictionary.pm6)|/Type /Subtype /BaseFont /CIDSystemInfo /FontDescriptor /DW /W /DW2 /W2 /CIDToGIDMap
[Table 120 – Additional entries in a CMap stream dictionary](gen/lib/ISO_32000/Table_120-Additional_entries_in_a_CMap_stream_dictionary.pm6)|/Type /CMapName /CIDSystemInfo /WMode /UseCMap
[Table 121 – Entries in a Type 0 font dictionary](gen/lib/ISO_32000/Table_121-Entries_in_a_Type_0_font_dictionary.pm6)|/Type /Subtype /BaseFont /Encoding /DescendantFonts /ToUnicode
[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.pm6)|/Type /FontName /FontFamily /FontStretch /FontWeight /Flags /FontBBox /ItalicAngle /Ascent /Descent /Leading /CapHeight /XHeight /StemV /StemH /AvgWidth /MaxWidth /MissingWidth /FontFile /FontFile2 /FontFile3 /CharSet
[Table 124 – Additional font descriptor entries for CIDFonts](gen/lib/ISO_32000/Table_124-Additional_font_descriptor_entries_for_CIDFonts.pm6)|/Style /Lang /FD /CIDSet
[Table 126 – Embedded font organization for various font types](gen/lib/ISO_32000/Table_126-Embedded_font_organization_for_various_font_types.pm6)|/FontFile /FontFile2 /FontFile3
[Table 127 – Additional entries in an embedded font stream dictionary](gen/lib/ISO_32000/Table_127-Additional_entries_in_an_embedded_font_stream_dictionary.pm6)|/Length1 /Length2 /Length3 /Subtype /Metadata
[Table 130 – Entries in a type 1 halftone dictionary](gen/lib/ISO_32000/Table_130-Entries_in_a_type_1_halftone_dictionary.pm6)|/Type /HalftoneType /HalftoneName /Frequency /Angle /SpotFunction /AccurateScreens /TransferFunction
[Table 131 – Additional entries specific to a type 6 halftone dictionary](gen/lib/ISO_32000/Table_131-Additional_entries_specific_to_a_type_6_halftone_dictionary.pm6)|/Type /HalftoneType /HalftoneName /Width /Height /TransferFunction
[Table 132 – Additional entries specific to a type 10 halftone dictionary](gen/lib/ISO_32000/Table_132-Additional_entries_specific_to_a_type_10_halftone_dictionary.pm6)|/Type /HalftoneType /HalftoneName /Xsquare /Ysquare /TransferFunction
[Table 133 – Additional entries specific to a type 16 halftone dictionary](gen/lib/ISO_32000/Table_133-Additional_entries_specific_to_a_type_16_halftone_dictionary.pm6)|/Type /HalftoneType /HalftoneName /Width /Height /Width2 /Height2 /TransferFunction
[Table 134 – Entries in a type 5 halftone dictionary](gen/lib/ISO_32000/Table_134-Entries_in_a_type_5_halftone_dictionary.pm6)|/Type /HalftoneType /HalftoneName /Default
[Table 144 – Entries in a soft-mask dictionary](gen/lib/ISO_32000/Table_144-Entries_in_a_soft-mask_dictionary.pm6)|/Type /S /G /BC /TR
[Table 146 – Additional entry in a soft-mask image dictionary](gen/lib/ISO_32000/Table_146-Additional_entry_in_a_soft-mask_image_dictionary.pm6)|/Matte
[Table 147 – Additional entries specific to a transparency group attributes dictionary](gen/lib/ISO_32000/Table_147-Additional_entries_specific_to_a_transparency_group_attributes_dictionary.pm6)|/S /CS /I /K
[Table 150 – Entries in a viewer preferences dictionary](gen/lib/ISO_32000/Table_150-Entries_in_a_viewer_preferences_dictionary.pm6)|/HideToolbar /HideMenubar /HideWindowUI /FitWindow /CenterWindow /DisplayDocTitle /NonFullScreenPageMode /Direction /ViewArea /ViewClip /PrintArea /PrintClip /PrintScaling /Duplex /PickTrayByPDFSize /PrintPageRange /NumCopies
[Table 152 – Entries in the outline dictionary](gen/lib/ISO_32000/Table_152-Entries_in_the_outline_dictionary.pm6)|/Type /First /Last /Count
[Table 153 – Entries in an outline item dictionary](gen/lib/ISO_32000/Table_153-Entries_in_an_outline_item_dictionary.pm6)|/Title /Parent /Prev /Next /First /Last /Count /Dest /A /SE /C /F
[Table 155 – Entries in a collection dictionary](gen/lib/ISO_32000/Table_155-Entries_in_a_collection_dictionary.pm6)|/Type /Schema /D /View /Sort
[Table 156 – Entries in a collection schema dictionary](gen/lib/ISO_32000/Table_156-Entries_in_a_collection_schema_dictionary.pm6)|/Type
[Table 157 – Entries in a collection field dictionary](gen/lib/ISO_32000/Table_157-Entries_in_a_collection_field_dictionary.pm6)|/Type /Subtype /N /O /V /E
[Table 158 – Entries in a collection sort dictionary](gen/lib/ISO_32000/Table_158-Entries_in_a_collection_sort_dictionary.pm6)|/Type /S /A
[Table 159 – Entries in a page label dictionary](gen/lib/ISO_32000/Table_159-Entries_in_a_page_label_dictionary.pm6)|/Type /S /P /St
[Table 160 – Entries in a thread dictionary](gen/lib/ISO_32000/Table_160-Entries_in_a_thread_dictionary.pm6)|/Type /F /I
[Table 161 – Entries in a bead dictionary](gen/lib/ISO_32000/Table_161-Entries_in_a_bead_dictionary.pm6)|/Type /T /N /V /P /R
[Table 162 – Entries in a transition dictionary](gen/lib/ISO_32000/Table_162-Entries_in_a_transition_dictionary.pm6)|/Type /S /D /Dm /M /Di /SS /B
[Table 163 – Entries in a navigation node dictionary](gen/lib/ISO_32000/Table_163-Entries_in_a_navigation_node_dictionary.pm6)|/Type /NA /PA /Next /Prev /Dur
[Table 164 – Entries common to all annotation dictionaries](gen/lib/ISO_32000/Table_164-Entries_common_to_all_annotation_dictionaries.pm6)|/Type /Subtype /Rect /Contents /P /NM /M /F /AP /AS /Border /C /StructParent /OC
[Table 166 – Entries in a border style dictionary](gen/lib/ISO_32000/Table_166-Entries_in_a_border_style_dictionary.pm6)|/Type /W /S /D
[Table 167 – Entries in a border effect dictionary](gen/lib/ISO_32000/Table_167-Entries_in_a_border_effect_dictionary.pm6)|/S /I
[Table 168 – Entries in an appearance dictionary](gen/lib/ISO_32000/Table_168-Entries_in_an_appearance_dictionary.pm6)|/N /R /D
[Table 170 – Additional entries specific to markup annotations](gen/lib/ISO_32000/Table_170-Additional_entries_specific_to_markup_annotations.pm6)|/T /Popup /CA /RC /CreationDate /IRT /Subj /RT /IT /ExData
[Table 172 – Additional entries specific to a text annotation](gen/lib/ISO_32000/Table_172-Additional_entries_specific_to_a_text_annotation.pm6)|/Subtype /Open /Name /State /StateModel
[Table 173 – Additional entries specific to a link annotation](gen/lib/ISO_32000/Table_173-Additional_entries_specific_to_a_link_annotation.pm6)|/Subtype /A /Dest /H /PA /QuadPoints /BS
[Table 174 – Additional entries specific to a free text annotation](gen/lib/ISO_32000/Table_174-Additional_entries_specific_to_a_free_text_annotation.pm6)|/Subtype /DA /Q /RC /DS /CL /IT /BE /RD /BS /LE
[Table 175 – Additional entries specific to a line annotation](gen/lib/ISO_32000/Table_175-Additional_entries_specific_to_a_line_annotation.pm6)|/Subtype /L /BS /LE /IC /LL /LLE /Cap /IT /LLO /CP /Measure /CO
[Table 177 – Additional entries specific to a square or circle annotation](gen/lib/ISO_32000/Table_177-Additional_entries_specific_to_a_square_or_circle_annotation.pm6)|/Subtype /BS /IC /BE /RD
[Table 178 – Additional entries specific to a polygon or polyline annotation](gen/lib/ISO_32000/Table_178-Additional_entries_specific_to_a_polygon_or_polyline_annotation.pm6)|/Subtype /Vertices /LE /BS /IC /BE /IT /Measure
[Table 179 – Additional entries specific to text markup annotations](gen/lib/ISO_32000/Table_179-Additional_entries_specific_to_text_markup_annotations.pm6)|/Subtype /QuadPoints
[Table 180 – Additional entries specific to a caret annotation](gen/lib/ISO_32000/Table_180-Additional_entries_specific_to_a_caret_annotation.pm6)|/Subtype /RD /Sy
[Table 181 – Additional entries specific to a rubber stamp annotation](gen/lib/ISO_32000/Table_181-Additional_entries_specific_to_a_rubber_stamp_annotation.pm6)|/Subtype /Name
[Table 182 – Additional entries specific to an ink annotation](gen/lib/ISO_32000/Table_182-Additional_entries_specific_to_an_ink_annotation.pm6)|/Subtype /InkList /BS
[Table 183 – Additional entries specific to a pop-up annotation](gen/lib/ISO_32000/Table_183-Additional_entries_specific_to_a_pop-up_annotation.pm6)|/Subtype /Parent /Open
[Table 184 – Additional entries specific to a file attachment annotation](gen/lib/ISO_32000/Table_184-Additional_entries_specific_to_a_file_attachment_annotation.pm6)|/Subtype /FS /Name
[Table 185 – Additional entries specific to a sound annotation](gen/lib/ISO_32000/Table_185-Additional_entries_specific_to_a_sound_annotation.pm6)|/Subtype /Sound /Name
[Table 186 – Additional entries specific to a movie annotation](gen/lib/ISO_32000/Table_186-Additional_entries_specific_to_a_movie_annotation.pm6)|/Subtype /T /Movie /A
[Table 187 – Additional entries specific to a screen annotation](gen/lib/ISO_32000/Table_187-Additional_entries_specific_to_a_screen_annotation.pm6)|/Subtype /T /MK /A /AA
[Table 188 – Additional entries specific to a widget annotation](gen/lib/ISO_32000/Table_188-Additional_entries_specific_to_a_widget_annotation.pm6)|/Subtype /H /MK /A /AA /BS /Parent
[Table 189 – Entries in an appearance characteristics dictionary](gen/lib/ISO_32000/Table_189-Entries_in_an_appearance_characteristics_dictionary.pm6)|/R /BC /BG /CA /RC /AC /I /RI /IX /IF /TP
[Table 190 – Additional entries specific to a watermark annotation](gen/lib/ISO_32000/Table_190-Additional_entries_specific_to_a_watermark_annotation.pm6)|/Subtype /FixedPrint
[Table 191 – Entries in a fixed print dictionary](gen/lib/ISO_32000/Table_191-Entries_in_a_fixed_print_dictionary.pm6)|/Type /Matrix /H /V
[Table 192 – Additional entries specific to a redaction annotation](gen/lib/ISO_32000/Table_192-Additional_entries_specific_to_a_redaction_annotation.pm6)|/Subtype /QuadPoints /IC /RO /OverlayText /Repeat /DA /Q
[Table 193 – Entries common to all action dictionaries](gen/lib/ISO_32000/Table_193-Entries_common_to_all_action_dictionaries.pm6)|/Type /S /Next
[Table 194 – Entries in an annotation’s additional-actions dictionary](gen/lib/ISO_32000/Table_194-Entries_in_an_annotations_additional-actions_dictionary.pm6)|/E /X /D /U /Fo /Bl /PO /PC /PV /PI
[Table 195 – Entries in a page object’s additional-actions dictionary](gen/lib/ISO_32000/Table_195-Entries_in_a_page_objects_additional-actions_dictionary.pm6)|/O /C
[Table 196 – Entries in a form field’s additional-actions dictionary](gen/lib/ISO_32000/Table_196-Entries_in_a_form_fields_additional-actions_dictionary.pm6)|/K /F /V /C
[Table 197 – Entries in the document catalog’s additional-actions dictionary](gen/lib/ISO_32000/Table_197-Entries_in_the_document_catalogs_additional-actions_dictionary.pm6)|/WC /WS /DS /WP /DP
[Table 199 – Additional entries specific to a go-to action](gen/lib/ISO_32000/Table_199-Additional_entries_specific_to_a_go-to_action.pm6)|/S /D
[Table 200 – Additional entries specific to a remote go-to action](gen/lib/ISO_32000/Table_200-Additional_entries_specific_to_a_remote_go-to_action.pm6)|/S /F /D /NewWindow
[Table 201 – Additional entries specific to an embedded go-to action](gen/lib/ISO_32000/Table_201-Additional_entries_specific_to_an_embedded_go-to_action.pm6)|/S /F /D /NewWindow /T
[Table 202 – Entries specific to a target dictionary](gen/lib/ISO_32000/Table_202-Entries_specific_to_a_target_dictionary.pm6)|/R /N /P /A /T
[Table 203 – Additional entries specific to a launch action](gen/lib/ISO_32000/Table_203-Additional_entries_specific_to_a_launch_action.pm6)|/S /F /Win /Mac /Unix /NewWindow
[Table 204 – Entries in a Windows launch parameter dictionary](gen/lib/ISO_32000/Table_204-Entries_in_a_Windows_launch_parameter_dictionary.pm6)|/F /D /O /P
[Table 205 – Additional entries specific to a thread action](gen/lib/ISO_32000/Table_205-Additional_entries_specific_to_a_thread_action.pm6)|/S /F /D /B
[Table 206 – Additional entries specific to a URI action](gen/lib/ISO_32000/Table_206-Additional_entries_specific_to_a_URI_action.pm6)|/S /URI /IsMap
[Table 207 – Entry in a URI dictionary](gen/lib/ISO_32000/Table_207-Entry_in_a_URI_dictionary.pm6)|/Base
[Table 208 – Additional entries specific to a sound action](gen/lib/ISO_32000/Table_208-Additional_entries_specific_to_a_sound_action.pm6)|/S /Sound /Volume /Synchronous /Repeat /Mix
[Table 209 – Additional entries specific to a movie action](gen/lib/ISO_32000/Table_209-Additional_entries_specific_to_a_movie_action.pm6)|/S /Annotation /T /Operation
[Table 210 – Additional entries specific to a hide action](gen/lib/ISO_32000/Table_210-Additional_entries_specific_to_a_hide_action.pm6)|/S /T /H
[Table 212 – Additional entries specific to named actions](gen/lib/ISO_32000/Table_212-Additional_entries_specific_to_named_actions.pm6)|/S /N
[Table 213 – Additional entries specific to a set-OCG-state action](gen/lib/ISO_32000/Table_213-Additional_entries_specific_to_a_set-OCG-state_action.pm6)|/S /State /PreserveRB
[Table 214 – Additional entries specific to a rendition action](gen/lib/ISO_32000/Table_214-Additional_entries_specific_to_a_rendition_action.pm6)|/S /R /AN /OP /JS
[Table 215 – Additional entries specific to a transition action](gen/lib/ISO_32000/Table_215-Additional_entries_specific_to_a_transition_action.pm6)|/S /Trans
[Table 216 – Additional entries specific to a go-to-3D-view action](gen/lib/ISO_32000/Table_216-Additional_entries_specific_to_a_go-to-ThreeD-view_action.pm6)|/S /TA /V
[Table 217 – Additional entries specific to a JavaScript action](gen/lib/ISO_32000/Table_217-Additional_entries_specific_to_a_JavaScript_action.pm6)|/S /JS
[Table 218 – Entries in the interactive form dictionary](gen/lib/ISO_32000/Table_218-Entries_in_the_interactive_form_dictionary.pm6)|/Fields /NeedAppearances /SigFlags /CO /DR /DA /Q /XFA
[Table 220 – Entries common to all field dictionaries](gen/lib/ISO_32000/Table_220-Entries_common_to_all_field_dictionaries.pm6)|/FT /Parent /Kids /T /TU /TM /Ff /V /DV /AA
[Table 222 – Additional entries common to all fields containing variable text](gen/lib/ISO_32000/Table_222-Additional_entries_common_to_all_fields_containing_variable_text.pm6)|/DA /Q /DS /RV
[Table 227 – Additional entry specific to check box and radio button fields](gen/lib/ISO_32000/Table_227-Additional_entry_specific_to_check_box_and_radio_button_fields.pm6)|/Opt
[Table 229 – Additional entry specific to a text field](gen/lib/ISO_32000/Table_229-Additional_entry_specific_to_a_text_field.pm6)|/MaxLen
[Table 231 – Additional entries specific to a choice field](gen/lib/ISO_32000/Table_231-Additional_entries_specific_to_a_choice_field.pm6)|/Opt /TI /I
[Table 232 – Additional entries specific to a signature field](gen/lib/ISO_32000/Table_232-Additional_entries_specific_to_a_signature_field.pm6)|/Lock /SV
[Table 233 – Entries in a signature field lock dictionary](gen/lib/ISO_32000/Table_233-Entries_in_a_signature_field_lock_dictionary.pm6)|/Type /Action /Fields
[Table 234 – Entries in a signature field seed value dictionary](gen/lib/ISO_32000/Table_234-Entries_in_a_signature_field_seed_value_dictionary.pm6)|/Type /Ff /Filter /SubFilter /DigestMethod /V /Cert /Reasons /MDP /TimeStamp /LegalAttestation /AddRevInfo
[Table 235 – Entries in a certificate seed value dictionary](gen/lib/ISO_32000/Table_235-Entries_in_a_certificate_seed_value_dictionary.pm6)|/Type /Ff /Subject /SubjectDN /KeyUsage /Issuer /OID /URL /URLType
[Table 236 – Additional entries specific to a submit-form action](gen/lib/ISO_32000/Table_236-Additional_entries_specific_to_a_submit-form_action.pm6)|/S /F /Fields /Flags
[Table 238 – Additional entries specific to a reset-form action](gen/lib/ISO_32000/Table_238-Additional_entries_specific_to_a_reset-form_action.pm6)|/S /Fields /Flags
[Table 240 – Additional entries specific to an import-data action](gen/lib/ISO_32000/Table_240-Additional_entries_specific_to_an_import-data_action.pm6)|/S /F
[Table 241 – Entry in the FDF trailer dictionary](gen/lib/ISO_32000/Table_241-Entry_in_the_FDF_trailer_dictionary.pm6)|/Root
[Table 242 – Entries in the FDF catalog dictionary](gen/lib/ISO_32000/Table_242-Entries_in_the_FDF_catalog_dictionary.pm6)|/Version /FDF
[Table 243 – Entries in the FDF dictionary](gen/lib/ISO_32000/Table_243-Entries_in_the_FDF_dictionary.pm6)|/F /ID /Fields /Status /Pages /Encoding /Annots /Differences /Target /EmbeddedFDFs /JavaScript
[Table 244 – Additional entry in an embedded file stream dictionary for an encrypted FDF file](gen/lib/ISO_32000/Table_244-Additional_entry_in_an_embedded_file_stream_dictionary_for_an_encrypted_FDF_file.pm6)|/EncryptionRevision
[Table 245 – Entries in the JavaScript dictionary](gen/lib/ISO_32000/Table_245-Entries_in_the_JavaScript_dictionary.pm6)|/Before /After /AfterPermsReady /Doc
[Table 246 – Entries in an FDF field dictionary](gen/lib/ISO_32000/Table_246-Entries_in_an_FDF_field_dictionary.pm6)|/Kids /T /V /Ff /SetFf /ClrFf /F /SetF /ClrF /AP /APRef /IF /Opt /A /AA /RV
[Table 247 – Entries in an icon fit dictionary](gen/lib/ISO_32000/Table_247-Entries_in_an_icon_fit_dictionary.pm6)|/SW /S /A /FB
[Table 248 – Entries in an FDF page dictionary](gen/lib/ISO_32000/Table_248-Entries_in_an_FDF_page_dictionary.pm6)|/Templates /Info
[Table 249 – Entries in an FDF template dictionary](gen/lib/ISO_32000/Table_249-Entries_in_an_FDF_template_dictionary.pm6)|/TRef /Fields /Rename
[Table 250 – Entries in an FDF named page reference dictionary](gen/lib/ISO_32000/Table_250-Entries_in_an_FDF_named_page_reference_dictionary.pm6)|/Name /F
[Table 251 – Additional entry for annotation dictionaries in an FDF file](gen/lib/ISO_32000/Table_251-Additional_entry_for_annotation_dictionaries_in_an_FDF_file.pm6)|/Page
[Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.pm6)|/Type /Filter /SubFilter /Contents /Cert /ByteRange /Reference /Changes /Name /M /Location /Reason /ContactInfo /R /V /Prop_Build /Prop_AuthTime /Prop_AuthType
[Table 253 – Entries in a signature reference dictionary](gen/lib/ISO_32000/Table_253-Entries_in_a_signature_reference_dictionary.pm6)|/Type /TransformMethod /TransformParams /Data /DigestMethod
[Table 254 – Entries in the DocMDP transform parameters dictionary](gen/lib/ISO_32000/Table_254-Entries_in_the_DocMDP_transform_parameters_dictionary.pm6)|/Type /P /V
[Table 255 – Entries in the UR transform parameters dictionary](gen/lib/ISO_32000/Table_255-Entries_in_the_UR_transform_parameters_dictionary.pm6)|/Type /Document /Msg /V /Annots /Form /Signature /EF /P
[Table 256 – Entries in the FieldMDP transform parameters dictionary](gen/lib/ISO_32000/Table_256-Entries_in_the_FieldMDP_transform_parameters_dictionary.pm6)|/Type /Action /Fields /V
[Table 258 – Entries in a permissions dictionary](gen/lib/ISO_32000/Table_258-Entries_in_a_permissions_dictionary.pm6)|/DocMDP /UR3
[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.pm6)|/JavaScriptActions /LaunchActions /URIActions /MovieActions /SoundActions /HideAnnotationActions /GoToRemoteActions /AlternateImages /ExternalStreams /TrueTypeFonts /ExternalRefXobjects /ExternalOPIdicts /NonEmbeddedFonts /DevDepGS_OP /DevDepGS_HT /DevDepGS_TR /DevDepGS_UCR /DevDepGS_BG /DevDepGS_FL /Annotations /OptionalContent /Attestation
[Table 260 – Entries in a viewport dictionary](gen/lib/ISO_32000/Table_260-Entries_in_a_viewport_dictionary.pm6)|/Type /BBox /Name /Measure
[Table 261 – Entries in a measure dictionary](gen/lib/ISO_32000/Table_261-Entries_in_a_measure_dictionary.pm6)|/Type /Subtype
[Table 262 – Additional entries in a rectilinear measure dictionary](gen/lib/ISO_32000/Table_262-Additional_entries_in_a_rectilinear_measure_dictionary.pm6)|/R /X /Y /D /A /T /S /O /CYX
[Table 263 – Entries in a number format dictionary](gen/lib/ISO_32000/Table_263-Entries_in_a_number_format_dictionary.pm6)|/Type /U /C /F /D /FD /RT /RD /PS /SS /O
[Table 264 – Entries common to all requirement dictionaries](gen/lib/ISO_32000/Table_264-Entries_common_to_all_requirement_dictionaries.pm6)|/Type /S /RH
[Table 265 – Entries in a requirement handler dictionary](gen/lib/ISO_32000/Table_265-Entries_in_a_requirement_handler_dictionary.pm6)|/Type /S /Script
[Table 266 – Entries common to all rendition dictionaries](gen/lib/ISO_32000/Table_266-Entries_common_to_all_rendition_dictionaries.pm6)|/Type /S /N /MH /BE
[Table 267 – Entries in a rendition MH/BE dictionary](gen/lib/ISO_32000/Table_267-Entries_in_a_rendition_MH-BE_dictionary.pm6)|/C
[Table 268 – Entries in a media criteria dictionary](gen/lib/ISO_32000/Table_268-Entries_in_a_media_criteria_dictionary.pm6)|/Type /A /C /O /S /R /D /Z /V /P /L
[Table 269 – Entries in a minimum bit depth dictionary](gen/lib/ISO_32000/Table_269-Entries_in_a_minimum_bit_depth_dictionary.pm6)|/Type /V /M
[Table 270 – Entries in a minimum screen size dictionary](gen/lib/ISO_32000/Table_270-Entries_in_a_minimum_screen_size_dictionary.pm6)|/Type /V /M
[Table 271 – Additional entries in a media rendition dictionary](gen/lib/ISO_32000/Table_271-Additional_entries_in_a_media_rendition_dictionary.pm6)|/C /P /SP
[Table 272 – Additional entries specific to a selector rendition dictionary](gen/lib/ISO_32000/Table_272-Additional_entries_specific_to_a_selector_rendition_dictionary.pm6)|/R
[Table 273 – Entries common to all media clip dictionaries](gen/lib/ISO_32000/Table_273-Entries_common_to_all_media_clip_dictionaries.pm6)|/Type /S /N
[Table 274 – Additional entries in a media clip data dictionary](gen/lib/ISO_32000/Table_274-Additional_entries_in_a_media_clip_data_dictionary.pm6)|/D /CT /P /Alt /PL /MH /BE
[Table 275 – Entries in a media permissions dictionary](gen/lib/ISO_32000/Table_275-Entries_in_a_media_permissions_dictionary.pm6)|/Type /TF
[Table 276 – Entries in a media clip data MH/BE dictionary](gen/lib/ISO_32000/Table_276-Entries_in_a_media_clip_data_MH-BE_dictionary.pm6)|/BU
[Table 277 – Additional entries in a media clip section dictionary](gen/lib/ISO_32000/Table_277-Additional_entries_in_a_media_clip_section_dictionary.pm6)|/D /Alt /MH /BE
[Table 278 – Entries in a media clip section MH/BE dictionary](gen/lib/ISO_32000/Table_278-Entries_in_a_media_clip_section_MH-BE_dictionary.pm6)|/B /E
[Table 279 – Entries in a media play parameters dictionary](gen/lib/ISO_32000/Table_279-Entries_in_a_media_play_parameters_dictionary.pm6)|/Type /PL /MH /BE
[Table 280 – Entries in a media play parameters MH/BE dictionary](gen/lib/ISO_32000/Table_280-Entries_in_a_media_play_parameters_MH-BE_dictionary.pm6)|/V /C /F /D /A /RC
[Table 281 – Entries in a media duration dictionary](gen/lib/ISO_32000/Table_281-Entries_in_a_media_duration_dictionary.pm6)|/Type /S /T
[Table 282 – Entries in a media screen parameters dictionary](gen/lib/ISO_32000/Table_282-Entries_in_a_media_screen_parameters_dictionary.pm6)|/Type /MH /BE
[Table 283 – Entries in a media screen parameters MH/BE dictionary](gen/lib/ISO_32000/Table_283-Entries_in_a_media_screen_parameters_MH-BE_dictionary.pm6)|/W /B /O /M /F
[Table 284 – Entries in a floating window parameters dictionary](gen/lib/ISO_32000/Table_284-Entries_in_a_floating_window_parameters_dictionary.pm6)|/Type /D /RT /P /O /T /UC /R /TT
[Table 285 – Entries common to all media offset dictionaries](gen/lib/ISO_32000/Table_285-Entries_common_to_all_media_offset_dictionaries.pm6)|/Type /S
[Table 286 – Additional entries in a media offset time dictionary](gen/lib/ISO_32000/Table_286-Additional_entries_in_a_media_offset_time_dictionary.pm6)|/T
[Table 287 – Additional entries in a media offset frame dictionary](gen/lib/ISO_32000/Table_287-Additional_entries_in_a_media_offset_frame_dictionary.pm6)|/F
[Table 288 – Additional entries in a media offset marker dictionary](gen/lib/ISO_32000/Table_288-Additional_entries_in_a_media_offset_marker_dictionary.pm6)|/M
[Table 289 – Entries in a timespan dictionary](gen/lib/ISO_32000/Table_289-Entries_in_a_timespan_dictionary.pm6)|/Type /S /V
[Table 290 – Entries in a media players dictionary](gen/lib/ISO_32000/Table_290-Entries_in_a_media_players_dictionary.pm6)|/Type /MU /A /NU
[Table 291 – Entries in a media player info dictionary](gen/lib/ISO_32000/Table_291-Entries_in_a_media_player_info_dictionary.pm6)|/Type /PID /MH /BE
[Table 292 – Entries in a software identifier dictionary](gen/lib/ISO_32000/Table_292-Entries_in_a_software_identifier_dictionary.pm6)|/Type /U /L /LI /H /HI /OS
[Table 294 – Additional entries specific to a sound object](gen/lib/ISO_32000/Table_294-Additional_entries_specific_to_a_sound_object.pm6)|/Type /R /C /B /E /CO /CP
[Table 295 – Entries in a movie dictionary](gen/lib/ISO_32000/Table_295-Entries_in_a_movie_dictionary.pm6)|/F /Aspect /Rotate /Poster
[Table 296 – Entries in a movie activation dictionary](gen/lib/ISO_32000/Table_296-Entries_in_a_movie_activation_dictionary.pm6)|/Start /Duration /Rate /Volume /ShowControls /Mode /Synchronous /FWScale /FWPosition
[Table 297 – Entries in a slideshow dictionary](gen/lib/ISO_32000/Table_297-Entries_in_a_slideshow_dictionary.pm6)|/Type /Subtype /Resources /StartResource
[Table 298 – Additional entries specific to a 3D annotation](gen/lib/ISO_32000/Table_298-Additional_entries_specific_to_a_ThreeD_annotation.pm6)|/Subtype /3DD /3DV /3DA /3DI /3DB
[Table 299 – Entries in a 3D activation dictionary](gen/lib/ISO_32000/Table_299-Entries_in_a_ThreeD_activation_dictionary.pm6)|/A /AIS /D /DIS /TB /NP
[Table 300 – Entries in a 3D stream dictionary](gen/lib/ISO_32000/Table_300-Entries_in_a_ThreeD_stream_dictionary.pm6)|/Type /Subtype /VA /DV /Resources /OnInstantiate /AN
[Table 301 – Entries in an 3D animation style dictionary](gen/lib/ISO_32000/Table_301-Entries_in_an_ThreeD_animation_style_dictionary.pm6)|/Type /Subtype /PC /TM
[Table 303 – Entries in a 3D reference dictionary](gen/lib/ISO_32000/Table_303-Entries_in_a_ThreeD_reference_dictionary.pm6)|/Type /3DD
[Table 304 – Entries in a 3D view dictionary](gen/lib/ISO_32000/Table_304-Entries_in_a_ThreeD_view_dictionary.pm6)|/Type /XN /IN /MS /C2W /U3DPath /CO /P /O /BG /RM /LS /SA /NA /NR
[Table 305 – Entries in a projection dictionary](gen/lib/ISO_32000/Table_305-Entries_in_a_projection_dictionary.pm6)|/Subtype /CS /F /N /FOV /PS /OS /OB
[Table 306 – Entries in a 3D background dictionary](gen/lib/ISO_32000/Table_306-Entries_in_a_ThreeD_background_dictionary.pm6)|/Type /Subtype /CS /C /EA
[Table 307 – Entries in a render mode dictionary](gen/lib/ISO_32000/Table_307-Entries_in_a_render_mode_dictionary.pm6)|/Type /Subtype /AC /FC /O /CV
[Table 309 – Entries in a 3D lighting scheme dictionary](gen/lib/ISO_32000/Table_309-Entries_in_a_ThreeD_lighting_scheme_dictionary.pm6)|/Type /Subtype
[Table 311 – Entries in a 3D cross section dictionary](gen/lib/ISO_32000/Table_311-Entries_in_a_ThreeD_cross_section_dictionary.pm6)|/Type /C /O /PO /PC /IV /IC
[Table 312 – Entries in a 3D node dictionary](gen/lib/ISO_32000/Table_312-Entries_in_a_ThreeD_node_dictionary.pm6)|/Type /N /O /V /M
[Table 313 – Entries in an external data dictionary used to markup 3D annotations](gen/lib/ISO_32000/Table_313-Entries_in_an_external_data_dictionary_used_to_markup_ThreeD_annotations.pm6)|/Type /Subtype /MD5 /3DA /3DV
[Table 315 – Additional entries in a metadata stream dictionary](gen/lib/ISO_32000/Table_315-Additional_entries_in_a_metadata_stream_dictionary.pm6)|/Type /Subtype
[Table 316 – Additional entry for components having metadata](gen/lib/ISO_32000/Table_316-Additional_entry_for_components_having_metadata.pm6)|/Metadata
[Table 317 – Entries in the document information dictionary](gen/lib/ISO_32000/Table_317-Entries_in_the_document_information_dictionary.pm6)|/Title /Author /Subject /Keywords /Creator /Producer /CreationDate /ModDate /Trapped
[Table 318 – Entries in a page-piece dictionary](gen/lib/ISO_32000/Table_318-Entries_in_a_page-piece_dictionary.pm6)|
[Table 319 – Entries in an data dictionary](gen/lib/ISO_32000/Table_319-Entries_in_an_data_dictionary.pm6)|/LastModified /Private
[Table 321 – Entries in the mark information dictionary](gen/lib/ISO_32000/Table_321-Entries_in_the_mark_information_dictionary.pm6)|/Marked /UserProperties /Suspects
[Table 322 – Entries in the structure tree root](gen/lib/ISO_32000/Table_322-Entries_in_the_structure_tree_root.pm6)|/Type /K /IDTree /ParentTree /ParentTreeNextKey /RoleMap /ClassMap
[Table 323 – Entries in a structure element dictionary](gen/lib/ISO_32000/Table_323-Entries_in_a_structure_element_dictionary.pm6)|/Type /S /P /ID /Pg /K /A /C /R /T /Lang /Alt /E /ActualText
[Table 324 – Entries in a marked-content reference dictionary](gen/lib/ISO_32000/Table_324-Entries_in_a_marked-content_reference_dictionary.pm6)|/Type /Pg /Stm /StmOwn /MCID
[Table 325 – Entries in an object reference dictionary](gen/lib/ISO_32000/Table_325-Entries_in_an_object_reference_dictionary.pm6)|/Type /Pg /Obj
[Table 326 – Additional dictionary entries for structure element access](gen/lib/ISO_32000/Table_326-Additional_dictionary_entries_for_structure_element_access.pm6)|/StructParent /StructParents
[Table 327 – Entry common to all attribute object dictionaries](gen/lib/ISO_32000/Table_327-Entry_common_to_all_attribute_object_dictionaries.pm6)|/O
[Table 328 – Additional entries in an attribute object dictionary for user properties](gen/lib/ISO_32000/Table_328-Additional_entries_in_an_attribute_object_dictionary_for_user_properties.pm6)|/O /P
[Table 329 – Entries in a user property dictionary](gen/lib/ISO_32000/Table_329-Entries_in_a_user_property_dictionary.pm6)|/N /V /F /H
[Table 330 – Property list entries for artifacts](gen/lib/ISO_32000/Table_330-Property_list_entries_for_artifacts.pm6)|/Type /BBox /Attached /Subtype
[Table 343 – Standard layout attributes common to all standard structure types](gen/lib/ISO_32000/Table_343-Standard_layout_attributes_common_to_all_standard_structure_types.pm6)|/Placement /WritingMode /BackgroundColor /BorderColor /BorderStyle /BorderThickness /Padding /Color
[Table 344 – Additional standard layout attributes specific to block-level structure elements](gen/lib/ISO_32000/Table_344-Additional_standard_layout_attributes_specific_to_block-level_structure_elements.pm6)|/SpaceBefore /SpaceAfter /StartIndent /EndIndent /TextIndent /TextAlign /BBox /Width /Height /BlockAlign /InlineAlign /TBorderStyle /TPadding
[Table 345 – Standard layout attributes specific to inline-level structure elements](gen/lib/ISO_32000/Table_345-Standard_layout_attributes_specific_to_inline-level_structure_elements.pm6)|/BaselineShift /LineHeight /TextDecorationColor /TextDecorationThickness /TextDecorationType /RubyAlign /RubyPosition /GlyphOrientationVertical
[Table 346 – Standard column attributes](gen/lib/ISO_32000/Table_346-Standard_column_attributes.pm6)|/ColumnCount /ColumnGap /ColumnWidths
[Table 347 – Standard list attribute](gen/lib/ISO_32000/Table_347-Standard_list_attribute.pm6)|/ListNumbering
[Table 348 – PrintField attributes](gen/lib/ISO_32000/Table_348-PrintField_attributes.pm6)|/Role /checked /Desc
[Table 349 – Standard table attributes](gen/lib/ISO_32000/Table_349-Standard_table_attributes.pm6)|/RowSpan /ColSpan /Headers /Scope /Summary
[Table 350 – Entries in the Web Capture information dictionary](gen/lib/ISO_32000/Table_350-Entries_in_the_Web_Capture_information_dictionary.pm6)|/V /C
[Table 352 – Entries common to all Web Capture content sets](gen/lib/ISO_32000/Table_352-Entries_common_to_all_Web_Capture_content_sets.pm6)|/Type /S /ID /O /SI /CT /TS
[Table 353 – Additional entries specific to a Web Capture page set](gen/lib/ISO_32000/Table_353-Additional_entries_specific_to_a_Web_Capture_page_set.pm6)|/S /T /TID
[Table 354 – Additional entries specific to a Web Capture image set](gen/lib/ISO_32000/Table_354-Additional_entries_specific_to_a_Web_Capture_image_set.pm6)|/S /R
[Table 355 – Entries in a source information dictionary](gen/lib/ISO_32000/Table_355-Entries_in_a_source_information_dictionary.pm6)|/AU /TS /E /S /C
[Table 356 – Entries in a URL alias dictionary](gen/lib/ISO_32000/Table_356-Entries_in_a_URL_alias_dictionary.pm6)|/U /C
[Table 357 – Entries in a Web Capture command dictionary](gen/lib/ISO_32000/Table_357-Entries_in_a_Web_Capture_command_dictionary.pm6)|/URL /L /F /P /CT /H /S
[Table 359 – Entries in a Web Capture command settings dictionary](gen/lib/ISO_32000/Table_359-Entries_in_a_Web_Capture_command_settings_dictionary.pm6)|/G /C
[Table 360 – Entries in a box colour information dictionary](gen/lib/ISO_32000/Table_360-Entries_in_a_box_colour_information_dictionary.pm6)|/CropBox /BleedBox /TrimBox /ArtBox
[Table 361 – Entries in a box style dictionary](gen/lib/ISO_32000/Table_361-Entries_in_a_box_style_dictionary.pm6)|/C /W /S /D
[Table 362 – Additional entries specific to a printer’s mark annotation](gen/lib/ISO_32000/Table_362-Additional_entries_specific_to_a_printers_mark_annotation.pm6)|/Subtype /MN
[Table 363 – Additional entries specific to a printer’s mark form dictionary](gen/lib/ISO_32000/Table_363-Additional_entries_specific_to_a_printers_mark_form_dictionary.pm6)|/MarkStyle /Colorants
[Table 364 – Entries in a separation dictionary](gen/lib/ISO_32000/Table_364-Entries_in_a_separation_dictionary.pm6)|/Pages /DeviceColorant /ColorSpace
[Table 365 – Entries in an output intent dictionary](gen/lib/ISO_32000/Table_365-Entries_in_an_output_intent_dictionary.pm6)|/Type /S /OutputCondition /OutputConditionIdentifier /RegistryName /Info /DestOutputProfile
[Table 366 – Additional entries specific to a trap network annotation](gen/lib/ISO_32000/Table_366-Additional_entries_specific_to_a_trap_network_annotation.pm6)|/Subtype /LastModified /Version /AnnotStates /FontFauxing
[Table 367 – Additional entries specific to a trap network appearance stream](gen/lib/ISO_32000/Table_367-Additional_entries_specific_to_a_trap_network_appearance_stream.pm6)|/PCM /SeparationColorNames /TrapRegions /TrapStyles
[Table 368 – Entry in an OPI version dictionary](gen/lib/ISO_32000/Table_368-Entry_in_an_OPI_version_dictionary.pm6)|
[Table F. 1 – Entries in the linearization parameter dictionary](gen/lib/ISO_32000/Table_F1-Entries_in_the_linearization_parameter_dictionary.pm6)|/Linearized /L /H /O /E /N /T /P

## Entry to role mappings

Entry|ISO_32000 Roles
----|-----
/3DA|[Table 298 – Additional entries specific to a 3D annotation](gen/lib/ISO_32000/Table_298-Additional_entries_specific_to_a_ThreeD_annotation.pm6) [Table 313 – Entries in an external data dictionary used to markup 3D annotations](gen/lib/ISO_32000/Table_313-Entries_in_an_external_data_dictionary_used_to_markup_ThreeD_annotations.pm6)
/3DB|[Table 298 – Additional entries specific to a 3D annotation](gen/lib/ISO_32000/Table_298-Additional_entries_specific_to_a_ThreeD_annotation.pm6)
/3DD|[Table 298 – Additional entries specific to a 3D annotation](gen/lib/ISO_32000/Table_298-Additional_entries_specific_to_a_ThreeD_annotation.pm6) [Table 303 – Entries in a 3D reference dictionary](gen/lib/ISO_32000/Table_303-Entries_in_a_ThreeD_reference_dictionary.pm6)
/3DI|[Table 298 – Additional entries specific to a 3D annotation](gen/lib/ISO_32000/Table_298-Additional_entries_specific_to_a_ThreeD_annotation.pm6)
/3DV|[Table 298 – Additional entries specific to a 3D annotation](gen/lib/ISO_32000/Table_298-Additional_entries_specific_to_a_ThreeD_annotation.pm6) [Table 313 – Entries in an external data dictionary used to markup 3D annotations](gen/lib/ISO_32000/Table_313-Entries_in_an_external_data_dictionary_used_to_markup_ThreeD_annotations.pm6)
/A|[Table 153 – Entries in an outline item dictionary](gen/lib/ISO_32000/Table_153-Entries_in_an_outline_item_dictionary.pm6) [Table 158 – Entries in a collection sort dictionary](gen/lib/ISO_32000/Table_158-Entries_in_a_collection_sort_dictionary.pm6) [Table 173 – Additional entries specific to a link annotation](gen/lib/ISO_32000/Table_173-Additional_entries_specific_to_a_link_annotation.pm6) [Table 186 – Additional entries specific to a movie annotation](gen/lib/ISO_32000/Table_186-Additional_entries_specific_to_a_movie_annotation.pm6) [Table 187 – Additional entries specific to a screen annotation](gen/lib/ISO_32000/Table_187-Additional_entries_specific_to_a_screen_annotation.pm6) [Table 188 – Additional entries specific to a widget annotation](gen/lib/ISO_32000/Table_188-Additional_entries_specific_to_a_widget_annotation.pm6) [Table 202 – Entries specific to a target dictionary](gen/lib/ISO_32000/Table_202-Entries_specific_to_a_target_dictionary.pm6) [Table 246 – Entries in an FDF field dictionary](gen/lib/ISO_32000/Table_246-Entries_in_an_FDF_field_dictionary.pm6) [Table 247 – Entries in an icon fit dictionary](gen/lib/ISO_32000/Table_247-Entries_in_an_icon_fit_dictionary.pm6) [Table 262 – Additional entries in a rectilinear measure dictionary](gen/lib/ISO_32000/Table_262-Additional_entries_in_a_rectilinear_measure_dictionary.pm6) [Table 268 – Entries in a media criteria dictionary](gen/lib/ISO_32000/Table_268-Entries_in_a_media_criteria_dictionary.pm6) [Table 280 – Entries in a media play parameters MH/BE dictionary](gen/lib/ISO_32000/Table_280-Entries_in_a_media_play_parameters_MH-BE_dictionary.pm6) [Table 290 – Entries in a media players dictionary](gen/lib/ISO_32000/Table_290-Entries_in_a_media_players_dictionary.pm6) [Table 299 – Entries in a 3D activation dictionary](gen/lib/ISO_32000/Table_299-Entries_in_a_ThreeD_activation_dictionary.pm6) [Table 323 – Entries in a structure element dictionary](gen/lib/ISO_32000/Table_323-Entries_in_a_structure_element_dictionary.pm6)
/AA|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.pm6) [Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.pm6) [Table 187 – Additional entries specific to a screen annotation](gen/lib/ISO_32000/Table_187-Additional_entries_specific_to_a_screen_annotation.pm6) [Table 188 – Additional entries specific to a widget annotation](gen/lib/ISO_32000/Table_188-Additional_entries_specific_to_a_widget_annotation.pm6) [Table 220 – Entries common to all field dictionaries](gen/lib/ISO_32000/Table_220-Entries_common_to_all_field_dictionaries.pm6) [Table 246 – Entries in an FDF field dictionary](gen/lib/ISO_32000/Table_246-Entries_in_an_FDF_field_dictionary.pm6)
/AC|[Table 189 – Entries in an appearance characteristics dictionary](gen/lib/ISO_32000/Table_189-Entries_in_an_appearance_characteristics_dictionary.pm6) [Table 307 – Entries in a render mode dictionary](gen/lib/ISO_32000/Table_307-Entries_in_a_render_mode_dictionary.pm6)
/AIS|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.pm6) [Table 299 – Entries in a 3D activation dictionary](gen/lib/ISO_32000/Table_299-Entries_in_a_ThreeD_activation_dictionary.pm6)
/AN|[Table 214 – Additional entries specific to a rendition action](gen/lib/ISO_32000/Table_214-Additional_entries_specific_to_a_rendition_action.pm6) [Table 300 – Entries in a 3D stream dictionary](gen/lib/ISO_32000/Table_300-Entries_in_a_ThreeD_stream_dictionary.pm6)
/AP|[Table 31 – Entries in the name dictionary](gen/lib/ISO_32000/Table_31-Entries_in_the_name_dictionary.pm6) [Table 164 – Entries common to all annotation dictionaries](gen/lib/ISO_32000/Table_164-Entries_common_to_all_annotation_dictionaries.pm6) [Table 246 – Entries in an FDF field dictionary](gen/lib/ISO_32000/Table_246-Entries_in_an_FDF_field_dictionary.pm6)
/APRef|[Table 246 – Entries in an FDF field dictionary](gen/lib/ISO_32000/Table_246-Entries_in_an_FDF_field_dictionary.pm6)
/AS|[Table 101 – Entries in an Optional Content Configuration Dictionary](gen/lib/ISO_32000/Table_101-Entries_in_an_Optional_Content_Configuration_Dictionary.pm6) [Table 164 – Entries common to all annotation dictionaries](gen/lib/ISO_32000/Table_164-Entries_common_to_all_annotation_dictionaries.pm6)
/AU|[Table 355 – Entries in a source information dictionary](gen/lib/ISO_32000/Table_355-Entries_in_a_source_information_dictionary.pm6)
/AccurateScreens|[Table 130 – Entries in a type 1 halftone dictionary](gen/lib/ISO_32000/Table_130-Entries_in_a_type_1_halftone_dictionary.pm6)
/AcroForm|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.pm6)
/Action|[Table 233 – Entries in a signature field lock dictionary](gen/lib/ISO_32000/Table_233-Entries_in_a_signature_field_lock_dictionary.pm6) [Table 256 – Entries in the FieldMDP transform parameters dictionary](gen/lib/ISO_32000/Table_256-Entries_in_the_FieldMDP_transform_parameters_dictionary.pm6)
/ActualText|[Table 323 – Entries in a structure element dictionary](gen/lib/ISO_32000/Table_323-Entries_in_a_structure_element_dictionary.pm6)
/AddRevInfo|[Table 234 – Entries in a signature field seed value dictionary](gen/lib/ISO_32000/Table_234-Entries_in_a_signature_field_seed_value_dictionary.pm6)
/After|[Table 245 – Entries in the JavaScript dictionary](gen/lib/ISO_32000/Table_245-Entries_in_the_JavaScript_dictionary.pm6)
/AfterPermsReady|[Table 245 – Entries in the JavaScript dictionary](gen/lib/ISO_32000/Table_245-Entries_in_the_JavaScript_dictionary.pm6)
/Alt|[Table 274 – Additional entries in a media clip data dictionary](gen/lib/ISO_32000/Table_274-Additional_entries_in_a_media_clip_data_dictionary.pm6) [Table 277 – Additional entries in a media clip section dictionary](gen/lib/ISO_32000/Table_277-Additional_entries_in_a_media_clip_section_dictionary.pm6) [Table 323 – Entries in a structure element dictionary](gen/lib/ISO_32000/Table_323-Entries_in_a_structure_element_dictionary.pm6)
/Alternate|[Table 66 – Additional Entries Specific to an ICC Profile Stream Dictionary](gen/lib/ISO_32000/Table_66-Additional_Entries_Specific_to_an_ICC_Profile_Stream_Dictionary.pm6)
/AlternateImages|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.pm6)
/AlternatePresentations|[Table 31 – Entries in the name dictionary](gen/lib/ISO_32000/Table_31-Entries_in_the_name_dictionary.pm6)
/Alternates|[Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.pm6)
/Angle|[Table 130 – Entries in a type 1 halftone dictionary](gen/lib/ISO_32000/Table_130-Entries_in_a_type_1_halftone_dictionary.pm6)
/AnnotStates|[Table 366 – Additional entries specific to a trap network annotation](gen/lib/ISO_32000/Table_366-Additional_entries_specific_to_a_trap_network_annotation.pm6)
/Annotation|[Table 209 – Additional entries specific to a movie action](gen/lib/ISO_32000/Table_209-Additional_entries_specific_to_a_movie_action.pm6)
/Annotations|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.pm6)
/Annots|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.pm6) [Table 243 – Entries in the FDF dictionary](gen/lib/ISO_32000/Table_243-Entries_in_the_FDF_dictionary.pm6) [Table 255 – Entries in the UR transform parameters dictionary](gen/lib/ISO_32000/Table_255-Entries_in_the_UR_transform_parameters_dictionary.pm6)
/AntiAlias|[Table 78 – Entries Common to All Shading Dictionaries](gen/lib/ISO_32000/Table_78-Entries_Common_to_All_Shading_Dictionaries.pm6)
/ArtBox|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.pm6) [Table 360 – Entries in a box colour information dictionary](gen/lib/ISO_32000/Table_360-Entries_in_a_box_colour_information_dictionary.pm6)
/Ascent|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.pm6)
/Aspect|[Table 295 – Entries in a movie dictionary](gen/lib/ISO_32000/Table_295-Entries_in_a_movie_dictionary.pm6)
/Attached|[Table 330 – Property list entries for artifacts](gen/lib/ISO_32000/Table_330-Property_list_entries_for_artifacts.pm6)
/Attestation|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.pm6)
/AuthEvent|[Table 25 – Entries common to all crypt filter dictionaries](gen/lib/ISO_32000/Table_25-Entries_common_to_all_crypt_filter_dictionaries.pm6)
/Author|[Table 317 – Entries in the document information dictionary](gen/lib/ISO_32000/Table_317-Entries_in_the_document_information_dictionary.pm6)
/AvgWidth|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.pm6)
/B|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.pm6) [Table 162 – Entries in a transition dictionary](gen/lib/ISO_32000/Table_162-Entries_in_a_transition_dictionary.pm6) [Table 205 – Additional entries specific to a thread action](gen/lib/ISO_32000/Table_205-Additional_entries_specific_to_a_thread_action.pm6) [Table 278 – Entries in a media clip section MH/BE dictionary](gen/lib/ISO_32000/Table_278-Entries_in_a_media_clip_section_MH-BE_dictionary.pm6) [Table 283 – Entries in a media screen parameters MH/BE dictionary](gen/lib/ISO_32000/Table_283-Entries_in_a_media_screen_parameters_MH-BE_dictionary.pm6) [Table 294 – Additional entries specific to a sound object](gen/lib/ISO_32000/Table_294-Additional_entries_specific_to_a_sound_object.pm6)
/BBox|[Table 75 – Additional Entries Specific to a Type 1 Pattern Dictionary](gen/lib/ISO_32000/Table_75-Additional_Entries_Specific_to_a_Type_1_Pattern_Dictionary.pm6) [Table 78 – Entries Common to All Shading Dictionaries](gen/lib/ISO_32000/Table_78-Entries_Common_to_All_Shading_Dictionaries.pm6) [Table 95 – Additional Entries Specific to a Type 1 Form Dictionary](gen/lib/ISO_32000/Table_95-Additional_Entries_Specific_to_a_Type_1_Form_Dictionary.pm6) [Table 260 – Entries in a viewport dictionary](gen/lib/ISO_32000/Table_260-Entries_in_a_viewport_dictionary.pm6) [Table 330 – Property list entries for artifacts](gen/lib/ISO_32000/Table_330-Property_list_entries_for_artifacts.pm6) [Table 344 – Additional standard layout attributes specific to block-level structure elements](gen/lib/ISO_32000/Table_344-Additional_standard_layout_attributes_specific_to_block-level_structure_elements.pm6)
/BC|[Table 144 – Entries in a soft-mask dictionary](gen/lib/ISO_32000/Table_144-Entries_in_a_soft-mask_dictionary.pm6) [Table 189 – Entries in an appearance characteristics dictionary](gen/lib/ISO_32000/Table_189-Entries_in_an_appearance_characteristics_dictionary.pm6)
/BE|[Table 174 – Additional entries specific to a free text annotation](gen/lib/ISO_32000/Table_174-Additional_entries_specific_to_a_free_text_annotation.pm6) [Table 177 – Additional entries specific to a square or circle annotation](gen/lib/ISO_32000/Table_177-Additional_entries_specific_to_a_square_or_circle_annotation.pm6) [Table 178 – Additional entries specific to a polygon or polyline annotation](gen/lib/ISO_32000/Table_178-Additional_entries_specific_to_a_polygon_or_polyline_annotation.pm6) [Table 266 – Entries common to all rendition dictionaries](gen/lib/ISO_32000/Table_266-Entries_common_to_all_rendition_dictionaries.pm6) [Table 274 – Additional entries in a media clip data dictionary](gen/lib/ISO_32000/Table_274-Additional_entries_in_a_media_clip_data_dictionary.pm6) [Table 277 – Additional entries in a media clip section dictionary](gen/lib/ISO_32000/Table_277-Additional_entries_in_a_media_clip_section_dictionary.pm6) [Table 279 – Entries in a media play parameters dictionary](gen/lib/ISO_32000/Table_279-Entries_in_a_media_play_parameters_dictionary.pm6) [Table 282 – Entries in a media screen parameters dictionary](gen/lib/ISO_32000/Table_282-Entries_in_a_media_screen_parameters_dictionary.pm6) [Table 291 – Entries in a media player info dictionary](gen/lib/ISO_32000/Table_291-Entries_in_a_media_player_info_dictionary.pm6)
/BG|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.pm6) [Table 189 – Entries in an appearance characteristics dictionary](gen/lib/ISO_32000/Table_189-Entries_in_an_appearance_characteristics_dictionary.pm6) [Table 304 – Entries in a 3D view dictionary](gen/lib/ISO_32000/Table_304-Entries_in_a_ThreeD_view_dictionary.pm6)
/BG2|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.pm6)
/BM|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.pm6)
/BS|[Table 173 – Additional entries specific to a link annotation](gen/lib/ISO_32000/Table_173-Additional_entries_specific_to_a_link_annotation.pm6) [Table 174 – Additional entries specific to a free text annotation](gen/lib/ISO_32000/Table_174-Additional_entries_specific_to_a_free_text_annotation.pm6) [Table 175 – Additional entries specific to a line annotation](gen/lib/ISO_32000/Table_175-Additional_entries_specific_to_a_line_annotation.pm6) [Table 177 – Additional entries specific to a square or circle annotation](gen/lib/ISO_32000/Table_177-Additional_entries_specific_to_a_square_or_circle_annotation.pm6) [Table 178 – Additional entries specific to a polygon or polyline annotation](gen/lib/ISO_32000/Table_178-Additional_entries_specific_to_a_polygon_or_polyline_annotation.pm6) [Table 182 – Additional entries specific to an ink annotation](gen/lib/ISO_32000/Table_182-Additional_entries_specific_to_an_ink_annotation.pm6) [Table 188 – Additional entries specific to a widget annotation](gen/lib/ISO_32000/Table_188-Additional_entries_specific_to_a_widget_annotation.pm6)
/BU|[Table 276 – Entries in a media clip data MH/BE dictionary](gen/lib/ISO_32000/Table_276-Entries_in_a_media_clip_data_MH-BE_dictionary.pm6)
/Background|[Table 78 – Entries Common to All Shading Dictionaries](gen/lib/ISO_32000/Table_78-Entries_Common_to_All_Shading_Dictionaries.pm6)
/BackgroundColor|[Table 343 – Standard layout attributes common to all standard structure types](gen/lib/ISO_32000/Table_343-Standard_layout_attributes_common_to_all_standard_structure_types.pm6)
/Base|[Table 207 – Entry in a URI dictionary](gen/lib/ISO_32000/Table_207-Entry_in_a_URI_dictionary.pm6)
/BaseEncoding|[Table 114 – Entries in an encoding dictionary](gen/lib/ISO_32000/Table_114-Entries_in_an_encoding_dictionary.pm6)
/BaseFont|[Table 111 – Entries in a Type 1 font dictionary](gen/lib/ISO_32000/Table_111-Entries_in_a_Type_1_font_dictionary.pm6) [Table 117 – Entries in a CIDFont dictionary](gen/lib/ISO_32000/Table_117-Entries_in_a_CIDFont_dictionary.pm6) [Table 121 – Entries in a Type 0 font dictionary](gen/lib/ISO_32000/Table_121-Entries_in_a_Type_0_font_dictionary.pm6)
/BaseState|[Table 101 – Entries in an Optional Content Configuration Dictionary](gen/lib/ISO_32000/Table_101-Entries_in_an_Optional_Content_Configuration_Dictionary.pm6)
/BaseVersion|[Table 50 – Entries in a developer extensions dictionary](gen/lib/ISO_32000/Table_50-Entries_in_a_developer_extensions_dictionary.pm6)
/BaselineShift|[Table 345 – Standard layout attributes specific to inline-level structure elements](gen/lib/ISO_32000/Table_345-Standard_layout_attributes_specific_to_inline-level_structure_elements.pm6)
/Before|[Table 245 – Entries in the JavaScript dictionary](gen/lib/ISO_32000/Table_245-Entries_in_the_JavaScript_dictionary.pm6)
/BitsPerComponent|[Table 8 – Optional parameters for LZWDecode and FlateDecode filters](gen/lib/ISO_32000/Table_8-Optional_parameters_for_LZWDecode_and_FlateDecode_filters.pm6) [Table 82 – Additional Entries Specific to a Type 4 Shading Dictionary](gen/lib/ISO_32000/Table_82-Additional_Entries_Specific_to_a_Type_4_Shading_Dictionary.pm6) [Table 83 – Additional Entries Specific to a Type 5 Shading Dictionary](gen/lib/ISO_32000/Table_83-Additional_Entries_Specific_to_a_Type_5_Shading_Dictionary.pm6) [Table 84 – Additional Entries Specific to a Type 6 Shading Dictionary](gen/lib/ISO_32000/Table_84-Additional_Entries_Specific_to_a_Type_6_Shading_Dictionary.pm6) [Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.pm6)
/BitsPerCoordinate|[Table 82 – Additional Entries Specific to a Type 4 Shading Dictionary](gen/lib/ISO_32000/Table_82-Additional_Entries_Specific_to_a_Type_4_Shading_Dictionary.pm6) [Table 83 – Additional Entries Specific to a Type 5 Shading Dictionary](gen/lib/ISO_32000/Table_83-Additional_Entries_Specific_to_a_Type_5_Shading_Dictionary.pm6) [Table 84 – Additional Entries Specific to a Type 6 Shading Dictionary](gen/lib/ISO_32000/Table_84-Additional_Entries_Specific_to_a_Type_6_Shading_Dictionary.pm6)
/BitsPerFlag|[Table 82 – Additional Entries Specific to a Type 4 Shading Dictionary](gen/lib/ISO_32000/Table_82-Additional_Entries_Specific_to_a_Type_4_Shading_Dictionary.pm6) [Table 84 – Additional Entries Specific to a Type 6 Shading Dictionary](gen/lib/ISO_32000/Table_84-Additional_Entries_Specific_to_a_Type_6_Shading_Dictionary.pm6)
/BitsPerSample|[Table 39 – Additional entries specific to a type 0 function dictionary](gen/lib/ISO_32000/Table_39-Additional_entries_specific_to_a_type_0_function_dictionary.pm6)
/Bl|[Table 194 – Entries in an annotation’s additional-actions dictionary](gen/lib/ISO_32000/Table_194-Entries_in_an_annotations_additional-actions_dictionary.pm6)
/BlackIs1|[Table 11 – Optional parameters for the CCITTFaxDecode filter](gen/lib/ISO_32000/Table_11-Optional_parameters_for_the_CCITTFaxDecode_filter.pm6)
/BlackPoint|[Table 63 – Entries in a CalGray Colour Space Dictionary](gen/lib/ISO_32000/Table_63-Entries_in_a_CalGray_Colour_Space_Dictionary.pm6) [Table 64 – Entries in a CalRGB Colour Space Dictionary](gen/lib/ISO_32000/Table_64-Entries_in_a_CalRGB_Colour_Space_Dictionary.pm6) [Table 65 – Entries in a Lab Colour Space Dictionary](gen/lib/ISO_32000/Table_65-Entries_in_a_Lab_Colour_Space_Dictionary.pm6)
/BleedBox|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.pm6) [Table 360 – Entries in a box colour information dictionary](gen/lib/ISO_32000/Table_360-Entries_in_a_box_colour_information_dictionary.pm6)
/BlockAlign|[Table 344 – Additional standard layout attributes specific to block-level structure elements](gen/lib/ISO_32000/Table_344-Additional_standard_layout_attributes_specific_to_block-level_structure_elements.pm6)
/Border|[Table 164 – Entries common to all annotation dictionaries](gen/lib/ISO_32000/Table_164-Entries_common_to_all_annotation_dictionaries.pm6)
/BorderColor|[Table 343 – Standard layout attributes common to all standard structure types](gen/lib/ISO_32000/Table_343-Standard_layout_attributes_common_to_all_standard_structure_types.pm6)
/BorderStyle|[Table 343 – Standard layout attributes common to all standard structure types](gen/lib/ISO_32000/Table_343-Standard_layout_attributes_common_to_all_standard_structure_types.pm6)
/BorderThickness|[Table 343 – Standard layout attributes common to all standard structure types](gen/lib/ISO_32000/Table_343-Standard_layout_attributes_common_to_all_standard_structure_types.pm6)
/Bounds|[Table 41 – Additional entries specific to a type 3 function dictionary](gen/lib/ISO_32000/Table_41-Additional_entries_specific_to_a_type_3_function_dictionary.pm6)
/BoxColorInfo|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.pm6)
/ByteRange|[Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.pm6)
/C|[Table 153 – Entries in an outline item dictionary](gen/lib/ISO_32000/Table_153-Entries_in_an_outline_item_dictionary.pm6) [Table 164 – Entries common to all annotation dictionaries](gen/lib/ISO_32000/Table_164-Entries_common_to_all_annotation_dictionaries.pm6) [Table 195 – Entries in a page object’s additional-actions dictionary](gen/lib/ISO_32000/Table_195-Entries_in_a_page_objects_additional-actions_dictionary.pm6) [Table 196 – Entries in a form field’s additional-actions dictionary](gen/lib/ISO_32000/Table_196-Entries_in_a_form_fields_additional-actions_dictionary.pm6) [Table 263 – Entries in a number format dictionary](gen/lib/ISO_32000/Table_263-Entries_in_a_number_format_dictionary.pm6) [Table 267 – Entries in a rendition MH/BE dictionary](gen/lib/ISO_32000/Table_267-Entries_in_a_rendition_MH-BE_dictionary.pm6) [Table 268 – Entries in a media criteria dictionary](gen/lib/ISO_32000/Table_268-Entries_in_a_media_criteria_dictionary.pm6) [Table 271 – Additional entries in a media rendition dictionary](gen/lib/ISO_32000/Table_271-Additional_entries_in_a_media_rendition_dictionary.pm6) [Table 280 – Entries in a media play parameters MH/BE dictionary](gen/lib/ISO_32000/Table_280-Entries_in_a_media_play_parameters_MH-BE_dictionary.pm6) [Table 294 – Additional entries specific to a sound object](gen/lib/ISO_32000/Table_294-Additional_entries_specific_to_a_sound_object.pm6) [Table 306 – Entries in a 3D background dictionary](gen/lib/ISO_32000/Table_306-Entries_in_a_ThreeD_background_dictionary.pm6) [Table 311 – Entries in a 3D cross section dictionary](gen/lib/ISO_32000/Table_311-Entries_in_a_ThreeD_cross_section_dictionary.pm6) [Table 323 – Entries in a structure element dictionary](gen/lib/ISO_32000/Table_323-Entries_in_a_structure_element_dictionary.pm6) [Table 350 – Entries in the Web Capture information dictionary](gen/lib/ISO_32000/Table_350-Entries_in_the_Web_Capture_information_dictionary.pm6) [Table 355 – Entries in a source information dictionary](gen/lib/ISO_32000/Table_355-Entries_in_a_source_information_dictionary.pm6) [Table 356 – Entries in a URL alias dictionary](gen/lib/ISO_32000/Table_356-Entries_in_a_URL_alias_dictionary.pm6) [Table 359 – Entries in a Web Capture command settings dictionary](gen/lib/ISO_32000/Table_359-Entries_in_a_Web_Capture_command_settings_dictionary.pm6) [Table 361 – Entries in a box style dictionary](gen/lib/ISO_32000/Table_361-Entries_in_a_box_style_dictionary.pm6)
/C0|[Table 40 – Additional entries specific to a type 2 function dictionary](gen/lib/ISO_32000/Table_40-Additional_entries_specific_to_a_type_2_function_dictionary.pm6)
/C1|[Table 40 – Additional entries specific to a type 2 function dictionary](gen/lib/ISO_32000/Table_40-Additional_entries_specific_to_a_type_2_function_dictionary.pm6)
/C2W|[Table 304 – Entries in a 3D view dictionary](gen/lib/ISO_32000/Table_304-Entries_in_a_ThreeD_view_dictionary.pm6)
/CA|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.pm6) [Table 170 – Additional entries specific to markup annotations](gen/lib/ISO_32000/Table_170-Additional_entries_specific_to_markup_annotations.pm6) [Table 189 – Entries in an appearance characteristics dictionary](gen/lib/ISO_32000/Table_189-Entries_in_an_appearance_characteristics_dictionary.pm6)
/CF|[Table 20 – Entries common to all encryption dictionaries](gen/lib/ISO_32000/Table_20-Entries_common_to_all_encryption_dictionaries.pm6)
/CFM|[Table 25 – Entries common to all crypt filter dictionaries](gen/lib/ISO_32000/Table_25-Entries_common_to_all_crypt_filter_dictionaries.pm6)
/CI|[Table 44 – Entries in a file specification dictionary](gen/lib/ISO_32000/Table_44-Entries_in_a_file_specification_dictionary.pm6)
/CIDSet|[Table 124 – Additional font descriptor entries for CIDFonts](gen/lib/ISO_32000/Table_124-Additional_font_descriptor_entries_for_CIDFonts.pm6)
/CIDSystemInfo|[Table 117 – Entries in a CIDFont dictionary](gen/lib/ISO_32000/Table_117-Entries_in_a_CIDFont_dictionary.pm6) [Table 120 – Additional entries in a CMap stream dictionary](gen/lib/ISO_32000/Table_120-Additional_entries_in_a_CMap_stream_dictionary.pm6)
/CIDToGIDMap|[Table 117 – Entries in a CIDFont dictionary](gen/lib/ISO_32000/Table_117-Entries_in_a_CIDFont_dictionary.pm6)
/CL|[Table 174 – Additional entries specific to a free text annotation](gen/lib/ISO_32000/Table_174-Additional_entries_specific_to_a_free_text_annotation.pm6)
/CMapName|[Table 120 – Additional entries in a CMap stream dictionary](gen/lib/ISO_32000/Table_120-Additional_entries_in_a_CMap_stream_dictionary.pm6)
/CO|[Table 175 – Additional entries specific to a line annotation](gen/lib/ISO_32000/Table_175-Additional_entries_specific_to_a_line_annotation.pm6) [Table 218 – Entries in the interactive form dictionary](gen/lib/ISO_32000/Table_218-Entries_in_the_interactive_form_dictionary.pm6) [Table 294 – Additional entries specific to a sound object](gen/lib/ISO_32000/Table_294-Additional_entries_specific_to_a_sound_object.pm6) [Table 304 – Entries in a 3D view dictionary](gen/lib/ISO_32000/Table_304-Entries_in_a_ThreeD_view_dictionary.pm6)
/CP|[Table 175 – Additional entries specific to a line annotation](gen/lib/ISO_32000/Table_175-Additional_entries_specific_to_a_line_annotation.pm6) [Table 294 – Additional entries specific to a sound object](gen/lib/ISO_32000/Table_294-Additional_entries_specific_to_a_sound_object.pm6)
/CS|[Table 147 – Additional entries specific to a transparency group attributes dictionary](gen/lib/ISO_32000/Table_147-Additional_entries_specific_to_a_transparency_group_attributes_dictionary.pm6) [Table 305 – Entries in a projection dictionary](gen/lib/ISO_32000/Table_305-Entries_in_a_projection_dictionary.pm6) [Table 306 – Entries in a 3D background dictionary](gen/lib/ISO_32000/Table_306-Entries_in_a_ThreeD_background_dictionary.pm6)
/CT|[Table 274 – Additional entries in a media clip data dictionary](gen/lib/ISO_32000/Table_274-Additional_entries_in_a_media_clip_data_dictionary.pm6) [Table 352 – Entries common to all Web Capture content sets](gen/lib/ISO_32000/Table_352-Entries_common_to_all_Web_Capture_content_sets.pm6) [Table 357 – Entries in a Web Capture command dictionary](gen/lib/ISO_32000/Table_357-Entries_in_a_Web_Capture_command_dictionary.pm6)
/CTM|[Table 52 – Device-Independent Graphics State Parameters](gen/lib/ISO_32000/Table_52-Device-Independent_Graphics_State_Parameters.pm6)
/CV|[Table 307 – Entries in a render mode dictionary](gen/lib/ISO_32000/Table_307-Entries_in_a_render_mode_dictionary.pm6)
/CYX|[Table 262 – Additional entries in a rectilinear measure dictionary](gen/lib/ISO_32000/Table_262-Additional_entries_in_a_rectilinear_measure_dictionary.pm6)
/Cap|[Table 175 – Additional entries specific to a line annotation](gen/lib/ISO_32000/Table_175-Additional_entries_specific_to_a_line_annotation.pm6)
/CapHeight|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.pm6)
/Category|[Table 103 – Entries in a Usage Application Dictionary](gen/lib/ISO_32000/Table_103-Entries_in_a_Usage_Application_Dictionary.pm6)
/CenterWindow|[Table 150 – Entries in a viewer preferences dictionary](gen/lib/ISO_32000/Table_150-Entries_in_a_viewer_preferences_dictionary.pm6)
/Cert|[Table 234 – Entries in a signature field seed value dictionary](gen/lib/ISO_32000/Table_234-Entries_in_a_signature_field_seed_value_dictionary.pm6) [Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.pm6)
/Changes|[Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.pm6)
/CharProcs|[Table 112 – Entries in a Type 3 font dictionary](gen/lib/ISO_32000/Table_112-Entries_in_a_Type_3_font_dictionary.pm6)
/CharSet|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.pm6)
/CheckSum|[Table 46 – Entries in an embedded file parameter dictionary](gen/lib/ISO_32000/Table_46-Entries_in_an_embedded_file_parameter_dictionary.pm6)
/ClassMap|[Table 322 – Entries in the structure tree root](gen/lib/ISO_32000/Table_322-Entries_in_the_structure_tree_root.pm6)
/ClrF|[Table 246 – Entries in an FDF field dictionary](gen/lib/ISO_32000/Table_246-Entries_in_an_FDF_field_dictionary.pm6)
/ClrFf|[Table 246 – Entries in an FDF field dictionary](gen/lib/ISO_32000/Table_246-Entries_in_an_FDF_field_dictionary.pm6)
/ColSpan|[Table 349 – Standard table attributes](gen/lib/ISO_32000/Table_349-Standard_table_attributes.pm6)
/Collection|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.pm6)
/Color|[Table 343 – Standard layout attributes common to all standard structure types](gen/lib/ISO_32000/Table_343-Standard_layout_attributes_common_to_all_standard_structure_types.pm6)
/ColorSpace|[Table 33 – Entries in a resource dictionary](gen/lib/ISO_32000/Table_33-Entries_in_a_resource_dictionary.pm6) [Table 72 – Entries in a DeviceN Process Dictionary](gen/lib/ISO_32000/Table_72-Entries_in_a_DeviceN_Process_Dictionary.pm6) [Table 78 – Entries Common to All Shading Dictionaries](gen/lib/ISO_32000/Table_78-Entries_Common_to_All_Shading_Dictionaries.pm6) [Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.pm6) [Table 364 – Entries in a separation dictionary](gen/lib/ISO_32000/Table_364-Entries_in_a_separation_dictionary.pm6)
/ColorTransform|[Table 13 – Optional parameter for the DCTDecode filter](gen/lib/ISO_32000/Table_13-Optional_parameter_for_the_DCTDecode_filter.pm6)
/Colorants|[Table 71 – Entries in a DeviceN Colour Space Attributes Dictionary](gen/lib/ISO_32000/Table_71-Entries_in_a_DeviceN_Colour_Space_Attributes_Dictionary.pm6) [Table 363 – Additional entries specific to a printer’s mark form dictionary](gen/lib/ISO_32000/Table_363-Additional_entries_specific_to_a_printers_mark_form_dictionary.pm6)
/Colors|[Table 8 – Optional parameters for LZWDecode and FlateDecode filters](gen/lib/ISO_32000/Table_8-Optional_parameters_for_LZWDecode_and_FlateDecode_filters.pm6)
/ColumnCount|[Table 346 – Standard column attributes](gen/lib/ISO_32000/Table_346-Standard_column_attributes.pm6)
/ColumnGap|[Table 346 – Standard column attributes](gen/lib/ISO_32000/Table_346-Standard_column_attributes.pm6)
/ColumnWidths|[Table 346 – Standard column attributes](gen/lib/ISO_32000/Table_346-Standard_column_attributes.pm6)
/Columns|[Table 8 – Optional parameters for LZWDecode and FlateDecode filters](gen/lib/ISO_32000/Table_8-Optional_parameters_for_LZWDecode_and_FlateDecode_filters.pm6) [Table 11 – Optional parameters for the CCITTFaxDecode filter](gen/lib/ISO_32000/Table_11-Optional_parameters_for_the_CCITTFaxDecode_filter.pm6)
/Components|[Table 72 – Entries in a DeviceN Process Dictionary](gen/lib/ISO_32000/Table_72-Entries_in_a_DeviceN_Process_Dictionary.pm6)
/Configs|[Table 100 – Entries in the Optional Content Properties Dictionary](gen/lib/ISO_32000/Table_100-Entries_in_the_Optional_Content_Properties_Dictionary.pm6)
/ContactInfo|[Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.pm6)
/Contents|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.pm6) [Table 164 – Entries common to all annotation dictionaries](gen/lib/ISO_32000/Table_164-Entries_common_to_all_annotation_dictionaries.pm6) [Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.pm6)
/Coords|[Table 80 – Additional Entries Specific to a Type 2 Shading Dictionary](gen/lib/ISO_32000/Table_80-Additional_Entries_Specific_to_a_Type_2_Shading_Dictionary.pm6) [Table 81 – Additional Entries Specific to a Type 3 Shading Dictionary](gen/lib/ISO_32000/Table_81-Additional_Entries_Specific_to_a_Type_3_Shading_Dictionary.pm6)
/Count|[Table 29 – Required entries in a page tree node](gen/lib/ISO_32000/Table_29-Required_entries_in_a_page_tree_node.pm6) [Table 152 – Entries in the outline dictionary](gen/lib/ISO_32000/Table_152-Entries_in_the_outline_dictionary.pm6) [Table 153 – Entries in an outline item dictionary](gen/lib/ISO_32000/Table_153-Entries_in_an_outline_item_dictionary.pm6)
/CreationDate|[Table 46 – Entries in an embedded file parameter dictionary](gen/lib/ISO_32000/Table_46-Entries_in_an_embedded_file_parameter_dictionary.pm6) [Table 170 – Additional entries specific to markup annotations](gen/lib/ISO_32000/Table_170-Additional_entries_specific_to_markup_annotations.pm6) [Table 317 – Entries in the document information dictionary](gen/lib/ISO_32000/Table_317-Entries_in_the_document_information_dictionary.pm6)
/Creator|[Table 47 – Entries in a Mac OS file information dictionary](gen/lib/ISO_32000/Table_47-Entries_in_a_Mac_OS_file_information_dictionary.pm6) [Table 101 – Entries in an Optional Content Configuration Dictionary](gen/lib/ISO_32000/Table_101-Entries_in_an_Optional_Content_Configuration_Dictionary.pm6) [Table 317 – Entries in the document information dictionary](gen/lib/ISO_32000/Table_317-Entries_in_the_document_information_dictionary.pm6)
/CreatorInfo|[Table 102 – Entries in an Optional Content Usage Dictionary](gen/lib/ISO_32000/Table_102-Entries_in_an_Optional_Content_Usage_Dictionary.pm6)
/CropBox|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.pm6) [Table 360 – Entries in a box colour information dictionary](gen/lib/ISO_32000/Table_360-Entries_in_a_box_colour_information_dictionary.pm6)
/D|[Table 49 – Entries in a collection subitem dictionary](gen/lib/ISO_32000/Table_49-Entries_in_a_collection_subitem_dictionary.pm6) [Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.pm6) [Table 100 – Entries in the Optional Content Properties Dictionary](gen/lib/ISO_32000/Table_100-Entries_in_the_Optional_Content_Properties_Dictionary.pm6) [Table 155 – Entries in a collection dictionary](gen/lib/ISO_32000/Table_155-Entries_in_a_collection_dictionary.pm6) [Table 162 – Entries in a transition dictionary](gen/lib/ISO_32000/Table_162-Entries_in_a_transition_dictionary.pm6) [Table 166 – Entries in a border style dictionary](gen/lib/ISO_32000/Table_166-Entries_in_a_border_style_dictionary.pm6) [Table 168 – Entries in an appearance dictionary](gen/lib/ISO_32000/Table_168-Entries_in_an_appearance_dictionary.pm6) [Table 194 – Entries in an annotation’s additional-actions dictionary](gen/lib/ISO_32000/Table_194-Entries_in_an_annotations_additional-actions_dictionary.pm6) [Table 199 – Additional entries specific to a go-to action](gen/lib/ISO_32000/Table_199-Additional_entries_specific_to_a_go-to_action.pm6) [Table 200 – Additional entries specific to a remote go-to action](gen/lib/ISO_32000/Table_200-Additional_entries_specific_to_a_remote_go-to_action.pm6) [Table 201 – Additional entries specific to an embedded go-to action](gen/lib/ISO_32000/Table_201-Additional_entries_specific_to_an_embedded_go-to_action.pm6) [Table 204 – Entries in a Windows launch parameter dictionary](gen/lib/ISO_32000/Table_204-Entries_in_a_Windows_launch_parameter_dictionary.pm6) [Table 205 – Additional entries specific to a thread action](gen/lib/ISO_32000/Table_205-Additional_entries_specific_to_a_thread_action.pm6) [Table 262 – Additional entries in a rectilinear measure dictionary](gen/lib/ISO_32000/Table_262-Additional_entries_in_a_rectilinear_measure_dictionary.pm6) [Table 263 – Entries in a number format dictionary](gen/lib/ISO_32000/Table_263-Entries_in_a_number_format_dictionary.pm6) [Table 268 – Entries in a media criteria dictionary](gen/lib/ISO_32000/Table_268-Entries_in_a_media_criteria_dictionary.pm6) [Table 274 – Additional entries in a media clip data dictionary](gen/lib/ISO_32000/Table_274-Additional_entries_in_a_media_clip_data_dictionary.pm6) [Table 277 – Additional entries in a media clip section dictionary](gen/lib/ISO_32000/Table_277-Additional_entries_in_a_media_clip_section_dictionary.pm6) [Table 280 – Entries in a media play parameters MH/BE dictionary](gen/lib/ISO_32000/Table_280-Entries_in_a_media_play_parameters_MH-BE_dictionary.pm6) [Table 284 – Entries in a floating window parameters dictionary](gen/lib/ISO_32000/Table_284-Entries_in_a_floating_window_parameters_dictionary.pm6) [Table 299 – Entries in a 3D activation dictionary](gen/lib/ISO_32000/Table_299-Entries_in_a_ThreeD_activation_dictionary.pm6) [Table 361 – Entries in a box style dictionary](gen/lib/ISO_32000/Table_361-Entries_in_a_box_style_dictionary.pm6)
/DA|[Table 174 – Additional entries specific to a free text annotation](gen/lib/ISO_32000/Table_174-Additional_entries_specific_to_a_free_text_annotation.pm6) [Table 192 – Additional entries specific to a redaction annotation](gen/lib/ISO_32000/Table_192-Additional_entries_specific_to_a_redaction_annotation.pm6) [Table 218 – Entries in the interactive form dictionary](gen/lib/ISO_32000/Table_218-Entries_in_the_interactive_form_dictionary.pm6) [Table 222 – Additional entries common to all fields containing variable text](gen/lib/ISO_32000/Table_222-Additional_entries_common_to_all_fields_containing_variable_text.pm6)
/DIS|[Table 299 – Entries in a 3D activation dictionary](gen/lib/ISO_32000/Table_299-Entries_in_a_ThreeD_activation_dictionary.pm6)
/DL|[Table 5 – Entries common to all stream dictionaries](gen/lib/ISO_32000/Table_5-Entries_common_to_all_stream_dictionaries.pm6)
/DOS|[Table 44 – Entries in a file specification dictionary](gen/lib/ISO_32000/Table_44-Entries_in_a_file_specification_dictionary.pm6)
/DP|[Table 197 – Entries in the document catalog’s additional-actions dictionary](gen/lib/ISO_32000/Table_197-Entries_in_the_document_catalogs_additional-actions_dictionary.pm6)
/DR|[Table 218 – Entries in the interactive form dictionary](gen/lib/ISO_32000/Table_218-Entries_in_the_interactive_form_dictionary.pm6)
/DS|[Table 174 – Additional entries specific to a free text annotation](gen/lib/ISO_32000/Table_174-Additional_entries_specific_to_a_free_text_annotation.pm6) [Table 197 – Entries in the document catalog’s additional-actions dictionary](gen/lib/ISO_32000/Table_197-Entries_in_the_document_catalogs_additional-actions_dictionary.pm6) [Table 222 – Additional entries common to all fields containing variable text](gen/lib/ISO_32000/Table_222-Additional_entries_common_to_all_fields_containing_variable_text.pm6)
/DV|[Table 220 – Entries common to all field dictionaries](gen/lib/ISO_32000/Table_220-Entries_common_to_all_field_dictionaries.pm6) [Table 300 – Entries in a 3D stream dictionary](gen/lib/ISO_32000/Table_300-Entries_in_a_ThreeD_stream_dictionary.pm6)
/DW|[Table 117 – Entries in a CIDFont dictionary](gen/lib/ISO_32000/Table_117-Entries_in_a_CIDFont_dictionary.pm6)
/DW2|[Table 117 – Entries in a CIDFont dictionary](gen/lib/ISO_32000/Table_117-Entries_in_a_CIDFont_dictionary.pm6)
/DamagedRowsBeforeError|[Table 11 – Optional parameters for the CCITTFaxDecode filter](gen/lib/ISO_32000/Table_11-Optional_parameters_for_the_CCITTFaxDecode_filter.pm6)
/Data|[Table 253 – Entries in a signature reference dictionary](gen/lib/ISO_32000/Table_253-Entries_in_a_signature_reference_dictionary.pm6)
/Decode|[Table 39 – Additional entries specific to a type 0 function dictionary](gen/lib/ISO_32000/Table_39-Additional_entries_specific_to_a_type_0_function_dictionary.pm6) [Table 82 – Additional Entries Specific to a Type 4 Shading Dictionary](gen/lib/ISO_32000/Table_82-Additional_Entries_Specific_to_a_Type_4_Shading_Dictionary.pm6) [Table 83 – Additional Entries Specific to a Type 5 Shading Dictionary](gen/lib/ISO_32000/Table_83-Additional_Entries_Specific_to_a_Type_5_Shading_Dictionary.pm6) [Table 84 – Additional Entries Specific to a Type 6 Shading Dictionary](gen/lib/ISO_32000/Table_84-Additional_Entries_Specific_to_a_Type_6_Shading_Dictionary.pm6) [Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.pm6)
/DecodeParms|[Table 5 – Entries common to all stream dictionaries](gen/lib/ISO_32000/Table_5-Entries_common_to_all_stream_dictionaries.pm6)
/Default|[Table 134 – Entries in a type 5 halftone dictionary](gen/lib/ISO_32000/Table_134-Entries_in_a_type_5_halftone_dictionary.pm6)
/DefaultForPrinting|[Table 91 – Entries in an Alternate Image Dictionary](gen/lib/ISO_32000/Table_91-Entries_in_an_Alternate_Image_Dictionary.pm6)
/Desc|[Table 44 – Entries in a file specification dictionary](gen/lib/ISO_32000/Table_44-Entries_in_a_file_specification_dictionary.pm6) [Table 348 – PrintField attributes](gen/lib/ISO_32000/Table_348-PrintField_attributes.pm6)
/DescendantFonts|[Table 121 – Entries in a Type 0 font dictionary](gen/lib/ISO_32000/Table_121-Entries_in_a_Type_0_font_dictionary.pm6)
/Descent|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.pm6)
/Dest|[Table 153 – Entries in an outline item dictionary](gen/lib/ISO_32000/Table_153-Entries_in_an_outline_item_dictionary.pm6) [Table 173 – Additional entries specific to a link annotation](gen/lib/ISO_32000/Table_173-Additional_entries_specific_to_a_link_annotation.pm6)
/DestOutputProfile|[Table 365 – Entries in an output intent dictionary](gen/lib/ISO_32000/Table_365-Entries_in_an_output_intent_dictionary.pm6)
/Dests|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.pm6) [Table 31 – Entries in the name dictionary](gen/lib/ISO_32000/Table_31-Entries_in_the_name_dictionary.pm6)
/DevDepGS_BG|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.pm6)
/DevDepGS_FL|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.pm6)
/DevDepGS_HT|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.pm6)
/DevDepGS_OP|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.pm6)
/DevDepGS_TR|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.pm6)
/DevDepGS_UCR|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.pm6)
/DeviceColorant|[Table 364 – Entries in a separation dictionary](gen/lib/ISO_32000/Table_364-Entries_in_a_separation_dictionary.pm6)
/Di|[Table 162 – Entries in a transition dictionary](gen/lib/ISO_32000/Table_162-Entries_in_a_transition_dictionary.pm6)
/Differences|[Table 114 – Entries in an encoding dictionary](gen/lib/ISO_32000/Table_114-Entries_in_an_encoding_dictionary.pm6) [Table 243 – Entries in the FDF dictionary](gen/lib/ISO_32000/Table_243-Entries_in_the_FDF_dictionary.pm6)
/DigestMethod|[Table 234 – Entries in a signature field seed value dictionary](gen/lib/ISO_32000/Table_234-Entries_in_a_signature_field_seed_value_dictionary.pm6) [Table 253 – Entries in a signature reference dictionary](gen/lib/ISO_32000/Table_253-Entries_in_a_signature_reference_dictionary.pm6)
/Direction|[Table 150 – Entries in a viewer preferences dictionary](gen/lib/ISO_32000/Table_150-Entries_in_a_viewer_preferences_dictionary.pm6)
/DisplayDocTitle|[Table 150 – Entries in a viewer preferences dictionary](gen/lib/ISO_32000/Table_150-Entries_in_a_viewer_preferences_dictionary.pm6)
/Dm|[Table 162 – Entries in a transition dictionary](gen/lib/ISO_32000/Table_162-Entries_in_a_transition_dictionary.pm6)
/Doc|[Table 245 – Entries in the JavaScript dictionary](gen/lib/ISO_32000/Table_245-Entries_in_the_JavaScript_dictionary.pm6)
/DocMDP|[Table 258 – Entries in a permissions dictionary](gen/lib/ISO_32000/Table_258-Entries_in_a_permissions_dictionary.pm6)
/Document|[Table 255 – Entries in the UR transform parameters dictionary](gen/lib/ISO_32000/Table_255-Entries_in_the_UR_transform_parameters_dictionary.pm6)
/Domain|[Table 38 – Entries common to all function dictionaries](gen/lib/ISO_32000/Table_38-Entries_common_to_all_function_dictionaries.pm6) [Table 79 – Additional Entries Specific to a Type 1 Shading Dictionary](gen/lib/ISO_32000/Table_79-Additional_Entries_Specific_to_a_Type_1_Shading_Dictionary.pm6) [Table 80 – Additional Entries Specific to a Type 2 Shading Dictionary](gen/lib/ISO_32000/Table_80-Additional_Entries_Specific_to_a_Type_2_Shading_Dictionary.pm6) [Table 81 – Additional Entries Specific to a Type 3 Shading Dictionary](gen/lib/ISO_32000/Table_81-Additional_Entries_Specific_to_a_Type_3_Shading_Dictionary.pm6)
/DotGain|[Table 73 – Entries in a DeviceN Mixing Hints Dictionary](gen/lib/ISO_32000/Table_73-Entries_in_a_DeviceN_Mixing_Hints_Dictionary.pm6)
/Duplex|[Table 150 – Entries in a viewer preferences dictionary](gen/lib/ISO_32000/Table_150-Entries_in_a_viewer_preferences_dictionary.pm6)
/Dur|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.pm6) [Table 163 – Entries in a navigation node dictionary](gen/lib/ISO_32000/Table_163-Entries_in_a_navigation_node_dictionary.pm6)
/Duration|[Table 296 – Entries in a movie activation dictionary](gen/lib/ISO_32000/Table_296-Entries_in_a_movie_activation_dictionary.pm6)
/E|[Table 157 – Entries in a collection field dictionary](gen/lib/ISO_32000/Table_157-Entries_in_a_collection_field_dictionary.pm6) [Table 194 – Entries in an annotation’s additional-actions dictionary](gen/lib/ISO_32000/Table_194-Entries_in_an_annotations_additional-actions_dictionary.pm6) [Table 278 – Entries in a media clip section MH/BE dictionary](gen/lib/ISO_32000/Table_278-Entries_in_a_media_clip_section_MH-BE_dictionary.pm6) [Table 294 – Additional entries specific to a sound object](gen/lib/ISO_32000/Table_294-Additional_entries_specific_to_a_sound_object.pm6) [Table 323 – Entries in a structure element dictionary](gen/lib/ISO_32000/Table_323-Entries_in_a_structure_element_dictionary.pm6) [Table 355 – Entries in a source information dictionary](gen/lib/ISO_32000/Table_355-Entries_in_a_source_information_dictionary.pm6) [Table F. 1 – Entries in the linearization parameter dictionary](gen/lib/ISO_32000/Table_F1-Entries_in_the_linearization_parameter_dictionary.pm6)
/EA|[Table 306 – Entries in a 3D background dictionary](gen/lib/ISO_32000/Table_306-Entries_in_a_ThreeD_background_dictionary.pm6)
/EF|[Table 44 – Entries in a file specification dictionary](gen/lib/ISO_32000/Table_44-Entries_in_a_file_specification_dictionary.pm6) [Table 255 – Entries in the UR transform parameters dictionary](gen/lib/ISO_32000/Table_255-Entries_in_the_UR_transform_parameters_dictionary.pm6)
/EFF|[Table 20 – Entries common to all encryption dictionaries](gen/lib/ISO_32000/Table_20-Entries_common_to_all_encryption_dictionaries.pm6)
/EarlyChange|[Table 8 – Optional parameters for LZWDecode and FlateDecode filters](gen/lib/ISO_32000/Table_8-Optional_parameters_for_LZWDecode_and_FlateDecode_filters.pm6)
/EmbeddedFDFs|[Table 243 – Entries in the FDF dictionary](gen/lib/ISO_32000/Table_243-Entries_in_the_FDF_dictionary.pm6)
/EmbeddedFiles|[Table 31 – Entries in the name dictionary](gen/lib/ISO_32000/Table_31-Entries_in_the_name_dictionary.pm6)
/Encode|[Table 39 – Additional entries specific to a type 0 function dictionary](gen/lib/ISO_32000/Table_39-Additional_entries_specific_to_a_type_0_function_dictionary.pm6) [Table 41 – Additional entries specific to a type 3 function dictionary](gen/lib/ISO_32000/Table_41-Additional_entries_specific_to_a_type_3_function_dictionary.pm6)
/EncodedByteAlign|[Table 11 – Optional parameters for the CCITTFaxDecode filter](gen/lib/ISO_32000/Table_11-Optional_parameters_for_the_CCITTFaxDecode_filter.pm6)
/Encoding|[Table 111 – Entries in a Type 1 font dictionary](gen/lib/ISO_32000/Table_111-Entries_in_a_Type_1_font_dictionary.pm6) [Table 112 – Entries in a Type 3 font dictionary](gen/lib/ISO_32000/Table_112-Entries_in_a_Type_3_font_dictionary.pm6) [Table 121 – Entries in a Type 0 font dictionary](gen/lib/ISO_32000/Table_121-Entries_in_a_Type_0_font_dictionary.pm6) [Table 243 – Entries in the FDF dictionary](gen/lib/ISO_32000/Table_243-Entries_in_the_FDF_dictionary.pm6)
/Encrypt|[Table 15 – Entries in the file trailer dictionary](gen/lib/ISO_32000/Table_15-Entries_in_the_file_trailer_dictionary.pm6)
/EncryptMetadata|[Table 21 – Additional encryption dictionary entries for the standard security handler](gen/lib/ISO_32000/Table_21-Additional_encryption_dictionary_entries_for_the_standard_security_handler.pm6) [Table 27 – Additional crypt filter dictionary entries for public-key security handlers](gen/lib/ISO_32000/Table_27-Additional_crypt_filter_dictionary_entries_for_public-key_security_handlers.pm6)
/EncryptionRevision|[Table 244 – Additional entry in an embedded file stream dictionary for an encrypted FDF file](gen/lib/ISO_32000/Table_244-Additional_entry_in_an_embedded_file_stream_dictionary_for_an_encrypted_FDF_file.pm6)
/EndIndent|[Table 344 – Additional standard layout attributes specific to block-level structure elements](gen/lib/ISO_32000/Table_344-Additional_standard_layout_attributes_specific_to_block-level_structure_elements.pm6)
/EndOfBlock|[Table 11 – Optional parameters for the CCITTFaxDecode filter](gen/lib/ISO_32000/Table_11-Optional_parameters_for_the_CCITTFaxDecode_filter.pm6)
/EndOfLine|[Table 11 – Optional parameters for the CCITTFaxDecode filter](gen/lib/ISO_32000/Table_11-Optional_parameters_for_the_CCITTFaxDecode_filter.pm6)
/Event|[Table 103 – Entries in a Usage Application Dictionary](gen/lib/ISO_32000/Table_103-Entries_in_a_Usage_Application_Dictionary.pm6)
/ExData|[Table 170 – Additional entries specific to markup annotations](gen/lib/ISO_32000/Table_170-Additional_entries_specific_to_markup_annotations.pm6)
/Export|[Table 102 – Entries in an Optional Content Usage Dictionary](gen/lib/ISO_32000/Table_102-Entries_in_an_Optional_Content_Usage_Dictionary.pm6)
/ExtGState|[Table 33 – Entries in a resource dictionary](gen/lib/ISO_32000/Table_33-Entries_in_a_resource_dictionary.pm6) [Table 76 – Entries in a Type 2 Pattern Dictionary](gen/lib/ISO_32000/Table_76-Entries_in_a_Type_2_Pattern_Dictionary.pm6)
/Extend|[Table 80 – Additional Entries Specific to a Type 2 Shading Dictionary](gen/lib/ISO_32000/Table_80-Additional_Entries_Specific_to_a_Type_2_Shading_Dictionary.pm6) [Table 81 – Additional Entries Specific to a Type 3 Shading Dictionary](gen/lib/ISO_32000/Table_81-Additional_Entries_Specific_to_a_Type_3_Shading_Dictionary.pm6)
/Extends|[Table 16 – Additional entries specific to an object stream dictionary](gen/lib/ISO_32000/Table_16-Additional_entries_specific_to_an_object_stream_dictionary.pm6)
/ExtensionLevel|[Table 50 – Entries in a developer extensions dictionary](gen/lib/ISO_32000/Table_50-Entries_in_a_developer_extensions_dictionary.pm6)
/Extensions|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.pm6)
/ExternalOPIdicts|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.pm6)
/ExternalRefXobjects|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.pm6)
/ExternalStreams|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.pm6)
/F|[Table 5 – Entries common to all stream dictionaries](gen/lib/ISO_32000/Table_5-Entries_common_to_all_stream_dictionaries.pm6) [Table 44 – Entries in a file specification dictionary](gen/lib/ISO_32000/Table_44-Entries_in_a_file_specification_dictionary.pm6) [Table 97 – Entries in a Reference Dictionary](gen/lib/ISO_32000/Table_97-Entries_in_a_Reference_Dictionary.pm6) [Table 153 – Entries in an outline item dictionary](gen/lib/ISO_32000/Table_153-Entries_in_an_outline_item_dictionary.pm6) [Table 160 – Entries in a thread dictionary](gen/lib/ISO_32000/Table_160-Entries_in_a_thread_dictionary.pm6) [Table 164 – Entries common to all annotation dictionaries](gen/lib/ISO_32000/Table_164-Entries_common_to_all_annotation_dictionaries.pm6) [Table 196 – Entries in a form field’s additional-actions dictionary](gen/lib/ISO_32000/Table_196-Entries_in_a_form_fields_additional-actions_dictionary.pm6) [Table 200 – Additional entries specific to a remote go-to action](gen/lib/ISO_32000/Table_200-Additional_entries_specific_to_a_remote_go-to_action.pm6) [Table 201 – Additional entries specific to an embedded go-to action](gen/lib/ISO_32000/Table_201-Additional_entries_specific_to_an_embedded_go-to_action.pm6) [Table 203 – Additional entries specific to a launch action](gen/lib/ISO_32000/Table_203-Additional_entries_specific_to_a_launch_action.pm6) [Table 204 – Entries in a Windows launch parameter dictionary](gen/lib/ISO_32000/Table_204-Entries_in_a_Windows_launch_parameter_dictionary.pm6) [Table 205 – Additional entries specific to a thread action](gen/lib/ISO_32000/Table_205-Additional_entries_specific_to_a_thread_action.pm6) [Table 236 – Additional entries specific to a submit-form action](gen/lib/ISO_32000/Table_236-Additional_entries_specific_to_a_submit-form_action.pm6) [Table 240 – Additional entries specific to an import-data action](gen/lib/ISO_32000/Table_240-Additional_entries_specific_to_an_import-data_action.pm6) [Table 243 – Entries in the FDF dictionary](gen/lib/ISO_32000/Table_243-Entries_in_the_FDF_dictionary.pm6) [Table 246 – Entries in an FDF field dictionary](gen/lib/ISO_32000/Table_246-Entries_in_an_FDF_field_dictionary.pm6) [Table 250 – Entries in an FDF named page reference dictionary](gen/lib/ISO_32000/Table_250-Entries_in_an_FDF_named_page_reference_dictionary.pm6) [Table 263 – Entries in a number format dictionary](gen/lib/ISO_32000/Table_263-Entries_in_a_number_format_dictionary.pm6) [Table 280 – Entries in a media play parameters MH/BE dictionary](gen/lib/ISO_32000/Table_280-Entries_in_a_media_play_parameters_MH-BE_dictionary.pm6) [Table 283 – Entries in a media screen parameters MH/BE dictionary](gen/lib/ISO_32000/Table_283-Entries_in_a_media_screen_parameters_MH-BE_dictionary.pm6) [Table 287 – Additional entries in a media offset frame dictionary](gen/lib/ISO_32000/Table_287-Additional_entries_in_a_media_offset_frame_dictionary.pm6) [Table 295 – Entries in a movie dictionary](gen/lib/ISO_32000/Table_295-Entries_in_a_movie_dictionary.pm6) [Table 305 – Entries in a projection dictionary](gen/lib/ISO_32000/Table_305-Entries_in_a_projection_dictionary.pm6) [Table 329 – Entries in a user property dictionary](gen/lib/ISO_32000/Table_329-Entries_in_a_user_property_dictionary.pm6) [Table 357 – Entries in a Web Capture command dictionary](gen/lib/ISO_32000/Table_357-Entries_in_a_Web_Capture_command_dictionary.pm6)
/FB|[Table 247 – Entries in an icon fit dictionary](gen/lib/ISO_32000/Table_247-Entries_in_an_icon_fit_dictionary.pm6)
/FC|[Table 307 – Entries in a render mode dictionary](gen/lib/ISO_32000/Table_307-Entries_in_a_render_mode_dictionary.pm6)
/FD|[Table 124 – Additional font descriptor entries for CIDFonts](gen/lib/ISO_32000/Table_124-Additional_font_descriptor_entries_for_CIDFonts.pm6) [Table 263 – Entries in a number format dictionary](gen/lib/ISO_32000/Table_263-Entries_in_a_number_format_dictionary.pm6)
/FDF|[Table 242 – Entries in the FDF catalog dictionary](gen/lib/ISO_32000/Table_242-Entries_in_the_FDF_catalog_dictionary.pm6)
/FDecodeParms|[Table 5 – Entries common to all stream dictionaries](gen/lib/ISO_32000/Table_5-Entries_common_to_all_stream_dictionaries.pm6)
/FFilter|[Table 5 – Entries common to all stream dictionaries](gen/lib/ISO_32000/Table_5-Entries_common_to_all_stream_dictionaries.pm6)
/FL|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.pm6)
/FOV|[Table 305 – Entries in a projection dictionary](gen/lib/ISO_32000/Table_305-Entries_in_a_projection_dictionary.pm6)
/FS|[Table 44 – Entries in a file specification dictionary](gen/lib/ISO_32000/Table_44-Entries_in_a_file_specification_dictionary.pm6) [Table 184 – Additional entries specific to a file attachment annotation](gen/lib/ISO_32000/Table_184-Additional_entries_specific_to_a_file_attachment_annotation.pm6)
/FT|[Table 220 – Entries common to all field dictionaries](gen/lib/ISO_32000/Table_220-Entries_common_to_all_field_dictionaries.pm6)
/FWPosition|[Table 296 – Entries in a movie activation dictionary](gen/lib/ISO_32000/Table_296-Entries_in_a_movie_activation_dictionary.pm6)
/FWScale|[Table 296 – Entries in a movie activation dictionary](gen/lib/ISO_32000/Table_296-Entries_in_a_movie_activation_dictionary.pm6)
/Ff|[Table 220 – Entries common to all field dictionaries](gen/lib/ISO_32000/Table_220-Entries_common_to_all_field_dictionaries.pm6) [Table 234 – Entries in a signature field seed value dictionary](gen/lib/ISO_32000/Table_234-Entries_in_a_signature_field_seed_value_dictionary.pm6) [Table 235 – Entries in a certificate seed value dictionary](gen/lib/ISO_32000/Table_235-Entries_in_a_certificate_seed_value_dictionary.pm6) [Table 246 – Entries in an FDF field dictionary](gen/lib/ISO_32000/Table_246-Entries_in_an_FDF_field_dictionary.pm6)
/Fields|[Table 218 – Entries in the interactive form dictionary](gen/lib/ISO_32000/Table_218-Entries_in_the_interactive_form_dictionary.pm6) [Table 233 – Entries in a signature field lock dictionary](gen/lib/ISO_32000/Table_233-Entries_in_a_signature_field_lock_dictionary.pm6) [Table 236 – Additional entries specific to a submit-form action](gen/lib/ISO_32000/Table_236-Additional_entries_specific_to_a_submit-form_action.pm6) [Table 238 – Additional entries specific to a reset-form action](gen/lib/ISO_32000/Table_238-Additional_entries_specific_to_a_reset-form_action.pm6) [Table 243 – Entries in the FDF dictionary](gen/lib/ISO_32000/Table_243-Entries_in_the_FDF_dictionary.pm6) [Table 249 – Entries in an FDF template dictionary](gen/lib/ISO_32000/Table_249-Entries_in_an_FDF_template_dictionary.pm6) [Table 256 – Entries in the FieldMDP transform parameters dictionary](gen/lib/ISO_32000/Table_256-Entries_in_the_FieldMDP_transform_parameters_dictionary.pm6)
/Filter|[Table 5 – Entries common to all stream dictionaries](gen/lib/ISO_32000/Table_5-Entries_common_to_all_stream_dictionaries.pm6) [Table 20 – Entries common to all encryption dictionaries](gen/lib/ISO_32000/Table_20-Entries_common_to_all_encryption_dictionaries.pm6) [Table 234 – Entries in a signature field seed value dictionary](gen/lib/ISO_32000/Table_234-Entries_in_a_signature_field_seed_value_dictionary.pm6) [Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.pm6)
/First|[Table 16 – Additional entries specific to an object stream dictionary](gen/lib/ISO_32000/Table_16-Additional_entries_specific_to_an_object_stream_dictionary.pm6) [Table 152 – Entries in the outline dictionary](gen/lib/ISO_32000/Table_152-Entries_in_the_outline_dictionary.pm6) [Table 153 – Entries in an outline item dictionary](gen/lib/ISO_32000/Table_153-Entries_in_an_outline_item_dictionary.pm6)
/FirstChar|[Table 111 – Entries in a Type 1 font dictionary](gen/lib/ISO_32000/Table_111-Entries_in_a_Type_1_font_dictionary.pm6) [Table 112 – Entries in a Type 3 font dictionary](gen/lib/ISO_32000/Table_112-Entries_in_a_Type_3_font_dictionary.pm6)
/FitWindow|[Table 150 – Entries in a viewer preferences dictionary](gen/lib/ISO_32000/Table_150-Entries_in_a_viewer_preferences_dictionary.pm6)
/FixedPrint|[Table 190 – Additional entries specific to a watermark annotation](gen/lib/ISO_32000/Table_190-Additional_entries_specific_to_a_watermark_annotation.pm6)
/Flags|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.pm6) [Table 236 – Additional entries specific to a submit-form action](gen/lib/ISO_32000/Table_236-Additional_entries_specific_to_a_submit-form_action.pm6) [Table 238 – Additional entries specific to a reset-form action](gen/lib/ISO_32000/Table_238-Additional_entries_specific_to_a_reset-form_action.pm6)
/Fo|[Table 194 – Entries in an annotation’s additional-actions dictionary](gen/lib/ISO_32000/Table_194-Entries_in_an_annotations_additional-actions_dictionary.pm6)
/Font|[Table 33 – Entries in a resource dictionary](gen/lib/ISO_32000/Table_33-Entries_in_a_resource_dictionary.pm6) [Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.pm6)
/FontBBox|[Table 112 – Entries in a Type 3 font dictionary](gen/lib/ISO_32000/Table_112-Entries_in_a_Type_3_font_dictionary.pm6) [Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.pm6)
/FontDescriptor|[Table 111 – Entries in a Type 1 font dictionary](gen/lib/ISO_32000/Table_111-Entries_in_a_Type_1_font_dictionary.pm6) [Table 112 – Entries in a Type 3 font dictionary](gen/lib/ISO_32000/Table_112-Entries_in_a_Type_3_font_dictionary.pm6) [Table 117 – Entries in a CIDFont dictionary](gen/lib/ISO_32000/Table_117-Entries_in_a_CIDFont_dictionary.pm6)
/FontFamily|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.pm6)
/FontFauxing|[Table 366 – Additional entries specific to a trap network annotation](gen/lib/ISO_32000/Table_366-Additional_entries_specific_to_a_trap_network_annotation.pm6)
/FontFile|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.pm6) [Table 126 – Embedded font organization for various font types](gen/lib/ISO_32000/Table_126-Embedded_font_organization_for_various_font_types.pm6)
/FontFile2|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.pm6) [Table 126 – Embedded font organization for various font types](gen/lib/ISO_32000/Table_126-Embedded_font_organization_for_various_font_types.pm6)
/FontFile3|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.pm6) [Table 126 – Embedded font organization for various font types](gen/lib/ISO_32000/Table_126-Embedded_font_organization_for_various_font_types.pm6)
/FontMatrix|[Table 112 – Entries in a Type 3 font dictionary](gen/lib/ISO_32000/Table_112-Entries_in_a_Type_3_font_dictionary.pm6)
/FontName|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.pm6)
/FontStretch|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.pm6)
/FontWeight|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.pm6)
/Form|[Table 255 – Entries in the UR transform parameters dictionary](gen/lib/ISO_32000/Table_255-Entries_in_the_UR_transform_parameters_dictionary.pm6)
/FormType|[Table 95 – Additional Entries Specific to a Type 1 Form Dictionary](gen/lib/ISO_32000/Table_95-Additional_Entries_Specific_to_a_Type_1_Form_Dictionary.pm6)
/Frequency|[Table 130 – Entries in a type 1 halftone dictionary](gen/lib/ISO_32000/Table_130-Entries_in_a_type_1_halftone_dictionary.pm6)
/Function|[Table 79 – Additional Entries Specific to a Type 1 Shading Dictionary](gen/lib/ISO_32000/Table_79-Additional_Entries_Specific_to_a_Type_1_Shading_Dictionary.pm6) [Table 80 – Additional Entries Specific to a Type 2 Shading Dictionary](gen/lib/ISO_32000/Table_80-Additional_Entries_Specific_to_a_Type_2_Shading_Dictionary.pm6) [Table 81 – Additional Entries Specific to a Type 3 Shading Dictionary](gen/lib/ISO_32000/Table_81-Additional_Entries_Specific_to_a_Type_3_Shading_Dictionary.pm6) [Table 82 – Additional Entries Specific to a Type 4 Shading Dictionary](gen/lib/ISO_32000/Table_82-Additional_Entries_Specific_to_a_Type_4_Shading_Dictionary.pm6) [Table 83 – Additional Entries Specific to a Type 5 Shading Dictionary](gen/lib/ISO_32000/Table_83-Additional_Entries_Specific_to_a_Type_5_Shading_Dictionary.pm6) [Table 84 – Additional Entries Specific to a Type 6 Shading Dictionary](gen/lib/ISO_32000/Table_84-Additional_Entries_Specific_to_a_Type_6_Shading_Dictionary.pm6)
/FunctionType|[Table 38 – Entries common to all function dictionaries](gen/lib/ISO_32000/Table_38-Entries_common_to_all_function_dictionaries.pm6)
/Functions|[Table 41 – Additional entries specific to a type 3 function dictionary](gen/lib/ISO_32000/Table_41-Additional_entries_specific_to_a_type_3_function_dictionary.pm6)
/G|[Table 144 – Entries in a soft-mask dictionary](gen/lib/ISO_32000/Table_144-Entries_in_a_soft-mask_dictionary.pm6) [Table 359 – Entries in a Web Capture command settings dictionary](gen/lib/ISO_32000/Table_359-Entries_in_a_Web_Capture_command_settings_dictionary.pm6)
/Gamma|[Table 63 – Entries in a CalGray Colour Space Dictionary](gen/lib/ISO_32000/Table_63-Entries_in_a_CalGray_Colour_Space_Dictionary.pm6) [Table 64 – Entries in a CalRGB Colour Space Dictionary](gen/lib/ISO_32000/Table_64-Entries_in_a_CalRGB_Colour_Space_Dictionary.pm6)
/GlyphOrientationVertical|[Table 345 – Standard layout attributes specific to inline-level structure elements](gen/lib/ISO_32000/Table_345-Standard_layout_attributes_specific_to_inline-level_structure_elements.pm6)
/GoToRemoteActions|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.pm6)
/Group|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.pm6) [Table 95 – Additional Entries Specific to a Type 1 Form Dictionary](gen/lib/ISO_32000/Table_95-Additional_Entries_Specific_to_a_Type_1_Form_Dictionary.pm6)
/H|[Table 173 – Additional entries specific to a link annotation](gen/lib/ISO_32000/Table_173-Additional_entries_specific_to_a_link_annotation.pm6) [Table 188 – Additional entries specific to a widget annotation](gen/lib/ISO_32000/Table_188-Additional_entries_specific_to_a_widget_annotation.pm6) [Table 191 – Entries in a fixed print dictionary](gen/lib/ISO_32000/Table_191-Entries_in_a_fixed_print_dictionary.pm6) [Table 210 – Additional entries specific to a hide action](gen/lib/ISO_32000/Table_210-Additional_entries_specific_to_a_hide_action.pm6) [Table 292 – Entries in a software identifier dictionary](gen/lib/ISO_32000/Table_292-Entries_in_a_software_identifier_dictionary.pm6) [Table 329 – Entries in a user property dictionary](gen/lib/ISO_32000/Table_329-Entries_in_a_user_property_dictionary.pm6) [Table 357 – Entries in a Web Capture command dictionary](gen/lib/ISO_32000/Table_357-Entries_in_a_Web_Capture_command_dictionary.pm6) [Table F. 1 – Entries in the linearization parameter dictionary](gen/lib/ISO_32000/Table_F1-Entries_in_the_linearization_parameter_dictionary.pm6)
/HI|[Table 292 – Entries in a software identifier dictionary](gen/lib/ISO_32000/Table_292-Entries_in_a_software_identifier_dictionary.pm6)
/HT|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.pm6)
/HalftoneName|[Table 130 – Entries in a type 1 halftone dictionary](gen/lib/ISO_32000/Table_130-Entries_in_a_type_1_halftone_dictionary.pm6) [Table 131 – Additional entries specific to a type 6 halftone dictionary](gen/lib/ISO_32000/Table_131-Additional_entries_specific_to_a_type_6_halftone_dictionary.pm6) [Table 132 – Additional entries specific to a type 10 halftone dictionary](gen/lib/ISO_32000/Table_132-Additional_entries_specific_to_a_type_10_halftone_dictionary.pm6) [Table 133 – Additional entries specific to a type 16 halftone dictionary](gen/lib/ISO_32000/Table_133-Additional_entries_specific_to_a_type_16_halftone_dictionary.pm6) [Table 134 – Entries in a type 5 halftone dictionary](gen/lib/ISO_32000/Table_134-Entries_in_a_type_5_halftone_dictionary.pm6)
/HalftoneType|[Table 130 – Entries in a type 1 halftone dictionary](gen/lib/ISO_32000/Table_130-Entries_in_a_type_1_halftone_dictionary.pm6) [Table 131 – Additional entries specific to a type 6 halftone dictionary](gen/lib/ISO_32000/Table_131-Additional_entries_specific_to_a_type_6_halftone_dictionary.pm6) [Table 132 – Additional entries specific to a type 10 halftone dictionary](gen/lib/ISO_32000/Table_132-Additional_entries_specific_to_a_type_10_halftone_dictionary.pm6) [Table 133 – Additional entries specific to a type 16 halftone dictionary](gen/lib/ISO_32000/Table_133-Additional_entries_specific_to_a_type_16_halftone_dictionary.pm6) [Table 134 – Entries in a type 5 halftone dictionary](gen/lib/ISO_32000/Table_134-Entries_in_a_type_5_halftone_dictionary.pm6)
/Headers|[Table 349 – Standard table attributes](gen/lib/ISO_32000/Table_349-Standard_table_attributes.pm6)
/Height|[Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.pm6) [Table 131 – Additional entries specific to a type 6 halftone dictionary](gen/lib/ISO_32000/Table_131-Additional_entries_specific_to_a_type_6_halftone_dictionary.pm6) [Table 133 – Additional entries specific to a type 16 halftone dictionary](gen/lib/ISO_32000/Table_133-Additional_entries_specific_to_a_type_16_halftone_dictionary.pm6) [Table 344 – Additional standard layout attributes specific to block-level structure elements](gen/lib/ISO_32000/Table_344-Additional_standard_layout_attributes_specific_to_block-level_structure_elements.pm6)
/Height2|[Table 133 – Additional entries specific to a type 16 halftone dictionary](gen/lib/ISO_32000/Table_133-Additional_entries_specific_to_a_type_16_halftone_dictionary.pm6)
/HideAnnotationActions|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.pm6)
/HideMenubar|[Table 150 – Entries in a viewer preferences dictionary](gen/lib/ISO_32000/Table_150-Entries_in_a_viewer_preferences_dictionary.pm6)
/HideToolbar|[Table 150 – Entries in a viewer preferences dictionary](gen/lib/ISO_32000/Table_150-Entries_in_a_viewer_preferences_dictionary.pm6)
/HideWindowUI|[Table 150 – Entries in a viewer preferences dictionary](gen/lib/ISO_32000/Table_150-Entries_in_a_viewer_preferences_dictionary.pm6)
/I|[Table 147 – Additional entries specific to a transparency group attributes dictionary](gen/lib/ISO_32000/Table_147-Additional_entries_specific_to_a_transparency_group_attributes_dictionary.pm6) [Table 160 – Entries in a thread dictionary](gen/lib/ISO_32000/Table_160-Entries_in_a_thread_dictionary.pm6) [Table 167 – Entries in a border effect dictionary](gen/lib/ISO_32000/Table_167-Entries_in_a_border_effect_dictionary.pm6) [Table 189 – Entries in an appearance characteristics dictionary](gen/lib/ISO_32000/Table_189-Entries_in_an_appearance_characteristics_dictionary.pm6) [Table 231 – Additional entries specific to a choice field](gen/lib/ISO_32000/Table_231-Additional_entries_specific_to_a_choice_field.pm6)
/IC|[Table 175 – Additional entries specific to a line annotation](gen/lib/ISO_32000/Table_175-Additional_entries_specific_to_a_line_annotation.pm6) [Table 177 – Additional entries specific to a square or circle annotation](gen/lib/ISO_32000/Table_177-Additional_entries_specific_to_a_square_or_circle_annotation.pm6) [Table 178 – Additional entries specific to a polygon or polyline annotation](gen/lib/ISO_32000/Table_178-Additional_entries_specific_to_a_polygon_or_polyline_annotation.pm6) [Table 192 – Additional entries specific to a redaction annotation](gen/lib/ISO_32000/Table_192-Additional_entries_specific_to_a_redaction_annotation.pm6) [Table 311 – Entries in a 3D cross section dictionary](gen/lib/ISO_32000/Table_311-Entries_in_a_ThreeD_cross_section_dictionary.pm6)
/ID|[Table 15 – Entries in the file trailer dictionary](gen/lib/ISO_32000/Table_15-Entries_in_the_file_trailer_dictionary.pm6) [Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.pm6) [Table 44 – Entries in a file specification dictionary](gen/lib/ISO_32000/Table_44-Entries_in_a_file_specification_dictionary.pm6) [Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.pm6) [Table 97 – Entries in a Reference Dictionary](gen/lib/ISO_32000/Table_97-Entries_in_a_Reference_Dictionary.pm6) [Table 243 – Entries in the FDF dictionary](gen/lib/ISO_32000/Table_243-Entries_in_the_FDF_dictionary.pm6) [Table 323 – Entries in a structure element dictionary](gen/lib/ISO_32000/Table_323-Entries_in_a_structure_element_dictionary.pm6) [Table 352 – Entries common to all Web Capture content sets](gen/lib/ISO_32000/Table_352-Entries_common_to_all_Web_Capture_content_sets.pm6)
/IDS|[Table 31 – Entries in the name dictionary](gen/lib/ISO_32000/Table_31-Entries_in_the_name_dictionary.pm6)
/IDTree|[Table 322 – Entries in the structure tree root](gen/lib/ISO_32000/Table_322-Entries_in_the_structure_tree_root.pm6)
/IF|[Table 189 – Entries in an appearance characteristics dictionary](gen/lib/ISO_32000/Table_189-Entries_in_an_appearance_characteristics_dictionary.pm6) [Table 246 – Entries in an FDF field dictionary](gen/lib/ISO_32000/Table_246-Entries_in_an_FDF_field_dictionary.pm6)
/IN|[Table 304 – Entries in a 3D view dictionary](gen/lib/ISO_32000/Table_304-Entries_in_a_ThreeD_view_dictionary.pm6)
/IRT|[Table 170 – Additional entries specific to markup annotations](gen/lib/ISO_32000/Table_170-Additional_entries_specific_to_markup_annotations.pm6)
/IT|[Table 170 – Additional entries specific to markup annotations](gen/lib/ISO_32000/Table_170-Additional_entries_specific_to_markup_annotations.pm6) [Table 174 – Additional entries specific to a free text annotation](gen/lib/ISO_32000/Table_174-Additional_entries_specific_to_a_free_text_annotation.pm6) [Table 175 – Additional entries specific to a line annotation](gen/lib/ISO_32000/Table_175-Additional_entries_specific_to_a_line_annotation.pm6) [Table 178 – Additional entries specific to a polygon or polyline annotation](gen/lib/ISO_32000/Table_178-Additional_entries_specific_to_a_polygon_or_polyline_annotation.pm6)
/IV|[Table 311 – Entries in a 3D cross section dictionary](gen/lib/ISO_32000/Table_311-Entries_in_a_ThreeD_cross_section_dictionary.pm6)
/IX|[Table 189 – Entries in an appearance characteristics dictionary](gen/lib/ISO_32000/Table_189-Entries_in_an_appearance_characteristics_dictionary.pm6)
/Image|[Table 91 – Entries in an Alternate Image Dictionary](gen/lib/ISO_32000/Table_91-Entries_in_an_Alternate_Image_Dictionary.pm6)
/ImageMask|[Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.pm6)
/Index|[Table 17 – Additional entries specific to a cross-reference stream dictionary](gen/lib/ISO_32000/Table_17-Additional_entries_specific_to_a_cross-reference_stream_dictionary.pm6)
/Info|[Table 15 – Entries in the file trailer dictionary](gen/lib/ISO_32000/Table_15-Entries_in_the_file_trailer_dictionary.pm6) [Table 248 – Entries in an FDF page dictionary](gen/lib/ISO_32000/Table_248-Entries_in_an_FDF_page_dictionary.pm6) [Table 365 – Entries in an output intent dictionary](gen/lib/ISO_32000/Table_365-Entries_in_an_output_intent_dictionary.pm6)
/InkList|[Table 182 – Additional entries specific to an ink annotation](gen/lib/ISO_32000/Table_182-Additional_entries_specific_to_an_ink_annotation.pm6)
/InlineAlign|[Table 344 – Additional standard layout attributes specific to block-level structure elements](gen/lib/ISO_32000/Table_344-Additional_standard_layout_attributes_specific_to_block-level_structure_elements.pm6)
/Intent|[Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.pm6) [Table 98 – Entries in an Optional Content Group Dictionary](gen/lib/ISO_32000/Table_98-Entries_in_an_Optional_Content_Group_Dictionary.pm6) [Table 101 – Entries in an Optional Content Configuration Dictionary](gen/lib/ISO_32000/Table_101-Entries_in_an_Optional_Content_Configuration_Dictionary.pm6)
/Interpolate|[Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.pm6)
/IsMap|[Table 206 – Additional entries specific to a URI action](gen/lib/ISO_32000/Table_206-Additional_entries_specific_to_a_URI_action.pm6)
/Issuer|[Table 235 – Entries in a certificate seed value dictionary](gen/lib/ISO_32000/Table_235-Entries_in_a_certificate_seed_value_dictionary.pm6)
/ItalicAngle|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.pm6)
/JBIG2Globals|[Table 12 – Optional parameter for the JBIG2Decode filter](gen/lib/ISO_32000/Table_12-Optional_parameter_for_the_JBIG2Decode_filter.pm6)
/JS|[Table 214 – Additional entries specific to a rendition action](gen/lib/ISO_32000/Table_214-Additional_entries_specific_to_a_rendition_action.pm6) [Table 217 – Additional entries specific to a JavaScript action](gen/lib/ISO_32000/Table_217-Additional_entries_specific_to_a_JavaScript_action.pm6)
/JavaScript|[Table 31 – Entries in the name dictionary](gen/lib/ISO_32000/Table_31-Entries_in_the_name_dictionary.pm6) [Table 243 – Entries in the FDF dictionary](gen/lib/ISO_32000/Table_243-Entries_in_the_FDF_dictionary.pm6)
/JavaScriptActions|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.pm6)
/K|[Table 11 – Optional parameters for the CCITTFaxDecode filter](gen/lib/ISO_32000/Table_11-Optional_parameters_for_the_CCITTFaxDecode_filter.pm6) [Table 147 – Additional entries specific to a transparency group attributes dictionary](gen/lib/ISO_32000/Table_147-Additional_entries_specific_to_a_transparency_group_attributes_dictionary.pm6) [Table 196 – Entries in a form field’s additional-actions dictionary](gen/lib/ISO_32000/Table_196-Entries_in_a_form_fields_additional-actions_dictionary.pm6) [Table 322 – Entries in the structure tree root](gen/lib/ISO_32000/Table_322-Entries_in_the_structure_tree_root.pm6) [Table 323 – Entries in a structure element dictionary](gen/lib/ISO_32000/Table_323-Entries_in_a_structure_element_dictionary.pm6)
/KeyUsage|[Table 235 – Entries in a certificate seed value dictionary](gen/lib/ISO_32000/Table_235-Entries_in_a_certificate_seed_value_dictionary.pm6)
/Keywords|[Table 317 – Entries in the document information dictionary](gen/lib/ISO_32000/Table_317-Entries_in_the_document_information_dictionary.pm6)
/Kids|[Table 29 – Required entries in a page tree node](gen/lib/ISO_32000/Table_29-Required_entries_in_a_page_tree_node.pm6) [Table 36 – Entries in a name tree node dictionary](gen/lib/ISO_32000/Table_36-Entries_in_a_name_tree_node_dictionary.pm6) [Table 37 – Entries in a number tree node dictionary](gen/lib/ISO_32000/Table_37-Entries_in_a_number_tree_node_dictionary.pm6) [Table 220 – Entries common to all field dictionaries](gen/lib/ISO_32000/Table_220-Entries_common_to_all_field_dictionaries.pm6) [Table 246 – Entries in an FDF field dictionary](gen/lib/ISO_32000/Table_246-Entries_in_an_FDF_field_dictionary.pm6)
/L|[Table 175 – Additional entries specific to a line annotation](gen/lib/ISO_32000/Table_175-Additional_entries_specific_to_a_line_annotation.pm6) [Table 268 – Entries in a media criteria dictionary](gen/lib/ISO_32000/Table_268-Entries_in_a_media_criteria_dictionary.pm6) [Table 292 – Entries in a software identifier dictionary](gen/lib/ISO_32000/Table_292-Entries_in_a_software_identifier_dictionary.pm6) [Table 357 – Entries in a Web Capture command dictionary](gen/lib/ISO_32000/Table_357-Entries_in_a_Web_Capture_command_dictionary.pm6) [Table F. 1 – Entries in the linearization parameter dictionary](gen/lib/ISO_32000/Table_F1-Entries_in_the_linearization_parameter_dictionary.pm6)
/LC|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.pm6)
/LE|[Table 174 – Additional entries specific to a free text annotation](gen/lib/ISO_32000/Table_174-Additional_entries_specific_to_a_free_text_annotation.pm6) [Table 175 – Additional entries specific to a line annotation](gen/lib/ISO_32000/Table_175-Additional_entries_specific_to_a_line_annotation.pm6) [Table 178 – Additional entries specific to a polygon or polyline annotation](gen/lib/ISO_32000/Table_178-Additional_entries_specific_to_a_polygon_or_polyline_annotation.pm6)
/LI|[Table 292 – Entries in a software identifier dictionary](gen/lib/ISO_32000/Table_292-Entries_in_a_software_identifier_dictionary.pm6)
/LJ|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.pm6)
/LL|[Table 175 – Additional entries specific to a line annotation](gen/lib/ISO_32000/Table_175-Additional_entries_specific_to_a_line_annotation.pm6)
/LLE|[Table 175 – Additional entries specific to a line annotation](gen/lib/ISO_32000/Table_175-Additional_entries_specific_to_a_line_annotation.pm6)
/LLO|[Table 175 – Additional entries specific to a line annotation](gen/lib/ISO_32000/Table_175-Additional_entries_specific_to_a_line_annotation.pm6)
/LS|[Table 304 – Entries in a 3D view dictionary](gen/lib/ISO_32000/Table_304-Entries_in_a_ThreeD_view_dictionary.pm6)
/LW|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.pm6)
/Lang|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.pm6) [Table 124 – Additional font descriptor entries for CIDFonts](gen/lib/ISO_32000/Table_124-Additional_font_descriptor_entries_for_CIDFonts.pm6) [Table 323 – Entries in a structure element dictionary](gen/lib/ISO_32000/Table_323-Entries_in_a_structure_element_dictionary.pm6)
/Language|[Table 102 – Entries in an Optional Content Usage Dictionary](gen/lib/ISO_32000/Table_102-Entries_in_an_Optional_Content_Usage_Dictionary.pm6)
/Last|[Table 152 – Entries in the outline dictionary](gen/lib/ISO_32000/Table_152-Entries_in_the_outline_dictionary.pm6) [Table 153 – Entries in an outline item dictionary](gen/lib/ISO_32000/Table_153-Entries_in_an_outline_item_dictionary.pm6)
/LastChar|[Table 111 – Entries in a Type 1 font dictionary](gen/lib/ISO_32000/Table_111-Entries_in_a_Type_1_font_dictionary.pm6) [Table 112 – Entries in a Type 3 font dictionary](gen/lib/ISO_32000/Table_112-Entries_in_a_Type_3_font_dictionary.pm6)
/LastModified|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.pm6) [Table 95 – Additional Entries Specific to a Type 1 Form Dictionary](gen/lib/ISO_32000/Table_95-Additional_Entries_Specific_to_a_Type_1_Form_Dictionary.pm6) [Table 319 – Entries in an data dictionary](gen/lib/ISO_32000/Table_319-Entries_in_an_data_dictionary.pm6) [Table 366 – Additional entries specific to a trap network annotation](gen/lib/ISO_32000/Table_366-Additional_entries_specific_to_a_trap_network_annotation.pm6)
/LaunchActions|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.pm6)
/Leading|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.pm6)
/Legal|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.pm6)
/LegalAttestation|[Table 234 – Entries in a signature field seed value dictionary](gen/lib/ISO_32000/Table_234-Entries_in_a_signature_field_seed_value_dictionary.pm6)
/Length|[Table 5 – Entries common to all stream dictionaries](gen/lib/ISO_32000/Table_5-Entries_common_to_all_stream_dictionaries.pm6) [Table 20 – Entries common to all encryption dictionaries](gen/lib/ISO_32000/Table_20-Entries_common_to_all_encryption_dictionaries.pm6) [Table 25 – Entries common to all crypt filter dictionaries](gen/lib/ISO_32000/Table_25-Entries_common_to_all_crypt_filter_dictionaries.pm6)
/Length1|[Table 127 – Additional entries in an embedded font stream dictionary](gen/lib/ISO_32000/Table_127-Additional_entries_in_an_embedded_font_stream_dictionary.pm6)
/Length2|[Table 127 – Additional entries in an embedded font stream dictionary](gen/lib/ISO_32000/Table_127-Additional_entries_in_an_embedded_font_stream_dictionary.pm6)
/Length3|[Table 127 – Additional entries in an embedded font stream dictionary](gen/lib/ISO_32000/Table_127-Additional_entries_in_an_embedded_font_stream_dictionary.pm6)
/Level1|[Table 88 – Additional Entries Specific to a PostScript XObject Dictionary](gen/lib/ISO_32000/Table_88-Additional_Entries_Specific_to_a_PostScript_XObject_Dictionary.pm6)
/Limits|[Table 36 – Entries in a name tree node dictionary](gen/lib/ISO_32000/Table_36-Entries_in_a_name_tree_node_dictionary.pm6) [Table 37 – Entries in a number tree node dictionary](gen/lib/ISO_32000/Table_37-Entries_in_a_number_tree_node_dictionary.pm6)
/LineHeight|[Table 345 – Standard layout attributes specific to inline-level structure elements](gen/lib/ISO_32000/Table_345-Standard_layout_attributes_specific_to_inline-level_structure_elements.pm6)
/Linearized|[Table F. 1 – Entries in the linearization parameter dictionary](gen/lib/ISO_32000/Table_F1-Entries_in_the_linearization_parameter_dictionary.pm6)
/ListMode|[Table 101 – Entries in an Optional Content Configuration Dictionary](gen/lib/ISO_32000/Table_101-Entries_in_an_Optional_Content_Configuration_Dictionary.pm6)
/ListNumbering|[Table 347 – Standard list attribute](gen/lib/ISO_32000/Table_347-Standard_list_attribute.pm6)
/Location|[Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.pm6)
/Lock|[Table 232 – Additional entries specific to a signature field](gen/lib/ISO_32000/Table_232-Additional_entries_specific_to_a_signature_field.pm6)
/Locked|[Table 101 – Entries in an Optional Content Configuration Dictionary](gen/lib/ISO_32000/Table_101-Entries_in_an_Optional_Content_Configuration_Dictionary.pm6)
/M|[Table 162 – Entries in a transition dictionary](gen/lib/ISO_32000/Table_162-Entries_in_a_transition_dictionary.pm6) [Table 164 – Entries common to all annotation dictionaries](gen/lib/ISO_32000/Table_164-Entries_common_to_all_annotation_dictionaries.pm6) [Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.pm6) [Table 269 – Entries in a minimum bit depth dictionary](gen/lib/ISO_32000/Table_269-Entries_in_a_minimum_bit_depth_dictionary.pm6) [Table 270 – Entries in a minimum screen size dictionary](gen/lib/ISO_32000/Table_270-Entries_in_a_minimum_screen_size_dictionary.pm6) [Table 283 – Entries in a media screen parameters MH/BE dictionary](gen/lib/ISO_32000/Table_283-Entries_in_a_media_screen_parameters_MH-BE_dictionary.pm6) [Table 288 – Additional entries in a media offset marker dictionary](gen/lib/ISO_32000/Table_288-Additional_entries_in_a_media_offset_marker_dictionary.pm6) [Table 312 – Entries in a 3D node dictionary](gen/lib/ISO_32000/Table_312-Entries_in_a_ThreeD_node_dictionary.pm6)
/MCID|[Table 324 – Entries in a marked-content reference dictionary](gen/lib/ISO_32000/Table_324-Entries_in_a_marked-content_reference_dictionary.pm6)
/MD5|[Table 313 – Entries in an external data dictionary used to markup 3D annotations](gen/lib/ISO_32000/Table_313-Entries_in_an_external_data_dictionary_used_to_markup_ThreeD_annotations.pm6)
/MDP|[Table 234 – Entries in a signature field seed value dictionary](gen/lib/ISO_32000/Table_234-Entries_in_a_signature_field_seed_value_dictionary.pm6)
/MH|[Table 266 – Entries common to all rendition dictionaries](gen/lib/ISO_32000/Table_266-Entries_common_to_all_rendition_dictionaries.pm6) [Table 274 – Additional entries in a media clip data dictionary](gen/lib/ISO_32000/Table_274-Additional_entries_in_a_media_clip_data_dictionary.pm6) [Table 277 – Additional entries in a media clip section dictionary](gen/lib/ISO_32000/Table_277-Additional_entries_in_a_media_clip_section_dictionary.pm6) [Table 279 – Entries in a media play parameters dictionary](gen/lib/ISO_32000/Table_279-Entries_in_a_media_play_parameters_dictionary.pm6) [Table 282 – Entries in a media screen parameters dictionary](gen/lib/ISO_32000/Table_282-Entries_in_a_media_screen_parameters_dictionary.pm6) [Table 291 – Entries in a media player info dictionary](gen/lib/ISO_32000/Table_291-Entries_in_a_media_player_info_dictionary.pm6)
/MK|[Table 187 – Additional entries specific to a screen annotation](gen/lib/ISO_32000/Table_187-Additional_entries_specific_to_a_screen_annotation.pm6) [Table 188 – Additional entries specific to a widget annotation](gen/lib/ISO_32000/Table_188-Additional_entries_specific_to_a_widget_annotation.pm6)
/ML|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.pm6)
/MN|[Table 362 – Additional entries specific to a printer’s mark annotation](gen/lib/ISO_32000/Table_362-Additional_entries_specific_to_a_printers_mark_annotation.pm6)
/MS|[Table 304 – Entries in a 3D view dictionary](gen/lib/ISO_32000/Table_304-Entries_in_a_ThreeD_view_dictionary.pm6)
/MU|[Table 290 – Entries in a media players dictionary](gen/lib/ISO_32000/Table_290-Entries_in_a_media_players_dictionary.pm6)
/Mac|[Table 44 – Entries in a file specification dictionary](gen/lib/ISO_32000/Table_44-Entries_in_a_file_specification_dictionary.pm6) [Table 46 – Entries in an embedded file parameter dictionary](gen/lib/ISO_32000/Table_46-Entries_in_an_embedded_file_parameter_dictionary.pm6) [Table 203 – Additional entries specific to a launch action](gen/lib/ISO_32000/Table_203-Additional_entries_specific_to_a_launch_action.pm6)
/MarkInfo|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.pm6)
/MarkStyle|[Table 363 – Additional entries specific to a printer’s mark form dictionary](gen/lib/ISO_32000/Table_363-Additional_entries_specific_to_a_printers_mark_form_dictionary.pm6)
/Marked|[Table 321 – Entries in the mark information dictionary](gen/lib/ISO_32000/Table_321-Entries_in_the_mark_information_dictionary.pm6)
/Mask|[Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.pm6)
/Matrix|[Table 64 – Entries in a CalRGB Colour Space Dictionary](gen/lib/ISO_32000/Table_64-Entries_in_a_CalRGB_Colour_Space_Dictionary.pm6) [Table 75 – Additional Entries Specific to a Type 1 Pattern Dictionary](gen/lib/ISO_32000/Table_75-Additional_Entries_Specific_to_a_Type_1_Pattern_Dictionary.pm6) [Table 76 – Entries in a Type 2 Pattern Dictionary](gen/lib/ISO_32000/Table_76-Entries_in_a_Type_2_Pattern_Dictionary.pm6) [Table 79 – Additional Entries Specific to a Type 1 Shading Dictionary](gen/lib/ISO_32000/Table_79-Additional_Entries_Specific_to_a_Type_1_Shading_Dictionary.pm6) [Table 95 – Additional Entries Specific to a Type 1 Form Dictionary](gen/lib/ISO_32000/Table_95-Additional_Entries_Specific_to_a_Type_1_Form_Dictionary.pm6) [Table 191 – Entries in a fixed print dictionary](gen/lib/ISO_32000/Table_191-Entries_in_a_fixed_print_dictionary.pm6)
/Matte|[Table 146 – Additional entry in a soft-mask image dictionary](gen/lib/ISO_32000/Table_146-Additional_entry_in_a_soft-mask_image_dictionary.pm6)
/MaxLen|[Table 229 – Additional entry specific to a text field](gen/lib/ISO_32000/Table_229-Additional_entry_specific_to_a_text_field.pm6)
/MaxWidth|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.pm6)
/Measure|[Table 175 – Additional entries specific to a line annotation](gen/lib/ISO_32000/Table_175-Additional_entries_specific_to_a_line_annotation.pm6) [Table 178 – Additional entries specific to a polygon or polyline annotation](gen/lib/ISO_32000/Table_178-Additional_entries_specific_to_a_polygon_or_polyline_annotation.pm6) [Table 260 – Entries in a viewport dictionary](gen/lib/ISO_32000/Table_260-Entries_in_a_viewport_dictionary.pm6)
/MediaBox|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.pm6)
/Metadata|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.pm6) [Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.pm6) [Table 66 – Additional Entries Specific to an ICC Profile Stream Dictionary](gen/lib/ISO_32000/Table_66-Additional_Entries_Specific_to_an_ICC_Profile_Stream_Dictionary.pm6) [Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.pm6) [Table 95 – Additional Entries Specific to a Type 1 Form Dictionary](gen/lib/ISO_32000/Table_95-Additional_Entries_Specific_to_a_Type_1_Form_Dictionary.pm6) [Table 127 – Additional entries in an embedded font stream dictionary](gen/lib/ISO_32000/Table_127-Additional_entries_in_an_embedded_font_stream_dictionary.pm6) [Table 316 – Additional entry for components having metadata](gen/lib/ISO_32000/Table_316-Additional_entry_for_components_having_metadata.pm6)
/MissingWidth|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.pm6)
/Mix|[Table 208 – Additional entries specific to a sound action](gen/lib/ISO_32000/Table_208-Additional_entries_specific_to_a_sound_action.pm6)
/MixingHints|[Table 71 – Entries in a DeviceN Colour Space Attributes Dictionary](gen/lib/ISO_32000/Table_71-Entries_in_a_DeviceN_Colour_Space_Attributes_Dictionary.pm6)
/ModDate|[Table 46 – Entries in an embedded file parameter dictionary](gen/lib/ISO_32000/Table_46-Entries_in_an_embedded_file_parameter_dictionary.pm6) [Table 317 – Entries in the document information dictionary](gen/lib/ISO_32000/Table_317-Entries_in_the_document_information_dictionary.pm6)
/Mode|[Table 296 – Entries in a movie activation dictionary](gen/lib/ISO_32000/Table_296-Entries_in_a_movie_activation_dictionary.pm6)
/Movie|[Table 186 – Additional entries specific to a movie annotation](gen/lib/ISO_32000/Table_186-Additional_entries_specific_to_a_movie_annotation.pm6)
/MovieActions|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.pm6)
/Msg|[Table 255 – Entries in the UR transform parameters dictionary](gen/lib/ISO_32000/Table_255-Entries_in_the_UR_transform_parameters_dictionary.pm6)
/N|[Table 16 – Additional entries specific to an object stream dictionary](gen/lib/ISO_32000/Table_16-Additional_entries_specific_to_an_object_stream_dictionary.pm6) [Table 40 – Additional entries specific to a type 2 function dictionary](gen/lib/ISO_32000/Table_40-Additional_entries_specific_to_a_type_2_function_dictionary.pm6) [Table 66 – Additional Entries Specific to an ICC Profile Stream Dictionary](gen/lib/ISO_32000/Table_66-Additional_Entries_Specific_to_an_ICC_Profile_Stream_Dictionary.pm6) [Table 157 – Entries in a collection field dictionary](gen/lib/ISO_32000/Table_157-Entries_in_a_collection_field_dictionary.pm6) [Table 161 – Entries in a bead dictionary](gen/lib/ISO_32000/Table_161-Entries_in_a_bead_dictionary.pm6) [Table 168 – Entries in an appearance dictionary](gen/lib/ISO_32000/Table_168-Entries_in_an_appearance_dictionary.pm6) [Table 202 – Entries specific to a target dictionary](gen/lib/ISO_32000/Table_202-Entries_specific_to_a_target_dictionary.pm6) [Table 212 – Additional entries specific to named actions](gen/lib/ISO_32000/Table_212-Additional_entries_specific_to_named_actions.pm6) [Table 266 – Entries common to all rendition dictionaries](gen/lib/ISO_32000/Table_266-Entries_common_to_all_rendition_dictionaries.pm6) [Table 273 – Entries common to all media clip dictionaries](gen/lib/ISO_32000/Table_273-Entries_common_to_all_media_clip_dictionaries.pm6) [Table 305 – Entries in a projection dictionary](gen/lib/ISO_32000/Table_305-Entries_in_a_projection_dictionary.pm6) [Table 312 – Entries in a 3D node dictionary](gen/lib/ISO_32000/Table_312-Entries_in_a_ThreeD_node_dictionary.pm6) [Table 329 – Entries in a user property dictionary](gen/lib/ISO_32000/Table_329-Entries_in_a_user_property_dictionary.pm6) [Table F. 1 – Entries in the linearization parameter dictionary](gen/lib/ISO_32000/Table_F1-Entries_in_the_linearization_parameter_dictionary.pm6)
/NA|[Table 163 – Entries in a navigation node dictionary](gen/lib/ISO_32000/Table_163-Entries_in_a_navigation_node_dictionary.pm6) [Table 304 – Entries in a 3D view dictionary](gen/lib/ISO_32000/Table_304-Entries_in_a_ThreeD_view_dictionary.pm6)
/NM|[Table 164 – Entries common to all annotation dictionaries](gen/lib/ISO_32000/Table_164-Entries_common_to_all_annotation_dictionaries.pm6)
/NP|[Table 299 – Entries in a 3D activation dictionary](gen/lib/ISO_32000/Table_299-Entries_in_a_ThreeD_activation_dictionary.pm6)
/NR|[Table 304 – Entries in a 3D view dictionary](gen/lib/ISO_32000/Table_304-Entries_in_a_ThreeD_view_dictionary.pm6)
/NU|[Table 290 – Entries in a media players dictionary](gen/lib/ISO_32000/Table_290-Entries_in_a_media_players_dictionary.pm6)
/Name|[Table 14 – Optional parameters for Crypt filters](gen/lib/ISO_32000/Table_14-Optional_parameters_for_Crypt_filters.pm6) [Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.pm6) [Table 95 – Additional Entries Specific to a Type 1 Form Dictionary](gen/lib/ISO_32000/Table_95-Additional_Entries_Specific_to_a_Type_1_Form_Dictionary.pm6) [Table 98 – Entries in an Optional Content Group Dictionary](gen/lib/ISO_32000/Table_98-Entries_in_an_Optional_Content_Group_Dictionary.pm6) [Table 101 – Entries in an Optional Content Configuration Dictionary](gen/lib/ISO_32000/Table_101-Entries_in_an_Optional_Content_Configuration_Dictionary.pm6) [Table 111 – Entries in a Type 1 font dictionary](gen/lib/ISO_32000/Table_111-Entries_in_a_Type_1_font_dictionary.pm6) [Table 112 – Entries in a Type 3 font dictionary](gen/lib/ISO_32000/Table_112-Entries_in_a_Type_3_font_dictionary.pm6) [Table 172 – Additional entries specific to a text annotation](gen/lib/ISO_32000/Table_172-Additional_entries_specific_to_a_text_annotation.pm6) [Table 181 – Additional entries specific to a rubber stamp annotation](gen/lib/ISO_32000/Table_181-Additional_entries_specific_to_a_rubber_stamp_annotation.pm6) [Table 184 – Additional entries specific to a file attachment annotation](gen/lib/ISO_32000/Table_184-Additional_entries_specific_to_a_file_attachment_annotation.pm6) [Table 185 – Additional entries specific to a sound annotation](gen/lib/ISO_32000/Table_185-Additional_entries_specific_to_a_sound_annotation.pm6) [Table 250 – Entries in an FDF named page reference dictionary](gen/lib/ISO_32000/Table_250-Entries_in_an_FDF_named_page_reference_dictionary.pm6) [Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.pm6) [Table 260 – Entries in a viewport dictionary](gen/lib/ISO_32000/Table_260-Entries_in_a_viewport_dictionary.pm6)
/Names|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.pm6) [Table 36 – Entries in a name tree node dictionary](gen/lib/ISO_32000/Table_36-Entries_in_a_name_tree_node_dictionary.pm6)
/NeedAppearances|[Table 218 – Entries in the interactive form dictionary](gen/lib/ISO_32000/Table_218-Entries_in_the_interactive_form_dictionary.pm6)
/NeedsRendering|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.pm6)
/NewWindow|[Table 200 – Additional entries specific to a remote go-to action](gen/lib/ISO_32000/Table_200-Additional_entries_specific_to_a_remote_go-to_action.pm6) [Table 201 – Additional entries specific to an embedded go-to action](gen/lib/ISO_32000/Table_201-Additional_entries_specific_to_an_embedded_go-to_action.pm6) [Table 203 – Additional entries specific to a launch action](gen/lib/ISO_32000/Table_203-Additional_entries_specific_to_a_launch_action.pm6)
/Next|[Table 153 – Entries in an outline item dictionary](gen/lib/ISO_32000/Table_153-Entries_in_an_outline_item_dictionary.pm6) [Table 163 – Entries in a navigation node dictionary](gen/lib/ISO_32000/Table_163-Entries_in_a_navigation_node_dictionary.pm6) [Table 193 – Entries common to all action dictionaries](gen/lib/ISO_32000/Table_193-Entries_common_to_all_action_dictionaries.pm6)
/NonEmbeddedFonts|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.pm6)
/NonFullScreenPageMode|[Table 150 – Entries in a viewer preferences dictionary](gen/lib/ISO_32000/Table_150-Entries_in_a_viewer_preferences_dictionary.pm6)
/NumCopies|[Table 150 – Entries in a viewer preferences dictionary](gen/lib/ISO_32000/Table_150-Entries_in_a_viewer_preferences_dictionary.pm6)
/Nums|[Table 37 – Entries in a number tree node dictionary](gen/lib/ISO_32000/Table_37-Entries_in_a_number_tree_node_dictionary.pm6)
/O|[Table 21 – Additional encryption dictionary entries for the standard security handler](gen/lib/ISO_32000/Table_21-Additional_encryption_dictionary_entries_for_the_standard_security_handler.pm6) [Table 157 – Entries in a collection field dictionary](gen/lib/ISO_32000/Table_157-Entries_in_a_collection_field_dictionary.pm6) [Table 195 – Entries in a page object’s additional-actions dictionary](gen/lib/ISO_32000/Table_195-Entries_in_a_page_objects_additional-actions_dictionary.pm6) [Table 204 – Entries in a Windows launch parameter dictionary](gen/lib/ISO_32000/Table_204-Entries_in_a_Windows_launch_parameter_dictionary.pm6) [Table 262 – Additional entries in a rectilinear measure dictionary](gen/lib/ISO_32000/Table_262-Additional_entries_in_a_rectilinear_measure_dictionary.pm6) [Table 263 – Entries in a number format dictionary](gen/lib/ISO_32000/Table_263-Entries_in_a_number_format_dictionary.pm6) [Table 268 – Entries in a media criteria dictionary](gen/lib/ISO_32000/Table_268-Entries_in_a_media_criteria_dictionary.pm6) [Table 283 – Entries in a media screen parameters MH/BE dictionary](gen/lib/ISO_32000/Table_283-Entries_in_a_media_screen_parameters_MH-BE_dictionary.pm6) [Table 284 – Entries in a floating window parameters dictionary](gen/lib/ISO_32000/Table_284-Entries_in_a_floating_window_parameters_dictionary.pm6) [Table 304 – Entries in a 3D view dictionary](gen/lib/ISO_32000/Table_304-Entries_in_a_ThreeD_view_dictionary.pm6) [Table 307 – Entries in a render mode dictionary](gen/lib/ISO_32000/Table_307-Entries_in_a_render_mode_dictionary.pm6) [Table 311 – Entries in a 3D cross section dictionary](gen/lib/ISO_32000/Table_311-Entries_in_a_ThreeD_cross_section_dictionary.pm6) [Table 312 – Entries in a 3D node dictionary](gen/lib/ISO_32000/Table_312-Entries_in_a_ThreeD_node_dictionary.pm6) [Table 327 – Entry common to all attribute object dictionaries](gen/lib/ISO_32000/Table_327-Entry_common_to_all_attribute_object_dictionaries.pm6) [Table 328 – Additional entries in an attribute object dictionary for user properties](gen/lib/ISO_32000/Table_328-Additional_entries_in_an_attribute_object_dictionary_for_user_properties.pm6) [Table 352 – Entries common to all Web Capture content sets](gen/lib/ISO_32000/Table_352-Entries_common_to_all_Web_Capture_content_sets.pm6) [Table F. 1 – Entries in the linearization parameter dictionary](gen/lib/ISO_32000/Table_F1-Entries_in_the_linearization_parameter_dictionary.pm6)
/OB|[Table 305 – Entries in a projection dictionary](gen/lib/ISO_32000/Table_305-Entries_in_a_projection_dictionary.pm6)
/OC|[Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.pm6) [Table 91 – Entries in an Alternate Image Dictionary](gen/lib/ISO_32000/Table_91-Entries_in_an_Alternate_Image_Dictionary.pm6) [Table 95 – Additional Entries Specific to a Type 1 Form Dictionary](gen/lib/ISO_32000/Table_95-Additional_Entries_Specific_to_a_Type_1_Form_Dictionary.pm6) [Table 164 – Entries common to all annotation dictionaries](gen/lib/ISO_32000/Table_164-Entries_common_to_all_annotation_dictionaries.pm6)
/OCGs|[Table 99 – Entries in an Optional Content Membership Dictionary](gen/lib/ISO_32000/Table_99-Entries_in_an_Optional_Content_Membership_Dictionary.pm6) [Table 100 – Entries in the Optional Content Properties Dictionary](gen/lib/ISO_32000/Table_100-Entries_in_the_Optional_Content_Properties_Dictionary.pm6) [Table 103 – Entries in a Usage Application Dictionary](gen/lib/ISO_32000/Table_103-Entries_in_a_Usage_Application_Dictionary.pm6)
/OCProperties|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.pm6)
/OFF|[Table 101 – Entries in an Optional Content Configuration Dictionary](gen/lib/ISO_32000/Table_101-Entries_in_an_Optional_Content_Configuration_Dictionary.pm6)
/OID|[Table 235 – Entries in a certificate seed value dictionary](gen/lib/ISO_32000/Table_235-Entries_in_a_certificate_seed_value_dictionary.pm6)
/ON|[Table 101 – Entries in an Optional Content Configuration Dictionary](gen/lib/ISO_32000/Table_101-Entries_in_an_Optional_Content_Configuration_Dictionary.pm6)
/OP|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.pm6) [Table 214 – Additional entries specific to a rendition action](gen/lib/ISO_32000/Table_214-Additional_entries_specific_to_a_rendition_action.pm6)
/OPI|[Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.pm6) [Table 95 – Additional Entries Specific to a Type 1 Form Dictionary](gen/lib/ISO_32000/Table_95-Additional_Entries_Specific_to_a_Type_1_Form_Dictionary.pm6)
/OPM|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.pm6)
/OS|[Table 292 – Entries in a software identifier dictionary](gen/lib/ISO_32000/Table_292-Entries_in_a_software_identifier_dictionary.pm6) [Table 305 – Entries in a projection dictionary](gen/lib/ISO_32000/Table_305-Entries_in_a_projection_dictionary.pm6)
/Obj|[Table 325 – Entries in an object reference dictionary](gen/lib/ISO_32000/Table_325-Entries_in_an_object_reference_dictionary.pm6)
/OnInstantiate|[Table 300 – Entries in a 3D stream dictionary](gen/lib/ISO_32000/Table_300-Entries_in_a_ThreeD_stream_dictionary.pm6)
/Open|[Table 172 – Additional entries specific to a text annotation](gen/lib/ISO_32000/Table_172-Additional_entries_specific_to_a_text_annotation.pm6) [Table 183 – Additional entries specific to a pop-up annotation](gen/lib/ISO_32000/Table_183-Additional_entries_specific_to_a_pop-up_annotation.pm6)
/OpenAction|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.pm6)
/Operation|[Table 209 – Additional entries specific to a movie action](gen/lib/ISO_32000/Table_209-Additional_entries_specific_to_a_movie_action.pm6)
/Opt|[Table 227 – Additional entry specific to check box and radio button fields](gen/lib/ISO_32000/Table_227-Additional_entry_specific_to_check_box_and_radio_button_fields.pm6) [Table 231 – Additional entries specific to a choice field](gen/lib/ISO_32000/Table_231-Additional_entries_specific_to_a_choice_field.pm6) [Table 246 – Entries in an FDF field dictionary](gen/lib/ISO_32000/Table_246-Entries_in_an_FDF_field_dictionary.pm6)
/OptionalContent|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.pm6)
/Order|[Table 39 – Additional entries specific to a type 0 function dictionary](gen/lib/ISO_32000/Table_39-Additional_entries_specific_to_a_type_0_function_dictionary.pm6) [Table 101 – Entries in an Optional Content Configuration Dictionary](gen/lib/ISO_32000/Table_101-Entries_in_an_Optional_Content_Configuration_Dictionary.pm6)
/Ordering|[Table 116 – Entries in a CIDSystemInfo dictionary](gen/lib/ISO_32000/Table_116-Entries_in_a_CIDSystemInfo_dictionary.pm6)
/Outlines|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.pm6)
/OutputCondition|[Table 365 – Entries in an output intent dictionary](gen/lib/ISO_32000/Table_365-Entries_in_an_output_intent_dictionary.pm6)
/OutputConditionIdentifier|[Table 365 – Entries in an output intent dictionary](gen/lib/ISO_32000/Table_365-Entries_in_an_output_intent_dictionary.pm6)
/OutputIntents|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.pm6)
/OverlayText|[Table 192 – Additional entries specific to a redaction annotation](gen/lib/ISO_32000/Table_192-Additional_entries_specific_to_a_redaction_annotation.pm6)
/P|[Table 21 – Additional encryption dictionary entries for the standard security handler](gen/lib/ISO_32000/Table_21-Additional_encryption_dictionary_entries_for_the_standard_security_handler.pm6) [Table 23 – Additional encryption dictionary entries for public-key security handlers](gen/lib/ISO_32000/Table_23-Additional_encryption_dictionary_entries_for_public-key_security_handlers.pm6) [Table 49 – Entries in a collection subitem dictionary](gen/lib/ISO_32000/Table_49-Entries_in_a_collection_subitem_dictionary.pm6) [Table 99 – Entries in an Optional Content Membership Dictionary](gen/lib/ISO_32000/Table_99-Entries_in_an_Optional_Content_Membership_Dictionary.pm6) [Table 159 – Entries in a page label dictionary](gen/lib/ISO_32000/Table_159-Entries_in_a_page_label_dictionary.pm6) [Table 161 – Entries in a bead dictionary](gen/lib/ISO_32000/Table_161-Entries_in_a_bead_dictionary.pm6) [Table 164 – Entries common to all annotation dictionaries](gen/lib/ISO_32000/Table_164-Entries_common_to_all_annotation_dictionaries.pm6) [Table 202 – Entries specific to a target dictionary](gen/lib/ISO_32000/Table_202-Entries_specific_to_a_target_dictionary.pm6) [Table 204 – Entries in a Windows launch parameter dictionary](gen/lib/ISO_32000/Table_204-Entries_in_a_Windows_launch_parameter_dictionary.pm6) [Table 254 – Entries in the DocMDP transform parameters dictionary](gen/lib/ISO_32000/Table_254-Entries_in_the_DocMDP_transform_parameters_dictionary.pm6) [Table 255 – Entries in the UR transform parameters dictionary](gen/lib/ISO_32000/Table_255-Entries_in_the_UR_transform_parameters_dictionary.pm6) [Table 268 – Entries in a media criteria dictionary](gen/lib/ISO_32000/Table_268-Entries_in_a_media_criteria_dictionary.pm6) [Table 271 – Additional entries in a media rendition dictionary](gen/lib/ISO_32000/Table_271-Additional_entries_in_a_media_rendition_dictionary.pm6) [Table 274 – Additional entries in a media clip data dictionary](gen/lib/ISO_32000/Table_274-Additional_entries_in_a_media_clip_data_dictionary.pm6) [Table 284 – Entries in a floating window parameters dictionary](gen/lib/ISO_32000/Table_284-Entries_in_a_floating_window_parameters_dictionary.pm6) [Table 304 – Entries in a 3D view dictionary](gen/lib/ISO_32000/Table_304-Entries_in_a_ThreeD_view_dictionary.pm6) [Table 323 – Entries in a structure element dictionary](gen/lib/ISO_32000/Table_323-Entries_in_a_structure_element_dictionary.pm6) [Table 328 – Additional entries in an attribute object dictionary for user properties](gen/lib/ISO_32000/Table_328-Additional_entries_in_an_attribute_object_dictionary_for_user_properties.pm6) [Table 357 – Entries in a Web Capture command dictionary](gen/lib/ISO_32000/Table_357-Entries_in_a_Web_Capture_command_dictionary.pm6) [Table F. 1 – Entries in the linearization parameter dictionary](gen/lib/ISO_32000/Table_F1-Entries_in_the_linearization_parameter_dictionary.pm6)
/PA|[Table 163 – Entries in a navigation node dictionary](gen/lib/ISO_32000/Table_163-Entries_in_a_navigation_node_dictionary.pm6) [Table 173 – Additional entries specific to a link annotation](gen/lib/ISO_32000/Table_173-Additional_entries_specific_to_a_link_annotation.pm6)
/PC|[Table 194 – Entries in an annotation’s additional-actions dictionary](gen/lib/ISO_32000/Table_194-Entries_in_an_annotations_additional-actions_dictionary.pm6) [Table 301 – Entries in an 3D animation style dictionary](gen/lib/ISO_32000/Table_301-Entries_in_an_ThreeD_animation_style_dictionary.pm6) [Table 311 – Entries in a 3D cross section dictionary](gen/lib/ISO_32000/Table_311-Entries_in_a_ThreeD_cross_section_dictionary.pm6)
/PCM|[Table 367 – Additional entries specific to a trap network appearance stream](gen/lib/ISO_32000/Table_367-Additional_entries_specific_to_a_trap_network_appearance_stream.pm6)
/PI|[Table 194 – Entries in an annotation’s additional-actions dictionary](gen/lib/ISO_32000/Table_194-Entries_in_an_annotations_additional-actions_dictionary.pm6)
/PID|[Table 291 – Entries in a media player info dictionary](gen/lib/ISO_32000/Table_291-Entries_in_a_media_player_info_dictionary.pm6)
/PL|[Table 274 – Additional entries in a media clip data dictionary](gen/lib/ISO_32000/Table_274-Additional_entries_in_a_media_clip_data_dictionary.pm6) [Table 279 – Entries in a media play parameters dictionary](gen/lib/ISO_32000/Table_279-Entries_in_a_media_play_parameters_dictionary.pm6)
/PO|[Table 194 – Entries in an annotation’s additional-actions dictionary](gen/lib/ISO_32000/Table_194-Entries_in_an_annotations_additional-actions_dictionary.pm6) [Table 311 – Entries in a 3D cross section dictionary](gen/lib/ISO_32000/Table_311-Entries_in_a_ThreeD_cross_section_dictionary.pm6)
/PS|[Table 263 – Entries in a number format dictionary](gen/lib/ISO_32000/Table_263-Entries_in_a_number_format_dictionary.pm6) [Table 305 – Entries in a projection dictionary](gen/lib/ISO_32000/Table_305-Entries_in_a_projection_dictionary.pm6)
/PV|[Table 194 – Entries in an annotation’s additional-actions dictionary](gen/lib/ISO_32000/Table_194-Entries_in_an_annotations_additional-actions_dictionary.pm6)
/PZ|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.pm6)
/Padding|[Table 343 – Standard layout attributes common to all standard structure types](gen/lib/ISO_32000/Table_343-Standard_layout_attributes_common_to_all_standard_structure_types.pm6)
/Page|[Table 97 – Entries in a Reference Dictionary](gen/lib/ISO_32000/Table_97-Entries_in_a_Reference_Dictionary.pm6) [Table 251 – Additional entry for annotation dictionaries in an FDF file](gen/lib/ISO_32000/Table_251-Additional_entry_for_annotation_dictionaries_in_an_FDF_file.pm6)
/PageElement|[Table 102 – Entries in an Optional Content Usage Dictionary](gen/lib/ISO_32000/Table_102-Entries_in_an_Optional_Content_Usage_Dictionary.pm6)
/PageLabels|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.pm6)
/PageLayout|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.pm6)
/PageMode|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.pm6)
/Pages|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.pm6) [Table 31 – Entries in the name dictionary](gen/lib/ISO_32000/Table_31-Entries_in_the_name_dictionary.pm6) [Table 243 – Entries in the FDF dictionary](gen/lib/ISO_32000/Table_243-Entries_in_the_FDF_dictionary.pm6) [Table 364 – Entries in a separation dictionary](gen/lib/ISO_32000/Table_364-Entries_in_a_separation_dictionary.pm6)
/PaintType|[Table 75 – Additional Entries Specific to a Type 1 Pattern Dictionary](gen/lib/ISO_32000/Table_75-Additional_Entries_Specific_to_a_Type_1_Pattern_Dictionary.pm6)
/Params|[Table 45 – Additional entries in an embedded file stream dictionary](gen/lib/ISO_32000/Table_45-Additional_entries_in_an_embedded_file_stream_dictionary.pm6)
/Parent|[Table 29 – Required entries in a page tree node](gen/lib/ISO_32000/Table_29-Required_entries_in_a_page_tree_node.pm6) [Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.pm6) [Table 153 – Entries in an outline item dictionary](gen/lib/ISO_32000/Table_153-Entries_in_an_outline_item_dictionary.pm6) [Table 183 – Additional entries specific to a pop-up annotation](gen/lib/ISO_32000/Table_183-Additional_entries_specific_to_a_pop-up_annotation.pm6) [Table 188 – Additional entries specific to a widget annotation](gen/lib/ISO_32000/Table_188-Additional_entries_specific_to_a_widget_annotation.pm6) [Table 220 – Entries common to all field dictionaries](gen/lib/ISO_32000/Table_220-Entries_common_to_all_field_dictionaries.pm6)
/ParentTree|[Table 322 – Entries in the structure tree root](gen/lib/ISO_32000/Table_322-Entries_in_the_structure_tree_root.pm6)
/ParentTreeNextKey|[Table 322 – Entries in the structure tree root](gen/lib/ISO_32000/Table_322-Entries_in_the_structure_tree_root.pm6)
/Pattern|[Table 33 – Entries in a resource dictionary](gen/lib/ISO_32000/Table_33-Entries_in_a_resource_dictionary.pm6)
/PatternType|[Table 75 – Additional Entries Specific to a Type 1 Pattern Dictionary](gen/lib/ISO_32000/Table_75-Additional_Entries_Specific_to_a_Type_1_Pattern_Dictionary.pm6) [Table 76 – Entries in a Type 2 Pattern Dictionary](gen/lib/ISO_32000/Table_76-Entries_in_a_Type_2_Pattern_Dictionary.pm6)
/Perms|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.pm6)
/Pg|[Table 323 – Entries in a structure element dictionary](gen/lib/ISO_32000/Table_323-Entries_in_a_structure_element_dictionary.pm6) [Table 324 – Entries in a marked-content reference dictionary](gen/lib/ISO_32000/Table_324-Entries_in_a_marked-content_reference_dictionary.pm6) [Table 325 – Entries in an object reference dictionary](gen/lib/ISO_32000/Table_325-Entries_in_an_object_reference_dictionary.pm6)
/PickTrayByPDFSize|[Table 150 – Entries in a viewer preferences dictionary](gen/lib/ISO_32000/Table_150-Entries_in_a_viewer_preferences_dictionary.pm6)
/PieceInfo|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.pm6) [Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.pm6) [Table 95 – Additional Entries Specific to a Type 1 Form Dictionary](gen/lib/ISO_32000/Table_95-Additional_Entries_Specific_to_a_Type_1_Form_Dictionary.pm6)
/Placement|[Table 343 – Standard layout attributes common to all standard structure types](gen/lib/ISO_32000/Table_343-Standard_layout_attributes_common_to_all_standard_structure_types.pm6)
/Popup|[Table 170 – Additional entries specific to markup annotations](gen/lib/ISO_32000/Table_170-Additional_entries_specific_to_markup_annotations.pm6)
/Poster|[Table 295 – Entries in a movie dictionary](gen/lib/ISO_32000/Table_295-Entries_in_a_movie_dictionary.pm6)
/Predictor|[Table 8 – Optional parameters for LZWDecode and FlateDecode filters](gen/lib/ISO_32000/Table_8-Optional_parameters_for_LZWDecode_and_FlateDecode_filters.pm6)
/PresSteps|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.pm6)
/PreserveRB|[Table 213 – Additional entries specific to a set-OCG-state action](gen/lib/ISO_32000/Table_213-Additional_entries_specific_to_a_set-OCG-state_action.pm6)
/Prev|[Table 15 – Entries in the file trailer dictionary](gen/lib/ISO_32000/Table_15-Entries_in_the_file_trailer_dictionary.pm6) [Table 17 – Additional entries specific to a cross-reference stream dictionary](gen/lib/ISO_32000/Table_17-Additional_entries_specific_to_a_cross-reference_stream_dictionary.pm6) [Table 153 – Entries in an outline item dictionary](gen/lib/ISO_32000/Table_153-Entries_in_an_outline_item_dictionary.pm6) [Table 163 – Entries in a navigation node dictionary](gen/lib/ISO_32000/Table_163-Entries_in_a_navigation_node_dictionary.pm6)
/Print|[Table 102 – Entries in an Optional Content Usage Dictionary](gen/lib/ISO_32000/Table_102-Entries_in_an_Optional_Content_Usage_Dictionary.pm6)
/PrintArea|[Table 150 – Entries in a viewer preferences dictionary](gen/lib/ISO_32000/Table_150-Entries_in_a_viewer_preferences_dictionary.pm6)
/PrintClip|[Table 150 – Entries in a viewer preferences dictionary](gen/lib/ISO_32000/Table_150-Entries_in_a_viewer_preferences_dictionary.pm6)
/PrintPageRange|[Table 150 – Entries in a viewer preferences dictionary](gen/lib/ISO_32000/Table_150-Entries_in_a_viewer_preferences_dictionary.pm6)
/PrintScaling|[Table 150 – Entries in a viewer preferences dictionary](gen/lib/ISO_32000/Table_150-Entries_in_a_viewer_preferences_dictionary.pm6)
/PrintingOrder|[Table 73 – Entries in a DeviceN Mixing Hints Dictionary](gen/lib/ISO_32000/Table_73-Entries_in_a_DeviceN_Mixing_Hints_Dictionary.pm6)
/Private|[Table 319 – Entries in an data dictionary](gen/lib/ISO_32000/Table_319-Entries_in_an_data_dictionary.pm6)
/ProcSet|[Table 33 – Entries in a resource dictionary](gen/lib/ISO_32000/Table_33-Entries_in_a_resource_dictionary.pm6)
/Process|[Table 71 – Entries in a DeviceN Colour Space Attributes Dictionary](gen/lib/ISO_32000/Table_71-Entries_in_a_DeviceN_Colour_Space_Attributes_Dictionary.pm6)
/Producer|[Table 317 – Entries in the document information dictionary](gen/lib/ISO_32000/Table_317-Entries_in_the_document_information_dictionary.pm6)
/Prop_AuthTime|[Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.pm6)
/Prop_AuthType|[Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.pm6)
/Prop_Build|[Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.pm6)
/Properties|[Table 33 – Entries in a resource dictionary](gen/lib/ISO_32000/Table_33-Entries_in_a_resource_dictionary.pm6)
/Q|[Table 174 – Additional entries specific to a free text annotation](gen/lib/ISO_32000/Table_174-Additional_entries_specific_to_a_free_text_annotation.pm6) [Table 192 – Additional entries specific to a redaction annotation](gen/lib/ISO_32000/Table_192-Additional_entries_specific_to_a_redaction_annotation.pm6) [Table 218 – Entries in the interactive form dictionary](gen/lib/ISO_32000/Table_218-Entries_in_the_interactive_form_dictionary.pm6) [Table 222 – Additional entries common to all fields containing variable text](gen/lib/ISO_32000/Table_222-Additional_entries_common_to_all_fields_containing_variable_text.pm6)
/QuadPoints|[Table 173 – Additional entries specific to a link annotation](gen/lib/ISO_32000/Table_173-Additional_entries_specific_to_a_link_annotation.pm6) [Table 179 – Additional entries specific to text markup annotations](gen/lib/ISO_32000/Table_179-Additional_entries_specific_to_text_markup_annotations.pm6) [Table 192 – Additional entries specific to a redaction annotation](gen/lib/ISO_32000/Table_192-Additional_entries_specific_to_a_redaction_annotation.pm6)
/R|[Table 21 – Additional encryption dictionary entries for the standard security handler](gen/lib/ISO_32000/Table_21-Additional_encryption_dictionary_entries_for_the_standard_security_handler.pm6) [Table 161 – Entries in a bead dictionary](gen/lib/ISO_32000/Table_161-Entries_in_a_bead_dictionary.pm6) [Table 168 – Entries in an appearance dictionary](gen/lib/ISO_32000/Table_168-Entries_in_an_appearance_dictionary.pm6) [Table 189 – Entries in an appearance characteristics dictionary](gen/lib/ISO_32000/Table_189-Entries_in_an_appearance_characteristics_dictionary.pm6) [Table 202 – Entries specific to a target dictionary](gen/lib/ISO_32000/Table_202-Entries_specific_to_a_target_dictionary.pm6) [Table 214 – Additional entries specific to a rendition action](gen/lib/ISO_32000/Table_214-Additional_entries_specific_to_a_rendition_action.pm6) [Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.pm6) [Table 262 – Additional entries in a rectilinear measure dictionary](gen/lib/ISO_32000/Table_262-Additional_entries_in_a_rectilinear_measure_dictionary.pm6) [Table 268 – Entries in a media criteria dictionary](gen/lib/ISO_32000/Table_268-Entries_in_a_media_criteria_dictionary.pm6) [Table 272 – Additional entries specific to a selector rendition dictionary](gen/lib/ISO_32000/Table_272-Additional_entries_specific_to_a_selector_rendition_dictionary.pm6) [Table 284 – Entries in a floating window parameters dictionary](gen/lib/ISO_32000/Table_284-Entries_in_a_floating_window_parameters_dictionary.pm6) [Table 294 – Additional entries specific to a sound object](gen/lib/ISO_32000/Table_294-Additional_entries_specific_to_a_sound_object.pm6) [Table 323 – Entries in a structure element dictionary](gen/lib/ISO_32000/Table_323-Entries_in_a_structure_element_dictionary.pm6) [Table 354 – Additional entries specific to a Web Capture image set](gen/lib/ISO_32000/Table_354-Additional_entries_specific_to_a_Web_Capture_image_set.pm6)
/RBGroups|[Table 101 – Entries in an Optional Content Configuration Dictionary](gen/lib/ISO_32000/Table_101-Entries_in_an_Optional_Content_Configuration_Dictionary.pm6)
/RC|[Table 170 – Additional entries specific to markup annotations](gen/lib/ISO_32000/Table_170-Additional_entries_specific_to_markup_annotations.pm6) [Table 174 – Additional entries specific to a free text annotation](gen/lib/ISO_32000/Table_174-Additional_entries_specific_to_a_free_text_annotation.pm6) [Table 189 – Entries in an appearance characteristics dictionary](gen/lib/ISO_32000/Table_189-Entries_in_an_appearance_characteristics_dictionary.pm6) [Table 280 – Entries in a media play parameters MH/BE dictionary](gen/lib/ISO_32000/Table_280-Entries_in_a_media_play_parameters_MH-BE_dictionary.pm6)
/RD|[Table 174 – Additional entries specific to a free text annotation](gen/lib/ISO_32000/Table_174-Additional_entries_specific_to_a_free_text_annotation.pm6) [Table 177 – Additional entries specific to a square or circle annotation](gen/lib/ISO_32000/Table_177-Additional_entries_specific_to_a_square_or_circle_annotation.pm6) [Table 180 – Additional entries specific to a caret annotation](gen/lib/ISO_32000/Table_180-Additional_entries_specific_to_a_caret_annotation.pm6) [Table 263 – Entries in a number format dictionary](gen/lib/ISO_32000/Table_263-Entries_in_a_number_format_dictionary.pm6)
/RF|[Table 44 – Entries in a file specification dictionary](gen/lib/ISO_32000/Table_44-Entries_in_a_file_specification_dictionary.pm6)
/RH|[Table 264 – Entries common to all requirement dictionaries](gen/lib/ISO_32000/Table_264-Entries_common_to_all_requirement_dictionaries.pm6)
/RI|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.pm6) [Table 189 – Entries in an appearance characteristics dictionary](gen/lib/ISO_32000/Table_189-Entries_in_an_appearance_characteristics_dictionary.pm6)
/RM|[Table 304 – Entries in a 3D view dictionary](gen/lib/ISO_32000/Table_304-Entries_in_a_ThreeD_view_dictionary.pm6)
/RO|[Table 192 – Additional entries specific to a redaction annotation](gen/lib/ISO_32000/Table_192-Additional_entries_specific_to_a_redaction_annotation.pm6)
/RT|[Table 170 – Additional entries specific to markup annotations](gen/lib/ISO_32000/Table_170-Additional_entries_specific_to_markup_annotations.pm6) [Table 263 – Entries in a number format dictionary](gen/lib/ISO_32000/Table_263-Entries_in_a_number_format_dictionary.pm6) [Table 284 – Entries in a floating window parameters dictionary](gen/lib/ISO_32000/Table_284-Entries_in_a_floating_window_parameters_dictionary.pm6)
/RV|[Table 222 – Additional entries common to all fields containing variable text](gen/lib/ISO_32000/Table_222-Additional_entries_common_to_all_fields_containing_variable_text.pm6) [Table 246 – Entries in an FDF field dictionary](gen/lib/ISO_32000/Table_246-Entries_in_an_FDF_field_dictionary.pm6)
/Range|[Table 38 – Entries common to all function dictionaries](gen/lib/ISO_32000/Table_38-Entries_common_to_all_function_dictionaries.pm6) [Table 65 – Entries in a Lab Colour Space Dictionary](gen/lib/ISO_32000/Table_65-Entries_in_a_Lab_Colour_Space_Dictionary.pm6) [Table 66 – Additional Entries Specific to an ICC Profile Stream Dictionary](gen/lib/ISO_32000/Table_66-Additional_Entries_Specific_to_an_ICC_Profile_Stream_Dictionary.pm6)
/Rate|[Table 296 – Entries in a movie activation dictionary](gen/lib/ISO_32000/Table_296-Entries_in_a_movie_activation_dictionary.pm6)
/Reason|[Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.pm6)
/Reasons|[Table 234 – Entries in a signature field seed value dictionary](gen/lib/ISO_32000/Table_234-Entries_in_a_signature_field_seed_value_dictionary.pm6)
/Recipients|[Table 23 – Additional encryption dictionary entries for public-key security handlers](gen/lib/ISO_32000/Table_23-Additional_encryption_dictionary_entries_for_public-key_security_handlers.pm6) [Table 27 – Additional crypt filter dictionary entries for public-key security handlers](gen/lib/ISO_32000/Table_27-Additional_crypt_filter_dictionary_entries_for_public-key_security_handlers.pm6)
/Rect|[Table 164 – Entries common to all annotation dictionaries](gen/lib/ISO_32000/Table_164-Entries_common_to_all_annotation_dictionaries.pm6)
/Ref|[Table 95 – Additional Entries Specific to a Type 1 Form Dictionary](gen/lib/ISO_32000/Table_95-Additional_Entries_Specific_to_a_Type_1_Form_Dictionary.pm6)
/Reference|[Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.pm6)
/Registry|[Table 116 – Entries in a CIDSystemInfo dictionary](gen/lib/ISO_32000/Table_116-Entries_in_a_CIDSystemInfo_dictionary.pm6)
/RegistryName|[Table 365 – Entries in an output intent dictionary](gen/lib/ISO_32000/Table_365-Entries_in_an_output_intent_dictionary.pm6)
/Rename|[Table 249 – Entries in an FDF template dictionary](gen/lib/ISO_32000/Table_249-Entries_in_an_FDF_template_dictionary.pm6)
/Renditions|[Table 31 – Entries in the name dictionary](gen/lib/ISO_32000/Table_31-Entries_in_the_name_dictionary.pm6)
/Repeat|[Table 192 – Additional entries specific to a redaction annotation](gen/lib/ISO_32000/Table_192-Additional_entries_specific_to_a_redaction_annotation.pm6) [Table 208 – Additional entries specific to a sound action](gen/lib/ISO_32000/Table_208-Additional_entries_specific_to_a_sound_action.pm6)
/Requirements|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.pm6)
/ResFork|[Table 47 – Entries in a Mac OS file information dictionary](gen/lib/ISO_32000/Table_47-Entries_in_a_Mac_OS_file_information_dictionary.pm6)
/Resources|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.pm6) [Table 75 – Additional Entries Specific to a Type 1 Pattern Dictionary](gen/lib/ISO_32000/Table_75-Additional_Entries_Specific_to_a_Type_1_Pattern_Dictionary.pm6) [Table 95 – Additional Entries Specific to a Type 1 Form Dictionary](gen/lib/ISO_32000/Table_95-Additional_Entries_Specific_to_a_Type_1_Form_Dictionary.pm6) [Table 112 – Entries in a Type 3 font dictionary](gen/lib/ISO_32000/Table_112-Entries_in_a_Type_3_font_dictionary.pm6) [Table 297 – Entries in a slideshow dictionary](gen/lib/ISO_32000/Table_297-Entries_in_a_slideshow_dictionary.pm6) [Table 300 – Entries in a 3D stream dictionary](gen/lib/ISO_32000/Table_300-Entries_in_a_ThreeD_stream_dictionary.pm6)
/Role|[Table 348 – PrintField attributes](gen/lib/ISO_32000/Table_348-PrintField_attributes.pm6)
/RoleMap|[Table 322 – Entries in the structure tree root](gen/lib/ISO_32000/Table_322-Entries_in_the_structure_tree_root.pm6)
/Root|[Table 15 – Entries in the file trailer dictionary](gen/lib/ISO_32000/Table_15-Entries_in_the_file_trailer_dictionary.pm6) [Table 241 – Entry in the FDF trailer dictionary](gen/lib/ISO_32000/Table_241-Entry_in_the_FDF_trailer_dictionary.pm6)
/Rotate|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.pm6) [Table 295 – Entries in a movie dictionary](gen/lib/ISO_32000/Table_295-Entries_in_a_movie_dictionary.pm6)
/RowSpan|[Table 349 – Standard table attributes](gen/lib/ISO_32000/Table_349-Standard_table_attributes.pm6)
/Rows|[Table 11 – Optional parameters for the CCITTFaxDecode filter](gen/lib/ISO_32000/Table_11-Optional_parameters_for_the_CCITTFaxDecode_filter.pm6)
/RubyAlign|[Table 345 – Standard layout attributes specific to inline-level structure elements](gen/lib/ISO_32000/Table_345-Standard_layout_attributes_specific_to_inline-level_structure_elements.pm6)
/RubyPosition|[Table 345 – Standard layout attributes specific to inline-level structure elements](gen/lib/ISO_32000/Table_345-Standard_layout_attributes_specific_to_inline-level_structure_elements.pm6)
/S|[Table 96 – Entries Common to all Group Attributes Dictionaries](gen/lib/ISO_32000/Table_96-Entries_Common_to_all_Group_Attributes_Dictionaries.pm6) [Table 144 – Entries in a soft-mask dictionary](gen/lib/ISO_32000/Table_144-Entries_in_a_soft-mask_dictionary.pm6) [Table 147 – Additional entries specific to a transparency group attributes dictionary](gen/lib/ISO_32000/Table_147-Additional_entries_specific_to_a_transparency_group_attributes_dictionary.pm6) [Table 158 – Entries in a collection sort dictionary](gen/lib/ISO_32000/Table_158-Entries_in_a_collection_sort_dictionary.pm6) [Table 159 – Entries in a page label dictionary](gen/lib/ISO_32000/Table_159-Entries_in_a_page_label_dictionary.pm6) [Table 162 – Entries in a transition dictionary](gen/lib/ISO_32000/Table_162-Entries_in_a_transition_dictionary.pm6) [Table 166 – Entries in a border style dictionary](gen/lib/ISO_32000/Table_166-Entries_in_a_border_style_dictionary.pm6) [Table 167 – Entries in a border effect dictionary](gen/lib/ISO_32000/Table_167-Entries_in_a_border_effect_dictionary.pm6) [Table 193 – Entries common to all action dictionaries](gen/lib/ISO_32000/Table_193-Entries_common_to_all_action_dictionaries.pm6) [Table 199 – Additional entries specific to a go-to action](gen/lib/ISO_32000/Table_199-Additional_entries_specific_to_a_go-to_action.pm6) [Table 200 – Additional entries specific to a remote go-to action](gen/lib/ISO_32000/Table_200-Additional_entries_specific_to_a_remote_go-to_action.pm6) [Table 201 – Additional entries specific to an embedded go-to action](gen/lib/ISO_32000/Table_201-Additional_entries_specific_to_an_embedded_go-to_action.pm6) [Table 203 – Additional entries specific to a launch action](gen/lib/ISO_32000/Table_203-Additional_entries_specific_to_a_launch_action.pm6) [Table 205 – Additional entries specific to a thread action](gen/lib/ISO_32000/Table_205-Additional_entries_specific_to_a_thread_action.pm6) [Table 206 – Additional entries specific to a URI action](gen/lib/ISO_32000/Table_206-Additional_entries_specific_to_a_URI_action.pm6) [Table 208 – Additional entries specific to a sound action](gen/lib/ISO_32000/Table_208-Additional_entries_specific_to_a_sound_action.pm6) [Table 209 – Additional entries specific to a movie action](gen/lib/ISO_32000/Table_209-Additional_entries_specific_to_a_movie_action.pm6) [Table 210 – Additional entries specific to a hide action](gen/lib/ISO_32000/Table_210-Additional_entries_specific_to_a_hide_action.pm6) [Table 212 – Additional entries specific to named actions](gen/lib/ISO_32000/Table_212-Additional_entries_specific_to_named_actions.pm6) [Table 213 – Additional entries specific to a set-OCG-state action](gen/lib/ISO_32000/Table_213-Additional_entries_specific_to_a_set-OCG-state_action.pm6) [Table 214 – Additional entries specific to a rendition action](gen/lib/ISO_32000/Table_214-Additional_entries_specific_to_a_rendition_action.pm6) [Table 215 – Additional entries specific to a transition action](gen/lib/ISO_32000/Table_215-Additional_entries_specific_to_a_transition_action.pm6) [Table 216 – Additional entries specific to a go-to-3D-view action](gen/lib/ISO_32000/Table_216-Additional_entries_specific_to_a_go-to-ThreeD-view_action.pm6) [Table 217 – Additional entries specific to a JavaScript action](gen/lib/ISO_32000/Table_217-Additional_entries_specific_to_a_JavaScript_action.pm6) [Table 236 – Additional entries specific to a submit-form action](gen/lib/ISO_32000/Table_236-Additional_entries_specific_to_a_submit-form_action.pm6) [Table 238 – Additional entries specific to a reset-form action](gen/lib/ISO_32000/Table_238-Additional_entries_specific_to_a_reset-form_action.pm6) [Table 240 – Additional entries specific to an import-data action](gen/lib/ISO_32000/Table_240-Additional_entries_specific_to_an_import-data_action.pm6) [Table 247 – Entries in an icon fit dictionary](gen/lib/ISO_32000/Table_247-Entries_in_an_icon_fit_dictionary.pm6) [Table 262 – Additional entries in a rectilinear measure dictionary](gen/lib/ISO_32000/Table_262-Additional_entries_in_a_rectilinear_measure_dictionary.pm6) [Table 264 – Entries common to all requirement dictionaries](gen/lib/ISO_32000/Table_264-Entries_common_to_all_requirement_dictionaries.pm6) [Table 265 – Entries in a requirement handler dictionary](gen/lib/ISO_32000/Table_265-Entries_in_a_requirement_handler_dictionary.pm6) [Table 266 – Entries common to all rendition dictionaries](gen/lib/ISO_32000/Table_266-Entries_common_to_all_rendition_dictionaries.pm6) [Table 268 – Entries in a media criteria dictionary](gen/lib/ISO_32000/Table_268-Entries_in_a_media_criteria_dictionary.pm6) [Table 273 – Entries common to all media clip dictionaries](gen/lib/ISO_32000/Table_273-Entries_common_to_all_media_clip_dictionaries.pm6) [Table 281 – Entries in a media duration dictionary](gen/lib/ISO_32000/Table_281-Entries_in_a_media_duration_dictionary.pm6) [Table 285 – Entries common to all media offset dictionaries](gen/lib/ISO_32000/Table_285-Entries_common_to_all_media_offset_dictionaries.pm6) [Table 289 – Entries in a timespan dictionary](gen/lib/ISO_32000/Table_289-Entries_in_a_timespan_dictionary.pm6) [Table 323 – Entries in a structure element dictionary](gen/lib/ISO_32000/Table_323-Entries_in_a_structure_element_dictionary.pm6) [Table 352 – Entries common to all Web Capture content sets](gen/lib/ISO_32000/Table_352-Entries_common_to_all_Web_Capture_content_sets.pm6) [Table 353 – Additional entries specific to a Web Capture page set](gen/lib/ISO_32000/Table_353-Additional_entries_specific_to_a_Web_Capture_page_set.pm6) [Table 354 – Additional entries specific to a Web Capture image set](gen/lib/ISO_32000/Table_354-Additional_entries_specific_to_a_Web_Capture_image_set.pm6) [Table 355 – Entries in a source information dictionary](gen/lib/ISO_32000/Table_355-Entries_in_a_source_information_dictionary.pm6) [Table 357 – Entries in a Web Capture command dictionary](gen/lib/ISO_32000/Table_357-Entries_in_a_Web_Capture_command_dictionary.pm6) [Table 361 – Entries in a box style dictionary](gen/lib/ISO_32000/Table_361-Entries_in_a_box_style_dictionary.pm6) [Table 365 – Entries in an output intent dictionary](gen/lib/ISO_32000/Table_365-Entries_in_an_output_intent_dictionary.pm6)
/SA|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.pm6) [Table 304 – Entries in a 3D view dictionary](gen/lib/ISO_32000/Table_304-Entries_in_a_ThreeD_view_dictionary.pm6)
/SE|[Table 153 – Entries in an outline item dictionary](gen/lib/ISO_32000/Table_153-Entries_in_an_outline_item_dictionary.pm6)
/SI|[Table 352 – Entries common to all Web Capture content sets](gen/lib/ISO_32000/Table_352-Entries_common_to_all_Web_Capture_content_sets.pm6)
/SM|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.pm6)
/SMask|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.pm6) [Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.pm6)
/SMaskInData|[Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.pm6)
/SP|[Table 271 – Additional entries in a media rendition dictionary](gen/lib/ISO_32000/Table_271-Additional_entries_in_a_media_rendition_dictionary.pm6)
/SS|[Table 162 – Entries in a transition dictionary](gen/lib/ISO_32000/Table_162-Entries_in_a_transition_dictionary.pm6) [Table 263 – Entries in a number format dictionary](gen/lib/ISO_32000/Table_263-Entries_in_a_number_format_dictionary.pm6)
/SV|[Table 232 – Additional entries specific to a signature field](gen/lib/ISO_32000/Table_232-Additional_entries_specific_to_a_signature_field.pm6)
/SW|[Table 247 – Entries in an icon fit dictionary](gen/lib/ISO_32000/Table_247-Entries_in_an_icon_fit_dictionary.pm6)
/Schema|[Table 155 – Entries in a collection dictionary](gen/lib/ISO_32000/Table_155-Entries_in_a_collection_dictionary.pm6)
/Scope|[Table 349 – Standard table attributes](gen/lib/ISO_32000/Table_349-Standard_table_attributes.pm6)
/Script|[Table 265 – Entries in a requirement handler dictionary](gen/lib/ISO_32000/Table_265-Entries_in_a_requirement_handler_dictionary.pm6)
/SeparationColorNames|[Table 367 – Additional entries specific to a trap network appearance stream](gen/lib/ISO_32000/Table_367-Additional_entries_specific_to_a_trap_network_appearance_stream.pm6)
/SeparationInfo|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.pm6)
/SetF|[Table 246 – Entries in an FDF field dictionary](gen/lib/ISO_32000/Table_246-Entries_in_an_FDF_field_dictionary.pm6)
/SetFf|[Table 246 – Entries in an FDF field dictionary](gen/lib/ISO_32000/Table_246-Entries_in_an_FDF_field_dictionary.pm6)
/Shading|[Table 33 – Entries in a resource dictionary](gen/lib/ISO_32000/Table_33-Entries_in_a_resource_dictionary.pm6) [Table 76 – Entries in a Type 2 Pattern Dictionary](gen/lib/ISO_32000/Table_76-Entries_in_a_Type_2_Pattern_Dictionary.pm6)
/ShadingType|[Table 78 – Entries Common to All Shading Dictionaries](gen/lib/ISO_32000/Table_78-Entries_Common_to_All_Shading_Dictionaries.pm6)
/ShowControls|[Table 296 – Entries in a movie activation dictionary](gen/lib/ISO_32000/Table_296-Entries_in_a_movie_activation_dictionary.pm6)
/SigFlags|[Table 218 – Entries in the interactive form dictionary](gen/lib/ISO_32000/Table_218-Entries_in_the_interactive_form_dictionary.pm6)
/Signature|[Table 255 – Entries in the UR transform parameters dictionary](gen/lib/ISO_32000/Table_255-Entries_in_the_UR_transform_parameters_dictionary.pm6)
/Size|[Table 15 – Entries in the file trailer dictionary](gen/lib/ISO_32000/Table_15-Entries_in_the_file_trailer_dictionary.pm6) [Table 17 – Additional entries specific to a cross-reference stream dictionary](gen/lib/ISO_32000/Table_17-Additional_entries_specific_to_a_cross-reference_stream_dictionary.pm6) [Table 39 – Additional entries specific to a type 0 function dictionary](gen/lib/ISO_32000/Table_39-Additional_entries_specific_to_a_type_0_function_dictionary.pm6) [Table 46 – Entries in an embedded file parameter dictionary](gen/lib/ISO_32000/Table_46-Entries_in_an_embedded_file_parameter_dictionary.pm6)
/Solidities|[Table 73 – Entries in a DeviceN Mixing Hints Dictionary](gen/lib/ISO_32000/Table_73-Entries_in_a_DeviceN_Mixing_Hints_Dictionary.pm6)
/Sort|[Table 155 – Entries in a collection dictionary](gen/lib/ISO_32000/Table_155-Entries_in_a_collection_dictionary.pm6)
/Sound|[Table 185 – Additional entries specific to a sound annotation](gen/lib/ISO_32000/Table_185-Additional_entries_specific_to_a_sound_annotation.pm6) [Table 208 – Additional entries specific to a sound action](gen/lib/ISO_32000/Table_208-Additional_entries_specific_to_a_sound_action.pm6)
/SoundActions|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.pm6)
/SpaceAfter|[Table 344 – Additional standard layout attributes specific to block-level structure elements](gen/lib/ISO_32000/Table_344-Additional_standard_layout_attributes_specific_to_block-level_structure_elements.pm6)
/SpaceBefore|[Table 344 – Additional standard layout attributes specific to block-level structure elements](gen/lib/ISO_32000/Table_344-Additional_standard_layout_attributes_specific_to_block-level_structure_elements.pm6)
/SpiderInfo|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.pm6)
/SpotFunction|[Table 130 – Entries in a type 1 halftone dictionary](gen/lib/ISO_32000/Table_130-Entries_in_a_type_1_halftone_dictionary.pm6)
/St|[Table 159 – Entries in a page label dictionary](gen/lib/ISO_32000/Table_159-Entries_in_a_page_label_dictionary.pm6)
/Start|[Table 296 – Entries in a movie activation dictionary](gen/lib/ISO_32000/Table_296-Entries_in_a_movie_activation_dictionary.pm6)
/StartIndent|[Table 344 – Additional standard layout attributes specific to block-level structure elements](gen/lib/ISO_32000/Table_344-Additional_standard_layout_attributes_specific_to_block-level_structure_elements.pm6)
/StartResource|[Table 297 – Entries in a slideshow dictionary](gen/lib/ISO_32000/Table_297-Entries_in_a_slideshow_dictionary.pm6)
/State|[Table 172 – Additional entries specific to a text annotation](gen/lib/ISO_32000/Table_172-Additional_entries_specific_to_a_text_annotation.pm6) [Table 213 – Additional entries specific to a set-OCG-state action](gen/lib/ISO_32000/Table_213-Additional_entries_specific_to_a_set-OCG-state_action.pm6)
/StateModel|[Table 172 – Additional entries specific to a text annotation](gen/lib/ISO_32000/Table_172-Additional_entries_specific_to_a_text_annotation.pm6)
/Status|[Table 243 – Entries in the FDF dictionary](gen/lib/ISO_32000/Table_243-Entries_in_the_FDF_dictionary.pm6)
/StemH|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.pm6)
/StemV|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.pm6)
/Stm|[Table 324 – Entries in a marked-content reference dictionary](gen/lib/ISO_32000/Table_324-Entries_in_a_marked-content_reference_dictionary.pm6)
/StmF|[Table 20 – Entries common to all encryption dictionaries](gen/lib/ISO_32000/Table_20-Entries_common_to_all_encryption_dictionaries.pm6)
/StmOwn|[Table 324 – Entries in a marked-content reference dictionary](gen/lib/ISO_32000/Table_324-Entries_in_a_marked-content_reference_dictionary.pm6)
/StrF|[Table 20 – Entries common to all encryption dictionaries](gen/lib/ISO_32000/Table_20-Entries_common_to_all_encryption_dictionaries.pm6)
/StructParent|[Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.pm6) [Table 95 – Additional Entries Specific to a Type 1 Form Dictionary](gen/lib/ISO_32000/Table_95-Additional_Entries_Specific_to_a_Type_1_Form_Dictionary.pm6) [Table 164 – Entries common to all annotation dictionaries](gen/lib/ISO_32000/Table_164-Entries_common_to_all_annotation_dictionaries.pm6) [Table 326 – Additional dictionary entries for structure element access](gen/lib/ISO_32000/Table_326-Additional_dictionary_entries_for_structure_element_access.pm6)
/StructParents|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.pm6) [Table 95 – Additional Entries Specific to a Type 1 Form Dictionary](gen/lib/ISO_32000/Table_95-Additional_Entries_Specific_to_a_Type_1_Form_Dictionary.pm6) [Table 326 – Additional dictionary entries for structure element access](gen/lib/ISO_32000/Table_326-Additional_dictionary_entries_for_structure_element_access.pm6)
/StructTreeRoot|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.pm6)
/Style|[Table 124 – Additional font descriptor entries for CIDFonts](gen/lib/ISO_32000/Table_124-Additional_font_descriptor_entries_for_CIDFonts.pm6)
/SubFilter|[Table 20 – Entries common to all encryption dictionaries](gen/lib/ISO_32000/Table_20-Entries_common_to_all_encryption_dictionaries.pm6) [Table 234 – Entries in a signature field seed value dictionary](gen/lib/ISO_32000/Table_234-Entries_in_a_signature_field_seed_value_dictionary.pm6) [Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.pm6)
/Subj|[Table 170 – Additional entries specific to markup annotations](gen/lib/ISO_32000/Table_170-Additional_entries_specific_to_markup_annotations.pm6)
/Subject|[Table 235 – Entries in a certificate seed value dictionary](gen/lib/ISO_32000/Table_235-Entries_in_a_certificate_seed_value_dictionary.pm6) [Table 317 – Entries in the document information dictionary](gen/lib/ISO_32000/Table_317-Entries_in_the_document_information_dictionary.pm6)
/SubjectDN|[Table 235 – Entries in a certificate seed value dictionary](gen/lib/ISO_32000/Table_235-Entries_in_a_certificate_seed_value_dictionary.pm6)
/Subtype|[Table 45 – Additional entries in an embedded file stream dictionary](gen/lib/ISO_32000/Table_45-Additional_entries_in_an_embedded_file_stream_dictionary.pm6) [Table 47 – Entries in a Mac OS file information dictionary](gen/lib/ISO_32000/Table_47-Entries_in_a_Mac_OS_file_information_dictionary.pm6) [Table 71 – Entries in a DeviceN Colour Space Attributes Dictionary](gen/lib/ISO_32000/Table_71-Entries_in_a_DeviceN_Colour_Space_Attributes_Dictionary.pm6) [Table 88 – Additional Entries Specific to a PostScript XObject Dictionary](gen/lib/ISO_32000/Table_88-Additional_Entries_Specific_to_a_PostScript_XObject_Dictionary.pm6) [Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.pm6) [Table 95 – Additional Entries Specific to a Type 1 Form Dictionary](gen/lib/ISO_32000/Table_95-Additional_Entries_Specific_to_a_Type_1_Form_Dictionary.pm6) [Table 111 – Entries in a Type 1 font dictionary](gen/lib/ISO_32000/Table_111-Entries_in_a_Type_1_font_dictionary.pm6) [Table 112 – Entries in a Type 3 font dictionary](gen/lib/ISO_32000/Table_112-Entries_in_a_Type_3_font_dictionary.pm6) [Table 117 – Entries in a CIDFont dictionary](gen/lib/ISO_32000/Table_117-Entries_in_a_CIDFont_dictionary.pm6) [Table 121 – Entries in a Type 0 font dictionary](gen/lib/ISO_32000/Table_121-Entries_in_a_Type_0_font_dictionary.pm6) [Table 127 – Additional entries in an embedded font stream dictionary](gen/lib/ISO_32000/Table_127-Additional_entries_in_an_embedded_font_stream_dictionary.pm6) [Table 157 – Entries in a collection field dictionary](gen/lib/ISO_32000/Table_157-Entries_in_a_collection_field_dictionary.pm6) [Table 164 – Entries common to all annotation dictionaries](gen/lib/ISO_32000/Table_164-Entries_common_to_all_annotation_dictionaries.pm6) [Table 172 – Additional entries specific to a text annotation](gen/lib/ISO_32000/Table_172-Additional_entries_specific_to_a_text_annotation.pm6) [Table 173 – Additional entries specific to a link annotation](gen/lib/ISO_32000/Table_173-Additional_entries_specific_to_a_link_annotation.pm6) [Table 174 – Additional entries specific to a free text annotation](gen/lib/ISO_32000/Table_174-Additional_entries_specific_to_a_free_text_annotation.pm6) [Table 175 – Additional entries specific to a line annotation](gen/lib/ISO_32000/Table_175-Additional_entries_specific_to_a_line_annotation.pm6) [Table 177 – Additional entries specific to a square or circle annotation](gen/lib/ISO_32000/Table_177-Additional_entries_specific_to_a_square_or_circle_annotation.pm6) [Table 178 – Additional entries specific to a polygon or polyline annotation](gen/lib/ISO_32000/Table_178-Additional_entries_specific_to_a_polygon_or_polyline_annotation.pm6) [Table 179 – Additional entries specific to text markup annotations](gen/lib/ISO_32000/Table_179-Additional_entries_specific_to_text_markup_annotations.pm6) [Table 180 – Additional entries specific to a caret annotation](gen/lib/ISO_32000/Table_180-Additional_entries_specific_to_a_caret_annotation.pm6) [Table 181 – Additional entries specific to a rubber stamp annotation](gen/lib/ISO_32000/Table_181-Additional_entries_specific_to_a_rubber_stamp_annotation.pm6) [Table 182 – Additional entries specific to an ink annotation](gen/lib/ISO_32000/Table_182-Additional_entries_specific_to_an_ink_annotation.pm6) [Table 183 – Additional entries specific to a pop-up annotation](gen/lib/ISO_32000/Table_183-Additional_entries_specific_to_a_pop-up_annotation.pm6) [Table 184 – Additional entries specific to a file attachment annotation](gen/lib/ISO_32000/Table_184-Additional_entries_specific_to_a_file_attachment_annotation.pm6) [Table 185 – Additional entries specific to a sound annotation](gen/lib/ISO_32000/Table_185-Additional_entries_specific_to_a_sound_annotation.pm6) [Table 186 – Additional entries specific to a movie annotation](gen/lib/ISO_32000/Table_186-Additional_entries_specific_to_a_movie_annotation.pm6) [Table 187 – Additional entries specific to a screen annotation](gen/lib/ISO_32000/Table_187-Additional_entries_specific_to_a_screen_annotation.pm6) [Table 188 – Additional entries specific to a widget annotation](gen/lib/ISO_32000/Table_188-Additional_entries_specific_to_a_widget_annotation.pm6) [Table 190 – Additional entries specific to a watermark annotation](gen/lib/ISO_32000/Table_190-Additional_entries_specific_to_a_watermark_annotation.pm6) [Table 192 – Additional entries specific to a redaction annotation](gen/lib/ISO_32000/Table_192-Additional_entries_specific_to_a_redaction_annotation.pm6) [Table 261 – Entries in a measure dictionary](gen/lib/ISO_32000/Table_261-Entries_in_a_measure_dictionary.pm6) [Table 297 – Entries in a slideshow dictionary](gen/lib/ISO_32000/Table_297-Entries_in_a_slideshow_dictionary.pm6) [Table 298 – Additional entries specific to a 3D annotation](gen/lib/ISO_32000/Table_298-Additional_entries_specific_to_a_ThreeD_annotation.pm6) [Table 300 – Entries in a 3D stream dictionary](gen/lib/ISO_32000/Table_300-Entries_in_a_ThreeD_stream_dictionary.pm6) [Table 301 – Entries in an 3D animation style dictionary](gen/lib/ISO_32000/Table_301-Entries_in_an_ThreeD_animation_style_dictionary.pm6) [Table 305 – Entries in a projection dictionary](gen/lib/ISO_32000/Table_305-Entries_in_a_projection_dictionary.pm6) [Table 306 – Entries in a 3D background dictionary](gen/lib/ISO_32000/Table_306-Entries_in_a_ThreeD_background_dictionary.pm6) [Table 307 – Entries in a render mode dictionary](gen/lib/ISO_32000/Table_307-Entries_in_a_render_mode_dictionary.pm6) [Table 309 – Entries in a 3D lighting scheme dictionary](gen/lib/ISO_32000/Table_309-Entries_in_a_ThreeD_lighting_scheme_dictionary.pm6) [Table 313 – Entries in an external data dictionary used to markup 3D annotations](gen/lib/ISO_32000/Table_313-Entries_in_an_external_data_dictionary_used_to_markup_ThreeD_annotations.pm6) [Table 315 – Additional entries in a metadata stream dictionary](gen/lib/ISO_32000/Table_315-Additional_entries_in_a_metadata_stream_dictionary.pm6) [Table 330 – Property list entries for artifacts](gen/lib/ISO_32000/Table_330-Property_list_entries_for_artifacts.pm6) [Table 362 – Additional entries specific to a printer’s mark annotation](gen/lib/ISO_32000/Table_362-Additional_entries_specific_to_a_printers_mark_annotation.pm6) [Table 366 – Additional entries specific to a trap network annotation](gen/lib/ISO_32000/Table_366-Additional_entries_specific_to_a_trap_network_annotation.pm6)
/Summary|[Table 349 – Standard table attributes](gen/lib/ISO_32000/Table_349-Standard_table_attributes.pm6)
/Supplement|[Table 116 – Entries in a CIDSystemInfo dictionary](gen/lib/ISO_32000/Table_116-Entries_in_a_CIDSystemInfo_dictionary.pm6)
/Suspects|[Table 321 – Entries in the mark information dictionary](gen/lib/ISO_32000/Table_321-Entries_in_the_mark_information_dictionary.pm6)
/Sy|[Table 180 – Additional entries specific to a caret annotation](gen/lib/ISO_32000/Table_180-Additional_entries_specific_to_a_caret_annotation.pm6)
/Synchronous|[Table 208 – Additional entries specific to a sound action](gen/lib/ISO_32000/Table_208-Additional_entries_specific_to_a_sound_action.pm6) [Table 296 – Entries in a movie activation dictionary](gen/lib/ISO_32000/Table_296-Entries_in_a_movie_activation_dictionary.pm6)
/T|[Table 161 – Entries in a bead dictionary](gen/lib/ISO_32000/Table_161-Entries_in_a_bead_dictionary.pm6) [Table 170 – Additional entries specific to markup annotations](gen/lib/ISO_32000/Table_170-Additional_entries_specific_to_markup_annotations.pm6) [Table 186 – Additional entries specific to a movie annotation](gen/lib/ISO_32000/Table_186-Additional_entries_specific_to_a_movie_annotation.pm6) [Table 187 – Additional entries specific to a screen annotation](gen/lib/ISO_32000/Table_187-Additional_entries_specific_to_a_screen_annotation.pm6) [Table 201 – Additional entries specific to an embedded go-to action](gen/lib/ISO_32000/Table_201-Additional_entries_specific_to_an_embedded_go-to_action.pm6) [Table 202 – Entries specific to a target dictionary](gen/lib/ISO_32000/Table_202-Entries_specific_to_a_target_dictionary.pm6) [Table 209 – Additional entries specific to a movie action](gen/lib/ISO_32000/Table_209-Additional_entries_specific_to_a_movie_action.pm6) [Table 210 – Additional entries specific to a hide action](gen/lib/ISO_32000/Table_210-Additional_entries_specific_to_a_hide_action.pm6) [Table 220 – Entries common to all field dictionaries](gen/lib/ISO_32000/Table_220-Entries_common_to_all_field_dictionaries.pm6) [Table 246 – Entries in an FDF field dictionary](gen/lib/ISO_32000/Table_246-Entries_in_an_FDF_field_dictionary.pm6) [Table 262 – Additional entries in a rectilinear measure dictionary](gen/lib/ISO_32000/Table_262-Additional_entries_in_a_rectilinear_measure_dictionary.pm6) [Table 281 – Entries in a media duration dictionary](gen/lib/ISO_32000/Table_281-Entries_in_a_media_duration_dictionary.pm6) [Table 284 – Entries in a floating window parameters dictionary](gen/lib/ISO_32000/Table_284-Entries_in_a_floating_window_parameters_dictionary.pm6) [Table 286 – Additional entries in a media offset time dictionary](gen/lib/ISO_32000/Table_286-Additional_entries_in_a_media_offset_time_dictionary.pm6) [Table 323 – Entries in a structure element dictionary](gen/lib/ISO_32000/Table_323-Entries_in_a_structure_element_dictionary.pm6) [Table 353 – Additional entries specific to a Web Capture page set](gen/lib/ISO_32000/Table_353-Additional_entries_specific_to_a_Web_Capture_page_set.pm6) [Table F. 1 – Entries in the linearization parameter dictionary](gen/lib/ISO_32000/Table_F1-Entries_in_the_linearization_parameter_dictionary.pm6)
/TA|[Table 216 – Additional entries specific to a go-to-3D-view action](gen/lib/ISO_32000/Table_216-Additional_entries_specific_to_a_go-to-ThreeD-view_action.pm6)
/TB|[Table 299 – Entries in a 3D activation dictionary](gen/lib/ISO_32000/Table_299-Entries_in_a_ThreeD_activation_dictionary.pm6)
/TBorderStyle|[Table 344 – Additional standard layout attributes specific to block-level structure elements](gen/lib/ISO_32000/Table_344-Additional_standard_layout_attributes_specific_to_block-level_structure_elements.pm6)
/TF|[Table 275 – Entries in a media permissions dictionary](gen/lib/ISO_32000/Table_275-Entries_in_a_media_permissions_dictionary.pm6)
/TI|[Table 231 – Additional entries specific to a choice field](gen/lib/ISO_32000/Table_231-Additional_entries_specific_to_a_choice_field.pm6)
/TID|[Table 353 – Additional entries specific to a Web Capture page set](gen/lib/ISO_32000/Table_353-Additional_entries_specific_to_a_Web_Capture_page_set.pm6)
/TK|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.pm6)
/TM|[Table 220 – Entries common to all field dictionaries](gen/lib/ISO_32000/Table_220-Entries_common_to_all_field_dictionaries.pm6) [Table 301 – Entries in an 3D animation style dictionary](gen/lib/ISO_32000/Table_301-Entries_in_an_ThreeD_animation_style_dictionary.pm6)
/TP|[Table 189 – Entries in an appearance characteristics dictionary](gen/lib/ISO_32000/Table_189-Entries_in_an_appearance_characteristics_dictionary.pm6)
/TPadding|[Table 344 – Additional standard layout attributes specific to block-level structure elements](gen/lib/ISO_32000/Table_344-Additional_standard_layout_attributes_specific_to_block-level_structure_elements.pm6)
/TR|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.pm6) [Table 144 – Entries in a soft-mask dictionary](gen/lib/ISO_32000/Table_144-Entries_in_a_soft-mask_dictionary.pm6)
/TR2|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.pm6)
/TRef|[Table 249 – Entries in an FDF template dictionary](gen/lib/ISO_32000/Table_249-Entries_in_an_FDF_template_dictionary.pm6)
/TS|[Table 352 – Entries common to all Web Capture content sets](gen/lib/ISO_32000/Table_352-Entries_common_to_all_Web_Capture_content_sets.pm6) [Table 355 – Entries in a source information dictionary](gen/lib/ISO_32000/Table_355-Entries_in_a_source_information_dictionary.pm6)
/TT|[Table 284 – Entries in a floating window parameters dictionary](gen/lib/ISO_32000/Table_284-Entries_in_a_floating_window_parameters_dictionary.pm6)
/TU|[Table 220 – Entries common to all field dictionaries](gen/lib/ISO_32000/Table_220-Entries_common_to_all_field_dictionaries.pm6)
/Tabs|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.pm6)
/Target|[Table 243 – Entries in the FDF dictionary](gen/lib/ISO_32000/Table_243-Entries_in_the_FDF_dictionary.pm6)
/TemplateInstantiated|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.pm6)
/Templates|[Table 31 – Entries in the name dictionary](gen/lib/ISO_32000/Table_31-Entries_in_the_name_dictionary.pm6) [Table 248 – Entries in an FDF page dictionary](gen/lib/ISO_32000/Table_248-Entries_in_an_FDF_page_dictionary.pm6)
/TextAlign|[Table 344 – Additional standard layout attributes specific to block-level structure elements](gen/lib/ISO_32000/Table_344-Additional_standard_layout_attributes_specific_to_block-level_structure_elements.pm6)
/TextDecorationColor|[Table 345 – Standard layout attributes specific to inline-level structure elements](gen/lib/ISO_32000/Table_345-Standard_layout_attributes_specific_to_inline-level_structure_elements.pm6)
/TextDecorationThickness|[Table 345 – Standard layout attributes specific to inline-level structure elements](gen/lib/ISO_32000/Table_345-Standard_layout_attributes_specific_to_inline-level_structure_elements.pm6)
/TextDecorationType|[Table 345 – Standard layout attributes specific to inline-level structure elements](gen/lib/ISO_32000/Table_345-Standard_layout_attributes_specific_to_inline-level_structure_elements.pm6)
/TextIndent|[Table 344 – Additional standard layout attributes specific to block-level structure elements](gen/lib/ISO_32000/Table_344-Additional_standard_layout_attributes_specific_to_block-level_structure_elements.pm6)
/Threads|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.pm6)
/Thumb|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.pm6)
/TilingType|[Table 75 – Additional Entries Specific to a Type 1 Pattern Dictionary](gen/lib/ISO_32000/Table_75-Additional_Entries_Specific_to_a_Type_1_Pattern_Dictionary.pm6)
/TimeStamp|[Table 234 – Entries in a signature field seed value dictionary](gen/lib/ISO_32000/Table_234-Entries_in_a_signature_field_seed_value_dictionary.pm6)
/Title|[Table 153 – Entries in an outline item dictionary](gen/lib/ISO_32000/Table_153-Entries_in_an_outline_item_dictionary.pm6) [Table 317 – Entries in the document information dictionary](gen/lib/ISO_32000/Table_317-Entries_in_the_document_information_dictionary.pm6)
/ToUnicode|[Table 111 – Entries in a Type 1 font dictionary](gen/lib/ISO_32000/Table_111-Entries_in_a_Type_1_font_dictionary.pm6) [Table 112 – Entries in a Type 3 font dictionary](gen/lib/ISO_32000/Table_112-Entries_in_a_Type_3_font_dictionary.pm6) [Table 121 – Entries in a Type 0 font dictionary](gen/lib/ISO_32000/Table_121-Entries_in_a_Type_0_font_dictionary.pm6)
/Trans|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.pm6) [Table 215 – Additional entries specific to a transition action](gen/lib/ISO_32000/Table_215-Additional_entries_specific_to_a_transition_action.pm6)
/TransferFunction|[Table 130 – Entries in a type 1 halftone dictionary](gen/lib/ISO_32000/Table_130-Entries_in_a_type_1_halftone_dictionary.pm6) [Table 131 – Additional entries specific to a type 6 halftone dictionary](gen/lib/ISO_32000/Table_131-Additional_entries_specific_to_a_type_6_halftone_dictionary.pm6) [Table 132 – Additional entries specific to a type 10 halftone dictionary](gen/lib/ISO_32000/Table_132-Additional_entries_specific_to_a_type_10_halftone_dictionary.pm6) [Table 133 – Additional entries specific to a type 16 halftone dictionary](gen/lib/ISO_32000/Table_133-Additional_entries_specific_to_a_type_16_halftone_dictionary.pm6)
/TransformMethod|[Table 253 – Entries in a signature reference dictionary](gen/lib/ISO_32000/Table_253-Entries_in_a_signature_reference_dictionary.pm6)
/TransformParams|[Table 253 – Entries in a signature reference dictionary](gen/lib/ISO_32000/Table_253-Entries_in_a_signature_reference_dictionary.pm6)
/TrapRegions|[Table 367 – Additional entries specific to a trap network appearance stream](gen/lib/ISO_32000/Table_367-Additional_entries_specific_to_a_trap_network_appearance_stream.pm6)
/TrapStyles|[Table 367 – Additional entries specific to a trap network appearance stream](gen/lib/ISO_32000/Table_367-Additional_entries_specific_to_a_trap_network_appearance_stream.pm6)
/Trapped|[Table 317 – Entries in the document information dictionary](gen/lib/ISO_32000/Table_317-Entries_in_the_document_information_dictionary.pm6)
/TrimBox|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.pm6) [Table 360 – Entries in a box colour information dictionary](gen/lib/ISO_32000/Table_360-Entries_in_a_box_colour_information_dictionary.pm6)
/TrueTypeFonts|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.pm6)
/Type|[Table 14 – Optional parameters for Crypt filters](gen/lib/ISO_32000/Table_14-Optional_parameters_for_Crypt_filters.pm6) [Table 16 – Additional entries specific to an object stream dictionary](gen/lib/ISO_32000/Table_16-Additional_entries_specific_to_an_object_stream_dictionary.pm6) [Table 17 – Additional entries specific to a cross-reference stream dictionary](gen/lib/ISO_32000/Table_17-Additional_entries_specific_to_a_cross-reference_stream_dictionary.pm6) [Table 25 – Entries common to all crypt filter dictionaries](gen/lib/ISO_32000/Table_25-Entries_common_to_all_crypt_filter_dictionaries.pm6) [Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.pm6) [Table 29 – Required entries in a page tree node](gen/lib/ISO_32000/Table_29-Required_entries_in_a_page_tree_node.pm6) [Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.pm6) [Table 44 – Entries in a file specification dictionary](gen/lib/ISO_32000/Table_44-Entries_in_a_file_specification_dictionary.pm6) [Table 45 – Additional entries in an embedded file stream dictionary](gen/lib/ISO_32000/Table_45-Additional_entries_in_an_embedded_file_stream_dictionary.pm6) [Table 48 – Entries in a collection item dictionary](gen/lib/ISO_32000/Table_48-Entries_in_a_collection_item_dictionary.pm6) [Table 49 – Entries in a collection subitem dictionary](gen/lib/ISO_32000/Table_49-Entries_in_a_collection_subitem_dictionary.pm6) [Table 50 – Entries in a developer extensions dictionary](gen/lib/ISO_32000/Table_50-Entries_in_a_developer_extensions_dictionary.pm6) [Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.pm6) [Table 75 – Additional Entries Specific to a Type 1 Pattern Dictionary](gen/lib/ISO_32000/Table_75-Additional_Entries_Specific_to_a_Type_1_Pattern_Dictionary.pm6) [Table 76 – Entries in a Type 2 Pattern Dictionary](gen/lib/ISO_32000/Table_76-Entries_in_a_Type_2_Pattern_Dictionary.pm6) [Table 88 – Additional Entries Specific to a PostScript XObject Dictionary](gen/lib/ISO_32000/Table_88-Additional_Entries_Specific_to_a_PostScript_XObject_Dictionary.pm6) [Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.pm6) [Table 95 – Additional Entries Specific to a Type 1 Form Dictionary](gen/lib/ISO_32000/Table_95-Additional_Entries_Specific_to_a_Type_1_Form_Dictionary.pm6) [Table 96 – Entries Common to all Group Attributes Dictionaries](gen/lib/ISO_32000/Table_96-Entries_Common_to_all_Group_Attributes_Dictionaries.pm6) [Table 98 – Entries in an Optional Content Group Dictionary](gen/lib/ISO_32000/Table_98-Entries_in_an_Optional_Content_Group_Dictionary.pm6) [Table 99 – Entries in an Optional Content Membership Dictionary](gen/lib/ISO_32000/Table_99-Entries_in_an_Optional_Content_Membership_Dictionary.pm6) [Table 111 – Entries in a Type 1 font dictionary](gen/lib/ISO_32000/Table_111-Entries_in_a_Type_1_font_dictionary.pm6) [Table 112 – Entries in a Type 3 font dictionary](gen/lib/ISO_32000/Table_112-Entries_in_a_Type_3_font_dictionary.pm6) [Table 114 – Entries in an encoding dictionary](gen/lib/ISO_32000/Table_114-Entries_in_an_encoding_dictionary.pm6) [Table 117 – Entries in a CIDFont dictionary](gen/lib/ISO_32000/Table_117-Entries_in_a_CIDFont_dictionary.pm6) [Table 120 – Additional entries in a CMap stream dictionary](gen/lib/ISO_32000/Table_120-Additional_entries_in_a_CMap_stream_dictionary.pm6) [Table 121 – Entries in a Type 0 font dictionary](gen/lib/ISO_32000/Table_121-Entries_in_a_Type_0_font_dictionary.pm6) [Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.pm6) [Table 130 – Entries in a type 1 halftone dictionary](gen/lib/ISO_32000/Table_130-Entries_in_a_type_1_halftone_dictionary.pm6) [Table 131 – Additional entries specific to a type 6 halftone dictionary](gen/lib/ISO_32000/Table_131-Additional_entries_specific_to_a_type_6_halftone_dictionary.pm6) [Table 132 – Additional entries specific to a type 10 halftone dictionary](gen/lib/ISO_32000/Table_132-Additional_entries_specific_to_a_type_10_halftone_dictionary.pm6) [Table 133 – Additional entries specific to a type 16 halftone dictionary](gen/lib/ISO_32000/Table_133-Additional_entries_specific_to_a_type_16_halftone_dictionary.pm6) [Table 134 – Entries in a type 5 halftone dictionary](gen/lib/ISO_32000/Table_134-Entries_in_a_type_5_halftone_dictionary.pm6) [Table 144 – Entries in a soft-mask dictionary](gen/lib/ISO_32000/Table_144-Entries_in_a_soft-mask_dictionary.pm6) [Table 152 – Entries in the outline dictionary](gen/lib/ISO_32000/Table_152-Entries_in_the_outline_dictionary.pm6) [Table 155 – Entries in a collection dictionary](gen/lib/ISO_32000/Table_155-Entries_in_a_collection_dictionary.pm6) [Table 156 – Entries in a collection schema dictionary](gen/lib/ISO_32000/Table_156-Entries_in_a_collection_schema_dictionary.pm6) [Table 157 – Entries in a collection field dictionary](gen/lib/ISO_32000/Table_157-Entries_in_a_collection_field_dictionary.pm6) [Table 158 – Entries in a collection sort dictionary](gen/lib/ISO_32000/Table_158-Entries_in_a_collection_sort_dictionary.pm6) [Table 159 – Entries in a page label dictionary](gen/lib/ISO_32000/Table_159-Entries_in_a_page_label_dictionary.pm6) [Table 160 – Entries in a thread dictionary](gen/lib/ISO_32000/Table_160-Entries_in_a_thread_dictionary.pm6) [Table 161 – Entries in a bead dictionary](gen/lib/ISO_32000/Table_161-Entries_in_a_bead_dictionary.pm6) [Table 162 – Entries in a transition dictionary](gen/lib/ISO_32000/Table_162-Entries_in_a_transition_dictionary.pm6) [Table 163 – Entries in a navigation node dictionary](gen/lib/ISO_32000/Table_163-Entries_in_a_navigation_node_dictionary.pm6) [Table 164 – Entries common to all annotation dictionaries](gen/lib/ISO_32000/Table_164-Entries_common_to_all_annotation_dictionaries.pm6) [Table 166 – Entries in a border style dictionary](gen/lib/ISO_32000/Table_166-Entries_in_a_border_style_dictionary.pm6) [Table 191 – Entries in a fixed print dictionary](gen/lib/ISO_32000/Table_191-Entries_in_a_fixed_print_dictionary.pm6) [Table 193 – Entries common to all action dictionaries](gen/lib/ISO_32000/Table_193-Entries_common_to_all_action_dictionaries.pm6) [Table 233 – Entries in a signature field lock dictionary](gen/lib/ISO_32000/Table_233-Entries_in_a_signature_field_lock_dictionary.pm6) [Table 234 – Entries in a signature field seed value dictionary](gen/lib/ISO_32000/Table_234-Entries_in_a_signature_field_seed_value_dictionary.pm6) [Table 235 – Entries in a certificate seed value dictionary](gen/lib/ISO_32000/Table_235-Entries_in_a_certificate_seed_value_dictionary.pm6) [Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.pm6) [Table 253 – Entries in a signature reference dictionary](gen/lib/ISO_32000/Table_253-Entries_in_a_signature_reference_dictionary.pm6) [Table 254 – Entries in the DocMDP transform parameters dictionary](gen/lib/ISO_32000/Table_254-Entries_in_the_DocMDP_transform_parameters_dictionary.pm6) [Table 255 – Entries in the UR transform parameters dictionary](gen/lib/ISO_32000/Table_255-Entries_in_the_UR_transform_parameters_dictionary.pm6) [Table 256 – Entries in the FieldMDP transform parameters dictionary](gen/lib/ISO_32000/Table_256-Entries_in_the_FieldMDP_transform_parameters_dictionary.pm6) [Table 260 – Entries in a viewport dictionary](gen/lib/ISO_32000/Table_260-Entries_in_a_viewport_dictionary.pm6) [Table 261 – Entries in a measure dictionary](gen/lib/ISO_32000/Table_261-Entries_in_a_measure_dictionary.pm6) [Table 263 – Entries in a number format dictionary](gen/lib/ISO_32000/Table_263-Entries_in_a_number_format_dictionary.pm6) [Table 264 – Entries common to all requirement dictionaries](gen/lib/ISO_32000/Table_264-Entries_common_to_all_requirement_dictionaries.pm6) [Table 265 – Entries in a requirement handler dictionary](gen/lib/ISO_32000/Table_265-Entries_in_a_requirement_handler_dictionary.pm6) [Table 266 – Entries common to all rendition dictionaries](gen/lib/ISO_32000/Table_266-Entries_common_to_all_rendition_dictionaries.pm6) [Table 268 – Entries in a media criteria dictionary](gen/lib/ISO_32000/Table_268-Entries_in_a_media_criteria_dictionary.pm6) [Table 269 – Entries in a minimum bit depth dictionary](gen/lib/ISO_32000/Table_269-Entries_in_a_minimum_bit_depth_dictionary.pm6) [Table 270 – Entries in a minimum screen size dictionary](gen/lib/ISO_32000/Table_270-Entries_in_a_minimum_screen_size_dictionary.pm6) [Table 273 – Entries common to all media clip dictionaries](gen/lib/ISO_32000/Table_273-Entries_common_to_all_media_clip_dictionaries.pm6) [Table 275 – Entries in a media permissions dictionary](gen/lib/ISO_32000/Table_275-Entries_in_a_media_permissions_dictionary.pm6) [Table 279 – Entries in a media play parameters dictionary](gen/lib/ISO_32000/Table_279-Entries_in_a_media_play_parameters_dictionary.pm6) [Table 281 – Entries in a media duration dictionary](gen/lib/ISO_32000/Table_281-Entries_in_a_media_duration_dictionary.pm6) [Table 282 – Entries in a media screen parameters dictionary](gen/lib/ISO_32000/Table_282-Entries_in_a_media_screen_parameters_dictionary.pm6) [Table 284 – Entries in a floating window parameters dictionary](gen/lib/ISO_32000/Table_284-Entries_in_a_floating_window_parameters_dictionary.pm6) [Table 285 – Entries common to all media offset dictionaries](gen/lib/ISO_32000/Table_285-Entries_common_to_all_media_offset_dictionaries.pm6) [Table 289 – Entries in a timespan dictionary](gen/lib/ISO_32000/Table_289-Entries_in_a_timespan_dictionary.pm6) [Table 290 – Entries in a media players dictionary](gen/lib/ISO_32000/Table_290-Entries_in_a_media_players_dictionary.pm6) [Table 291 – Entries in a media player info dictionary](gen/lib/ISO_32000/Table_291-Entries_in_a_media_player_info_dictionary.pm6) [Table 292 – Entries in a software identifier dictionary](gen/lib/ISO_32000/Table_292-Entries_in_a_software_identifier_dictionary.pm6) [Table 294 – Additional entries specific to a sound object](gen/lib/ISO_32000/Table_294-Additional_entries_specific_to_a_sound_object.pm6) [Table 297 – Entries in a slideshow dictionary](gen/lib/ISO_32000/Table_297-Entries_in_a_slideshow_dictionary.pm6) [Table 300 – Entries in a 3D stream dictionary](gen/lib/ISO_32000/Table_300-Entries_in_a_ThreeD_stream_dictionary.pm6) [Table 301 – Entries in an 3D animation style dictionary](gen/lib/ISO_32000/Table_301-Entries_in_an_ThreeD_animation_style_dictionary.pm6) [Table 303 – Entries in a 3D reference dictionary](gen/lib/ISO_32000/Table_303-Entries_in_a_ThreeD_reference_dictionary.pm6) [Table 304 – Entries in a 3D view dictionary](gen/lib/ISO_32000/Table_304-Entries_in_a_ThreeD_view_dictionary.pm6) [Table 306 – Entries in a 3D background dictionary](gen/lib/ISO_32000/Table_306-Entries_in_a_ThreeD_background_dictionary.pm6) [Table 307 – Entries in a render mode dictionary](gen/lib/ISO_32000/Table_307-Entries_in_a_render_mode_dictionary.pm6) [Table 309 – Entries in a 3D lighting scheme dictionary](gen/lib/ISO_32000/Table_309-Entries_in_a_ThreeD_lighting_scheme_dictionary.pm6) [Table 311 – Entries in a 3D cross section dictionary](gen/lib/ISO_32000/Table_311-Entries_in_a_ThreeD_cross_section_dictionary.pm6) [Table 312 – Entries in a 3D node dictionary](gen/lib/ISO_32000/Table_312-Entries_in_a_ThreeD_node_dictionary.pm6) [Table 313 – Entries in an external data dictionary used to markup 3D annotations](gen/lib/ISO_32000/Table_313-Entries_in_an_external_data_dictionary_used_to_markup_ThreeD_annotations.pm6) [Table 315 – Additional entries in a metadata stream dictionary](gen/lib/ISO_32000/Table_315-Additional_entries_in_a_metadata_stream_dictionary.pm6) [Table 322 – Entries in the structure tree root](gen/lib/ISO_32000/Table_322-Entries_in_the_structure_tree_root.pm6) [Table 323 – Entries in a structure element dictionary](gen/lib/ISO_32000/Table_323-Entries_in_a_structure_element_dictionary.pm6) [Table 324 – Entries in a marked-content reference dictionary](gen/lib/ISO_32000/Table_324-Entries_in_a_marked-content_reference_dictionary.pm6) [Table 325 – Entries in an object reference dictionary](gen/lib/ISO_32000/Table_325-Entries_in_an_object_reference_dictionary.pm6) [Table 330 – Property list entries for artifacts](gen/lib/ISO_32000/Table_330-Property_list_entries_for_artifacts.pm6) [Table 352 – Entries common to all Web Capture content sets](gen/lib/ISO_32000/Table_352-Entries_common_to_all_Web_Capture_content_sets.pm6) [Table 365 – Entries in an output intent dictionary](gen/lib/ISO_32000/Table_365-Entries_in_an_output_intent_dictionary.pm6)
/U|[Table 21 – Additional encryption dictionary entries for the standard security handler](gen/lib/ISO_32000/Table_21-Additional_encryption_dictionary_entries_for_the_standard_security_handler.pm6) [Table 194 – Entries in an annotation’s additional-actions dictionary](gen/lib/ISO_32000/Table_194-Entries_in_an_annotations_additional-actions_dictionary.pm6) [Table 263 – Entries in a number format dictionary](gen/lib/ISO_32000/Table_263-Entries_in_a_number_format_dictionary.pm6) [Table 292 – Entries in a software identifier dictionary](gen/lib/ISO_32000/Table_292-Entries_in_a_software_identifier_dictionary.pm6) [Table 356 – Entries in a URL alias dictionary](gen/lib/ISO_32000/Table_356-Entries_in_a_URL_alias_dictionary.pm6)
/U3DPath|[Table 304 – Entries in a 3D view dictionary](gen/lib/ISO_32000/Table_304-Entries_in_a_ThreeD_view_dictionary.pm6)
/UC|[Table 284 – Entries in a floating window parameters dictionary](gen/lib/ISO_32000/Table_284-Entries_in_a_floating_window_parameters_dictionary.pm6)
/UCR|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.pm6)
/UCR2|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.pm6)
/UF|[Table 44 – Entries in a file specification dictionary](gen/lib/ISO_32000/Table_44-Entries_in_a_file_specification_dictionary.pm6)
/UR3|[Table 258 – Entries in a permissions dictionary](gen/lib/ISO_32000/Table_258-Entries_in_a_permissions_dictionary.pm6)
/URI|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.pm6) [Table 206 – Additional entries specific to a URI action](gen/lib/ISO_32000/Table_206-Additional_entries_specific_to_a_URI_action.pm6)
/URIActions|[Table 259 – Entries in a legal attestation dictionary](gen/lib/ISO_32000/Table_259-Entries_in_a_legal_attestation_dictionary.pm6)
/URL|[Table 235 – Entries in a certificate seed value dictionary](gen/lib/ISO_32000/Table_235-Entries_in_a_certificate_seed_value_dictionary.pm6) [Table 357 – Entries in a Web Capture command dictionary](gen/lib/ISO_32000/Table_357-Entries_in_a_Web_Capture_command_dictionary.pm6)
/URLS|[Table 31 – Entries in the name dictionary](gen/lib/ISO_32000/Table_31-Entries_in_the_name_dictionary.pm6)
/URLType|[Table 235 – Entries in a certificate seed value dictionary](gen/lib/ISO_32000/Table_235-Entries_in_a_certificate_seed_value_dictionary.pm6)
/Unix|[Table 44 – Entries in a file specification dictionary](gen/lib/ISO_32000/Table_44-Entries_in_a_file_specification_dictionary.pm6) [Table 203 – Additional entries specific to a launch action](gen/lib/ISO_32000/Table_203-Additional_entries_specific_to_a_launch_action.pm6)
/Usage|[Table 98 – Entries in an Optional Content Group Dictionary](gen/lib/ISO_32000/Table_98-Entries_in_an_Optional_Content_Group_Dictionary.pm6)
/UseCMap|[Table 120 – Additional entries in a CMap stream dictionary](gen/lib/ISO_32000/Table_120-Additional_entries_in_a_CMap_stream_dictionary.pm6)
/User|[Table 102 – Entries in an Optional Content Usage Dictionary](gen/lib/ISO_32000/Table_102-Entries_in_an_Optional_Content_Usage_Dictionary.pm6)
/UserProperties|[Table 321 – Entries in the mark information dictionary](gen/lib/ISO_32000/Table_321-Entries_in_the_mark_information_dictionary.pm6)
/UserUnit|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.pm6)
/V|[Table 20 – Entries common to all encryption dictionaries](gen/lib/ISO_32000/Table_20-Entries_common_to_all_encryption_dictionaries.pm6) [Table 44 – Entries in a file specification dictionary](gen/lib/ISO_32000/Table_44-Entries_in_a_file_specification_dictionary.pm6) [Table 157 – Entries in a collection field dictionary](gen/lib/ISO_32000/Table_157-Entries_in_a_collection_field_dictionary.pm6) [Table 161 – Entries in a bead dictionary](gen/lib/ISO_32000/Table_161-Entries_in_a_bead_dictionary.pm6) [Table 191 – Entries in a fixed print dictionary](gen/lib/ISO_32000/Table_191-Entries_in_a_fixed_print_dictionary.pm6) [Table 196 – Entries in a form field’s additional-actions dictionary](gen/lib/ISO_32000/Table_196-Entries_in_a_form_fields_additional-actions_dictionary.pm6) [Table 216 – Additional entries specific to a go-to-3D-view action](gen/lib/ISO_32000/Table_216-Additional_entries_specific_to_a_go-to-ThreeD-view_action.pm6) [Table 220 – Entries common to all field dictionaries](gen/lib/ISO_32000/Table_220-Entries_common_to_all_field_dictionaries.pm6) [Table 234 – Entries in a signature field seed value dictionary](gen/lib/ISO_32000/Table_234-Entries_in_a_signature_field_seed_value_dictionary.pm6) [Table 246 – Entries in an FDF field dictionary](gen/lib/ISO_32000/Table_246-Entries_in_an_FDF_field_dictionary.pm6) [Table 252 – Entries in a signature dictionary](gen/lib/ISO_32000/Table_252-Entries_in_a_signature_dictionary.pm6) [Table 254 – Entries in the DocMDP transform parameters dictionary](gen/lib/ISO_32000/Table_254-Entries_in_the_DocMDP_transform_parameters_dictionary.pm6) [Table 255 – Entries in the UR transform parameters dictionary](gen/lib/ISO_32000/Table_255-Entries_in_the_UR_transform_parameters_dictionary.pm6) [Table 256 – Entries in the FieldMDP transform parameters dictionary](gen/lib/ISO_32000/Table_256-Entries_in_the_FieldMDP_transform_parameters_dictionary.pm6) [Table 268 – Entries in a media criteria dictionary](gen/lib/ISO_32000/Table_268-Entries_in_a_media_criteria_dictionary.pm6) [Table 269 – Entries in a minimum bit depth dictionary](gen/lib/ISO_32000/Table_269-Entries_in_a_minimum_bit_depth_dictionary.pm6) [Table 270 – Entries in a minimum screen size dictionary](gen/lib/ISO_32000/Table_270-Entries_in_a_minimum_screen_size_dictionary.pm6) [Table 280 – Entries in a media play parameters MH/BE dictionary](gen/lib/ISO_32000/Table_280-Entries_in_a_media_play_parameters_MH-BE_dictionary.pm6) [Table 289 – Entries in a timespan dictionary](gen/lib/ISO_32000/Table_289-Entries_in_a_timespan_dictionary.pm6) [Table 312 – Entries in a 3D node dictionary](gen/lib/ISO_32000/Table_312-Entries_in_a_ThreeD_node_dictionary.pm6) [Table 329 – Entries in a user property dictionary](gen/lib/ISO_32000/Table_329-Entries_in_a_user_property_dictionary.pm6) [Table 350 – Entries in the Web Capture information dictionary](gen/lib/ISO_32000/Table_350-Entries_in_the_Web_Capture_information_dictionary.pm6)
/VA|[Table 300 – Entries in a 3D stream dictionary](gen/lib/ISO_32000/Table_300-Entries_in_a_ThreeD_stream_dictionary.pm6)
/VE|[Table 99 – Entries in an Optional Content Membership Dictionary](gen/lib/ISO_32000/Table_99-Entries_in_an_Optional_Content_Membership_Dictionary.pm6)
/VP|[Table 30 – Entries in a page object](gen/lib/ISO_32000/Table_30-Entries_in_a_page_object.pm6)
/Version|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.pm6) [Table 242 – Entries in the FDF catalog dictionary](gen/lib/ISO_32000/Table_242-Entries_in_the_FDF_catalog_dictionary.pm6) [Table 366 – Additional entries specific to a trap network annotation](gen/lib/ISO_32000/Table_366-Additional_entries_specific_to_a_trap_network_annotation.pm6)
/Vertices|[Table 178 – Additional entries specific to a polygon or polyline annotation](gen/lib/ISO_32000/Table_178-Additional_entries_specific_to_a_polygon_or_polyline_annotation.pm6)
/VerticesPerRow|[Table 83 – Additional Entries Specific to a Type 5 Shading Dictionary](gen/lib/ISO_32000/Table_83-Additional_Entries_Specific_to_a_Type_5_Shading_Dictionary.pm6)
/View|[Table 102 – Entries in an Optional Content Usage Dictionary](gen/lib/ISO_32000/Table_102-Entries_in_an_Optional_Content_Usage_Dictionary.pm6) [Table 155 – Entries in a collection dictionary](gen/lib/ISO_32000/Table_155-Entries_in_a_collection_dictionary.pm6)
/ViewArea|[Table 150 – Entries in a viewer preferences dictionary](gen/lib/ISO_32000/Table_150-Entries_in_a_viewer_preferences_dictionary.pm6)
/ViewClip|[Table 150 – Entries in a viewer preferences dictionary](gen/lib/ISO_32000/Table_150-Entries_in_a_viewer_preferences_dictionary.pm6)
/ViewerPreferences|[Table 28 – Entries in the catalog dictionary](gen/lib/ISO_32000/Table_28-Entries_in_the_catalog_dictionary.pm6)
/Volume|[Table 208 – Additional entries specific to a sound action](gen/lib/ISO_32000/Table_208-Additional_entries_specific_to_a_sound_action.pm6) [Table 296 – Entries in a movie activation dictionary](gen/lib/ISO_32000/Table_296-Entries_in_a_movie_activation_dictionary.pm6)
/W|[Table 17 – Additional entries specific to a cross-reference stream dictionary](gen/lib/ISO_32000/Table_17-Additional_entries_specific_to_a_cross-reference_stream_dictionary.pm6) [Table 117 – Entries in a CIDFont dictionary](gen/lib/ISO_32000/Table_117-Entries_in_a_CIDFont_dictionary.pm6) [Table 166 – Entries in a border style dictionary](gen/lib/ISO_32000/Table_166-Entries_in_a_border_style_dictionary.pm6) [Table 283 – Entries in a media screen parameters MH/BE dictionary](gen/lib/ISO_32000/Table_283-Entries_in_a_media_screen_parameters_MH-BE_dictionary.pm6) [Table 361 – Entries in a box style dictionary](gen/lib/ISO_32000/Table_361-Entries_in_a_box_style_dictionary.pm6)
/W2|[Table 117 – Entries in a CIDFont dictionary](gen/lib/ISO_32000/Table_117-Entries_in_a_CIDFont_dictionary.pm6)
/WC|[Table 197 – Entries in the document catalog’s additional-actions dictionary](gen/lib/ISO_32000/Table_197-Entries_in_the_document_catalogs_additional-actions_dictionary.pm6)
/WMode|[Table 120 – Additional entries in a CMap stream dictionary](gen/lib/ISO_32000/Table_120-Additional_entries_in_a_CMap_stream_dictionary.pm6)
/WP|[Table 197 – Entries in the document catalog’s additional-actions dictionary](gen/lib/ISO_32000/Table_197-Entries_in_the_document_catalogs_additional-actions_dictionary.pm6)
/WS|[Table 197 – Entries in the document catalog’s additional-actions dictionary](gen/lib/ISO_32000/Table_197-Entries_in_the_document_catalogs_additional-actions_dictionary.pm6)
/WhitePoint|[Table 63 – Entries in a CalGray Colour Space Dictionary](gen/lib/ISO_32000/Table_63-Entries_in_a_CalGray_Colour_Space_Dictionary.pm6) [Table 64 – Entries in a CalRGB Colour Space Dictionary](gen/lib/ISO_32000/Table_64-Entries_in_a_CalRGB_Colour_Space_Dictionary.pm6) [Table 65 – Entries in a Lab Colour Space Dictionary](gen/lib/ISO_32000/Table_65-Entries_in_a_Lab_Colour_Space_Dictionary.pm6)
/Width|[Table 89 – Additional Entries Specific to an Image Dictionary](gen/lib/ISO_32000/Table_89-Additional_Entries_Specific_to_an_Image_Dictionary.pm6) [Table 131 – Additional entries specific to a type 6 halftone dictionary](gen/lib/ISO_32000/Table_131-Additional_entries_specific_to_a_type_6_halftone_dictionary.pm6) [Table 133 – Additional entries specific to a type 16 halftone dictionary](gen/lib/ISO_32000/Table_133-Additional_entries_specific_to_a_type_16_halftone_dictionary.pm6) [Table 344 – Additional standard layout attributes specific to block-level structure elements](gen/lib/ISO_32000/Table_344-Additional_standard_layout_attributes_specific_to_block-level_structure_elements.pm6)
/Width2|[Table 133 – Additional entries specific to a type 16 halftone dictionary](gen/lib/ISO_32000/Table_133-Additional_entries_specific_to_a_type_16_halftone_dictionary.pm6)
/Widths|[Table 111 – Entries in a Type 1 font dictionary](gen/lib/ISO_32000/Table_111-Entries_in_a_Type_1_font_dictionary.pm6) [Table 112 – Entries in a Type 3 font dictionary](gen/lib/ISO_32000/Table_112-Entries_in_a_Type_3_font_dictionary.pm6)
/Win|[Table 203 – Additional entries specific to a launch action](gen/lib/ISO_32000/Table_203-Additional_entries_specific_to_a_launch_action.pm6)
/WritingMode|[Table 343 – Standard layout attributes common to all standard structure types](gen/lib/ISO_32000/Table_343-Standard_layout_attributes_common_to_all_standard_structure_types.pm6)
/X|[Table 194 – Entries in an annotation’s additional-actions dictionary](gen/lib/ISO_32000/Table_194-Entries_in_an_annotations_additional-actions_dictionary.pm6) [Table 262 – Additional entries in a rectilinear measure dictionary](gen/lib/ISO_32000/Table_262-Additional_entries_in_a_rectilinear_measure_dictionary.pm6)
/XFA|[Table 218 – Entries in the interactive form dictionary](gen/lib/ISO_32000/Table_218-Entries_in_the_interactive_form_dictionary.pm6)
/XHeight|[Table 122 – Entries common to all font descriptors](gen/lib/ISO_32000/Table_122-Entries_common_to_all_font_descriptors.pm6)
/XN|[Table 304 – Entries in a 3D view dictionary](gen/lib/ISO_32000/Table_304-Entries_in_a_ThreeD_view_dictionary.pm6)
/XObject|[Table 33 – Entries in a resource dictionary](gen/lib/ISO_32000/Table_33-Entries_in_a_resource_dictionary.pm6)
/XRefStm|[Table 19 – Additional entries in a hybrid-reference file’s trailer dictionary](gen/lib/ISO_32000/Table_19-Additional_entries_in_a_hybrid-reference_files_trailer_dictionary.pm6)
/XStep|[Table 75 – Additional Entries Specific to a Type 1 Pattern Dictionary](gen/lib/ISO_32000/Table_75-Additional_Entries_Specific_to_a_Type_1_Pattern_Dictionary.pm6)
/Xsquare|[Table 132 – Additional entries specific to a type 10 halftone dictionary](gen/lib/ISO_32000/Table_132-Additional_entries_specific_to_a_type_10_halftone_dictionary.pm6)
/Y|[Table 262 – Additional entries in a rectilinear measure dictionary](gen/lib/ISO_32000/Table_262-Additional_entries_in_a_rectilinear_measure_dictionary.pm6)
/YStep|[Table 75 – Additional Entries Specific to a Type 1 Pattern Dictionary](gen/lib/ISO_32000/Table_75-Additional_Entries_Specific_to_a_Type_1_Pattern_Dictionary.pm6)
/Ysquare|[Table 132 – Additional entries specific to a type 10 halftone dictionary](gen/lib/ISO_32000/Table_132-Additional_entries_specific_to_a_type_10_halftone_dictionary.pm6)
/Z|[Table 268 – Entries in a media criteria dictionary](gen/lib/ISO_32000/Table_268-Entries_in_a_media_criteria_dictionary.pm6)
/Zoom|[Table 102 – Entries in an Optional Content Usage Dictionary](gen/lib/ISO_32000/Table_102-Entries_in_an_Optional_Content_Usage_Dictionary.pm6)
/ca|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.pm6)
/checked|[Table 348 – PrintField attributes](gen/lib/ISO_32000/Table_348-PrintField_attributes.pm6)
/color|[Table 52 – Device-Independent Graphics State Parameters](gen/lib/ISO_32000/Table_52-Device-Independent_Graphics_State_Parameters.pm6)
/flatness|[Table 53 – Device-Dependent Graphics State Parameters](gen/lib/ISO_32000/Table_53-Device-Dependent_Graphics_State_Parameters.pm6)
/halftone|[Table 53 – Device-Dependent Graphics State Parameters](gen/lib/ISO_32000/Table_53-Device-Dependent_Graphics_State_Parameters.pm6)
/op|[Table 58 – Entries in a Graphics State Parameter Dictionary](gen/lib/ISO_32000/Table_58-Entries_in_a_Graphics_State_Parameter_Dictionary.pm6)
/overprint|[Table 53 – Device-Dependent Graphics State Parameters](gen/lib/ISO_32000/Table_53-Device-Dependent_Graphics_State_Parameters.pm6)
/smoothness|[Table 53 – Device-Dependent Graphics State Parameters](gen/lib/ISO_32000/Table_53-Device-Dependent_Graphics_State_Parameters.pm6)
/transfer|[Table 53 – Device-Dependent Graphics State Parameters](gen/lib/ISO_32000/Table_53-Device-Dependent_Graphics_State_Parameters.pm6)
