#!perl
#   - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
#
#   file: t/hook.t #
#
#   Copyright © 2015 Van de Bugger
#
#   This file is part of perl-Dist-Zilla-Plugin-Hook.
#
#   perl-Dist-Zilla-Plugin-Hook is free software: you can redistribute it and/or modify it under
#   the terms of the GNU General Public License as published by the Free Software Foundation,
#   either version 3 of the License, or (at your option) any later version.
#
#   perl-Dist-Zilla-Plugin-Hook is distributed in the hope that it will be useful, but WITHOUT ANY
#   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#   PURPOSE. See the GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License along with
#   perl-Dist-Zilla-Plugin-Hook. If not, see <http://www.gnu.org/licenses/>.
#
#   - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

use strict;
use warnings;
use lib 't';

use Test::More;
use Test::Routine::Util;
use Test::Deep qw{ re };

# AutoPrereqs hints:
use Dist::Zilla::Plugin::ReportPhase    ();
use Throwable                           ();

plan tests => 19;

my $role  = 'HookTester';
my $abort = 'Dist::Zilla died in construction: Aborting...';

#   `$self` and `$zilla` variables are defined.
run_tests( '$self and $zilla', $role, {
    ini_body => [
        [ 'Hook::Init', {
            'hook' => [ '$self->log( $zilla->name );' ],
        } ],
        'GatherDir',
    ],
    exp_messages => [
        '[Hook::Init] Dummy'
    ],
} );

#   `$plugin` variable is defined too.
run_tests( '$plugin == $self', $role, {
    ini_body => [
        [ 'Hook::Init', {
            'hook' => [ '$self->log( $plugin == $self ? "OK" : "NOT OK" );' ],
        } ],
        'GatherDir',
    ],
    exp_messages => [
        '[Hook::Init] OK',
    ],
} );

#   `$dist` variable is defined too.
run_tests( '$dist == $zilla', $role, {
    ini_body => [
        [ 'Hook::Init', {
            'hook' => [ '$self->log( $dist == $zilla ? "OK" : "NOT OK" );' ],
        } ],
        'GatherDir',
    ],
    exp_messages => [
        '[Hook::Init] OK',
    ],
} );

#   `$arg` variable is defined if method is called with an argument.
run_tests( '$arg defined', $role, {
    ini_body => [
        [ 'Hook::LicenseProvider', {    # `Dist::Zilla` provides an argument for `LicenseProvider`.
            'hook' => [
                '$plugin->log( [',
                '    "Copyright (C) %d %s", $arg->{ copyright_year }, $arg->{ copyright_holder }',
                '] );',
            ],
        } ],
        'GatherDir',
    ],
    exp_messages => [
        '[Hook::LicenseProvider] Copyright (C) 2007 John Doe',
    ],
} );

SKIP: {
    skip 'Not yet decided', 1;      # TODO: Should I declare the variable and let it undefined or
                                    #   not declare variable at all? What about warnings?
    run_tests( '$arg not defined', $role, {
        #   Check $arg variable is not defined if plugin does receives an argument.
        ini_body => [
            [ 'Hook::BeforeBuild', {
                'hook' => [
                    '$plugin->log( $arg );',
                ],
            } ],
            'GatherDir',
        ],
        exp_exception => 'Aborting...',
        exp_messages => [
            re( qr{^\[Hook::BeforeBuild\] Global symbol "\$arg" requires explicit package name.* at Hook::BeforeBuild line 1\b} ),
        ],
    } );
};

#   `@_` variable defined, if method is called with argument.
#   `provide_license` method of `LicenseProvider` role is called with `HashRef` with two keys:
#   `copyright_holder` and `copyright_year`. Let us check it.
run_tests( '@_', $role, {
    ini_body => [
        [ 'Hook::LicenseProvider', {
            'hook' => [
                'my ( $args ) = @_;',
                'my ( $holder, $year ) = map( $args->{ "copyright_$_" }, qw{ holder year } );',
                '$plugin->log( [ "Copyright (C) %d %s", $year, $holder ] );',
            ],
        } ],
        'GatherDir',
    ],
    exp_messages => [
        '[Hook::LicenseProvider] Copyright (C) 2007 John Doe',
    ],
} );

#   `use strict;` is in effect (thanks to `Moose`).
run_tests( '"use strict;" is in effect', $role, {
    ini_body => [
        [ 'Hook::BeforeBuild', {
            'hook' => [
                '$assa = 123;',
            ],
        } ],
        'GatherDir',
    ],
    exp_exception => 'Aborting...',
    exp_messages => [
        re( qr{^\[Hook::BeforeBuild\] Global symbol "\$assa" requires explicit package name.* at Hook::BeforeBuild line 1\b} ),
    ],
} );

#   `use warnings;` is in effect (thanks to `Moose`).
run_tests( '"use warnings;" is in effect', $role, {
    #   Using undefined variable causes warning in log, but does not break execution.
    ini_body => [
        [ 'Hook::BeforeBuild', {
            'hook' => [
                'my $assa;',
                'my $qwerty = $assa + 1;',
            ],
        } ],
        'GatherDir',
    ],
    exp_messages => [
        '[Hook::BeforeBuild] Use of uninitialized value $assa in addition (+) at Hook::BeforeBuild line 2.',
    ],
} );

#   Semicolon (without preceeding space) works as statement separator.
run_tests( 'semicolon not preceeded by space', $role, {
    ini_body => [
        [ 'Hook::Init', {
            'hook' => [ '$plugin->log( "Assa" ); $plugin->log( "Qwerty" );' ],
        } ],
        'GatherDir',
    ],
    exp_messages => [
        '[Hook::Init] Assa',
        '[Hook::Init] Qwerty',
    ],
} );

#   Semicolon (with preceeding space) works as comment starter.
run_tests( 'semicolon preceeded by space', $role, {
    ini_body => [
        [ 'Hook::Init', {
            'hook' => [ '$plugin->log( "Assa" ) ; $plugin->log( "Qwerty" );' ],
        } ],
        'GatherDir',
    ],
    exp_messages => [
        '[Hook::Init] Assa',    # Only one message, no "Qwerty".
    ],
} );

my $hook = { hook => [
    '$plugin->log( "hook" );',
    'if ( $plugin->plugin_name eq "Hook::MetaProvider" ) {',
    '    return {};',
    '} else {',
    '    return undef;',
    '};',
] };

run_tests( 'Phases', $role, {
    msg_grepper => sub {
        return $_ =~ m{^\[(?:Phase_(?:Begins|Ends)|Hook::.+?)\] };
    },
    ini_body => [
        'ReportPhase/Phase_Begins',
        [ 'Hook::AfterBuild',            $hook ],
        [ 'Hook::AfterMint',             $hook ],
        [ 'Hook::AfterRelease',          $hook ],
        [ 'Hook::BeforeArchive',         $hook ],
        [ 'Hook::BeforeBuild',           $hook ],
        [ 'Hook::BeforeMint',            $hook ],
        [ 'Hook::BeforeRelease',         $hook ],
        [ 'Hook::FileGatherer',          $hook ],
        [ 'Hook::FileMunger',            $hook ],
        [ 'Hook::FilePruner',            $hook ],
        [ 'Hook::Init',                  $hook ],
        [ 'Hook::InstallTool',           $hook ],
        [ 'Hook::LicenseProvider',       $hook ],
        [ 'Hook::MetaProvider',          $hook ],
        [ 'Hook::NameProvider',          $hook ],   # TODO: Why this is not shown in the log?
        [ 'Hook::PrereqSource',          $hook ],
        [ 'Hook::Releaser',              $hook ],
        [ 'Hook::ReleaseStatusProvider', $hook ],
        [ 'Hook::VersionProvider',       $hook ],
        'GatherDir',
        'ReportPhase/Phase_Ends',
    ],
    exp_messages => [
        '[Hook::Init] hook',
        '[Phase_Begins] ########## Before Build ##########',
        '[Hook::BeforeBuild] hook',
        '[Phase_Ends] ########## Before Build ##########',
        '[Phase_Begins] ########## Gather Files ##########',
        '[Hook::FileGatherer] hook',
        '[Phase_Ends] ########## Gather Files ##########',
        '[Phase_Begins] ########## Prune Files ##########',
        '[Hook::FilePruner] hook',
        '[Phase_Ends] ########## Prune Files ##########',
        '[Phase_Begins] ########## Provide Version ##########',
        '[Hook::VersionProvider] hook',
        '[Phase_Ends] ########## Provide Version ##########',
        '[Phase_Begins] ########## Munge Files ##########',
        '[Hook::FileMunger] hook',
        '[Phase_Ends] ########## Munge Files ##########',
        '[Phase_Begins] ########## Bundle Config ##########',   # TODO: Support `PluginBundle`?
        '[Hook::PrereqSource] hook',
        '[Phase_Ends] ########## Bundle Config ##########',
        '[Hook::LicenseProvider] hook',                 # ReportPhase does not have such phase
        '[Hook::ReleaseStatusProvider] hook',
        '[Phase_Begins] ########## Metadata ##########',
        '[Hook::MetaProvider] hook',
        '[Phase_Ends] ########## Metadata ##########',
        '[Phase_Begins] ########## Setup Installer ##########',
        '[Hook::InstallTool] hook',
        '[Phase_Ends] ########## Setup Installer ##########',
        '[Phase_Begins] ########## After Build ##########',
        '[Hook::AfterBuild] hook',
        '[Phase_Ends] ########## After Build ##########',
    ],
} );

#   Hook dies, line number reported correctly.
run_tests( 'die in hook', $role, {
    ini_body => [
        [ 'Hook::Init', {
            'hook' => [
                '#     this is line 1',
                'die "oops"; # line 2',                         # Die is in line 2.
                '#     this is line 3',
            ],
        } ],
        'GatherDir',
    ],
    exp_exception => $abort,
    exp_messages  => [
        re( qr{^\[Hook::Init\] oops at Hook::Init line 2\b} ),  # Verify the line.
    ],
} );

#   Hook dies, but throws not a string but an object.
run_tests( 'die with object', $role, {
    ini_body => [
        [ 'Hook::Init', {
            'hook' => [
                'use strict;',
                '{   package Exception;',
                '    use Moose;',
                '    with "Throwable";',
                '    has message => ( is => "ro" );',
                '    sub string { shift->message };',
                '    use overload q{""} => \\&string;',
                '}',
                'Exception->throw( { message => "Assa" } );',
            ],
        } ],
        'GatherDir',
    ],
    exp_exception => $abort,
    exp_messages  => [
        re( qr{^\[Hook::Init\] Assa\b} ),   # Object stringified.
    ],
} );

#   Named hook dies, line number reported correctly.
run_tests( 'die in named hook', $role, {
    ini_body => [
        [ 'Hook::Init', 'HookName', {
            #   Hook name must include "hook" word,
            #   otherwise messages will be filtered out by `HookTester`.
            'hook' => [
                '#     this is line 1',
                '#     this is line 2',
                'die "oops"; # line 3',                         # Die is in line 3.
            ],
        } ],
        'GatherDir',
    ],
    exp_exception => $abort,
    exp_messages  => [
        re( qr{^\[HookName\] oops at HookName line 3\b} ),  # Verify the line.
    ],
} );

#   Named hook dies, hook name contains spaces.
run_tests( 'hook name contains space', $role, {
    ini_body => [
        [ 'Hook::Init', 'hook name', {
            #   Hook name must include "hook" word,
            #   otherwise messages will be filtered out by `HookTester`.
            'hook' => [
                '#     this is line 1',
                '#     this is line 2',
                'die "oops"; # line 3',                         # Die is in line 3.
            ],
        } ],
        'GatherDir',
    ],
    exp_exception => $abort,
    exp_messages  => [
        re( qr{^\[hook name\] oops at hook name line 3\b} ),  # Verify the line.
    ],
} );

#   Named hook dies, hook name contains quote.
#   Perl `#line` directive does not allow (escaped) quotes in filename. Following directive is
#   incorrect and will be ignored by Perl:
#       #line 1 "hook \"name\""
#   To avoid totally wrong line numbers, `Hooker` replaces quotes with apostrophes.
run_tests( 'hook name contains quote', $role, {
    ini_body => [
        [ 'Hook::Init', 'hook "name"', {
            #                ^^^  ^^^ Note quotes
            'hook' => [
                '#     this is line 1',
                '#     this is line 2',
                'die "oops"; # line 3',                         # Die is in line 3.
            ],
        } ],
        'GatherDir',
    ],
    exp_exception => $abort,
    exp_messages  => [
        re( qr{^\[hook "name"\] oops at hook 'name' line 3\b} ),
        #             ^^^  ^^^              ^^^  ^^^ Note apostrophes.
    ],
} );

#   Prologue is executed in the beginning of every hook.
run_tests( 'prologue', $role, {
    ini_body => [
        [ 'Hook/prologue', {
            'hook' => [
                '$self->log( "prologue" );',
            ],
        } ],
        [ 'Hook::Init', {
            'hook' => [
                '$self->log( "hook" );',
            ],
        } ],
        [ 'Hook::BeforeBuild', {
            'hook' => [
                '$self->log( "hook" );',
            ],
        } ],
        'GatherDir',
    ],
    exp_messages  => [
        '[Hook::Init] prologue',            # Prologue before `Hook::Init`.
        '[Hook::Init] hook',
        '[Hook::BeforeBuild] prologue',     # Prologue before `Hook::BeforeBuild`.
        '[Hook::BeforeBuild] hook',
    ],
} );

#   Hook dies in prologue. Message printed from the appropriate plugin, but error location
#   is inn prologue.
run_tests( 'prologue dies', $role, {
    ini_body => [
        [ 'Hook/prologue', {
            'hook' => [
                '$self->log( "prologue" );',
                'die "oops";',
            ],
        } ],
        [ 'Hook::Init', {
            'hook' => [
                '$self->log( "init" );',
            ],
        } ],
        'GatherDir',
    ],
    exp_exception => $abort,
    exp_messages  => [
        '[Hook::Init] prologue',
        re( qr{\[Hook::Init\] oops at prologue line 2\b} ),
        #       ^^^^^^^^^^^^^         ^^^^^^^^
    ],
} );

#   Hook dies in "main body", prologue does not affect line numbers.
run_tests( 'prologue + body dies', $role, {
    ini_body => [
        [ 'Hook/prologue', {
            'hook' => [
                '$self->log( "prologue" );',
            ],
        } ],
        [ 'Hook::Init', {
            'hook' => [
                '$self->log( "init" );',
                'die "oops";',
            ],
        } ],
        'GatherDir',
    ],
    exp_exception => $abort,
    exp_messages  => [
        '[Hook::Init] prologue',
        '[Hook::Init] init',
        re( qr{\[Hook::Init\] oops at Hook::Init line 2\b} ),
        #                             ^^^^^^^^^^
    ],
} );

done_testing;

exit( 0 );

# end of file #
